/*
 * Decompiled with CFR 0.152.
 */
package tdl.record.sourcecode.record;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.eclipse.jgit.api.Git;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tdl.record.sourcecode.content.SourceCodeProvider;
import tdl.record.sourcecode.metrics.SourceCodeRecordingListener;
import tdl.record.sourcecode.metrics.SourceCodeRecordingMetricsCollector;
import tdl.record.sourcecode.record.SourceCodeRecorderException;
import tdl.record.sourcecode.snapshot.Snapshot;
import tdl.record.sourcecode.snapshot.SnapshotRecorderException;
import tdl.record.sourcecode.snapshot.file.Writer;
import tdl.record.sourcecode.time.SystemMonotonicTimeSource;
import tdl.record.sourcecode.time.TimeSource;

public class SourceCodeRecorder {
    private static final Logger log = LoggerFactory.getLogger(SourceCodeRecorder.class);
    private final SourceCodeProvider sourceCodeProvider;
    private final Path outputRecordingFilePath;
    private final TimeSource timeSource;
    private final long snapshotIntervalMillis;
    private final long recordingStartTimestamp;
    private final int keySnapshotSpacing;
    private final AtomicBoolean shouldStopJob;
    private final SourceCodeRecordingListener sourceCodeRecordingListener;
    private Queue<String> tagQueue;

    public static void runSanityCheck() {
        try {
            Path gitDirectory = Files.createTempDirectory("sanity_check", new FileAttribute[0]);
            Git.init().setDirectory(gitDirectory.toFile()).call();
        }
        catch (Exception e) {
            throw new IllegalStateException("Not able to run the \"git init\" on temporary folder.", e);
        }
    }

    private SourceCodeRecorder(SourceCodeProvider sourceCodeProvider, Path outputRecordingFilePath, TimeSource timeSource, long recordedTimestamp, long snapshotIntervalMillis, int keySnapshotSpacing, SourceCodeRecordingListener sourceCodeRecordingListener) {
        this.sourceCodeProvider = sourceCodeProvider;
        this.outputRecordingFilePath = outputRecordingFilePath;
        this.timeSource = timeSource;
        this.recordingStartTimestamp = recordedTimestamp;
        this.snapshotIntervalMillis = snapshotIntervalMillis;
        this.keySnapshotSpacing = keySnapshotSpacing;
        this.sourceCodeRecordingListener = sourceCodeRecordingListener;
        this.shouldStopJob = new AtomicBoolean(false);
        this.tagQueue = new ConcurrentLinkedQueue<String>();
    }

    public void start(Duration recordingDuration) throws SourceCodeRecorderException {
        Writer writer;
        Path lockFilePath = Paths.get(this.outputRecordingFilePath + ".lock", new String[0]);
        try {
            Files.write(lockFilePath, new byte[0], StandardOpenOption.CREATE);
            writer = new Writer(this.outputRecordingFilePath, this.sourceCodeProvider, this.timeSource, this.recordingStartTimestamp, this.keySnapshotSpacing, false);
        }
        catch (IOException | SnapshotRecorderException e) {
            throw new SourceCodeRecorderException("Failed to open destination", e);
        }
        try {
            this.sourceCodeRecordingListener.notifyRecordingStart(this.outputRecordingFilePath);
            this.doRecord(writer, recordingDuration);
        }
        catch (SnapshotRecorderException e) {
            throw new SourceCodeRecorderException("Exception encountered while recording", e);
        }
        finally {
            this.sourceCodeRecordingListener.notifyRecordingEnd();
        }
    }

    public void tagCurrentState(String tag) {
        log.info("Tag state with: " + tag);
        this.tagQueue.offer(tag);
        this.timeSource.wakeUpNow();
    }

    private void doRecord(Writer writer, Duration recordingDuration) throws SnapshotRecorderException {
        while (this.timeSource.currentTimeNano() < recordingDuration.toNanos()) {
            long timestampBeforeProcessing = this.timeSource.currentTimeNano();
            this.sourceCodeRecordingListener.notifySnapshotStart(timestampBeforeProcessing, TimeUnit.NANOSECONDS);
            Snapshot snapshot = writer.takeSnapshot();
            writer.writeSnapshotWithTags(snapshot, this.getAllEnqueuedTags());
            this.sourceCodeRecordingListener.notifySnapshotEnd(this.timeSource.currentTimeNano(), TimeUnit.NANOSECONDS);
            if (this.shouldStopJob.get() && !this.hasTags()) break;
            long timeToSleep = TimeUnit.MILLISECONDS.toNanos(this.snapshotIntervalMillis);
            long nextTimestamp = timestampBeforeProcessing + timeToSleep;
            try {
                this.timeSource.wakeUpAt(nextTimestamp, TimeUnit.NANOSECONDS);
            }
            catch (InterruptedException | BrokenBarrierException e) {
                log.debug("Interrupted while sleeping", (Throwable)e);
            }
        }
    }

    private List<String> getAllEnqueuedTags() {
        ArrayList<String> tags = new ArrayList<String>();
        while (this.tagQueue.peek() != null) {
            tags.add(this.tagQueue.poll());
        }
        return tags;
    }

    private boolean hasTags() {
        return this.tagQueue.size() > 0;
    }

    public void stop() {
        if (!this.shouldStopJob.get()) {
            log.info("Stopping recording");
            this.shouldStopJob.set(true);
            this.timeSource.wakeUpNow();
        } else {
            log.info("Recording already stopping");
        }
    }

    public void close() {
        try {
            log.info("Closing the source code stream");
            Path lockFilePath = Paths.get(this.outputRecordingFilePath + ".lock", new String[0]);
            Files.delete(lockFilePath);
        }
        catch (IOException e) {
            throw new RuntimeException("Can't delete *.lock file.", e);
        }
    }

    public static class Builder {
        private final SourceCodeProvider bSourceCodeProvider;
        private final Path bOutputRecordingFilePath;
        private TimeSource bTimeSource;
        private long bSnapshotIntervalMillis;
        private long bRecordingStartTimestampSec;
        private int bKeySnapshotSpacing;
        private SourceCodeRecordingListener bSourceCodeRecordingListener;

        public Builder(SourceCodeProvider sourceCodeProvider, Path outputRecordingFilePath) {
            this.bSourceCodeProvider = sourceCodeProvider;
            this.bOutputRecordingFilePath = outputRecordingFilePath;
            this.bRecordingStartTimestampSec = System.currentTimeMillis() / 1000L;
            this.bTimeSource = new SystemMonotonicTimeSource();
            this.bSnapshotIntervalMillis = TimeUnit.MINUTES.toMillis(5L);
            this.bKeySnapshotSpacing = 5;
            this.bSourceCodeRecordingListener = new SourceCodeRecordingMetricsCollector();
        }

        public Builder withRecordingStartTimestampSec(long recordingStartTimestampSec) {
            this.bRecordingStartTimestampSec = recordingStartTimestampSec;
            return this;
        }

        public Builder withTimeSource(TimeSource timeSource) {
            this.bTimeSource = timeSource;
            return this;
        }

        public Builder withSnapshotEvery(int interval, TimeUnit timeUnit) {
            this.bSnapshotIntervalMillis = timeUnit.toMillis(interval);
            return this;
        }

        public Builder withKeySnapshotSpacing(int keySnapshotSpacing) {
            this.bKeySnapshotSpacing = keySnapshotSpacing;
            return this;
        }

        public Builder withRecordingListener(SourceCodeRecordingListener sourceCodeRecordingListener) {
            this.bSourceCodeRecordingListener = sourceCodeRecordingListener;
            return this;
        }

        public SourceCodeRecorder build() {
            return new SourceCodeRecorder(this.bSourceCodeProvider, this.bOutputRecordingFilePath, this.bTimeSource, this.bRecordingStartTimestampSec, this.bSnapshotIntervalMillis, this.bKeySnapshotSpacing, this.bSourceCodeRecordingListener);
        }
    }
}

