/*
 * Decompiled with CFR 0.152.
 */
package rs.ltt.jmap.client.api;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.EOFException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.UUID;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import rs.ltt.jmap.client.JmapRequest;
import rs.ltt.jmap.client.Services;
import rs.ltt.jmap.client.api.AbstractJmapApiClient;
import rs.ltt.jmap.client.api.SessionStateListener;
import rs.ltt.jmap.client.api.WebSocketClosedException;
import rs.ltt.jmap.client.event.State;
import rs.ltt.jmap.client.http.HttpAuthentication;
import rs.ltt.jmap.client.session.Session;
import rs.ltt.jmap.common.GenericResponse;
import rs.ltt.jmap.common.websocket.AbstractApiWebSocketMessage;
import rs.ltt.jmap.common.websocket.RequestWebSocketMessage;
import rs.ltt.jmap.common.websocket.WebSocketMessage;

public class WebSocketJmapApiClient
extends AbstractJmapApiClient
implements Closeable {
    protected static final Logger LOGGER = LoggerFactory.getLogger(WebSocketJmapApiClient.class);
    private static final String JMAP = "jmap";
    protected final List<Long> connectionDurations = new ArrayList<Long>();
    private final HttpUrl webSocketUrl;
    private final HttpAuthentication authentication;
    private final ArrayList<JmapRequest> requestQueue = new ArrayList();
    private final HashMap<String, JmapRequest> inFlightRequests = new HashMap();
    protected int attempt = 0;
    protected State state = State.CLOSED;
    protected ScheduledFuture<?> reconnectionFuture;
    private WebSocket currentWebSocket;
    private long lastFrameReceived = 0L;

    public WebSocketJmapApiClient(HttpUrl webSocketUrl, HttpAuthentication httpAuthentication, @Nullable SessionStateListener sessionStateListener) {
        super(sessionStateListener);
        this.webSocketUrl = (HttpUrl)Preconditions.checkNotNull((Object)webSocketUrl, (Object)"This WebSocket URL must not be null");
        this.authentication = httpAuthentication;
    }

    @Override
    public synchronized void execute(JmapRequest jmapRequest) {
        if (this.readyToSend()) {
            this.send(jmapRequest);
        } else {
            LOGGER.info("Queued up JmapRequest because not ready to send in state {}", (Object)this.state);
            this.requestQueue.add(jmapRequest);
        }
    }

    @Override
    public boolean isValidFor(Session session) {
        return true;
    }

    private void send(JmapRequest jmapRequest) {
        String requestId = UUID.randomUUID().toString();
        this.inFlightRequests.put(requestId, jmapRequest);
        RequestWebSocketMessage message = RequestWebSocketMessage.builder().id(requestId).request(jmapRequest.getRequest()).build();
        if (this.send((WebSocketMessage)message)) {
            return;
        }
        jmapRequest.setException(new Exception("Unable to send. WebSocket was closed"));
        this.inFlightRequests.remove(requestId);
    }

    protected boolean send(WebSocketMessage message) {
        if (Services.OK_HTTP_LOGGER.isDebugEnabled()) {
            Services.OK_HTTP_LOGGER.debug("--> {}", (Object)Services.GSON.toJson((Object)message));
        }
        return this.requireWebSocket().send(Services.GSON.toJson((Object)message));
    }

    private WebSocket requireWebSocket() {
        WebSocket current = this.currentWebSocket;
        if (current == null) {
            throw new IllegalStateException(String.format("WebSocket was unexpectedly null even though we are in state %s", new Object[]{this.state}));
        }
        return current;
    }

    protected boolean readyToSend() {
        if (this.state == State.CONNECTED) {
            return true;
        }
        if (this.state.needsReconnect()) {
            this.connectWebSocket();
            return false;
        }
        if (this.state == State.CONNECTING) {
            return false;
        }
        throw new IllegalArgumentException(String.format("WebSocketClient is %s", new Object[]{this.state}));
    }

    protected void connectWebSocket() {
        ++this.attempt;
        this.cancelReconnectionFuture();
        this.transitionTo(State.CONNECTING);
        this.startWebSocket();
    }

    protected void transitionTo(State state) {
        LOGGER.info("transition to {}", (Object)state);
        this.state = state;
    }

    private void cancelReconnectionFuture() {
        ScheduledFuture<?> future = this.reconnectionFuture;
        if (future != null && !future.isDone()) {
            future.cancel(false);
        }
    }

    private void startWebSocket() {
        LOGGER.info("Using WebSocket URL {}", (Object)this.webSocketUrl);
        Request.Builder requestBuilder = new Request.Builder();
        requestBuilder.url(this.webSocketUrl);
        this.authentication.authenticate(requestBuilder);
        requestBuilder.header("Sec-WebSocket-Protocol", JMAP);
        Request request = requestBuilder.build();
        OkHttpClient okHttpClient = Services.OK_HTTP_CLIENT.newBuilder().callTimeout(30L, TimeUnit.SECONDS).pingInterval(this.getPingInterval()).build();
        this.setCurrentWebSocket(okHttpClient.newWebSocket(request, (WebSocketListener)new WebSocketProcessor(this)));
    }

    protected Duration getPingInterval() {
        return Duration.ZERO;
    }

    private void setCurrentWebSocket(WebSocket webSocket) {
        if (this.currentWebSocket != null) {
            throw new IllegalStateException("Unable to set current WebSocket. One already exists");
        }
        this.currentWebSocket = webSocket;
    }

    private synchronized void onMessage(WebSocket webSocket, String text) {
        WebSocketMessage message;
        this.lastFrameReceived = System.nanoTime();
        if (Services.OK_HTTP_LOGGER.isDebugEnabled()) {
            Services.OK_HTTP_LOGGER.debug("<-- {}", (Object)text);
        }
        try {
            message = (WebSocketMessage)Services.GSON.fromJson(text, WebSocketMessage.class);
        }
        catch (Exception e) {
            LOGGER.error("Unable to parse incoming WebSocketMessage", (Throwable)e);
            this.policyViolation(e);
            return;
        }
        this.onWebSocketMessage(message);
    }

    protected boolean onWebSocketMessage(WebSocketMessage message) {
        if (message instanceof AbstractApiWebSocketMessage) {
            return this.onApiMessage((AbstractApiWebSocketMessage)message);
        }
        return false;
    }

    protected boolean onApiMessage(AbstractApiWebSocketMessage apiMessage) {
        String requestId = apiMessage.getRequestId();
        if (requestId == null) {
            this.policyViolation(new IllegalStateException(String.format("Server sent %s w/o requestId", apiMessage.getClass().getSimpleName())));
            return false;
        }
        JmapRequest jmapRequest = this.inFlightRequests.remove(requestId);
        if (jmapRequest == null) {
            this.policyViolation(new IllegalStateException(String.format("Could not find in flight request with id %s", requestId)));
            return false;
        }
        Object payload = apiMessage.getPayload();
        if (payload instanceof GenericResponse) {
            this.processResponse(jmapRequest, (GenericResponse)payload);
            return false;
        }
        return false;
    }

    private void disconnect(State state) {
        WebSocket currentWebSocket = this.currentWebSocket;
        if (currentWebSocket != null) {
            currentWebSocket.cancel();
            this.currentWebSocket = null;
            this.transitionTo(state);
        }
    }

    private void policyViolation(Throwable throwable) {
        this.disconnect(State.FAILED);
        this.failPendingRequests(throwable);
    }

    private void failPendingRequests(Throwable throwable) {
        WebSocketJmapApiClient.failPendingRequests(this.requestQueue.listIterator(), throwable);
        WebSocketJmapApiClient.failPendingRequests(this.inFlightRequests.values().iterator(), throwable);
    }

    private static void failPendingRequests(Iterator<JmapRequest> iterator, Throwable throwable) {
        while (iterator.hasNext()) {
            JmapRequest jmapRequest = iterator.next();
            jmapRequest.setException(throwable);
            iterator.remove();
        }
    }

    protected synchronized void onOpen() {
        this.attempt = 0;
        this.transitionTo(State.CONNECTED);
        this.lastFrameReceived = System.nanoTime();
        ListIterator<JmapRequest> iterator = this.requestQueue.listIterator();
        while (iterator.hasNext()) {
            JmapRequest jmapRequest = iterator.next();
            this.send(jmapRequest);
            iterator.remove();
        }
    }

    private synchronized void onFailure(Throwable throwable, Response response) {
        boolean showFailure = this.state != State.FAILED;
        boolean wasConnected = this.state == State.CONNECTED;
        this.disconnect(State.FAILED);
        if (showFailure) {
            LOGGER.info("Unable to connect to WebSocket URL", throwable);
        }
        if (throwable instanceof EOFException && wasConnected) {
            this.connectionDurations.add(System.nanoTime() - this.lastFrameReceived);
        }
        this.failPendingRequests(throwable);
    }

    private void onClosing(WebSocket webSocket, int code, String reason) {
        LOGGER.info("Server closed the connection with code {} and reason {}", (Object)code, (Object)reason);
        this.disconnect(State.CLOSED);
        this.failPendingRequests(new WebSocketClosedException(code, reason));
    }

    @Override
    public synchronized void close() {
        this.disconnect(State.CLOSED);
        this.cancelReconnectionFuture();
    }

    private static class WebSocketProcessor
    extends WebSocketListener {
        private final WebSocketJmapApiClient client;

        private WebSocketProcessor(WebSocketJmapApiClient client) {
            this.client = client;
        }

        public void onClosing(@NotNull WebSocket webSocket, int code, @NotNull String reason) {
            super.onClosing(webSocket, code, reason);
            this.client.onClosing(webSocket, code, reason);
        }

        public void onFailure(@NotNull WebSocket webSocket, @NotNull Throwable t, Response response) {
            super.onFailure(webSocket, t, response);
            this.client.onFailure(t, response);
        }

        public void onMessage(@NotNull WebSocket webSocket, @NotNull String text) {
            super.onMessage(webSocket, text);
            this.client.onMessage(webSocket, text);
        }

        public void onOpen(@NotNull WebSocket webSocket, @NotNull Response response) {
            super.onOpen(webSocket, response);
            this.client.onOpen();
        }
    }
}

