/*
 * Decompiled with CFR 0.152.
 */
package rs.ltt.jmap.client.blob;

import com.google.common.base.Strings;
import com.google.common.primitives.Longs;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.HttpUrl;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import rs.ltt.jmap.client.Services;
import rs.ltt.jmap.client.blob.BlobTransferException;
import rs.ltt.jmap.client.blob.Download;
import rs.ltt.jmap.client.blob.Progress;
import rs.ltt.jmap.client.blob.RequestBodies;
import rs.ltt.jmap.client.blob.ResumptionFailedException;
import rs.ltt.jmap.client.blob.Uploadable;
import rs.ltt.jmap.client.http.HttpAuthentication;
import rs.ltt.jmap.client.util.SettableCallFuture;
import rs.ltt.jmap.common.ErrorResponse;
import rs.ltt.jmap.common.entity.Upload;

public class BinaryDataClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(BinaryDataClient.class);
    private static final String HTTP_HEADER_RANGE = "Range";
    private static final String HTTP_HEADER_CONTENT_RANGE = "Content-Range";
    private static final String HTTP_HEADER_CONTENT_LENGTH = "Content-Length";
    private static final Pattern CONTENT_RANGE_PATTERN = Pattern.compile("(^[a-zA-Z][\\w]*)\\s+(\\d+)\\s?-\\s?(\\d+)?\\s?/?\\s?(\\d+|\\*)?");
    private final HttpAuthentication httpAuthentication;

    public BinaryDataClient(HttpAuthentication httpAuthentication) {
        this.httpAuthentication = httpAuthentication;
    }

    public ListenableFuture<Download> download(HttpUrl httpUrl, final long rangeStart) {
        Request.Builder requestBuilder = new Request.Builder();
        requestBuilder.url(httpUrl);
        if (rangeStart > 0L) {
            requestBuilder.header(HTTP_HEADER_RANGE, String.format("bytes=%d-", rangeStart));
        }
        this.httpAuthentication.authenticate(requestBuilder);
        Call call = Services.OK_HTTP_CLIENT.newCall(requestBuilder.build());
        final SettableCallFuture settableFuture = SettableCallFuture.create(call);
        LOGGER.info("Downloading blob from {}", (Object)httpUrl);
        call.enqueue(new Callback(){

            public void onFailure(@NotNull Call call, @NotNull IOException e) {
                settableFuture.setException(e);
            }

            public void onResponse(@NotNull Call call, @NotNull Response response) {
                settableFuture.setFuture(BinaryDataClient.this.onDownloadResponse(call, response, rangeStart));
            }
        });
        return settableFuture;
    }

    private ListenableFuture<Download> onDownloadResponse(@NotNull Call call, @NotNull Response response, long rangeStart) {
        ErrorResponse errorResponse;
        ResponseBody body = response.body();
        if (body == null) {
            return Futures.immediateFailedFuture((Throwable)new IllegalStateException("response body was empty"));
        }
        if (response.isSuccessful()) {
            Download download;
            boolean resumed;
            ContentRange contentRange;
            String contentLengthHeader = response.header(HTTP_HEADER_CONTENT_LENGTH);
            try {
                contentRange = ContentRange.of(response.header(HTTP_HEADER_CONTENT_RANGE));
            }
            catch (IllegalArgumentException e) {
                return Futures.immediateFailedFuture((Throwable)new ResumptionFailedException(e));
            }
            Long contentLength = contentLengthHeader == null ? null : Longs.tryParse((String)contentLengthHeader);
            boolean bl = resumed = rangeStart > 0L && contentRange != null;
            if (resumed) {
                if (rangeStart != contentRange.getStart()) {
                    return Futures.immediateFailedFuture((Throwable)new ResumptionFailedException(String.format("Requested start %d did not match actual start %d", rangeStart, contentRange.getStart())));
                }
                download = new Download(call, true, contentRange.getEnd(), body.byteStream());
            } else {
                long length = contentLength == null ? 0L : contentLength;
                download = new Download(call, false, length, body.byteStream());
            }
            return Futures.immediateFuture((Object)download);
        }
        try (InputStreamReader reader = new InputStreamReader(body.byteStream());){
            errorResponse = (ErrorResponse)Services.GSON.fromJson((Reader)reader, ErrorResponse.class);
        }
        catch (Exception e) {
            return Futures.immediateFailedFuture((Throwable)e);
        }
        return Futures.immediateFailedFuture((Throwable)new BlobTransferException(response.code(), errorResponse));
    }

    public ListenableFuture<Upload> upload(HttpUrl httpUrl, Uploadable uploadable, Progress progress) {
        RequestBody requestBody = RequestBodies.of(uploadable, progress);
        Request.Builder requestBuilder = new Request.Builder();
        requestBuilder.url(httpUrl);
        this.httpAuthentication.authenticate(requestBuilder);
        requestBuilder.post(requestBody);
        Call call = Services.OK_HTTP_CLIENT.newCall(requestBuilder.build());
        final SettableCallFuture settableFuture = SettableCallFuture.create(call);
        call.enqueue(new Callback(){

            public void onFailure(@NotNull Call call, @NotNull IOException e) {
                settableFuture.setException(e);
            }

            public void onResponse(@NotNull Call call, @NotNull Response response) {
                settableFuture.setFuture(BinaryDataClient.this.onUploadResponse(response));
            }
        });
        return settableFuture;
    }

    private ListenableFuture<Upload> onUploadResponse(@NotNull Response response) {
        ErrorResponse errorResponse;
        ResponseBody body = response.body();
        if (body == null) {
            return Futures.immediateFailedFuture((Throwable)new IllegalStateException("response body was empty"));
        }
        if (response.isSuccessful()) {
            Upload upload;
            try (InputStreamReader reader = new InputStreamReader(body.byteStream());){
                upload = (Upload)Services.GSON.fromJson((Reader)reader, Upload.class);
                this.validate(upload);
            }
            catch (Exception e) {
                return Futures.immediateFailedFuture((Throwable)e);
            }
            return Futures.immediateFuture((Object)upload);
        }
        try (InputStreamReader reader = new InputStreamReader(body.byteStream());){
            errorResponse = (ErrorResponse)Services.GSON.fromJson((Reader)reader, ErrorResponse.class);
        }
        catch (Exception e) {
            return Futures.immediateFailedFuture((Throwable)e);
        }
        return Futures.immediateFailedFuture((Throwable)new BlobTransferException(response.code(), errorResponse));
    }

    private void validate(Upload upload) {
        if (Strings.isNullOrEmpty((String)upload.getBlobId())) {
            throw new IllegalStateException("Upload object is missing blobId");
        }
        if (upload.getSize() == null) {
            throw new IllegalStateException("Upload object is missing size");
        }
        if (Strings.isNullOrEmpty((String)upload.getType())) {
            throw new IllegalStateException("Upload object is missing type");
        }
    }

    public static class ContentRange {
        private final String unit;
        private final long start;
        private final long end;
        private final long contentLength;

        private ContentRange(String unit, Long start, Long end, Long contentLength) {
            this.unit = unit;
            this.start = start == null ? 0L : start;
            this.end = end == null ? 0L : end;
            this.contentLength = contentLength == null ? 0L : contentLength;
        }

        public static ContentRange of(String header) {
            if (header == null) {
                return null;
            }
            Matcher matcher = CONTENT_RANGE_PATTERN.matcher(header);
            if (matcher.matches()) {
                return new ContentRange(matcher.group(1), Longs.tryParse((String)matcher.group(2)), Longs.tryParse((String)matcher.group(3)), Longs.tryParse((String)matcher.group(4)));
            }
            throw new IllegalArgumentException(String.format("Invalid content range %s", header));
        }

        public String getUnit() {
            return this.unit;
        }

        public long getStart() {
            return this.start;
        }

        public long getEnd() {
            return this.end;
        }

        public long getContentLength() {
            return this.contentLength;
        }
    }
}

