package ru.tbank.posterminal.p2psdk

import android.app.Application
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.os.Handler
import android.os.IBinder
import android.os.Looper
import android.os.Message
import android.os.Messenger
import android.os.RemoteException

private const val TLOG_TAG = "TSoftposManager"
private const val P2P_PACKAGE_NAME = "ru.tinkoff.posterminal"

/**
 * The class is for handling Pay to Phone transactions using a bound service.
 *
 * The `TSoftposManager` is responsible for managing the binding to the SoftPOS service,
 * initiating transactions, validating input data, and handling communication with the service.
 * It also provides logging capabilities for debugging and diagnostic purposes.
 *
 * ### Important Lifecycle Notes:
 *
 * - **Lifecycle Management**: The lifecycle management of the `TSoftposManager` instance is the responsibility of the developer.
 * - It is recommended to sync the lifecycle of the `TSoftposManager` instance with the lifecycle of its associated `Context` (e.g., an `Activity`).
 * - Ensure to call [unbindSoftpos] when the `Context` is destroyed (e.g., in `onDestroy`) to avoid memory leaks and ensure proper cleanup of resources.
 *
 * ### Example Usage
 * ```kotlin
 * class MyActivity : AppCompatActivity() {
 *     private lateinit var tSoftposManager: TSoftposManager
 *
 *     override fun onCreate(savedInstanceState: Bundle?) {
 *         super.onCreate(savedInstanceState)
 *         tSoftposManager = TSoftposManager(this).apply {
 *             initLogger(MyLogger()) // Initialize the logger if needed
 *         }
 *     }
 *
 *     fun startTransaction() {
 *         val transactionData = TransactionData(...) // populate with required data
 *         tSoftposManager.payToPhone(transactionData, object : Callback {
 *             override fun onTransactionRegistered(result: Result): Boolean {
 *                 // Handle success
 *                 return true
 *             }
 *
 *             override fun onError(exception: Exception) {
 *                 // Handle error
 *             }
 *         })
 *     }
 *
 *     override fun onDestroy() {
 *         super.onDestroy()
 *         tSoftposManager.unbindSoftpos() // Ensure the manager unbinds when the Activity is destroyed
 *     }
 * }
 * ```
 *
 * ### Features:
 * - Binds to the Softpos service for communication.
 * - Handles input validation (amount ranges, transaction IDs, etc.).
 * - Executes payment transactions and provides callbacks for results and errors.
 * - Logs debugging and informational messages using an optional logger.
 *
 * @param context The [Context] used to bind to the SoftPOS service. Typically an [android.app.Activity] or [Application].
 *
 * @constructor Creates a new instance of `TSoftposManager`.
 */

public class TSoftposManager(
    private val context: Context,
) {

    @Volatile
    private var callback: Callback? = null

    @Volatile
    private var softposMessenger: Messenger? = null

    @Volatile
    private var tLogger: TLogger? = null

    @Volatile
    private var clientPackage: String? = null

    /**
     * Indicates whether a transaction is currently in progress.
     * @return `true` if there is an ongoing transaction, otherwise `false`.
     */

    public val isTransactionInProgress: Boolean get() = callback != null

    /**
     * Unbinds the Softpos service and clears any associated resources.
     * This should be called to clean up after interactions with the Softpos service are complete.
     *
     * If the service is not bound, it safely catches and logs the exception.
     */

    public fun unbindSoftpos() {
        tLogger?.logDebug(TLOG_TAG, "Unbind service")
        handler.removeCallbacksAndMessages(null)
        softposMessenger = null
        callback = null
        try {
            context.unbindService(serviceConnection)
        } catch (e: IllegalArgumentException) {
            tLogger?.logDebug(TLOG_TAG, "Service was not bound: $e")
        }
    }

    /**
     * Initiates a Pay to Phone transaction.
     *
     * Performs validation on the transaction data (amount, merchant ID, etc.) and binds to the SoftPOS
     * service to begin the transaction. Executes a callback on success or failure.
     *
     * @param transactionData The data required to perform the transaction.
     * @param callback A callback to handle the transaction result or errors.
     *
     */

    public fun payToPhone(
        transactionData: TransactionData,
        callback: Callback,
    ) {
        tLogger?.logDebug(
            TLOG_TAG,
            "payToPhone(): $transactionData, isTransactionInProgress=$isTransactionInProgress"
        )
        if (isTransactionInProgress) {
            callback.onError(IllegalStateException("Transaction is already in progress!"))
            return
        }
        validateAmount(transactionData, context)?.let { error ->
            callback.onError(error)
            return
        }
        validateMidAndTransactionId(transactionData)?.let { error ->
            callback.onError(error)
            return
        }
        this.callback = callback

        try {
            bindSoftpos(context) {
                startActivityIfPossible(
                    context = context,
                    transactionData = transactionData
                )
            }
        } catch (e: Exception) {
            callback.onError(e)
            this.callback = null
        }
    }

    /**
     * Initializes the logger for TSoftposManager
     *
     * The logger is used for debugging and informational logs during service operations.
     *
     * @param tLogger The logger to be used for logging.
     */

    public fun initLogger(tLogger: TLogger) {
        this.tLogger = tLogger
    }

    private val serviceConnection = object : ServiceConnection {

        var onReadyToTransact: Runnable? = null

        override fun onServiceConnected(name: ComponentName?, service: IBinder?) {
            tLogger?.logInfo(TLOG_TAG, "Service connected")
            softposMessenger = Messenger(service)
            onReadyToTransact?.run()
            onReadyToTransact = null
        }

        override fun onServiceDisconnected(name: ComponentName?) {
            tLogger?.logInfo(TLOG_TAG, "onServiceDisconnected: $name")
            unbindSoftpos()
        }

        override fun onBindingDied(name: ComponentName?) {
            tLogger?.logInfo(TLOG_TAG, "onBindingDied: $name")
            unbindSoftpos()
        }

        override fun onNullBinding(name: ComponentName?) {
            tLogger?.logInfo(TLOG_TAG, "onNullBinding: $name")
            unbindSoftpos()
        }
    }

    private val handler = object : Handler(Looper.getMainLooper()) {
        override fun handleMessage(msg: Message) {
            tLogger?.logDebug(
                TLOG_TAG,
                "Received message what: ${msg.what}, arg1: ${msg.arg1}, arg2: ${msg.arg2}, data: ${msg.data.toStringFormat()}"
            )
            when (msg.what) {
                IPC_RESULT_SUCCESS -> try {
                    val isRegistered =
                        callback?.onTransactionRegistered(msg.toSoftposResult()) ?: false
                    tLogger?.logDebug(TLOG_TAG, "isRegistered: $isRegistered")
                    if (msg.isPayment()) {
                        sendHealthCheckResult(isRegistered, clientPackage)
                    }
                } catch (e: Exception) {
                    callback?.onError(e)
                } finally {
                    callback = null
                }

                IPC_RESULT_FAILURE -> {
                    callback?.onError(msg.toTransactionException())
                    callback = null
                }
            }
        }
    }

    private val clientMessenger = Messenger(handler)

    private fun bindSoftpos(activityContext: Context, onReadyToTransact: Runnable) {
        if (softposMessenger.isBound()) {
            onReadyToTransact.run()
        } else {
            val intent = Intent(IPC_PAYMENT_SERVICE_ACTION).apply {
                setPackage(P2P_PACKAGE_NAME)
            }
            serviceConnection.onReadyToTransact = onReadyToTransact
            val isBound = activityContext.bindService(
                intent,
                serviceConnection,
                Context.BIND_AUTO_CREATE or Context.BIND_IMPORTANT
            )
            if (!isBound) {
                callback?.onError(IllegalStateException("Failed to bind to Pay to phone ($P2P_PACKAGE_NAME)!"))
                unbindSoftpos()
            }
        }
    }

    private fun startActivityIfPossible(
        context: Context,
        transactionData: TransactionData,
    ) {
        payToPhoneIPC(transactionData, context)

        val intent = Intent(PAYMENT_ACTION).apply {
            addFlags(Intent.FLAG_ACTIVITY_EXCLUDE_FROM_RECENTS)
            if (context is Application) {
                addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            }
        }
        context.startActivity(intent)
    }

    private fun payToPhoneIPC(
        transactionData: TransactionData,
        context: Context,
    ) {
        val messenger = softposMessenger
        check(messenger.isBound()) { "Messenger is not bound. Did you forget to call bindSoftpos()?" }

        val paymentMethod = transactionData.paymentMethod
        val transactionId =
            if (transactionData is RefundTransactionData) transactionData.transactionId else null
        this.clientPackage = context.packageName
        val message = Message.obtain(null, IPC_PAY).apply {
            clientPackage = context.packageName
            replyTo = clientMessenger
            arg1 = if (transactionId != null) IPC_A1_REFUND else IPC_A1_PAYMENT
            arg2 = paymentMethod.ipcArgValue
            data.appendData(transactionData)
        }
        try {
            messenger.send(message)
        } catch (e: RemoteException) {
            tLogger?.logError(TLOG_TAG, "Send message", e)
            throw e
        } finally {
            message.recycle()
        }
    }

    private fun validateAmount(
        transactionData: TransactionData,
        context: Context,
    ): SoftposException? {
        val amount = transactionData.amount
        val paymentMethod = transactionData.paymentMethod
        if (amount < paymentMethod.minAmount || amount > paymentMethod.maxAmount) {
            val applicationName = context.packageManager
                .getApplicationLabel(context.applicationInfo)
            return SoftposException(
                code = VALIDATION_ERROR_CODE,
                details = context.getString(
                    R.string.pay_to_phone_sdk_invalid_amount,
                    getAmountString(paymentMethod.minAmount, context),
                    getAmountString(paymentMethod.maxAmount, context),
                    applicationName
                )
            )
        }
        return null
    }

    private fun validateMidAndTransactionId(
        transactionData: TransactionData,
    ): SoftposException? {
        if (transactionData is RefundTransactionData) {
            val tid = transactionData.transactionId
            val mid = transactionData.mid
            if (tid <= 0L) {
                return SoftposException(
                    code = VALIDATION_ERROR_CODE,
                    details = "Incorrect transactionId=$tid"
                )
            }
            if (mid <= 0L) {
                return SoftposException(
                    code = VALIDATION_ERROR_CODE,
                    details = "Incorrect mid=$mid"
                )
            }
        }
        return null
    }

    private fun getAmountString(amount: Long, context: Context): String {
        val rub: Long = amount / 100
        val cop: Long = amount % 100
        return if (cop == 0L) {
            context.getString(R.string.pay_to_phone_sdk_amount_part_rub, rub)
        } else {
            if (rub == 0L) {
                context.getString(R.string.pay_to_phone_sdk_amount_part_cop, cop)
            } else {
                context.getString(R.string.pay_to_phone_sdk_amount_part_rub_cop, rub, cop)
            }
        }
    }

    private fun sendHealthCheckResult(isRegistered: Boolean, packageName: String?) {
        val messenger = softposMessenger
        check(messenger.isBound()) { "Messenger is not bound. Did you forget to call bindSoftpos()?" }
        check(!packageName.isNullOrBlank()) { "Package name is null or empty" }

        val message = Message.obtain(null, IPC_HEALTH_CHECK).apply {
            clientPackage = packageName
            replyTo = clientMessenger
            arg1 = if (isRegistered) IPC_A1_HEALTH_CHECK_SUCCESS else IPC_A1_HEALTH_CHECK_FAILURE
        }
        try {
            messenger.send(message)
        } catch (e: RemoteException) {
            tLogger?.logError(TLOG_TAG, "Send health check message failed", e)
            throw e
        } finally {
            message.recycle()
        }
    }
}
