/*
 * Decompiled with CFR 0.152.
 */
package run.halo.gradle;

import java.io.Closeable;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.EnumSet;
import java.util.function.Consumer;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import run.halo.gradle.Assert;

public abstract class FileUtils {
    private static final Logger log = LoggerFactory.getLogger(FileUtils.class);

    public static void unzip(ZipInputStream zis, Path targetPath) throws IOException {
        Assert.notNull(zis, "Zip input stream must not be null");
        Assert.notNull(targetPath, "Target path must not be null");
        FileUtils.createIfAbsent(targetPath);
        FileUtils.ensureEmpty(targetPath);
        ZipEntry zipEntry = zis.getNextEntry();
        while (zipEntry != null) {
            Path entryPath = targetPath.resolve(zipEntry.getName());
            FileUtils.checkDirectoryTraversal(targetPath, entryPath);
            if (Files.notExists(entryPath.getParent(), new LinkOption[0])) {
                Files.createDirectories(entryPath.getParent(), new FileAttribute[0]);
            }
            if (zipEntry.isDirectory()) {
                Files.createDirectory(entryPath, new FileAttribute[0]);
            } else {
                Files.copy(zis, entryPath, new CopyOption[0]);
            }
            zipEntry = zis.getNextEntry();
        }
    }

    public static void copyRecursively(final Path src, final Path dest) throws IOException {
        Assert.notNull(src, "Source Path must not be null");
        Assert.notNull(dest, "Destination Path must not be null");
        BasicFileAttributes srcAttr = Files.readAttributes(src, BasicFileAttributes.class, new LinkOption[0]);
        if (srcAttr.isDirectory()) {
            Files.walkFileTree(src, EnumSet.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    Files.createDirectories(dest.resolve(src.relativize(dir)), new FileAttribute[0]);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.copy(file, dest.resolve(src.relativize(file)), StandardCopyOption.REPLACE_EXISTING);
                    return FileVisitResult.CONTINUE;
                }
            });
        } else if (srcAttr.isRegularFile()) {
            Files.copy(src, dest, new CopyOption[0]);
        } else {
            throw new IllegalArgumentException("Source File must denote a directory or file");
        }
    }

    public static boolean deleteRecursively(Path root) throws IOException {
        if (root == null) {
            return false;
        }
        if (!Files.exists(root, new LinkOption[0])) {
            return false;
        }
        Files.walkFileTree(root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        });
        return true;
    }

    public static void createIfAbsent(Path path) throws IOException {
        Assert.notNull(path, "Path must not be null");
        if (Files.notExists(path, new LinkOption[0])) {
            Files.createDirectories(path, new FileAttribute[0]);
            log.debug("Created directory: [{}]", (Object)path);
        }
    }

    public static void ensureEmpty(Path path) throws IOException {
        if (!FileUtils.isEmpty(path)) {
            throw new DirectoryNotEmptyException("Target directory: " + path + " was not empty");
        }
    }

    public static boolean isEmpty(Path path) throws IOException {
        Assert.notNull(path, "Path must not be null");
        if (!Files.isDirectory(path, new LinkOption[0]) || Files.notExists(path, new LinkOption[0])) {
            return true;
        }
        try (Stream<Path> pathStream = Files.list(path);){
            boolean bl = pathStream.findAny().isEmpty();
            return bl;
        }
    }

    public static void closeQuietly(Closeable closeable) {
        FileUtils.closeQuietly(closeable, null);
    }

    public static void closeQuietly(Closeable closeable, Consumer<IOException> consumer) {
        block3: {
            if (closeable != null) {
                try {
                    closeable.close();
                }
                catch (IOException e) {
                    if (consumer == null) break block3;
                    consumer.accept(e);
                }
            }
        }
    }

    public static void checkDirectoryTraversal(Path parentPath, Path pathToCheck) {
        Assert.notNull(parentPath, "Parent path must not be null");
        Assert.notNull(pathToCheck, "Path to check must not be null");
        if (pathToCheck.normalize().startsWith(parentPath)) {
            return;
        }
        throw new IllegalStateException(pathToCheck.toString());
    }

    public static void checkDirectoryTraversal(String parentPath, String pathToCheck) {
        FileUtils.checkDirectoryTraversal(Paths.get(parentPath, new String[0]), Paths.get(pathToCheck, new String[0]));
    }

    public static void checkDirectoryTraversal(Path parentPath, String pathToCheck) {
        FileUtils.checkDirectoryTraversal(parentPath, Paths.get(pathToCheck, new String[0]));
    }
}

