/*
 * Decompiled with CFR 0.152.
 */
package se.digg.dgc.signatures.impl;

import com.upokecenter.cbor.CBORException;
import com.upokecenter.cbor.CBORObject;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.time.Instant;
import java.util.Optional;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1String;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x509.Certificate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import se.digg.dgc.signatures.DGCSigner;
import se.digg.dgc.signatures.cose.CoseSign1_Object;
import se.digg.dgc.signatures.cose.HeaderParameterKey;
import se.digg.dgc.signatures.cose.SignatureAlgorithm;
import se.digg.dgc.signatures.cwt.Cwt;
import se.swedenconnect.security.credential.PkiCredential;

public class DefaultDGCSigner
implements DGCSigner {
    private static final Logger log = LoggerFactory.getLogger(DefaultDGCSigner.class);
    private final CredentialWrapper signerCredential;
    private Provider securityProvider;
    private final String country;
    private final byte[] keyIdentifier;
    private final Instant signerExpiration;
    private SignatureAlgorithm algorithmIdentifier;
    private boolean includeCoseTag = true;
    private boolean includeCwtTag = false;

    public DefaultDGCSigner(PrivateKey signerKey, X509Certificate signerCertificate) throws CertificateException {
        this(new CredentialWrapper(signerKey, signerCertificate));
    }

    public DefaultDGCSigner(PkiCredential signerCredential) throws CertificateException {
        this(new CredentialWrapper(signerCredential));
    }

    private DefaultDGCSigner(CredentialWrapper signerCredential) throws CertificateException {
        this.signerCredential = signerCredential;
        this.country = DefaultDGCSigner.getCountry(signerCredential.getCertificate());
        this.keyIdentifier = DefaultDGCSigner.calculateKid(signerCredential.getCertificate());
        this.signerExpiration = Instant.ofEpochMilli(signerCredential.getCertificate().getNotAfter().getTime());
        if (ECPublicKey.class.isInstance(this.signerCredential.getPublicKey())) {
            this.algorithmIdentifier = SignatureAlgorithm.ES256;
        } else if (RSAPublicKey.class.isInstance(this.signerCredential.getPublicKey())) {
            this.algorithmIdentifier = SignatureAlgorithm.PS256;
        } else {
            throw new SecurityException("Unsupported key");
        }
    }

    @Override
    public byte[] sign(byte[] dccPayload, Instant expiration) throws SignatureException {
        if (expiration.isAfter(this.signerExpiration)) {
            log.warn("Expiration of DCC goes beyond the signer certificate validity");
        }
        try {
            Cwt cwt = Cwt.builder().issuer(this.country).issuedAt(Instant.now()).expiration(expiration).dgcV1(dccPayload).build();
            CoseSign1_Object coseObject = CoseSign1_Object.builder().includeMessageTag(this.includeCoseTag).protectedAttribute(HeaderParameterKey.ALG.getCborObject(), this.algorithmIdentifier.getCborObject()).protectedAttribute(HeaderParameterKey.KID.getCborObject(), CBORObject.FromObject((byte[])this.keyIdentifier)).content(cwt.encode()).build();
            coseObject.sign(this.signerCredential.getPrivateKey(), this.securityProvider);
            byte[] coseEncoding = coseObject.encode();
            if (this.includeCwtTag) {
                CBORObject obj = CBORObject.DecodeFromBytes((byte[])coseEncoding);
                return CBORObject.FromObjectAndTag((Object)obj, (int)61).EncodeToBytes();
            }
            return coseEncoding;
        }
        catch (CBORException e) {
            throw new SignatureException("CBOR error - " + e.getMessage(), e);
        }
    }

    @Override
    public Instant getSignerExpiration() {
        return this.signerExpiration;
    }

    @Override
    public String getSignerCountry() {
        return this.country;
    }

    public void setAlgorithmIdentifier(SignatureAlgorithm algorithmIdentifier) {
        if (algorithmIdentifier != null) {
            this.algorithmIdentifier = algorithmIdentifier;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static String getCountry(X509Certificate cert) throws CertificateException {
        try (ByteArrayInputStream bis = new ByteArrayInputStream(cert.getEncoded());
             ASN1InputStream as = new ASN1InputStream((InputStream)bis);){
            ASN1Sequence seq = (ASN1Sequence)as.readObject();
            Certificate asn1Cert = Certificate.getInstance((Object)seq);
            if (asn1Cert.getSubject() == null) throw new CertificateException("Missing country in certificate subject");
            if (asn1Cert.getSubject().getRDNs() == null) {
                throw new CertificateException("Missing country in certificate subject");
            }
            RDN[] rdns = asn1Cert.getSubject().getRDNs(new ASN1ObjectIdentifier("2.5.4.6"));
            if (rdns == null) throw new CertificateException("Missing country in certificate subject");
            if (rdns.length == 0) {
                throw new CertificateException("Missing country in certificate subject");
            }
            ASN1Primitive p = rdns[0].getFirst().getValue().toASN1Primitive();
            if (!(p instanceof ASN1String)) throw new CertificateException("Missing country in certificate subject");
            String string = ((ASN1String)p).getString();
            return string;
        }
        catch (IOException e) {
            throw new CertificateException("Failed to read certificate", e);
        }
    }

    private static byte[] calculateKid(X509Certificate cert) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] sha256 = digest.digest(cert.getEncoded());
            byte[] kid = new byte[8];
            System.arraycopy(sha256, 0, kid, 0, 8);
            return kid;
        }
        catch (NoSuchAlgorithmException | CertificateEncodingException e) {
            throw new SecurityException(e);
        }
    }

    public void setSecurityProvider(Provider securityProvider) {
        this.securityProvider = securityProvider;
    }

    public void setIncludeCoseTag(boolean includeCoseTag) {
        this.includeCoseTag = includeCoseTag;
    }

    public void setIncludeCwtTag(boolean includeCwtTag) {
        this.includeCwtTag = includeCwtTag;
    }

    private static class CredentialWrapper {
        private PrivateKey signerKey;
        private X509Certificate signerCertificate;
        private PkiCredential signerCredential;

        public CredentialWrapper(PrivateKey signerKey, X509Certificate signerCertificate) {
            this.signerKey = Optional.ofNullable(signerKey).orElseThrow(() -> new IllegalArgumentException("signerKey must not be null"));
            this.signerCertificate = Optional.ofNullable(signerCertificate).orElseThrow(() -> new IllegalArgumentException("signerCertificate must not be null"));
        }

        public CredentialWrapper(PkiCredential signerCredential) {
            this.signerCredential = Optional.ofNullable(signerCredential).orElseThrow(() -> new IllegalArgumentException("signerCredential must not be null"));
        }

        public PublicKey getPublicKey() {
            return this.getCertificate().getPublicKey();
        }

        public X509Certificate getCertificate() {
            return this.signerCredential != null ? this.signerCredential.getCertificate() : this.signerCertificate;
        }

        public PrivateKey getPrivateKey() {
            return this.signerCredential != null ? this.signerCredential.getPrivateKey() : this.signerKey;
        }
    }
}

