/*****************************************************************************************
* Copyright (c) 2008 Hewlett-Packard Development Company, L.P.
* Permission is hereby granted, free of charge, to any person obtaining a copy of
* this software and associated documentation files (the "Software"), to deal in
* the Software without restriction, including without limitation the rights to use,
* copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
* Software, and to permit persons to whom the Software is furnished to do so,
* subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
* INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
* PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
* HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
* CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
* OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*****************************************************************************************/
/****************************************************************************************
 * SVN MACROS
 *
 * $Revision: 217 $
 * $Author: abla626 $
 * $LastChangedDate: 2012-02-16 17:30:49 +1300 (Thu, 16 Feb 2012) $
 ************************************************************************************/
package com.hp.hpl.inkml;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Observable;
import java.util.logging.Logger;
/**
 * This class models the <ink> element (root element) of InkML document.
 * @author Muthuselvam Selvaraj
 * @version 0.5.0
 * Creation date : 7th May, 2007
 */

public class Ink extends Observable {

	/**
	 * This is an abstract class used to notify observers 
	 * the changes within the ink
	 * 
	 * @author Arturo Blas Jimenez
	 */
	public abstract class TraceDataChanged
	{
		/**
		 * Elements changed within the ink
		 */
		private final Collection<TraceDataElement> changed;
		
		/**
		 * Construcotr
		 */
		private TraceDataChanged()
		{
			changed = new LinkedList<TraceDataElement>();
		}
		
		/**
		 * Constructor
		 * @param changed Changed elements
		 */
		private TraceDataChanged(final Collection<TraceDataElement> changed)
		{
			this.changed = changed;
		}
		
		/**
		 * Constructor
		 * @param changed Changed elements
		 */
		private TraceDataChanged(final TraceDataElement changed)
		{
			this();
			this.changed.add(changed);
		}
		
		/**
		 * @return The list of changes
		 */
		public final Collection<TraceDataElement> getChanges()
		{
			return changed;
		}
		
		/**
		 * @return The list of traces within the changes
		 */
		public final Collection<Trace> getChangedTraces()
		{
			Collection<Trace> changedTraces = new ArrayList<Trace>();
			for(TraceDataElement e : changed)
			{
				if(e instanceof Trace)
				{
					changedTraces.add((Trace)e);
				}
				else if(e instanceof TraceGroup)
				{
					TraceGroup group = (TraceGroup) e;
					changedTraces.addAll(group.getTraceList());
				}
				else if(e instanceof TraceView)
				{
					TraceView view = (TraceView) e;
					try 
					{
						changedTraces.addAll(view.getTraceList());
					} 
					catch (InkMLException e1) 
					{
						e1.printStackTrace();
					}
				}	
			}
			return changedTraces;
		}
	}
	
	/**
	 * This is used to notify observers about new trace
	 * data added to the ink
	 * @author Arturo Blas Jimenez
	 */
	public final class TraceDataAdded extends TraceDataChanged
	{
		/**
		 * Constructor
		 * @param changed Changed elements
		 */
		private TraceDataAdded(final Collection<TraceDataElement> changed)
		{
			super(changed);
		}
		
		/**
		 * Constructor
		 * @param changed Changed elements
		 */
		private TraceDataAdded(final TraceDataElement changed)
		{
			super(changed);
		}
	}
	
	/**
	 * This is used to notify observers about trace
	 * data removed from the ink
	 * @author Arturo Blas Jimenez
	 */
	public class TraceDataRemoved extends TraceDataChanged
	{
		/**
		 * Constructor
		 */
		private TraceDataRemoved()
		{
			super();
		}
		
		/**
		 * Constructor
		 * @param changed Changed elements
		 */
		private TraceDataRemoved(final Collection<TraceDataElement> changed)
		{
			super(changed);
		}
		
		/**
		 * Constructor
		 * @param changed Changed elements
		 */
		private TraceDataRemoved(final TraceDataElement changed)
		{
			super(changed);
		}
	}
	
	/**
	 * This is used to notify observers that all trace
	 * data has been cleared from the ink
	 * @author Arturo Blas Jimenez
	 */
	public final class TraceDataCleared extends TraceDataRemoved
	{
		private TraceDataCleared()
		{
			super();
		}
	}
	
	public static final String INKML_NAMESPACE = "http://www.w3.org/2003/InkML";

	public static final String XSD_NAMESPACE = "http://www.w3.org/2001/XMLSchema-datatypes";
	/**
	 * This indicate what kind of context change happened.
	 * The change in context happens wheneither or many of the following happens,
	 * brush, traceFormat, inkSource, canvas, canvasTransform, timestamp.
	 * Default value none which is used to indicate no change in context.
	 *
	 */
	public enum contextChangeStatus {
		/**
		 * Brush is changed
		 */
		isBrushChanged, 
		/**
		 * traceFormat is changed
		 */
		isTraceFormatChanged, 
		/**
		 * inkSource is changed
		 */
		isInkSourceChanged,
		/**
		 * canvas is changed
		 */
		isCanvasChanged, 
		/**
		 * canvasTransform is changed
		 */
		isCanvasTransformChanged,
		/**
		 * timestamp is changed
		 */
		isTimestampChanged, 
		/**
		 * no context change
		 */
		NONE
	};

	private LinkedList<TraceDataElement> traceDataList;
	private Definitions definitions;
	private Context currentContext;
	private Annotation annotation;
	private AnnotationXML annotationXML;
	// Create logger instance for logging
	private static Logger logger =
		 Logger.getLogger(Ink.class.getName());

	/**
	 *
	 */
	private String docID="";

	/**
	 * No argument Constructor creates a 'blank' InkML Ink object.
	 */
	public Ink() {
		this.definitions = Definitions.getDefaultDefinitions();
		this.currentContext = Context.getDefaultContext();
		traceDataList = new LinkedList<TraceDataElement>();
	}
	/**
	 * Compare the Context object in the parameter with the current context of this Ink Object and returns the status.
	 * @param context Context object to be compared with.
	 * @throws InkMLException
	 */
	public ArrayList<contextChangeStatus> getContextChanges (Context context) throws InkMLException{
		ArrayList<contextChangeStatus> ctxStatus = new ArrayList<contextChangeStatus>();
		if(null == context)
			return ctxStatus;

		// check and populate the status of all the contextual elements
			Brush brush = context.getBrush();
			TraceFormat traceFormat =context.getTraceFormat();
			InkSource inkSource = context.getInkSource();
			Canvas canvas = context.getCanvas();
			CanvasTransform canvasTransform = context.getCanvasTransform();
			Timestamp timestamp = context.getTimestamp();

			if(brush!=null && !this.currentContext.getBrush().equals(brush)){
				ctxStatus.add(contextChangeStatus.isBrushChanged);
			}
			if(traceFormat!=null && !this.currentContext.getTraceFormat().equals(traceFormat)){
				ctxStatus.add(contextChangeStatus.isTraceFormatChanged);
			}
			if(inkSource!=null && !this.currentContext.getInkSource().equals(inkSource)){
				ctxStatus.add(contextChangeStatus.isInkSourceChanged);
			}
			if(canvas!=null && !this.currentContext.getCanvas().equals(canvas)){
				ctxStatus.add(contextChangeStatus.isCanvasChanged);
			}
			if(canvasTransform!=null && !this.currentContext.getCanvasTransform().equals(canvasTransform)){
				ctxStatus.add(contextChangeStatus.isCanvasTransformChanged);
			}
			if(timestamp!=null && !this.currentContext.getTimestamp().equals(timestamp)){
				ctxStatus.add(contextChangeStatus.isTimestampChanged);
			}
		return ctxStatus;
	}
	/**
	 * Method to add a trace or traceGroup or a traceView to Ink document data object
	 * @param traceData
	 */
	public void addToTraceDataList(TraceDataElement traceData){
		traceDataList.add(traceData);
		notifyChange(new TraceDataAdded(traceData));
	}

	/**
	 * This method gives an Iterator for the List of Trace objects that -
	 * the Ink object contains. It includes the Trace object contained in
	 * Trace collections such as TraceGroup and TraceView.
	 * @return The Itearator object that can be used to navigate the list of -
	 *         Trace objects that contained within the current Ink document.
	 * @throws InkMLException
	 */
	public Iterator getTraceIterator() throws InkMLException {
		ArrayList<Trace> traceList = new ArrayList<Trace>();
		if(traceDataList != null){
			Iterator inkChildrenListIterator = traceDataList.iterator();
			while(inkChildrenListIterator.hasNext()) {
				InkElement object = (InkElement)inkChildrenListIterator.next();
				String inkElmntType = object.getInkElementType();
				if("Trace".equals(inkElmntType)){
					traceList.add((Trace)object);
				} if("TraceGroup".equals(inkElmntType)) {
					traceList.addAll(((TraceGroup)object).getTraceList());
				} if("TraceGroup".equals(inkElmntType)){
					traceList.addAll(((TraceView)object).getTraceList());
				}
			}
		}
		return traceList.iterator();
	}

	/**
	 * This method gives the global definition state that is associated with -
	 * the current <ink> document.
	 * @return the Definition object.
	 */
	public Definitions getDefinitions() {
		return this.definitions;
	}

	/**
	 * This method gives the 'ID' of the current Ink Document's root <ink> element ID.
	 * @return the ID string of the <ink> element.
	 */
	public String getDocID() {
		return docID;
	}

	/**
	 * Method to add a Trace
	 * @param trace
	 */
	public void addTrace(Trace trace) {
		addToTraceDataList(trace);
	}

	/**
	 * Method to add a TraceGroup
	 * @param traceGroup
	 */
	public void addTraceGroup(TraceGroup traceGroup) {
		addToTraceDataList(traceGroup);
	}

	/**
	 * Method to add a TraceView
	 * @param traceView
	 */
	public void addTraceView(TraceView traceView) {
		addToTraceDataList(traceView);
	}
	
	/**
	 * Method to add all traceData instances given in the parameter
	 * @param traceDataList
	 * @return boolean
	 * @see java.util.AbstractCollection#removeAll(java.util.Collection)
	 */
	public boolean addAllTraceData(Collection<TraceDataElement> traceDataList) {
		boolean returnVal = this.traceDataList.addAll(traceDataList);
		notifyChange(new TraceDataAdded(traceDataList));
		return returnVal;
	}

	/**
	 * Method to empty the trace list
	 */
	public void clearTraceDataList() {
		traceDataList.clear();
		notifyChange(new TraceDataCleared());
	}

	/**
	 * @param traceData
	 * @return boolean status
	 */
	public boolean containsInTraceDataList(TraceDataElement traceData) {
		return traceDataList.contains(traceData);
	}

	/**
	 * @param traceData collection
	 * @return boolean status
	 */
	public boolean containsAllInTraceDataList(Collection<TraceDataElement> traceData) {
		return traceDataList.containsAll(traceData);
	}

	/**
	 * Method to check if traceData list of Ink is empty
	 * @return boolean status
	 */
	public boolean isTraceDataListEmpty() {
		return traceDataList.isEmpty();
	}

	/**
	 * Method to remove traceData at givem index parameter
	 * @param index
	 * @return InkElement
	 */
	public InkElement removeFromTraceDataList(int index) {
		TraceDataElement element =  traceDataList.remove(index);
		notifyChange(new TraceDataRemoved(element));
		return (InkElement) element;
	}

	/**
	 * Method to remove given traceData in the parameter
	 * @param traceData to be removed
	 * @return status
	 * @see java.util.ArrayList#remove(java.lang.Object)
	 */
	public boolean removeFromTraceDataList(TraceDataElement traceData) {
		boolean element = traceDataList.remove(traceData);
		notifyChange(new TraceDataRemoved(traceData));
		return element;
	}

	/**
	 * Method to remove all traceData instances given in the parameter
	 * @param traceDataList
	 * @return boolean
	 * @see java.util.AbstractCollection#removeAll(java.util.Collection)
	 */
	public boolean removeAllTraceData(Collection<TraceDataElement> traceDataList) {
		boolean returnVal = this.traceDataList.removeAll(traceDataList);
		notifyChange(new TraceDataRemoved(traceDataList));
		return returnVal;
	}

	/**
	 * Method to remove other traceData but retain them given in the parameter
	 * @param traceDataList collection
	 * @return boolean status
	 * @see java.util.AbstractCollection#retainAll(java.util.Collection)
	 */
	public boolean retainAllTraceData(Collection<TraceDataElement> traceDataList) {
		Collection<TraceDataElement> removed = new ArrayList<TraceDataElement>();
		for(TraceDataElement elem : this.traceDataList)
		{
			if(!traceDataList.contains(elem))
				removed.add(elem);
		}
		boolean returnVal =  this.traceDataList.retainAll(traceDataList);
		notifyChange(new TraceDataRemoved(removed));
		return returnVal;
	}

	/**
	 * Method to get the global 'current context' in the scope of the ink document data object
	 * @return Context
	 */

	public Context getCurrentContext() {
		return this.currentContext;
	}

	/**
	 * Method to set the documentID
	 * @param String the documentID URI as string
	 */
	void setDocID(String id) {
		this.docID = id;
		notifyChange();
	}
	/**
	 * Method to add annotation to the ink document, to add semantic label to it.
	 * It is an optional element and users can utilize this to tag the ink document with some meta data information.
	 * @param annotation data it corresponds to a textual description wrapped by {@code <annotation>} element in the markup.
	 */
	public void addAnnotation(Annotation annotation) {
		this.annotation = annotation;
		notifyChange();
	}
	/**
     * Method to add annotationXML data to the ink document, to add semantic label to it.
	 * It is an optional element and users can utilize this to tag the ink document with some meta data information in XML format.
	 * @param aXml AnnotationXML data, it corresponds to an XML tree wrapped by {@code <annotationXML>} element in the markup.
	 * @see com.hp.hpl.inkml.AnnotationXML
	 */
	public void addAnnotationXML(AnnotationXML aXml) {
		this.annotationXML = aXml;
		notifyChange();
	}

	/**
	 * Method to get the list of trace, traceGroup and traceView data that belongs to the ink data object
	 * @return LinkedList<TraceDataElement>
	 */
	public LinkedList<TraceDataElement> getTraceDataList() {
		return traceDataList;
	}

	/**
	 * Method to get the annotation/label of the ink document data object
	 * @return Annotation
	 */
	public Annotation getAnnotation() {
		return annotation;
	}

	/**
	 * Method to assign an annotation, which is also called as semantic label.
	 * @param annotation the annotation to set
	 */
	public void setAnnotation(Annotation annotation) {
		this.annotation = annotation;
		notifyChange();
	}

	/**
	 * Method to get the annotationXML/label XML data of the ink document data object
	 * @return the annotationXML
	 */
	public AnnotationXML getAnnotationXML() {
		return annotationXML;
	}

	/**
	 * Method to assign an annotationXML, which is also called as semantic label XML data.
	 * @param annotationXML the annotationXML to set
	 */
	public void setAnnotationXML(AnnotationXML annotationXML) {
		this.annotationXML = annotationXML;
		notifyChange();
	}

	/**
	 * Method to assign a list of trace data to the ink document.
	 * Any existing trace data is overwritten by the new list.
	 * @param traceDataList the traceDataList to set
	 */
	public void setTraceDataList(LinkedList<TraceDataElement> traceDataList) 
	{
		LinkedList<TraceDataElement> oldTraces = new LinkedList<TraceDataElement>(this.traceDataList);

		this.traceDataList = traceDataList;
		
		if(!oldTraces.isEmpty())
			notifyChange(new TraceDataRemoved(oldTraces));
		if(!this.traceDataList.isEmpty())
			notifyChange(new TraceDataAdded(this.traceDataList));
	}
	/**
	 * Method to set the associated current context object.
	 * @param context
	 */
	public void setCurrentContext(Context context) {
		this.currentContext=context;
		notifyChange();
	}
	
	private void notifyChange()
	{
		notifyChange(null);
	}
	
	private void notifyChange(final Object change)
	{
		setChanged();
		notifyObservers(change);
		clearChanged();
	}

	/**
	 *  Method used by the Archiver component (InkMLWriter) to save the markup data of the Ink data object to file or other data stream
	 *  @param writer
	 */

	public void writeXML(InkMLWriter writer) throws IOException, InkMLException {
		if(writer == null){
			logger.severe("Ink:writeXML, InkMLWriter object not available (null)!!!");
			throw new InkMLException("Ink:writeXML, InkMLWriter object not available (null)!!!");
		}
		LinkedHashMap<String, String> attrMap = new LinkedHashMap<String, String>();
		if(!"".equals(this.docID)){
			attrMap.put("documentID", this.docID);
		}
		attrMap.put("xmlns", INKML_NAMESPACE);
		writer.writeStartTag("ink", attrMap);
		writer.incrementTagLevel();
		// write definitions
		this.definitions.writeXML(writer);

		// write Trace data list
		Iterator<TraceDataElement> iterator = this.traceDataList.iterator();
		while(iterator.hasNext()){
			TraceDataElement data = iterator.next();
			data.writeXML(writer);
		}
		writer.decrementTagLevel();
		writer.writeEndTag("ink");
	}

	/**
	 * Method to give the markup string data of the Ink data object
	 * @return String markup string
	 */
	public String toInkML() {
		StringBuffer xml = new StringBuffer();
		LinkedHashMap<String, String> attrMap = new LinkedHashMap<String, String>();
		if(!"".equals(this.docID)){
			attrMap.put("documentID", this.docID);
		}
		xml.append("<ink>");
		// write definitions
		xml.append(this.definitions.toInkML());

		// write Trace data list
		Iterator<TraceDataElement> iterator = this.traceDataList.iterator();
		while(iterator.hasNext()){
			TraceDataElement data = iterator.next();
			xml.append(data.toInkML());
		}
		xml.append("</ink>");
		return xml.toString();
	} 
}