/**
 * <h2>Amazon API Gateway Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Experimental-important.svg?style=for-the-badge" alt="Stability: Experimental" /></p>
 * <blockquote>
 * <p><strong>This is a <em>developer preview</em> (public beta) module. Releases might lack important features and might have
 * future breaking changes.</strong></p>
 * <p>This API is still under active development and subject to non-backward
 * compatible changes or removal in any future version. Use of the API is not recommended in production
 * environments. Experimental APIs are not subject to the Semantic Versioning model.</p>
 * </blockquote>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>Amazon API Gateway is a fully managed service that makes it easy for developers
 * to publish, maintain, monitor, and secure APIs at any scale. Create an API to
 * access data, business logic, or functionality from your back-end services, such
 * as applications running on Amazon Elastic Compute Cloud (Amazon EC2), code
 * running on AWS Lambda, or any web application.</p>
 * <h3>Defining APIs</h3>
 * <p>APIs are defined as a hierarchy of resources and methods. <code>addResource</code> and
 * <code>addMethod</code> can be used to build this hierarchy. The root resource is
 * <code>api.root</code>.</p>
 * <p>For example, the following code defines an API that includes the following HTTP
 * endpoints: <code>ANY /, GET /books</code>, <code>POST /books</code>, <code>GET /books/{book_id}</code>, <code>DELETE /books/{book_id}</code>.</p>
 * <pre><code class="language-ts">const api = new apigateway.RestApi(this, 'books-api');
 * 
 * api.root.addMethod('ANY');
 * 
 * const books = api.root.addResource('books');
 * books.addMethod('GET');
 * books.addMethod('POST');
 * 
 * const book = books.addResource('{book_id}');
 * book.addMethod('GET');
 * book.addMethod('DELETE');
 * </code></pre>
 * <h3>AWS Lambda-backed APIs</h3>
 * <p>A very common practice is to use Amazon API Gateway with AWS Lambda as the
 * backend integration. The <code>LambdaRestApi</code> construct makes it easy:</p>
 * <p>The following code defines a REST API that routes all requests to the
 * specified AWS Lambda function:</p>
 * <pre><code class="language-ts">const backend = new lambda.Function(...);
 * new apigateway.LambdaRestApi(this, 'myapi', {
 *   handler: backend,
 * });
 * </code></pre>
 * <p>You can also supply <code>proxy: false</code>, in which case you will have to explicitly
 * define the API model:</p>
 * <pre><code class="language-ts">const backend = new lambda.Function(...);
 * const api = new apigateway.LambdaRestApi(this, 'myapi', {
 *   handler: backend,
 *   proxy: false
 * });
 * 
 * const items = api.root.addResource('items');
 * items.addMethod('GET');  // GET /items
 * items.addMethod('POST'); // POST /items
 * 
 * const item = items.addResource('{item}');
 * item.addMethod('GET');   // GET /items/{item}
 * 
 * // the default integration for methods is &quot;handler&quot;, but one can
 * // customize this behavior per method or even a sub path.
 * item.addMethod('DELETE', new apigateway.HttpIntegration('http://amazon.com'));
 * </code></pre>
 * <h3>Integration Targets</h3>
 * <p>Methods are associated with backend integrations, which are invoked when this
 * method is called. API Gateway supports the following integrations:</p>
 * <ul>
 * <li><code>MockIntegration</code> - can be used to test APIs. This is the default
 * integration if one is not specified.</li>
 * <li><code>LambdaIntegration</code> - can be used to invoke an AWS Lambda function.</li>
 * <li><code>AwsIntegration</code> - can be used to invoke arbitrary AWS service APIs.</li>
 * <li><code>HttpIntegration</code> - can be used to invoke HTTP endpoints.</li>
 * </ul>
 * <p>The following example shows how to integrate the <code>GET /book/{book_id}</code> method to
 * an AWS Lambda function:</p>
 * <pre><code class="language-ts">const getBookHandler = new lambda.Function(...);
 * const getBookIntegration = new apigateway.LambdaIntegration(getBookHandler);
 * book.addMethod('GET', getBookIntegration);
 * </code></pre>
 * <p>Integration options can be optionally be specified:</p>
 * <pre><code class="language-ts">const getBookIntegration = new apigateway.LambdaIntegration(getBookHandler, {
 *   contentHandling: apigateway.ContentHandling.ConvertToText, // convert to base64
 *   credentialsPassthrough: true, // use caller identity to invoke the function
 * });
 * </code></pre>
 * <p>Method options can optionally be specified when adding methods:</p>
 * <pre><code class="language-ts">book.addMethod('GET', getBookIntegration, {
 *   authorizationType: apigateway.AuthorizationType.IAM,
 *   apiKeyRequired: true
 * });
 * </code></pre>
 * <p>The following example shows how to use an API Key with a usage plan:</p>
 * <pre><code class="language-ts">const hello = new lambda.Function(this, 'hello', {
 *   runtime: lambda.Runtime.NodeJS810,
 *   handler: 'hello.handler',
 *   code: lambda.Code.asset('lambda')
 * });
 * 
 * const api = new apigateway.RestApi(this, 'hello-api', { });
 * const integration = new apigateway.LambdaIntegration(hello);
 * 
 * const v1 = api.root.addResource('v1');
 * const echo = v1.addResource('echo');
 * const echoMethod = echo.addMethod('GET', integration, { apiKeyRequired: true });
 * const key = api.addApiKey('ApiKey');
 * 
 * const plan = api.addUsagePlan('UsagePlan', {
 *   name: 'Easy',
 *   apiKey: key
 * });
 * 
 * plan.addApiStage({
 *   stage: api.deploymentStage,
 *   throttle: [
 *     {
 *       method: echoMethod,
 *       throttle: {
 *         rateLimit: 10,
 *         burstLimit: 2
 *       }
 *     }
 *   ]
 * });
 * </code></pre>
 * <h4>Default Integration and Method Options</h4>
 * <p>The <code>defaultIntegration</code> and <code>defaultMethodOptions</code> properties can be used to
 * configure a default integration at any resource level. These options will be
 * used when defining method under this resource (recursively) with undefined
 * integration or options.</p>
 * <blockquote>
 * <p>If not defined, the default integration is <code>MockIntegration</code>. See reference
 * documentation for default method options.</p>
 * </blockquote>
 * <p>The following example defines the <code>booksBackend</code> integration as a default
 * integration. This means that all API methods that do not explicitly define an
 * integration will be routed to this AWS Lambda function.</p>
 * <pre><code class="language-ts">const booksBackend = new apigateway.LambdaIntegration(...);
 * const api = new apigateway.RestApi(this, 'books', {
 *   defaultIntegration: booksBackend
 * });
 * 
 * const books = new api.root.addResource('books');
 * books.addMethod('GET');  // integrated with `booksBackend`
 * books.addMethod('POST'); // integrated with `booksBackend`
 * 
 * const book = books.addResource('{book_id}');
 * book.addMethod('GET');   // integrated with `booksBackend`
 * </code></pre>
 * <h3>Proxy Routes</h3>
 * <p>The <code>addProxy</code> method can be used to install a greedy <code>{proxy+}</code> resource
 * on a path. By default, this also installs an <code>&quot;ANY&quot;</code> method:</p>
 * <pre><code class="language-ts">const proxy = resource.addProxy({
 *   defaultIntegration: new LambdaIntegration(handler),
 * 
 *   // &quot;false&quot; will require explicitly adding methods on the `proxy` resource
 *   anyMethod: true // &quot;true&quot; is the default
 * });
 * </code></pre>
 * <h3>Deployments</h3>
 * <p>By default, the <code>RestApi</code> construct will automatically create an API Gateway
 * <a href="https://docs.aws.amazon.com/apigateway/api-reference/resource/deployment/">Deployment</a> and a &quot;prod&quot; <a href="https://docs.aws.amazon.com/apigateway/api-reference/resource/stage/">Stage</a> which represent the API configuration you
 * defined in your CDK app. This means that when you deploy your app, your API will
 * be have open access from the internet via the stage URL.</p>
 * <p>The URL of your API can be obtained from the attribute <code>restApi.url</code>, and is
 * also exported as an <code>Output</code> from your stack, so it's printed when you <code>cdk deploy</code> your app:</p>
 * <pre><code>$ cdk deploy
 * ...
 * books.booksapiEndpointE230E8D5 = https://6lyktd4lpk.execute-api.us-east-1.amazonaws.com/prod/
 * </code></pre>
 * <p>To disable this behavior, you can set <code>{ deploy: false }</code> when creating your
 * API. This means that the API will not be deployed and a stage will not be
 * created for it. You will need to manually define a <code>apigateway.Deployment</code> and
 * <code>apigateway.Stage</code> resources.</p>
 * <p>Use the <code>deployOptions</code> property to customize the deployment options of your
 * API.</p>
 * <p>The following example will configure API Gateway to emit logs and data traces to
 * AWS CloudWatch for all API calls:</p>
 * <blockquote>
 * <p>By default, an IAM role will be created and associated with API Gateway to
 * allow it to write logs and metrics to AWS CloudWatch unless <code>cloudWatchRole</code> is
 * set to <code>false</code>.</p>
 * </blockquote>
 * <pre><code class="language-ts">const api = new apigateway.RestApi(this, 'books', {
 *   deployOptions: {
 *     loggingLevel: apigateway.MethodLoggingLevel.Info,
 *     dataTraceEnabled: true
 *   }
 * })
 * </code></pre>
 * <h4>Deeper dive: invalidation of deployments</h4>
 * <p>API Gateway deployments are an immutable snapshot of the API. This means that we
 * want to automatically create a new deployment resource every time the API model
 * defined in our CDK app changes.</p>
 * <p>In order to achieve that, the AWS CloudFormation logical ID of the
 * <code>AWS::ApiGateway::Deployment</code> resource is dynamically calculated by hashing the
 * API configuration (resources, methods). This means that when the configuration
 * changes (i.e. a resource or method are added, configuration is changed), a new
 * logical ID will be assigned to the deployment resource. This will cause
 * CloudFormation to create a new deployment resource.</p>
 * <p>By default, old deployments are <em>deleted</em>. You can set <code>retainDeployments: true</code>
 * to allow users revert the stage to an old deployment manually.</p>
 * <h3>Missing Features</h3>
 * <p>See <a href="https://github.com/awslabs/aws-cdk/issues/723">awslabs/aws-cdk#723</a> for a
 * list of missing features.</p>
 * <h3>Roadmap</h3>
 * <ul>
 * <li>[ ] Support defining REST API Models <a href="https://github.com/awslabs/aws-cdk/issues/1695">#1695</a></li>
 * </ul>
 * <hr />
 * <p>This module is part of the <a href="https://github.com/awslabs/aws-cdk">AWS Cloud Development Kit</a> project.</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.apigateway;
