/**
 * <h2>Amazon API Gateway Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge" alt="Stability: Stable" /></p>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>Amazon API Gateway is a fully managed service that makes it easy for developers
 * to publish, maintain, monitor, and secure APIs at any scale. Create an API to
 * access data, business logic, or functionality from your back-end services, such
 * as applications running on Amazon Elastic Compute Cloud (Amazon EC2), code
 * running on AWS Lambda, or any web application.</p>
 * <h3>Defining APIs</h3>
 * <p>APIs are defined as a hierarchy of resources and methods. <code>addResource</code> and
 * <code>addMethod</code> can be used to build this hierarchy. The root resource is
 * <code>api.root</code>.</p>
 * <p>For example, the following code defines an API that includes the following HTTP
 * endpoints: <code>ANY /, GET /books</code>, <code>POST /books</code>, <code>GET /books/{book_id}</code>, <code>DELETE /books/{book_id}</code>.</p>
 * <pre><code class="language-ts">const api = new apigateway.RestApi(this, 'books-api');
 * 
 * api.root.addMethod('ANY');
 * 
 * const books = api.root.addResource('books');
 * books.addMethod('GET');
 * books.addMethod('POST');
 * 
 * const book = books.addResource('{book_id}');
 * book.addMethod('GET');
 * book.addMethod('DELETE');
 * </code></pre>
 * <h3>AWS Lambda-backed APIs</h3>
 * <p>A very common practice is to use Amazon API Gateway with AWS Lambda as the
 * backend integration. The <code>LambdaRestApi</code> construct makes it easy:</p>
 * <p>The following code defines a REST API that routes all requests to the
 * specified AWS Lambda function:</p>
 * <pre><code class="language-ts">const backend = new lambda.Function(...);
 * new apigateway.LambdaRestApi(this, 'myapi', {
 *   handler: backend,
 * });
 * </code></pre>
 * <p>You can also supply <code>proxy: false</code>, in which case you will have to explicitly
 * define the API model:</p>
 * <pre><code class="language-ts">const backend = new lambda.Function(...);
 * const api = new apigateway.LambdaRestApi(this, 'myapi', {
 *   handler: backend,
 *   proxy: false
 * });
 * 
 * const items = api.root.addResource('items');
 * items.addMethod('GET');  // GET /items
 * items.addMethod('POST'); // POST /items
 * 
 * const item = items.addResource('{item}');
 * item.addMethod('GET');   // GET /items/{item}
 * 
 * // the default integration for methods is &quot;handler&quot;, but one can
 * // customize this behavior per method or even a sub path.
 * item.addMethod('DELETE', new apigateway.HttpIntegration('http://amazon.com'));
 * </code></pre>
 * <h3>Integration Targets</h3>
 * <p>Methods are associated with backend integrations, which are invoked when this
 * method is called. API Gateway supports the following integrations:</p>
 * <ul>
 * <li><code>MockIntegration</code> - can be used to test APIs. This is the default
 * integration if one is not specified.</li>
 * <li><code>LambdaIntegration</code> - can be used to invoke an AWS Lambda function.</li>
 * <li><code>AwsIntegration</code> - can be used to invoke arbitrary AWS service APIs.</li>
 * <li><code>HttpIntegration</code> - can be used to invoke HTTP endpoints.</li>
 * </ul>
 * <p>The following example shows how to integrate the <code>GET /book/{book_id}</code> method to
 * an AWS Lambda function:</p>
 * <pre><code class="language-ts">const getBookHandler = new lambda.Function(...);
 * const getBookIntegration = new apigateway.LambdaIntegration(getBookHandler);
 * book.addMethod('GET', getBookIntegration);
 * </code></pre>
 * <p>Integration options can be optionally be specified:</p>
 * <pre><code class="language-ts">const getBookIntegration = new apigateway.LambdaIntegration(getBookHandler, {
 *   contentHandling: apigateway.ContentHandling.CONVERT_TO_TEXT, // convert to base64
 *   credentialsPassthrough: true, // use caller identity to invoke the function
 * });
 * </code></pre>
 * <p>Method options can optionally be specified when adding methods:</p>
 * <pre><code class="language-ts">book.addMethod('GET', getBookIntegration, {
 *   authorizationType: apigateway.AuthorizationType.IAM,
 *   apiKeyRequired: true
 * });
 * </code></pre>
 * <p>The following example shows how to use an API Key with a usage plan:</p>
 * <pre><code class="language-ts">const hello = new lambda.Function(this, 'hello', {
 *   runtime: lambda.Runtime.NODEJS_10_X,
 *   handler: 'hello.handler',
 *   code: lambda.Code.fromAsset('lambda')
 * });
 * 
 * const api = new apigateway.RestApi(this, 'hello-api', { });
 * const integration = new apigateway.LambdaIntegration(hello);
 * 
 * const v1 = api.root.addResource('v1');
 * const echo = v1.addResource('echo');
 * const echoMethod = echo.addMethod('GET', integration, { apiKeyRequired: true });
 * const key = api.addApiKey('ApiKey');
 * 
 * const plan = api.addUsagePlan('UsagePlan', {
 *   name: 'Easy',
 *   apiKey: key
 * });
 * 
 * plan.addApiStage({
 *   stage: api.deploymentStage,
 *   throttle: [
 *     {
 *       method: echoMethod,
 *       throttle: {
 *         rateLimit: 10,
 *         burstLimit: 2
 *       }
 *     }
 *   ]
 * });
 * </code></pre>
 * <h3>Working with models</h3>
 * <p>When you work with Lambda integrations that are not Proxy integrations, you
 * have to define your models and mappings for the request, response, and integration.</p>
 * <pre><code class="language-ts">const hello = new lambda.Function(this, 'hello', {
 *   runtime: lambda.Runtime.NODEJS_10_X,
 *   handler: 'hello.handler',
 *   code: lambda.Code.fromAsset('lambda')
 * });
 * 
 * const api = new apigateway.RestApi(this, 'hello-api', { });
 * const resource = api.root.addResource('v1');
 * </code></pre>
 * <p>You can define more parameters on the integration to tune the behavior of API Gateway</p>
 * <pre><code class="language-ts">const integration = new LambdaIntegration(hello, {
 *   proxy: false,
 *   requestParameters: {
 *     // You can define mapping parameters from your method to your integration
 *     // - Destination parameters (the key) are the integration parameters (used in mappings)
 *     // - Source parameters (the value) are the source request parameters or expressions
 *     // @see: https://docs.aws.amazon.com/apigateway/latest/developerguide/request-response-data-mappings.html
 *     'integration.request.querystring.who': 'method.request.querystring.who'
 *   },
 *   allowTestInvoke: true,
 *   requestTemplates: {
 *     // You can define a mapping that will build a payload for your integration, based
 *     //  on the integration parameters that you have specified
 *     // Check: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
 *     'application/json': JSON.stringify({ action: 'sayHello', pollId: &quot;$util.escapeJavaScript($input.params('who'))&quot; })
 *   },
 *   // This parameter defines the behavior of the engine is no suitable response template is found
 *   passthroughBehavior: PassthroughBehavior.NEVER,
 *   integrationResponses: [
 *     {
 *       // Successful response from the Lambda function, no filter defined
 *       //  - the selectionPattern filter only tests the error message
 *       // We will set the response status code to 200
 *       statusCode: &quot;200&quot;,
 *       responseTemplates: {
 *         // This template takes the &quot;message&quot; result from the Lambda function, adn embeds it in a JSON response
 *         // Check https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
 *         'application/json': JSON.stringify({ state: 'ok', greeting: '$util.escapeJavaScript($input.body)' })
 *       },
 *       responseParameters: {
 *         // We can map response parameters
 *         // - Destination parameters (the key) are the response parameters (used in mappings)
 *         // - Source parameters (the value) are the integration response parameters or expressions
 *         'method.response.header.Content-Type': &quot;'application/json'&quot;,
 *         'method.response.header.Access-Control-Allow-Origin': &quot;'*'&quot;,
 *         'method.response.header.Access-Control-Allow-Credentials': &quot;'true'&quot;
 *       }
 *     },
 *     {
 *       // For errors, we check if the error message is not empty, get the error data
 *       selectionPattern: '(\n|.)+',
 *       // We will set the response status code to 200
 *       statusCode: &quot;400&quot;,
 *       responseTemplates: {
 *           'application/json': JSON.stringify({ state: 'error', message: &quot;$util.escapeJavaScript($input.path('$.errorMessage'))&quot; })
 *       },
 *       responseParameters: {
 *           'method.response.header.Content-Type': &quot;'application/json'&quot;,
 *           'method.response.header.Access-Control-Allow-Origin': &quot;'*'&quot;,
 *           'method.response.header.Access-Control-Allow-Credentials': &quot;'true'&quot;
 *       }
 *     }
 *   ]
 * });
 * 
 * </code></pre>
 * <p>You can define models for your responses (and requests)</p>
 * <pre><code class="language-ts">// We define the JSON Schema for the transformed valid response
 * const responseModel = api.addModel('ResponseModel', {
 *   contentType: 'application/json',
 *   modelName: 'ResponseModel',
 *   schema: { '$schema': 'http://json-schema.org/draft-04/schema#', 'title': 'pollResponse', 'type': 'object', 'properties': { 'state': { 'type': 'string' }, 'greeting': { 'type': 'string' } } }
 * });
 * 
 * // We define the JSON Schema for the transformed error response
 * const errorResponseModel = api.addModel('ErrorResponseModel', {
 *   contentType: 'application/json',
 *   modelName: 'ErrorResponseModel',
 *   schema: { '$schema': 'http://json-schema.org/draft-04/schema#', 'title': 'errorResponse', 'type': 'object', 'properties': { 'state': { 'type': 'string' }, 'message': { 'type': 'string' } } }
 * });
 * 
 * </code></pre>
 * <p>And reference all on your method definition.</p>
 * <pre><code class="language-ts">// If you want to define parameter mappings for the request, you need a validator
 * const validator = api.addRequestValidator('DefaultValidator', {
 *   validateRequestBody: false,
 *   validateRequestParameters: true
 * });
 * resource.addMethod('GET', integration, {
 *   // We can mark the parameters as required
 *   requestParameters: {
 *     'method.request.querystring.who': true
 *   },
 *   // We need to set the validator for ensuring they are passed
 *   requestValidator: validator,
 *   methodResponses: [
 *     {
 *       // Successful response from the integration
 *       statusCode: '200',
 *       // Define what parameters are allowed or not
 *       responseParameters: {
 *         'method.response.header.Content-Type': true,
 *         'method.response.header.Access-Control-Allow-Origin': true,
 *         'method.response.header.Access-Control-Allow-Credentials': true
 *       },
 *       // Validate the schema on the response
 *       responseModels: {
 *         'application/json': responseModel
 *       }
 *     },
 *     {
 *       // Same thing for the error responses
 *       statusCode: '400',
 *       responseParameters: {
 *         'method.response.header.Content-Type': true,
 *         'method.response.header.Access-Control-Allow-Origin': true,
 *         'method.response.header.Access-Control-Allow-Credentials': true
 *       },
 *       responseModels: {
 *         'application/json': errorResponseModel
 *       }
 *     }
 *   ]
 * });
 * </code></pre>
 * <h4>Default Integration and Method Options</h4>
 * <p>The <code>defaultIntegration</code> and <code>defaultMethodOptions</code> properties can be used to
 * configure a default integration at any resource level. These options will be
 * used when defining method under this resource (recursively) with undefined
 * integration or options.</p>
 * <blockquote>
 * <p>If not defined, the default integration is <code>MockIntegration</code>. See reference
 * documentation for default method options.</p>
 * </blockquote>
 * <p>The following example defines the <code>booksBackend</code> integration as a default
 * integration. This means that all API methods that do not explicitly define an
 * integration will be routed to this AWS Lambda function.</p>
 * <pre><code class="language-ts">const booksBackend = new apigateway.LambdaIntegration(...);
 * const api = new apigateway.RestApi(this, 'books', {
 *   defaultIntegration: booksBackend
 * });
 * 
 * const books = new api.root.addResource('books');
 * books.addMethod('GET');  // integrated with `booksBackend`
 * books.addMethod('POST'); // integrated with `booksBackend`
 * 
 * const book = books.addResource('{book_id}');
 * book.addMethod('GET');   // integrated with `booksBackend`
 * </code></pre>
 * <h3>Proxy Routes</h3>
 * <p>The <code>addProxy</code> method can be used to install a greedy <code>{proxy+}</code> resource
 * on a path. By default, this also installs an <code>&quot;ANY&quot;</code> method:</p>
 * <pre><code class="language-ts">const proxy = resource.addProxy({
 *   defaultIntegration: new LambdaIntegration(handler),
 * 
 *   // &quot;false&quot; will require explicitly adding methods on the `proxy` resource
 *   anyMethod: true // &quot;true&quot; is the default
 * });
 * </code></pre>
 * <h3>Deployments</h3>
 * <p>By default, the <code>RestApi</code> construct will automatically create an API Gateway
 * <a href="https://docs.aws.amazon.com/apigateway/api-reference/resource/deployment/">Deployment</a> and a &quot;prod&quot; <a href="https://docs.aws.amazon.com/apigateway/api-reference/resource/stage/">Stage</a> which represent the API configuration you
 * defined in your CDK app. This means that when you deploy your app, your API will
 * be have open access from the internet via the stage URL.</p>
 * <p>The URL of your API can be obtained from the attribute <code>restApi.url</code>, and is
 * also exported as an <code>Output</code> from your stack, so it's printed when you <code>cdk deploy</code> your app:</p>
 * <pre><code>$ cdk deploy
 * ...
 * books.booksapiEndpointE230E8D5 = https://6lyktd4lpk.execute-api.us-east-1.amazonaws.com/prod/
 * </code></pre>
 * <p>To disable this behavior, you can set <code>{ deploy: false }</code> when creating your
 * API. This means that the API will not be deployed and a stage will not be
 * created for it. You will need to manually define a <code>apigateway.Deployment</code> and
 * <code>apigateway.Stage</code> resources.</p>
 * <p>Use the <code>deployOptions</code> property to customize the deployment options of your
 * API.</p>
 * <p>The following example will configure API Gateway to emit logs and data traces to
 * AWS CloudWatch for all API calls:</p>
 * <blockquote>
 * <p>By default, an IAM role will be created and associated with API Gateway to
 * allow it to write logs and metrics to AWS CloudWatch unless <code>cloudWatchRole</code> is
 * set to <code>false</code>.</p>
 * </blockquote>
 * <pre><code class="language-ts">const api = new apigateway.RestApi(this, 'books', {
 *   deployOptions: {
 *     loggingLevel: apigateway.MethodLoggingLevel.INFO,
 *     dataTraceEnabled: true
 *   }
 * })
 * </code></pre>
 * <h4>Deeper dive: invalidation of deployments</h4>
 * <p>API Gateway deployments are an immutable snapshot of the API. This means that we
 * want to automatically create a new deployment resource every time the API model
 * defined in our CDK app changes.</p>
 * <p>In order to achieve that, the AWS CloudFormation logical ID of the
 * <code>AWS::ApiGateway::Deployment</code> resource is dynamically calculated by hashing the
 * API configuration (resources, methods). This means that when the configuration
 * changes (i.e. a resource or method are added, configuration is changed), a new
 * logical ID will be assigned to the deployment resource. This will cause
 * CloudFormation to create a new deployment resource.</p>
 * <p>By default, old deployments are <em>deleted</em>. You can set <code>retainDeployments: true</code>
 * to allow users revert the stage to an old deployment manually.</p>
 * <h3>Custom Domains</h3>
 * <p>To associate an API with a custom domain, use the <code>domainName</code> configuration when
 * you define your API:</p>
 * <pre><code class="language-ts">const api = new apigw.RestApi(this, 'MyDomain', {
 *   domainName: {
 *     domainName: 'example.com',
 *     certificate: acmCertificateForExampleCom,
 *   },
 * });
 * </code></pre>
 * <p>This will define a <code>DomainName</code> resource for you, along with a <code>BasePathMapping</code>
 * from the root of the domain to the deployment stage of the API. This is a common
 * set up.</p>
 * <p>To route domain traffic to an API Gateway API, use Amazon Route 53 to create an
 * alias record. An alias record is a Route 53 extension to DNS. It's similar to a
 * CNAME record, but you can create an alias record both for the root domain, such
 * as <code>example.com</code>, and for subdomains, such as <code>www.example.com</code>. (You can create
 * CNAME records only for subdomains.)</p>
 * <pre><code class="language-ts">new route53.ARecord(this, 'CustomDomainAliasRecord', {
 *   zone: hostedZoneForExampleCom,
 *   target: route53.AddressRecordTarget.fromAlias(new route53_targets.ApiGateway(api))
 * });
 * </code></pre>
 * <p>You can also define a <code>DomainName</code> resource directly in order to customize the default behavior:</p>
 * <pre><code class="language-ts">new apigw.DomainName(this, 'custom-domain', {
 *   domainName: 'example.com',
 *   certificate: acmCertificateForExampleCom,
 *   endpointType: apigw.EndpointType.EDGE // default is REGIONAL
 * });
 * </code></pre>
 * <p>Once you have a domain, you can map base paths of the domain to APIs.
 * The following example will map the URL https://example.com/go-to-api1
 * to the <code>api1</code> API and https://example.com/boom to the <code>api2</code> API.</p>
 * <pre><code class="language-ts">domain.addBasePathMapping(api1, { basePath: 'go-to-api1' });
 * domain.addBasePathMapping(api2, { basePath: 'boom' });
 * </code></pre>
 * <p>NOTE: currently, the mapping will always be assigned to the APIs
 * <code>deploymentStage</code>, which will automatically assigned to the latest API
 * deployment. Raise a GitHub issue if you require more granular control over
 * mapping base paths to stages.</p>
 * <p>If you don't specify <code>basePath</code>, all URLs under this domain will be mapped
 * to the API, and you won't be able to map another API to the same domain:</p>
 * <pre><code class="language-ts">domain.addBasePathMapping(api);
 * </code></pre>
 * <p>This can also be achieved through the <code>mapping</code> configuration when defining the
 * domain as demonstrated above.</p>
 * <p>If you wish to setup this domain with an Amazon Route53 alias, use the <code>route53_targets.ApiGatewayDomain</code>:</p>
 * <pre><code class="language-ts">new route53.ARecord(this, 'CustomDomainAliasRecord', {
 *   zone: hostedZoneForExampleCom,
 *   target: route53.AddressRecordTarget.fromAlias(new route53_targets.ApiGatewayDomain(domainName))
 * });
 * </code></pre>
 * <hr />
 * <p>This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.apigateway;
