/**
 * <h2>Amazon API Gateway Construct Library</h2>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cfn-resources: Stable" src="https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge">
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * Amazon API Gateway is a fully managed service that makes it easy for developers
 * to publish, maintain, monitor, and secure APIs at any scale. Create an API to
 * access data, business logic, or functionality from your back-end services, such
 * as applications running on Amazon Elastic Compute Cloud (Amazon EC2), code
 * running on AWS Lambda, or any web application.
 * <p>
 * <h2>Table of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#defining-apis">Defining APIs</a>
 * <p>
 * <ul>
 * <li><a href="#breaking-up-methods-and-resources-across-stacks">Breaking up Methods and Resources across Stacks</a></li>
 * </ul></li>
 * <li><a href="#aws-lambda-backed-apis">AWS Lambda-backed APIs</a></li>
 * <li><a href="#integration-targets">Integration Targets</a></li>
 * <li><a href="#working-with-models">Working with models</a></li>
 * <li><a href="#default-integration-and-method-options">Default Integration and Method Options</a></li>
 * <li><a href="#proxy-routes">Proxy Routes</a></li>
 * <li><a href="#authorizers">Authorizers</a>
 * <p>
 * <ul>
 * <li><a href="#iam-based-authorizer">IAM-based authorizer</a></li>
 * <li><a href="#lambda-based-token-authorizer">Lambda-based token authorizer</a></li>
 * <li><a href="#lambda-based-request-authorizer">Lambda-based request authorizer</a></li>
 * </ul></li>
 * <li><a href="#deployments">Deployments</a>
 * <p>
 * <ul>
 * <li><a href="#deep-dive-invalidation-of-deployments">Deep dive: Invalidation of deployments</a></li>
 * </ul></li>
 * <li><a href="#custom-domains">Custom Domains</a></li>
 * <li><a href="#access-logging">Access Logging</a></li>
 * <li><a href="#cross-origin-resource-sharing-cors">Cross Origin Resource Sharing (CORS)</a></li>
 * <li><a href="#endpoint-configuration">Endpoint Configuration</a></li>
 * <li><a href="#private-integrations">Private Integrations</a></li>
 * <li><a href="#gateway-response">Gateway Response</a></li>
 * <li><a href="#openapi-definition">OpenAPI Definition</a></li>
 * <li><a href="#apigateway-v2">APIGateway v2</a></li>
 * </ul>
 * <p>
 * <h2>Defining APIs</h2>
 * <p>
 * APIs are defined as a hierarchy of resources and methods. <code>addResource</code> and
 * <code>addMethod</code> can be used to build this hierarchy. The root resource is
 * <code>api.root</code>.
 * <p>
 * For example, the following code defines an API that includes the following HTTP
 * endpoints: <code>ANY /, GET /books</code>, <code>POST /books</code>, <code>GET /books/{book_id}</code>, <code>DELETE /books/{book_id}</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var api = new RestApi(this, "books-api");
 * 
 * api.root.addMethod("ANY");
 * 
 * var books = api.root.addResource("books");
 * books.addMethod("GET");
 * books.addMethod("POST");
 * 
 * var book = books.addResource("{book_id}");
 * book.addMethod("GET");
 * book.addMethod("DELETE");
 * </pre></blockquote>
 * <p>
 * <h2>AWS Lambda-backed APIs</h2>
 * <p>
 * A very common practice is to use Amazon API Gateway with AWS Lambda as the
 * backend integration. The <code>LambdaRestApi</code> construct makes it easy:
 * <p>
 * The following code defines a REST API that routes all requests to the
 * specified AWS Lambda function:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var backend = new Function(...);
 * LambdaRestApi.Builder.create(this, "myapi")
 *         .handler(backend)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also supply <code>proxy: false</code>, in which case you will have to explicitly
 * define the API model:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var backend = new Function(...);
 * var api = LambdaRestApi.Builder.create(this, "myapi")
 *         .handler(backend)
 *         .proxy(false)
 *         .build();
 * 
 * var items = api.root.addResource("items");
 * items.addMethod("GET");// GET /items
 * items.addMethod("POST");// POST /items
 * 
 * var item = items.addResource("{item}");
 * item.addMethod("GET");// GET /items/{item}
 * 
 * // the default integration for methods is "handler", but one can
 * // customize this behavior per method or even a sub path.
 * item.addMethod("DELETE", new HttpIntegration("http://amazon.com"));
 * </pre></blockquote>
 * <p>
 * <h3>Breaking up Methods and Resources across Stacks</h3>
 * <p>
 * It is fairly common for REST APIs with a large number of Resources and Methods to hit the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html">CloudFormation
 * limit</a> of 200 resources per
 * stack.
 * <p>
 * To help with this, Resources and Methods for the same REST API can be re-organized across multiple stacks. A common
 * way to do this is to have a stack per Resource or groups of Resources, but this is not the only possible way.
 * The following example uses sets up two Resources '/pets' and '/books' in separate stacks using nested stacks:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.core.App;
 * import software.amazon.awscdk.core.CfnOutput;
 * import software.amazon.awscdk.core.Construct;
 * import software.amazon.awscdk.core.NestedStack;
 * import software.amazon.awscdk.core.NestedStackProps;
 * import software.amazon.awscdk.core.Stack;
 * import lib.Deployment;
 * import lib.Method;
 * import lib.MockIntegration;
 * import lib.PassthroughBehavior;
 * import lib.RestApi;
 * import lib.Stage;
 * 
 * /**
 *  * This file showcases how to split up a RestApi's Resources and Methods across nested stacks.
 *  *
 *  * The root stack 'RootStack' first defines a RestApi.
 *  * Two nested stacks BooksStack and PetsStack, create corresponding Resources '/books' and '/pets'.
 *  * They are then deployed to a 'prod' Stage via a third nested stack - DeployStack.
 *  *
 *  * To verify this worked, go to the APIGateway
 *  *{@literal /}
 * 
 * public class RootStack extends Stack {
 *     public RootStack(Construct scope) {
 *         super(scope, "integ-restapi-import-RootStack");
 * 
 *         RestApi restApi = new RestApi(this, "RestApi", new RestApiProps()
 *                 .deploy(false));
 *         restApi.root.addMethod("ANY");
 * 
 *         PetsStack petsStack = new PetsStack(this, new ResourceNestedStackProps()
 *                 .restApiId(restApi.getRestApiId())
 *                 .rootResourceId(restApi.getRestApiRootResourceId()));
 *         BooksStack booksStack = new BooksStack(this, new ResourceNestedStackProps()
 *                 .restApiId(restApi.getRestApiId())
 *                 .rootResourceId(restApi.getRestApiRootResourceId()));
 *         new DeployStack(this, new DeployStackProps()
 *                 .restApiId(restApi.getRestApiId())
 *                 .methods(asList((SpreadElement ...petsStack.methods
 *                   petsStack.getMethods()), (SpreadElement ...booksStack.methods
 *                   booksStack.getMethods()))));
 * 
 *         new CfnOutput(this, "PetsURL", new CfnOutputProps()
 *                 .value("https://" + restApi.restApiId + ".execute-api." + this.region + ".amazonaws.com/prod/pets"));
 * 
 *         new CfnOutput(this, "BooksURL", new CfnOutputProps()
 *                 .value("https://" + restApi.restApiId + ".execute-api." + this.region + ".amazonaws.com/prod/books"));
 *     }
 * }
 * 
 * public class ResourceNestedStackProps extends NestedStackProps {
 *     private String restApiId;
 *     public String getRestApiId() {
 *         return this.restApiId;
 *     }
 *     public ResourceNestedStackProps restApiId(String restApiId) {
 *         this.restApiId = restApiId;
 *         return this;
 *     }
 * 
 *     private String rootResourceId;
 *     public String getRootResourceId() {
 *         return this.rootResourceId;
 *     }
 *     public ResourceNestedStackProps rootResourceId(String rootResourceId) {
 *         this.rootResourceId = rootResourceId;
 *         return this;
 *     }
 * }
 * 
 * public class PetsStack extends NestedStack {
 *     public final Array methods;
 * 
 *     public PetsStack(Construct scope, ResourceNestedStackProps props) {
 *         super(scope, "integ-restapi-import-PetsStack", props);
 * 
 *         IRestApi api = RestApi.fromRestApiAttributes(this, "RestApi", new RestApiAttributes()
 *                 .restApiId(props.getRestApiId())
 *                 .rootResourceId(props.getRootResourceId()));
 * 
 *         Method method = api.root.addResource('pets').addMethod("GET", MockIntegration.Builder.create()Map.of(
 *                 "integrationResponses", asList(Map.of(
 *                         "statusCode", "200")),
 *                 "passthroughBehavior", PassthroughBehavior.getNEVER(),
 *                 "requestTemplates", Map.of(
 *                         "application/json", "{ \"statusCode\": 200 }")), new MethodOptions()
 *                 .methodResponses(asList(new MethodResponse().statusCode("200"))));
 * 
 *         this.methods.push(method);
 *     }
 * }
 * 
 * public class BooksStack extends NestedStack {
 *     public final Array methods;
 * 
 *     public BooksStack(Construct scope, ResourceNestedStackProps props) {
 *         super(scope, "integ-restapi-import-BooksStack", props);
 * 
 *         IRestApi api = RestApi.fromRestApiAttributes(this, "RestApi", new RestApiAttributes()
 *                 .restApiId(props.getRestApiId())
 *                 .rootResourceId(props.getRootResourceId()));
 * 
 *         Method method = api.root.addResource('books').addMethod("GET", MockIntegration.Builder.create()Map.of(
 *                 "integrationResponses", asList(Map.of(
 *                         "statusCode", "200")),
 *                 "passthroughBehavior", PassthroughBehavior.getNEVER(),
 *                 "requestTemplates", Map.of(
 *                         "application/json", "{ \"statusCode\": 200 }")), new MethodOptions()
 *                 .methodResponses(asList(new MethodResponse().statusCode("200"))));
 * 
 *         this.methods.push(method);
 *     }
 * }
 * 
 * public class DeployStackProps extends NestedStackProps {
 *     private String restApiId;
 *     public String getRestApiId() {
 *         return this.restApiId;
 *     }
 *     public DeployStackProps restApiId(String restApiId) {
 *         this.restApiId = restApiId;
 *         return this;
 *     }
 * 
 *     private Array methods;
 *     public Array getMethods() {
 *         return this.methods;
 *     }
 *     public DeployStackProps methods(Array methods) {
 *         this.methods = methods;
 *         return this;
 *     }
 * }
 * 
 * public class DeployStack extends NestedStack {
 *     public DeployStack(Construct scope, DeployStackProps props) {
 *         super(scope, "integ-restapi-import-DeployStack", props);
 * 
 *         Deployment deployment = new Deployment(this, "Deployment", new DeploymentProps()
 *                 .api(RestApi.fromRestApiId(this, "RestApi", props.getRestApiId())));
 *         (props.methods ?? []).forEach((method) =&gt; deployment.node.addDependency(method));
 *         new Stage(this, "Stage", new StageProps().deployment(deployment));
 *     }
 * }
 * 
 * new RootStack(new App());
 * </pre></blockquote>
 * <p>
 * <h2>Integration Targets</h2>
 * <p>
 * Methods are associated with backend integrations, which are invoked when this
 * method is called. API Gateway supports the following integrations:
 * <p>
 * <ul>
 * <li><code>MockIntegration</code> - can be used to test APIs. This is the default
 * integration if one is not specified.</li>
 * <li><code>LambdaIntegration</code> - can be used to invoke an AWS Lambda function.</li>
 * <li><code>AwsIntegration</code> - can be used to invoke arbitrary AWS service APIs.</li>
 * <li><code>HttpIntegration</code> - can be used to invoke HTTP endpoints.</li>
 * </ul>
 * <p>
 * The following example shows how to integrate the <code>GET /book/{book_id}</code> method to
 * an AWS Lambda function:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var getBookHandler = new Function(...);
 * var getBookIntegration = new LambdaIntegration(getBookHandler);
 * book.addMethod("GET", getBookIntegration);
 * </pre></blockquote>
 * <p>
 * Integration options can be optionally be specified:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var getBookIntegration = LambdaIntegration.Builder.create(getBookHandler)
 *         .contentHandling(apigateway.ContentHandling.getCONVERT_TO_TEXT())// convert to base64
 *         .credentialsPassthrough(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Method options can optionally be specified when adding methods:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * book.addMethod("GET", getBookIntegration, Map.of(
 *         "authorizationType", apigateway.AuthorizationType.getIAM(),
 *         "apiKeyRequired", true));
 * </pre></blockquote>
 * <p>
 * The following example shows how to use an API Key with a usage plan:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var hello = Function.Builder.create(this, "hello")
 *         .runtime(lambda.Runtime.getNODEJS_10_X())
 *         .handler("hello.handler")
 *         .code(lambda.Code.fromAsset("lambda"))
 *         .build();
 * 
 * var api = RestApi.Builder.create(this, "hello-api").build();
 * var integration = new LambdaIntegration(hello);
 * 
 * var v1 = api.root.addResource("v1");
 * var echo = v1.addResource("echo");
 * var echoMethod = echo.addMethod("GET", integration, Map.of("apiKeyRequired", true));
 * var key = api.addApiKey("ApiKey");
 * 
 * var plan = api.addUsagePlan("UsagePlan", Map.of(
 *         "name", "Easy",
 *         "apiKey", key,
 *         "throttle", Map.of(
 *                 "rateLimit", 10,
 *                 "burstLimit", 2)));
 * 
 * plan.addApiStage(Map.of(
 *         "stage", api.getDeploymentStage(),
 *         "throttle", asList(Map.of(
 *                 "method", echoMethod,
 *                 "throttle", Map.of(
 *                         "rateLimit", 10,
 *                         "burstLimit", 2)))));
 * </pre></blockquote>
 * <p>
 * The name and value of the API Key can be specified at creation; if not
 * provided, a name and value will be automatically generated by API Gateway.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var key = api.addApiKey("ApiKey", Map.of(
 *         "apiKeyName", "myApiKey1",
 *         "value", "MyApiKeyThatIsAtLeast20Characters"));
 * </pre></blockquote>
 * <p>
 * In scenarios where you need to create a single api key and configure rate limiting for it, you can use <code>RateLimitedApiKey</code>.
 * This construct lets you specify rate limiting properties which should be applied only to the api key being created.
 * The API key created has the specified rate limits, such as quota and throttles, applied.
 * <p>
 * The following example shows how to use a rate limited api key :
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var hello = Function.Builder.create(this, "hello")
 *         .runtime(lambda.Runtime.getNODEJS_10_X())
 *         .handler("hello.handler")
 *         .code(lambda.Code.fromAsset("lambda"))
 *         .build();
 * 
 * var api = RestApi.Builder.create(this, "hello-api").build();
 * var integration = new LambdaIntegration(hello);
 * 
 * var v1 = api.root.addResource("v1");
 * var echo = v1.addResource("echo");
 * var echoMethod = echo.addMethod("GET", integration, Map.of("apiKeyRequired", true));
 * 
 * var key = RateLimitedApiKey.Builder.create(this, "rate-limited-api-key")
 *         .customerId("hello-customer")
 *         .resources(asList(api))
 *         .quota(Map.of(
 *                 "limit", 10000,
 *                 "period", apigateway.Period.getMONTH()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Working with models</h2>
 * <p>
 * When you work with Lambda integrations that are not Proxy integrations, you
 * have to define your models and mappings for the request, response, and integration.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var hello = Function.Builder.create(this, "hello")
 *         .runtime(lambda.Runtime.getNODEJS_10_X())
 *         .handler("hello.handler")
 *         .code(lambda.Code.fromAsset("lambda"))
 *         .build();
 * 
 * var api = RestApi.Builder.create(this, "hello-api").build();
 * var resource = api.root.addResource("v1");
 * </pre></blockquote>
 * <p>
 * You can define more parameters on the integration to tune the behavior of API Gateway
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var integration = LambdaIntegration.Builder.create(hello)
 *         .proxy(false)
 *         .requestParameters(Map.of(
 *                 // You can define mapping parameters from your method to your integration
 *                 // - Destination parameters (the key) are the integration parameters (used in mappings)
 *                 // - Source parameters (the value) are the source request parameters or expressions
 *                 // &#64;see: https://docs.aws.amazon.com/apigateway/latest/developerguide/request-response-data-mappings.html
 *                 "integration.request.querystring.who", "method.request.querystring.who"))
 *         .allowTestInvoke(true)
 *         .requestTemplates(Map.of(
 *                 // You can define a mapping that will build a payload for your integration, based
 *                 //  on the integration parameters that you have specified
 *                 // Check: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
 *                 "application/json", JSON.stringify(Map.of("action", "sayHello", "pollId", "$util.escapeJavaScript($input.params('who'))"))))
 *         // This parameter defines the behavior of the engine is no suitable response template is found
 *         .passthroughBehavior(PassthroughBehavior.getNEVER())
 *         .integrationResponses(asList(Map.of(
 *                 // Successful response from the Lambda function, no filter defined
 *                 //  - the selectionPattern filter only tests the error message
 *                 // We will set the response status code to 200
 *                 "statusCode", "200",
 *                 "responseTemplates", Map.of(
 *                         // This template takes the "message" result from the Lambda function, adn embeds it in a JSON response
 *                         // Check https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
 *                         "application/json", JSON.stringify(Map.of("state", "ok", "greeting", "$util.escapeJavaScript($input.body)"))),
 *                 "responseParameters", Map.of(
 *                         // We can map response parameters
 *                         // - Destination parameters (the key) are the response parameters (used in mappings)
 *                         // - Source parameters (the value) are the integration response parameters or expressions
 *                         "method.response.header.Content-Type", "'application/json'",
 *                         "method.response.header.Access-Control-Allow-Origin", "'*'",
 *                         "method.response.header.Access-Control-Allow-Credentials", "'true'")), Map.of(
 *                 // For errors, we check if the error message is not empty, get the error data
 *                 "selectionPattern", "(\n|.)+",
 *                 // We will set the response status code to 200
 *                 "statusCode", "400",
 *                 "responseTemplates", Map.of(
 *                         "application/json", JSON.stringify(Map.of("state", "error", "message", "$util.escapeJavaScript($input.path('$.errorMessage'))"))),
 *                 "responseParameters", Map.of(
 *                         "method.response.header.Content-Type", "'application/json'",
 *                         "method.response.header.Access-Control-Allow-Origin", "'*'",
 *                         "method.response.header.Access-Control-Allow-Credentials", "'true'"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can define models for your responses (and requests)
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // We define the JSON Schema for the transformed valid response
 * var responseModel = api.addModel("ResponseModel", Map.of(
 *         "contentType", "application/json",
 *         "modelName", "ResponseModel",
 *         "schema", Map.of(
 *                 "schema", JsonSchemaVersion.getDRAFT4(),
 *                 "title", "pollResponse",
 *                 "type", JsonSchemaType.getOBJECT(),
 *                 "properties", Map.of(
 *                         "state", Map.of("type", JsonSchemaType.getSTRING()),
 *                         "greeting", Map.of("type", JsonSchemaType.getSTRING())))));
 * 
 * // We define the JSON Schema for the transformed error response
 * var errorResponseModel = api.addModel("ErrorResponseModel", Map.of(
 *         "contentType", "application/json",
 *         "modelName", "ErrorResponseModel",
 *         "schema", Map.of(
 *                 "schema", JsonSchemaVersion.getDRAFT4(),
 *                 "title", "errorResponse",
 *                 "type", JsonSchemaType.getOBJECT(),
 *                 "properties", Map.of(
 *                         "state", Map.of("type", JsonSchemaType.getSTRING()),
 *                         "message", Map.of("type", JsonSchemaType.getSTRING())))));
 * </pre></blockquote>
 * <p>
 * And reference all on your method definition.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * resource.addMethod("GET", integration, Map.of(
 *         // We can mark the parameters as required
 *         "requestParameters", Map.of(
 *                 "method.request.querystring.who", true),
 *         // we can set request validator options like below
 *         "requestValidatorOptions", Map.of(
 *                 "requestValidatorName", "test-validator",
 *                 "validateRequestBody", true,
 *                 "validateRequestParameters", false),
 *         "methodResponses", asList(Map.of(
 *                 // Successful response from the integration
 *                 "statusCode", "200",
 *                 // Define what parameters are allowed or not
 *                 "responseParameters", Map.of(
 *                         "method.response.header.Content-Type", true,
 *                         "method.response.header.Access-Control-Allow-Origin", true,
 *                         "method.response.header.Access-Control-Allow-Credentials", true),
 *                 // Validate the schema on the response
 *                 "responseModels", Map.of(
 *                         "application/json", responseModel)), Map.of(
 *                 // Same thing for the error responses
 *                 "statusCode", "400",
 *                 "responseParameters", Map.of(
 *                         "method.response.header.Content-Type", true,
 *                         "method.response.header.Access-Control-Allow-Origin", true,
 *                         "method.response.header.Access-Control-Allow-Credentials", true),
 *                 "responseModels", Map.of(
 *                         "application/json", errorResponseModel)))));
 * </pre></blockquote>
 * <p>
 * Specifying <code>requestValidatorOptions</code> automatically creates the RequestValidator construct with the given options.
 * However, if you have your RequestValidator already initialized or imported, use the <code>requestValidator</code> option instead.
 * <p>
 * <h2>Default Integration and Method Options</h2>
 * <p>
 * The <code>defaultIntegration</code> and <code>defaultMethodOptions</code> properties can be used to
 * configure a default integration at any resource level. These options will be
 * used when defining method under this resource (recursively) with undefined
 * integration or options.
 * <p>
 * <blockquote>
 * <p>
 * If not defined, the default integration is <code>MockIntegration</code>. See reference
 * documentation for default method options.
 * <p>
 * </blockquote>
 * <p>
 * The following example defines the <code>booksBackend</code> integration as a default
 * integration. This means that all API methods that do not explicitly define an
 * integration will be routed to this AWS Lambda function.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var booksBackend = new LambdaIntegration(...);
 * var api = RestApi.Builder.create(this, "books")
 *         .defaultIntegration(booksBackend)
 *         .build();
 * 
 * var books = new addResource("books");
 * books.addMethod("GET");// integrated with `booksBackend`
 * books.addMethod("POST");// integrated with `booksBackend`
 * 
 * var book = books.addResource("{book_id}");
 * book.addMethod("GET");
 * </pre></blockquote>
 * <p>
 * A Method can be configured with authorization scopes. Authorization scopes are
 * used in conjunction with an <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html#apigateway-enable-cognito-user-pool">authorizer that uses Amazon Cognito user
 * pools</a>.
 * Read more about authorization scopes
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationscopes">here</a>.
 * <p>
 * Authorization scopes for a Method can be configured using the <code>authorizationScopes</code> property as shown below -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * books.addMethod("GET", new HttpIntegration("http://amazon.com"), Map.of(
 *         "authorizationType", AuthorizationType.getCOGNITO(),
 *         "authorizationScopes", asList("Scope1", "Scope2")));
 * </pre></blockquote>
 * <p>
 * <h2>Proxy Routes</h2>
 * <p>
 * The <code>addProxy</code> method can be used to install a greedy <code>{proxy+}</code> resource
 * on a path. By default, this also installs an <code>"ANY"</code> method:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var proxy = resource.addProxy(Map.of(
 *         "defaultIntegration", new LambdaIntegration(handler),
 * 
 *         // "false" will require explicitly adding methods on the `proxy` resource
 *         "anyMethod", true));
 * </pre></blockquote>
 * <p>
 * <h2>Authorizers</h2>
 * <p>
 * API Gateway <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-to-api.html">supports several different authorization types</a>
 * that can be used for controlling access to your REST APIs.
 * <p>
 * <h3>IAM-based authorizer</h3>
 * <p>
 * The following CDK code provides 'execute-api' permission to an IAM user, via IAM policies, for the 'GET' method on the <code>books</code> resource:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var getBooks = books.addMethod("GET", new HttpIntegration("http://amazon.com"), Map.of(
 *         "authorizationType", apigateway.AuthorizationType.getIAM()));
 * 
 * iamUser.attachInlinePolicy(Policy.Builder.create(this, "AllowBooks")
 *         .statements(asList(
 *             PolicyStatement.Builder.create()
 *                     .actions(asList("execute-api:Invoke"))
 *                     .effect(iam.Effect.getAllow())
 *                     .resources(asList(getBooks.methodArn()))
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Lambda-based token authorizer</h3>
 * <p>
 * API Gateway also allows <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">lambda functions to be used as authorizers</a>.
 * <p>
 * This module provides support for token-based Lambda authorizers. When a client makes a request to an API's methods configured with such
 * an authorizer, API Gateway calls the Lambda authorizer, which takes the caller's identity as input and returns an IAM policy as output.
 * A token-based Lambda authorizer (also called a token authorizer) receives the caller's identity in a bearer token, such as
 * a JSON Web Token (JWT) or an OAuth token.
 * <p>
 * API Gateway interacts with the authorizer Lambda function handler by passing input and expecting the output in a specific format.
 * The event object that the handler is called with contains the <code>authorizationToken</code> and the <code>methodArn</code> from the request to the
 * API Gateway endpoint. The handler is expected to return the <code>principalId</code> (i.e. the client identifier) and a <code>policyDocument</code> stating
 * what the client is authorizer to perform.
 * See <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">here</a> for a detailed specification on
 * inputs and outputs of the Lambda handler.
 * <p>
 * The following code attaches a token-based Lambda authorizer to the 'GET' Method of the Book resource:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var authFn = Function.Builder.create(this, "booksAuthorizerLambda").build();
 * 
 * var auth = TokenAuthorizer.Builder.create(this, "booksAuthorizer")
 *         .handler(authFn)
 *         .build();
 * 
 * books.addMethod("GET", new HttpIntegration("http://amazon.com"), Map.of(
 *         "authorizer", auth));
 * </pre></blockquote>
 * <p>
 * You can find a full working example <a href="test/authorizers/integ.token-authorizer.ts">here</a>.
 * <p>
 * By default, the <code>TokenAuthorizer</code> looks for the authorization token in the request header with the key 'Authorization'. This can,
 * however, be modified by changing the <code>identitySource</code> property.
 * <p>
 * Authorizers can also be passed via the <code>defaultMethodOptions</code> property within the <code>RestApi</code> construct or the <code>Method</code> construct. Unless
 * explicitly overridden, the specified defaults will be applied across all <code>Method</code>s across the <code>RestApi</code> or across all <code>Resource</code>s,
 * depending on where the defaults were specified.
 * <p>
 * <h3>Lambda-based request authorizer</h3>
 * <p>
 * This module provides support for request-based Lambda authorizers. When a client makes a request to an API's methods configured with such
 * an authorizer, API Gateway calls the Lambda authorizer, which takes specified parts of the request, known as identity sources,
 * as input and returns an IAM policy as output. A request-based Lambda authorizer (also called a request authorizer) receives
 * the identity sources in a series of values pulled from the request, from the headers, stage variables, query strings, and the context.
 * <p>
 * API Gateway interacts with the authorizer Lambda function handler by passing input and expecting the output in a specific format.
 * The event object that the handler is called with contains the body of the request and the <code>methodArn</code> from the request to the
 * API Gateway endpoint. The handler is expected to return the <code>principalId</code> (i.e. the client identifier) and a <code>policyDocument</code> stating
 * what the client is authorizer to perform.
 * See <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">here</a> for a detailed specification on
 * inputs and outputs of the Lambda handler.
 * <p>
 * The following code attaches a request-based Lambda authorizer to the 'GET' Method of the Book resource:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var authFn = Function.Builder.create(this, "booksAuthorizerLambda").build();
 * 
 * var auth = RequestAuthorizer.Builder.create(this, "booksAuthorizer")
 *         .handler(authFn)
 *         .identitySources(asList(IdentitySource.header("Authorization")))
 *         .build();
 * 
 * books.addMethod("GET", new HttpIntegration("http://amazon.com"), Map.of(
 *         "authorizer", auth));
 * </pre></blockquote>
 * <p>
 * You can find a full working example <a href="test/authorizers/integ.request-authorizer.ts">here</a>.
 * <p>
 * By default, the <code>RequestAuthorizer</code> does not pass any kind of information from the request. This can,
 * however, be modified by changing the <code>identitySource</code> property, and is required when specifying a value for caching.
 * <p>
 * Authorizers can also be passed via the <code>defaultMethodOptions</code> property within the <code>RestApi</code> construct or the <code>Method</code> construct. Unless
 * explicitly overridden, the specified defaults will be applied across all <code>Method</code>s across the <code>RestApi</code> or across all <code>Resource</code>s,
 * depending on where the defaults were specified.
 * <p>
 * <h2>Deployments</h2>
 * <p>
 * By default, the <code>RestApi</code> construct will automatically create an API Gateway
 * <a href="https://docs.aws.amazon.com/apigateway/api-reference/resource/deployment/">Deployment</a> and a "prod" <a href="https://docs.aws.amazon.com/apigateway/api-reference/resource/stage/">Stage</a> which represent the API configuration you
 * defined in your CDK app. This means that when you deploy your app, your API will
 * be have open access from the internet via the stage URL.
 * <p>
 * The URL of your API can be obtained from the attribute <code>restApi.url</code>, and is
 * also exported as an <code>Output</code> from your stack, so it's printed when you <code>cdk deploy</code> your app:
 * <p>
 * <blockquote><pre>
 * $ cdk deploy
 * ...
 * books.booksapiEndpointE230E8D5 = https://6lyktd4lpk.execute-api.us-east-1.amazonaws.com/prod/
 * </pre></blockquote>
 * <p>
 * To disable this behavior, you can set <code>{ deploy: false }</code> when creating your
 * API. This means that the API will not be deployed and a stage will not be
 * created for it. You will need to manually define a <code>apigateway.Deployment</code> and
 * <code>apigateway.Stage</code> resources.
 * <p>
 * Use the <code>deployOptions</code> property to customize the deployment options of your
 * API.
 * <p>
 * The following example will configure API Gateway to emit logs and data traces to
 * AWS CloudWatch for all API calls:
 * <p>
 * <blockquote>
 * <p>
 * By default, an IAM role will be created and associated with API Gateway to
 * allow it to write logs and metrics to AWS CloudWatch unless <code>cloudWatchRole</code> is
 * set to <code>false</code>.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var api = RestApi.Builder.create(this, "books")
 *         .deployOptions(Map.of(
 *                 "loggingLevel", apigateway.MethodLoggingLevel.getINFO(),
 *                 "dataTraceEnabled", true))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deep dive: Invalidation of deployments</h3>
 * <p>
 * API Gateway deployments are an immutable snapshot of the API. This means that we
 * want to automatically create a new deployment resource every time the API model
 * defined in our CDK app changes.
 * <p>
 * In order to achieve that, the AWS CloudFormation logical ID of the
 * <code>AWS::ApiGateway::Deployment</code> resource is dynamically calculated by hashing the
 * API configuration (resources, methods). This means that when the configuration
 * changes (i.e. a resource or method are added, configuration is changed), a new
 * logical ID will be assigned to the deployment resource. This will cause
 * CloudFormation to create a new deployment resource.
 * <p>
 * By default, old deployments are <em>deleted</em>. You can set <code>retainDeployments: true</code>
 * to allow users revert the stage to an old deployment manually.
 * <p>
 * <h2>Custom Domains</h2>
 * <p>
 * To associate an API with a custom domain, use the <code>domainName</code> configuration when
 * you define your API:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var api = RestApi.Builder.create(this, "MyDomain")
 *         .domainName(Map.of(
 *                 "domainName", "example.com",
 *                 "certificate", acmCertificateForExampleCom))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will define a <code>DomainName</code> resource for you, along with a <code>BasePathMapping</code>
 * from the root of the domain to the deployment stage of the API. This is a common
 * set up.
 * <p>
 * To route domain traffic to an API Gateway API, use Amazon Route 53 to create an
 * alias record. An alias record is a Route 53 extension to DNS. It's similar to a
 * CNAME record, but you can create an alias record both for the root domain, such
 * as <code>example.com</code>, and for subdomains, such as <code>www.example.com</code>. (You can create
 * CNAME records only for subdomains.)
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.route53.*;
 * import software.amazon.awscdk.services.route53.targets.*;
 * 
 * new ARecord(this, "CustomDomainAliasRecord", new ARecordProps()
 *         .zone(hostedZoneForExampleCom)
 *         .target(route53.RecordTarget.fromAlias(new ApiGateway(api))));
 * </pre></blockquote>
 * <p>
 * You can also define a <code>DomainName</code> resource directly in order to customize the default behavior:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * DomainName.Builder.create(this, "custom-domain")
 *         .domainName("example.com")
 *         .certificate(acmCertificateForExampleCom)
 *         .endpointType(apigw.EndpointType.getEDGE())// default is REGIONAL
 *         .securityPolicy(apigw.SecurityPolicy.getTLS_1_2())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Once you have a domain, you can map base paths of the domain to APIs.
 * The following example will map the URL <a href="https://example.com/go-to-api1">https://example.com/go-to-api1</a>
 * to the <code>api1</code> API and <a href="https://example.com/boom">https://example.com/boom</a> to the <code>api2</code> API.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * domain.addBasePathMapping(api1, Map.of("basePath", "go-to-api1"));
 * domain.addBasePathMapping(api2, Map.of("basePath", "boom"));
 * </pre></blockquote>
 * <p>
 * You can specify the API <code>Stage</code> to which this base path URL will map to. By default, this will be the
 * <code>deploymentStage</code> of the <code>RestApi</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var betaDeploy = Deployment.Builder.create(this, "beta-deployment")
 *         .api(restapi)
 *         .build();
 * var betaStage = Stage.Builder.create(this, "beta-stage")
 *         .deployment(betaDeploy)
 *         .build();
 * domain.addBasePathMapping(restapi, Map.of("basePath", "api/beta", "stage", betaStage));
 * </pre></blockquote>
 * <p>
 * If you don't specify <code>basePath</code>, all URLs under this domain will be mapped
 * to the API, and you won't be able to map another API to the same domain:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * domain.addBasePathMapping(api);
 * </pre></blockquote>
 * <p>
 * This can also be achieved through the <code>mapping</code> configuration when defining the
 * domain as demonstrated above.
 * <p>
 * If you wish to setup this domain with an Amazon Route53 alias, use the <code>targets.ApiGatewayDomain</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.route53.*;
 * import software.amazon.awscdk.services.route53.targets.*;
 * 
 * new ARecord(this, "CustomDomainAliasRecord", new ARecordProps()
 *         .zone(hostedZoneForExampleCom)
 *         .target(route53.RecordTarget.fromAlias(new ApiGatewayDomain(domainName))));
 * </pre></blockquote>
 * <p>
 * <h2>Access Logging</h2>
 * <p>
 * Access logging creates logs everytime an API method is accessed. Access logs can have information on
 * who has accessed the API, how the caller accessed the API and what responses were generated.
 * Access logs are configured on a Stage of the RestApi.
 * Access logs can be expressed in a format of your choosing, and can contain any access details, with a
 * minimum that it must include the 'requestId'. The list of  variables that can be expressed in the access
 * log can be found
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference">here</a>.
 * Read more at <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/set-up-logging.html">Setting Up CloudWatch API Logging in API
 * Gateway</a>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // production stage
 * var prdLogGroup = new LogGroup(this, "PrdLogs");
 * var api = RestApi.Builder.create(this, "books")
 *         .deployOptions(Map.of(
 *                 "accessLogDestination", new LogGroupLogDestination(prdLogGroup),
 *                 "accessLogFormat", apigateway.AccessLogFormat.jsonWithStandardFields()))
 *         .build();
 * var deployment = Deployment.Builder.create(stack, "Deployment").api(api).build();
 * 
 * // development stage
 * var devLogGroup = new LogGroup(this, "DevLogs");
 * Stage.Builder.create(this, "dev")
 *         .deployment(deployment)
 *         .accessLogDestination(new LogGroupLogDestination(devLogGroup))
 *         .accessLogFormat(apigateway.AccessLogFormat.jsonWithStandardFields(Map.of(
 *                 "caller", false,
 *                 "httpMethod", true,
 *                 "ip", true,
 *                 "protocol", true,
 *                 "requestTime", true,
 *                 "resourcePath", true,
 *                 "responseLength", true,
 *                 "status", true,
 *                 "user", true)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following code will generate the access log in the <a href="https://en.wikipedia.org/wiki/Common_Log_Format">CLF format</a>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var logGroup = new LogGroup(this, "ApiGatewayAccessLogs");
 * var api = RestApi.Builder.create(this, "books")
 *         .deployOptions(Map.of(
 *                 "accessLogDestination", new LogGroupLogDestination(logGroup),
 *                 "accessLogFormat", apigateway.AccessLogFormat.clf()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also configure your own access log format by using the <code>AccessLogFormat.custom()</code> API.
 * <code>AccessLogField</code> provides commonly used fields. The following code configures access log to contain.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var logGroup = new LogGroup(this, "ApiGatewayAccessLogs");
 * RestApi.Builder.create(this, "books")
 *         .deployOptions(Map.of(
 *                 "accessLogDestination", new LogGroupLogDestination(logGroup),
 *                 "accessLogFormat", apigateway.AccessLogFormat.custom(AccessLogFormat.contextRequestId() + " " + AccessLogField.contextErrorMessage() + " " + AccessLogField.contextErrorMessageString())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can use the <code>methodOptions</code> property to configure
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html#apigateway-api-level-throttling-in-usage-plan">default method throttling</a>
 * for a stage. The following snippet configures the a stage that accepts
 * 100 requests per minute, allowing burst up to 200 requests per minute.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var api = new RestApi(this, "books");
 * var deployment = Deployment.Builder.create(this, "my-deployment").api(api).build();
 * var stage = Stage.Builder.create(this, "my-stage")
 *         .deployment(deployment)
 *         .methodOptions(Map.of(
 *                 "/*{@literal /}*", Map.of(// This special path applies to all resource paths and all HTTP methods
 *                         "throttlingRateLimit", 100,
 *                         "throttlingBurstLimit", 200)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Configuring <code>methodOptions</code> on the <code>deployOptions</code> of <code>RestApi</code> will set the
 * throttling behaviors on the default stage that is automatically created.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var api = RestApi.Builder.create(this, "books")
 *         .deployOptions(Map.of(
 *                 "methodOptions", Map.of(
 *                         "/*{@literal /}*", Map.of(// This special path applies to all resource paths and all HTTP methods
 *                                 "throttlingRateLimit", 100,
 *                                 "throttlingBurstLimit", 1000))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Cross Origin Resource Sharing (CORS)</h2>
 * <p>
 * <a href="https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS">Cross-Origin Resource Sharing (CORS)</a> is a mechanism
 * that uses additional HTTP headers to tell browsers to give a web application
 * running at one origin, access to selected resources from a different origin. A
 * web application executes a cross-origin HTTP request when it requests a resource
 * that has a different origin (domain, protocol, or port) from its own.
 * <p>
 * You can add the CORS <a href="https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS#Preflighted_requests">preflight</a> OPTIONS
 * HTTP method to any API resource via the <code>defaultCorsPreflightOptions</code> option or by calling the <code>addCorsPreflight</code> on a specific resource.
 * <p>
 * The following example will enable CORS for all methods and all origins on all resources of the API:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * RestApi.Builder.create(this, "api")
 *         .defaultCorsPreflightOptions(Map.of(
 *                 "allowOrigins", apigateway.Cors.getALL_ORIGINS(),
 *                 "allowMethods", apigateway.Cors.getALL_METHODS()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following example will add an OPTIONS method to the <code>myResource</code> API resource, which
 * only allows GET and PUT HTTP requests from the origin <a href="https://amazon.com.">https://amazon.com.</a>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * myResource.addCorsPreflight(Map.of(
 *         "allowOrigins", asList("https://amazon.com"),
 *         "allowMethods", asList("GET", "PUT")));
 * </pre></blockquote>
 * <p>
 * See the
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-apigateway.CorsOptions.html"><code>CorsOptions</code></a>
 * API reference for a detailed list of supported configuration options.
 * <p>
 * You can specify defaults this at the resource level, in which case they will be applied to the entire resource sub-tree:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var subtree = resource.addResource("subtree", Map.of(
 *         "defaultCorsPreflightOptions", Map.of(
 *                 "allowOrigins", asList("https://amazon.com"))));
 * </pre></blockquote>
 * <p>
 * This means that all resources under <code>subtree</code> (inclusive) will have a preflight
 * OPTIONS added to them.
 * <p>
 * See <a href="https://github.com/aws/aws-cdk/issues/906">#906</a> for a list of CORS
 * features which are not yet supported.
 * <p>
 * <h2>Endpoint Configuration</h2>
 * <p>
 * API gateway allows you to specify an
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-endpoint-types.html">API Endpoint Type</a>.
 * To define an endpoint type for the API gateway, use <code>endpointConfiguration</code> property:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var api = RestApi.Builder.create(stack, "api")
 *         .endpointConfiguration(Map.of(
 *                 "types", asList(apigw.EndpointType.getEDGE())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also create an association between your Rest API and a VPC endpoint. By doing so,
 * API Gateway will generate a new
 * Route53 Alias DNS record which you can use to invoke your private APIs. More info can be found
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/associate-private-api-with-vpc-endpoint.html">here</a>.
 * <p>
 * Here is an example:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var someEndpoint = ;/* Get or Create endpoint here *{@literal /}
 * var api = RestApi.Builder.create(stack, "api")
 *         .endpointConfiguration(Map.of(
 *                 "types", asList(apigw.EndpointType.getPRIVATE()),
 *                 "vpcEndpoints", asList(someEndpoint)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * By performing this association, we can invoke the API gateway using the following format:
 * <p>
 * <blockquote><pre>
 * https://{rest-api-id}-{vpce-id}.execute-api.{region}.amazonaws.com/{stage}
 * </pre></blockquote>
 * <p>
 * <h2>Private Integrations</h2>
 * <p>
 * A private integration makes it simple to expose HTTP/HTTPS resources behind an
 * Amazon VPC for access by clients outside of the VPC. The private integration uses
 * an API Gateway resource of <code>VpcLink</code> to encapsulate connections between API
 * Gateway and targeted VPC resources.
 * The <code>VpcLink</code> is then attached to the <code>Integration</code> of a specific API Gateway
 * Method. The following code sets up a private integration with a network load
 * balancer -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var vpc = new Vpc(stack, "VPC");
 * var nlb = NetworkLoadBalancer.Builder.create(stack, "NLB")
 *         .vpc(vpc)
 *         .build();
 * var link = VpcLink.Builder.create(stack, "link")
 *         .targets(asList(nlb))
 *         .build();
 * 
 * var integration = Integration.Builder.create()
 *         .type(apigw.IntegrationType.getHTTP_PROXY())
 *         .options(Map.of(
 *                 "connectionType", apigw.ConnectionType.getVPC_LINK(),
 *                 "vpcLink", link))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Any existing <code>VpcLink</code> resource can be imported into the CDK app via the <code>VpcLink.fromVpcLinkId()</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var stack = new Stack(app, "my-stack");
 * 
 * var awesomeLink = VpcLink.fromVpcLinkId(stack, "awesome-vpc-link", "us-east-1_oiuR12Abd");
 * </pre></blockquote>
 * <p>
 * <h2>Gateway response</h2>
 * <p>
 * If the Rest API fails to process an incoming request, it returns to the client an error response without forwarding the
 * request to the integration backend. API Gateway has a set of standard response messages that are sent to the client for
 * each type of error. These error responses can be configured on the Rest API. The list of Gateway responses that can be
 * configured can be found <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html">here</a>.
 * Learn more about <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-gatewayResponse-definition.html">Gateway
 * Responses</a>.
 * <p>
 * The following code configures a Gateway Response when the response is 'access denied':
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var api = new RestApi(this, "books-api");
 * api.addGatewayResponse("test-response", Map.of(
 *         "type", ResponseType.getACCESS_DENIED(),
 *         "statusCode", "500",
 *         "responseHeaders", Map.of(
 *                 "Access-Control-Allow-Origin", "test.com",
 *                 "test-key", "test-value"),
 *         "templates", Map.of(
 *                 "application/json", "{ \"message\": $context.error.messageString, \"statusCode\": \"488\", \"type\": \"$context.error.responseType\" }")));
 * </pre></blockquote>
 * <p>
 * <h2>OpenAPI Definition</h2>
 * <p>
 * CDK supports creating a REST API by importing an OpenAPI definition file. It currently supports OpenAPI v2.0 and OpenAPI
 * v3.0 definition files. Read more about <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-import-api.html">Configuring a REST API using
 * OpenAPI</a>.
 * <p>
 * The following code creates a REST API using an external OpenAPI definition JSON file -
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * var api = SpecRestApi.Builder.create(this, "books-api")
 *         .apiDefinition(apigateway.ApiDefinition.fromAsset("path-to-file.json"))
 *         .build();
 * 
 * var booksResource = api.root.addResource("books");
 * booksResource.addMethod("GET", ...);
 * </pre></blockquote>
 * <p>
 * It is possible to use the <code>addResource()</code> API to define additional API Gateway Resources.
 * <p>
 * <strong>Note:</strong> Deployment will fail if a Resource of the same name is already defined in the Open API specification.
 * <p>
 * <strong>Note:</strong> Any default properties configured, such as <code>defaultIntegration</code>, <code>defaultMethodOptions</code>, etc. will only be
 * applied to Resources and Methods defined in the CDK, and not the ones defined in the spec. Use the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions.html">API Gateway
 * extensions to OpenAPI</a>
 * to configure these.
 * <p>
 * There are a number of limitations in using OpenAPI definitions in API Gateway. Read the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-known-issues.html#api-gateway-known-issues-rest-apis">Amazon API Gateway important
 * notes for REST APIs</a>
 * for more details.
 * <p>
 * <strong>Note:</strong> When starting off with an OpenAPI definition using <code>SpecRestApi</code>, it is not possible to configure some
 * properties that can be configured directly in the OpenAPI specification file. This is to prevent people duplication
 * of these properties and potential confusion.
 * <p>
 * <h2>APIGateway v2</h2>
 * <p>
 * APIGateway v2 APIs are now moved to its own package named <code>aws-apigatewayv2</code>. For backwards compatibility, existing
 * APIGateway v2 "CFN resources" (such as <code>CfnApi</code>) that were previously exported as part of this package, are still
 * exported from here and have been marked deprecated. However, updates to these CloudFormation resources, such as new
 * properties and new resource types will not be available.
 * <p>
 * Move to using <code>aws-apigatewayv2</code> to get the latest APIs and updates.
 * <p>
 * <hr>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.apigateway;
