/**
 * <h1>AWS::APIGatewayv2 Construct Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * Features                                   | Stability
 * -------------------------------------------|--------------------------------------------------------
 * Higher level constructs for HTTP APIs      | <img alt="Experimental" src="https://img.shields.io/badge/experimental-important.svg?style=for-the-badge">
 * Higher level constructs for Websocket APIs | <img alt="Experimental" src="https://img.shields.io/badge/experimental-important.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * <strong>Experimental:</strong> Higher level constructs in this module that are marked as experimental are
 * under active development. They are subject to non-backward compatible changes or removal in any
 * future version. These are not subject to the <a href="https://semver.org/">Semantic Versioning</a> model and
 * breaking changes will be announced in the release notes. This means that while you may use them,
 * you may need to update your source code when upgrading to a newer version of this package.
 * <p>
 * </blockquote>
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * <h2>Table of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#introduction">Introduction</a></li>
 * <li><a href="#http-api">HTTP API</a>
 * <p>
 * <ul>
 * <li><a href="#defining-http-apis">Defining HTTP APIs</a></li>
 * <li><a href="#cross-origin-resource-sharing-cors">Cross Origin Resource Sharing (CORS)</a></li>
 * <li><a href="#publishing-http-apis">Publishing HTTP APIs</a></li>
 * <li><a href="#custom-domain">Custom Domain</a></li>
 * <li><a href="#mutual-tls-mtls">Mutual TLS</a></li>
 * <li><a href="#managing-access-to-http-apis">Managing access to HTTP APIs</a></li>
 * <li><a href="#metrics">Metrics</a></li>
 * <li><a href="#vpc-link">VPC Link</a></li>
 * <li><a href="#private-integration">Private Integration</a></li>
 * </ul></li>
 * <li><a href="#websocket-api">WebSocket API</a>
 * <p>
 * <ul>
 * <li><a href="#manage-connections-permission">Manage Connections Permission</a></li>
 * <li><a href="#managing-access-to-websocket-apis">Managing access to WebSocket APIs</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>Introduction</h2>
 * <p>
 * Amazon API Gateway is an AWS service for creating, publishing, maintaining, monitoring, and securing REST, HTTP, and WebSocket
 * APIs at any scale. API developers can create APIs that access AWS or other web services, as well as data stored in the AWS Cloud.
 * As an API Gateway API developer, you can create APIs for use in your own client applications. Read the
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/welcome.html">Amazon API Gateway Developer Guide</a>.
 * <p>
 * This module supports features under <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/AWS_ApiGatewayV2.html">API Gateway v2</a>
 * that lets users set up Websocket and HTTP APIs.
 * REST APIs can be created using the <code>&#64;aws-cdk/aws-apigateway</code> module.
 * <p>
 * <h2>HTTP API</h2>
 * <p>
 * HTTP APIs enable creation of RESTful APIs that integrate with AWS Lambda functions, known as Lambda proxy integration,
 * or to any routable HTTP endpoint, known as HTTP proxy integration.
 * <p>
 * <h3>Defining HTTP APIs</h3>
 * <p>
 * HTTP APIs have two fundamental concepts - Routes and Integrations.
 * <p>
 * Routes direct incoming API requests to backend resources. Routes consist of two parts: an HTTP method and a resource
 * path, such as, <code>GET /books</code>. Learn more at <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-develop-routes.html">Working with
 * routes</a>. Use the <code>ANY</code> method
 * to match any methods for a route that are not explicitly defined.
 * <p>
 * Integrations define how the HTTP API responds when a client reaches a specific Route. HTTP APIs support Lambda proxy
 * integration, HTTP proxy integration and, AWS service integrations, also known as private integrations. Learn more at
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-develop-integrations.html">Configuring integrations</a>.
 * <p>
 * Integrations are available at the <code>aws-apigatewayv2-integrations</code> module and more information is available in that module.
 * As an early example, the following code snippet configures a route <code>GET /books</code> with an HTTP proxy integration all
 * configures all other HTTP method calls to <code>/books</code> to a lambda proxy.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.integrations.alpha.HttpUrlIntegration;
 * import software.amazon.awscdk.services.apigatewayv2.integrations.alpha.HttpLambdaIntegration;
 * 
 * Function booksDefaultFn;
 * 
 * 
 * HttpUrlIntegration getBooksIntegration = new HttpUrlIntegration("GetBooksIntegration", "https://get-books-proxy.myproxy.internal");
 * HttpLambdaIntegration booksDefaultIntegration = new HttpLambdaIntegration("BooksIntegration", booksDefaultFn);
 * 
 * HttpApi httpApi = new HttpApi(this, "HttpApi");
 * 
 * httpApi.addRoutes(AddRoutesOptions.builder()
 *         .path("/books")
 *         .methods(List.of(HttpMethod.GET))
 *         .integration(getBooksIntegration)
 *         .build());
 * httpApi.addRoutes(AddRoutesOptions.builder()
 *         .path("/books")
 *         .methods(List.of(HttpMethod.ANY))
 *         .integration(booksDefaultIntegration)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The URL to the endpoint can be retrieved via the <code>apiEndpoint</code> attribute. By default this URL is enabled for clients. Use <code>disableExecuteApiEndpoint</code> to disable it.
 * <p>
 * <blockquote><pre>
 * HttpApi httpApi = HttpApi.Builder.create(this, "HttpApi")
 *         .disableExecuteApiEndpoint(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>defaultIntegration</code> option while defining HTTP APIs lets you create a default catch-all integration that is
 * matched when a client reaches a route that is not explicitly defined.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.integrations.alpha.HttpUrlIntegration;
 * 
 * 
 * HttpApi.Builder.create(this, "HttpProxyApi")
 *         .defaultIntegration(new HttpUrlIntegration("DefaultIntegration", "https://example.com"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cross Origin Resource Sharing (CORS)</h3>
 * <p>
 * <a href="https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS">Cross-origin resource sharing (CORS)</a> is a browser security
 * feature that restricts HTTP requests that are initiated from scripts running in the browser. Enabling CORS will allow
 * requests to your API from a web application hosted in a domain different from your API domain.
 * <p>
 * When configured CORS for an HTTP API, API Gateway automatically sends a response to preflight <code>OPTIONS</code> requests, even
 * if there isn't an <code>OPTIONS</code> route configured. Note that, when this option is used, API Gateway will ignore CORS headers
 * returned from your backend integration. Learn more about <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-cors.html">Configuring CORS for an HTTP
 * API</a>.
 * <p>
 * The <code>corsPreflight</code> option lets you specify a CORS configuration for an API.
 * <p>
 * <blockquote><pre>
 * HttpApi.Builder.create(this, "HttpProxyApi")
 *         .corsPreflight(CorsPreflightOptions.builder()
 *                 .allowHeaders(List.of("Authorization"))
 *                 .allowMethods(List.of(CorsHttpMethod.GET, CorsHttpMethod.HEAD, CorsHttpMethod.OPTIONS, CorsHttpMethod.POST))
 *                 .allowOrigins(List.of("*"))
 *                 .maxAge(Duration.days(10))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Publishing HTTP APIs</h3>
 * <p>
 * A Stage is a logical reference to a lifecycle state of your API (for example, <code>dev</code>, <code>prod</code>, <code>beta</code>, or <code>v2</code>). API
 * stages are identified by their stage name. Each stage is a named reference to a deployment of the API made available for
 * client applications to call.
 * <p>
 * Use <code>HttpStage</code> to create a Stage resource for HTTP APIs. The following code sets up a Stage, whose URL is available at
 * <code>https://{api_id}.execute-api.{region}.amazonaws.com/beta</code>.
 * <p>
 * <blockquote><pre>
 * HttpApi api;
 * 
 * 
 * HttpStage.Builder.create(this, "Stage")
 *         .httpApi(api)
 *         .stageName("beta")
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you omit the <code>stageName</code> will create a <code>$default</code> stage. A <code>$default</code> stage is one that is served from the base of
 * the API's URL - <code>https://{api_id}.execute-api.{region}.amazonaws.com/</code>.
 * <p>
 * Note that, <code>HttpApi</code> will always creates a <code>$default</code> stage, unless the <code>createDefaultStage</code> property is unset.
 * <p>
 * <h3>Custom Domain</h3>
 * <p>
 * Custom domain names are simpler and more intuitive URLs that you can provide to your API users. Custom domain name are associated to API stages.
 * <p>
 * The code snippet below creates a custom domain and configures a default domain mapping for your API that maps the
 * custom domain to the <code>$default</code> stage of the API.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.*;
 * import software.amazon.awscdk.services.apigatewayv2.integrations.alpha.HttpLambdaIntegration;
 * 
 * Function handler;
 * 
 * 
 * String certArn = "arn:aws:acm:us-east-1:111111111111:certificate";
 * String domainName = "example.com";
 * 
 * DomainName dn = DomainName.Builder.create(this, "DN")
 *         .domainName(domainName)
 *         .certificate(Certificate.fromCertificateArn(this, "cert", certArn))
 *         .build();
 * HttpApi api = HttpApi.Builder.create(this, "HttpProxyProdApi")
 *         .defaultIntegration(new HttpLambdaIntegration("DefaultIntegration", handler))
 *         // https://${dn.domainName}/foo goes to prodApi $default stage
 *         .defaultDomainMapping(DomainMappingOptions.builder()
 *                 .domainName(dn)
 *                 .mappingKey("foo")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * To migrate a domain endpoint from one type to another, you can add a new endpoint configuration via <code>addEndpoint()</code>
 * and then configure DNS records to route traffic to the new endpoint. After that, you can remove the previous endpoint configuration.
 * Learn more at <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-regional-api-custom-domain-migrate.html">Migrating a custom domain name</a>
 * <p>
 * To associate a specific <code>Stage</code> to a custom domain mapping -
 * <p>
 * <blockquote><pre>
 * HttpApi api;
 * DomainName dn;
 * 
 * 
 * api.addStage("beta", HttpStageOptions.builder()
 *         .stageName("beta")
 *         .autoDeploy(true)
 *         // https://${dn.domainName}/bar goes to the beta stage
 *         .domainMapping(DomainMappingOptions.builder()
 *                 .domainName(dn)
 *                 .mappingKey("bar")
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * The same domain name can be associated with stages across different <code>HttpApi</code> as so -
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.integrations.alpha.HttpLambdaIntegration;
 * 
 * Function handler;
 * DomainName dn;
 * 
 * 
 * HttpApi apiDemo = HttpApi.Builder.create(this, "DemoApi")
 *         .defaultIntegration(new HttpLambdaIntegration("DefaultIntegration", handler))
 *         // https://${dn.domainName}/demo goes to apiDemo $default stage
 *         .defaultDomainMapping(DomainMappingOptions.builder()
 *                 .domainName(dn)
 *                 .mappingKey("demo")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>mappingKey</code> determines the base path of the URL with the custom domain. Each custom domain is only allowed
 * to have one API mapping with undefined <code>mappingKey</code>. If more than one API mappings are specified, <code>mappingKey</code> will be required for all of them. In the sample above, the custom domain is associated
 * with 3 API mapping resources across different APIs and Stages.
 * <p>
 * |        API     |     Stage   |   URL  |
 * | :------------: | :---------: | :----: |
 * | api | $default  |   <code>https://${domainName}/foo</code>  |
 * | api | beta  |   <code>https://${domainName}/bar</code>  |
 * | apiDemo | $default  |   <code>https://${domainName}/demo</code>  |
 * <p>
 * You can retrieve the full domain URL with mapping key using the <code>domainUrl</code> property as so -
 * <p>
 * <blockquote><pre>
 * HttpApi apiDemo;
 * 
 * String demoDomainUrl = apiDemo.getDefaultStage().getDomainUrl();
 * </pre></blockquote>
 * <p>
 * <h3>Mutual TLS (mTLS)</h3>
 * <p>
 * Mutual TLS can be configured to limit access to your API based by using client certificates instead of (or as an extension of) using authorization headers.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.services.certificatemanager.*;
 * Bucket bucket;
 * 
 * 
 * String certArn = "arn:aws:acm:us-east-1:111111111111:certificate";
 * String domainName = "example.com";
 * 
 * DomainName.Builder.create(this, "DomainName")
 *         .domainName(domainName)
 *         .certificate(Certificate.fromCertificateArn(this, "cert", certArn))
 *         .mtls(MTLSConfig.builder()
 *                 .bucket(bucket)
 *                 .key("someca.pem")
 *                 .version("version")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Instructions for configuring your trust store can be found <a href="https://aws.amazon.com/blogs/compute/introducing-mutual-tls-authentication-for-amazon-api-gateway/">here</a>
 * <p>
 * <h3>Managing access to HTTP APIs</h3>
 * <p>
 * API Gateway supports multiple mechanisms for <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-access-control.html">controlling and managing access to your HTTP
 * API</a> through authorizers.
 * <p>
 * These authorizers can be found in the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-apigatewayv2-authorizers-readme.html">APIGatewayV2-Authorizers</a> constructs library.
 * <p>
 * <h3>Metrics</h3>
 * <p>
 * The API Gateway v2 service sends metrics around the performance of HTTP APIs to Amazon CloudWatch.
 * These metrics can be referred to using the metric APIs available on the <code>HttpApi</code> construct.
 * The APIs with the <code>metric</code> prefix can be used to get reference to specific metrics for this API. For example,
 * the method below refers to the client side errors metric for this API.
 * <p>
 * <blockquote><pre>
 * HttpApi api = new HttpApi(this, "my-api");
 * Metric clientErrorMetric = api.metricClientError();
 * </pre></blockquote>
 * <p>
 * Please note that this will return a metric for all the stages defined in the api. It is also possible to refer to metrics for a specific Stage using
 * the <code>metric</code> methods from the <code>Stage</code> construct.
 * <p>
 * <blockquote><pre>
 * HttpApi api = new HttpApi(this, "my-api");
 * HttpStage stage = HttpStage.Builder.create(this, "Stage")
 *         .httpApi(api)
 *         .build();
 * Metric clientErrorMetric = stage.metricClientError();
 * </pre></blockquote>
 * <p>
 * <h3>VPC Link</h3>
 * <p>
 * Private integrations let HTTP APIs connect with AWS resources that are placed behind a VPC. These are usually Application
 * Load Balancers, Network Load Balancers or a Cloud Map service. The <code>VpcLink</code> construct enables this integration.
 * The following code creates a <code>VpcLink</code> to a private VPC.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * 
 * 
 * Vpc vpc = new Vpc(this, "VPC");
 * VpcLink vpcLink = VpcLink.Builder.create(this, "VpcLink").vpc(vpc).build();
 * </pre></blockquote>
 * <p>
 * Any existing <code>VpcLink</code> resource can be imported into the CDK app via the <code>VpcLink.fromVpcLinkAttributes()</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * 
 * Vpc vpc;
 * 
 * IVpcLink awesomeLink = VpcLink.fromVpcLinkAttributes(this, "awesome-vpc-link", VpcLinkAttributes.builder()
 *         .vpcLinkId("us-east-1_oiuR12Abd")
 *         .vpc(vpc)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Private Integration</h3>
 * <p>
 * Private integrations enable integrating an HTTP API route with private resources in a VPC, such as Application Load Balancers or
 * Amazon ECS container-based applications.  Using private integrations, resources in a VPC can be exposed for access by
 * clients outside of the VPC.
 * <p>
 * These integrations can be found in the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-apigatewayv2-integrations-readme.html">aws-apigatewayv2-integrations</a> constructs library.
 * <p>
 * <h2>WebSocket API</h2>
 * <p>
 * A WebSocket API in API Gateway is a collection of WebSocket routes that are integrated with backend HTTP endpoints,
 * Lambda functions, or other AWS services. You can use API Gateway features to help you with all aspects of the API
 * lifecycle, from creation through monitoring your production APIs. <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-websocket-api-overview.html">Read more</a>
 * <p>
 * WebSocket APIs have two fundamental concepts - Routes and Integrations.
 * <p>
 * WebSocket APIs direct JSON messages to backend integrations based on configured routes. (Non-JSON messages are directed
 * to the configured <code>$default</code> route.)
 * <p>
 * Integrations define how the WebSocket API behaves when a client reaches a specific Route. Learn more at
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-websocket-api-integration-requests.html">Configuring integrations</a>.
 * <p>
 * Integrations are available in the <code>aws-apigatewayv2-integrations</code> module and more information is available in that module.
 * <p>
 * To add the default WebSocket routes supported by API Gateway (<code>$connect</code>, <code>$disconnect</code> and <code>$default</code>), configure them as part of api props:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.integrations.alpha.WebSocketLambdaIntegration;
 * 
 * Function connectHandler;
 * Function disconnectHandler;
 * Function defaultHandler;
 * 
 * 
 * WebSocketApi webSocketApi = WebSocketApi.Builder.create(this, "mywsapi")
 *         .connectRouteOptions(WebSocketRouteOptions.builder().integration(new WebSocketLambdaIntegration("ConnectIntegration", connectHandler)).build())
 *         .disconnectRouteOptions(WebSocketRouteOptions.builder().integration(new WebSocketLambdaIntegration("DisconnectIntegration", disconnectHandler)).build())
 *         .defaultRouteOptions(WebSocketRouteOptions.builder().integration(new WebSocketLambdaIntegration("DefaultIntegration", defaultHandler)).build())
 *         .build();
 * 
 * WebSocketStage.Builder.create(this, "mystage")
 *         .webSocketApi(webSocketApi)
 *         .stageName("dev")
 *         .autoDeploy(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To retrieve a websocket URL and a callback URL:
 * <p>
 * <blockquote><pre>
 * WebSocketStage webSocketStage;
 * 
 * 
 * String webSocketURL = webSocketStage.getUrl();
 * // wss://${this.api.apiId}.execute-api.${s.region}.${s.urlSuffix}/${urlPath}
 * String callbackURL = webSocketStage.getCallbackUrl();
 * </pre></blockquote>
 * <p>
 * To add any other route:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.integrations.alpha.WebSocketLambdaIntegration;
 * 
 * Function messageHandler;
 * 
 * WebSocketApi webSocketApi = new WebSocketApi(this, "mywsapi");
 * webSocketApi.addRoute("sendmessage", WebSocketRouteOptions.builder()
 *         .integration(new WebSocketLambdaIntegration("SendMessageIntegration", messageHandler))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To import an existing WebSocketApi:
 * <p>
 * <blockquote><pre>
 * IWebSocketApi webSocketApi = WebSocketApi.fromWebSocketApiAttributes(this, "mywsapi", WebSocketApiAttributes.builder().webSocketId("api-1234").build());
 * </pre></blockquote>
 * <p>
 * <h3>Manage Connections Permission</h3>
 * <p>
 * Grant permission to use API Gateway Management API of a WebSocket API by calling the <code>grantManageConnections</code> API.
 * You can use Management API to send a callback message to a connected client, get connection information, or disconnect the client. Learn more at <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-how-to-call-websocket-api-connections.html">Use &#64;connections commands in your backend service</a>.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * WebSocketApi webSocketApi = new WebSocketApi(this, "mywsapi");
 * WebSocketStage stage = WebSocketStage.Builder.create(this, "mystage")
 *         .webSocketApi(webSocketApi)
 *         .stageName("dev")
 *         .build();
 * // per stage permission
 * stage.grantManagementApiAccess(fn);
 * // for all the stages permission
 * webSocketApi.grantManageConnections(fn);
 * </pre></blockquote>
 * <p>
 * <h3>Managing access to WebSocket APIs</h3>
 * <p>
 * API Gateway supports multiple mechanisms for <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-websocket-api-control-access.html">controlling and managing access to a WebSocket API</a> through authorizers.
 * <p>
 * These authorizers can be found in the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-apigatewayv2-authorizers-readme.html">APIGatewayV2-Authorizers</a> constructs library.
 * <p>
 * <h3>API Keys</h3>
 * <p>
 * Websocket APIs also support usage of API Keys. An API Key is a key that is used to grant access to an API. These are useful for controlling and tracking access to an API, when used together with <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html">usage plans</a>. These together allow you to configure controls around API access such as quotas and throttling, along with per-API Key metrics on usage.
 * <p>
 * To require an API Key when accessing the Websocket API:
 * <p>
 * <blockquote><pre>
 * WebSocketApi webSocketApi = WebSocketApi.Builder.create(this, "mywsapi")
 *         .apiKeySelectionExpression(WebSocketApiKeySelectionExpression.HEADER_X_API_KEY)
 *         .build();
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.apigatewayv2.alpha;
