/**
 * <h1>AWS APIGatewayv2 Integrations</h1>
 * <p>
 * <h2>Table of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#http-apis">HTTP APIs</a>
 * <p>
 * <ul>
 * <li><a href="#lambda">Lambda Integration</a></li>
 * <li><a href="#http-proxy">HTTP Proxy Integration</a></li>
 * <li><a href="#private-integration">Private Integration</a></li>
 * <li><a href="#request-parameters">Request Parameters</a></li>
 * </ul></li>
 * <li><a href="#websocket-apis">WebSocket APIs</a>
 * <p>
 * <ul>
 * <li><a href="#lambda-websocket-integration">Lambda WebSocket Integration</a></li>
 * <li><a href="#aws-websocket-integration">AWS WebSocket Integration</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>HTTP APIs</h2>
 * <p>
 * Integrations connect a route to backend resources. HTTP APIs support Lambda proxy, AWS service, and HTTP proxy integrations. HTTP proxy integrations are also known as private integrations.
 * <p>
 * <h3>Lambda</h3>
 * <p>
 * Lambda integrations enable integrating an HTTP API route with a Lambda function. When a client invokes the route, the
 * API Gateway service forwards the request to the Lambda function and returns the function's response to the client.
 * <p>
 * The API Gateway service will invoke the Lambda function with an event payload of a specific format. The service expects
 * the function to respond in a specific format. The details on this format are available at <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-develop-integrations-lambda.html">Working with AWS Lambda
 * proxy integrations</a>.
 * <p>
 * The following code configures a route <code>GET /books</code> with a Lambda proxy integration.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.HttpLambdaIntegration;
 * 
 * Function booksDefaultFn;
 * 
 * HttpLambdaIntegration booksIntegration = new HttpLambdaIntegration("BooksIntegration", booksDefaultFn);
 * 
 * HttpApi httpApi = new HttpApi(this, "HttpApi");
 * 
 * httpApi.addRoutes(AddRoutesOptions.builder()
 *         .path("/books")
 *         .methods(List.of(HttpMethod.GET))
 *         .integration(booksIntegration)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>HTTP Proxy</h3>
 * <p>
 * HTTP Proxy integrations enables connecting an HTTP API route to a publicly routable HTTP endpoint. When a client
 * invokes the route, the API Gateway service forwards the entire request and response between the API Gateway endpoint
 * and the integrating HTTP endpoint. More information can be found at <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-develop-integrations-http.html">Working with HTTP proxy
 * integrations</a>.
 * <p>
 * The following code configures a route <code>GET /books</code> with an HTTP proxy integration to an HTTP endpoint
 * <code>get-books-proxy.example.com</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.HttpUrlIntegration;
 * 
 * 
 * HttpUrlIntegration booksIntegration = new HttpUrlIntegration("BooksIntegration", "https://get-books-proxy.example.com");
 * 
 * HttpApi httpApi = new HttpApi(this, "HttpApi");
 * 
 * httpApi.addRoutes(AddRoutesOptions.builder()
 *         .path("/books")
 *         .methods(List.of(HttpMethod.GET))
 *         .integration(booksIntegration)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>StepFunctions Integration</h3>
 * <p>
 * Step Functions integrations enable integrating an HTTP API route with AWS Step Functions.
 * This allows the HTTP API to start state machine executions synchronously or asynchronously, or to stop executions.
 * <p>
 * When a client invokes the route configured with a Step Functions integration, the API Gateway service interacts with the specified state machine according to the integration subtype (e.g., starts a new execution, synchronously starts an execution, or stops an execution) and returns the response to the client.
 * <p>
 * The following code configures a Step Functions integrations:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.HttpStepFunctionsIntegration;
 * import software.amazon.awscdk.services.stepfunctions.*;
 * 
 * StateMachine stateMachine;
 * HttpApi httpApi;
 * 
 * 
 * httpApi.addRoutes(AddRoutesOptions.builder()
 *         .path("/start")
 *         .methods(List.of(HttpMethod.POST))
 *         .integration(HttpStepFunctionsIntegration.Builder.create("StartExecutionIntegration")
 *                 .stateMachine(stateMachine)
 *                 .subtype(HttpIntegrationSubtype.STEPFUNCTIONS_START_EXECUTION)
 *                 .build())
 *         .build());
 * 
 * httpApi.addRoutes(AddRoutesOptions.builder()
 *         .path("/start-sync")
 *         .methods(List.of(HttpMethod.POST))
 *         .integration(HttpStepFunctionsIntegration.Builder.create("StartSyncExecutionIntegration")
 *                 .stateMachine(stateMachine)
 *                 .subtype(HttpIntegrationSubtype.STEPFUNCTIONS_START_SYNC_EXECUTION)
 *                 .build())
 *         .build());
 * 
 * httpApi.addRoutes(AddRoutesOptions.builder()
 *         .path("/stop")
 *         .methods(List.of(HttpMethod.POST))
 *         .integration(HttpStepFunctionsIntegration.Builder.create("StopExecutionIntegration")
 *                 .stateMachine(stateMachine)
 *                 .subtype(HttpIntegrationSubtype.STEPFUNCTIONS_STOP_EXECUTION)
 *                 // For the `STOP_EXECUTION` subtype, it is necessary to specify the `executionArn`.
 *                 .parameterMapping(new ParameterMapping().custom("ExecutionArn", "$request.querystring.executionArn"))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>:
 * <p>
 * <ul>
 * <li>The <code>executionArn</code> parameter is required for the <code>STOP_EXECUTION</code> subtype. It is necessary to specify the <code>executionArn</code> in the <code>parameterMapping</code> property of the <code>HttpStepFunctionsIntegration</code> object.</li>
 * <li><code>START_SYNC_EXECUTION</code> subtype is only supported for EXPRESS type state machine.</li>
 * </ul>
 * <p>
 * <h3>Private Integration</h3>
 * <p>
 * Private integrations enable integrating an HTTP API route with private resources in a VPC, such as Application Load Balancers or
 * Amazon ECS container-based applications.  Using private integrations, resources in a VPC can be exposed for access by
 * clients outside of the VPC.
 * <p>
 * The following integrations are supported for private resources in a VPC.
 * <p>
 * <h4>Application Load Balancer</h4>
 * <p>
 * The following code is a basic application load balancer private integration of HTTP API:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.HttpAlbIntegration;
 * 
 * 
 * Vpc vpc = new Vpc(this, "VPC");
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "lb").vpc(vpc).build();
 * ApplicationListener listener = lb.addListener("listener", BaseApplicationListenerProps.builder().port(80).build());
 * listener.addTargets("target", AddApplicationTargetsProps.builder()
 *         .port(80)
 *         .build());
 * 
 * HttpApi httpEndpoint = HttpApi.Builder.create(this, "HttpProxyPrivateApi")
 *         .defaultIntegration(new HttpAlbIntegration("DefaultIntegration", listener))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When an imported load balancer is used, the <code>vpc</code> option must be specified for <code>HttpAlbIntegration</code>.
 * <p>
 * <h4>Network Load Balancer</h4>
 * <p>
 * The following code is a basic network load balancer private integration of HTTP API:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.HttpNlbIntegration;
 * 
 * 
 * Vpc vpc = new Vpc(this, "VPC");
 * NetworkLoadBalancer lb = NetworkLoadBalancer.Builder.create(this, "lb").vpc(vpc).build();
 * NetworkListener listener = lb.addListener("listener", BaseNetworkListenerProps.builder().port(80).build());
 * listener.addTargets("target", AddNetworkTargetsProps.builder()
 *         .port(80)
 *         .build());
 * 
 * HttpApi httpEndpoint = HttpApi.Builder.create(this, "HttpProxyPrivateApi")
 *         .defaultIntegration(new HttpNlbIntegration("DefaultIntegration", listener))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When an imported load balancer is used, the <code>vpc</code> option must be specified for <code>HttpNlbIntegration</code>.
 * <p>
 * <h4>Cloud Map Service Discovery</h4>
 * <p>
 * The following code is a basic discovery service private integration of HTTP API:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.servicediscovery.*;
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.HttpServiceDiscoveryIntegration;
 * 
 * 
 * Vpc vpc = new Vpc(this, "VPC");
 * VpcLink vpcLink = VpcLink.Builder.create(this, "VpcLink").vpc(vpc).build();
 * PrivateDnsNamespace namespace = PrivateDnsNamespace.Builder.create(this, "Namespace")
 *         .name("boobar.com")
 *         .vpc(vpc)
 *         .build();
 * Service service = namespace.createService("Service");
 * 
 * HttpApi httpEndpoint = HttpApi.Builder.create(this, "HttpProxyPrivateApi")
 *         .defaultIntegration(HttpServiceDiscoveryIntegration.Builder.create("DefaultIntegration", service)
 *                 .vpcLink(vpcLink)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Request Parameters</h3>
 * <p>
 * Request parameter mapping allows API requests from clients to be modified before they reach backend integrations.
 * Parameter mapping can be used to specify modifications to request parameters. See <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-parameter-mapping.html">Transforming API requests and
 * responses</a>.
 * <p>
 * The following example creates a new header - <code>header2</code> - as a copy of <code>header1</code> and removes <code>header1</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.HttpAlbIntegration;
 * 
 * ApplicationLoadBalancer lb;
 * 
 * ApplicationListener listener = lb.addListener("listener", BaseApplicationListenerProps.builder().port(80).build());
 * listener.addTargets("target", AddApplicationTargetsProps.builder()
 *         .port(80)
 *         .build());
 * 
 * HttpApi httpEndpoint = HttpApi.Builder.create(this, "HttpProxyPrivateApi")
 *         .defaultIntegration(HttpAlbIntegration.Builder.create("DefaultIntegration", listener)
 *                 .parameterMapping(new ParameterMapping().appendHeader("header2", MappingValue.requestHeader("header1")).removeHeader("header1"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add mapping keys and values not yet supported by the CDK, use the <code>custom()</code> method:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.HttpAlbIntegration;
 * 
 * ApplicationLoadBalancer lb;
 * 
 * ApplicationListener listener = lb.addListener("listener", BaseApplicationListenerProps.builder().port(80).build());
 * listener.addTargets("target", AddApplicationTargetsProps.builder()
 *         .port(80)
 *         .build());
 * 
 * HttpApi httpEndpoint = HttpApi.Builder.create(this, "HttpProxyPrivateApi")
 *         .defaultIntegration(HttpAlbIntegration.Builder.create("DefaultIntegration", listener)
 *                 .parameterMapping(new ParameterMapping().custom("myKey", "myValue"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>WebSocket APIs</h2>
 * <p>
 * WebSocket integrations connect a route to backend resources. The following integrations are supported in the CDK.
 * <p>
 * <h3>Lambda WebSocket Integration</h3>
 * <p>
 * Lambda integrations enable integrating a WebSocket API route with a Lambda function. When a client connects/disconnects
 * or sends a message specific to a route, the API Gateway service forwards the request to the Lambda function
 * <p>
 * The API Gateway service will invoke the Lambda function with an event payload of a specific format.
 * <p>
 * The following code configures a <code>sendMessage</code> route with a Lambda integration
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.WebSocketLambdaIntegration;
 * 
 * Function messageHandler;
 * 
 * 
 * WebSocketApi webSocketApi = new WebSocketApi(this, "mywsapi");
 * WebSocketStage.Builder.create(this, "mystage")
 *         .webSocketApi(webSocketApi)
 *         .stageName("dev")
 *         .autoDeploy(true)
 *         .build();
 * webSocketApi.addRoute("sendMessage", WebSocketRouteOptions.builder()
 *         .integration(new WebSocketLambdaIntegration("SendMessageIntegration", messageHandler))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>AWS WebSocket Integration</h3>
 * <p>
 * AWS type integrations enable integrating with any supported AWS service. This is only supported for WebSocket APIs. When a client
 * connects/disconnects or sends a message specific to a route, the API Gateway service forwards the request to the specified AWS service.
 * <p>
 * The following code configures a <code>$connect</code> route with a AWS integration that integrates with a dynamodb table. On websocket api connect,
 * it will write new entry to the dynamodb table.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.aws_apigatewayv2_integrations.WebSocketAwsIntegration;
 * import software.amazon.awscdk.services.dynamodb.*;
 * import software.amazon.awscdk.services.iam.*;
 * 
 * Role apiRole;
 * Table table;
 * 
 * 
 * WebSocketApi webSocketApi = new WebSocketApi(this, "mywsapi");
 * WebSocketStage.Builder.create(this, "mystage")
 *         .webSocketApi(webSocketApi)
 *         .stageName("dev")
 *         .autoDeploy(true)
 *         .build();
 * webSocketApi.addRoute("$connect", WebSocketRouteOptions.builder()
 *         .integration(WebSocketAwsIntegration.Builder.create("DynamodbPutItem")
 *                 .integrationUri(String.format("arn:aws:apigateway:%s:dynamodb:action/PutItem", this.region))
 *                 .integrationMethod(HttpMethod.POST)
 *                 .credentialsRole(apiRole)
 *                 .requestTemplates(Map.of(
 *                         "application/json", JSON.stringify(Map.of(
 *                                 "TableName", table.getTableName(),
 *                                 "Item", Map.of(
 *                                         "id", Map.of(
 *                                                 "S", "$context.requestId"))))))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also set additional properties to change the behavior of your integration, such as <code>contentHandling</code>.
 * See <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/websocket-api-develop-binary-media-types.html">Working with binary media types for WebSocket APIs</a>.
 */
package software.amazon.awscdk.aws_apigatewayv2_integrations;
