/**
 * <h1>AWS AppSync Construct Library</h1>
 * <p>
 * The <code>aws-cdk-lib/aws-appsync</code> package contains constructs for building flexible
 * APIs that use GraphQL.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.appsync.*;
 * </pre></blockquote>
 * <p>
 * <h2>Example</h2>
 * <p>
 * <h3>DynamoDB</h3>
 * <p>
 * Example of a GraphQL API with <code>AWS_IAM</code> <a href="#authorization">authorization</a> resolving into a DynamoDb
 * backend data source.
 * <p>
 * GraphQL schema file <code>schema.graphql</code>:
 * <p>
 * <blockquote><pre>
 * type demo {
 *   id: String!
 *   version: String!
 * }
 * type Query {
 *   getDemos: [ demo! ]
 * }
 * input DemoInput {
 *   version: String!
 * }
 * type Mutation {
 *   addDemo(input: DemoInput!): demo
 * }
 * </pre></blockquote>
 * <p>
 * CDK stack file <code>app-stack.ts</code>:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "Api")
 *         .name("demo")
 *         .definition(Definition.fromFile(join(__dirname, "schema.graphql")))
 *         .authorizationConfig(AuthorizationConfig.builder()
 *                 .defaultAuthorization(AuthorizationMode.builder()
 *                         .authorizationType(AuthorizationType.IAM)
 *                         .build())
 *                 .build())
 *         .xrayEnabled(true)
 *         .build();
 * 
 * Table demoTable = Table.Builder.create(this, "DemoTable")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .build();
 * 
 * DynamoDbDataSource demoDS = api.addDynamoDbDataSource("demoDataSource", demoTable);
 * 
 * // Resolver for the Query "getDemos" that scans the DynamoDb table and returns the entire list.
 * // Resolver Mapping Template Reference:
 * // https://docs.aws.amazon.com/appsync/latest/devguide/resolver-mapping-template-reference-dynamodb.html
 * demoDS.createResolver("QueryGetDemosResolver", BaseResolverProps.builder()
 *         .typeName("Query")
 *         .fieldName("getDemos")
 *         .requestMappingTemplate(MappingTemplate.dynamoDbScanTable())
 *         .responseMappingTemplate(MappingTemplate.dynamoDbResultList())
 *         .build());
 * 
 * // Resolver for the Mutation "addDemo" that puts the item into the DynamoDb table.
 * demoDS.createResolver("MutationAddDemoResolver", BaseResolverProps.builder()
 *         .typeName("Mutation")
 *         .fieldName("addDemo")
 *         .requestMappingTemplate(MappingTemplate.dynamoDbPutItem(PrimaryKey.partition("id").auto(), Values.projecting("input")))
 *         .responseMappingTemplate(MappingTemplate.dynamoDbResultItem())
 *         .build());
 * 
 * //To enable DynamoDB read consistency with the `MappingTemplate`:
 * demoDS.createResolver("QueryGetDemosConsistentResolver", BaseResolverProps.builder()
 *         .typeName("Query")
 *         .fieldName("getDemosConsistent")
 *         .requestMappingTemplate(MappingTemplate.dynamoDbScanTable(true))
 *         .responseMappingTemplate(MappingTemplate.dynamoDbResultList())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Aurora Serverless</h3>
 * <p>
 * AppSync provides a data source for executing SQL commands against Amazon Aurora
 * Serverless clusters. You can use AppSync resolvers to execute SQL statements
 * against the Data API with GraphQL queries, mutations, and subscriptions.
 * <p>
 * <blockquote><pre>
 * // Build a data source for AppSync to access the database.
 * GraphqlApi api;
 * // Create username and password secret for DB Cluster
 * DatabaseSecret secret = DatabaseSecret.Builder.create(this, "AuroraSecret")
 *         .username("clusteradmin")
 *         .build();
 * 
 * // The VPC to place the cluster in
 * Vpc vpc = new Vpc(this, "AuroraVpc");
 * 
 * // Create the serverless cluster, provide all values needed to customise the database.
 * ServerlessCluster cluster = ServerlessCluster.Builder.create(this, "AuroraCluster")
 *         .engine(DatabaseClusterEngine.AURORA_MYSQL)
 *         .vpc(vpc)
 *         .credentials(Map.of("username", "clusteradmin"))
 *         .clusterIdentifier("db-endpoint-test")
 *         .defaultDatabaseName("demos")
 *         .build();
 * RdsDataSource rdsDS = api.addRdsDataSource("rds", cluster, secret, "demos");
 * 
 * // Set up a resolver for an RDS query.
 * rdsDS.createResolver("QueryGetDemosRdsResolver", BaseResolverProps.builder()
 *         .typeName("Query")
 *         .fieldName("getDemosRds")
 *         .requestMappingTemplate(MappingTemplate.fromString("\n  {\n    \"version\": \"2018-05-29\",\n    \"statements\": [\n      \"SELECT * FROM demos\"\n    ]\n  }\n  "))
 *         .responseMappingTemplate(MappingTemplate.fromString("\n    $utils.toJson($utils.rds.toJsonObject($ctx.result)[0])\n  "))
 *         .build());
 * 
 * // Set up a resolver for an RDS mutation.
 * rdsDS.createResolver("MutationAddDemoRdsResolver", BaseResolverProps.builder()
 *         .typeName("Mutation")
 *         .fieldName("addDemoRds")
 *         .requestMappingTemplate(MappingTemplate.fromString("\n  {\n    \"version\": \"2018-05-29\",\n    \"statements\": [\n      \"INSERT INTO demos VALUES (:id, :version)\",\n      \"SELECT * WHERE id = :id\"\n    ],\n    \"variableMap\": {\n      \":id\": $util.toJson($util.autoId()),\n      \":version\": $util.toJson($ctx.args.version)\n    }\n  }\n  "))
 *         .responseMappingTemplate(MappingTemplate.fromString("\n    $utils.toJson($utils.rds.toJsonObject($ctx.result)[1][0])\n  "))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>HTTP Endpoints</h3>
 * <p>
 * GraphQL schema file <code>schema.graphql</code>:
 * <p>
 * <blockquote><pre>
 * type job {
 *   id: String!
 *   version: String!
 * }
 * 
 * input DemoInput {
 *   version: String!
 * }
 * 
 * type Mutation {
 *   callStepFunction(input: DemoInput!): job
 * }
 * </pre></blockquote>
 * <p>
 * GraphQL request mapping template <code>request.vtl</code>:
 * <p>
 * <blockquote><pre>
 * {
 *   "version": "2018-05-29",
 *   "method": "POST",
 *   "resourcePath": "/",
 *   "params": {
 *     "headers": {
 *       "content-type": "application/x-amz-json-1.0",
 *       "x-amz-target":"AWSStepFunctions.StartExecution"
 *     },
 *     "body": {
 *       "stateMachineArn": "&lt;your step functions arn&gt;",
 *       "input": "{ \"id\": \"$context.arguments.id\" }"
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * GraphQL response mapping template <code>response.vtl</code>:
 * <p>
 * <blockquote><pre>
 * {
 *   "id": "${context.result.id}"
 * }
 * </pre></blockquote>
 * <p>
 * CDK stack file <code>app-stack.ts</code>:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("api")
 *         .definition(Definition.fromFile(join(__dirname, "schema.graphql")))
 *         .build();
 * 
 * HttpDataSource httpDs = api.addHttpDataSource("ds", "https://states.amazonaws.com", HttpDataSourceOptions.builder()
 *         .name("httpDsWithStepF")
 *         .description("from appsync to StepFunctions Workflow")
 *         .authorizationConfig(AwsIamConfig.builder()
 *                 .signingRegion("us-east-1")
 *                 .signingServiceName("states")
 *                 .build())
 *         .build());
 * 
 * httpDs.createResolver("MutationCallStepFunctionResolver", BaseResolverProps.builder()
 *         .typeName("Mutation")
 *         .fieldName("callStepFunction")
 *         .requestMappingTemplate(MappingTemplate.fromFile("request.vtl"))
 *         .responseMappingTemplate(MappingTemplate.fromFile("response.vtl"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>EventBridge</h3>
 * <p>
 * Integrating AppSync with EventBridge enables developers to use EventBridge rules to route commands for GraphQL mutations
 * that need to perform any one of a variety of asynchronous tasks. More broadly, it enables teams to expose an event bus
 * as a part of a GraphQL schema.
 * <p>
 * GraphQL schema file <code>schema.graphql</code>:
 * <p>
 * <blockquote><pre>
 * schema {
 *     query: Query
 *     mutation: Mutation
 * }
 * 
 * type Query {
 *     event(id:ID!): Event
 * }
 * 
 * type Mutation {
 *     emitEvent(id: ID!, name: String): PutEventsResult!
 * }
 * 
 * type Event {
 *     id: ID!
 *     name: String!
 * }
 * 
 * type Entry {
 *     ErrorCode: String
 *     ErrorMessage: String
 *     EventId: String
 * }
 * 
 * type PutEventsResult {
 *     Entries: [Entry!]
 *     FailedEntry: Int
 * }
 * </pre></blockquote>
 * <p>
 * GraphQL request mapping template <code>request.vtl</code>:
 * <p>
 * <blockquote><pre>
 * {
 *     "version" : "2018-05-29",
 *     "operation": "PutEvents",
 *     "events" : [
 *         {
 *             "source": "integ.appsync.eventbridge",
 *             "detailType": "Mutation.emitEvent",
 *             "detail": $util.toJson($context.arguments)
 *         }
 *     ]
 * }
 * </pre></blockquote>
 * <p>
 * GraphQL response mapping template <code>response.vtl</code>:
 * <p>
 * <blockquote><pre>
 * $util.toJson($ctx.result)'
 * </pre></blockquote>
 * <p>
 * This response mapping template simply converts the EventBridge PutEvents result to JSON.
 * For details about the response see the
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html">documentation</a>.
 * Additional logic can be added to the response template to map the response type, or to error in the event of failed
 * events. More information can be found
 * <a href="https://docs.aws.amazon.com/appsync/latest/devguide/resolver-mapping-template-reference-eventbridge.html">here</a>.
 * <p>
 * CDK stack file <code>app-stack.ts</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.events.*;
 * 
 * 
 * GraphqlApi api = GraphqlApi.Builder.create(this, "EventBridgeApi")
 *         .name("EventBridgeApi")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.eventbridge.graphql")))
 *         .build();
 * 
 * EventBus bus = EventBus.Builder.create(this, "DestinationEventBus").build();
 * 
 * EventBridgeDataSource dataSource = api.addEventBridgeDataSource("NoneDS", bus);
 * 
 * dataSource.createResolver("EventResolver", BaseResolverProps.builder()
 *         .typeName("Mutation")
 *         .fieldName("emitEvent")
 *         .requestMappingTemplate(MappingTemplate.fromFile("request.vtl"))
 *         .responseMappingTemplate(MappingTemplate.fromFile("response.vtl"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Amazon OpenSearch Service</h3>
 * <p>
 * AppSync has builtin support for Amazon OpenSearch Service (successor to Amazon
 * Elasticsearch Service) from domains that are provisioned through your AWS account. You can
 * use AppSync resolvers to perform GraphQL operations such as queries, mutations, and
 * subscriptions.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.opensearchservice.*;
 * 
 * GraphqlApi api;
 * 
 * 
 * User user = new User(this, "User");
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_2_3)
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .fineGrainedAccessControl(AdvancedSecurityOptions.builder().masterUserArn(user.getUserArn()).build())
 *         .encryptionAtRest(EncryptionAtRestOptions.builder().enabled(true).build())
 *         .nodeToNodeEncryption(true)
 *         .enforceHttps(true)
 *         .build();
 * OpenSearchDataSource ds = api.addOpenSearchDataSource("ds", domain);
 * 
 * ds.createResolver("QueryGetTestsResolver", BaseResolverProps.builder()
 *         .typeName("Query")
 *         .fieldName("getTests")
 *         .requestMappingTemplate(MappingTemplate.fromString(JSON.stringify(Map.of(
 *                 "version", "2017-02-28",
 *                 "operation", "GET",
 *                 "path", "/id/post/_search",
 *                 "params", Map.of(
 *                         "headers", Map.of(),
 *                         "queryString", Map.of(),
 *                         "body", Map.of("from", 0, "size", 50))))))
 *         .responseMappingTemplate(MappingTemplate.fromString("[\n    #foreach($entry in $context.result.hits.hits)\n    #if( $velocityCount &gt; 1 ) , #end\n    $utils.toJson($entry.get(\"_source\"))\n    #end\n  ]"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Merged APIs</h2>
 * <p>
 * AppSync supports <a href="https://docs.aws.amazon.com/appsync/latest/devguide/merged-api.html">Merged APIs</a> which can be used to merge multiple source APIs into a single API.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * // first source API
 * GraphqlApi firstApi = GraphqlApi.Builder.create(this, "FirstSourceAPI")
 *         .name("FirstSourceAPI")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.merged-api-1.graphql")))
 *         .build();
 * 
 * // second source API
 * GraphqlApi secondApi = GraphqlApi.Builder.create(this, "SecondSourceAPI")
 *         .name("SecondSourceAPI")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.merged-api-2.graphql")))
 *         .build();
 * 
 * // Merged API
 * GraphqlApi mergedApi = GraphqlApi.Builder.create(this, "MergedAPI")
 *         .name("MergedAPI")
 *         .definition(Definition.fromSourceApis(SourceApiOptions.builder()
 *                 .sourceApis(List.of(SourceApi.builder()
 *                         .sourceApi(firstApi)
 *                         .mergeType(MergeType.MANUAL_MERGE)
 *                         .build(), SourceApi.builder()
 *                         .sourceApi(secondApi)
 *                         .mergeType(MergeType.AUTO_MERGE)
 *                         .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Merged APIs Across Different Stacks</h2>
 * <p>
 * The SourceApiAssociation construct allows you to define a SourceApiAssociation to a Merged API in a different stack or account. This allows a source API owner the ability to associate it to an existing Merged API itself.
 * <p>
 * <blockquote><pre>
 * GraphqlApi sourceApi = GraphqlApi.Builder.create(this, "FirstSourceAPI")
 *         .name("FirstSourceAPI")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.merged-api-1.graphql")))
 *         .build();
 * 
 * IGraphqlApi importedMergedApi = GraphqlApi.fromGraphqlApiAttributes(this, "ImportedMergedApi", GraphqlApiAttributes.builder()
 *         .graphqlApiId("MyApiId")
 *         .graphqlApiArn("MyApiArn")
 *         .build());
 * 
 * IRole importedExecutionRole = Role.fromRoleArn(this, "ExecutionRole", "arn:aws:iam::ACCOUNT:role/MyExistingRole");
 * SourceApiAssociation.Builder.create(this, "SourceApiAssociation2")
 *         .sourceApi(sourceApi)
 *         .mergedApi(importedMergedApi)
 *         .mergeType(MergeType.MANUAL_MERGE)
 *         .mergedApiExecutionRole(importedExecutionRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Merge Source API Update Within CDK Deployment</h2>
 * <p>
 * The SourceApiAssociationMergeOperation construct available in the <a href="https://github.com/cdklabs/awscdk-appsync-utils">awscdk-appsync-utils</a> package provides the ability to merge a source API to a Merged API via a custom
 * resource. If the merge operation fails with a conflict, the stack update will fail and rollback the changes to the source API in the stack in order to prevent merge conflicts and ensure the source API changes are always propagated to the Merged API.
 * <p>
 * <h2>Custom Domain Names</h2>
 * <p>
 * For many use cases you may want to associate a custom domain name with your
 * GraphQL API. This can be done during the API creation.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.*;
 * import software.amazon.awscdk.services.route53.*;
 * 
 * // hosted zone and route53 features
 * String hostedZoneId;
 * String zoneName = "example.com";
 * 
 * 
 * String myDomainName = "api.example.com";
 * Certificate certificate = Certificate.Builder.create(this, "cert").domainName(myDomainName).build();
 * SchemaFile schema = SchemaFile.Builder.create().filePath("mySchemaFile").build();
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("myApi")
 *         .definition(Definition.fromSchema(schema))
 *         .domainName(DomainOptions.builder()
 *                 .certificate(certificate)
 *                 .domainName(myDomainName)
 *                 .build())
 *         .build();
 * 
 * // hosted zone for adding appsync domain
 * IHostedZone zone = HostedZone.fromHostedZoneAttributes(this, "HostedZone", HostedZoneAttributes.builder()
 *         .hostedZoneId(hostedZoneId)
 *         .zoneName(zoneName)
 *         .build());
 * 
 * // create a cname to the appsync domain. will map to something like xxxx.cloudfront.net
 * // create a cname to the appsync domain. will map to something like xxxx.cloudfront.net
 * CnameRecord.Builder.create(this, "CnameApiRecord")
 *         .recordName("api")
 *         .zone(zone)
 *         .domainName(api.getAppSyncDomainName())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Log Group</h2>
 * <p>
 * AppSync automatically create a log group with the name <code>/aws/appsync/apis/&lt;graphql_api_id&gt;</code> upon deployment with
 * log data set to never expire. If you want to set a different expiration period, use the <code>logConfig.retention</code> property.
 * <p>
 * To obtain the GraphQL API's log group as a <code>logs.ILogGroup</code> use the <code>logGroup</code> property of the
 * <code>GraphqlApi</code> construct.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * 
 * 
 * LogConfig logConfig = LogConfig.builder()
 *         .retention(RetentionDays.ONE_WEEK)
 *         .build();
 * 
 * GraphqlApi.Builder.create(this, "api")
 *         .authorizationConfig(AuthorizationConfig.builder().build())
 *         .name("myApi")
 *         .definition(Definition.fromFile(join(__dirname, "myApi.graphql")))
 *         .logConfig(logConfig)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Schema</h2>
 * <p>
 * You can define a schema using from a local file using <code>Definition.fromFile</code>
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("myApi")
 *         .definition(Definition.fromFile(join(__dirname, "schema.graphl")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>ISchema</h3>
 * <p>
 * Alternative schema sources can be defined by implementing the <code>ISchema</code>
 * interface. An example of this is the <code>CodeFirstSchema</code> class provided in
 * <a href="https://github.com/cdklabs/awscdk-appsync-utils">awscdk-appsync-utils</a>
 * <p>
 * <h2>Imports</h2>
 * <p>
 * Any GraphQL Api that has been created outside the stack can be imported from
 * another stack into your CDK app. Utilizing the <code>fromXxx</code> function, you have
 * the ability to add data sources and resolvers through a <code>IGraphqlApi</code> interface.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * Table table;
 * 
 * IGraphqlApi importedApi = GraphqlApi.fromGraphqlApiAttributes(this, "IApi", GraphqlApiAttributes.builder()
 *         .graphqlApiId(api.getApiId())
 *         .graphqlApiArn(api.getArn())
 *         .build());
 * importedApi.addDynamoDbDataSource("TableDataSource", table);
 * </pre></blockquote>
 * <p>
 * If you don't specify <code>graphqlArn</code> in <code>fromXxxAttributes</code>, CDK will autogenerate
 * the expected <code>arn</code> for the imported api, given the <code>apiId</code>. For creating data
 * sources and resolvers, an <code>apiId</code> is sufficient.
 * <p>
 * <h2>Private APIs</h2>
 * <p>
 * By default all AppSync GraphQL APIs are public and can be accessed from the internet.
 * For customers that want to limit access to be from their VPC, the optional API <code>visibility</code> property can be set to <code>Visibility.PRIVATE</code>
 * at creation time. To explicitly create a public API, the <code>visibility</code> property should be set to <code>Visibility.GLOBAL</code>.
 * If visibility is not set, the service will default to <code>GLOBAL</code>.
 * <p>
 * CDK stack file <code>app-stack.ts</code>:
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("MyPrivateAPI")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.schema.graphql")))
 *         .visibility(Visibility.PRIVATE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/appsync/latest/devguide/using-private-apis.html">documentation</a>
 * for more details about Private APIs
 * <p>
 * <h2>Authorization</h2>
 * <p>
 * There are multiple authorization types available for GraphQL API to cater to different
 * access use cases. They are:
 * <p>
 * <ul>
 * <li>API Keys (<code>AuthorizationType.API_KEY</code>)</li>
 * <li>Amazon Cognito User Pools (<code>AuthorizationType.USER_POOL</code>)</li>
 * <li>OpenID Connect (<code>AuthorizationType.OPENID_CONNECT</code>)</li>
 * <li>AWS Identity and Access Management (<code>AuthorizationType.AWS_IAM</code>)</li>
 * <li>AWS Lambda (<code>AuthorizationType.AWS_LAMBDA</code>)</li>
 * </ul>
 * <p>
 * These types can be used simultaneously in a single API, allowing different types of clients to
 * access data. When you specify an authorization type, you can also specify the corresponding
 * authorization mode to finish defining your authorization. For example, this is a GraphQL API
 * with AWS Lambda Authorization.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * Function authFunction;
 * 
 * 
 * GraphqlApi.Builder.create(this, "api")
 *         .name("api")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.test.graphql")))
 *         .authorizationConfig(AuthorizationConfig.builder()
 *                 .defaultAuthorization(AuthorizationMode.builder()
 *                         .authorizationType(AuthorizationType.LAMBDA)
 *                         .lambdaAuthorizerConfig(LambdaAuthorizerConfig.builder()
 *                                 .handler(authFunction)
 *                                 .build())
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Permissions</h2>
 * <p>
 * When using <code>AWS_IAM</code> as the authorization type for GraphQL API, an IAM Role
 * with correct permissions must be used for access to API.
 * <p>
 * When configuring permissions, you can specify specific resources to only be
 * accessible by <code>IAM</code> authorization. For example, if you want to only allow mutability
 * for <code>IAM</code> authorized access you would configure the following.
 * <p>
 * In <code>schema.graphql</code>:
 * <p>
 * <blockquote><pre>
 * type Mutation {
 *   updateExample(...): ...
 *     &#64;aws_iam
 * }
 * </pre></blockquote>
 * <p>
 * In <code>IAM</code>:
 * <p>
 * <blockquote><pre>
 * {
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {
 *       "Effect": "Allow",
 *       "Action": [
 *         "appsync:GraphQL"
 *       ],
 *       "Resource": [
 *         "arn:aws:appsync:REGION:ACCOUNT_ID:apis/GRAPHQL_ID/types/Mutation/fields/updateExample"
 *       ]
 *     }
 *   ]
 * }
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/appsync/latest/devguide/security.html#aws-iam-authorization">documentation</a> for more details.
 * <p>
 * To make this easier, CDK provides <code>grant</code> API.
 * <p>
 * Use the <code>grant</code> function for more granular authorization.
 * <p>
 * <blockquote><pre>
 * IGraphqlApi api;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * 
 * api.grant(role, IamResource.custom("types/Mutation/fields/updateExample"), "appsync:GraphQL");
 * </pre></blockquote>
 * <p>
 * <h3>IamResource</h3>
 * <p>
 * In order to use the <code>grant</code> functions, you need to use the class <code>IamResource</code>.
 * <p>
 * <ul>
 * <li><code>IamResource.custom(...arns)</code> permits custom ARNs and requires an argument.</li>
 * <li><code>IamResouce.ofType(type, ...fields)</code> permits ARNs for types and their fields.</li>
 * <li><code>IamResource.all()</code> permits ALL resources.</li>
 * </ul>
 * <p>
 * <h3>Generic Permissions</h3>
 * <p>
 * Alternatively, you can use more generic <code>grant</code> functions to accomplish the same usage.
 * <p>
 * These include:
 * <p>
 * <ul>
 * <li>grantMutation (use to grant access to Mutation fields)</li>
 * <li>grantQuery (use to grant access to Query fields)</li>
 * <li>grantSubscription (use to grant access to Subscription fields)</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * IGraphqlApi api;
 * Role role;
 * 
 * 
 * // For generic types
 * api.grantMutation(role, "updateExample");
 * 
 * // For custom types and granular design
 * api.grant(role, IamResource.ofType("Mutation", "updateExample"), "appsync:GraphQL");
 * </pre></blockquote>
 * <p>
 * <h2>Pipeline Resolvers and AppSync Functions</h2>
 * <p>
 * AppSync Functions are local functions that perform certain operations onto a
 * backend data source. Developers can compose operations (Functions) and execute
 * them in sequence with Pipeline Resolvers.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * 
 * 
 * AppsyncFunction appsyncFunction = AppsyncFunction.Builder.create(this, "function")
 *         .name("appsync_function")
 *         .api(api)
 *         .dataSource(api.addNoneDataSource("none"))
 *         .requestMappingTemplate(MappingTemplate.fromFile("request.vtl"))
 *         .responseMappingTemplate(MappingTemplate.fromFile("response.vtl"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * AppSync Functions are used in tandem with pipeline resolvers to compose multiple
 * operations.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * AppsyncFunction appsyncFunction;
 * 
 * 
 * Resolver pipelineResolver = Resolver.Builder.create(this, "pipeline")
 *         .api(api)
 *         .dataSource(api.addNoneDataSource("none"))
 *         .typeName("typeName")
 *         .fieldName("fieldName")
 *         .requestMappingTemplate(MappingTemplate.fromFile("beforeRequest.vtl"))
 *         .pipelineConfig(List.of(appsyncFunction))
 *         .responseMappingTemplate(MappingTemplate.fromFile("afterResponse.vtl"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>JS Functions and Resolvers</h3>
 * <p>
 * JS Functions and resolvers are also supported. You can use a <code>.js</code> file within your CDK project, or specify your function code inline.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api;
 * 
 * 
 * AppsyncFunction myJsFunction = AppsyncFunction.Builder.create(this, "function")
 *         .name("my_js_function")
 *         .api(api)
 *         .dataSource(api.addNoneDataSource("none"))
 *         .code(Code.fromAsset("directory/function_code.js"))
 *         .runtime(FunctionRuntime.JS_1_0_0)
 *         .build();
 * 
 * Resolver.Builder.create(this, "PipelineResolver")
 *         .api(api)
 *         .typeName("typeName")
 *         .fieldName("fieldName")
 *         .code(Code.fromInline("\n    // The before step\n    export function request(...args) {\n      console.log(args);\n      return {}\n    }\n\n    // The after step\n    export function response(ctx) {\n      return ctx.prev.result\n    }\n  "))
 *         .runtime(FunctionRuntime.JS_1_0_0)
 *         .pipelineConfig(List.of(myJsFunction))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Learn more about Pipeline Resolvers and AppSync Functions <a href="https://docs.aws.amazon.com/appsync/latest/devguide/pipeline-resolvers.html">here</a>.
 * <p>
 * <h2>Introspection</h2>
 * <p>
 * By default, AppSync allows you to use introspection queries.
 * <p>
 * For customers that want to limit access to be introspection queries, the <code>introspectionConfig</code> property can be set to <code>IntrospectionConfig.DISABLED</code> at creation time.
 * If <code>introspectionConfig</code> is not set, the service will default to <code>ENABLED</code>.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("DisableIntrospectionApi")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.schema.graphql")))
 *         .introspectionConfig(IntrospectionConfig.DISABLED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Query Depth Limits</h2>
 * <p>
 * By default, queries are able to process an unlimited amount of nested levels.
 * Limiting queries to a specified amount of nested levels has potential implications for the performance and flexibility of your project.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("LimitQueryDepths")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.schema.graphql")))
 *         .queryDepthLimit(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Resolver Count Limits</h2>
 * <p>
 * You can control how many resolvers each query can process.
 * By default, each query can process up to 10000 resolvers.
 * By setting a limit AppSync will not handle any resolvers past a certain number limit.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("LimitResolverCount")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.schema.graphql")))
 *         .resolverCountLimit(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Environment Variables</h2>
 * <p>
 * To use environment variables in resolvers, you can use the <code>environmentVariables</code> property and
 * the <code>addEnvironmentVariable</code> method.
 * <p>
 * <blockquote><pre>
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("api")
 *         .definition(Definition.fromFile(join(__dirname, "appsync.schema.graphql")))
 *         .environmentVariables(Map.of(
 *                 "EnvKey1", "non-empty-1"))
 *         .build();
 * 
 * api.addEnvironmentVariable("EnvKey2", "non-empty-2");
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.appsync;
