/**
 * <h1>AWS Key Management Service Construct Library</h1>
 * <p>
 * Define a KMS key:
 * <p>
 * <blockquote><pre>
 * Key.Builder.create(this, "MyKey")
 *         .enableKeyRotation(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Define a KMS key with waiting period:
 * <p>
 * Specifies the number of days in the waiting period before AWS KMS deletes a CMK that has been removed from a CloudFormation stack.
 * <p>
 * <blockquote><pre>
 * Key key = Key.Builder.create(this, "MyKey")
 *         .pendingWindow(Duration.days(10))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Add a couple of aliases:
 * <p>
 * <blockquote><pre>
 * Key key = new Key(this, "MyKey");
 * key.addAlias("alias/foo");
 * key.addAlias("alias/bar");
 * </pre></blockquote>
 * <p>
 * Define a key with specific key spec and key usage:
 * <p>
 * Valid <code>keySpec</code> values depends on <code>keyUsage</code> value.
 * <p>
 * <blockquote><pre>
 * Key key = Key.Builder.create(this, "MyKey")
 *         .keySpec(KeySpec.ECC_SECG_P256K1) // Default to SYMMETRIC_DEFAULT
 *         .keyUsage(KeyUsage.SIGN_VERIFY)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Sharing keys between stacks</h2>
 * <p>
 * To use a KMS key in a different stack in the same CDK application,
 * pass the construct to the other stack:
 * <p>
 * <blockquote><pre>
 * /**
 *  * Stack that defines the key
 *  *&#47;
 * public class KeyStack extends Stack {
 *     public final Key key;
 * 
 *     public KeyStack(App scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public KeyStack(App scope, String id, StackProps props) {
 *         super(scope, id, props);
 *         this.key = Key.Builder.create(this, "MyKey").removalPolicy(RemovalPolicy.DESTROY).build();
 *     }
 * }
 * 
 * public class UseStackProps extends StackProps {
 *     private IKey key;
 *     public IKey getKey() {
 *         return this.key;
 *     }
 *     public UseStackProps key(IKey key) {
 *         this.key = key;
 *         return this;
 *     }
 * }
 * 
 * /**
 *  * Stack that uses the key
 *  *&#47;
 * public class UseStack extends Stack {
 *     public UseStack(App scope, String id, UseStackProps props) {
 *         super(scope, id, props);
 * 
 *         // Use the IKey object here.
 *         // Use the IKey object here.
 *         Alias.Builder.create(this, "Alias")
 *                 .aliasName("alias/foo")
 *                 .targetKey(props.getKey())
 *                 .build();
 *     }
 * }
 * 
 * KeyStack keyStack = new KeyStack(app, "KeyStack");
 * new UseStack(app, "UseStack", new UseStackProps().key(keyStack.getKey()));
 * </pre></blockquote>
 * <p>
 * <h2>Importing existing keys</h2>
 * <p>
 * <h3>Import key by ARN</h3>
 * <p>
 * To use a KMS key that is not defined in this CDK app, but is created through other means, use
 * <code>Key.fromKeyArn(parent, name, ref)</code>:
 * <p>
 * <blockquote><pre>
 * IKey myKeyImported = Key.fromKeyArn(this, "MyImportedKey", "arn:aws:...");
 * 
 * // you can do stuff with this imported key.
 * myKeyImported.addAlias("alias/foo");
 * </pre></blockquote>
 * <p>
 * Note that a call to <code>.addToResourcePolicy(statement)</code> on <code>myKeyImported</code> will not have
 * an affect on the key's policy because it is not owned by your stack. The call
 * will be a no-op.
 * <p>
 * <h3>Import key by alias</h3>
 * <p>
 * If a Key has an associated Alias, the Alias can be imported by name and used in place
 * of the Key as a reference. A common scenario for this is in referencing AWS managed keys.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudtrail.*;
 * 
 * 
 * IAlias myKeyAlias = Alias.fromAliasName(this, "myKey", "alias/aws/s3");
 * Trail trail = Trail.Builder.create(this, "myCloudTrail")
 *         .sendToCloudWatchLogs(true)
 *         .encryptionKey(myKeyAlias)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that calls to <code>addToResourcePolicy</code> and <code>grant*</code> methods on <code>myKeyAlias</code> will be
 * no-ops, and <code>addAlias</code> and <code>aliasTargetKey</code> will fail, as the imported alias does not
 * have a reference to the underlying KMS Key.
 * <p>
 * <h3>Lookup key by alias</h3>
 * <p>
 * If you can't use a KMS key imported by alias (e.g. because you need access to the key id), you can lookup the key with <code>Key.fromLookup()</code>.
 * <p>
 * In general, the preferred method would be to use <code>Alias.fromAliasName()</code> which returns an <code>IAlias</code> object which extends <code>IKey</code>. However, some services need to have access to the underlying key id. In this case, <code>Key.fromLookup()</code> allows to lookup the key id.
 * <p>
 * The result of the <code>Key.fromLookup()</code> operation will be written to a file
 * called <code>cdk.context.json</code>. You must commit this file to source control so
 * that the lookup values are available in non-privileged environments such
 * as CI build steps, and to ensure your template builds are repeatable.
 * <p>
 * Here's how <code>Key.fromLookup()</code> can be used:
 * <p>
 * <blockquote><pre>
 * IKey myKeyLookup = Key.fromLookup(this, "MyKeyLookup", KeyLookupOptions.builder()
 *         .aliasName("alias/KeyAlias")
 *         .build());
 * 
 * Role role = Role.Builder.create(this, "MyRole")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * myKeyLookup.grantEncryptDecrypt(role);
 * </pre></blockquote>
 * <p>
 * Note that a call to <code>.addToResourcePolicy(statement)</code> on <code>myKeyLookup</code> will not have
 * an affect on the key's policy because it is not owned by your stack. The call
 * will be a no-op.
 * <p>
 * <h2>Key Policies</h2>
 * <p>
 * Controlling access and usage of KMS Keys requires the use of key policies (resource-based policies attached to the key);
 * this is in contrast to most other AWS resources where access can be entirely controlled with IAM policies,
 * and optionally complemented with resource policies. For more in-depth understanding of KMS key access and policies, see
 * <p>
 * <ul>
 * <li>https://docs.aws.amazon.com/kms/latest/developerguide/control-access-overview.html</li>
 * <li>https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html</li>
 * </ul>
 * <p>
 * KMS keys can be created to trust IAM policies. This is the default behavior for both the KMS APIs and in
 * the console. This behavior is enabled by the '&#64;aws-cdk/aws-kms:defaultKeyPolicies' feature flag,
 * which is set for all new projects; for existing projects, this same behavior can be enabled by
 * passing the <code>trustAccountIdentities</code> property as <code>true</code> when creating the key:
 * <p>
 * <blockquote><pre>
 * Key.Builder.create(this, "MyKey").trustAccountIdentities(true).build();
 * </pre></blockquote>
 * <p>
 * With either the <code>&#64;aws-cdk/aws-kms:defaultKeyPolicies</code> feature flag set,
 * or the <code>trustAccountIdentities</code> prop set, the Key will be given the following default key policy:
 * <p>
 * <blockquote><pre>
 * {
 *   "Effect": "Allow",
 *   "Principal": {"AWS": "arn:aws:iam::111122223333:root"},
 *   "Action": "kms:*",
 *   "Resource": "*"
 * }
 * </pre></blockquote>
 * <p>
 * This policy grants full access to the key to the root account user.
 * This enables the root account user -- via IAM policies -- to grant access to other IAM principals.
 * With the above default policy, future permissions can be added to either the key policy or IAM principal policy.
 * <p>
 * <blockquote><pre>
 * Key key = new Key(this, "MyKey");
 * User user = new User(this, "MyUser");
 * key.grantEncrypt(user);
 * </pre></blockquote>
 * <p>
 * Adopting the default KMS key policy (and so trusting account identities)
 * solves many issues around cyclic dependencies between stacks.
 * Without this default key policy, future permissions must be added to both the key policy and IAM principal policy,
 * which can cause cyclic dependencies if the permissions cross stack boundaries.
 * (For example, an encrypted bucket in one stack, and Lambda function that accesses it in another.)
 * <p>
 * <h3>Appending to or replacing the default key policy</h3>
 * <p>
 * The default key policy can be amended or replaced entirely, depending on your use case and requirements.
 * A common addition to the key policy would be to add other key admins that are allowed to administer the key
 * (e.g., change permissions, revoke, delete). Additional key admins can be specified at key creation or after
 * via the <code>grantAdmin</code> method.
 * <p>
 * <blockquote><pre>
 * IRole myTrustedAdminRole = Role.fromRoleArn(this, "TrustedRole", "arn:aws:iam:....");
 * Key key = Key.Builder.create(this, "MyKey")
 *         .admins(List.of(myTrustedAdminRole))
 *         .build();
 * 
 * Key secondKey = new Key(this, "MyKey2");
 * secondKey.grantAdmin(myTrustedAdminRole);
 * </pre></blockquote>
 * <p>
 * Alternatively, a custom key policy can be specified, which will replace the default key policy.
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note</strong>: In applications without the '&#64;aws-cdk/aws-kms:defaultKeyPolicies' feature flag set
 * and with <code>trustedAccountIdentities</code> set to false (the default), specifying a policy at key creation <em>appends</em> the
 * provided policy to the default key policy, rather than <em>replacing</em> the default policy.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * IRole myTrustedAdminRole = Role.fromRoleArn(this, "TrustedRole", "arn:aws:iam:....");
 * // Creates a limited admin policy and assigns to the account root.
 * PolicyDocument myCustomPolicy = PolicyDocument.Builder.create()
 *         .statements(List.of(PolicyStatement.Builder.create()
 *                 .actions(List.of("kms:Create*", "kms:Describe*", "kms:Enable*", "kms:List*", "kms:Put*"))
 *                 .principals(List.of(new AccountRootPrincipal()))
 *                 .resources(List.of("*"))
 *                 .build()))
 *         .build();
 * Key key = Key.Builder.create(this, "MyKey")
 *         .policy(myCustomPolicy)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Warning:</strong> Replacing the default key policy with one that only grants access to a specific user or role
 * runs the risk of the key becoming unmanageable if that user or role is deleted.
 * It is highly recommended that the key policy grants access to the account root, rather than specific principals.
 * See https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html for more information.
 * <p>
 * </blockquote>
 * <p>
 * <h3>HMAC specific key policies</h3>
 * <p>
 * HMAC keys have a different key policy than other KMS keys. They have a policy for generating and for verifying a MAC.
 * The respective policies can be attached to a principal via the <code>grantGenerateMac</code> and <code>grantVerifyMac</code> methods.
 * <p>
 * <blockquote><pre>
 * Key key = new Key(this, "MyKey");
 * User user = new User(this, "MyUser");
 * key.grantGenerateMac(user); // Adds 'kms:GenerateMac' to the principal's policy
 * key.grantVerifyMac(user);
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.kms;
