/**
 * <h1>Amazon Simple Notification Service Construct Library</h1>
 * <p>
 * Add an SNS Topic to your stack:
 * <p>
 * <blockquote><pre>
 * Topic topic = Topic.Builder.create(this, "Topic")
 *         .displayName("Customer subscription topic")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Add a FIFO SNS topic with content-based de-duplication to your stack:
 * <p>
 * <blockquote><pre>
 * Topic topic = Topic.Builder.create(this, "Topic")
 *         .contentBasedDeduplication(true)
 *         .displayName("Customer subscription topic")
 *         .fifo(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that FIFO topics require a topic name to be provided. The required <code>.fifo</code> suffix will be automatically generated and added to the topic name if it is not explicitly provided.
 * <p>
 * <h2>Subscriptions</h2>
 * <p>
 * Various subscriptions can be added to the topic by calling the
 * <code>.addSubscription(...)</code> method on the topic. It accepts a <em>subscription</em> object,
 * default implementations of which can be found in the
 * <code>aws-cdk-lib/aws-sns-subscriptions</code> package:
 * <p>
 * Add an HTTPS Subscription to your topic:
 * <p>
 * <blockquote><pre>
 * Topic myTopic = new Topic(this, "MyTopic");
 * 
 * myTopic.addSubscription(new UrlSubscription("https://foobar.com/"));
 * </pre></blockquote>
 * <p>
 * Subscribe a queue to the topic:
 * <p>
 * <blockquote><pre>
 * Queue queue;
 * 
 * Topic myTopic = new Topic(this, "MyTopic");
 * 
 * myTopic.addSubscription(new SqsSubscription(queue));
 * </pre></blockquote>
 * <p>
 * Note that subscriptions of queues in different accounts need to be manually confirmed by
 * reading the initial message from the queue and visiting the link found in it.
 * <p>
 * <h3>Filter policy</h3>
 * <p>
 * A filter policy can be specified when subscribing an endpoint to a topic.
 * <p>
 * Example with a Lambda subscription:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * Function fn;
 * 
 * 
 * Topic myTopic = new Topic(this, "MyTopic");
 * 
 * // Lambda should receive only message matching the following conditions on attributes:
 * // color: 'red' or 'orange' or begins with 'bl'
 * // size: anything but 'small' or 'medium'
 * // price: between 100 and 200 or greater than 300
 * // store: attribute must be present
 * myTopic.addSubscription(LambdaSubscription.Builder.create(fn)
 *         .filterPolicy(Map.of(
 *                 "color", SubscriptionFilter.stringFilter(StringConditions.builder()
 *                         .allowlist(List.of("red", "orange"))
 *                         .matchPrefixes(List.of("bl"))
 *                         .matchSuffixes(List.of("ue"))
 *                         .build()),
 *                 "size", SubscriptionFilter.stringFilter(StringConditions.builder()
 *                         .denylist(List.of("small", "medium"))
 *                         .build()),
 *                 "price", SubscriptionFilter.numericFilter(NumericConditions.builder()
 *                         .between(BetweenCondition.builder().start(100).stop(200).build())
 *                         .greaterThan(300)
 *                         .build()),
 *                 "store", SubscriptionFilter.existsFilter()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Payload-based filtering</h4>
 * <p>
 * To filter messages based on the payload or body of the message, use the <code>filterPolicyWithMessageBody</code> property. This type of filter policy supports creating filters on nested objects.
 * <p>
 * Example with a Lambda subscription:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * Function fn;
 * 
 * 
 * Topic myTopic = new Topic(this, "MyTopic");
 * 
 * // Lambda should receive only message matching the following conditions on message body:
 * // color: 'red' or 'orange'
 * myTopic.addSubscription(LambdaSubscription.Builder.create(fn)
 *         .filterPolicyWithMessageBody(Map.of(
 *                 "background", FilterOrPolicy.policy(Map.of(
 *                         "color", FilterOrPolicy.filter(SubscriptionFilter.stringFilter(StringConditions.builder()
 *                                 .allowlist(List.of("red", "orange"))
 *                                 .build()))))))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Example of Firehose Subscription</h3>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kinesisfirehose.alpha.DeliveryStream;
 * DeliveryStream stream;
 * 
 * 
 * Topic topic = new Topic(this, "Topic");
 * 
 * Subscription.Builder.create(this, "Subscription")
 *         .topic(topic)
 *         .endpoint(stream.getDeliveryStreamArn())
 *         .protocol(SubscriptionProtocol.FIREHOSE)
 *         .subscriptionRoleArn("SAMPLE_ARN")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>DLQ setup for SNS Subscription</h2>
 * <p>
 * CDK can attach provided Queue as DLQ for your SNS subscription.
 * See the <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-configure-dead-letter-queue.html">SNS DLQ configuration docs</a> for more information about this feature.
 * <p>
 * Example of usage with user provided DLQ.
 * <p>
 * <blockquote><pre>
 * Topic topic = new Topic(this, "Topic");
 * Queue dlQueue = Queue.Builder.create(this, "DeadLetterQueue")
 *         .queueName("MySubscription_DLQ")
 *         .retentionPeriod(Duration.days(14))
 *         .build();
 * 
 * Subscription.Builder.create(this, "Subscription")
 *         .endpoint("endpoint")
 *         .protocol(SubscriptionProtocol.LAMBDA)
 *         .topic(topic)
 *         .deadLetterQueue(dlQueue)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>CloudWatch Event Rule Target</h2>
 * <p>
 * SNS topics can be used as targets for CloudWatch event rules.
 * <p>
 * Use the <code>aws-cdk-lib/aws-events-targets.SnsTopic</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codecommit.*;
 * import software.amazon.awscdk.services.events.targets.*;
 * 
 * Repository repo;
 * 
 * Topic myTopic = new Topic(this, "Topic");
 * 
 * repo.onCommit("OnCommit", OnCommitOptions.builder()
 *         .target(new SnsTopic(myTopic))
 *         .build());
 * </pre></blockquote>
 * <p>
 * This will result in adding a target to the event rule and will also modify the
 * topic resource policy to allow CloudWatch events to publish to the topic.
 * <p>
 * <h2>Topic Policy</h2>
 * <p>
 * A topic policy is automatically created when <code>addToResourcePolicy</code> is called, if
 * one doesn't already exist. Using <code>addToResourcePolicy</code> is the simplest way to
 * add policies, but a <code>TopicPolicy</code> can also be created manually.
 * <p>
 * <blockquote><pre>
 * Topic topic = new Topic(this, "Topic");
 * TopicPolicy topicPolicy = TopicPolicy.Builder.create(this, "TopicPolicy")
 *         .topics(List.of(topic))
 *         .build();
 * 
 * topicPolicy.document.addStatements(PolicyStatement.Builder.create()
 *         .actions(List.of("sns:Subscribe"))
 *         .principals(List.of(new AnyPrincipal()))
 *         .resources(List.of(topic.getTopicArn()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * A policy document can also be passed on <code>TopicPolicy</code> construction
 * <p>
 * <blockquote><pre>
 * Topic topic = new Topic(this, "Topic");
 * PolicyDocument policyDocument = PolicyDocument.Builder.create()
 *         .assignSids(true)
 *         .statements(List.of(
 *             PolicyStatement.Builder.create()
 *                     .actions(List.of("sns:Subscribe"))
 *                     .principals(List.of(new AnyPrincipal()))
 *                     .resources(List.of(topic.getTopicArn()))
 *                     .build()))
 *         .build();
 * 
 * TopicPolicy topicPolicy = TopicPolicy.Builder.create(this, "Policy")
 *         .topics(List.of(topic))
 *         .policyDocument(policyDocument)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Enforce encryption of data in transit when publishing to a topic</h3>
 * <p>
 * You can enforce SSL when creating a topic policy by setting the <code>enforceSSL</code> flag:
 * <p>
 * <blockquote><pre>
 * Topic topic = new Topic(this, "Topic");
 * PolicyDocument policyDocument = PolicyDocument.Builder.create()
 *         .assignSids(true)
 *         .statements(List.of(
 *             PolicyStatement.Builder.create()
 *                     .actions(List.of("sns:Publish"))
 *                     .principals(List.of(new ServicePrincipal("s3.amazonaws.com")))
 *                     .resources(List.of(topic.getTopicArn()))
 *                     .build()))
 *         .build();
 * 
 * TopicPolicy topicPolicy = TopicPolicy.Builder.create(this, "Policy")
 *         .topics(List.of(topic))
 *         .policyDocument(policyDocument)
 *         .enforceSSL(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Similiarly you can enforce SSL by setting the <code>enforceSSL</code> flag on the topic:
 * <p>
 * <blockquote><pre>
 * Topic topic = Topic.Builder.create(this, "TopicAddPolicy")
 *         .enforceSSL(true)
 *         .build();
 * 
 * topic.addToResourcePolicy(PolicyStatement.Builder.create()
 *         .principals(List.of(new ServicePrincipal("s3.amazonaws.com")))
 *         .actions(List.of("sns:Publish"))
 *         .resources(List.of(topic.getTopicArn()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Delivery status logging</h2>
 * <p>
 * Amazon SNS provides support to log the delivery status of notification messages sent to topics with the following Amazon SNS endpoints:
 * <p>
 * <ul>
 * <li>HTTP</li>
 * <li>Amazon Kinesis Data Firehose</li>
 * <li>AWS Lambda</li>
 * <li>Platform application endpoint</li>
 * <li>Amazon Simple Queue Service</li>
 * </ul>
 * <p>
 * Example with a delivery status logging configuration for SQS:
 * <p>
 * <blockquote><pre>
 * Role role;
 * 
 * Topic topic = Topic.Builder.create(this, "MyTopic")
 *         .loggingConfigs(List.of(LoggingConfig.builder()
 *                 .protocol(LoggingProtocol.SQS)
 *                 .failureFeedbackRole(role)
 *                 .successFeedbackRole(role)
 *                 .successFeedbackSampleRate(50)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * A delivery status logging configuration can also be added to your topic by <code>addLoggingConfig</code> method:
 * <p>
 * <blockquote><pre>
 * Role role;
 * 
 * Topic topic = new Topic(this, "MyTopic");
 * 
 * topic.addLoggingConfig(LoggingConfig.builder()
 *         .protocol(LoggingProtocol.SQS)
 *         .failureFeedbackRole(role)
 *         .successFeedbackRole(role)
 *         .successFeedbackSampleRate(50)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Note that valid values for <code>successFeedbackSampleRate</code> are integer between 0-100.
 * <p>
 * <h2>Archive Policy</h2>
 * <p>
 * Message archiving provides the ability to archive a single copy of all messages published to your topic.
 * You can store published messages within your topic by enabling the message archive policy on the topic, which enables message archiving for all subscriptions linked to that topic.
 * Messages can be archived for a minimum of one day to a maximum of 365 days.
 * <p>
 * Example with a archive policy for SQS:
 * <p>
 * <blockquote><pre>
 * Role role;
 * 
 * Topic topic = Topic.Builder.create(this, "MyTopic")
 *         .fifo(true)
 *         .messageRetentionPeriodInDays(7)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: The <code>messageRetentionPeriodInDays</code> property is only available for FIFO topics.
 */
package software.amazon.awscdk.services.sns;
