/**
 * <h1>Amazon ECR Construct Library</h1>
 * <p>
 * This package contains constructs for working with Amazon Elastic Container Registry.
 * <p>
 * <h2>Repositories</h2>
 * <p>
 * Define a repository by creating a new instance of <code>Repository</code>. A repository
 * holds multiple versions of a single container image.
 * <p>
 * <blockquote><pre>
 * Repository repository = new Repository(this, "Repository");
 * </pre></blockquote>
 * <p>
 * <h2>Image scanning</h2>
 * <p>
 * Amazon ECR image scanning helps in identifying software vulnerabilities in your container images.
 * You can manually scan container images stored in Amazon ECR, or you can configure your repositories
 * to scan images when you push them to a repository. To create a new repository to scan on push, simply
 * enable <code>imageScanOnPush</code> in the properties.
 * <p>
 * <blockquote><pre>
 * Repository repository = Repository.Builder.create(this, "Repo")
 *         .imageScanOnPush(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To create an <code>onImageScanCompleted</code> event rule and trigger the event target
 * <p>
 * <blockquote><pre>
 * Repository repository;
 * SomeTarget target;
 * 
 * 
 * repository.onImageScanCompleted("ImageScanComplete").addTarget(target);
 * </pre></blockquote>
 * <p>
 * <h3>Authorization Token</h3>
 * <p>
 * Besides the Amazon ECR APIs, ECR also allows the Docker CLI or a language-specific Docker library to push and pull
 * images from an ECR repository. However, the Docker CLI does not support native IAM authentication methods and
 * additional steps must be taken so that Amazon ECR can authenticate and authorize Docker push and pull requests.
 * More information can be found at at <a href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/Registries.html#registry_auth">Registry Authentication</a>.
 * <p>
 * A Docker authorization token can be obtained using the <code>GetAuthorizationToken</code> ECR API. The following code snippets
 * grants an IAM user access to call this API.
 * <p>
 * <blockquote><pre>
 * User user = new User(this, "User");
 * AuthorizationToken.grantRead(user);
 * </pre></blockquote>
 * <p>
 * If you access images in the <a href="https://gallery.ecr.aws/">Public ECR Gallery</a> as well, it is recommended you authenticate to the registry to benefit from
 * higher rate and bandwidth limits.
 * <p>
 * <blockquote>
 * <p>
 * See <code>Pricing</code> in https://aws.amazon.com/blogs/aws/amazon-ecr-public-a-new-public-container-registry/ and <a href="https://docs.aws.amazon.com/AmazonECR/latest/public/public-service-quotas.html">Service quotas</a>.
 * <p>
 * </blockquote>
 * <p>
 * The following code snippet grants an IAM user access to retrieve an authorization token for the public gallery.
 * <p>
 * <blockquote><pre>
 * User user = new User(this, "User");
 * PublicGalleryAuthorizationToken.grantRead(user);
 * </pre></blockquote>
 * <p>
 * This user can then proceed to login to the registry using one of the <a href="https://docs.aws.amazon.com/AmazonECR/latest/public/public-registries.html#public-registry-auth">authentication methods</a>.
 * <p>
 * <h3>Other Grantee</h3>
 * <p>
 * <h4>grantPush</h4>
 * <p>
 * The grantPush method grants the specified IAM entity (the grantee) permission to push images to the ECR repository. Specifically, it grants permissions for the following actions:
 * <p>
 * <ul>
 * <li>'ecr:CompleteLayerUpload'</li>
 * <li>'ecr:UploadLayerPart'</li>
 * <li>'ecr:InitiateLayerUpload'</li>
 * <li>'ecr:BatchCheckLayerAvailability'</li>
 * <li>'ecr:PutImage'</li>
 * <li>'ecr:GetAuthorizationToken'</li>
 * </ul>
 * <p>
 * Here is an example of granting a user push permissions:
 * <p>
 * <blockquote><pre>
 * Repository repository;
 * 
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("codebuild.amazonaws.com"))
 *         .build();
 * repository.grantPush(role);
 * </pre></blockquote>
 * <p>
 * <h4>grantPull</h4>
 * <p>
 * The grantPull method grants the specified IAM entity (the grantee) permission to pull images from the ECR repository. Specifically, it grants permissions for the following actions:
 * <p>
 * <ul>
 * <li>'ecr:BatchCheckLayerAvailability'</li>
 * <li>'ecr:GetDownloadUrlForLayer'</li>
 * <li>'ecr:BatchGetImage'</li>
 * <li>'ecr:GetAuthorizationToken'</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Repository repository;
 * 
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("codebuild.amazonaws.com"))
 *         .build();
 * repository.grantPull(role);
 * </pre></blockquote>
 * <p>
 * <h4>grantPullPush</h4>
 * <p>
 * The grantPullPush method grants the specified IAM entity (the grantee) permission to both pull and push images from/to the ECR repository. Specifically, it grants permissions for all the actions required for pull and push permissions.
 * <p>
 * Here is an example of granting a user both pull and push permissions:
 * <p>
 * <blockquote><pre>
 * Repository repository;
 * 
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("codebuild.amazonaws.com"))
 *         .build();
 * repository.grantPullPush(role);
 * </pre></blockquote>
 * <p>
 * By using these methods, you can grant specific operational permissions on the ECR repository to IAM entities. This allows for proper management of access to the repository and ensures security.
 * <p>
 * <h3>Image tag immutability</h3>
 * <p>
 * You can set tag immutability on images in our repository using the <code>imageTagMutability</code> construct prop.
 * <p>
 * <blockquote><pre>
 * Repository.Builder.create(this, "Repo").imageTagMutability(TagMutability.IMMUTABLE).build();
 * </pre></blockquote>
 * <p>
 * <h3>Encryption</h3>
 * <p>
 * By default, Amazon ECR uses server-side encryption with Amazon S3-managed encryption keys which encrypts your data at rest using an AES-256 encryption algorithm. For more control over the encryption for your Amazon ECR repositories, you can use server-side encryption with KMS keys stored in AWS Key Management Service (AWS KMS). Read more about this feature in the <a href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/encryption-at-rest.html">ECR Developer Guide</a>.
 * <p>
 * When you use AWS KMS to encrypt your data, you can either use the default AWS managed key, which is managed by Amazon ECR, by specifying <code>RepositoryEncryption.KMS</code> in the <code>encryption</code> property. Or specify your own customer managed KMS key, by specifying the <code>encryptionKey</code> property.
 * <p>
 * When <code>encryptionKey</code> is set, the <code>encryption</code> property must be <code>KMS</code> or empty.
 * <p>
 * In the case <code>encryption</code> is set to <code>KMS</code> but no <code>encryptionKey</code> is set, an AWS managed KMS key is used.
 * <p>
 * <blockquote><pre>
 * Repository.Builder.create(this, "Repo")
 *         .encryption(RepositoryEncryption.KMS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Otherwise, a customer-managed KMS key is used if <code>encryptionKey</code> was set and <code>encryption</code> was optionally set to <code>KMS</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * Repository.Builder.create(this, "Repo")
 *         .encryptionKey(new Key(this, "Key"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Automatically clean up repositories</h2>
 * <p>
 * You can set life cycle rules to automatically clean up old images from your
 * repository. The first life cycle rule that matches an image will be applied
 * against that image. For example, the following deletes images older than
 * 30 days, while keeping all images tagged with prod (note that the order
 * is important here):
 * <p>
 * <blockquote><pre>
 * Repository repository;
 * 
 * repository.addLifecycleRule(LifecycleRule.builder().tagPrefixList(List.of("prod")).maxImageCount(9999).build());
 * repository.addLifecycleRule(LifecycleRule.builder().maxImageAge(Duration.days(30)).build());
 * </pre></blockquote>
 * <p>
 * When using <code>tagPatternList</code>, an image is successfully matched if it matches
 * the wildcard filter.
 * <p>
 * <blockquote><pre>
 * Repository repository;
 * 
 * repository.addLifecycleRule(LifecycleRule.builder().tagPatternList(List.of("prod*")).maxImageCount(9999).build());
 * </pre></blockquote>
 * <p>
 * <h3>Repository deletion</h3>
 * <p>
 * When a repository is removed from a stack (or the stack is deleted), the ECR
 * repository will be removed according to its removal policy (which by default will
 * simply orphan the repository and leave it in your AWS account). If the removal
 * policy is set to <code>RemovalPolicy.DESTROY</code>, the repository will be deleted as long
 * as it does not contain any images.
 * <p>
 * To override this and force all images to get deleted during repository deletion,
 * enable the <code>emptyOnDelete</code> option as well as setting the removal policy to
 * <code>RemovalPolicy.DESTROY</code>.
 * <p>
 * <blockquote><pre>
 * Repository repository = Repository.Builder.create(this, "MyTempRepo")
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .emptyOnDelete(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Managing the Resource Policy</h2>
 * <p>
 * You can add statements to the resource policy of the repository using the
 * <code>addToResourcePolicy</code> method. However, be advised that you must not include
 * a <code>resources</code> section in the <code>PolicyStatement</code>.
 * <p>
 * <blockquote><pre>
 * Repository repository;
 * 
 * repository.addToResourcePolicy(PolicyStatement.Builder.create()
 *         .actions(List.of("ecr:GetDownloadUrlForLayer"))
 *         // resources: ['*'], // not currently allowed!
 *         .principals(List.of(new AnyPrincipal()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>CloudWatch event rules</h2>
 * <p>
 * You can publish repository events to a CloudWatch event rule with <code>onEvent</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * import software.amazon.awscdk.services.events.targets.LambdaFunction;
 * 
 * 
 * Repository repo = new Repository(this, "Repo");
 * Function lambdaHandler = Function.Builder.create(this, "LambdaFunction")
 *         .runtime(Runtime.PYTHON_3_12)
 *         .code(Code.fromInline("# dummy func"))
 *         .handler("index.handler")
 *         .build();
 * 
 * repo.onEvent("OnEventTargetLambda", OnEventOptions.builder()
 *         .target(new LambdaFunction(lambdaHandler))
 *         .build());
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ecr;
