/**
 * <h1>Amazon EventBridge Construct Library</h1>
 * <p>
 * Amazon EventBridge delivers a near real-time stream of system events that
 * describe changes in AWS resources. For example, an AWS CodePipeline emits the
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/event-types.html#codepipeline-event-type">State
 * Change</a>
 * event when the pipeline changes its state.
 * <p>
 * <ul>
 * <li><strong>Events</strong>: An event indicates a change in your AWS environment. AWS resources
 * can generate events when their state changes. For example, Amazon EC2
 * generates an event when the state of an EC2 instance changes from pending to
 * running, and Amazon EC2 Auto Scaling generates events when it launches or
 * terminates instances. AWS CloudTrail publishes events when you make API calls.
 * You can generate custom application-level events and publish them to
 * EventBridge. You can also set up scheduled events that are generated on
 * a periodic basis. For a list of services that generate events, and sample
 * events from each service, see <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/event-types.html">EventBridge Event Examples From Each
 * Supported
 * Service</a>.</li>
 * <li><strong>Targets</strong>: A target processes events. Targets can include Amazon EC2
 * instances, AWS Lambda functions, Kinesis streams, Amazon ECS tasks, Step
 * Functions state machines, Amazon SNS topics, Amazon SQS queues, Amazon CloudWatch LogGroups, and built-in
 * targets. A target receives events in JSON format.</li>
 * <li><strong>Rules</strong>: A rule matches incoming events and routes them to targets for
 * processing. A single rule can route to multiple targets, all of which are
 * processed in parallel. Rules are not processed in a particular order. This
 * enables different parts of an organization to look for and process the events
 * that are of interest to them. A rule can customize the JSON sent to the
 * target, by passing only certain parts or by overwriting it with a constant.</li>
 * <li><strong>EventBuses</strong>: An event bus can receive events from your own custom applications
 * or it can receive events from applications and services created by AWS SaaS partners.
 * See <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/create-event-bus.html">Creating an Event Bus</a>.</li>
 * </ul>
 * <p>
 * <h2>Rule</h2>
 * <p>
 * The <code>Rule</code> construct defines an EventBridge rule which monitors an
 * event based on an <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/filtering-examples-structure.html">event
 * pattern</a>
 * and invoke <strong>event targets</strong> when the pattern is matched against a triggered
 * event. Event targets are objects that implement the <code>IRuleTarget</code> interface.
 * <p>
 * Normally, you will use one of the <code>source.onXxx(name[, target[, options]]) -&gt; Rule</code> methods on the event source to define an event rule associated with
 * the specific activity. You can targets either via props, or add targets using
 * <code>rule.addTarget</code>.
 * <p>
 * For example, to define an rule that triggers a CodeBuild project build when a
 * commit is pushed to the "master" branch of a CodeCommit repository:
 * <p>
 * <blockquote><pre>
 * Repository repo;
 * Project project;
 * 
 * 
 * Rule onCommitRule = repo.onCommit("OnCommit", OnCommitOptions.builder()
 *         .target(new CodeBuildProject(project))
 *         .branches(List.of("master"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can add additional targets, with optional <a href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_InputTransformer.html">input
 * transformer</a>
 * using <code>eventRule.addTarget(target[, input])</code>. For example, we can add a SNS
 * topic target which formats a human-readable message for the commit.
 * <p>
 * For example, this adds an SNS topic as a target:
 * <p>
 * <blockquote><pre>
 * Rule onCommitRule;
 * Topic topic;
 * 
 * 
 * onCommitRule.addTarget(SnsTopic.Builder.create(topic)
 *         .message(RuleTargetInput.fromText(String.format("A commit was pushed to the repository %s on branch %s", ReferenceEvent.getRepositoryName(), ReferenceEvent.getReferenceName())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Or using an Object:
 * <p>
 * <blockquote><pre>
 * Rule onCommitRule;
 * Topic topic;
 * 
 * 
 * onCommitRule.addTarget(SnsTopic.Builder.create(topic)
 *         .message(RuleTargetInput.fromObject(Map.of(
 *                 "DataType", String.format("custom_%s", EventField.fromPath("$.detail-type")))))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To define a pattern, use the matcher API, which provides a number of factory methods to declare different logical predicates. For example, to match all S3 events for objects larger than 1024 bytes, stored using one of the storage classes Glacier, Glacier IR or Deep Archive and coming from any region other than the AWS GovCloud ones:
 * <p>
 * <blockquote><pre>
 * Rule rule = Rule.Builder.create(this, "rule")
 *         .eventPattern(EventPattern.builder()
 *                 .detail(Map.of(
 *                         "object", Map.of(
 *                                 // Matchers may appear at any level
 *                                 "size", Match.greaterThan(1024)),
 * 
 *                         // 'OR' condition
 *                         "source-storage-class", Match.anyOf(Match.prefix("GLACIER"), Match.exactString("DEEP_ARCHIVE"))))
 *                 .detailType(Match.equalsIgnoreCase("object created"))
 * 
 *                 // If you prefer, you can use a low level array of strings, as directly consumed by EventBridge
 *                 .source(List.of("aws.s3"))
 * 
 *                 .region(Match.anythingButPrefix("us-gov"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Scheduling</h2>
 * <p>
 * You can configure a Rule to run on a schedule (cron or rate).
 * Rate must be specified in minutes, hours or days.
 * <p>
 * The following example runs a task every day at 4am:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.events.Rule;
 * import software.amazon.awscdk.services.events.Schedule;
 * import software.amazon.awscdk.services.events.targets.EcsTask;
 * import software.amazon.awscdk.services.ecs.Cluster;
 * import software.amazon.awscdk.services.ecs.TaskDefinition;
 * import software.amazon.awscdk.services.iam.Role;
 * 
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * Role role;
 * 
 * 
 * EcsTask ecsTaskTarget = EcsTask.Builder.create().cluster(cluster).taskDefinition(taskDefinition).role(role).build();
 * 
 * Rule.Builder.create(this, "ScheduleRule")
 *         .schedule(Schedule.cron(CronOptions.builder().minute("0").hour("4").build()))
 *         .targets(List.of(ecsTaskTarget))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to specify Fargate platform version, set <code>platformVersion</code> in EcsTask's props like the following example:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * Role role;
 * 
 * 
 * FargatePlatformVersion platformVersion = FargatePlatformVersion.VERSION1_4;
 * EcsTask ecsTaskTarget = EcsTask.Builder.create().cluster(cluster).taskDefinition(taskDefinition).role(role).platformVersion(platformVersion).build();
 * </pre></blockquote>
 * <p>
 * <h2>Event Targets</h2>
 * <p>
 * The <code>aws-cdk-lib/aws-events-targets</code> module includes classes that implement the <code>IRuleTarget</code>
 * interface for various AWS services.
 * <p>
 * See the README of the <a href="https://github.com/aws/aws-cdk/tree/main/packages/aws-cdk-lib/aws-events-targets"><code>aws-cdk-lib/aws-events-targets</code></a> module for more information on supported targets.
 * <p>
 * <h3>Cross-account and cross-region targets</h3>
 * <p>
 * It's possible to have the source of the event and a target in separate AWS accounts and regions:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.App;
 * import software.amazon.awscdk.Stack;
 * import software.amazon.awscdk.services.codebuild.*;
 * import software.amazon.awscdk.services.codecommit.*;
 * import software.amazon.awscdk.services.events.targets.*;
 * 
 * App app = new App();
 * 
 * String account1 = "11111111111";
 * String account2 = "22222222222";
 * 
 * Stack stack1 = Stack.Builder.create(app, "Stack1").env(Environment.builder().account(account1).region("us-west-1").build()).build();
 * Repository repo = Repository.Builder.create(stack1, "Repository")
 *         .repositoryName("myrepository")
 *         .build();
 * 
 * Stack stack2 = Stack.Builder.create(app, "Stack2").env(Environment.builder().account(account2).region("us-east-1").build()).build();
 * Project project = Project.Builder.create(stack2, "Project").build();
 * 
 * repo.onCommit("OnCommit", OnCommitOptions.builder()
 *         .target(new CodeBuildProject(project))
 *         .build());
 * </pre></blockquote>
 * <p>
 * In this situation, the CDK will wire the 2 accounts together:
 * <p>
 * <ul>
 * <li>It will generate a rule in the source stack with the event bus of the target account as the target</li>
 * <li>It will generate a rule in the target stack, with the provided target</li>
 * <li>It will generate a separate stack that gives the source account permissions to publish events
 * to the event bus of the target account in the given region,
 * and make sure its deployed before the source stack</li>
 * </ul>
 * <p>
 * For more information, see the
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html">AWS documentation on cross-account events</a>.
 * <p>
 * <h2>Archiving</h2>
 * <p>
 * It is possible to archive all or some events sent to an event bus. It is then possible to <a href="https://aws.amazon.com/blogs/aws/new-archive-and-replay-events-with-amazon-eventbridge/">replay these events</a>.
 * <p>
 * <blockquote><pre>
 * EventBus bus = EventBus.Builder.create(this, "bus")
 *         .eventBusName("MyCustomEventBus")
 *         .build();
 * 
 * bus.archive("MyArchive", BaseArchiveProps.builder()
 *         .archiveName("MyCustomEventBusArchive")
 *         .description("MyCustomerEventBus Archive")
 *         .eventPattern(EventPattern.builder()
 *                 .account(List.of(Stack.of(this).getAccount()))
 *                 .build())
 *         .retention(Duration.days(365))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Granting PutEvents to an existing EventBus</h2>
 * <p>
 * To import an existing EventBus into your CDK application, use <code>EventBus.fromEventBusArn</code>, <code>EventBus.fromEventBusAttributes</code>
 * or <code>EventBus.fromEventBusName</code> factory method.
 * <p>
 * Then, you can use the <code>grantPutEventsTo</code> method to grant <code>event:PutEvents</code> to the eventBus.
 * <p>
 * <blockquote><pre>
 * Function lambdaFunction;
 * 
 * 
 * IEventBus eventBus = EventBus.fromEventBusArn(this, "ImportedEventBus", "arn:aws:events:us-east-1:111111111:event-bus/my-event-bus");
 * 
 * // now you can just call methods on the eventbus
 * eventBus.grantPutEventsTo(lambdaFunction);
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.events;
