/**
 * <h1>Amazon CloudFront Construct Library</h1>
 * <p>
 * Amazon CloudFront is a web service that speeds up distribution of your static and dynamic web content, such as .html, .css, .js, and image files, to
 * your users. CloudFront delivers your content through a worldwide network of data centers called edge locations. When a user requests content that
 * you're serving with CloudFront, the user is routed to the edge location that provides the lowest latency, so that content is delivered with the best
 * possible performance.
 * <p>
 * <h2>Distribution API</h2>
 * <p>
 * The <code>Distribution</code> API is currently being built to replace the existing <code>CloudFrontWebDistribution</code> API. The <code>Distribution</code> API is optimized for the
 * most common use cases of CloudFront distributions (e.g., single origin and behavior, few customizations) while still providing the ability for more
 * advanced use cases. The API focuses on simplicity for the common use cases, and convenience methods for creating the behaviors and origins necessary
 * for more complex use cases.
 * <p>
 * <h3>Creating a distribution</h3>
 * <p>
 * CloudFront distributions deliver your content from one or more origins; an origin is the location where you store the original version of your
 * content. Origins can be created from S3 buckets or a custom origin (HTTP server). Constructs to define origins are in the <code>aws-cdk-lib/aws-cloudfront-origins</code> module.
 * <p>
 * Each distribution has a default behavior which applies to all requests to that distribution, and routes requests to a primary origin.
 * Additional behaviors may be specified for an origin with a given URL path pattern. Behaviors allow routing with multiple origins,
 * controlling which HTTP methods to support, whether to require users to use HTTPS, and what query strings or cookies to forward to your origin,
 * among other settings.
 * <p>
 * <h4>From an S3 Bucket</h4>
 * <p>
 * An S3 bucket can be added as an origin. If the bucket is configured as a website endpoint, the distribution can use S3 redirects and S3 custom error
 * documents.
 * <p>
 * <blockquote><pre>
 * // Creates a distribution from an S3 bucket.
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new S3Origin(myBucket)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The above will treat the bucket differently based on if <code>IBucket.isWebsite</code> is set or not. If the bucket is configured as a website, the bucket is
 * treated as an HTTP origin, and the built-in S3 redirects and error pages can be used. Otherwise, the bucket is handled as a bucket origin and
 * CloudFront's redirect and error handling will be used. In the latter case, the Origin will create an origin access identity and grant it access to the
 * underlying bucket. This can be used in conjunction with a bucket that is not public to require that your users access your content using CloudFront
 * URLs and not S3 URLs directly.
 * <p>
 * <h4>ELBv2 Load Balancer</h4>
 * <p>
 * An Elastic Load Balancing (ELB) v2 load balancer may be used as an origin. In order for a load balancer to serve as an origin, it must be publicly
 * accessible (<code>internetFacing</code> is true). Both Application and Network load balancers are supported.
 * <p>
 * <blockquote><pre>
 * // Creates a distribution from an ELBv2 load balancer
 * Vpc vpc;
 * 
 * // Create an application load balancer in a VPC. 'internetFacing' must be 'true'
 * // for CloudFront to access the load balancer and use it as an origin.
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB")
 *         .vpc(vpc)
 *         .internetFacing(true)
 *         .build();
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new LoadBalancerV2Origin(lb)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>From an HTTP endpoint</h4>
 * <p>
 * Origins can also be created from any other HTTP endpoint, given the domain name, and optionally, other origin properties.
 * <p>
 * <blockquote><pre>
 * // Creates a distribution from an HTTP endpoint
 * // Creates a distribution from an HTTP endpoint
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new HttpOrigin("www.example.com")).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Domain Names and Certificates</h3>
 * <p>
 * When you create a distribution, CloudFront assigns a domain name for the distribution, for example: <code>d111111abcdef8.cloudfront.net</code>; this value can
 * be retrieved from <code>distribution.distributionDomainName</code>. CloudFront distributions use a default certificate (<code>*.cloudfront.net</code>) to support HTTPS by
 * default. If you want to use your own domain name, such as <code>www.example.com</code>, you must associate a certificate with your distribution that contains
 * your domain name, and provide one (or more) domain names from the certificate for the distribution.
 * <p>
 * The certificate must be present in the AWS Certificate Manager (ACM) service in the US East (N. Virginia) region; the certificate
 * may either be created by ACM, or created elsewhere and imported into ACM. When a certificate is used, the distribution will support HTTPS connections
 * from SNI only and a minimum protocol version of TLSv1.2_2021 if the <code>&#64;aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021</code> feature flag is set, and TLSv1.2_2019 otherwise.
 * <p>
 * <blockquote><pre>
 * // To use your own domain name in a Distribution, you must associate a certificate
 * import software.amazon.awscdk.services.certificatemanager.*;
 * import software.amazon.awscdk.services.route53.*;
 * 
 * HostedZone hostedZone;
 * 
 * Bucket myBucket;
 * 
 * Certificate myCertificate = Certificate.Builder.create(this, "mySiteCert")
 *         .domainName("www.example.com")
 *         .validation(CertificateValidation.fromDns(hostedZone))
 *         .build();
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new S3Origin(myBucket)).build())
 *         .domainNames(List.of("www.example.com"))
 *         .certificate(myCertificate)
 *         .build();
 * </pre></blockquote>
 * <p>
 * However, you can customize the minimum protocol version for the certificate while creating the distribution using <code>minimumProtocolVersion</code> property.
 * <p>
 * <blockquote><pre>
 * // Create a Distribution with a custom domain name and a minimum protocol version.
 * Bucket myBucket;
 * 
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new S3Origin(myBucket)).build())
 *         .domainNames(List.of("www.example.com"))
 *         .minimumProtocolVersion(SecurityPolicyProtocol.TLS_V1_2016)
 *         .sslSupportMethod(SSLMethod.SNI)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Cross Region Certificates</h4>
 * <p>
 * <blockquote>
 * <p>
 * <strong>This feature is currently experimental</strong>
 * <p>
 * </blockquote>
 * <p>
 * You can enable the Stack property <code>crossRegionReferences</code>
 * in order to access resources in a different stack <em>and</em> region. With this feature flag
 * enabled it is possible to do something like creating a CloudFront distribution in <code>us-east-2</code> and
 * an ACM certificate in <code>us-east-1</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.*;
 * import software.amazon.awscdk.services.route53.*;
 * 
 * App app;
 * 
 * 
 * Stack stack1 = Stack.Builder.create(app, "Stack1")
 *         .env(Environment.builder()
 *                 .region("us-east-1")
 *                 .build())
 *         .crossRegionReferences(true)
 *         .build();
 * Certificate cert = Certificate.Builder.create(stack1, "Cert")
 *         .domainName("*.example.com")
 *         .validation(CertificateValidation.fromDns(PublicHostedZone.fromHostedZoneId(stack1, "Zone", "Z0329774B51CGXTDQV3X")))
 *         .build();
 * 
 * Stack stack2 = Stack.Builder.create(app, "Stack2")
 *         .env(Environment.builder()
 *                 .region("us-east-2")
 *                 .build())
 *         .crossRegionReferences(true)
 *         .build();
 * Distribution.Builder.create(stack2, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new HttpOrigin("example.com"))
 *                 .build())
 *         .domainNames(List.of("dev.example.com"))
 *         .certificate(cert)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Multiple Behaviors &amp; Origins</h3>
 * <p>
 * Each distribution has a default behavior which applies to all requests to that distribution; additional behaviors may be specified for a
 * given URL path pattern. Behaviors allow routing with multiple origins, controlling which HTTP methods to support, whether to require users to
 * use HTTPS, and what query strings or cookies to forward to your origin, among others.
 * <p>
 * The properties of the default behavior can be adjusted as part of the distribution creation. The following example shows configuring the HTTP
 * methods and viewer protocol policy of the cache.
 * <p>
 * <blockquote><pre>
 * // Create a Distribution with configured HTTP methods and viewer protocol policy of the cache.
 * Bucket myBucket;
 * 
 * Distribution myWebDistribution = Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new S3Origin(myBucket))
 *                 .allowedMethods(AllowedMethods.ALLOW_ALL)
 *                 .viewerProtocolPolicy(ViewerProtocolPolicy.REDIRECT_TO_HTTPS)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Additional behaviors can be specified at creation, or added after the initial creation. Each additional behavior is associated with an origin,
 * and enable customization for a specific set of resources based on a URL path pattern. For example, we can add a behavior to <code>myWebDistribution</code> to
 * override the default viewer protocol policy for all of the images.
 * <p>
 * <blockquote><pre>
 * // Add a behavior to a Distribution after initial creation.
 * Bucket myBucket;
 * Distribution myWebDistribution;
 * 
 * myWebDistribution.addBehavior("/images/*.jpg", new S3Origin(myBucket), AddBehaviorOptions.builder()
 *         .viewerProtocolPolicy(ViewerProtocolPolicy.REDIRECT_TO_HTTPS)
 *         .build());
 * </pre></blockquote>
 * <p>
 * These behaviors can also be specified at distribution creation time.
 * <p>
 * <blockquote><pre>
 * // Create a Distribution with additional behaviors at creation time.
 * Bucket myBucket;
 * 
 * S3Origin bucketOrigin = new S3Origin(myBucket);
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(bucketOrigin)
 *                 .allowedMethods(AllowedMethods.ALLOW_ALL)
 *                 .viewerProtocolPolicy(ViewerProtocolPolicy.REDIRECT_TO_HTTPS)
 *                 .build())
 *         .additionalBehaviors(Map.of(
 *                 "/images/*.jpg", BehaviorOptions.builder()
 *                         .origin(bucketOrigin)
 *                         .viewerProtocolPolicy(ViewerProtocolPolicy.REDIRECT_TO_HTTPS)
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Customizing Cache Keys and TTLs with Cache Policies</h3>
 * <p>
 * You can use a cache policy to improve your cache hit ratio by controlling the values (URL query strings, HTTP headers, and cookies)
 * that are included in the cache key, and/or adjusting how long items remain in the cache via the time-to-live (TTL) settings.
 * CloudFront provides some predefined cache policies, known as managed policies, for common use cases. You can use these managed policies,
 * or you can create your own cache policy that’s specific to your needs.
 * See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html for more details.
 * <p>
 * <blockquote><pre>
 * // Using an existing cache policy for a Distribution
 * S3Origin bucketOrigin;
 * 
 * Distribution.Builder.create(this, "myDistManagedPolicy")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(bucketOrigin)
 *                 .cachePolicy(CachePolicy.CACHING_OPTIMIZED)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * // Creating a custom cache policy for a Distribution -- all parameters optional
 * S3Origin bucketOrigin;
 * 
 * CachePolicy myCachePolicy = CachePolicy.Builder.create(this, "myCachePolicy")
 *         .cachePolicyName("MyPolicy")
 *         .comment("A default policy")
 *         .defaultTtl(Duration.days(2))
 *         .minTtl(Duration.minutes(1))
 *         .maxTtl(Duration.days(10))
 *         .cookieBehavior(CacheCookieBehavior.all())
 *         .headerBehavior(CacheHeaderBehavior.allowList("X-CustomHeader"))
 *         .queryStringBehavior(CacheQueryStringBehavior.denyList("username"))
 *         .enableAcceptEncodingGzip(true)
 *         .enableAcceptEncodingBrotli(true)
 *         .build();
 * Distribution.Builder.create(this, "myDistCustomPolicy")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(bucketOrigin)
 *                 .cachePolicy(myCachePolicy)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Customizing Origin Requests with Origin Request Policies</h3>
 * <p>
 * When CloudFront makes a request to an origin, the URL path, request body (if present), and a few standard headers are included.
 * Other information from the viewer request, such as URL query strings, HTTP headers, and cookies, is not included in the origin request by default.
 * You can use an origin request policy to control the information that’s included in an origin request.
 * CloudFront provides some predefined origin request policies, known as managed policies, for common use cases. You can use these managed policies,
 * or you can create your own origin request policy that’s specific to your needs.
 * See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html for more details.
 * <p>
 * <blockquote><pre>
 * // Using an existing origin request policy for a Distribution
 * S3Origin bucketOrigin;
 * 
 * Distribution.Builder.create(this, "myDistManagedPolicy")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(bucketOrigin)
 *                 .originRequestPolicy(OriginRequestPolicy.CORS_S3_ORIGIN)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * // Creating a custom origin request policy for a Distribution -- all parameters optional
 * S3Origin bucketOrigin;
 * 
 * OriginRequestPolicy myOriginRequestPolicy = OriginRequestPolicy.Builder.create(this, "OriginRequestPolicy")
 *         .originRequestPolicyName("MyPolicy")
 *         .comment("A default policy")
 *         .cookieBehavior(OriginRequestCookieBehavior.none())
 *         .headerBehavior(OriginRequestHeaderBehavior.all("CloudFront-Is-Android-Viewer"))
 *         .queryStringBehavior(OriginRequestQueryStringBehavior.allowList("username"))
 *         .build();
 * 
 * Distribution.Builder.create(this, "myDistCustomPolicy")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(bucketOrigin)
 *                 .originRequestPolicy(myOriginRequestPolicy)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Customizing Response Headers with Response Headers Policies</h3>
 * <p>
 * You can configure CloudFront to add one or more HTTP headers to the responses that it sends to viewers (web browsers or other clients), without making any changes to the origin or writing any code.
 * To specify the headers that CloudFront adds to HTTP responses, you use a response headers policy. CloudFront adds the headers regardless of whether it serves the object from the cache or has to retrieve the object from the origin. If the origin response includes one or more of the headers that’s in a response headers policy, the policy can specify whether CloudFront uses the header it received from the origin or overwrites it with the one in the policy.
 * See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/adding-response-headers.html
 * <p>
 * <blockquote><pre>
 * // Using an existing managed response headers policy
 * S3Origin bucketOrigin;
 * 
 * Distribution.Builder.create(this, "myDistManagedPolicy")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(bucketOrigin)
 *                 .responseHeadersPolicy(ResponseHeadersPolicy.CORS_ALLOW_ALL_ORIGINS)
 *                 .build())
 *         .build();
 * 
 * // Creating a custom response headers policy -- all parameters optional
 * ResponseHeadersPolicy myResponseHeadersPolicy = ResponseHeadersPolicy.Builder.create(this, "ResponseHeadersPolicy")
 *         .responseHeadersPolicyName("MyPolicy")
 *         .comment("A default policy")
 *         .corsBehavior(ResponseHeadersCorsBehavior.builder()
 *                 .accessControlAllowCredentials(false)
 *                 .accessControlAllowHeaders(List.of("X-Custom-Header-1", "X-Custom-Header-2"))
 *                 .accessControlAllowMethods(List.of("GET", "POST"))
 *                 .accessControlAllowOrigins(List.of("*"))
 *                 .accessControlExposeHeaders(List.of("X-Custom-Header-1", "X-Custom-Header-2"))
 *                 .accessControlMaxAge(Duration.seconds(600))
 *                 .originOverride(true)
 *                 .build())
 *         .customHeadersBehavior(ResponseCustomHeadersBehavior.builder()
 *                 .customHeaders(List.of(ResponseCustomHeader.builder().header("X-Amz-Date").value("some-value").override(true).build(), ResponseCustomHeader.builder().header("X-Amz-Security-Token").value("some-value").override(false).build()))
 *                 .build())
 *         .securityHeadersBehavior(ResponseSecurityHeadersBehavior.builder()
 *                 .contentSecurityPolicy(ResponseHeadersContentSecurityPolicy.builder().contentSecurityPolicy("default-src https:;").override(true).build())
 *                 .contentTypeOptions(ResponseHeadersContentTypeOptions.builder().override(true).build())
 *                 .frameOptions(ResponseHeadersFrameOptions.builder().frameOption(HeadersFrameOption.DENY).override(true).build())
 *                 .referrerPolicy(ResponseHeadersReferrerPolicy.builder().referrerPolicy(HeadersReferrerPolicy.NO_REFERRER).override(true).build())
 *                 .strictTransportSecurity(ResponseHeadersStrictTransportSecurity.builder().accessControlMaxAge(Duration.seconds(600)).includeSubdomains(true).override(true).build())
 *                 .xssProtection(ResponseHeadersXSSProtection.builder().protection(true).modeBlock(true).reportUri("https://example.com/csp-report").override(true).build())
 *                 .build())
 *         .removeHeaders(List.of("Server"))
 *         .serverTimingSamplingRate(50)
 *         .build();
 * Distribution.Builder.create(this, "myDistCustomPolicy")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(bucketOrigin)
 *                 .responseHeadersPolicy(myResponseHeadersPolicy)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Validating signed URLs or signed cookies with Trusted Key Groups</h3>
 * <p>
 * CloudFront Distribution supports validating signed URLs or signed cookies using key groups.
 * When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed
 * cookies for all requests that match the cache behavior.
 * <p>
 * <blockquote><pre>
 * // Validating signed URLs or signed cookies with Trusted Key Groups
 * 
 * // public key in PEM format
 * String publicKey;
 * 
 * PublicKey pubKey = PublicKey.Builder.create(this, "MyPubKey")
 *         .encodedKey(publicKey)
 *         .build();
 * 
 * KeyGroup keyGroup = KeyGroup.Builder.create(this, "MyKeyGroup")
 *         .items(List.of(pubKey))
 *         .build();
 * 
 * Distribution.Builder.create(this, "Dist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new HttpOrigin("www.example.com"))
 *                 .trustedKeyGroups(List.of(keyGroup))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Lambda&#64;Edge</h3>
 * <p>
 * Lambda&#64;Edge is an extension of AWS Lambda, a compute service that lets you execute
 * functions that customize the content that CloudFront delivers. You can author Node.js
 * or Python functions in the US East (N. Virginia) region, and then execute them in AWS
 * locations globally that are closer to the viewer, without provisioning or managing servers.
 * Lambda&#64;Edge functions are associated with a specific behavior and event type. Lambda&#64;Edge
 * can be used to rewrite URLs, alter responses based on headers or cookies, or authorize
 * requests based on headers or authorization tokens.
 * <p>
 * The following shows a Lambda&#64;Edge function added to the default behavior and triggered
 * on every request:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket;
 * // A Lambda&#64;Edge function added to default behavior of a Distribution
 * // and triggered on every request
 * EdgeFunction myFunc = EdgeFunction.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new S3Origin(myBucket))
 *                 .edgeLambdas(List.of(EdgeLambda.builder()
 *                         .functionVersion(myFunc.getCurrentVersion())
 *                         .eventType(LambdaEdgeEventType.VIEWER_REQUEST)
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> Lambda&#64;Edge functions must be created in the <code>us-east-1</code> region, regardless of the region of the CloudFront distribution and stack.
 * To make it easier to request functions for Lambda&#64;Edge, the <code>EdgeFunction</code> construct can be used.
 * The <code>EdgeFunction</code> construct will automatically request a function in <code>us-east-1</code>, regardless of the region of the current stack.
 * <code>EdgeFunction</code> has the same interface as <code>Function</code> and can be created and used interchangeably.
 * Please note that using <code>EdgeFunction</code> requires that the <code>us-east-1</code> region has been bootstrapped.
 * See https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html for more about bootstrapping regions.
 * <p>
 * </blockquote>
 * <p>
 * If the stack is in <code>us-east-1</code>, a "normal" <code>lambda.Function</code> can be used instead of an <code>EdgeFunction</code>.
 * <p>
 * <blockquote><pre>
 * // Using a lambda Function instead of an EdgeFunction for stacks in `us-east-`.
 * Function myFunc = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If the stack is not in <code>us-east-1</code>, and you need references from different applications on the same account,
 * you can also set a specific stack ID for each Lambda&#64;Edge.
 * <p>
 * <blockquote><pre>
 * // Setting stackIds for EdgeFunctions that can be referenced from different applications
 * // on the same account.
 * EdgeFunction myFunc1 = EdgeFunction.Builder.create(this, "MyFunction1")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler1")))
 *         .stackId("edge-lambda-stack-id-1")
 *         .build();
 * 
 * EdgeFunction myFunc2 = EdgeFunction.Builder.create(this, "MyFunction2")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler2")))
 *         .stackId("edge-lambda-stack-id-2")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Lambda&#64;Edge functions can also be associated with additional behaviors,
 * either at or after Distribution creation time.
 * <p>
 * <blockquote><pre>
 * // Associating a Lambda&#64;Edge function with additional behaviors.
 * 
 * EdgeFunction myFunc;
 * // assigning at Distribution creation
 * Bucket myBucket;
 * 
 * S3Origin myOrigin = new S3Origin(myBucket);
 * Distribution myDistribution = Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(myOrigin).build())
 *         .additionalBehaviors(Map.of(
 *                 "images/*", BehaviorOptions.builder()
 *                         .origin(myOrigin)
 *                         .edgeLambdas(List.of(EdgeLambda.builder()
 *                                 .functionVersion(myFunc.getCurrentVersion())
 *                                 .eventType(LambdaEdgeEventType.ORIGIN_REQUEST)
 *                                 .includeBody(true)
 *                                 .build()))
 *                         .build()))
 *         .build();
 * 
 * // assigning after creation
 * myDistribution.addBehavior("images/*", myOrigin, AddBehaviorOptions.builder()
 *         .edgeLambdas(List.of(EdgeLambda.builder()
 *                 .functionVersion(myFunc.getCurrentVersion())
 *                 .eventType(LambdaEdgeEventType.VIEWER_RESPONSE)
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Adding an existing Lambda&#64;Edge function created in a different stack to a CloudFront distribution.
 * <p>
 * <blockquote><pre>
 * // Adding an existing Lambda&#64;Edge function created in a different stack
 * // to a CloudFront distribution.
 * Bucket s3Bucket;
 * 
 * IVersion functionVersion = Version.fromVersionArn(this, "Version", "arn:aws:lambda:us-east-1:123456789012:function:functionName:1");
 * 
 * Distribution.Builder.create(this, "distro")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new S3Origin(s3Bucket))
 *                 .edgeLambdas(List.of(EdgeLambda.builder()
 *                         .functionVersion(functionVersion)
 *                         .eventType(LambdaEdgeEventType.VIEWER_REQUEST)
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>CloudFront Function</h3>
 * <p>
 * You can also deploy CloudFront functions and add them to a CloudFront distribution.
 * <p>
 * <blockquote><pre>
 * Bucket s3Bucket;
 * // Add a cloudfront Function to a Distribution
 * Function cfFunction = Function.Builder.create(this, "Function")
 *         .code(FunctionCode.fromInline("function handler(event) { return event.request }"))
 *         .runtime(FunctionRuntime.JS_2_0)
 *         .build();
 * Distribution.Builder.create(this, "distro")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new S3Origin(s3Bucket))
 *                 .functionAssociations(List.of(FunctionAssociation.builder()
 *                         .function(cfFunction)
 *                         .eventType(FunctionEventType.VIEWER_REQUEST)
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * It will auto-generate the name of the function and deploy it to the <code>live</code> stage.
 * <p>
 * Additionally, you can load the function's code from a file using the <code>FunctionCode.fromFile()</code> method.
 * <p>
 * If you set <code>autoPublish</code> to false, the function will not be automatically published to the LIVE stage when it’s created.
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "Function")
 *         .code(FunctionCode.fromInline("function handler(event) { return event.request }"))
 *         .runtime(FunctionRuntime.JS_2_0)
 *         .autoPublish(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Key Value Store</h3>
 * <p>
 * A CloudFront Key Value Store can be created and optionally have data imported from a JSON file
 * by default.
 * <p>
 * To create an empty Key Value Store:
 * <p>
 * <blockquote><pre>
 * KeyValueStore store = new KeyValueStore(this, "KeyValueStore");
 * </pre></blockquote>
 * <p>
 * To also include an initial set of values, the <code>source</code> property can be specified, either from a
 * local file or an inline string. For the structure of this file, see <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/kvs-with-functions-create-s3-kvp.html">Creating a file of key value pairs</a>.
 * <p>
 * <blockquote><pre>
 * KeyValueStore storeAsset = KeyValueStore.Builder.create(this, "KeyValueStoreAsset")
 *         .keyValueStoreName("KeyValueStoreAsset")
 *         .source(ImportSource.fromAsset("path-to-data.json"))
 *         .build();
 * 
 * KeyValueStore storeInline = KeyValueStore.Builder.create(this, "KeyValueStoreInline")
 *         .keyValueStoreName("KeyValueStoreInline")
 *         .source(ImportSource.fromInline(JSON.stringify(Map.of(
 *                 "data", List.of(Map.of(
 *                         "key", "key1",
 *                         "value", "value1"), Map.of(
 *                         "key", "key2",
 *                         "value", "value2"))))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The Key Value Store can then be associated to a function using the <code>cloudfront-js-2.0</code> runtime
 * or newer:
 * <p>
 * <blockquote><pre>
 * KeyValueStore store = new KeyValueStore(this, "KeyValueStore");
 * Function.Builder.create(this, "Function")
 *         .code(FunctionCode.fromInline("function handler(event) { return event.request }"))
 *         // Note that JS_2_0 must be used for Key Value Store support
 *         .runtime(FunctionRuntime.JS_2_0)
 *         .keyValueStore(store)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Logging</h3>
 * <p>
 * You can configure CloudFront to create log files that contain detailed information about every user request that CloudFront receives.
 * The logs can go to either an existing bucket, or a bucket will be created for you.
 * <p>
 * <blockquote><pre>
 * // Configure logging for Distributions
 * 
 * // Simplest form - creates a new bucket and logs to it.
 * // Configure logging for Distributions
 * // Simplest form - creates a new bucket and logs to it.
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new HttpOrigin("www.example.com")).build())
 *         .enableLogging(true)
 *         .build();
 * 
 * // You can optionally log to a specific bucket, configure whether cookies are logged, and give the log files a prefix.
 * // You can optionally log to a specific bucket, configure whether cookies are logged, and give the log files a prefix.
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new HttpOrigin("www.example.com")).build())
 *         .enableLogging(true) // Optional, this is implied if logBucket is specified
 *         .logBucket(Bucket.Builder.create(this, "LogBucket")
 *                 .objectOwnership(ObjectOwnership.OBJECT_WRITER)
 *                 .build())
 *         .logFilePrefix("distribution-access-logs/")
 *         .logIncludesCookies(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>CloudFront Distribution Metrics</h3>
 * <p>
 * You can view operational metrics about your CloudFront distributions.
 * <p>
 * <h4>Default CloudFront Distribution Metrics</h4>
 * <p>
 * The <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/viewing-cloudfront-metrics.html#monitoring-console.distributions">following metrics are available by default</a> for all CloudFront distributions:
 * <p>
 * <ul>
 * <li>Total requests: The total number of viewer requests received by CloudFront for all HTTP methods and for both HTTP and HTTPS requests.</li>
 * <li>Total bytes uploaded: The total number of bytes that viewers uploaded to your origin with CloudFront, using POST and PUT requests.</li>
 * <li>Total bytes downloaded: The total number of bytes downloaded by viewers for GET, HEAD, and OPTIONS requests.</li>
 * <li>Total error rate: The percentage of all viewer requests for which the response's HTTP status code was 4xx or 5xx.</li>
 * <li>4xx error rate: The percentage of all viewer requests for which the response's HTTP status code was 4xx.</li>
 * <li>5xx error rate: The percentage of all viewer requests for which the response's HTTP status code was 5xx.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Distribution dist = Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new HttpOrigin("www.example.com")).build())
 *         .build();
 * 
 * // Retrieving default distribution metrics
 * Metric requestsMetric = dist.metricRequests();
 * Metric bytesUploadedMetric = dist.metricBytesUploaded();
 * Metric bytesDownloadedMetric = dist.metricBytesDownloaded();
 * Metric totalErrorRateMetric = dist.metricTotalErrorRate();
 * Metric http4xxErrorRateMetric = dist.metric4xxErrorRate();
 * Metric http5xxErrorRateMetric = dist.metric5xxErrorRate();
 * </pre></blockquote>
 * <p>
 * <h4>Additional CloudFront Distribution Metrics</h4>
 * <p>
 * You can enable <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/viewing-cloudfront-metrics.html#monitoring-console.distributions-additional">additional CloudFront distribution metrics</a>, which include the following metrics:
 * <p>
 * <ul>
 * <li>4xx and 5xx error rates: View 4xx and 5xx error rates by the specific HTTP status code, as a percentage of total requests.</li>
 * <li>Origin latency: See the total time spent from when CloudFront receives a request to when it provides a response to the network (not the viewer), for responses that are served from the origin, not the CloudFront cache.</li>
 * <li>Cache hit rate: View cache hits as a percentage of total cacheable requests, excluding errors.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Distribution dist = Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new HttpOrigin("www.example.com")).build())
 *         .publishAdditionalMetrics(true)
 *         .build();
 * 
 * // Retrieving additional distribution metrics
 * Metric latencyMetric = dist.metricOriginLatency();
 * Metric cacheHitRateMetric = dist.metricCacheHitRate();
 * Metric http401ErrorRateMetric = dist.metric401ErrorRate();
 * Metric http403ErrorRateMetric = dist.metric403ErrorRate();
 * Metric http404ErrorRateMetric = dist.metric404ErrorRate();
 * Metric http502ErrorRateMetric = dist.metric502ErrorRate();
 * Metric http503ErrorRateMetric = dist.metric503ErrorRate();
 * Metric http504ErrorRateMetric = dist.metric504ErrorRate();
 * </pre></blockquote>
 * <p>
 * <h3>HTTP Versions</h3>
 * <p>
 * You can configure CloudFront to use a particular version of the HTTP protocol. By default,
 * newly created distributions use HTTP/2 but can be configured to use both HTTP/2 and HTTP/3 or
 * just HTTP/3. For all supported HTTP versions, see the <code>HttpVerson</code> enum.
 * <p>
 * <blockquote><pre>
 * // Configure a distribution to use HTTP/2 and HTTP/3
 * // Configure a distribution to use HTTP/2 and HTTP/3
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new HttpOrigin("www.example.com")).build())
 *         .httpVersion(HttpVersion.HTTP2_AND_3)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Importing Distributions</h3>
 * <p>
 * Existing distributions can be imported as well; note that like most imported constructs, an imported distribution cannot be modified.
 * However, it can be used as a reference for other higher-level constructs.
 * <p>
 * <blockquote><pre>
 * // Using a reference to an imported Distribution
 * IDistribution distribution = Distribution.fromDistributionAttributes(this, "ImportedDist", DistributionAttributes.builder()
 *         .domainName("d111111abcdef8.cloudfront.net")
 *         .distributionId("012345ABCDEF")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Permissions</h3>
 * <p>
 * Use the <code>grant()</code> method to allow actions on the distribution.
 * <code>grantCreateInvalidation()</code> is a shorthand to allow <code>CreateInvalidation</code>.
 * <p>
 * <blockquote><pre>
 * Distribution distribution;
 * Function lambdaFn;
 * 
 * distribution.grant(lambdaFn, "cloudfront:ListInvalidations", "cloudfront:GetInvalidation");
 * distribution.grantCreateInvalidation(lambdaFn);
 * </pre></blockquote>
 * <p>
 * <h3>Realtime Log Config</h3>
 * <p>
 * CloudFront supports realtime log delivery from your distribution to a Kinesis stream.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html">Real-time logs</a> in the CloudFront User Guide.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Adding realtime logs config to a Cloudfront Distribution on default behavior.
 * import software.amazon.awscdk.services.kinesis.*;
 * 
 * Stream stream;
 * 
 * 
 * RealtimeLogConfig realTimeConfig = RealtimeLogConfig.Builder.create(this, "realtimeLog")
 *         .endPoints(List.of(Endpoint.fromKinesisStream(stream)))
 *         .fields(List.of("timestamp", "c-ip", "time-to-first-byte", "sc-status"))
 *         .realtimeLogConfigName("my-delivery-stream")
 *         .samplingRate(100)
 *         .build();
 * 
 * Distribution.Builder.create(this, "myCdn")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new HttpOrigin("www.example.com"))
 *                 .realtimeLogConfig(realTimeConfig)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Migrating from the original CloudFrontWebDistribution to the newer Distribution construct</h2>
 * <p>
 * It's possible to migrate a distribution from the original to the modern API.
 * The changes necessary are the following:
 * <p>
 * <h3>The Distribution</h3>
 * <p>
 * Replace <code>new CloudFrontWebDistribution</code> with <code>new Distribution</code>. Some
 * configuration properties have been changed:
 * <p>
 * | Old API                | New API                                                                                        |
 * | ---------------------- | ---------------------------------------------------------------------------------------------- |
 * | <code>originConfigs</code>        | <code>defaultBehavior</code>; use <code>additionalBehaviors</code> if necessary                                      |
 * | <code>viewerCertificate</code>    | <code>certificate</code>; use <code>domainNames</code> for aliases                                                   |
 * | <code>errorConfigurations</code>  | <code>errorResponses</code>                                                                               |
 * | <code>loggingConfig</code>        | <code>enableLogging</code>; configure with <code>logBucket</code> <code>logFilePrefix</code> and <code>logIncludesCookies</code>           |
 * | <code>viewerProtocolPolicy</code> | removed; set on each behavior instead. default changed from <code>REDIRECT_TO_HTTPS</code> to <code>ALLOW_ALL</code> |
 * <p>
 * After switching constructs, you need to maintain the same logical ID for the underlying <a href="https://docs.aws.amazon.com/cdk/api/v1/docs/&#64;aws-cdk_aws-cloudfront.CfnDistribution.html">CfnDistribution</a> if you wish to avoid the deletion and recreation of your distribution.
 * To do this, use <a href="https://docs.aws.amazon.com/cdk/v2/guide/cfn_layer.html">escape hatches</a> to override the logical ID created by the new Distribution construct with the logical ID created by the old construct.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Bucket sourceBucket;
 * 
 * 
 * Distribution myDistribution = Distribution.Builder.create(this, "MyCfWebDistribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new S3Origin(sourceBucket))
 *                 .build())
 *         .build();
 * CfnDistribution cfnDistribution = (CfnDistribution)myDistribution.getNode().getDefaultChild();
 * cfnDistribution.overrideLogicalId("MyDistributionCFDistribution3H55TI9Q");
 * </pre></blockquote>
 * <p>
 * <h3>Behaviors</h3>
 * <p>
 * The modern API makes use of the <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_cloudfront_origins-readme.html">CloudFront Origins</a> module to easily configure your origin. Replace your origin configuration with the relevant CloudFront Origins class. For example, here's a behavior with an S3 origin:
 * <p>
 * <blockquote><pre>
 * Bucket sourceBucket;
 * OriginAccessIdentity oai;
 * 
 * 
 * CloudFrontWebDistribution.Builder.create(this, "MyCfWebDistribution")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder()
 *                         .s3BucketSource(sourceBucket)
 *                         .originAccessIdentity(oai)
 *                         .build())
 *                 .behaviors(List.of(Behavior.builder().isDefaultBehavior(true).build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Becomes:
 * <p>
 * <blockquote><pre>
 * Bucket sourceBucket;
 * 
 * 
 * Distribution distribution = Distribution.Builder.create(this, "MyCfWebDistribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new S3Origin(sourceBucket))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * In the original API all behaviors are defined in the <code>originConfigs</code> property. The new API is optimized for a single origin and behavior, so the default behavior and additional behaviors will be defined separately.
 * <p>
 * <blockquote><pre>
 * Bucket sourceBucket;
 * OriginAccessIdentity oai;
 * 
 * 
 * CloudFrontWebDistribution.Builder.create(this, "MyCfWebDistribution")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder()
 *                         .s3BucketSource(sourceBucket)
 *                         .originAccessIdentity(oai)
 *                         .build())
 *                 .behaviors(List.of(Behavior.builder().isDefaultBehavior(true).build()))
 *                 .build(), SourceConfiguration.builder()
 *                 .customOriginSource(CustomOriginConfig.builder()
 *                         .domainName("MYALIAS")
 *                         .build())
 *                 .behaviors(List.of(Behavior.builder().pathPattern("/somewhere").build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Becomes:
 * <p>
 * <blockquote><pre>
 * Bucket sourceBucket;
 * 
 * 
 * Distribution distribution = Distribution.Builder.create(this, "MyCfWebDistribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new S3Origin(sourceBucket))
 *                 .build())
 *         .additionalBehaviors(Map.of(
 *                 "/somewhere", BehaviorOptions.builder()
 *                         .origin(new HttpOrigin("MYALIAS"))
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Certificates</h3>
 * <p>
 * If you are using an ACM certificate, you can pass the certificate directly to the <code>certificate</code> prop.
 * Any aliases used before in the <code>ViewerCertificate</code> class should be passed in to the <code>domainNames</code> prop in the modern API.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.*;
 * Certificate certificate;
 * Bucket sourceBucket;
 * 
 * 
 * ViewerCertificate viewerCertificate = ViewerCertificate.fromAcmCertificate(certificate, ViewerCertificateOptions.builder()
 *         .aliases(List.of("MYALIAS"))
 *         .build());
 * 
 * CloudFrontWebDistribution.Builder.create(this, "MyCfWebDistribution")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder()
 *                         .s3BucketSource(sourceBucket)
 *                         .build())
 *                 .behaviors(List.of(Behavior.builder().isDefaultBehavior(true).build()))
 *                 .build()))
 *         .viewerCertificate(viewerCertificate)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Becomes:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.*;
 * Certificate certificate;
 * Bucket sourceBucket;
 * 
 * 
 * Distribution distribution = Distribution.Builder.create(this, "MyCfWebDistribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new S3Origin(sourceBucket))
 *                 .build())
 *         .domainNames(List.of("MYALIAS"))
 *         .certificate(certificate)
 *         .build();
 * </pre></blockquote>
 * <p>
 * IAM certificates aren't directly supported by the new API, but can be easily configured through <a href="https://docs.aws.amazon.com/cdk/v2/guide/cfn_layer.html">escape hatches</a>
 * <p>
 * <blockquote><pre>
 * Bucket sourceBucket;
 * 
 * ViewerCertificate viewerCertificate = ViewerCertificate.fromIamCertificate("MYIAMROLEIDENTIFIER", ViewerCertificateOptions.builder()
 *         .aliases(List.of("MYALIAS"))
 *         .build());
 * 
 * CloudFrontWebDistribution.Builder.create(this, "MyCfWebDistribution")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder()
 *                         .s3BucketSource(sourceBucket)
 *                         .build())
 *                 .behaviors(List.of(Behavior.builder().isDefaultBehavior(true).build()))
 *                 .build()))
 *         .viewerCertificate(viewerCertificate)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Becomes:
 * <p>
 * <blockquote><pre>
 * Bucket sourceBucket;
 * 
 * Distribution distribution = Distribution.Builder.create(this, "MyCfWebDistribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new S3Origin(sourceBucket))
 *                 .build())
 *         .domainNames(List.of("MYALIAS"))
 *         .build();
 * 
 * CfnDistribution cfnDistribution = (CfnDistribution)distribution.getNode().getDefaultChild();
 * 
 * cfnDistribution.addPropertyOverride("ViewerCertificate.IamCertificateId", "MYIAMROLEIDENTIFIER");
 * cfnDistribution.addPropertyOverride("ViewerCertificate.SslSupportMethod", "sni-only");
 * </pre></blockquote>
 * <p>
 * <h3>Other changes</h3>
 * <p>
 * A number of default settings have changed on the new API when creating a new distribution, behavior, and origin.
 * After making the major changes needed for the migration, run <code>cdk diff</code> to see what settings have changed.
 * If no changes are desired during migration, you will at the least be able to use <a href="https://docs.aws.amazon.com/cdk/v2/guide/cfn_layer.html">escape hatches</a> to override what the CDK synthesizes, if you can't change the properties directly.
 * <p>
 * <h2>CloudFrontWebDistribution API</h2>
 * <p>
 * <blockquote>
 * <p>
 * The <code>CloudFrontWebDistribution</code> construct is the original construct written for working with CloudFront distributions.
 * Users are encouraged to use the newer <code>Distribution</code> instead, as it has a simpler interface and receives new features faster.
 * <p>
 * </blockquote>
 * <p>
 * Example usage:
 * <p>
 * <blockquote><pre>
 * // Using a CloudFrontWebDistribution construct.
 * 
 * Bucket sourceBucket;
 * 
 * CloudFrontWebDistribution distribution = CloudFrontWebDistribution.Builder.create(this, "MyDistribution")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder()
 *                         .s3BucketSource(sourceBucket)
 *                         .build())
 *                 .behaviors(List.of(Behavior.builder().isDefaultBehavior(true).build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Viewer certificate</h3>
 * <p>
 * By default, CloudFront Web Distributions will answer HTTPS requests with CloudFront's default certificate,
 * only containing the distribution <code>domainName</code> (e.g. d111111abcdef8.cloudfront.net).
 * You can customize the viewer certificate property to provide a custom certificate and/or list of domain name aliases to fit your needs.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https-alternate-domain-names.html">Using Alternate Domain Names and HTTPS</a> in the CloudFront User Guide.
 * <p>
 * <h4>Default certificate</h4>
 * <p>
 * You can customize the default certificate aliases. This is intended to be used in combination with CNAME records in your DNS zone.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Bucket s3BucketSource = new Bucket(this, "Bucket");
 * 
 * CloudFrontWebDistribution distribution = CloudFrontWebDistribution.Builder.create(this, "AnAmazingWebsiteProbably")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder().s3BucketSource(s3BucketSource).build())
 *                 .behaviors(List.of(Behavior.builder().isDefaultBehavior(true).build()))
 *                 .build()))
 *         .viewerCertificate(ViewerCertificate.fromCloudFrontDefaultCertificate("www.example.com"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>ACM certificate</h4>
 * <p>
 * You can change the default certificate by one stored AWS Certificate Manager, or ACM.
 * Those certificate can either be generated by AWS, or purchased by another CA imported into ACM.
 * <p>
 * For more information, see
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-certificatemanager-readme.html">the aws-certificatemanager module documentation</a>
 * or <a href="https://docs.aws.amazon.com/acm/latest/userguide/import-certificate.html">Importing Certificates into AWS Certificate Manager</a>
 * in the AWS Certificate Manager User Guide.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Bucket s3BucketSource = new Bucket(this, "Bucket");
 * 
 * Certificate certificate = Certificate.Builder.create(this, "Certificate")
 *         .domainName("example.com")
 *         .subjectAlternativeNames(List.of("*.example.com"))
 *         .build();
 * 
 * CloudFrontWebDistribution distribution = CloudFrontWebDistribution.Builder.create(this, "AnAmazingWebsiteProbably")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder().s3BucketSource(s3BucketSource).build())
 *                 .behaviors(List.of(Behavior.builder().isDefaultBehavior(true).build()))
 *                 .build()))
 *         .viewerCertificate(ViewerCertificate.fromAcmCertificate(certificate, ViewerCertificateOptions.builder()
 *                 .aliases(List.of("example.com", "www.example.com"))
 *                 .securityPolicy(SecurityPolicyProtocol.TLS_V1) // default
 *                 .sslMethod(SSLMethod.SNI)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>IAM certificate</h4>
 * <p>
 * You can also import a certificate into the IAM certificate store.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-and-https-procedures.html#cnames-and-https-uploading-certificates">Importing an SSL/TLS Certificate</a> in the CloudFront User Guide.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Bucket s3BucketSource = new Bucket(this, "Bucket");
 * 
 * CloudFrontWebDistribution distribution = CloudFrontWebDistribution.Builder.create(this, "AnAmazingWebsiteProbably")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder().s3BucketSource(s3BucketSource).build())
 *                 .behaviors(List.of(Behavior.builder().isDefaultBehavior(true).build()))
 *                 .build()))
 *         .viewerCertificate(ViewerCertificate.fromIamCertificate("certificateId", ViewerCertificateOptions.builder()
 *                 .aliases(List.of("example.com"))
 *                 .securityPolicy(SecurityPolicyProtocol.SSL_V3) // default
 *                 .sslMethod(SSLMethod.SNI)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Trusted Key Groups</h3>
 * <p>
 * CloudFront Web Distributions supports validating signed URLs or signed cookies using key groups.
 * When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Using trusted key groups for Cloudfront Web Distributions.
 * Bucket sourceBucket;
 * String publicKey;
 * 
 * PublicKey pubKey = PublicKey.Builder.create(this, "MyPubKey")
 *         .encodedKey(publicKey)
 *         .build();
 * 
 * KeyGroup keyGroup = KeyGroup.Builder.create(this, "MyKeyGroup")
 *         .items(List.of(pubKey))
 *         .build();
 * 
 * CloudFrontWebDistribution.Builder.create(this, "AnAmazingWebsiteProbably")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder()
 *                         .s3BucketSource(sourceBucket)
 *                         .build())
 *                 .behaviors(List.of(Behavior.builder()
 *                         .isDefaultBehavior(true)
 *                         .trustedKeyGroups(List.of(keyGroup))
 *                         .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Restrictions</h3>
 * <p>
 * CloudFront supports adding restrictions to your distribution.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/georestrictions.html">Restricting the Geographic Distribution of Your Content</a> in the CloudFront User Guide.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Adding restrictions to a Cloudfront Web Distribution.
 * Bucket sourceBucket;
 * 
 * CloudFrontWebDistribution.Builder.create(this, "MyDistribution")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder()
 *                         .s3BucketSource(sourceBucket)
 *                         .build())
 *                 .behaviors(List.of(Behavior.builder().isDefaultBehavior(true).build()))
 *                 .build()))
 *         .geoRestriction(GeoRestriction.allowlist("US", "GB"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Connection behaviors between CloudFront and your origin</h3>
 * <p>
 * CloudFront provides you even more control over the connection behaviors between CloudFront and your origin.
 * You can now configure the number of connection attempts CloudFront will make to your origin and the origin connection timeout for each attempt.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-attempts">Origin Connection Attempts</a>
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-timeout">Origin Connection Timeout</a>
 * <p>
 * Example usage:
 * <p>
 * <blockquote><pre>
 * // Configuring connection behaviors between Cloudfront and your origin
 * CloudFrontWebDistribution distribution = CloudFrontWebDistribution.Builder.create(this, "MyDistribution")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .connectionAttempts(3)
 *                 .connectionTimeout(Duration.seconds(10))
 *                 .behaviors(List.of(Behavior.builder()
 *                         .isDefaultBehavior(true)
 *                         .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Origin Fallback</h4>
 * <p>
 * In case the origin source is not available and answers with one of the
 * specified status codes the failover origin source will be used.
 * <p>
 * <blockquote><pre>
 * // Configuring origin fallback options for the CloudFrontWebDistribution
 * // Configuring origin fallback options for the CloudFrontWebDistribution
 * CloudFrontWebDistribution.Builder.create(this, "ADistribution")
 *         .originConfigs(List.of(SourceConfiguration.builder()
 *                 .s3OriginSource(S3OriginConfig.builder()
 *                         .s3BucketSource(Bucket.fromBucketName(this, "aBucket", "myoriginbucket"))
 *                         .originPath("/")
 *                         .originHeaders(Map.of(
 *                                 "myHeader", "42"))
 *                         .originShieldRegion("us-west-2")
 *                         .build())
 *                 .failoverS3OriginSource(S3OriginConfig.builder()
 *                         .s3BucketSource(Bucket.fromBucketName(this, "aBucketFallback", "myoriginbucketfallback"))
 *                         .originPath("/somewhere")
 *                         .originHeaders(Map.of(
 *                                 "myHeader2", "21"))
 *                         .originShieldRegion("us-east-1")
 *                         .build())
 *                 .failoverCriteriaStatusCodes(List.of(FailoverStatusCode.INTERNAL_SERVER_ERROR))
 *                 .behaviors(List.of(Behavior.builder()
 *                         .isDefaultBehavior(true)
 *                         .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>KeyGroup &amp; PublicKey API</h2>
 * <p>
 * You can create a key group to use with CloudFront signed URLs and signed cookies
 * You can add public keys to use with CloudFront features such as signed URLs, signed cookies, and field-level encryption.
 * <p>
 * The following example command uses OpenSSL to generate an RSA key pair with a length of 2048 bits and save to the file named <code>private_key.pem</code>.
 * <p>
 * <blockquote><pre>
 * openssl genrsa -out private_key.pem 2048
 * </pre></blockquote>
 * <p>
 * The resulting file contains both the public and the private key. The following example command extracts the public key from the file named <code>private_key.pem</code> and stores it in <code>public_key.pem</code>.
 * <p>
 * <blockquote><pre>
 * openssl rsa -pubout -in private_key.pem -out public_key.pem
 * </pre></blockquote>
 * <p>
 * Note: Don't forget to copy/paste the contents of <code>public_key.pem</code> file including <code>-----BEGIN PUBLIC KEY-----</code> and <code>-----END PUBLIC KEY-----</code> lines into <code>encodedKey</code> parameter when creating a <code>PublicKey</code>.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Create a key group to use with CloudFront signed URLs and signed cookies.
 * // Create a key group to use with CloudFront signed URLs and signed cookies.
 * KeyGroup.Builder.create(this, "MyKeyGroup")
 *         .items(List.of(
 *             PublicKey.Builder.create(this, "MyPublicKey")
 *                     .encodedKey("...")
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See:
 * <p>
 * <ul>
 * <li>https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html</li>
 * <li>https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-trusted-signers.html</li>
 * </ul>
 */
package software.amazon.awscdk.services.cloudfront;
