/**
 * <h1>Actions for AWS Elastic Load Balancing V2</h1>
 * <p>
 * This package contains integration actions for ELBv2. See the README of the <code>aws-cdk-lib/aws-elasticloadbalancingv2</code> library.
 * <p>
 * <h2>Cognito</h2>
 * <p>
 * ELB allows for requests to be authenticated against a Cognito user pool using
 * the <code>AuthenticateCognitoAction</code>. For details on the setup's requirements,
 * read <a href="https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-authenticate-users.html#cognito-requirements">Prepare to use Amazon
 * Cognito</a>.
 * Here's an example:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.*;
 * 
 * Vpc vpc;
 * Certificate certificate;
 * 
 * 
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB")
 *         .vpc(vpc)
 *         .internetFacing(true)
 *         .build();
 * 
 * UserPool userPool = new UserPool(this, "UserPool");
 * UserPoolClient userPoolClient = UserPoolClient.Builder.create(this, "Client")
 *         .userPool(userPool)
 * 
 *         // Required minimal configuration for use with an ELB
 *         .generateSecret(true)
 *         .authFlows(AuthFlow.builder()
 *                 .userPassword(true)
 *                 .build())
 *         .oAuth(OAuthSettings.builder()
 *                 .flows(OAuthFlows.builder()
 *                         .authorizationCodeGrant(true)
 *                         .build())
 *                 .scopes(List.of(OAuthScope.EMAIL))
 *                 .callbackUrls(List.of(String.format("https://%s/oauth2/idpresponse", lb.getLoadBalancerDnsName())))
 *                 .build())
 *         .build();
 * CfnUserPoolClient cfnClient = (CfnUserPoolClient)userPoolClient.getNode().getDefaultChild();
 * cfnClient.addPropertyOverride("RefreshTokenValidity", 1);
 * cfnClient.addPropertyOverride("SupportedIdentityProviders", List.of("COGNITO"));
 * 
 * UserPoolDomain userPoolDomain = UserPoolDomain.Builder.create(this, "Domain")
 *         .userPool(userPool)
 *         .cognitoDomain(CognitoDomainOptions.builder()
 *                 .domainPrefix("test-cdk-prefix")
 *                 .build())
 *         .build();
 * 
 * lb.addListener("Listener", BaseApplicationListenerProps.builder()
 *         .port(443)
 *         .certificates(List.of(certificate))
 *         .defaultAction(AuthenticateCognitoAction.Builder.create()
 *                 .userPool(userPool)
 *                 .userPoolClient(userPoolClient)
 *                 .userPoolDomain(userPoolDomain)
 *                 .next(ListenerAction.fixedResponse(200, FixedResponseOptions.builder()
 *                         .contentType("text/plain")
 *                         .messageBody("Authenticated")
 *                         .build()))
 *                 .build())
 *         .build());
 * 
 * CfnOutput.Builder.create(this, "DNS")
 *         .value(lb.getLoadBalancerDnsName())
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.elasticloadbalancingv2.actions;
