/**
 * <h1>Amazon Route53 Construct Library</h1>
 * <p>
 * To add a public hosted zone:
 * <p>
 * <blockquote><pre>
 * PublicHostedZone.Builder.create(this, "HostedZone")
 *         .zoneName("fully.qualified.domain.com")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add a private hosted zone, use <code>PrivateHostedZone</code>. Note that
 * <code>enableDnsHostnames</code> and <code>enableDnsSupport</code> must have been enabled for the
 * VPC you're configuring for private hosted zones.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * PrivateHostedZone zone = PrivateHostedZone.Builder.create(this, "HostedZone")
 *         .zoneName("fully.qualified.domain.com")
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Additional VPCs can be added with <code>zone.addVpc()</code>.
 * <p>
 * <h2>Adding Records</h2>
 * <p>
 * To add a TXT record to your zone:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * TxtRecord.Builder.create(this, "TXTRecord")
 *         .zone(myZone)
 *         .recordName("_foo") // If the name ends with a ".", it will be used as-is;
 *         // if it ends with a "." followed by the zone name, a trailing "." will be added automatically;
 *         // otherwise, a ".", the zone name, and a trailing "." will be added automatically.
 *         // Defaults to zone root if not specified.
 *         .values(List.of("Bar!", "Baz?"))
 *         .ttl(Duration.minutes(90))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add a NS record to your zone:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * NsRecord.Builder.create(this, "NSRecord")
 *         .zone(myZone)
 *         .recordName("foo")
 *         .values(List.of("ns-1.awsdns.co.uk.", "ns-2.awsdns.com."))
 *         .ttl(Duration.minutes(90))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add a DS record to your zone:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * DsRecord.Builder.create(this, "DSRecord")
 *         .zone(myZone)
 *         .recordName("foo")
 *         .values(List.of("12345 3 1 123456789abcdef67890123456789abcdef67890"))
 *         .ttl(Duration.minutes(90))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add an A record to your zone:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * ARecord.Builder.create(this, "ARecord")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.4", "5.6.7.8"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add an A record for an EC2 instance with an Elastic IP (EIP) to your zone:
 * <p>
 * <blockquote><pre>
 * Instance instance;
 * 
 * HostedZone myZone;
 * 
 * 
 * CfnEIP elasticIp = CfnEIP.Builder.create(this, "EIP")
 *         .domain("vpc")
 *         .instanceId(instance.getInstanceId())
 *         .build();
 * ARecord.Builder.create(this, "ARecord")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses(elasticIp.getRef()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To create an A record of type alias with target set to another record created outside CDK:
 * <p>
 * <h3>This function registers the given input i.e. DNS Name(string) of an existing record as an AliasTarget to the new ARecord. To register a target that is created as part of CDK use this instead https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_route53_targets-readme.html</h3>
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * String targetRecord = "existing.record.cdk.local";
 * ARecord record = ARecord.fromARecordAttributes(this, "A", ARecordAttrs.builder()
 *         .zone(myZone)
 *         .recordName("test")
 *         .targetDNS(targetRecord)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To add an AAAA record pointing to a CloudFront distribution:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudfront.*;
 * 
 * HostedZone myZone;
 * CloudFrontWebDistribution distribution;
 * 
 * AaaaRecord.Builder.create(this, "Alias")
 *         .zone(myZone)
 *         .target(RecordTarget.fromAlias(new CloudFrontTarget(distribution)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <a href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy-geo.html">Geolocation routing</a> can be enabled for continent, country or subdivision:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * // continent
 * // continent
 * ARecord.Builder.create(this, "ARecordGeoLocationContinent")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.0", "5.6.7.0"))
 *         .geoLocation(GeoLocation.continent(Continent.EUROPE))
 *         .build();
 * 
 * // country
 * // country
 * ARecord.Builder.create(this, "ARecordGeoLocationCountry")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.1", "5.6.7.1"))
 *         .geoLocation(GeoLocation.country("DE"))
 *         .build();
 * 
 * // subdivision
 * // subdivision
 * ARecord.Builder.create(this, "ARecordGeoLocationSubDividion")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.2", "5.6.7.2"))
 *         .geoLocation(GeoLocation.subdivision("WA"))
 *         .build();
 * 
 * // default (wildcard record if no specific record is found)
 * // default (wildcard record if no specific record is found)
 * ARecord.Builder.create(this, "ARecordGeoLocationDefault")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.3", "5.6.7.3"))
 *         .geoLocation(GeoLocation.default())
 *         .build();
 * </pre></blockquote>
 * <p>
 * To enable <a href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy-weighted.html">weighted routing</a>, use the <code>weight</code> parameter:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * ARecord.Builder.create(this, "ARecordWeighted1")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.4"))
 *         .weight(10)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To enable <a href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy-latency.html">latency based routing</a>, use the <code>region</code> parameter:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * ARecord.Builder.create(this, "ARecordLatency1")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.4"))
 *         .region("us-east-1")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To enable <a href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy-multivalue.html">multivalue answer routing</a>, use the <code>multivalueAnswer</code> parameter:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * ARecord.Builder.create(this, "ARecordMultiValue1")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.4"))
 *         .multiValueAnswer(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To specify a unique identifier to differentiate among multiple resource record sets that have the same combination of name and type, use the <code>setIdentifier</code> parameter:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * ARecord.Builder.create(this, "ARecordWeighted1")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.4"))
 *         .weight(10)
 *         .setIdentifier("weighted-record-id")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Warning</strong> It is not possible to specify <code>setIdentifier</code> for a simple routing policy.
 * <p>
 * Constructs are available for A, AAAA, CAA, CNAME, MX, NS, SRV and TXT records.
 * <p>
 * Use the <code>CaaAmazonRecord</code> construct to easily restrict certificate authorities
 * allowed to issue certificates for a domain to Amazon only.
 * <p>
 * <h3>Replacing existing record sets (dangerous!)</h3>
 * <p>
 * Use the <code>deleteExisting</code> prop to delete an existing record set before deploying the new one.
 * This is useful if you want to minimize downtime and avoid "manual" actions while deploying a
 * stack with a record set that already exists. This is typically the case for record sets that
 * are not already "owned" by CloudFormation or "owned" by another stack or construct that is
 * going to be deleted (migration).
 * <p>
 * <blockquote>
 * <p>
 * <strong>N.B.:</strong> this feature is dangerous, use with caution! It can only be used safely when
 * <code>deleteExisting</code> is set to <code>true</code> as soon as the resource is added to the stack. Changing
 * an existing Record Set's <code>deleteExisting</code> property from <code>false -&gt; true</code> after deployment
 * will delete the record!
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * ARecord.Builder.create(this, "ARecord")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.4", "5.6.7.8"))
 *         .deleteExisting(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cross Account Zone Delegation</h3>
 * <p>
 * If you want to have your root domain hosted zone in one account and your subdomain hosted
 * zone in a different one, you can use <code>CrossAccountZoneDelegationRecord</code> to set up delegation
 * between them.
 * <p>
 * In the account containing the parent hosted zone:
 * <p>
 * <blockquote><pre>
 * PublicHostedZone parentZone = PublicHostedZone.Builder.create(this, "HostedZone")
 *         .zoneName("someexample.com")
 *         .build();
 * Role crossAccountRole = Role.Builder.create(this, "CrossAccountRole")
 *         // The role name must be predictable
 *         .roleName("MyDelegationRole")
 *         // The other account
 *         .assumedBy(new AccountPrincipal("12345678901"))
 *         // You can scope down this role policy to be least privileged.
 *         // If you want the other account to be able to manage specific records,
 *         // you can scope down by resource and/or normalized record names
 *         .inlinePolicies(Map.of(
 *                 "crossAccountPolicy", PolicyDocument.Builder.create()
 *                         .statements(List.of(
 *                             PolicyStatement.Builder.create()
 *                                     .sid("ListHostedZonesByName")
 *                                     .effect(Effect.ALLOW)
 *                                     .actions(List.of("route53:ListHostedZonesByName"))
 *                                     .resources(List.of("*"))
 *                                     .build(),
 *                             PolicyStatement.Builder.create()
 *                                     .sid("GetHostedZoneAndChangeResourceRecordSets")
 *                                     .effect(Effect.ALLOW)
 *                                     .actions(List.of("route53:GetHostedZone", "route53:ChangeResourceRecordSets"))
 *                                     // This example assumes the RecordSet subdomain.somexample.com
 *                                     // is contained in the HostedZone
 *                                     .resources(List.of("arn:aws:route53:::hostedzone/HZID00000000000000000"))
 *                                     .conditions(Map.of(
 *                                             "ForAllValues:StringLike", Map.of(
 *                                                     "route53:ChangeResourceRecordSetsNormalizedRecordNames", List.of("subdomain.someexample.com"))))
 *                                     .build()))
 *                         .build()))
 *         .build();
 * parentZone.grantDelegation(crossAccountRole);
 * </pre></blockquote>
 * <p>
 * In the account containing the child zone to be delegated:
 * <p>
 * <blockquote><pre>
 * PublicHostedZone subZone = PublicHostedZone.Builder.create(this, "SubZone")
 *         .zoneName("sub.someexample.com")
 *         .build();
 * 
 * // import the delegation role by constructing the roleArn
 * String delegationRoleArn = Stack.of(this).formatArn(ArnComponents.builder()
 *         .region("") // IAM is global in each partition
 *         .service("iam")
 *         .account("parent-account-id")
 *         .resource("role")
 *         .resourceName("MyDelegationRole")
 *         .build());
 * IRole delegationRole = Role.fromRoleArn(this, "DelegationRole", delegationRoleArn);
 * 
 * // create the record
 * // create the record
 * CrossAccountZoneDelegationRecord.Builder.create(this, "delegate")
 *         .delegatedZone(subZone)
 *         .parentHostedZoneName("someexample.com") // or you can use parentHostedZoneId
 *         .delegationRole(delegationRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Delegating the hosted zone requires assuming a role in the parent hosted zone's account.
 * In order for the assumed credentials to be valid, the resource must assume the role using
 * an STS endpoint in a region where both the subdomain's account and the parent's account
 * are opted-in. By default, this region is determined automatically, but if you need to
 * change the region used for the AssumeRole call, specify <code>assumeRoleRegion</code>:
 * <p>
 * <blockquote><pre>
 * PublicHostedZone subZone = PublicHostedZone.Builder.create(this, "SubZone")
 *         .zoneName("sub.someexample.com")
 *         .build();
 * 
 * // import the delegation role by constructing the roleArn
 * String delegationRoleArn = Stack.of(this).formatArn(ArnComponents.builder()
 *         .region("") // IAM is global in each partition
 *         .service("iam")
 *         .account("parent-account-id")
 *         .resource("role")
 *         .resourceName("MyDelegationRole")
 *         .build());
 * IRole delegationRole = Role.fromRoleArn(this, "DelegationRole", delegationRoleArn);
 * 
 * CrossAccountZoneDelegationRecord.Builder.create(this, "delegate")
 *         .delegatedZone(subZone)
 *         .parentHostedZoneName("someexample.com") // or you can use parentHostedZoneId
 *         .delegationRole(delegationRole)
 *         .assumeRoleRegion("us-east-1")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Add Trailing Dot to Domain Names</h3>
 * <p>
 * In order to continue managing existing domain names with trailing dots using CDK, you can set <code>addTrailingDot: false</code> to prevent the Construct from adding a dot at the end of the domain name.
 * <p>
 * <blockquote><pre>
 * PublicHostedZone.Builder.create(this, "HostedZone")
 *         .zoneName("fully.qualified.domain.com.")
 *         .addTrailingDot(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Enabling DNSSEC</h2>
 * <p>
 * DNSSEC can be enabled for Hosted Zones. For detailed information, see
 * <a href="https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/dns-configuring-dnssec.html">Configuring DNSSEC signing in Amazon Route 53</a>.
 * <p>
 * Enabling DNSSEC requires an asymmetric KMS Customer-Managed Key using the <code>ECC_NIST_P256</code> key spec.
 * Additionally, that KMS key must be in <code>us-east-1</code>.
 * <p>
 * <blockquote><pre>
 * Key kmsKey = Key.Builder.create(this, "KmsCMK")
 *         .keySpec(KeySpec.ECC_NIST_P256)
 *         .keyUsage(KeyUsage.SIGN_VERIFY)
 *         .build();
 * HostedZone hostedZone = HostedZone.Builder.create(this, "HostedZone")
 *         .zoneName("example.com")
 *         .build();
 * // Enable DNSSEC signing for the zone
 * hostedZone.enableDnssec(ZoneSigningOptions.builder().kmsKey(kmsKey).build());
 * </pre></blockquote>
 * <p>
 * The necessary permissions for Route 53 to use the key will automatically be added when using
 * this configuration. If it is necessary to create a key signing key manually, that can be done
 * using the <code>KeySigningKey</code> construct:
 * <p>
 * <blockquote><pre>
 * HostedZone hostedZone;
 * Key kmsKey;
 * 
 * KeySigningKey.Builder.create(this, "KeySigningKey")
 *         .hostedZone(hostedZone)
 *         .kmsKey(kmsKey)
 *         .keySigningKeyName("ksk")
 *         .status(KeySigningKeyStatus.ACTIVE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When directly constructing the <code>KeySigningKey</code> resource, enabling DNSSEC signing for the hosted
 * zone will be need to be done explicitly (either using the <code>CfnDNSSEC</code> construct or via another
 * means).
 * <p>
 * <h2>Imports</h2>
 * <p>
 * If you don't know the ID of the Hosted Zone to import, you can use the
 * <code>HostedZone.fromLookup</code>:
 * <p>
 * <blockquote><pre>
 * HostedZone.fromLookup(this, "MyZone", HostedZoneProviderProps.builder()
 *         .domainName("example.com")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <code>HostedZone.fromLookup</code> requires an environment to be configured. Check
 * out the <a href="https://docs.aws.amazon.com/cdk/latest/guide/environments.html">documentation</a> for more documentation and examples. CDK
 * automatically looks into your <code>~/.aws/config</code> file for the <code>[default]</code> profile.
 * If you want to specify a different account run <code>cdk deploy --profile [profile]</code>.
 * <p>
 * <blockquote><pre>
 * new MyDevStack(app, 'dev', {
 *   env: {
 *     account: process.env.CDK_DEFAULT_ACCOUNT,
 *     region: process.env.CDK_DEFAULT_REGION,
 *   },
 * });
 * </pre></blockquote>
 * <p>
 * If you know the ID and Name of a Hosted Zone, you can import it directly:
 * <p>
 * <blockquote><pre>
 * IHostedZone zone = HostedZone.fromHostedZoneAttributes(this, "MyZone", HostedZoneAttributes.builder()
 *         .zoneName("example.com")
 *         .hostedZoneId("ZOJJZC49E0EPZ")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Alternatively, use the <code>HostedZone.fromHostedZoneId</code> to import hosted zones if
 * you know the ID and the retrieval for the <code>zoneName</code> is undesirable.
 * <p>
 * <blockquote><pre>
 * IHostedZone zone = HostedZone.fromHostedZoneId(this, "MyZone", "ZOJJZC49E0EPZ");
 * </pre></blockquote>
 * <p>
 * You can import a Public Hosted Zone as well with the similar <code>PublicHostedZone.fromPublicHostedZoneId</code> and <code>PublicHostedZone.fromPublicHostedZoneAttributes</code> methods:
 * <p>
 * <blockquote><pre>
 * IPublicHostedZone zoneFromAttributes = PublicHostedZone.fromPublicHostedZoneAttributes(this, "MyZone", PublicHostedZoneAttributes.builder()
 *         .zoneName("example.com")
 *         .hostedZoneId("ZOJJZC49E0EPZ")
 *         .build());
 * 
 * // Does not know zoneName
 * IPublicHostedZone zoneFromId = PublicHostedZone.fromPublicHostedZoneId(this, "MyZone", "ZOJJZC49E0EPZ");
 * </pre></blockquote>
 * <p>
 * You can use <code>CrossAccountZoneDelegationRecord</code> on imported Hosted Zones with the <code>grantDelegation</code> method:
 * <p>
 * <blockquote><pre>
 * Role crossAccountRole = Role.Builder.create(this, "CrossAccountRole")
 *         // The role name must be predictable
 *         .roleName("MyDelegationRole")
 *         // The other account
 *         .assumedBy(new AccountPrincipal("12345678901"))
 *         .build();
 * 
 * IHostedZone zoneFromId = HostedZone.fromHostedZoneId(this, "MyZone", "zone-id");
 * zoneFromId.grantDelegation(crossAccountRole);
 * 
 * IPublicHostedZone publicZoneFromId = PublicHostedZone.fromPublicHostedZoneId(this, "MyPublicZone", "public-zone-id");
 * publicZoneFromId.grantDelegation(crossAccountRole);
 * 
 * IPrivateHostedZone privateZoneFromId = PrivateHostedZone.fromPrivateHostedZoneId(this, "MyPrivateZone", "private-zone-id");
 * privateZoneFromId.grantDelegation(crossAccountRole);
 * </pre></blockquote>
 * <p>
 * <h2>VPC Endpoint Service Private DNS</h2>
 * <p>
 * When you create a VPC endpoint service, AWS generates endpoint-specific DNS hostnames that consumers use to communicate with the service.
 * For example, vpce-1234-abcdev-us-east-1.vpce-svc-123345.us-east-1.vpce.amazonaws.com.
 * By default, your consumers access the service with that DNS name.
 * This can cause problems with HTTPS traffic because the DNS will not match the backend certificate:
 * <p>
 * <blockquote><pre>
 * curl: (60) SSL: no alternative certificate subject name matches target host name 'vpce-abcdefghijklmnopq-rstuvwx.vpce-svc-abcdefghijklmnopq.us-east-1.vpce.amazonaws.com'
 * </pre></blockquote>
 * <p>
 * Effectively, the endpoint appears untrustworthy. To mitigate this, clients have to create an alias for this DNS name in Route53.
 * <p>
 * Private DNS for an endpoint service lets you configure a private DNS name so consumers can
 * access the service using an existing DNS name without creating this Route53 DNS alias
 * This DNS name can also be guaranteed to match up with the backend certificate.
 * <p>
 * Before consumers can use the private DNS name, you must verify that you have control of the domain/subdomain.
 * <p>
 * Assuming your account has ownership of the particular domain/subdomain,
 * this construct sets up the private DNS configuration on the endpoint service,
 * creates all the necessary Route53 entries, and verifies domain ownership.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.elasticloadbalancingv2.NetworkLoadBalancer;
 * 
 * 
 * Vpc vpc = new Vpc(this, "VPC");
 * NetworkLoadBalancer nlb = NetworkLoadBalancer.Builder.create(this, "NLB")
 *         .vpc(vpc)
 *         .build();
 * VpcEndpointService vpces = VpcEndpointService.Builder.create(this, "VPCES")
 *         .vpcEndpointServiceLoadBalancers(List.of(nlb))
 *         .build();
 * // You must use a public hosted zone so domain ownership can be verified
 * PublicHostedZone zone = PublicHostedZone.Builder.create(this, "PHZ")
 *         .zoneName("aws-cdk.dev")
 *         .build();
 * VpcEndpointServiceDomainName.Builder.create(this, "EndpointDomain")
 *         .endpointService(vpces)
 *         .domainName("my-stuff.aws-cdk.dev")
 *         .publicHostedZone(zone)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.route53;
