/**
 * <h1>CloudFront Origins for the CDK CloudFront Library</h1>
 * <p>
 * This library contains convenience methods for defining origins for a CloudFront distribution. You can use this library to create origins from
 * S3 buckets, Elastic Load Balancing v2 load balancers, or any other domain name.
 * <p>
 * <h2>S3 Bucket</h2>
 * <p>
 * An S3 bucket can be used as an origin. An S3 bucket origin can either be configured using a standard S3 bucket or using a S3 bucket that's configured as a website endpoint (see AWS docs for <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DownloadDistS3AndCustomOrigins.html#using-s3-as-origin">Using an S3 Bucket</a>).
 * <p>
 * <blockquote>
 * <p>
 * Note: <code>S3Origin</code> has been deprecated. Use <code>S3BucketOrigin</code> for standard S3 origins and <code>S3StaticWebsiteOrigin</code> for static website S3 origins.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Standard S3 Bucket</h3>
 * <p>
 * To set up an origin using a standard S3 bucket, use the <code>S3BucketOrigin</code> class. The bucket
 * is handled as a bucket origin and
 * CloudFront's redirect and error handling will be used. It is recommended to use <code>S3BucketOrigin.withOriginAccessControl()</code> to configure OAC for your origin.
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(S3BucketOrigin.withOriginAccessControl(myBucket)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: When you use CloudFront OAC with Amazon S3 bucket origins, you must set Amazon S3 Object Ownership to Bucket owner enforced (the default for new Amazon S3 buckets). If you require ACLs, use the Bucket owner preferred setting to maintain control over objects uploaded via CloudFront.
 * <p>
 * </blockquote>
 * <p>
 * <h3>S3 Bucket Configured as a Website Endpoint</h3>
 * <p>
 * To set up an origin using an S3 bucket configured as a website endpoint, use the <code>S3StaticWebsiteOrigin</code> class. When the bucket is configured as a
 * website endpoint, the bucket is treated as an HTTP origin,
 * and the distribution can use built-in S3 redirects and S3 custom error pages.
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new S3StaticWebsiteOrigin(myBucket)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Restricting access to a standard S3 Origin</h3>
 * <p>
 * CloudFront provides two ways to send authenticated requests to a standard Amazon S3 origin:
 * <p>
 * <ul>
 * <li>origin access control (OAC) and</li>
 * <li>origin access identity (OAI)</li>
 * </ul>
 * <p>
 * OAI is considered legacy due to limited functionality and regional
 * limitations, whereas OAC is recommended because it supports all Amazon S3
 * buckets in all AWS Regions, Amazon S3 server-side encryption with AWS KMS (SSE-KMS), and dynamic requests (PUT and DELETE) to Amazon S3. Additionally,
 * OAC provides stronger security posture with short term credentials,
 * and more frequent credential rotations as compared to OAI. OAI and OAC can be used in conjunction with a bucket that is not public to
 * require that your users access your content using CloudFront URLs and not S3 URLs directly.
 * <p>
 * See AWS docs on <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html">Restricting access to an Amazon S3 Origin</a> for more details.
 * <p>
 * <blockquote>
 * <p>
 * Note: OAC and OAI can only be used with an regular S3 bucket origin (not a bucket configured as a website endpoint).
 * <p>
 * </blockquote>
 * <p>
 * The <code>S3BucketOrigin</code> class supports creating a standard S3 origin with OAC, OAI, and no access control (using your bucket access settings) via
 * the <code>withOriginAccessControl()</code>, <code>withOriginAccessIdentity()</code>, and <code>withBucketDefaults()</code> methods respectively.
 * <p>
 * <h4>Setting up a new origin access control (OAC)</h4>
 * <p>
 * Setup a standard S3 origin with origin access control as follows:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(S3BucketOrigin.withOriginAccessControl(myBucket))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * When creating a standard S3 origin using <code>origins.S3BucketOrigin.withOriginAccessControl()</code>, an <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originaccesscontrol-originaccesscontrolconfig.html">Origin Access Control resource</a> is automatically created with the origin type set to <code>s3</code> and signing behavior set to <code>always</code>.
 * <p>
 * You can grant read, write or delete access to the OAC using the <code>originAccessLevels</code> property:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * IOrigin s3Origin = S3BucketOrigin.withOriginAccessControl(myBucket, S3BucketOriginWithOACProps.builder()
 *         .originAccessLevels(List.of(AccessLevel.READ, AccessLevel.WRITE, AccessLevel.DELETE))
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also pass in a custom S3 origin access control:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * S3OriginAccessControl oac = S3OriginAccessControl.Builder.create(this, "MyOAC")
 *         .signing(Signing.SIGV4_NO_OVERRIDE)
 *         .build();
 * IOrigin s3Origin = S3BucketOrigin.withOriginAccessControl(myBucket, S3BucketOriginWithOACProps.builder()
 *         .originAccessControl(oac)
 *         .build());
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(s3Origin)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * An existing S3 origin access control can be imported using the <code>fromOriginAccessControlId</code> method:
 * <p>
 * <blockquote><pre>
 * IOriginAccessControl importedOAC = S3OriginAccessControl.fromOriginAccessControlId(this, "myImportedOAC", "ABC123ABC123AB");
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html">Note</a>: When you use OAC with S3
 * bucket origins, the bucket's object ownership must be either set to Bucket owner enforced (default for new S3 buckets) or Bucket owner preferred (only if you require ACLs).
 * <p>
 * </blockquote>
 * <p>
 * <h4>Setting up OAC with a SSE-KMS encrypted S3 origin</h4>
 * <p>
 * If the objects in the S3 bucket origin are encrypted using server-side encryption with
 * AWS Key Management Service (SSE-KMS), the OAC must have permission to use the KMS key.
 * <p>
 * Setting up a standard S3 origin using <code>S3BucketOrigin.withOriginAccessControl()</code> will automatically add the statement to the KMS key policy
 * to give the OAC permission to use the KMS key.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * Key myKmsKey = new Key(this, "myKMSKey");
 * Bucket myBucket = Bucket.Builder.create(this, "mySSEKMSEncryptedBucket")
 *         .encryption(BucketEncryption.KMS)
 *         .encryptionKey(myKmsKey)
 *         .objectOwnership(ObjectOwnership.BUCKET_OWNER_ENFORCED)
 *         .build();
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(S3BucketOrigin.withOriginAccessControl(myBucket))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h5>Scoping down the key policy</h5>
 * <p>
 * I saw this warning message during synth time. What do I do?
 * <p>
 * <blockquote><pre>
 * To avoid a circular dependency between the KMS key, Bucket, and Distribution during the initial deployment, a wildcard is used in the Key policy condition to match all Distribution IDs.
 * After deploying once, it is strongly recommended to further scope down the policy for best security practices by following the guidance in the "Using OAC for a SSE-KMS encrypted S3 origin" section in the module README.
 * </pre></blockquote>
 * <p>
 * If the S3 bucket has an <code>encryptionKey</code> defined, <code>S3BucketOrigin.withOriginAccessControl()</code>
 * will automatically add the following policy statement to the KMS key policy to allow CloudFront read-only access (unless otherwise specified in the <code>originAccessLevels</code> property).
 * <p>
 * <blockquote><pre>
 * {
 *     "Statement": {
 *         "Effect": "Allow",
 *         "Principal": {
 *             "Service": "cloudfront.amazonaws.com"
 *         },
 *         "Action": "kms:Decrypt",
 *         "Resource": "*",
 *         "Condition": {
 *             "ArnLike": {
 *                 "AWS:SourceArn": "arn:aws:cloudfront::&lt;account ID&gt;:distribution/*"
 *             }
 *         }
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * This policy uses a wildcard to match all distribution IDs in the account instead of referencing the specific distribution ID to resolve the circular dependency. The policy statement is not as scoped down as the example in the AWS CloudFront docs (see <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#create-oac-overview-s3">SSE-KMS section</a>).
 * <p>
 * After you have deployed the Distribution, you should follow these steps to only grant permissions to the specific distribution according to AWS best practices:
 * <p>
 * <strong>Step 1.</strong> Copy the key policy
 * <p>
 * <strong>Step 2.</strong> Use an escape hatch to update the policy statement condition so that
 * <p>
 * <blockquote><pre>
 *   "Condition": {
 *       "ArnLike": {
 *           "AWS:SourceArn": "arn:aws:cloudfront::&lt;account ID&gt;:distribution/*"
 *       }
 *   }
 * </pre></blockquote>
 * <p>
 * ...becomes...
 * <p>
 * <blockquote><pre>
 *   "Condition": {
 *       "StringEquals": {
 *           "AWS:SourceArn": "arn:aws:cloudfront::111122223333:distribution/&lt;CloudFront distribution ID&gt;"
 *       }
 *   }
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note the change of condition operator from <code>ArnLike</code> to <code>StringEquals</code> in addition to replacing the wildcard (<code>*</code>) with the distribution ID.
 * <p>
 * </blockquote>
 * <p>
 * To set the key policy using an escape hatch:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * Key kmsKey = new Key(this, "myKMSKey");
 * Bucket myBucket = Bucket.Builder.create(this, "mySSEKMSEncryptedBucket")
 *         .encryption(BucketEncryption.KMS)
 *         .encryptionKey(kmsKey)
 *         .objectOwnership(ObjectOwnership.BUCKET_OWNER_ENFORCED)
 *         .build();
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(S3BucketOrigin.withOriginAccessControl(myBucket))
 *                 .build())
 *         .build();
 * 
 * // Add the following to scope down the key policy
 * Map&lt;String, Object&gt; scopedDownKeyPolicy = Map.of(
 *         "Version", "2012-10-17",
 *         "Statement", List.of(Map.of(
 *                 "Effect", "Allow",
 *                 "Principal", Map.of(
 *                         "AWS", "arn:aws:iam::111122223333:root"),
 *                 "Action", "kms:*",
 *                 "Resource", "*"), Map.of(
 *                 "Effect", "Allow",
 *                 "Principal", Map.of(
 *                         "Service", "cloudfront.amazonaws.com"),
 *                 "Action", List.of("kms:Decrypt", "kms:Encrypt", "kms:GenerateDataKey*"),
 *                 "Resource", "*",
 *                 "Condition", Map.of(
 *                         "StringEquals", Map.of(
 *                                 "AWS:SourceArn", "arn:aws:cloudfront::111122223333:distribution/&lt;CloudFront distribution ID&gt;")))));
 * CfnKey cfnKey = ((CfnKey)kmsKey.getNode().getDefaultChild());
 * cfnKey.getKeyPolicy() = scopedDownKeyPolicy;
 * </pre></blockquote>
 * <p>
 * <strong>Step 3.</strong> Deploy the stack
 * <p>
 * <blockquote>
 * <p>
 * Tip: Run <code>cdk diff</code> before deploying to verify the
 * changes to your stack.
 * <p>
 * </blockquote>
 * <p>
 * <strong>Step 4.</strong> Verify your final key policy includes the following statement after deploying:
 * <p>
 * <blockquote><pre>
 * {
 *     "Effect": "Allow",
 *     "Principal": {
 *         "Service": [
 *             "cloudfront.amazonaws.com"
 *         ]
 *      },
 *     "Action": [
 *         "kms:Decrypt",
 *         "kms:Encrypt",
 *         "kms:GenerateDataKey*"
 *     ],
 *     "Resource": "*",
 *     "Condition": {
 *             "StringEquals": {
 *                 "AWS:SourceArn": "arn:aws:cloudfront::111122223333:distribution/&lt;CloudFront distribution ID&gt;"
 *             }
 *         }
 * }
 * </pre></blockquote>
 * <p>
 * <h5>Updating imported key policies</h5>
 * <p>
 * If you are using an imported KMS key to encrypt your S3 bucket and want to use OAC, you will need to update the
 * key policy manually to allow CloudFront to use the key. Like most imported resources, CDK apps cannot modify the configuration of imported keys.
 * <p>
 * After deploying the distribution, add the following policy statement to your key policy to allow CloudFront OAC to access your KMS key for SSE-KMS:
 * <p>
 * <blockquote><pre>
 * {
 *     "Sid": "AllowCloudFrontServicePrincipalSSE-KMS",
 *     "Effect": "Allow",
 *     "Principal": {
 *         "Service": [
 *             "cloudfront.amazonaws.com"
 *         ]
 *      },
 *     "Action": [
 *         "kms:Decrypt",
 *         "kms:Encrypt",
 *         "kms:GenerateDataKey*"
 *     ],
 *     "Resource": "*",
 *     "Condition": {
 *             "StringEquals": {
 *                 "AWS:SourceArn": "arn:aws:cloudfront::111122223333:distribution/&lt;CloudFront distribution ID&gt;"
 *             }
 *         }
 * }
 * </pre></blockquote>
 * <p>
 * See CloudFront docs on <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#create-oac-overview-s3">SSE-KMS</a> for more details.
 * <p>
 * <h4>Setting up OAC with imported S3 buckets</h4>
 * <p>
 * If you are using an imported bucket for your S3 Origin and want to use OAC,
 * you will need to update
 * the S3 bucket policy manually to allow the OAC to access the S3 origin. Like most imported resources, CDK apps cannot modify the configuration of imported buckets.
 * <p>
 * After deploying the distribution, add the following
 * policy statement to your
 * S3 bucket to allow CloudFront read-only access
 * (or additional S3 permissions as required):
 * <p>
 * <blockquote><pre>
 * {
 *     "Version": "2012-10-17",
 *     "Statement": {
 *         "Effect": "Allow",
 *         "Principal": {
 *             "Service": "cloudfront.amazonaws.com"
 *         },
 *         "Action": "s3:GetObject",
 *         "Resource": "arn:aws:s3:::&lt;S3 bucket name&gt;/*",
 *         "Condition": {
 *             "StringEquals": {
 *                 "AWS:SourceArn": "arn:aws:cloudfront::111122223333:distribution/&lt;CloudFront distribution ID&gt;"
 *             }
 *         }
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * See CloudFront docs on <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#create-oac-overview-s3">Giving the origin access control permission to access the S3 bucket</a> for more details.
 * <p>
 * <blockquote>
 * <p>
 * Note: If your bucket previously used OAI, you will need to manually remove the policy statement
 * that gives the OAI access to your bucket after setting up OAC.
 * <p>
 * </blockquote>
 * <p>
 * <h4>Setting up an OAI (legacy)</h4>
 * <p>
 * Setup an S3 origin with origin access identity (legacy) as follows:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(S3BucketOrigin.withOriginAccessIdentity(myBucket))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also pass in a custom S3 origin access identity:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * OriginAccessIdentity myOai = OriginAccessIdentity.Builder.create(this, "myOAI")
 *         .comment("My custom OAI")
 *         .build();
 * IOrigin s3Origin = S3BucketOrigin.withOriginAccessIdentity(myBucket, S3BucketOriginWithOAIProps.builder()
 *         .originAccessIdentity(myOai)
 *         .build());
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(s3Origin)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Setting up OAI with imported S3 buckets (legacy)</h4>
 * <p>
 * If you are using an imported bucket for your S3 Origin and want to use OAI,
 * you will need to update
 * the S3 bucket policy manually to allow the OAI to access the S3 origin. Like most imported resources, CDK apps cannot modify the configuration of imported buckets.
 * <p>
 * Add the following
 * policy statement to your
 * S3 bucket to allow the OAI read access:
 * <p>
 * <blockquote><pre>
 * {
 *     "Version": "2012-10-17",
 *     "Id": "PolicyForCloudFrontPrivateContent",
 *     "Statement": [
 *         {
 *             "Effect": "Allow",
 *             "Principal": {
 *                 "AWS": "arn:aws:iam::cloudfront:user/CloudFront Origin Access Identity &lt;origin access identity ID&gt;"
 *             },
 *             "Action": "s3:GetObject",
 *             "Resource": "arn:aws:s3:::&lt;S3 bucket name&gt;/*"
 *         }
 *     ]
 * }
 * </pre></blockquote>
 * <p>
 * See AWS docs on <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#private-content-restricting-access-to-s3-oai">Giving an origin access identity permission to read files in the Amazon S3 bucket</a> for more details.
 * <p>
 * <h3>Setting up a S3 origin with no origin access control</h3>
 * <p>
 * To setup a standard S3 origin with no access control (no OAI nor OAC), use <code>origins.S3BucketOrigin.withBucketDefaults()</code>:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(S3BucketOrigin.withBucketDefaults(myBucket))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Migrating from OAI to OAC</h3>
 * <p>
 * If you are currently using OAI for your S3 origin and wish to migrate to OAC,
 * replace the <code>S3Origin</code> construct (deprecated) with <code>S3BucketOrigin.withOriginAccessControl()</code> which automatically
 * creates and sets up an OAC for you.
 * <p>
 * Existing setup using OAI and <code>S3Origin</code>:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * S3Origin s3Origin = new S3Origin(myBucket);
 * Distribution distribution = Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(s3Origin).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Step 1:</strong>
 * <p>
 * To ensure CloudFront doesn't lose access to the bucket during the transition, add a statement to bucket policy to grant OAC access to the S3 origin. Deploy the stack. If you are okay with downtime during the transition, you can skip this step.
 * <p>
 * <blockquote>
 * <p>
 * Tip: Run <code>cdk diff</code> before deploying to verify the
 * changes to your stack.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.iam.*;
 * 
 * 
 * Stack stack = new Stack();
 * Bucket myBucket = new Bucket(this, "myBucket");
 * S3Origin s3Origin = new S3Origin(myBucket);
 * Distribution distribution = Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(s3Origin).build())
 *         .build();
 * 
 * // Construct the bucket policy statement
 * String distributionArn = stack.formatArn(ArnComponents.builder()
 *         .service("cloudfront")
 *         .region("")
 *         .resource("distribution")
 *         .resourceName(distribution.getDistributionId())
 *         .arnFormat(ArnFormat.SLASH_RESOURCE_NAME)
 *         .build());
 * 
 * ServicePrincipal cloudfrontSP = new ServicePrincipal("cloudfront.amazonaws.com");
 * 
 * PolicyStatement oacBucketPolicyStatement = PolicyStatement.Builder.create()
 *         .effect(Effect.ALLOW)
 *         .principals(List.of(cloudfrontSP))
 *         .actions(List.of("s3:GetObject"))
 *         .resources(List.of(myBucket.arnForObjects("*")))
 *         .conditions(Map.of(
 *                 "StringEquals", Map.of(
 *                         "AWS:SourceArn", distributionArn)))
 *         .build();
 * 
 * // Add statement to bucket policy
 * myBucket.addToResourcePolicy(oacBucketPolicyStatement);
 * </pre></blockquote>
 * <p>
 * The following changes will take place:
 * <p>
 * <ol>
 * <li>The bucket policy will be modified to grant the CloudFront distribution access. At this point the bucket policy allows both an OAI and an OAC to access the S3 origin.</li>
 * </ol>
 * <p>
 * <strong>Step 2:</strong>
 * <p>
 * Replace <code>S3Origin</code> with <code>S3BucketOrigin.withOriginAccessControl()</code>, which creates an OAC and attaches it to the distribution. You can remove the code from Step 1 which updated the bucket policy, as <code>S3BucketOrigin.withOriginAccessControl()</code> updates the bucket policy automatically with the same statement when defined in the <code>Distribution</code> (no net difference).
 * <p>
 * Run <code>cdk diff</code> before deploying to verify the changes to your stack.
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "Bucket");
 * IOrigin s3Origin = S3BucketOrigin.withOriginAccessControl(bucket);
 * Distribution distribution = Distribution.Builder.create(this, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder().origin(s3Origin).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following changes will take place:
 * <p>
 * <ol>
 * <li>A <code>AWS::CloudFront::OriginAccessControl</code> resource will be created.</li>
 * <li>The <code>Origin</code> property of the <code>AWS::CloudFront::Distribution</code> will set <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-originaccesscontrolid"><code>OriginAccessControlId</code></a> to the OAC ID after it is created. It will also set <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-s3originconfig.html#aws-properties-cloudfront-distribution-s3originconfig-properties"><code>S3OriginConfig</code></a> to <code>{"OriginAccessIdentity": ""}</code>, which deletes the origin access identity from the existing distribution.</li>
 * <li>The <code>AWS::CloudFront::CloudFrontOriginAccessIdentity</code> resource will be deleted.</li>
 * </ol>
 * <p>
 * <strong>Will migrating from OAI to OAC cause any resource replacement?</strong>
 * <p>
 * No, following the migration steps does not cause any replacement of the existing <code>AWS::CloudFront::Distribution</code>, <code>AWS::S3::Bucket</code> nor <code>AWS::S3::BucketPolicy</code> resources. It will modify the bucket policy, create a <code>AWS::CloudFront::OriginAccessControl</code> resource, and delete the existing <code>AWS::CloudFront::CloudFrontOriginAccessIdentity</code>.
 * <p>
 * <strong>Will migrating from OAI to OAC have any availability implications for my application?</strong>
 * <p>
 * Updates to bucket policies are eventually consistent. Therefore, removing OAI permissions and setting up OAC in the same CloudFormation stack deployment is not recommended as it may cause downtime where CloudFront loses access to the bucket. Following the steps outlined above lowers the risk of downtime as the bucket policy is updated to have both OAI and OAC permissions, then in a subsequent deployment, the OAI permissions are removed.
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#migrate-from-oai-to-oac">Migrating from origin access identity (OAI) to origin access control (OAC)</a>.
 * <p>
 * <h3>Adding Custom Headers</h3>
 * <p>
 * You can configure CloudFront to add custom headers to the requests that it sends to your origin. These custom headers enable you to send and gather information from your origin that you don’t get with typical viewer requests. These headers can even be customized for each origin. CloudFront supports custom headers for both for custom and Amazon S3 origins.
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(S3BucketOrigin.withOriginAccessControl(myBucket, S3BucketOriginWithOACProps.builder()
 *                 .customHeaders(Map.of(
 *                         "Foo", "bar"))
 *                 .build())).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>ELBv2 Load Balancer</h2>
 * <p>
 * An Elastic Load Balancing (ELB) v2 load balancer may be used as an origin. In order for a load balancer to serve as an origin, it must be publicly
 * accessible (<code>internetFacing</code> is true). Both Application and Network load balancers are supported.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * Vpc vpc;
 * 
 * // Create an application load balancer in a VPC. 'internetFacing' must be 'true'
 * // for CloudFront to access the load balancer and use it as an origin.
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB")
 *         .vpc(vpc)
 *         .internetFacing(true)
 *         .build();
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new LoadBalancerV2Origin(lb)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The origin can also be customized to respond on different ports, have different connection properties, etc.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * ApplicationLoadBalancer loadBalancer;
 * 
 * LoadBalancerV2Origin origin = LoadBalancerV2Origin.Builder.create(loadBalancer)
 *         .connectionAttempts(3)
 *         .connectionTimeout(Duration.seconds(5))
 *         .readTimeout(Duration.seconds(45))
 *         .keepaliveTimeout(Duration.seconds(45))
 *         .protocolPolicy(OriginProtocolPolicy.MATCH_VIEWER)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that the <code>readTimeout</code> and <code>keepaliveTimeout</code> properties can extend their values over 60 seconds only if a limit increase request for CloudFront origin response timeout
 * quota has been approved in the target account; otherwise, values over 60 seconds will produce an error at deploy time. Consider that this value is
 * still limited to a maximum value of 180 seconds, which is a hard limit for that quota.
 * <p>
 * <h2>From an HTTP endpoint</h2>
 * <p>
 * Origins can also be created from any other HTTP endpoint, given the domain name, and optionally, other origin properties.
 * <p>
 * <blockquote><pre>
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new HttpOrigin("www.example.com")).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * See the documentation of <code>aws-cdk-lib/aws-cloudfront</code> for more information.
 * <p>
 * <h2>Failover Origins (Origin Groups)</h2>
 * <p>
 * You can set up CloudFront with origin failover for scenarios that require high availability.
 * To get started, you create an origin group with two origins: a primary and a secondary.
 * If the primary origin is unavailable, or returns specific HTTP response status codes that indicate a failure,
 * CloudFront automatically switches to the secondary origin.
 * You achieve that behavior in the CDK using the <code>OriginGroup</code> class:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(OriginGroup.Builder.create()
 *                         .primaryOrigin(S3BucketOrigin.withOriginAccessControl(myBucket))
 *                         .fallbackOrigin(new HttpOrigin("www.example.com"))
 *                         // optional, defaults to: 500, 502, 503 and 504
 *                         .fallbackStatusCodes(List.of(404))
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>From an API Gateway REST API</h2>
 * <p>
 * Origins can be created from an API Gateway REST API. It is recommended to use a
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-endpoint-types.html">regional API</a> in this case. The origin path will automatically be set as the stage name.
 * <p>
 * <blockquote><pre>
 * RestApi api;
 * 
 * Distribution.Builder.create(this, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new RestApiOrigin(api)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to use a different origin path, you can specify it in the <code>originPath</code> property.
 * <p>
 * <blockquote><pre>
 * RestApi api;
 * 
 * Distribution.Builder.create(this, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder().origin(RestApiOrigin.Builder.create(api).originPath("/custom-origin-path").build()).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>From a Lambda Function URL</h2>
 * <p>
 * Lambda Function URLs enable direct invocation of Lambda functions via HTTP(S), without intermediaries. They can be set as CloudFront origins for streamlined function execution behind a CDN, leveraging caching and custom domains.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function fn;
 * 
 * FunctionUrl fnUrl = fn.addFunctionUrl(FunctionUrlOptions.builder().authType(FunctionUrlAuthType.NONE).build());
 * 
 * Distribution.Builder.create(this, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new FunctionUrlOrigin(fnUrl)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Lambda Function URL with Origin Access Control (OAC)</h3>
 * <p>
 * You can configure the Lambda Function URL with Origin Access Control (OAC) for enhanced security. When using OAC with Signing SIGV4_ALWAYS, it is recommended to set the Lambda Function URL authType to AWS_IAM to ensure proper authorization.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * Function fn;
 * 
 * 
 * FunctionUrl fnUrl = fn.addFunctionUrl(FunctionUrlOptions.builder()
 *         .authType(FunctionUrlAuthType.AWS_IAM)
 *         .build());
 * 
 * Distribution.Builder.create(this, "MyDistribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(FunctionUrlOrigin.withOriginAccessControl(fnUrl))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to explicitly add OAC for more customized access control, you can use the originAccessControl option as shown below.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * Function fn;
 * 
 * 
 * FunctionUrl fnUrl = fn.addFunctionUrl(FunctionUrlOptions.builder()
 *         .authType(FunctionUrlAuthType.AWS_IAM)
 *         .build());
 * 
 * // Define a custom OAC
 * FunctionUrlOriginAccessControl oac = FunctionUrlOriginAccessControl.Builder.create(this, "MyOAC")
 *         .originAccessControlName("CustomLambdaOAC")
 *         .signing(Signing.SIGV4_ALWAYS)
 *         .build();
 * 
 * // Set up Lambda Function URL with OAC in CloudFront Distribution
 * // Set up Lambda Function URL with OAC in CloudFront Distribution
 * Distribution.Builder.create(this, "MyDistribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(FunctionUrlOrigin.withOriginAccessControl(fnUrl, FunctionUrlOriginWithOACProps.builder()
 *                         .originAccessControl(oac)
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.cloudfront.origins;
