/**
 * <h1>Amazon Cognito Construct Library</h1>
 * <p>
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/what-is-amazon-cognito.html">Amazon Cognito</a> provides
 * authentication, authorization, and user management for your web and mobile apps. Your users can sign in directly with a
 * user name and password, or through a third party such as Facebook, Amazon, Google or Apple.
 * <p>
 * The two main components of Amazon Cognito are <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html">user
 * pools</a> and <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html">identity
 * pools</a>. User pools are user directories
 * that provide sign-up and sign-in options for your app users. Identity pools enable you to grant your users access to
 * other AWS services. Identity Pool L2 Constructs can be found <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cognito-identitypool-alpha-readme.html">here</a>.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Table of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#amazon-cognito-construct-library">Amazon Cognito Construct Library</a>
 * <p>
 * <ul>
 * <li><a href="#table-of-contents">Table of Contents</a></li>
 * <li><a href="#user-pools">User Pools</a>
 * <p>
 * <ul>
 * <li><a href="#sign-up">Sign Up</a></li>
 * <li><a href="#sign-in">Sign In</a></li>
 * <li><a href="#attributes">Attributes</a></li>
 * <li><a href="#attribute-verification">Attribute verification</a></li>
 * <li><a href="#security">Security</a>
 * <p>
 * <ul>
 * <li><a href="#multi-factor-authentication-mfa">Multi-factor Authentication (MFA)</a></li>
 * <li><a href="#account-recovery-settings">Account Recovery Settings</a></li>
 * <li><a href="#advanced-security-mode">Advanced Security Mode</a></li>
 * </ul></li>
 * <li><a href="#emails">Emails</a></li>
 * <li><a href="#device-tracking">Device Tracking</a></li>
 * <li><a href="#lambda-triggers">Lambda Triggers</a>
 * <p>
 * <ul>
 * <li><a href="#trigger-permissions">Trigger Permissions</a></li>
 * </ul></li>
 * <li><a href="#importing-user-pools">Importing User Pools</a></li>
 * <li><a href="#identity-providers">Identity Providers</a></li>
 * <li><a href="#app-clients">App Clients</a></li>
 * <li><a href="#resource-servers">Resource Servers</a></li>
 * <li><a href="#domains">Domains</a></li>
 * <li><a href="#deletion-protection">Deletion protection</a></li>
 * </ul></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>User Pools</h2>
 * <p>
 * User pools allow creating and managing your own directory of users that can sign up and sign in. They enable easy
 * integration with social identity providers such as Facebook, Google, Amazon, Microsoft Active Directory, etc. through
 * SAML.
 * <p>
 * Using the CDK, a new user pool can be created as part of the stack using the construct's constructor. You may specify
 * the <code>userPoolName</code> to give your own identifier to the user pool. If not, CloudFormation will generate a name.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         .userPoolName("myawesomeapp-userpool")
 *         .signInCaseSensitive(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, usernames and email addresses in user pools are case sensitive, which means <code>user&#64;example.com</code> and <code>User&#64;example.com</code>
 * are considered different. In most situations it is preferred to have usernames and email addresses be case insensitive so that
 * capitalization differences are ignored. As shown above, you can make a user pool case insensitive by setting <code>signInCaseSensitive</code>
 * to <code>false</code>. The case sensitivity cannot be changed once a user pool is created.
 * <p>
 * The default set up for the user pool is configured such that only administrators will be allowed
 * to create users. Features such as Multi-factor authentication (MFAs) and Lambda Triggers are not
 * configured by default.
 * <p>
 * Use the <code>grant()</code> method to add an IAM policy statement associated with the user pool to an
 * IAM principal's policy.
 * <p>
 * <blockquote><pre>
 * UserPool userPool = new UserPool(this, "myuserpool");
 * Role role = Role.Builder.create(this, "role")
 *         .assumedBy(new ServicePrincipal("foo"))
 *         .build();
 * userPool.grant(role, "cognito-idp:AdminCreateUser");
 * </pre></blockquote>
 * <p>
 * <h3>Sign Up</h3>
 * <p>
 * Users can either be signed up by the app's administrators or can sign themselves up. Once a user has signed up, their
 * account needs to be confirmed. Cognito provides several ways to sign users up and confirm their accounts. Learn more
 * about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html">user sign up here</a>.
 * <p>
 * To verify the email address of a user in your user pool with Amazon Cognito, you can send the user an email message
 * with a link that they can select, or you can send them a code that they can enter.
 * <p>
 * <h4>Code Verification</h4>
 * <p>
 * When a user signs up, email and SMS messages are used to verify their account and contact methods. The following code
 * snippet configures a user pool with properties relevant to these verification messages -
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .selfSignUpEnabled(true)
 *         .userVerification(UserVerificationConfig.builder()
 *                 .emailSubject("Verify your email for our awesome app!")
 *                 .emailBody("Thanks for signing up to our awesome app! Your verification code is {####}")
 *                 .emailStyle(VerificationEmailStyle.CODE)
 *                 .smsMessage("Thanks for signing up to our awesome app! Your verification code is {####}")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, self sign up is disabled. Learn more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-customizations.html">email and SMS verification messages
 * here</a>.
 * <p>
 * Besides users signing themselves up, an administrator of any user pool can sign users up. The user then receives an
 * invitation to join the user pool. The following code snippet configures a user pool with properties relevant to the
 * invitation messages -
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .userInvitation(UserInvitationConfig.builder()
 *                 .emailSubject("Invite to join our awesome app!")
 *                 .emailBody("Hello {username}, you have been invited to join our awesome app! Your temporary password is {####}")
 *                 .smsMessage("Hello {username}, your temporary password for our awesome app is {####}")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Link Verification</h4>
 * <p>
 * Alternatively, users can use link as a verification method. The following code snippet configures a user pool with
 * properties relevant to these verification messages and link verification method.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         .userVerification(UserVerificationConfig.builder()
 *                 .emailStyle(VerificationEmailStyle.LINK)
 *                 .emailSubject("Invite to join our awesome app!")
 *                 .emailBody("You have been invited to join our awesome app! {##Verify Your Email##}")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * All email subjects, bodies and SMS messages for both invitation and verification support Cognito's message templating.
 * Learn more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-templates.html">message templates
 * here</a>.
 * <p>
 * <h3>Sign In</h3>
 * <p>
 * Users registering or signing in into your application can do so with multiple identifiers. There are 4 options
 * available:
 * <p>
 * <ul>
 * <li><code>username</code>: Allow signing in using the one time immutable user name that the user chose at the time of sign up.</li>
 * <li><code>email</code>: Allow signing in using the email address that is associated with the account.</li>
 * <li><code>phone</code>: Allow signing in using the phone number that is associated with the account.</li>
 * <li><code>preferredUsername</code>: Allow signing in with an alternate user name that the user can change at any time. However, this
 * is not available if the <code>username</code> option is not chosen.</li>
 * </ul>
 * <p>
 * The following code sets up a user pool so that the user can sign in with either their username or their email address -
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         // ...
 *         .signInAliases(SignInAliases.builder()
 *                 .username(true)
 *                 .email(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * User pools can either be configured so that user name is primary sign in form, but also allows for the other three to be
 * used additionally; or it can be configured so that email and/or phone numbers are the only ways a user can register and
 * sign in. Read more about this
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases-settings">here</a>.
 * <p>
 * ⚠️ The Cognito service prevents changing the <code>signInAlias</code> property for an existing user pool.
 * <p>
 * To match with 'Option 1' in the above link, with a verified email, <code>signInAliases</code> should be set to
 * <code>{ username: true, email: true }</code>. To match with 'Option 2' in the above link with both a verified
 * email and phone number, this property should be set to <code>{ email: true, phone: true }</code>.
 * <p>
 * Cognito recommends that email and phone number be automatically verified, if they are one of the sign in methods for
 * the user pool. Read more about that
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases">here</a>.
 * The CDK does this by default, when email and/or phone number are specified as part of <code>signInAliases</code>. This can be
 * overridden by specifying the <code>autoVerify</code> property.
 * <p>
 * The following code snippet sets up only email as a sign in alias, but both email and phone number to be auto-verified.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         // ...
 *         .signInAliases(SignInAliases.builder().username(true).email(true).build())
 *         .autoVerify(AutoVerifiedAttrs.builder().email(true).phone(true).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * A user pool can optionally ignore case when evaluating sign-ins. When <code>signInCaseSensitive</code> is false, Cognito will not
 * check the capitalization of the alias when signing in. Default is true.
 * <p>
 * <h3>Attributes</h3>
 * <p>
 * Attributes represent the various properties of each user that's collected and stored in the user pool. Cognito
 * provides a set of standard attributes that are available for all user pools. Users are allowed to select any of these
 * standard attributes to be required. Users will not be able to sign up to the user pool without providing the required
 * attributes. Besides these, additional attributes can be further defined, and are known as custom attributes.
 * <p>
 * Learn more on <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html">attributes in Cognito's
 * documentation</a>.
 * <p>
 * The following code configures a user pool with two standard attributes (name and address) as required and mutable, and adds
 * four custom attributes.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .standardAttributes(StandardAttributes.builder()
 *                 .fullname(StandardAttribute.builder()
 *                         .required(true)
 *                         .mutable(false)
 *                         .build())
 *                 .address(StandardAttribute.builder()
 *                         .required(false)
 *                         .mutable(true)
 *                         .build())
 *                 .build())
 *         .customAttributes(Map.of(
 *                 "myappid", StringAttribute.Builder.create().minLen(5).maxLen(15).mutable(false).build(),
 *                 "callingcode", NumberAttribute.Builder.create().min(1).max(3).mutable(true).build(),
 *                 "isEmployee", BooleanAttribute.Builder.create().mutable(true).build(),
 *                 "joinedOn", new DateTimeAttribute()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * As shown in the code snippet, there are data types that are available for custom attributes. The 'String' and 'Number'
 * data types allow for further constraints on their length and values, respectively.
 * <p>
 * Custom attributes cannot be marked as required.
 * <p>
 * All custom attributes share the property <code>mutable</code> that specifies whether the value of the attribute can be changed.
 * The default value is <code>false</code>.
 * <p>
 * User pools come with two 'built-in' attributes - <code>email_verified</code> and <code>phone_number_verified</code>. These cannot be
 * configured (required-ness or mutability) as part of user pool creation. However, user pool administrators can modify
 * them for specific users using the <a href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html">AdminUpdateUserAttributes API</a>.
 * <p>
 * <h3>Attribute verification</h3>
 * <p>
 * When your user updates an email address or phone number attribute, Amazon Cognito marks it unverified until they verify the new value.
 * You can’t send messages to an unverified email address or phone number.
 * Your user can’t sign in with an unverified alias attribute.
 * You can choose how Amazon Cognito handles an updated email address or phone number after the update and before the verification.
 * <p>
 * Learn more on <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html?icmpid=docs_cognito_console_help_panel">configuring email or phone verification in Cognito's documentation.</a>
 * <p>
 * The following code configures a user pool that keeps the original value for the two standard attributes (email and phone_number) until the new values are verified.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .signInAliases(SignInAliases.builder().username(true).build())
 *         .autoVerify(AutoVerifiedAttrs.builder().email(true).phone(true).build())
 *         .keepOriginal(KeepOriginalAttrs.builder()
 *                 .email(true)
 *                 .phone(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Security</h3>
 * <p>
 * Cognito sends various messages to its users via SMS, for different actions, ranging from account verification to
 * marketing. In order to send SMS messages, Cognito needs an IAM role that it can assume, with permissions that allow it
 * to send SMS messages.
 * <p>
 * By default, the CDK looks at all of the specified properties (and their defaults when not explicitly specified) and
 * automatically creates an SMS role, when needed. For example, if MFA second factor by SMS is enabled, the CDK will
 * create a new role. The <code>smsRole</code> property can be used to specify the user supplied role that should be used instead.
 * Additionally, the property <code>enableSmsRole</code> can be used to override the CDK's default behaviour to either enable or
 * suppress automatic role creation.
 * <p>
 * <blockquote><pre>
 * Role poolSmsRole = Role.Builder.create(this, "userpoolsmsrole")
 *         .assumedBy(new ServicePrincipal("foo"))
 *         .build();
 * 
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .smsRole(poolSmsRole)
 *         .smsRoleExternalId("c87467be-4f34-11ea-b77f-2e728ce88125")
 *         .build();
 * </pre></blockquote>
 * <p>
 * When the <code>smsRole</code> property is specified, the <code>smsRoleExternalId</code> may also be specified. The value of
 * <code>smsRoleExternalId</code> will be used as the <code>sts:ExternalId</code> when the Cognito service assumes the role. In turn, the role's
 * assume role policy should be configured to accept this value as the ExternalId. Learn more about <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html">ExternalId
 * here</a>.
 * <p>
 * <h4>Multi-factor Authentication (MFA)</h4>
 * <p>
 * User pools can be configured to enable multi-factor authentication (MFA). It can either be turned off, set to optional
 * or made required. Setting MFA to optional means that individual users can choose to enable it.
 * Additionally, the MFA code can be sent either via SMS text message or via a time-based software token.
 * See the <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html">documentation on MFA</a> to
 * learn more.
 * <p>
 * The following code snippet marks MFA for the user pool as required. This means that all users are required to
 * configure an MFA token and use it for sign in. It also allows for the users to use both SMS based MFA, as well,
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa-totp.html">time-based one time password
 * (TOTP)</a>.
 * <p>
 * If you want to enable email-based MFA, set <code>email</code> propety to the Amazon SES email-sending configuration and set <code>advancedSecurityMode</code> to <code>AdvancedSecurity.ENFORCED</code> or <code>AdvancedSecurity.AUDIT</code>.
 * For more information, see <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-advanced-security-email-mfa.html">Email MFA</a>.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .mfa(Mfa.REQUIRED)
 *         .mfaSecondFactor(MfaSecondFactor.builder()
 *                 .sms(true)
 *                 .otp(true)
 *                 .email(false)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * User pools can be configured with policies around a user's password. This includes the password length and the
 * character sets that they must contain.
 * <p>
 * Further to this, it can also be configured with the validity of the auto-generated temporary password. A temporary
 * password is generated by the user pool either when an admin signs up a user or when a password reset is requested.
 * The validity of this password dictates how long to give the user to use this password before expiring it.
 * <p>
 * The following code snippet configures these properties -
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .passwordPolicy(PasswordPolicy.builder()
 *                 .minLength(12)
 *                 .requireLowercase(true)
 *                 .requireUppercase(true)
 *                 .requireDigits(true)
 *                 .requireSymbols(true)
 *                 .tempPasswordValidity(Duration.days(3))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that, <code>tempPasswordValidity</code> can be specified only in whole days. Specifying fractional days would throw an error.
 * <p>
 * <h4>Account Recovery Settings</h4>
 * <p>
 * User pools can be configured on which method a user should use when recovering the password for their account. This
 * can either be email and/or SMS. Read more at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html">Recovering User Accounts</a>
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "UserPool")
 *         // ...
 *         .accountRecovery(AccountRecovery.EMAIL_ONLY)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The default for account recovery is by phone if available and by email otherwise.
 * A user will not be allowed to reset their password via phone if they are also using it for MFA.
 * <p>
 * <h4>Advanced Security Mode</h4>
 * <p>
 * User pools can be configured to use Advanced security. You can turn the user pool advanced security features on, and customize the actions that are taken in response to different risks. Or you can use audit mode to gather metrics on detected risks without taking action. In audit mode, the advanced security features publish metrics to Amazon CloudWatch. See the <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-advanced-security.html">documentation on Advanced security</a> to learn more.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .advancedSecurityMode(AdvancedSecurityMode.ENFORCED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Emails</h3>
 * <p>
 * Cognito sends emails to users in the user pool, when particular actions take place, such as welcome emails, invitation
 * emails, password resets, etc. The address from which these emails are sent can be configured on the user pool.
 * Read more at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html">Email settings for User Pools</a>.
 * <p>
 * By default, user pools are configured to use Cognito's built in email capability, which will send emails
 * from <code>no-reply&#64;verificationemail.com</code>. If you want to use a custom email address you can configure
 * Cognito to send emails through Amazon SES, which is detailed below.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         .email(UserPoolEmail.withCognito("support&#64;myawesomeapp.com"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * For typical production environments, the default email limit is below the required delivery volume.
 * To enable a higher delivery volume, you can configure the UserPool to send emails through Amazon SES. To do
 * so, follow the steps in the <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html#user-pool-email-developer">Cognito Developer Guide</a>
 * to verify an email address, move the account out of the SES sandbox, and grant Cognito email permissions via an
 * authorization policy.
 * <p>
 * Once the SES setup is complete, the UserPool can be configured to use the SES email.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         .email(UserPoolEmail.withSES(UserPoolSESOptions.builder()
 *                 .fromEmail("noreply&#64;myawesomeapp.com")
 *                 .fromName("Awesome App")
 *                 .replyTo("support&#64;myawesomeapp.com")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Sending emails through SES requires that SES be configured (as described above) in a valid SES region.
 * If the UserPool is being created in a different region, <code>sesRegion</code> must be used to specify the correct SES region.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         .email(UserPoolEmail.withSES(UserPoolSESOptions.builder()
 *                 .sesRegion("us-east-1")
 *                 .fromEmail("noreply&#64;myawesomeapp.com")
 *                 .fromName("Awesome App")
 *                 .replyTo("support&#64;myawesomeapp.com")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When sending emails from an SES verified domain, <code>sesVerifiedDomain</code> can be used to specify the domain.
 * The email address does not need to be verified when sending emails from a verified domain, because the identity of the email configuration is can be determined from the domain alone.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         .email(UserPoolEmail.withSES(UserPoolSESOptions.builder()
 *                 .sesRegion("us-east-1")
 *                 .fromEmail("noreply&#64;myawesomeapp.com")
 *                 .fromName("Awesome App")
 *                 .replyTo("support&#64;myawesomeapp.com")
 *                 .sesVerifiedDomain("myawesomeapp.com")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If <code>fromName</code> does not comply RFC 5322 atom or quoted-string, it will be quoted or mime-encoded.
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         .email(UserPoolEmail.withSES(UserPoolSESOptions.builder()
 *                 .fromEmail("noreply&#64;myawesomeapp.com")
 *                 .fromName("myname&#64;mycompany.com")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Device Tracking</h3>
 * <p>
 * User pools can be configured to track devices that users have logged in to.
 * Read more at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html">Device Tracking</a>
 * <p>
 * <blockquote><pre>
 * UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .deviceTracking(DeviceTracking.builder()
 *                 .challengeRequiredOnNewDevice(true)
 *                 .deviceOnlyRememberedOnUserPrompt(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The default is to not track devices.
 * <p>
 * <h3>Lambda Triggers</h3>
 * <p>
 * User pools can be configured such that AWS Lambda functions can be triggered when certain user operations or actions
 * occur, such as, sign up, user confirmation, sign in, etc. They can also be used to add custom authentication
 * challenges, user migrations and custom verification messages. Learn more about triggers at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html">User Pool Workflows with
 * Triggers</a>.
 * <p>
 * Lambda triggers can either be specified as part of the <code>UserPool</code> initialization, or it can be added later, via methods
 * on the construct, as so -
 * <p>
 * <blockquote><pre>
 * Function authChallengeFn = Function.Builder.create(this, "authChallengeFn")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "path/to/asset")))
 *         .build();
 * 
 * UserPool userpool = UserPool.Builder.create(this, "myuserpool")
 *         // ...
 *         .lambdaTriggers(UserPoolTriggers.builder()
 *                 .createAuthChallenge(authChallengeFn)
 *                 .build())
 *         .build();
 * 
 * userpool.addTrigger(UserPoolOperation.USER_MIGRATION, Function.Builder.create(this, "userMigrationFn")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "path/to/asset")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Additionally, only the pre token generation Lambda trigger supports trigger events with lambda version V2.0:
 * <p>
 * <blockquote><pre>
 * UserPool userpool;
 * Function preTokenGenerationFn;
 * 
 * 
 * userpool.addTrigger(UserPoolOperation.PRE_TOKEN_GENERATION_CONFIG, preTokenGenerationFn, LambdaVersion.V2_0);
 * </pre></blockquote>
 * <p>
 * The following table lists the set of triggers available, and their corresponding method to add it to the user pool.
 * For more information on the function of these triggers and how to configure them, read <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html">User Pool Workflows with
 * Triggers</a>.
 * <p>
 * <h4>Trigger Permissions</h4>
 * <p>
 * The <code>function.attachToRolePolicy()</code> API can be used to add additional IAM permissions to the lambda trigger
 * as necessary.
 * <p>
 * ⚠️ Using the <code>attachToRolePolicy</code> API to provide permissions to your user pool will result in a circular dependency. See <a href="https://github.com/aws/aws-cdk/issues/7016">aws/aws-cdk#7016</a>.
 * Error message when running <code>cdk synth</code> or <code>cdk deploy</code>:
 * <p>
 * <blockquote>
 * <p>
 * Circular dependency between resources: [pool056F3F7E, fnPostAuthFnCognitoA630A2B1, ...]
 * <p>
 * </blockquote>
 * <p>
 * To work around the circular dependency issue, use the <code>attachInlinePolicy()</code> API instead, as shown below.
 * <p>
 * <blockquote><pre>
 * Function postAuthFn;
 * 
 * 
 * UserPool userpool = UserPool.Builder.create(this, "myuserpool")
 *         .lambdaTriggers(UserPoolTriggers.builder()
 *                 .postAuthentication(postAuthFn)
 *                 .build())
 *         .build();
 * 
 * // provide permissions to describe the user pool scoped to the ARN the user pool
 * postAuthFn.role.attachInlinePolicy(Policy.Builder.create(this, "userpool-policy")
 *         .statements(List.of(PolicyStatement.Builder.create()
 *                 .actions(List.of("cognito-idp:DescribeUserPool"))
 *                 .resources(List.of(userpool.getUserPoolArn()))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Importing User Pools</h3>
 * <p>
 * Any user pool that has been created outside of this stack, can be imported into the CDK app. Importing a user pool
 * allows for it to be used in other parts of the CDK app that reference an <code>IUserPool</code>. However, imported user pools have
 * limited configurability. As a rule of thumb, none of the properties that are part of the
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html"><code>AWS::Cognito::UserPool</code></a>
 * CloudFormation resource can be configured.
 * <p>
 * User pools can be imported either using their id via the <code>UserPool.fromUserPoolId()</code>, or by using their ARN, via the
 * <code>UserPool.fromUserPoolArn()</code> API.
 * <p>
 * <blockquote><pre>
 * IUserPool awesomePool = UserPool.fromUserPoolId(this, "awesome-user-pool", "us-east-1_oiuR12Abd");
 * 
 * IUserPool otherAwesomePool = UserPool.fromUserPoolArn(this, "other-awesome-user-pool", "arn:aws:cognito-idp:eu-west-1:123456789012:userpool/us-east-1_mtRyYQ14D");
 * </pre></blockquote>
 * <p>
 * <h3>Identity Providers</h3>
 * <p>
 * Users that are part of a user pool can sign in either directly through a user pool, or federate through a third-party
 * identity provider. Once configured, the Cognito backend will take care of integrating with the third-party provider.
 * Read more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-identity-federation.html">Adding User Pool Sign-in Through a Third
 * Party</a>.
 * <p>
 * The following third-party identity providers are currently supported in the CDK -
 * <p>
 * <ul>
 * <li><a href="https://developer.amazon.com/apps-and-games/login-with-amazon">Login With Amazon</a></li>
 * <li><a href="https://developers.facebook.com/docs/facebook-login/">Facebook Login</a></li>
 * <li><a href="https://developers.google.com/identity/sign-in/web/sign-in">Google Login</a></li>
 * <li><a href="https://developer.apple.com/sign-in-with-apple/get-started/">Sign In With Apple</a></li>
 * <li><a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-oidc-idp.html">OpenID Connect</a></li>
 * <li><a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-saml-idp.html">SAML</a></li>
 * </ul>
 * <p>
 * The following code configures a user pool to federate with the third party provider, 'Login with Amazon'. The identity
 * provider needs to be configured with a set of credentials that the Cognito backend can use to federate with the
 * third-party identity provider.
 * <p>
 * <blockquote><pre>
 * UserPool userpool = new UserPool(this, "Pool");
 * 
 * UserPoolIdentityProviderAmazon provider = UserPoolIdentityProviderAmazon.Builder.create(this, "Amazon")
 *         .clientId("amzn-client-id")
 *         .clientSecret("amzn-client-secret")
 *         .userPool(userpool)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Using Google identity provider is possible to use clientSecretValue with SecretValue from secrets manager.
 * <p>
 * <blockquote><pre>
 * UserPool userpool = new UserPool(this, "Pool");
 * SecretValue secret = Secret.fromSecretAttributes(this, "CognitoClientSecret", SecretAttributes.builder()
 *         .secretCompleteArn("arn:aws:secretsmanager:xxx:xxx:secret:xxx-xxx")
 *         .build()).getSecretValue();
 * 
 * UserPoolIdentityProviderGoogle provider = UserPoolIdentityProviderGoogle.Builder.create(this, "Google")
 *         .clientId("amzn-client-id")
 *         .clientSecretValue(secret)
 *         .userPool(userpool)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Using SAML identity provider is possible to use SAML metadata file content or SAML metadata file url.
 * <p>
 * <blockquote><pre>
 * UserPool userpool = new UserPool(this, "Pool");
 * 
 * // specify the metadata as a file content
 * // specify the metadata as a file content
 * UserPoolIdentityProviderSaml.Builder.create(this, "userpoolIdpFile")
 *         .userPool(userpool)
 *         .metadata(UserPoolIdentityProviderSamlMetadata.file("my-file-contents"))
 *         // Whether to require encrypted SAML assertions from IdP
 *         .encryptedResponses(true)
 *         // The signing algorithm for the SAML requests
 *         .requestSigningAlgorithm(SigningAlgorithm.RSA_SHA256)
 *         // Enable IdP initiated SAML auth flow
 *         .idpInitiated(true)
 *         .build();
 * 
 * // specify the metadata as a URL
 * // specify the metadata as a URL
 * UserPoolIdentityProviderSaml.Builder.create(this, "userpoolidpUrl")
 *         .userPool(userpool)
 *         .metadata(UserPoolIdentityProviderSamlMetadata.url("https://my-metadata-url.com"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Attribute mapping allows mapping attributes provided by the third-party identity providers to <a href="#Attributes">standard and custom
 * attributes</a> of the user pool. Learn more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html">Specifying Identity Provider Attribute Mappings for Your
 * User Pool</a>.
 * <p>
 * The following code shows how different attributes provided by 'Login With Amazon' can be mapped to standard and custom
 * user pool attributes.
 * <p>
 * <blockquote><pre>
 * UserPool userpool = new UserPool(this, "Pool");
 * 
 * UserPoolIdentityProviderAmazon.Builder.create(this, "Amazon")
 *         .clientId("amzn-client-id")
 *         .clientSecret("amzn-client-secret")
 *         .userPool(userpool)
 *         .attributeMapping(AttributeMapping.builder()
 *                 .email(ProviderAttribute.AMAZON_EMAIL)
 *                 .website(ProviderAttribute.other("url")) // use other() when an attribute is not pre-defined in the CDK
 *                 .custom(Map.of(
 *                         // custom user pool attributes go here
 *                         "uniqueId", ProviderAttribute.AMAZON_USER_ID))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>App Clients</h3>
 * <p>
 * An app is an entity within a user pool that has permission to call unauthenticated APIs (APIs that do not have an
 * authenticated user), such as APIs to register, sign in, and handle forgotten passwords. To call these APIs, you need an
 * app client ID and an optional client secret. Read <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-client-apps.html">Configuring a User Pool App
 * Client</a> to learn more.
 * <p>
 * The following code creates an app client and retrieves the client id -
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "pool");
 * UserPoolClient client = pool.addClient("customer-app-client");
 * String clientId = client.getUserPoolClientId();
 * </pre></blockquote>
 * <p>
 * Existing app clients can be imported into the CDK app using the <code>UserPoolClient.fromUserPoolClientId()</code> API. For new
 * and imported user pools, clients can also be created via the <code>UserPoolClient</code> constructor, as so -
 * <p>
 * <blockquote><pre>
 * IUserPool importedPool = UserPool.fromUserPoolId(this, "imported-pool", "us-east-1_oiuR12Abd");
 * UserPoolClient.Builder.create(this, "customer-app-client")
 *         .userPool(importedPool)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Clients can be configured with authentication flows. Authentication flows allow users on a client to be authenticated
 * with a user pool. Cognito user pools provide several different types of authentication, such as, SRP (Secure
 * Remote Password) authentication, username-and-password authentication, etc. Learn more about this at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html">UserPool Authentication
 * Flow</a>.
 * <p>
 * The following code configures a client to use both SRP and username-and-password authentication -
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "pool");
 * pool.addClient("app-client", UserPoolClientOptions.builder()
 *         .authFlows(AuthFlow.builder()
 *                 .userPassword(true)
 *                 .userSrp(true)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * Custom authentication protocols can be configured by setting the <code>custom</code> property under <code>authFlow</code> and defining lambda
 * functions for the corresponding user pool <a href="#lambda-triggers">triggers</a>. Learn more at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html#amazon-cognito-user-pools-custom-authentication-flow">Custom Authentication
 * Flow</a>.
 * <p>
 * In addition to these authentication mechanisms, Cognito user pools also support using OAuth 2.0 framework for
 * authenticating users. User pool clients can be configured with OAuth 2.0 authorization flows and scopes. Learn more
 * about the <a href="https://tools.ietf.org/html/rfc6749">OAuth 2.0 authorization framework</a> and <a href="https://aws.amazon.com/blogs/mobile/understanding-amazon-cognito-user-pool-oauth-2-0-grants/">Cognito user pool's
 * implementation of
 * OAuth2.0</a>.
 * <p>
 * The following code configures an app client with the authorization code grant flow and registers the the app's welcome
 * page as a callback (or redirect) URL. It also configures the access token scope to 'openid'. All of these concepts can
 * be found in the <a href="https://tools.ietf.org/html/rfc6749">OAuth 2.0 RFC</a>.
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", UserPoolClientOptions.builder()
 *         .oAuth(OAuthSettings.builder()
 *                 .flows(OAuthFlows.builder()
 *                         .authorizationCodeGrant(true)
 *                         .build())
 *                 .scopes(List.of(OAuthScope.OPENID))
 *                 .callbackUrls(List.of("https://my-app-domain.com/welcome"))
 *                 .logoutUrls(List.of("https://my-app-domain.com/signin"))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * To set a default redirect URI, use the <code>defaultRedirectUri</code> property.
 * Its value must be present in the <code>callbackUrls</code> list.
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", UserPoolClientOptions.builder()
 *         .oAuth(OAuthSettings.builder()
 *                 .flows(OAuthFlows.builder()
 *                         .authorizationCodeGrant(true)
 *                         .build())
 *                 .scopes(List.of(OAuthScope.OPENID))
 *                 .defaultRedirectUri("https://my-app-domain.com/welcome")
 *                 .callbackUrls(List.of("https://my-app-domain.com/welcome", "https://my-app-domain.com/hello"))
 *                 .logoutUrls(List.of("https://my-app-domain.com/signin"))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * An app client can be configured to prevent user existence errors. This
 * instructs the Cognito authentication API to return generic authentication
 * failure responses instead of an UserNotFoundException. By default, the flag
 * is not set, which means the CloudFormation default (false) will be used. See the
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-managing-errors.html">documentation</a>
 * for the full details on the behavior of this flag.
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", UserPoolClientOptions.builder()
 *         .preventUserExistenceErrors(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * All identity providers created in the CDK app are automatically registered into the corresponding user pool. All app
 * clients created in the CDK have all of the identity providers enabled by default. The 'Cognito' identity provider,
 * that allows users to register and sign in directly with the Cognito user pool, is also enabled by default.
 * Alternatively, the list of supported identity providers for a client can be explicitly specified -
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", UserPoolClientOptions.builder()
 *         // ...
 *         .supportedIdentityProviders(List.of(UserPoolClientIdentityProvider.AMAZON, UserPoolClientIdentityProvider.COGNITO))
 *         .build());
 * </pre></blockquote>
 * <p>
 * If the identity provider and the app client are created in the same stack, specify the dependency between both constructs to
 * make sure that the identity provider already exists when the app client will be created. The app client cannot handle the
 * dependency to the identity provider automatically because the client does not have access to the provider's construct.
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * UserPoolIdentityProviderAmazon provider = UserPoolIdentityProviderAmazon.Builder.create(this, "Amazon")
 *         .userPool(pool)
 *         .clientId("amzn-client-id")
 *         .clientSecret("amzn-client-secret")
 *         .build();
 * 
 * UserPoolClient client = pool.addClient("app-client", UserPoolClientOptions.builder()
 *         // ...
 *         .supportedIdentityProviders(List.of(UserPoolClientIdentityProvider.AMAZON))
 *         .build());
 * 
 * client.node.addDependency(provider);
 * </pre></blockquote>
 * <p>
 * The property <code>authSessionValidity</code> is the session token for each API request in the authentication flow.
 * Valid duration is from 3 to 15 minutes.
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", UserPoolClientOptions.builder()
 *         // ...
 *         .authSessionValidity(Duration.minutes(15))
 *         .build());
 * </pre></blockquote>
 * <p>
 * In accordance with the OIDC open standard, Cognito user pool clients provide access tokens, ID tokens and refresh tokens.
 * More information is available at <a href="https://docs.aws.amazon.com/en_us/cognito/latest/developerguide/amazon-cognito-user-pools-using-tokens-with-identity-providers.html">Using Tokens with User Pools</a>.
 * The expiration time for these tokens can be configured as shown below.
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", UserPoolClientOptions.builder()
 *         // ...
 *         .accessTokenValidity(Duration.minutes(60))
 *         .idTokenValidity(Duration.minutes(60))
 *         .refreshTokenValidity(Duration.days(30))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Clients can (and should) be allowed to read and write relevant user attributes only. Usually every client can be allowed to
 * read the <code>given_name</code> attribute but not every client should be allowed to set the <code>email_verified</code> attribute.
 * The same criteria applies for both standard and custom attributes, more info is available at
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-attribute-permissions-and-scopes">Attribute Permissions and Scopes</a>.
 * The default behaviour is to allow read and write permissions on all attributes. The following code shows how this can be
 * configured for a client.
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * 
 * ClientAttributes clientWriteAttributes = (new ClientAttributes()).withStandardAttributes(StandardAttributesMask.builder().fullname(true).email(true).build()).withCustomAttributes("favouritePizza", "favouriteBeverage");
 * 
 * ClientAttributes clientReadAttributes = clientWriteAttributes.withStandardAttributes(StandardAttributesMask.builder().emailVerified(true).build()).withCustomAttributes("pointsEarned");
 * 
 * pool.addClient("app-client", UserPoolClientOptions.builder()
 *         // ...
 *         .readAttributes(clientReadAttributes)
 *         .writeAttributes(clientWriteAttributes)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/token-revocation.html">Token revocation</a>
 * can be configured to be able to revoke refresh tokens in app clients. By default, token revocation is enabled for new user
 * pools. The property can be used to enable the token revocation in existing app clients or to change the default behavior.
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * pool.addClient("app-client", UserPoolClientOptions.builder()
 *         // ...
 *         .enableTokenRevocation(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * User Pool clients can generate a client ID as well as a client secret, to support more advanced authentication workflows.
 * <p>
 * To create a client with an autogenerated client secret, pass the <code>generateSecret: true</code> prop:
 * <p>
 * <blockquote><pre>
 * UserPool importedPool;
 * 
 * 
 * UserPoolClient userPoolClient = UserPoolClient.Builder.create(this, "UserPoolClient")
 *         .userPool(importedPool)
 *         .generateSecret(true)
 *         .build();
 * 
 * // Allows you to pass the generated secret to other pieces of infrastructure
 * SecretValue secret = userPoolClient.getUserPoolClientSecret();
 * </pre></blockquote>
 * <p>
 * If you set <code>enablePropagateAdditionalUserContextData: true</code>, you can collect and pass
 * information about your user's session to Amazon Cognito advanced security
 * when you use the API to sign them up, sign them in, and reset their password.
 * <p>
 * <blockquote><pre>
 * UserPool importedPool;
 * 
 * 
 * UserPoolClient userPoolClient = UserPoolClient.Builder.create(this, "UserPoolClient")
 *         .userPool(importedPool)
 *         .generateSecret(true)
 *         .enablePropagateAdditionalUserContextData(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-adaptive-authentication.html#user-pool-settings-adaptive-authentication-device-fingerprint">Adding user device and session data to API requests</a> for more information.
 * <p>
 * <h3>Resource Servers</h3>
 * <p>
 * A resource server is a server for access-protected resources. It handles authenticated requests from an app that has an
 * access token. See <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html">Defining Resource
 * Servers</a>
 * for more information.
 * <p>
 * An application may choose to model custom permissions via OAuth. Resource Servers provide this capability via custom scopes
 * that are attached to an app client. The following example sets up a resource server for the 'users' resource for two different
 * app clients and configures the clients to use these scopes.
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * 
 * ResourceServerScope readOnlyScope = ResourceServerScope.Builder.create().scopeName("read").scopeDescription("Read-only access").build();
 * ResourceServerScope fullAccessScope = ResourceServerScope.Builder.create().scopeName("*").scopeDescription("Full access").build();
 * 
 * UserPoolResourceServer userServer = pool.addResourceServer("ResourceServer", UserPoolResourceServerOptions.builder()
 *         .identifier("users")
 *         .scopes(List.of(readOnlyScope, fullAccessScope))
 *         .build());
 * 
 * UserPoolClient readOnlyClient = pool.addClient("read-only-client", UserPoolClientOptions.builder()
 *         // ...
 *         .oAuth(OAuthSettings.builder()
 *                 // ...
 *                 .scopes(List.of(OAuthScope.resourceServer(userServer, readOnlyScope)))
 *                 .build())
 *         .build());
 * 
 * UserPoolClient fullAccessClient = pool.addClient("full-access-client", UserPoolClientOptions.builder()
 *         // ...
 *         .oAuth(OAuthSettings.builder()
 *                 // ...
 *                 .scopes(List.of(OAuthScope.resourceServer(userServer, fullAccessScope)))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Domains</h3>
 * <p>
 * After setting up an <a href="#app-clients">app client</a>, the address for the user pool's sign-up and sign-in webpages can be
 * configured using domains. There are two ways to set up a domain - either the Amazon Cognito hosted domain can be chosen
 * with an available domain prefix, or a custom domain name can be chosen. The custom domain must be one that is already
 * owned, and whose certificate is registered in AWS Certificate Manager.
 * <p>
 * The following code sets up a user pool domain in Amazon Cognito hosted domain with the prefix 'my-awesome-app', and
 * another domain with the custom domain 'user.myapp.com' -
 * <p>
 * <blockquote><pre>
 * UserPool pool = new UserPool(this, "Pool");
 * 
 * pool.addDomain("CognitoDomain", UserPoolDomainOptions.builder()
 *         .cognitoDomain(CognitoDomainOptions.builder()
 *                 .domainPrefix("my-awesome-app")
 *                 .build())
 *         .build());
 * 
 * String certificateArn = "arn:aws:acm:us-east-1:123456789012:certificate/11-3336f1-44483d-adc7-9cd375c5169d";
 * 
 * ICertificate domainCert = Certificate.fromCertificateArn(this, "domainCert", certificateArn);
 * pool.addDomain("CustomDomain", UserPoolDomainOptions.builder()
 *         .customDomain(CustomDomainOptions.builder()
 *                 .domainName("user.myapp.com")
 *                 .certificate(domainCert)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * Read more about <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-assign-domain-prefix.html">Using the Amazon Cognito
 * Domain</a> and <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-add-custom-domain.html">Using Your Own
 * Domain</a>.
 * <p>
 * The <code>signInUrl()</code> methods returns the fully qualified URL to the login page for the user pool. This page comes from the
 * hosted UI configured with Cognito. Learn more at <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-integration.html#cognito-user-pools-create-an-app-integration">Hosted UI with the Amazon Cognito
 * Console</a>.
 * <p>
 * <blockquote><pre>
 * UserPool userpool = UserPool.Builder.create(this, "UserPool").build();
 * UserPoolClient client = userpool.addClient("Client", UserPoolClientOptions.builder()
 *         // ...
 *         .oAuth(OAuthSettings.builder()
 *                 .flows(OAuthFlows.builder()
 *                         .implicitCodeGrant(true)
 *                         .build())
 *                 .callbackUrls(List.of("https://myapp.com/home", "https://myapp.com/users"))
 *                 .build())
 *         .build());
 * UserPoolDomain domain = userpool.addDomain("Domain", UserPoolDomainOptions.builder().build());
 * String signInUrl = domain.signInUrl(client, SignInUrlOptions.builder()
 *         .redirectUri("https://myapp.com/home")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Existing domains can be imported into CDK apps using <code>UserPoolDomain.fromDomainName()</code> API
 * <p>
 * <blockquote><pre>
 * IUserPoolDomain myUserPoolDomain = UserPoolDomain.fromDomainName(this, "my-user-pool-domain", "domain-name");
 * </pre></blockquote>
 * <p>
 * <h3>Deletion protection</h3>
 * <p>
 * Deletion protection can be enabled on a user pool to prevent accidental deletion:
 * <p>
 * <blockquote><pre>
 * UserPool userpool = UserPool.Builder.create(this, "UserPool")
 *         // ...
 *         .deletionProtection(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default deletion protection is disabled.
 * <p>
 * <h3><code>email_verified</code> Attribute Mapping</h3>
 * <p>
 * If you use a third-party identity provider, you can specify the <code>email_verified</code> attribute in attributeMapping.
 * <p>
 * <blockquote><pre>
 * UserPool userpool = new UserPool(this, "Pool");
 * 
 * UserPoolIdentityProviderGoogle.Builder.create(this, "google")
 *         .userPool(userpool)
 *         .clientId("google-client-id")
 *         .attributeMapping(AttributeMapping.builder()
 *                 .email(ProviderAttribute.GOOGLE_EMAIL)
 *                 .emailVerified(ProviderAttribute.GOOGLE_EMAIL_VERIFIED)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>User Pool Group</h3>
 * <p>
 * Support for groups in Amazon Cognito user pools enables you to create and manage groups and add users to groups.
 * Use groups to create collections of users to manage their permissions or to represent different types of users.
 * <p>
 * You can assign an AWS Identity and Access Management (IAM) role to a group to define the permissions for members of a group.
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-user-groups.html">Adding groups to a user pool</a>.
 * <p>
 * <blockquote><pre>
 * UserPool userPool;
 * Role role;
 * 
 * 
 * UserPoolGroup.Builder.create(this, "UserPoolGroup")
 *         .userPool(userPool)
 *         .groupName("my-group-name")
 *         .precedence(1)
 *         .role(role)
 *         .build();
 * 
 * // You can also add a group by using addGroup method.
 * userPool.addGroup("AnotherUserPoolGroup", UserPoolGroupOptions.builder()
 *         .groupName("another-group-name")
 *         .build());
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.cognito;
