/**
 * <h1>AWS Config Construct Library</h1>
 * <p>
 * <a href="https://docs.aws.amazon.com/config/latest/developerguide/WhatIsConfig.html">AWS Config</a> provides a detailed view of the configuration of AWS resources in your AWS account.
 * This includes how the resources are related to one another and how they were configured in the
 * past so that you can see how the configurations and relationships change over time.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Initial Setup</h2>
 * <p>
 * Before using the constructs provided in this module, you need to set up AWS Config
 * in the region in which it will be used. This setup includes the one-time creation of the
 * following resources per region:
 * <p>
 * <ul>
 * <li><code>ConfigurationRecorder</code>: Configure which resources will be recorded for config changes.</li>
 * <li><code>DeliveryChannel</code>: Configure where to store the recorded data.</li>
 * </ul>
 * <p>
 * The following guides provide the steps for getting started with AWS Config:
 * <p>
 * <ul>
 * <li><a href="https://docs.aws.amazon.com/config/latest/developerguide/gs-console.html">Using the AWS Console</a></li>
 * <li><a href="https://docs.aws.amazon.com/config/latest/developerguide/gs-cli.html">Using the AWS CLI</a></li>
 * </ul>
 * <p>
 * <h2>Rules</h2>
 * <p>
 * AWS Config can evaluate the configuration settings of your AWS resources by creating AWS Config rules,
 * which represent your ideal configuration settings.
 * <p>
 * See <a href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating Resources with AWS Config Rules</a> to learn more about AWS Config rules.
 * <p>
 * <h3>AWS Managed Rules</h3>
 * <p>
 * AWS Config provides AWS managed rules, which are predefined, customizable rules that AWS Config
 * uses to evaluate whether your AWS resources comply with common best practices.
 * <p>
 * For example, you could create a managed rule that checks whether active access keys are rotated
 * within the number of days specified.
 * <p>
 * <blockquote><pre>
 * // https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 * // https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 * ManagedRule.Builder.create(this, "AccessKeysRotated")
 *         .identifier(ManagedRuleIdentifiers.ACCESS_KEYS_ROTATED)
 *         .inputParameters(Map.of(
 *                 "maxAccessKeyAge", 60))
 * 
 *         // default is 24 hours
 *         .maximumExecutionFrequency(MaximumExecutionFrequency.TWELVE_HOURS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Identifiers for AWS managed rules are available through static constants in the <code>ManagedRuleIdentifiers</code> class.
 * You can find supported input parameters in the <a href="https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html">List of AWS Config Managed Rules</a>.
 * <p>
 * The following higher level constructs for AWS managed rules are available.
 * <p>
 * <h4>Access Key rotation</h4>
 * <p>
 * Checks whether your active access keys are rotated within the number of days specified.
 * <p>
 * <blockquote><pre>
 * // compliant if access keys have been rotated within the last 90 days
 * // compliant if access keys have been rotated within the last 90 days
 * new AccessKeysRotated(this, "AccessKeyRotated");
 * </pre></blockquote>
 * <p>
 * <h4>CloudFormation Stack drift detection</h4>
 * <p>
 * Checks whether your CloudFormation stack's actual configuration differs, or has drifted,
 * from it's expected configuration.
 * <p>
 * <blockquote><pre>
 * // compliant if stack's status is 'IN_SYNC'
 * // non-compliant if the stack's drift status is 'DRIFTED'
 * // compliant if stack's status is 'IN_SYNC'
 * // non-compliant if the stack's drift status is 'DRIFTED'
 * CloudFormationStackDriftDetectionCheck.Builder.create(this, "Drift")
 *         .ownStackOnly(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>CloudFormation Stack notifications</h4>
 * <p>
 * Checks whether your CloudFormation stacks are sending event notifications to a SNS topic.
 * <p>
 * <blockquote><pre>
 * // topics to which CloudFormation stacks may send event notifications
 * Topic topic1 = new Topic(this, "AllowedTopic1");
 * Topic topic2 = new Topic(this, "AllowedTopic2");
 * 
 * // non-compliant if CloudFormation stack does not send notifications to 'topic1' or 'topic2'
 * // non-compliant if CloudFormation stack does not send notifications to 'topic1' or 'topic2'
 * CloudFormationStackNotificationCheck.Builder.create(this, "NotificationCheck")
 *         .topics(List.of(topic1, topic2))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Custom rules</h3>
 * <p>
 * You can develop custom rules and add them to AWS Config. You associate each custom rule with an
 * AWS Lambda function, which contains the logic that evaluates whether your AWS resources comply
 * with the rule.
 * <p>
 * <h3>Triggers</h3>
 * <p>
 * AWS Lambda executes functions in response to events that are published by AWS Services.
 * The function for a custom Config rule receives an event that is published by AWS Config,
 * and is responsible for evaluating the compliance of the rule.
 * <p>
 * Evaluations can be triggered by configuration changes, periodically, or both.
 * To create a custom rule, define a <code>CustomRule</code> and specify the Lambda Function
 * to run and the trigger types.
 * <p>
 * <blockquote><pre>
 * Function evalComplianceFn;
 * 
 * 
 * CustomRule.Builder.create(this, "CustomRule")
 *         .lambdaFunction(evalComplianceFn)
 *         .configurationChanges(true)
 *         .periodic(true)
 * 
 *         // default is 24 hours
 *         .maximumExecutionFrequency(MaximumExecutionFrequency.SIX_HOURS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When the trigger for a rule occurs, the Lambda function is invoked by publishing an event.
 * See <a href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules_example-events.html">example events for AWS Config Rules</a>
 * <p>
 * The AWS documentation has examples of Lambda functions for evaluations that are
 * <a href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules_nodejs-sample.html#event-based-example-rule">triggered by configuration changes</a> and <a href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_develop-rules_nodejs-sample.html#periodic-example-rule">triggered periodically</a>
 * <p>
 * <h3>Scope</h3>
 * <p>
 * By default rules are triggered by changes to all <a href="https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources">resources</a>.
 * <p>
 * Use the <code>RuleScope</code> APIs (<code>fromResource()</code>, <code>fromResources()</code> or <code>fromTag()</code>) to restrict
 * the scope of both managed and custom rules:
 * <p>
 * <blockquote><pre>
 * Function evalComplianceFn;
 * ManagedRule sshRule = ManagedRule.Builder.create(this, "SSH")
 *         .identifier(ManagedRuleIdentifiers.EC2_SECURITY_GROUPS_INCOMING_SSH_DISABLED)
 *         .ruleScope(RuleScope.fromResource(ResourceType.EC2_SECURITY_GROUP, "sg-1234567890abcdefgh"))
 *         .build();
 * CustomRule customRule = CustomRule.Builder.create(this, "Lambda")
 *         .lambdaFunction(evalComplianceFn)
 *         .configurationChanges(true)
 *         .ruleScope(RuleScope.fromResources(List.of(ResourceType.CLOUDFORMATION_STACK, ResourceType.S3_BUCKET)))
 *         .build();
 * 
 * CustomRule tagRule = CustomRule.Builder.create(this, "CostCenterTagRule")
 *         .lambdaFunction(evalComplianceFn)
 *         .configurationChanges(true)
 *         .ruleScope(RuleScope.fromTag("Cost Center", "MyApp"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Events</h3>
 * <p>
 * You can define Amazon EventBridge event rules which trigger when a compliance check fails
 * or when a rule is re-evaluated.
 * <p>
 * Use the <code>onComplianceChange()</code> APIs to trigger an EventBridge event when a compliance check
 * of your AWS Config Rule fails:
 * <p>
 * <blockquote><pre>
 * // Topic to which compliance notification events will be published
 * Topic complianceTopic = new Topic(this, "ComplianceTopic");
 * 
 * CloudFormationStackDriftDetectionCheck rule = new CloudFormationStackDriftDetectionCheck(this, "Drift");
 * rule.onComplianceChange("TopicEvent", OnEventOptions.builder()
 *         .target(new SnsTopic(complianceTopic))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Use the <code>onReEvaluationStatus()</code> status to trigger an EventBridge event when an AWS Config
 * rule is re-evaluated.
 * <p>
 * <blockquote><pre>
 * // Topic to which re-evaluation notification events will be published
 * Topic reEvaluationTopic = new Topic(this, "ComplianceTopic");
 * 
 * CloudFormationStackDriftDetectionCheck rule = new CloudFormationStackDriftDetectionCheck(this, "Drift");
 * rule.onReEvaluationStatus("ReEvaluationEvent", OnEventOptions.builder()
 *         .target(new SnsTopic(reEvaluationTopic))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Example</h3>
 * <p>
 * The following example creates a custom rule that evaluates whether EC2 instances are compliant.
 * Compliance events are published to an SNS topic.
 * <p>
 * <blockquote><pre>
 * // Lambda function containing logic that evaluates compliance with the rule.
 * Function evalComplianceFn = Function.Builder.create(this, "CustomFunction")
 *         .code(AssetCode.fromInline("exports.handler = (event) =&gt; console.log(event);"))
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_12_X)
 *         .build();
 * 
 * // A custom rule that runs on configuration changes of EC2 instances
 * CustomRule customRule = CustomRule.Builder.create(this, "Custom")
 *         .configurationChanges(true)
 *         .lambdaFunction(evalComplianceFn)
 *         .ruleScope(RuleScope.fromResource(ResourceType.EC2_INSTANCE))
 *         .build();
 * 
 * // A rule to detect stack drifts
 * CloudFormationStackDriftDetectionCheck driftRule = new CloudFormationStackDriftDetectionCheck(this, "Drift");
 * 
 * // Topic to which compliance notification events will be published
 * Topic complianceTopic = new Topic(this, "ComplianceTopic");
 * 
 * // Send notification on compliance change events
 * driftRule.onComplianceChange("ComplianceChange", OnEventOptions.builder()
 *         .target(new SnsTopic(complianceTopic))
 *         .build());
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.config;
