/**
 * <h1>Amazon CloudWatch Logs Construct Library</h1>
 * <p>
 * This library supplies constructs for working with CloudWatch Logs.
 * <p>
 * <h2>Log Groups/Streams</h2>
 * <p>
 * The basic unit of CloudWatch is a <em>Log Group</em>. Every log group typically has the
 * same kind of data logged to it, in the same format. If there are multiple
 * applications or services logging into the Log Group, each of them creates a new
 * <em>Log Stream</em>.
 * <p>
 * Every log operation creates a "log event", which can consist of a simple string
 * or a single-line JSON object. JSON objects have the advantage that they afford
 * more filtering abilities (see below).
 * <p>
 * The only configurable attribute for log streams is the retention period, which
 * configures after how much time the events in the log stream expire and are
 * deleted.
 * <p>
 * The default retention period if not supplied is 2 years, but it can be set to
 * one of the values in the <code>RetentionDays</code> enum to configure a different
 * retention period (including infinite retention).
 * <p>
 * <blockquote><pre>
 * // Configure log group for short retention
 * LogGroup logGroup = LogGroup.Builder.create(stack, "LogGroup")
 *         .retention(RetentionDays.ONE_WEEK)
 *         .build();// Configure log group for infinite retention
 * LogGroup logGroup = LogGroup.Builder.create(stack, "LogGroup")
 *         .retention(Infinity)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>LogRetention</h2>
 * <p>
 * The <code>LogRetention</code> construct is a way to control the retention period of log groups that are created outside of the CDK. The construct is usually
 * used on log groups that are auto created by AWS services, such as <a href="https://docs.aws.amazon.com/lambda/latest/dg/monitoring-cloudwatchlogs.html">AWS
 * lambda</a>.
 * <p>
 * This is implemented using a <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html">CloudFormation custom
 * resource</a>
 * which pre-creates the log group if it doesn't exist, and sets the specified log retention period (never expire, by default).
 * <p>
 * By default, the log group will be created in the same region as the stack. The <code>logGroupRegion</code> property can be used to configure
 * log groups in other regions. This is typically useful when controlling retention for log groups auto-created by global services that
 * publish their log group to a specific region, such as AWS Chatbot creating a log group in <code>us-east-1</code>.
 * <p>
 * <h2>Resource Policy</h2>
 * <p>
 * CloudWatch Resource Policies allow other AWS services or IAM Principals to put log events into the log groups.
 * A resource policy is automatically created when <code>addToResourcePolicy</code> is called on the LogGroup for the first time:
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup = new LogGroup(this, "LogGroup");
 * logGroup.addToResourcePolicy(PolicyStatement.Builder.create()
 *         .actions(List.of("logs:CreateLogStream", "logs:PutLogEvents"))
 *         .principals(List.of(new ServicePrincipal("es.amazonaws.com")))
 *         .resources(List.of(logGroup.getLogGroupArn()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Or more conveniently, write permissions to the log group can be granted as follows which gives same result as in the above example.
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup = new LogGroup(this, "LogGroup");
 * logGroup.grantWrite(new ServicePrincipal("es.amazonaws.com"));
 * </pre></blockquote>
 * <p>
 * <h2>Encrypting Log Groups</h2>
 * <p>
 * By default, log group data is always encrypted in CloudWatch Logs. You have the
 * option to encrypt log group data using a AWS KMS customer master key (CMK) should
 * you not wish to use the default AWS encryption. Keep in mind that if you decide to
 * encrypt a log group, any service or IAM identity that needs to read the encrypted
 * log streams in the future will require the same CMK to decrypt the data.
 * <p>
 * Here's a simple example of creating an encrypted Log Group using a KMS CMK.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * LogGroup.Builder.create(this, "LogGroup")
 *         .encryptionKey(new Key(this, "Key"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See the AWS documentation for more detailed information about <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/encrypt-log-data-kms.html">encrypting CloudWatch
 * Logs</a>.
 * <p>
 * <h2>Subscriptions and Destinations</h2>
 * <p>
 * Log events matching a particular filter can be sent to either a Lambda function
 * or a Kinesis stream.
 * <p>
 * If the Kinesis stream lives in a different account, a <code>CrossAccountDestination</code>
 * object needs to be added in the destination account which will act as a proxy
 * for the remote Kinesis stream. This object is automatically created for you
 * if you use the CDK Kinesis library.
 * <p>
 * Create a <code>SubscriptionFilter</code>, initialize it with an appropriate <code>Pattern</code> (see
 * below) and supply the intended destination:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.destinations.*;
 * Function fn;
 * LogGroup logGroup;
 * 
 * 
 * SubscriptionFilter.Builder.create(this, "Subscription")
 *         .logGroup(logGroup)
 *         .destination(new LambdaDestination(fn))
 *         .filterPattern(FilterPattern.allTerms("ERROR", "MainThread"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Metric Filters</h2>
 * <p>
 * CloudWatch Logs can extract and emit metrics based on a textual log stream.
 * Depending on your needs, this may be a more convenient way of generating metrics
 * for you application than making calls to CloudWatch Metrics yourself.
 * <p>
 * A <code>MetricFilter</code> either emits a fixed number every time it sees a log event
 * matching a particular pattern (see below), or extracts a number from the log
 * event and uses that as the metric value.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * MetricFilter.Builder.create(this, "MetricFilter")
 *         .logGroup(logGroup)
 *         .metricNamespace("MyApp")
 *         .metricName("Latency")
 *         .filterPattern(FilterPattern.exists("$.latency"))
 *         .metricValue("$.latency")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Remember that if you want to use a value from the log event as the metric value,
 * you must mention it in your pattern somewhere.
 * <p>
 * A very simple MetricFilter can be created by using the <code>logGroup.extractMetric()</code>
 * helper function:
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup;
 * 
 * logGroup.extractMetric("$.jsonField", "Namespace", "MetricName");
 * </pre></blockquote>
 * <p>
 * Will extract the value of <code>jsonField</code> wherever it occurs in JSON-structed
 * log records in the LogGroup, and emit them to CloudWatch Metrics under
 * the name <code>Namespace/MetricName</code>.
 * <p>
 * <h3>Exposing Metric on a Metric Filter</h3>
 * <p>
 * You can expose a metric on a metric filter by calling the <code>MetricFilter.metric()</code> API.
 * This has a default of <code>statistic = 'avg'</code> if the statistic is not set in the <code>props</code>.
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup;
 * 
 * MetricFilter mf = MetricFilter.Builder.create(this, "MetricFilter")
 *         .logGroup(logGroup)
 *         .metricNamespace("MyApp")
 *         .metricName("Latency")
 *         .filterPattern(FilterPattern.exists("$.latency"))
 *         .metricValue("$.latency")
 *         .build();
 * 
 * //expose a metric from the metric filter
 * Metric metric = mf.metric();
 * 
 * //you can use the metric to create a new alarm
 * //you can use the metric to create a new alarm
 * Alarm.Builder.create(this, "alarm from metric filter")
 *         .metric(metric)
 *         .threshold(100)
 *         .evaluationPeriods(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Patterns</h2>
 * <p>
 * Patterns describe which log events match a subscription or metric filter. There
 * are three types of patterns:
 * <p>
 * <ul>
 * <li>Text patterns</li>
 * <li>JSON patterns</li>
 * <li>Space-delimited table patterns</li>
 * </ul>
 * <p>
 * All patterns are constructed by using static functions on the <code>FilterPattern</code>
 * class.
 * <p>
 * In addition to the patterns above, the following special patterns exist:
 * <p>
 * <ul>
 * <li><code>FilterPattern.allEvents()</code>: matches all log events.</li>
 * <li><code>FilterPattern.literal(string)</code>: if you already know what pattern expression to
 * use, this function takes a string and will use that as the log pattern. For
 * more information, see the <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html">Filter and Pattern
 * Syntax</a>.</li>
 * </ul>
 * <p>
 * <h3>Text Patterns</h3>
 * <p>
 * Text patterns match if the literal strings appear in the text form of the log
 * line.
 * <p>
 * <ul>
 * <li><code>FilterPattern.allTerms(term, term, ...)</code>: matches if all of the given terms
 * (substrings) appear in the log event.</li>
 * <li><code>FilterPattern.anyTerm(term, term, ...)</code>: matches if all of the given terms
 * (substrings) appear in the log event.</li>
 * <li><code>FilterPattern.anyTermGroup([term, term, ...], [term, term, ...], ...)</code>: matches if
 * all of the terms in any of the groups (specified as arrays) matches. This is
 * an OR match.</li>
 * </ul>
 * <p>
 * Examples:
 * <p>
 * <blockquote><pre>
 * // Search for lines that contain both "ERROR" and "MainThread"
 * IFilterPattern pattern1 = FilterPattern.allTerms("ERROR", "MainThread");
 * 
 * // Search for lines that either contain both "ERROR" and "MainThread", or
 * // both "WARN" and "Deadlock".
 * IFilterPattern pattern2 = FilterPattern.anyTermGroup(List.of("ERROR", "MainThread"), List.of("WARN", "Deadlock"));
 * </pre></blockquote>
 * <p>
 * <h2>JSON Patterns</h2>
 * <p>
 * JSON patterns apply if the log event is the JSON representation of an object
 * (without any other characters, so it cannot include a prefix such as timestamp
 * or log level). JSON patterns can make comparisons on the values inside the
 * fields.
 * <p>
 * <ul>
 * <li><strong>Strings</strong>: the comparison operators allowed for strings are <code>=</code> and <code>!=</code>.
 * String values can start or end with a <code>*</code> wildcard.</li>
 * <li><strong>Numbers</strong>: the comparison operators allowed for numbers are <code>=</code>, <code>!=</code>,
 * <code>&lt;</code>, <code>&lt;=</code>, <code>&gt;</code>, <code>&gt;=</code>.</li>
 * </ul>
 * <p>
 * Fields in the JSON structure are identified by identifier the complete object as <code>$</code>
 * and then descending into it, such as <code>$.field</code> or <code>$.list[0].field</code>.
 * <p>
 * <ul>
 * <li><code>FilterPattern.stringValue(field, comparison, string)</code>: matches if the given
 * field compares as indicated with the given string value.</li>
 * <li><code>FilterPattern.numberValue(field, comparison, number)</code>: matches if the given
 * field compares as indicated with the given numerical value.</li>
 * <li><code>FilterPattern.isNull(field)</code>: matches if the given field exists and has the
 * value <code>null</code>.</li>
 * <li><code>FilterPattern.notExists(field)</code>: matches if the given field is not in the JSON
 * structure.</li>
 * <li><code>FilterPattern.exists(field)</code>: matches if the given field is in the JSON
 * structure.</li>
 * <li><code>FilterPattern.booleanValue(field, boolean)</code>: matches if the given field
 * is exactly the given boolean value.</li>
 * <li><code>FilterPattern.all(jsonPattern, jsonPattern, ...)</code>: matches if all of the
 * given JSON patterns match. This makes an AND combination of the given
 * patterns.</li>
 * <li><code>FilterPattern.any(jsonPattern, jsonPattern, ...)</code>: matches if any of the
 * given JSON patterns match. This makes an OR combination of the given
 * patterns.</li>
 * </ul>
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Search for all events where the component field is equal to
 * // "HttpServer" and either error is true or the latency is higher
 * // than 1000.
 * JsonPattern pattern = FilterPattern.all(FilterPattern.stringValue("$.component", "=", "HttpServer"), FilterPattern.any(FilterPattern.booleanValue("$.error", true), FilterPattern.numberValue("$.latency", "&gt;", 1000)));
 * </pre></blockquote>
 * <p>
 * <h2>Space-delimited table patterns</h2>
 * <p>
 * If the log events are rows of a space-delimited table, this pattern can be used
 * to identify the columns in that structure and add conditions on any of them. The
 * canonical example where you would apply this type of pattern is Apache server
 * logs.
 * <p>
 * Text that is surrounded by <code>"..."</code> quotes or <code>[...]</code> square brackets will
 * be treated as one column.
 * <p>
 * <ul>
 * <li><code>FilterPattern.spaceDelimited(column, column, ...)</code>: construct a
 * <code>SpaceDelimitedTextPattern</code> object with the indicated columns. The columns
 * map one-by-one the columns found in the log event. The string <code>"..."</code> may
 * be used to specify an arbitrary number of unnamed columns anywhere in the
 * name list (but may only be specified once).</li>
 * </ul>
 * <p>
 * After constructing a <code>SpaceDelimitedTextPattern</code>, you can use the following
 * two members to add restrictions:
 * <p>
 * <ul>
 * <li><code>pattern.whereString(field, comparison, string)</code>: add a string condition.
 * The rules are the same as for JSON patterns.</li>
 * <li><code>pattern.whereNumber(field, comparison, number)</code>: add a numerical condition.
 * The rules are the same as for JSON patterns.</li>
 * </ul>
 * <p>
 * Multiple restrictions can be added on the same column; they must all apply.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Search for all events where the component is "HttpServer" and the
 * // result code is not equal to 200.
 * SpaceDelimitedTextPattern pattern = FilterPattern.spaceDelimited("time", "component", "...", "result_code", "latency").whereString("component", "=", "HttpServer").whereNumber("result_code", "!=", 200);
 * </pre></blockquote>
 * <p>
 * <h2>Notes</h2>
 * <p>
 * Be aware that Log Group ARNs will always have the string <code>:*</code> appended to
 * them, to match the behavior of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#aws-resource-logs-loggroup-return-values">the CloudFormation <code>AWS::Logs::LogGroup</code>
 * resource</a>.
 */
package software.amazon.awscdk.services.logs;
