/**
 * <h1>Triggers</h1>
 * <p>
 * Triggers allows you to execute code during deployments. This can be used for a
 * variety of use cases such as:
 * <p>
 * <ul>
 * <li>Self tests: validate something after a resource/construct been provisioned</li>
 * <li>Data priming: add initial data to resources after they are created</li>
 * <li>Preconditions: check things such as account limits or external dependencies
 * before deployment.</li>
 * </ul>
 * <p>
 * <h2>Usage</h2>
 * <p>
 * The <code>TriggerFunction</code> construct will define an AWS Lambda function which is
 * triggered <em>during</em> deployment:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import software.amazon.awscdk.services.lambda.*;
 * import software.amazon.awscdk.triggers.*;
 * import software.amazon.awscdk.Stack;
 * 
 * Stack stack;
 * 
 * 
 * TriggerFunction.Builder.create(stack, "MyTrigger")
 *         .runtime(Runtime.NODEJS_12_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(__dirname + "/my-trigger"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * In the above example, the AWS Lambda function defined in <code>myLambdaFunction</code> will
 * be invoked when the stack is deployed.
 * <p>
 * <h2>Trigger Failures</h2>
 * <p>
 * If the trigger handler fails (e.g. an exception is raised), the CloudFormation
 * deployment will fail, as if a resource failed to provision. This makes it easy
 * to implement "self tests" via triggers by simply making a set of assertions on
 * some provisioned infrastructure.
 * <p>
 * <h2>Order of Execution</h2>
 * <p>
 * By default, a trigger will be executed by CloudFormation after the associated
 * handler is provisioned. This means that if the handler takes an implicit
 * dependency on other resources (e.g. via environment variables), those resources
 * will be provisioned <em>before</em> the trigger is executed.
 * <p>
 * In most cases, implicit ordering should be sufficient, but you can also use
 * <code>executeAfter</code> and <code>executeBefore</code> to control the order of execution.
 * <p>
 * The following example defines the following order: <code>(hello, world) =&gt; myTrigger =&gt; goodbye</code>.
 * The resources under <code>hello</code> and <code>world</code> will be provisioned in
 * parallel, and then the trigger <code>myTrigger</code> will be executed. Only then the
 * resources under <code>goodbye</code> will be provisioned:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * import constructs.Construct;
 * import constructs.Node;
 * import software.amazon.awscdk.triggers.*;
 * 
 * Trigger myTrigger;
 * Construct hello;
 * Construct world;
 * Construct goodbye;
 * 
 * 
 * myTrigger.executeAfter(hello, world);
 * myTrigger.executeBefore(goodbye);
 * </pre></blockquote>
 * <p>
 * Note that <code>hello</code> and <code>world</code> are construct <em>scopes</em>. This means that they can
 * be specific resources (such as an <code>s3.Bucket</code> object) or groups of resources
 * composed together into constructs.
 * <p>
 * <h2>Re-execution of Triggers</h2>
 * <p>
 * By default, <code>executeOnHandlerChange</code> is enabled. This implies that the trigger
 * is re-executed every time the handler function code or configuration changes. If
 * this option is disabled, the trigger will be executed only once upon first
 * deployment.
 * <p>
 * In the future we will consider adding support for additional re-execution modes:
 * <p>
 * <ul>
 * <li><code>executeOnEveryDeployment: boolean</code> - re-executes every time the stack is
 * deployed (add random "salt" during synthesis).</li>
 * <li><code>executeOnResourceChange: Construct[]</code> - re-executes when one of the resources
 * under the specified scopes has changed (add the hash the CloudFormation
 * resource specs).</li>
 * </ul>
 */
package software.amazon.awscdk.triggers;
