/**
 * <h1>Amazon DynamoDB Construct Library</h1>
 * <p>
 * <blockquote>
 * <p>
 * The DynamoDB construct library has two table constructs - <code>Table</code> and <code>TableV2</code>. <code>TableV2</code> is the preferred construct for all use cases, including creating a single table or a table with multiple <code>replicas</code>.
 * <p>
 * </blockquote>
 * <p>
 * <a href="./TABLE_V1_API.md"><code>Table</code> API documentation</a>
 * <p>
 * Here is a minimal deployable DynamoDB table using <code>TableV2</code>:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, <code>TableV2</code> will create a single table in the main deployment region referred to as the primary table. The properties of the primary table are configurable via <code>TableV2</code> properties. For example, consider the following DynamoDB table created using the <code>TableV2</code> construct defined in a <code>Stack</code> being deployed to <code>us-west-2</code>:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .contributorInsights(true)
 *         .tableClass(TableClass.STANDARD_INFREQUENT_ACCESS)
 *         .pointInTimeRecovery(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The above <code>TableV2</code> definition will result in the provisioning of a single table in <code>us-west-2</code> with properties that match the properties set on the <code>TableV2</code> instance.
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/GlobalTables.html
 * <p>
 * <h2>Replicas</h2>
 * <p>
 * The <code>TableV2</code> construct can be configured with replica tables. This will enable you to work with your table as a global table. To do this, the <code>TableV2</code> construct must be defined in a <code>Stack</code> with a defined region. The main deployment region must not be given as a replica because this is created by default with the <code>TableV2</code> construct. The following is a minimal example of defining <code>TableV2</code> with <code>replicas</code>. This <code>TableV2</code> definition will provision three copies of the table - one in <code>us-west-2</code> (primary deployment region), one in <code>us-east-1</code>, and one in <code>us-east-2</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder().region("us-east-2").build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can add new <code>replicas</code> to an instance of the <code>TableV2</code> construct using the <code>addReplica</code> method:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build()))
 *         .build();
 * 
 * globalTable.addReplica(ReplicaTableProps.builder().region("us-east-2").deletionProtection(true).build());
 * </pre></blockquote>
 * <p>
 * The following properties are configurable on a per-replica basis, but will be inherited from the <code>TableV2</code> properties if not specified:
 * <p>
 * <ul>
 * <li>contributorInsights</li>
 * <li>deletionProtection</li>
 * <li>pointInTimeRecovery</li>
 * <li>tableClass</li>
 * <li>readCapacity (only configurable if the <code>TableV2</code> billing mode is <code>PROVISIONED</code>)</li>
 * <li>globalSecondaryIndexes (only <code>contributorInsights</code> and <code>readCapacity</code>)</li>
 * </ul>
 * <p>
 * The following example shows how to define properties on a per-replica basis:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .contributorInsights(true)
 *         .pointInTimeRecovery(true)
 *         .replicas(List.of(ReplicaTableProps.builder()
 *                 .region("us-east-1")
 *                 .tableClass(TableClass.STANDARD_INFREQUENT_ACCESS)
 *                 .pointInTimeRecovery(false)
 *                 .build(), ReplicaTableProps.builder()
 *                 .region("us-east-2")
 *                 .contributorInsights(false)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To obtain an <code>ITableV2</code> reference to a specific replica table, call the <code>replica</code> method on an instance of the <code>TableV2</code> construct and pass the replica region as an argument:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * User user;
 * 
 * 
 * public class FooStack extends Stack {
 *     public final TableV2 globalTable;
 * 
 *     public FooStack(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         this.globalTable = TableV2.Builder.create(this, "GlobalTable")
 *                 .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *                 .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder().region("us-east-2").build()))
 *                 .build();
 *     }
 * }
 * 
 * public class BarStackProps extends StackProps {
 *     private ITableV2 replicaTable;
 *     public ITableV2 getReplicaTable() {
 *         return this.replicaTable;
 *     }
 *     public BarStackProps replicaTable(ITableV2 replicaTable) {
 *         this.replicaTable = replicaTable;
 *         return this;
 *     }
 * }
 * 
 * public class BarStack extends Stack {
 *     public BarStack(Construct scope, String id, BarStackProps props) {
 *         super(scope, id, props);
 * 
 *         // user is given grantWriteData permissions to replica in us-east-1
 *         props.replicaTable.grantWriteData(user);
 *     }
 * }
 * 
 * App app = new App();
 * 
 * FooStack fooStack = FooStack.Builder.create(app, "FooStack").env(Environment.builder().region("us-west-2").build()).build();
 * BarStack barStack = new BarStack(app, "BarStack", new BarStackProps()
 *         .replicaTable(fooStack.globalTable.replica("us-east-1"))
 *         .env(Environment.builder().region("us-east-1").build())
 *         );
 * </pre></blockquote>
 * <p>
 * Note: You can create an instance of the <code>TableV2</code> construct with as many <code>replicas</code> as needed as long as there is only one replica per region. After table creation you can add or remove <code>replicas</code>, but you can only add or remove a single replica in each update.
 * <p>
 * <h2>Billing</h2>
 * <p>
 * The <code>TableV2</code> construct can be configured with on-demand or provisioned billing:
 * <p>
 * <ul>
 * <li>On-demand - The default option. This is a flexible billing option capable of serving requests without capacity planning. The billing mode will be <code>PAY_PER_REQUEST</code>.</li>
 * <li>You can optionally specify the <code>maxReadRequestUnits</code> or <code>maxWriteRequestUnits</code> on individual tables and associated global secondary indexes (GSIs). When you configure maximum throughput for an on-demand table, throughput requests that exceed the maximum amount specified will be throttled.</li>
 * <li>Provisioned - Specify the <code>readCapacity</code> and <code>writeCapacity</code> that you need for your application. The billing mode will be <code>PROVISIONED</code>. Capacity can be configured using one of the following modes:
 * <p>
 * <ul>
 * <li>Fixed - provisioned throughput capacity is configured with a fixed number of I/O operations per second.</li>
 * <li>Autoscaled - provisioned throughput capacity is dynamically adjusted on your behalf in response to actual traffic patterns.</li>
 * </ul></li>
 * </ul>
 * <p>
 * Note: <code>writeCapacity</code> can only be configured using autoscaled capacity.
 * <p>
 * The following example shows how to configure <code>TableV2</code> with on-demand billing:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .billing(Billing.onDemand())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following example shows how to configure <code>TableV2</code> with on-demand billing with optional maximum throughput configured:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .billing(Billing.onDemand(MaxThroughputProps.builder()
 *                 .maxReadRequestUnits(100)
 *                 .maxWriteRequestUnits(115)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When using provisioned billing, you must also specify <code>readCapacity</code> and <code>writeCapacity</code>. You can choose to configure <code>readCapacity</code> with fixed capacity or autoscaled capacity, but <code>writeCapacity</code> can only be configured with autoscaled capacity. The following example shows how to configure <code>TableV2</code> with provisioned billing:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .billing(Billing.provisioned(ThroughputProps.builder()
 *                 .readCapacity(Capacity.fixed(10))
 *                 .writeCapacity(Capacity.autoscaled(AutoscaledCapacityOptions.builder().maxCapacity(15).build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When using provisioned billing, you can configure the <code>readCapacity</code> on a per-replica basis:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .billing(Billing.provisioned(ThroughputProps.builder()
 *                 .readCapacity(Capacity.fixed(10))
 *                 .writeCapacity(Capacity.autoscaled(AutoscaledCapacityOptions.builder().maxCapacity(15).build()))
 *                 .build()))
 *         .replicas(List.of(ReplicaTableProps.builder()
 *                 .region("us-east-1")
 *                 .build(), ReplicaTableProps.builder()
 *                 .region("us-east-2")
 *                 .readCapacity(Capacity.autoscaled(AutoscaledCapacityOptions.builder().maxCapacity(20).targetUtilizationPercent(50).build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When changing the billing for a table from provisioned to on-demand or from on-demand to provisioned, <code>seedCapacity</code> must be configured for each autoscaled resource:
 * <p>
 * <blockquote><pre>
 * TableV2 globalTable = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .billing(Billing.provisioned(ThroughputProps.builder()
 *                 .readCapacity(Capacity.fixed(10))
 *                 .writeCapacity(Capacity.autoscaled(AutoscaledCapacityOptions.builder().maxCapacity(10).seedCapacity(20).build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.ReadWriteCapacityMode.html
 * <p>
 * <h2>Warm Throughput</h2>
 * <p>
 * Warm throughput refers to the number of read and write operations your DynamoDB table can instantaneously support.
 * <p>
 * This optional configuration allows you to pre-warm your table or index to handle anticipated throughput, ensuring optimal performance under expected load.
 * <p>
 * The Warm Throughput configuration settings are automatically replicated across all Global Table replicas.
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("id").type(AttributeType.STRING).build())
 *         .warmThroughput(WarmThroughput.builder()
 *                 .readUnitsPerSecond(15000)
 *                 .writeUnitsPerSecond(20000)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/warm-throughput.html
 * <p>
 * <h2>Encryption</h2>
 * <p>
 * All user data stored in a DynamoDB table is fully encrypted at rest. When creating an instance of the <code>TableV2</code> construct, you can select the following table encryption options:
 * <p>
 * <ul>
 * <li>AWS owned keys - Default encryption type. The keys are owned by DynamoDB (no additional charge).</li>
 * <li>AWS managed keys - The keys are stored in your account and are managed by AWS KMS (AWS KMS charges apply).</li>
 * <li>Customer managed keys - The keys are stored in your account and are created, owned, and managed by you. You have full control over the KMS keys (AWS KMS charges apply).</li>
 * </ul>
 * <p>
 * The following is an example of how to configure <code>TableV2</code> with encryption using an AWS owned key:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .encryption(TableEncryptionV2.dynamoOwnedKey())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following is an example of how to configure <code>TableV2</code> with encryption using an AWS managed key:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .encryption(TableEncryptionV2.awsManagedKey())
 *         .build();
 * </pre></blockquote>
 * <p>
 * When configuring <code>TableV2</code> with encryption using customer managed keys, you must specify the KMS key for the primary table as the <code>tableKey</code>. A map of <code>replicaKeyArns</code> must be provided containing each replica region and the associated KMS key ARN:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * Key tableKey = new Key(stack, "Key");
 * Map&lt;String, String&gt; replicaKeyArns = Map.of(
 *         "us-east-1", "arn:aws:kms:us-east-1:123456789012:key/g24efbna-az9b-42ro-m3bp-cq249l94fca6",
 *         "us-east-2", "arn:aws:kms:us-east-2:123456789012:key/h90bkasj-bs1j-92wp-s2ka-bh857d60bkj8");
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .encryption(TableEncryptionV2.customerManagedKey(tableKey, replicaKeyArns))
 *         .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder().region("us-east-2").build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note: When encryption is configured with customer managed keys, you must have a key already created in each replica region.
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-mgmt
 * <p>
 * <h2>Secondary Indexes</h2>
 * <p>
 * Secondary indexes allow efficient access to data with attributes other than the <code>primaryKey</code>. DynamoDB supports two types of secondary indexes:
 * <p>
 * <ul>
 * <li>Global secondary index - An index with a <code>partitionKey</code> and a <code>sortKey</code> that can be different from those on the base table. A <code>globalSecondaryIndex</code> is considered "global" because queries on the index can span all of the data in the base table, across all partitions. A <code>globalSecondaryIndex</code> is stored in its own partition space away from the base table and scales separately from the base table.</li>
 * <li>Local secondary index - An index that has the same <code>partitionKey</code> as the base table, but a different <code>sortKey</code>. A <code>localSecondaryIndex</code> is "local" in the sense that every partition of a <code>localSecondaryIndex</code> is scoped to a base table partition that has the same <code>partitionKey</code> value.</li>
 * </ul>
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/SecondaryIndexes.html
 * <p>
 * <h3>Global Secondary Indexes</h3>
 * <p>
 * <code>TableV2</code> can be configured with <code>globalSecondaryIndexes</code> by providing them as a <code>TableV2</code> property:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .globalSecondaryIndexes(List.of(GlobalSecondaryIndexPropsV2.builder()
 *                 .indexName("gsi")
 *                 .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can add a <code>globalSecondaryIndex</code> using the <code>addGlobalSecondaryIndex</code> method:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .globalSecondaryIndexes(List.of(GlobalSecondaryIndexPropsV2.builder()
 *                 .indexName("gsi1")
 *                 .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *                 .build()))
 *         .build();
 * 
 * table.addGlobalSecondaryIndex(GlobalSecondaryIndexPropsV2.builder()
 *         .indexName("gsi2")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can configure <code>readCapacity</code> and <code>writeCapacity</code> on a <code>globalSecondaryIndex</code> when an <code>TableV2</code> is configured with provisioned <code>billing</code>. If <code>TableV2</code> is configured with provisioned <code>billing</code> but <code>readCapacity</code> or <code>writeCapacity</code> are not configured on a <code>globalSecondaryIndex</code>, then they will be inherited from the capacity settings specified with the <code>billing</code> configuration:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .billing(Billing.provisioned(ThroughputProps.builder()
 *                 .readCapacity(Capacity.fixed(10))
 *                 .writeCapacity(Capacity.autoscaled(AutoscaledCapacityOptions.builder().maxCapacity(10).build()))
 *                 .build()))
 *         .globalSecondaryIndexes(List.of(GlobalSecondaryIndexPropsV2.builder()
 *                 .indexName("gsi1")
 *                 .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *                 .readCapacity(Capacity.fixed(15))
 *                 .build(), GlobalSecondaryIndexPropsV2.builder()
 *                 .indexName("gsi2")
 *                 .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *                 .writeCapacity(Capacity.autoscaled(AutoscaledCapacityOptions.builder().minCapacity(5).maxCapacity(20).build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * All <code>globalSecondaryIndexes</code> for replica tables are inherited from the primary table. You can configure <code>contributorInsights</code> and <code>readCapacity</code> for each <code>globalSecondaryIndex</code> on a per-replica basis:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .contributorInsights(true)
 *         .billing(Billing.provisioned(ThroughputProps.builder()
 *                 .readCapacity(Capacity.fixed(10))
 *                 .writeCapacity(Capacity.autoscaled(AutoscaledCapacityOptions.builder().maxCapacity(10).build()))
 *                 .build()))
 *         // each global secondary index will inherit contributor insights as true
 *         .globalSecondaryIndexes(List.of(GlobalSecondaryIndexPropsV2.builder()
 *                 .indexName("gsi1")
 *                 .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *                 .readCapacity(Capacity.fixed(15))
 *                 .build(), GlobalSecondaryIndexPropsV2.builder()
 *                 .indexName("gsi2")
 *                 .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *                 .writeCapacity(Capacity.autoscaled(AutoscaledCapacityOptions.builder().minCapacity(5).maxCapacity(20).build()))
 *                 .build()))
 *         .replicas(List.of(ReplicaTableProps.builder()
 *                 .region("us-east-1")
 *                 .globalSecondaryIndexOptions(Map.of(
 *                         "gsi1", ReplicaGlobalSecondaryIndexOptions.builder()
 *                                 .readCapacity(Capacity.autoscaled(AutoscaledCapacityOptions.builder().minCapacity(1).maxCapacity(10).build()))
 *                                 .build()))
 *                 .build(), ReplicaTableProps.builder()
 *                 .region("us-east-2")
 *                 .globalSecondaryIndexOptions(Map.of(
 *                         "gsi2", ReplicaGlobalSecondaryIndexOptions.builder()
 *                                 .contributorInsights(false)
 *                                 .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Local Secondary Indexes</h3>
 * <p>
 * <code>TableV2</code> can only be configured with <code>localSecondaryIndexes</code> when a <code>sortKey</code> is defined as a <code>TableV2</code> property.
 * <p>
 * You can provide <code>localSecondaryIndexes</code> as a <code>TableV2</code> property:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .sortKey(Attribute.builder().name("sk").type(AttributeType.NUMBER).build())
 *         .localSecondaryIndexes(List.of(LocalSecondaryIndexProps.builder()
 *                 .indexName("lsi")
 *                 .sortKey(Attribute.builder().name("sk").type(AttributeType.NUMBER).build())
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can add a <code>localSecondaryIndex</code> using the <code>addLocalSecondaryIndex</code> method:
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .sortKey(Attribute.builder().name("sk").type(AttributeType.NUMBER).build())
 *         .localSecondaryIndexes(List.of(LocalSecondaryIndexProps.builder()
 *                 .indexName("lsi1")
 *                 .sortKey(Attribute.builder().name("sk").type(AttributeType.NUMBER).build())
 *                 .build()))
 *         .build();
 * 
 * table.addLocalSecondaryIndex(LocalSecondaryIndexProps.builder()
 *         .indexName("lsi2")
 *         .sortKey(Attribute.builder().name("sk").type(AttributeType.NUMBER).build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Streams</h2>
 * <p>
 * Each DynamoDB table produces an independent stream based on all its writes, regardless of the origination point for those writes. DynamoDB supports two stream types:
 * <p>
 * <ul>
 * <li>DynamoDB streams - Capture item-level changes in your table, and push the changes to a DynamoDB stream. You then can access the change information through the DynamoDB Streams API.</li>
 * <li>Kinesis streams - Amazon Kinesis Data Streams for DynamoDB captures item-level changes in your table, and replicates the changes to a Kinesis data stream. You then can consume and manage the change information from Kinesis.</li>
 * </ul>
 * <p>
 * <h3>DynamoDB Streams</h3>
 * <p>
 * A <code>dynamoStream</code> can be configured as a <code>TableV2</code> property. If the <code>TableV2</code> instance has replica tables, then all replica tables will inherit the <code>dynamoStream</code> setting from the primary table.  If replicas are configured, but <code>dynamoStream</code> is not configured, then the primary table and all replicas will be automatically configured with the <code>NEW_AND_OLD_IMAGES</code> stream view type.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.kinesis.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * TableV2 globalTable = TableV2.Builder.create(this, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("id").type(AttributeType.STRING).build())
 *         .dynamoStream(StreamViewType.OLD_IMAGE)
 *         // tables in us-west-2, us-east-1, and us-east-2 all have dynamo stream type of OLD_IMAGES
 *         .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder().region("us-east-2").build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Streams.html
 * <p>
 * <h3>Kinesis Streams</h3>
 * <p>
 * A <code>kinesisStream</code> can be configured as a <code>TableV2</code> property. Replica tables will not inherit the <code>kinesisStream</code> configured for the primary table and should added on a per-replica basis.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.kinesis.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * Stream stream1 = new Stream(stack, "Stream1");
 * IStream stream2 = Stream.fromStreamArn(stack, "Stream2", "arn:aws:kinesis:us-east-2:123456789012:stream/my-stream");
 * 
 * TableV2 globalTable = TableV2.Builder.create(this, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("id").type(AttributeType.STRING).build())
 *         .kinesisStream(stream1) // for table in us-west-2
 *         .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder()
 *                 .region("us-east-2")
 *                 .kinesisStream(stream2)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/kds.html
 * <p>
 * <h2>Keys</h2>
 * <p>
 * When an instance of the <code>TableV2</code> construct is defined, you must define its schema using the <code>partitionKey</code> (required) and <code>sortKey</code> (optional) properties.
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .sortKey(Attribute.builder().name("sk").type(AttributeType.NUMBER).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Contributor Insights</h2>
 * <p>
 * Enabling <code>contributorInsights</code> for <code>TableV2</code> will provide information about the most accessed and throttled items in a table or <code>globalSecondaryIndex</code>. DynamoDB delivers this information to you via CloudWatch Contributor Insights rules, reports, and graphs of report data.
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .contributorInsights(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When you use <code>Table</code>, you can enable contributor insights for a table or specific global secondary index by setting <code>contributorInsightsEnabled</code> to <code>true</code>.
 * <p>
 * <blockquote><pre>
 * Table table = Table.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .contributorInsightsEnabled(true)
 *         .build();
 * 
 * table.addGlobalSecondaryIndex(GlobalSecondaryIndexProps.builder()
 *         .contributorInsightsEnabled(true) // for a specific global secondary index
 *         .indexName("gsi")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/contributorinsights_HowItWorks.html
 * <p>
 * <h2>Deletion Protection</h2>
 * <p>
 * <code>deletionProtection</code> determines if your DynamoDB table is protected from deletion and is configurable as a <code>TableV2</code> property. When enabled, the table cannot be deleted by any user or process.
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .deletionProtection(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also specify the <code>removalPolicy</code> as a property of the <code>TableV2</code> construct. This property allows you to control what happens to tables provisioned using <code>TableV2</code> during <code>stack</code> deletion. By default, the <code>removalPolicy</code> is <code>RETAIN</code> which will cause all tables provisioned using <code>TableV2</code> to be retained in the account, but orphaned from the <code>stack</code> they were created in. You can also set the <code>removalPolicy</code> to <code>DESTROY</code> which will delete all tables created using <code>TableV2</code> during <code>stack</code> deletion:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         // applies to all replicas, i.e., us-west-2, us-east-1, us-east-2
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder().region("us-east-2").build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <code>deletionProtection</code> is configurable on a per-replica basis. If the <code>removalPolicy</code> is set to <code>DESTROY</code>, but some <code>replicas</code> have <code>deletionProtection</code> enabled, then only the <code>replicas</code> without <code>deletionProtection</code> will be deleted during <code>stack</code> deletion:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .deletionProtection(true)
 *         // only the replica in us-east-1 will be deleted during stack deletion
 *         .replicas(List.of(ReplicaTableProps.builder()
 *                 .region("us-east-1")
 *                 .deletionProtection(false)
 *                 .build(), ReplicaTableProps.builder()
 *                 .region("us-east-2")
 *                 .deletionProtection(true)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Point-in-Time Recovery</h2>
 * <p>
 * <code>pointInTimeRecovery</code> provides automatic backups of your DynamoDB table data which helps protect your tables from accidental write or delete operations.
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .pointInTimeRecovery(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Table Class</h2>
 * <p>
 * You can configure a <code>TableV2</code> instance with table classes:
 * <p>
 * <ul>
 * <li>STANDARD - the default mode, and is recommended for the vast majority of workloads.</li>
 * <li>STANDARD_INFREQUENT_ACCESS - optimized for tables where storage is the dominant cost.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .tableClass(TableClass.STANDARD_INFREQUENT_ACCESS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.TableClasses.html
 * <p>
 * <h2>Tags</h2>
 * <p>
 * You can add tags to a <code>TableV2</code> in several ways. By adding the tags to the construct itself it will apply the tags to the
 * primary table.
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .tags(List.of(CfnTag.builder().key("primaryTableTagKey").value("primaryTableTagValue").build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also add tags to replica tables by specifying them within the replica table properties.
 * <p>
 * <blockquote><pre>
 * TableV2 table = TableV2.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .replicas(List.of(ReplicaTableProps.builder()
 *                 .region("us-west-1")
 *                 .tags(List.of(CfnTag.builder().key("replicaTableTagKey").value("replicaTableTagValue").build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Referencing Existing Global Tables</h2>
 * <p>
 * To reference an existing DynamoDB table in your CDK application, use the <code>TableV2.fromTableName</code>, <code>TableV2.fromTableArn</code>, or <code>TableV2.fromTableAttributes</code>
 * factory methods:
 * <p>
 * <blockquote><pre>
 * User user;
 * 
 * 
 * ITableV2 table = TableV2.fromTableArn(this, "ImportedTable", "arn:aws:dynamodb:us-east-1:123456789012:table/my-table");
 * // now you can call methods on the referenced table
 * table.grantReadWriteData(user);
 * </pre></blockquote>
 * <p>
 * If you intend to use the <code>tableStreamArn</code> (including indirectly, for example by creating an
 * <code>aws-cdk-lib/aws-lambda-event-sources.DynamoEventSource</code> on the referenced table), you <em>must</em> use the
 * <code>TableV2.fromTableAttributes</code> method and the <code>tableStreamArn</code> property <em>must</em> be populated.
 * <p>
 * To grant permissions to indexes for a referenced table you can either set <code>grantIndexPermissions</code> to <code>true</code>, or you can provide the indexes via the <code>globalIndexes</code> or <code>localIndexes</code> properties. This will enable <code>grant*</code> methods to also grant permissions to <em>all</em> table indexes.
 * <p>
 * <h2>Resource Policy</h2>
 * <p>
 * Using <code>resourcePolicy</code> you can add a <a href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/access-control-resource-based.html">resource policy</a> to a table in the form of a <code>PolicyDocument</code>:
 * <p>
 * <blockquote><pre>
 *     // resource policy document
 *     const policy = new iam.PolicyDocument({
 *       statements: [
 *         new iam.PolicyStatement({
 *           actions: ['dynamodb:GetItem'],
 *           principals: [new iam.AccountRootPrincipal()],
 *           resources: ['*'],
 *         }),
 *       ],
 *     });
 * 
 *     // table with resource policy
 *     new dynamodb.TableV2(this, 'TableTestV2-1', {
 *       partitionKey: {
 *         name: 'id',
 *         type: dynamodb.AttributeType.STRING,
 *       },
 *       removalPolicy: RemovalPolicy.DESTROY,
 *       resourcePolicy: policy,
 *     });
 * </pre></blockquote>
 * <p>
 * TableV2 doesn’t support creating a replica and adding a resource-based policy to that replica in the same stack update in Regions other than the Region where you deploy the stack update.
 * To incorporate a resource-based policy into a replica, you'll need to initially deploy the replica without the policy, followed by a subsequent update to include the desired policy.
 * <p>
 * <h2>Grants</h2>
 * <p>
 * Using any of the <code>grant*</code> methods on an instance of the <code>TableV2</code> construct will only apply to the primary table, its indexes, and any associated <code>encryptionKey</code>. As an example, <code>grantReadData</code> used below will only apply the table in <code>us-west-2</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.kms.*;
 * 
 * User user;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * Key tableKey = new Key(stack, "Key");
 * Map&lt;String, String&gt; replicaKeyArns = Map.of(
 *         "us-east-1", "arn:aws:kms:us-east-1:123456789012:key/g24efbna-az9b-42ro-m3bp-cq249l94fca6",
 *         "us-east-2", "arn:aws:kms:us-east-2:123456789012:key/g24efbna-az9b-42ro-m3bp-cq249l94fca6");
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .encryption(TableEncryptionV2.customerManagedKey(tableKey, replicaKeyArns))
 *         .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder().region("us-east-2").build()))
 *         .build();
 * 
 * // grantReadData only applies to the table in us-west-2 and the tableKey
 * globalTable.grantReadData(user);
 * </pre></blockquote>
 * <p>
 * The <code>replica</code> method can be used to grant to a specific replica table:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.kms.*;
 * 
 * User user;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * Key tableKey = new Key(stack, "Key");
 * Map&lt;String, String&gt; replicaKeyArns = Map.of(
 *         "us-east-1", "arn:aws:kms:us-east-1:123456789012:key/g24efbna-az9b-42ro-m3bp-cq249l94fca6",
 *         "us-east-2", "arn:aws:kms:us-east-2:123456789012:key/g24efbna-az9b-42ro-m3bp-cq249l94fca6");
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .encryption(TableEncryptionV2.customerManagedKey(tableKey, replicaKeyArns))
 *         .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder().region("us-east-2").build()))
 *         .build();
 * 
 * // grantReadData applies to the table in us-east-2 and the key arn for the key in us-east-2
 * globalTable.replica("us-east-2").grantReadData(user);
 * </pre></blockquote>
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * You can use <code>metric*</code> methods to generate metrics for a table that can be used when configuring an <code>Alarm</code> or <code>Graphs</code>. The <code>metric*</code> methods only apply to the primary table provisioned using the <code>TableV2</code> construct. As an example, <code>metricConsumedReadCapacityUnits</code> used below is only for the table in <code>us-west-2</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * 
 * App app = new App();
 * Stack stack = Stack.Builder.create(app, "Stack").env(Environment.builder().region("us-west-2").build()).build();
 * 
 * TableV2 globalTable = TableV2.Builder.create(stack, "GlobalTable")
 *         .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *         .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder().region("us-east-2").build()))
 *         .build();
 * 
 * // metric is only for the table in us-west-2
 * Metric metric = globalTable.metricConsumedReadCapacityUnits();
 * 
 * Alarm.Builder.create(this, "Alarm")
 *         .metric(metric)
 *         .evaluationPeriods(1)
 *         .threshold(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>replica</code> method can be used to generate a metric for a specific replica table:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * 
 * public class FooStack extends Stack {
 *     public final TableV2 globalTable;
 * 
 *     public FooStack(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         this.globalTable = TableV2.Builder.create(this, "GlobalTable")
 *                 .partitionKey(Attribute.builder().name("pk").type(AttributeType.STRING).build())
 *                 .replicas(List.of(ReplicaTableProps.builder().region("us-east-1").build(), ReplicaTableProps.builder().region("us-east-2").build()))
 *                 .build();
 *     }
 * }
 * 
 * public class BarStackProps extends StackProps {
 *     private ITableV2 replicaTable;
 *     public ITableV2 getReplicaTable() {
 *         return this.replicaTable;
 *     }
 *     public BarStackProps replicaTable(ITableV2 replicaTable) {
 *         this.replicaTable = replicaTable;
 *         return this;
 *     }
 * }
 * 
 * public class BarStack extends Stack {
 *     public BarStack(Construct scope, String id, BarStackProps props) {
 *         super(scope, id, props);
 * 
 *         // metric is only for the table in us-east-1
 *         Metric metric = props.replicaTable.metricConsumedReadCapacityUnits();
 * 
 *         Alarm.Builder.create(this, "Alarm")
 *                 .metric(metric)
 *                 .evaluationPeriods(1)
 *                 .threshold(1)
 *                 .build();
 *     }
 * }
 * 
 * App app = new App();
 * FooStack fooStack = FooStack.Builder.create(app, "FooStack").env(Environment.builder().region("us-west-2").build()).build();
 * BarStack barStack = new BarStack(app, "BarStack", new BarStackProps()
 *         .replicaTable(fooStack.globalTable.replica("us-east-1"))
 *         .env(Environment.builder().region("us-east-1").build())
 *         );
 * </pre></blockquote>
 * <p>
 * <h2>import from S3 Bucket</h2>
 * <p>
 * You can import data in S3 when creating a Table using the <code>Table</code> construct.
 * To import data into DynamoDB, it is required that your data is in a CSV, DynamoDB JSON, or Amazon Ion format within an Amazon S3 bucket.
 * The data may be compressed using ZSTD or GZIP formats, or you may choose to import it without compression.
 * The data source can be a single S3 object or multiple S3 objects sharing a common prefix.
 * <p>
 * Further reading:
 * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/S3DataImport.HowItWorks.html
 * <p>
 * <h3>use CSV format</h3>
 * <p>
 * The <code>InputFormat.csv</code> method accepts <code>delimiter</code> and <code>headerList</code> options as arguments.
 * If <code>delimiter</code> is not specified, <code>,</code> is used by default.
 * And if <code>headerList</code> is specified, the first line of CSV is treated as data instead of header.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.s3.*;
 * 
 * IBucket bucket;
 * 
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "Stack");
 * 
 * Table.Builder.create(stack, "Table")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .importSource(ImportSourceSpecification.builder()
 *                 .compressionType(InputCompressionType.GZIP)
 *                 .inputFormat(InputFormat.csv(CsvOptions.builder()
 *                         .delimiter(",")
 *                         .headerList(List.of("id", "name"))
 *                         .build()))
 *                 .bucket(bucket)
 *                 .keyPrefix("prefix")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>use DynamoDB JSON format</h3>
 * <p>
 * Use the <code>InputFormat.dynamoDBJson()</code> method to specify the <code>inputFormat</code> property.
 * There are currently no options available.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.s3.*;
 * 
 * IBucket bucket;
 * 
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "Stack");
 * 
 * Table.Builder.create(stack, "Table")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .importSource(ImportSourceSpecification.builder()
 *                 .compressionType(InputCompressionType.GZIP)
 *                 .inputFormat(InputFormat.dynamoDBJson())
 *                 .bucket(bucket)
 *                 .keyPrefix("prefix")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>use Amazon Ion format</h3>
 * <p>
 * Use the <code>InputFormat.ion()</code> method to specify the <code>inputFormat</code> property.
 * There are currently no options available.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.s3.*;
 * 
 * IBucket bucket;
 * 
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "Stack");
 * 
 * Table.Builder.create(stack, "Table")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .importSource(ImportSourceSpecification.builder()
 *                 .compressionType(InputCompressionType.GZIP)
 *                 .inputFormat(InputFormat.ion())
 *                 .bucket(bucket)
 *                 .keyPrefix("prefix")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.dynamodb;
