/**
 * <h1>Amazon ECS Construct Library</h1>
 * <p>
 * This package contains constructs for working with <strong>Amazon Elastic Container
 * Service</strong> (Amazon ECS).
 * <p>
 * Amazon Elastic Container Service (Amazon ECS) is a fully managed container orchestration service.
 * <p>
 * For further information on Amazon ECS,
 * see the <a href="https://docs.aws.amazon.com/ecs">Amazon ECS documentation</a>
 * <p>
 * The following example creates an Amazon ECS cluster, adds capacity to it, and
 * runs a service on it:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * // Create an ECS cluster
 * Cluster cluster = Cluster.Builder.create(this, "Cluster").vpc(vpc).build();
 * 
 * // Add capacity to it
 * cluster.addCapacity("DefaultAutoScalingGroupCapacity", AddCapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .desiredCapacity(3)
 *         .build());
 * 
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * 
 * taskDefinition.addContainer("DefaultContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(512)
 *         .build());
 * 
 * // Instantiate an Amazon ECS Service
 * Ec2Service ecsService = Ec2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * For a set of constructs defining common ECS architectural patterns, see the <code>aws-cdk-lib/aws-ecs-patterns</code> package.
 * <p>
 * <h2>Launch Types: AWS Fargate vs Amazon EC2 vs AWS ECS Anywhere</h2>
 * <p>
 * There are three sets of constructs in this library:
 * <p>
 * <ul>
 * <li>Use the <code>Ec2TaskDefinition</code> and <code>Ec2Service</code> constructs to run tasks on Amazon EC2 instances running in your account.</li>
 * <li>Use the <code>FargateTaskDefinition</code> and <code>FargateService</code> constructs to run tasks on
 * instances that are managed for you by AWS.</li>
 * <li>Use the <code>ExternalTaskDefinition</code> and <code>ExternalService</code> constructs to run AWS ECS Anywhere tasks on self-managed infrastructure.</li>
 * </ul>
 * <p>
 * Here are the main differences:
 * <p>
 * <ul>
 * <li><strong>Amazon EC2</strong>: instances are under your control. Complete control of task to host
 * allocation. Required to specify at least a memory reservation or limit for
 * every container. Can use Host, Bridge and AwsVpc networking modes. Can attach
 * Classic Load Balancer. Can share volumes between container and host.</li>
 * <li><strong>AWS Fargate</strong>: tasks run on AWS-managed instances, AWS manages task to host
 * allocation for you. Requires specification of memory and cpu sizes at the
 * taskdefinition level. Only supports AwsVpc networking modes and
 * Application/Network Load Balancers. Only the AWS log driver is supported.
 * Many host features are not supported such as adding kernel capabilities
 * and mounting host devices/volumes inside the container.</li>
 * <li><strong>AWS ECS Anywhere</strong>: tasks are run and managed by AWS ECS Anywhere on infrastructure
 * owned by the customer. Bridge, Host and None networking modes are supported. Does not
 * support autoscaling, load balancing, cloudmap or attachment of volumes.</li>
 * </ul>
 * <p>
 * For more information on Amazon EC2 vs AWS Fargate, networking and ECS Anywhere see the AWS Documentation:
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate</a>,
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task Networking</a>,
 * <a href="https://aws.amazon.com/ecs/anywhere/">ECS Anywhere</a>
 * <p>
 * <h2>Clusters</h2>
 * <p>
 * A <code>Cluster</code> defines the infrastructure to run your
 * tasks on. You can run many tasks on a single cluster.
 * <p>
 * The following code creates a cluster that can run AWS Fargate tasks:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To encrypt the fargate ephemeral storage configure a KMS key.
 * <p>
 * <blockquote><pre>
 * Key key;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .managedStorageConfiguration(ManagedStorageConfiguration.builder()
 *                 .fargateEphemeralStorageKmsKey(key)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following code imports an existing cluster using the ARN which can be used to
 * import an Amazon ECS service either EC2 or Fargate.
 * <p>
 * <blockquote><pre>
 * String clusterArn = "arn:aws:ecs:us-east-1:012345678910:cluster/clusterName";
 * 
 * ICluster cluster = Cluster.fromClusterArn(this, "Cluster", clusterArn);
 * </pre></blockquote>
 * <p>
 * To use tasks with Amazon EC2 launch-type, you have to add capacity to
 * the cluster in order for tasks to be scheduled on your instances.  Typically,
 * you add an AutoScalingGroup with instances running the latest
 * Amazon ECS-optimized AMI to the cluster. There is a method to build and add such an
 * AutoScalingGroup automatically, or you can supply a customized AutoScalingGroup
 * that you construct yourself. It's possible to add multiple AutoScalingGroups
 * with various instance types.
 * <p>
 * The following example creates an Amazon ECS cluster and adds capacity to it:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * 
 * // Either add default capacity
 * cluster.addCapacity("DefaultAutoScalingGroupCapacity", AddCapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .desiredCapacity(3)
 *         .build());
 * 
 * // Or add customized capacity. Be sure to start the Amazon ECS-optimized AMI.
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .machineImage(EcsOptimizedImage.amazonLinux())
 *         // Or use Amazon ECS-Optimized Amazon Linux 2 AMI
 *         // machineImage: EcsOptimizedImage.amazonLinux2(),
 *         .desiredCapacity(3)
 *         .build();
 * 
 * AsgCapacityProvider capacityProvider = AsgCapacityProvider.Builder.create(this, "AsgCapacityProvider")
 *         .autoScalingGroup(autoScalingGroup)
 *         .build();
 * cluster.addAsgCapacityProvider(capacityProvider);
 * </pre></blockquote>
 * <p>
 * If you omit the property <code>vpc</code>, the construct will create a new VPC with two AZs.
 * <p>
 * By default, all machine images will auto-update to the latest version
 * on each deployment, causing a replacement of the instances in your AutoScalingGroup
 * if the AMI has been updated since the last deployment.
 * <p>
 * If task draining is enabled, ECS will transparently reschedule tasks on to the new
 * instances before terminating your old instances. If you have disabled task draining,
 * the tasks will be terminated along with the instance. To prevent that, you
 * can pick a non-updating AMI by passing <code>cacheInContext: true</code>, but be sure
 * to periodically update to the latest AMI manually by using the <a href="https://docs.aws.amazon.com/cdk/latest/guide/context.html">CDK CLI
 * context management commands</a>:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .machineImage(EcsOptimizedImage.amazonLinux(EcsOptimizedImageOptions.builder().cachedInContext(true).build()))
 *         .vpc(vpc)
 *         .instanceType(new InstanceType("t2.micro"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To use <code>LaunchTemplate</code> with <code>AsgCapacityProvider</code>, make sure to specify the <code>userData</code> in the <code>LaunchTemplate</code>:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * LaunchTemplate launchTemplate = LaunchTemplate.Builder.create(this, "ASG-LaunchTemplate")
 *         .instanceType(new InstanceType("t3.medium"))
 *         .machineImage(EcsOptimizedImage.amazonLinux2())
 *         .userData(UserData.forLinux())
 *         .build();
 * 
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .mixedInstancesPolicy(MixedInstancesPolicy.builder()
 *                 .instancesDistribution(InstancesDistribution.builder()
 *                         .onDemandPercentageAboveBaseCapacity(50)
 *                         .build())
 *                 .launchTemplate(launchTemplate)
 *                 .build())
 *         .build();
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster").vpc(vpc).build();
 * 
 * AsgCapacityProvider capacityProvider = AsgCapacityProvider.Builder.create(this, "AsgCapacityProvider")
 *         .autoScalingGroup(autoScalingGroup)
 *         .machineImageType(MachineImageType.AMAZON_LINUX_2)
 *         .build();
 * 
 * cluster.addAsgCapacityProvider(capacityProvider);
 * </pre></blockquote>
 * <p>
 * The following code retrieve the Amazon Resource Names (ARNs) of tasks that are a part of a specified ECS cluster.
 * It's useful when you want to grant permissions to a task to access other AWS resources.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * String taskARNs = cluster.arnForTasks("*"); // arn:aws:ecs:&lt;region&gt;:&lt;regionId&gt;:task/&lt;clusterName&gt;/*
 * 
 * // Grant the task permission to access other AWS resources
 * taskDefinition.addToTaskRolePolicy(
 * PolicyStatement.Builder.create()
 *         .actions(List.of("ecs:UpdateTaskProtection"))
 *         .resources(List.of(taskARNs))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To manage task protection settings in an ECS cluster, you can use the <code>grantTaskProtection</code> method.
 * This method grants the <code>ecs:UpdateTaskProtection</code> permission to a specified IAM entity.
 * <p>
 * <blockquote><pre>
 * // Assume 'cluster' is an instance of ecs.Cluster
 * Cluster cluster;
 * Role taskRole;
 * 
 * 
 * // Grant ECS Task Protection permissions to the role
 * // Now 'taskRole' has the 'ecs:UpdateTaskProtection' permission on all tasks in the cluster
 * cluster.grantTaskProtection(taskRole);
 * </pre></blockquote>
 * <p>
 * <h3>Bottlerocket</h3>
 * <p>
 * <a href="https://aws.amazon.com/bottlerocket/">Bottlerocket</a> is a Linux-based open source operating system that is
 * purpose-built by AWS for running containers. You can launch Amazon ECS container instances with the Bottlerocket AMI.
 * <p>
 * The following example will create a capacity with self-managed Amazon EC2 capacity of 2 <code>c5.large</code> Linux instances running with <code>Bottlerocket</code> AMI.
 * <p>
 * The following example adds Bottlerocket capacity to the cluster:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * cluster.addCapacity("bottlerocket-asg", AddCapacityOptions.builder()
 *         .minCapacity(2)
 *         .instanceType(new InstanceType("c5.large"))
 *         .machineImage(new BottleRocketImage())
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also specify an NVIDIA-compatible AMI such as in this example:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * cluster.addCapacity("bottlerocket-asg", AddCapacityOptions.builder()
 *         .instanceType(new InstanceType("p3.2xlarge"))
 *         .machineImage(BottleRocketImage.Builder.create()
 *                 .variant(BottlerocketEcsVariant.AWS_ECS_2_NVIDIA)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>ARM64 (Graviton) Instances</h3>
 * <p>
 * To launch instances with ARM64 hardware, you can use the Amazon ECS-optimized
 * Amazon Linux 2 (arm64) AMI. Based on Amazon Linux 2, this AMI is recommended
 * for use when launching your EC2 instances that are powered by Arm-based AWS
 * Graviton Processors.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * cluster.addCapacity("graviton-cluster", AddCapacityOptions.builder()
 *         .minCapacity(2)
 *         .instanceType(new InstanceType("c6g.large"))
 *         .machineImage(EcsOptimizedImage.amazonLinux2(AmiHardwareType.ARM))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Bottlerocket is also supported:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * cluster.addCapacity("graviton-cluster", AddCapacityOptions.builder()
 *         .minCapacity(2)
 *         .instanceType(new InstanceType("c6g.large"))
 *         .machineImageType(MachineImageType.BOTTLEROCKET)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Amazon Linux 2 (Neuron) Instances</h3>
 * <p>
 * To launch Amazon EC2 Inf1, Trn1 or Inf2 instances, you can use the Amazon ECS optimized Amazon Linux 2 (Neuron) AMI. It comes pre-configured with AWS Inferentia and AWS Trainium drivers and the AWS Neuron runtime for Docker which makes running machine learning inference workloads easier on Amazon ECS.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * cluster.addCapacity("neuron-cluster", AddCapacityOptions.builder()
 *         .minCapacity(2)
 *         .instanceType(new InstanceType("inf1.xlarge"))
 *         .machineImage(EcsOptimizedImage.amazonLinux2(AmiHardwareType.NEURON))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Spot Instances</h3>
 * <p>
 * To add spot instances into the cluster, you must specify the <code>spotPrice</code> in the <code>ecs.AddCapacityOptions</code> and optionally enable the <code>spotInstanceDraining</code> property.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * // Add an AutoScalingGroup with spot instances to the existing cluster
 * cluster.addCapacity("AsgSpot", AddCapacityOptions.builder()
 *         .maxCapacity(2)
 *         .minCapacity(2)
 *         .desiredCapacity(2)
 *         .instanceType(new InstanceType("c5.xlarge"))
 *         .spotPrice("0.0735")
 *         // Enable the Automated Spot Draining support for Amazon ECS
 *         .spotInstanceDraining(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>SNS Topic Encryption</h3>
 * <p>
 * When the <code>ecs.AddCapacityOptions</code> that you provide has a non-zero <code>taskDrainTime</code> (the default) then an SNS topic and Lambda are created to ensure that the
 * cluster's instances have been properly drained of tasks before terminating. The SNS Topic is sent the instance-terminating lifecycle event from the AutoScalingGroup,
 * and the Lambda acts on that event. If you wish to engage <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html">server-side encryption</a> for this SNS Topic
 * then you may do so by providing a KMS key for the <code>topicEncryptionKey</code> property of <code>ecs.AddCapacityOptions</code>.
 * <p>
 * <blockquote><pre>
 * // Given
 * Cluster cluster;
 * Key key;
 * 
 * // Then, use that key to encrypt the lifecycle-event SNS Topic.
 * cluster.addCapacity("ASGEncryptedSNS", AddCapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .desiredCapacity(3)
 *         .topicEncryptionKey(key)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Container Insights</h3>
 * <p>
 * On a cluster, CloudWatch Container Insights can be enabled by setting the <code>containerInsightsV2</code> property. <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cloudwatch-container-insights.html">Container Insights</a>
 * can be disabled, enabled, or enhanced.
 * <p>
 * <blockquote><pre>
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .containerInsightsV2(ContainerInsights.ENHANCED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Task definitions</h2>
 * <p>
 * A task definition describes what a single copy of a <strong>task</strong> should look like.
 * A task definition has one or more containers; typically, it has one
 * main container (the <em>default container</em> is the first one that's added
 * to the task definition, and it is marked <em>essential</em>) and optionally
 * some supporting containers which are used to support the main container,
 * doings things like upload logs or metrics to monitoring services.
 * <p>
 * To run a task or service with Amazon EC2 launch type, use the <code>Ec2TaskDefinition</code>. For AWS Fargate tasks/services, use the
 * <code>FargateTaskDefinition</code>. For AWS ECS Anywhere use the <code>ExternalTaskDefinition</code>. These classes
 * provide simplified APIs that only contain properties relevant for each specific launch type.
 * <p>
 * For a <code>FargateTaskDefinition</code>, specify the task size (<code>memoryLimitMiB</code> and <code>cpu</code>):
 * <p>
 * <blockquote><pre>
 * FargateTaskDefinition fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .build();
 * </pre></blockquote>
 * <p>
 * On Fargate Platform Version 1.4.0 or later, you may specify up to 200GiB of
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-task-storage.html#fargate-task-storage-pv14">ephemeral storage</a>:
 * <p>
 * <blockquote><pre>
 * FargateTaskDefinition fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .ephemeralStorageGiB(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To specify the process namespace to use for the containers in the task, use the <code>pidMode</code> property:
 * <p>
 * <blockquote><pre>
 * FargateTaskDefinition fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .runtimePlatform(RuntimePlatform.builder()
 *                 .operatingSystemFamily(OperatingSystemFamily.LINUX)
 *                 .cpuArchitecture(CpuArchitecture.ARM64)
 *                 .build())
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .pidMode(PidMode.TASK)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note:</strong> <code>pidMode</code> is only supported for tasks that are hosted on AWS Fargate if the tasks are using platform version 1.4.0
 * or later (Linux). Only the <code>task</code> option is supported for Linux containers. <code>pidMode</code> isn't supported for Windows containers on Fargate.
 * If <code>pidMode</code> is specified for a Fargate task, then <code>runtimePlatform.operatingSystemFamily</code> must also be specified.
 * <p>
 * To add containers to a task definition, call <code>addContainer()</code>:
 * <p>
 * <blockquote><pre>
 * FargateTaskDefinition fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .build();
 * ContainerDefinition container = fargateTaskDefinition.addContainer("WebContainer", ContainerDefinitionOptions.builder()
 *         // Use an image from DockerHub
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * For an <code>Ec2TaskDefinition</code>:
 * <p>
 * <blockquote><pre>
 * Ec2TaskDefinition ec2TaskDefinition = Ec2TaskDefinition.Builder.create(this, "TaskDef")
 *         .networkMode(NetworkMode.BRIDGE)
 *         .build();
 * 
 * ContainerDefinition container = ec2TaskDefinition.addContainer("WebContainer", ContainerDefinitionOptions.builder()
 *         // Use an image from DockerHub
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .build());
 * </pre></blockquote>
 * <p>
 * For an <code>ExternalTaskDefinition</code>:
 * <p>
 * <blockquote><pre>
 * ExternalTaskDefinition externalTaskDefinition = new ExternalTaskDefinition(this, "TaskDef");
 * 
 * ContainerDefinition container = externalTaskDefinition.addContainer("WebContainer", ContainerDefinitionOptions.builder()
 *         // Use an image from DockerHub
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can specify container properties when you add them to the task definition, or with various methods, e.g.:
 * <p>
 * To add a port mapping when adding a container to the task definition, specify the <code>portMappings</code> option:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * 
 * taskDefinition.addContainer("WebContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .portMappings(List.of(PortMapping.builder().containerPort(3000).build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To add port mappings directly to a container definition, call <code>addPortMappings()</code>:
 * <p>
 * <blockquote><pre>
 * ContainerDefinition container;
 * 
 * 
 * container.addPortMappings(PortMapping.builder()
 *         .containerPort(3000)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Sometimes it is useful to be able to configure port ranges for a container, e.g. to run applications such as game servers
 * and real-time streaming which typically require multiple ports to be opened simultaneously. This feature is supported on
 * both Linux and Windows operating systems for both the EC2 and AWS Fargate launch types. There is a maximum limit of 100
 * port ranges per container, and you cannot specify overlapping port ranges.
 * <p>
 * Docker recommends that you turn off the <code>docker-proxy</code> in the Docker daemon config file when you have a large number of ports.
 * For more information, see <a href="https://github.com/moby/moby/issues/11185">Issue #11185</a> on the GitHub website.
 * <p>
 * <blockquote><pre>
 * ContainerDefinition container;
 * 
 * 
 * container.addPortMappings(PortMapping.builder()
 *         .containerPort(ContainerDefinition.CONTAINER_PORT_USE_RANGE)
 *         .containerPortRange("8080-8081")
 *         .build());
 * </pre></blockquote>
 * <p>
 * To add data volumes to a task definition, call <code>addVolume()</code>:
 * <p>
 * <blockquote><pre>
 * FargateTaskDefinition fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .build();
 * Map&lt;String, Object&gt; volume = Map.of(
 *         // Use an Elastic FileSystem
 *         "name", "mydatavolume",
 *         "efsVolumeConfiguration", Map.of(
 *                 "fileSystemId", "EFS"));
 * 
 * void container = fargateTaskDefinition.addVolume(volume);
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: ECS Anywhere doesn't support volume attachments in the task definition.
 * <p>
 * </blockquote>
 * <p>
 * To use a TaskDefinition that can be used with either Amazon EC2 or
 * AWS Fargate launch types, use the <code>TaskDefinition</code> construct.
 * <p>
 * When creating a task definition you have to specify what kind of
 * tasks you intend to run: Amazon EC2, AWS Fargate, or both.
 * The following example uses both:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition = TaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryMiB("512")
 *         .cpu("256")
 *         .networkMode(NetworkMode.AWS_VPC)
 *         .compatibility(Compatibility.EC2_AND_FARGATE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To grant a principal permission to run your <code>TaskDefinition</code>, you can use the <code>TaskDefinition.grantRun()</code> method:
 * <p>
 * <blockquote><pre>
 * IGrantable role;
 * 
 * TaskDefinition taskDef = TaskDefinition.Builder.create(this, "TaskDef")
 *         .cpu("512")
 *         .memoryMiB("512")
 *         .compatibility(Compatibility.EC2_AND_FARGATE)
 *         .build();
 * 
 * // Gives role required permissions to run taskDef
 * taskDef.grantRun(role);
 * </pre></blockquote>
 * <p>
 * To deploy containerized applications that require the allocation of standard input (stdin) or a terminal (tty), use the <code>interactive</code> property.
 * <p>
 * This parameter corresponds to <code>OpenStdin</code> in the <a href="https://docs.docker.com/engine/api/v1.35/#tag/Container/operation/ContainerCreate">Create a container</a> section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a>
 * and the <code>--interactive</code> option to <a href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * 
 * taskDefinition.addContainer("Container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .interactive(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Images</h3>
 * <p>
 * Images supply the software that runs inside the container. Images can be
 * obtained from either DockerHub or from ECR repositories, built directly from a local Dockerfile, or use an existing tarball.
 * <p>
 * <ul>
 * <li><code>ecs.ContainerImage.fromRegistry(imageName)</code>: use a public image.</li>
 * <li><code>ecs.ContainerImage.fromRegistry(imageName, { credentials: mySecret })</code>: use a private image that requires credentials.</li>
 * <li><code>ecs.ContainerImage.fromEcrRepository(repo, tagOrDigest)</code>: use the given ECR repository as the image
 * to start. If no tag or digest is provided, "latest" is assumed.</li>
 * <li><code>ecs.ContainerImage.fromAsset('./image')</code>: build and upload an
 * image directly from a <code>Dockerfile</code> in your source directory.</li>
 * <li><code>ecs.ContainerImage.fromDockerImageAsset(asset)</code>: uses an existing
 * <code>aws-cdk-lib/aws-ecr-assets.DockerImageAsset</code> as a container image.</li>
 * <li><code>ecs.ContainerImage.fromTarball(file)</code>: use an existing tarball.</li>
 * <li><code>new ecs.TagParameterContainerImage(repository)</code>: use the given ECR repository as the image
 * but a CloudFormation parameter as the tag.</li>
 * </ul>
 * <p>
 * <h3>Environment variables</h3>
 * <p>
 * To pass environment variables to the container, you can use the <code>environment</code>, <code>environmentFiles</code>, and <code>secrets</code> props.
 * <p>
 * <blockquote><pre>
 * Secret secret;
 * Secret dbSecret;
 * StringParameter parameter;
 * TaskDefinition taskDefinition;
 * Bucket s3Bucket;
 * 
 * 
 * ContainerDefinition newContainer = taskDefinition.addContainer("container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .environment(Map.of( // clear text, not for sensitive data
 *                 "STAGE", "prod"))
 *         .environmentFiles(List.of(EnvironmentFile.fromAsset("./demo-env-file.env"), EnvironmentFile.fromBucket(s3Bucket, "assets/demo-env-file.env")))
 *         .secrets(Map.of( // Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store at container start-up.
 *                 "SECRET", Secret.fromSecretsManager(secret),
 *                 "DB_PASSWORD", Secret.fromSecretsManager(dbSecret, "password"),  // Reference a specific JSON field, (requires platform version 1.4.0 or later for Fargate tasks)
 *                 "API_KEY", Secret.fromSecretsManagerVersion(secret, SecretVersionInfo.builder().versionId("12345").build(), "apiKey"),  // Reference a specific version of the secret by its version id or version stage (requires platform version 1.4.0 or later for Fargate tasks)
 *                 "PARAMETER", Secret.fromSsmParameter(parameter)))
 *         .build());
 * newContainer.addEnvironment("QUEUE_NAME", "MyQueue");
 * newContainer.addSecret("API_KEY", Secret.fromSecretsManager(secret));
 * newContainer.addSecret("DB_PASSWORD", Secret.fromSecretsManager(secret, "password"));
 * </pre></blockquote>
 * <p>
 * The task execution role is automatically granted read permissions on the secrets/parameters. Further details provided in the AWS documentation
 * about <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">specifying environment variables</a>.
 * <p>
 * <h3>Linux parameters</h3>
 * <p>
 * To apply additional linux-specific options related to init process and memory management to the container, use the <code>linuxParameters</code> property:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * 
 * taskDefinition.addContainer("container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .linuxParameters(LinuxParameters.Builder.create(this, "LinuxParameters")
 *                 .initProcessEnabled(true)
 *                 .sharedMemorySize(1024)
 *                 .maxSwap(Size.mebibytes(5000))
 *                 .swappiness(90)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>System controls</h3>
 * <p>
 * To set system controls (kernel parameters) on the container, use the <code>systemControls</code> prop:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * 
 * taskDefinition.addContainer("container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .systemControls(List.of(SystemControl.builder()
 *                 .namespace("net.ipv6.conf.all.default.disable_ipv6")
 *                 .value("1")
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Restart policy</h3>
 * <p>
 * To enable a restart policy for the container, set <code>enableRestartPolicy</code> to true and also specify
 * <code>restartIgnoredExitCodes</code> and <code>restartAttemptPeriod</code> if necessary.
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * 
 * taskDefinition.addContainer("container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .enableRestartPolicy(true)
 *         .restartIgnoredExitCodes(List.of(0, 127))
 *         .restartAttemptPeriod(Duration.seconds(360))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Enable Fault Injection</h3>
 * <p>
 * You can utilize fault injection with Amazon ECS on both Amazon EC2 and Fargate to test how their application responds to certain impairment scenarios. These tests provide information you can use to optimize your application's performance and resiliency.
 * <p>
 * When fault injection is enabled, the Amazon ECS container agent allows tasks access to new fault injection endpoints.
 * Fault injection only works with tasks using the <code>AWS_VPC</code> or <code>HOST</code> network modes.
 * <p>
 * For more infomation, see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fault-injection.html">Use fault injection with your Amazon ECS and Fargate workloads</a>.
 * <p>
 * To enable Fault Injection for the task definiton, set <code>enableFaultInjection</code> to true.
 * <p>
 * <blockquote><pre>
 * Ec2TaskDefinition.Builder.create(this, "Ec2TaskDefinition")
 *         .enableFaultInjection(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Docker labels</h2>
 * <p>
 * You can add labels to the container with the <code>dockerLabels</code> property or with the <code>addDockerLabel</code> method:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * 
 * ContainerDefinition container = taskDefinition.addContainer("cont", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .dockerLabels(Map.of(
 *                 "foo", "bar"))
 *         .build());
 * 
 * container.addDockerLabel("label", "value");
 * </pre></blockquote>
 * <p>
 * <h3>Using Windows containers on Fargate</h3>
 * <p>
 * AWS Fargate supports Amazon ECS Windows containers. For more details, please see this <a href="https://aws.amazon.com/tw/blogs/containers/running-windows-containers-with-amazon-ecs-on-aws-fargate/">blog post</a>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the Windows container to start
 * FargateTaskDefinition taskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .runtimePlatform(RuntimePlatform.builder()
 *                 .operatingSystemFamily(OperatingSystemFamily.WINDOWS_SERVER_2019_CORE)
 *                 .cpuArchitecture(CpuArchitecture.X86_64)
 *                 .build())
 *         .cpu(1024)
 *         .memoryLimitMiB(2048)
 *         .build();
 * 
 * taskDefinition.addContainer("windowsservercore", ContainerDefinitionOptions.builder()
 *         .logging(LogDriver.awsLogs(AwsLogDriverProps.builder().streamPrefix("win-iis-on-fargate").build()))
 *         .portMappings(List.of(PortMapping.builder().containerPort(80).build()))
 *         .image(ContainerImage.fromRegistry("mcr.microsoft.com/windows/servercore/iis:windowsservercore-ltsc2019"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Using Windows authentication with gMSA</h3>
 * <p>
 * Amazon ECS supports Active Directory authentication for Linux containers through a special kind of service account called a group Managed Service Account (gMSA). For more details, please see the <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">product documentation on how to implement on Windows containers</a>, or this <a href="https://aws.amazon.com/blogs/containers/using-windows-authentication-with-gmsa-on-linux-containers-on-amazon-ecs/">blog post on how to implement on  Linux containers</a>.
 * <p>
 * There are two types of CredentialSpecs, domained-join or domainless. Both types support creation from a S3 bucket, a SSM parameter, or by directly specifying a location for the file in the constructor.
 * <p>
 * A domian-joined gMSA container looks like:
 * <p>
 * <blockquote><pre>
 * // Make sure the task definition's execution role has permissions to read from the S3 bucket or SSM parameter where the CredSpec file is stored.
 * IParameter parameter;
 * TaskDefinition taskDefinition;
 * 
 * 
 * // Domain-joined gMSA container from a SSM parameter
 * taskDefinition.addContainer("gmsa-domain-joined-container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .cpu(128)
 *         .memoryLimitMiB(256)
 *         .credentialSpecs(List.of(DomainJoinedCredentialSpec.fromSsmParameter(parameter)))
 *         .build());
 * </pre></blockquote>
 * <p>
 * A domianless gMSA container looks like:
 * <p>
 * <blockquote><pre>
 * // Make sure the task definition's execution role has permissions to read from the S3 bucket or SSM parameter where the CredSpec file is stored.
 * Bucket bucket;
 * TaskDefinition taskDefinition;
 * 
 * 
 * // Domainless gMSA container from a S3 bucket object.
 * taskDefinition.addContainer("gmsa-domainless-container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .cpu(128)
 *         .memoryLimitMiB(256)
 *         .credentialSpecs(List.of(DomainlessCredentialSpec.fromS3Bucket(bucket, "credSpec")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Using Graviton2 with Fargate</h3>
 * <p>
 * AWS Graviton2 supports AWS Fargate. For more details, please see this <a href="https://aws.amazon.com/blogs/aws/announcing-aws-graviton2-support-for-aws-fargate-get-up-to-40-better-price-performance-for-your-serverless-containers/">blog post</a>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for running container on Graviton Runtime.
 * FargateTaskDefinition taskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .runtimePlatform(RuntimePlatform.builder()
 *                 .operatingSystemFamily(OperatingSystemFamily.LINUX)
 *                 .cpuArchitecture(CpuArchitecture.ARM64)
 *                 .build())
 *         .cpu(1024)
 *         .memoryLimitMiB(2048)
 *         .build();
 * 
 * taskDefinition.addContainer("webarm64", ContainerDefinitionOptions.builder()
 *         .logging(LogDriver.awsLogs(AwsLogDriverProps.builder().streamPrefix("graviton2-on-fargate").build()))
 *         .portMappings(List.of(PortMapping.builder().containerPort(80).build()))
 *         .image(ContainerImage.fromRegistry("public.ecr.aws/nginx/nginx:latest-arm64v8"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Service</h2>
 * <p>
 * A <code>Service</code> instantiates a <code>TaskDefinition</code> on a <code>Cluster</code> a given number of
 * times, optionally associating them with a load balancer.
 * If a task fails,
 * Amazon ECS automatically restarts the task.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .desiredCount(5)
 *         .minHealthyPercent(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * ECS Anywhere service definition looks like:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * ExternalService service = ExternalService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .desiredCount(5)
 *         .minHealthyPercent(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <code>Services</code> by default will create a security group if not provided.
 * If you'd like to specify which security groups to use you can override the <code>securityGroups</code> property.
 * <p>
 * By default, the service will use the revision of the passed task definition generated when the <code>TaskDefinition</code>
 * is deployed by CloudFormation. However, this may not be desired if the revision is externally managed,
 * for example through CodeDeploy.
 * <p>
 * To set a specific revision number or the special <code>latest</code> revision, use the <code>taskDefinitionRevision</code> parameter:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * ExternalService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .desiredCount(5)
 *         .minHealthyPercent(100)
 *         .taskDefinitionRevision(TaskDefinitionRevision.of(1))
 *         .build();
 * 
 * ExternalService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .desiredCount(5)
 *         .minHealthyPercent(100)
 *         .taskDefinitionRevision(TaskDefinitionRevision.LATEST)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deployment circuit breaker and rollback</h3>
 * <p>
 * Amazon ECS <a href="https://aws.amazon.com/tw/blogs/containers/announcing-amazon-ecs-deployment-circuit-breaker/">deployment circuit breaker</a>
 * automatically rolls back unhealthy service deployments, eliminating the need for manual intervention.
 * <p>
 * Use <code>circuitBreaker</code> to enable the deployment circuit breaker which determines whether a service deployment
 * will fail if the service can't reach a steady state.
 * You can optionally enable <code>rollback</code> for automatic rollback.
 * <p>
 * See <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Using the deployment circuit breaker</a> for more details.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .circuitBreaker(DeploymentCircuitBreaker.builder()
 *                 .enable(true)
 *                 .rollback(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: ECS Anywhere doesn't support deployment circuit breakers and rollback.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Deployment alarms</h3>
 * <p>
 * Amazon ECS [deployment alarms]
 * (https://aws.amazon.com/blogs/containers/automate-rollbacks-for-amazon-ecs-rolling-deployments-with-cloudwatch-alarms/)
 * allow monitoring and automatically reacting to changes during a rolling update
 * by using Amazon CloudWatch metric alarms.
 * <p>
 * Amazon ECS starts monitoring the configured deployment alarms as soon as one or
 * more tasks of the updated service are in a running state. The deployment process
 * continues until the primary deployment is healthy and has reached the desired
 * count and the active deployment has been scaled down to 0. Then, the deployment
 * remains in the IN_PROGRESS state for an additional "bake time." The length the
 * bake time is calculated based on the evaluation periods and period of the alarms.
 * After the bake time, if none of the alarms have been activated, then Amazon ECS
 * considers this to be a successful update and deletes the active deployment and
 * changes the status of the primary deployment to COMPLETED.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * Alarm elbAlarm;
 * 
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .deploymentAlarms(DeploymentAlarmConfig.builder()
 *                 .alarmNames(List.of(elbAlarm.getAlarmName()))
 *                 .behavior(AlarmBehavior.ROLLBACK_ON_ALARM)
 *                 .build())
 *         .build();
 * 
 * // Defining a deployment alarm after the service has been created
 * String cpuAlarmName = "MyCpuMetricAlarm";
 * Alarm.Builder.create(this, "CPUAlarm")
 *         .alarmName(cpuAlarmName)
 *         .metric(service.metricCpuUtilization())
 *         .evaluationPeriods(2)
 *         .threshold(80)
 *         .build();
 * service.enableDeploymentAlarms(List.of(cpuAlarmName), DeploymentAlarmOptions.builder()
 *         .behavior(AlarmBehavior.FAIL_ON_ALARM)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: Deployment alarms are only available when <code>deploymentController</code> is set
 * to <code>DeploymentControllerType.ECS</code>, which is the default.
 * <p>
 * </blockquote>
 * <p>
 * <h4>Troubleshooting circular dependencies</h4>
 * <p>
 * I saw this info message during synth time. What do I do?
 * <p>
 * <blockquote><pre>
 * Deployment alarm ({"Ref":"MyAlarmABC1234"}) enabled on MyEcsService may cause a
 * circular dependency error when this stack deploys. The alarm name references the
 * alarm's logical id, or another resource. See the 'Deployment alarms' section in
 * the module README for more details.
 * </pre></blockquote>
 * <p>
 * If your app deploys successfully with this message, you can disregard it. But it
 * indicates that you could encounter a circular dependency error when you try to
 * deploy. If you want to alarm on metrics produced by the service, there will be a
 * circular dependency between the service and its deployment alarms. In this case,
 * there are two options to avoid the circular dependency.
 * <p>
 * <ol>
 * <li>Define the physical name for the alarm. Use a defined physical name that is
 * unique within the deployment environment for the alarm name when creating the
 * alarm, and re-use the defined name. This name could be a hardcoded string, a
 * string generated based on the environment, or could reference another
 * resource that does not depend on the service.</li>
 * <li>Define the physical name for the service. Then, don't use
 * <code>metricCpuUtilization()</code> or similar methods. Create the metric object
 * separately by referencing the service metrics using this name.</li>
 * </ol>
 * <p>
 * Option 1, defining a physical name for the alarm:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .build();
 * 
 * String cpuAlarmName = "MyCpuMetricAlarm";
 * Alarm myAlarm = Alarm.Builder.create(this, "CPUAlarm")
 *         .alarmName(cpuAlarmName)
 *         .metric(service.metricCpuUtilization())
 *         .evaluationPeriods(2)
 *         .threshold(80)
 *         .build();
 * 
 * // Using `myAlarm.alarmName` here will cause a circular dependency
 * service.enableDeploymentAlarms(List.of(cpuAlarmName), DeploymentAlarmOptions.builder()
 *         .behavior(AlarmBehavior.FAIL_ON_ALARM)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Option 2, defining a physical name for the service:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * String serviceName = "MyFargateService";
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .serviceName(serviceName)
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .build();
 * 
 * Metric cpuMetric = Metric.Builder.create()
 *         .metricName("CPUUtilization")
 *         .namespace("AWS/ECS")
 *         .period(Duration.minutes(5))
 *         .statistic("Average")
 *         .dimensionsMap(Map.of(
 *                 "ClusterName", cluster.getClusterName(),
 *                 // Using `service.serviceName` here will cause a circular dependency
 *                 "ServiceName", serviceName))
 *         .build();
 * Alarm myAlarm = Alarm.Builder.create(this, "CPUAlarm")
 *         .alarmName("cpuAlarmName")
 *         .metric(cpuMetric)
 *         .evaluationPeriods(2)
 *         .threshold(80)
 *         .build();
 * 
 * service.enableDeploymentAlarms(List.of(myAlarm.getAlarmName()), DeploymentAlarmOptions.builder()
 *         .behavior(AlarmBehavior.FAIL_ON_ALARM)
 *         .build());
 * </pre></blockquote>
 * <p>
 * This issue only applies if the metrics to alarm on are emitted by the service
 * itself. If the metrics are emitted by a different resource, that does not depend
 * on the service, there will be no restrictions on the alarm name.
 * <p>
 * <h3>Include an application/network load balancer</h3>
 * <p>
 * <code>Services</code> are load balancing targets and can be added to a target group, which will be attached to an application/network load balancers:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * FargateService service = FargateService.Builder.create(this, "Service").cluster(cluster).taskDefinition(taskDefinition).minHealthyPercent(100).build();
 * 
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB").vpc(vpc).internetFacing(true).build();
 * ApplicationListener listener = lb.addListener("Listener", BaseApplicationListenerProps.builder().port(80).build());
 * ApplicationTargetGroup targetGroup1 = listener.addTargets("ECS1", AddApplicationTargetsProps.builder()
 *         .port(80)
 *         .targets(List.of(service))
 *         .build());
 * ApplicationTargetGroup targetGroup2 = listener.addTargets("ECS2", AddApplicationTargetsProps.builder()
 *         .port(80)
 *         .targets(List.of(service.loadBalancerTarget(LoadBalancerTargetOptions.builder()
 *                 .containerName("MyContainer")
 *                 .containerPort(8080)
 *                 .build())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: ECS Anywhere doesn't support application/network load balancers.
 * <p>
 * </blockquote>
 * <p>
 * Note that in the example above, the default <code>service</code> only allows you to register the first essential container or the first mapped port on the container as a target and add it to a new target group. To have more control over which container and port to register as targets, you can use <code>service.loadBalancerTarget()</code> to return a load balancing target for a specific container and port.
 * <p>
 * Alternatively, you can also create all load balancer targets to be registered in this service, add them to target groups, and attach target groups to listeners accordingly.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * Vpc vpc;
 * 
 * FargateService service = FargateService.Builder.create(this, "Service").cluster(cluster).taskDefinition(taskDefinition).minHealthyPercent(100).build();
 * 
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB").vpc(vpc).internetFacing(true).build();
 * ApplicationListener listener = lb.addListener("Listener", BaseApplicationListenerProps.builder().port(80).build());
 * service.registerLoadBalancerTargets(EcsTarget.builder()
 *         .containerName("web")
 *         .containerPort(80)
 *         .newTargetGroupId("ECS")
 *         .listener(ListenerConfig.applicationListener(listener, AddApplicationTargetsProps.builder()
 *                 .protocol(ApplicationProtocol.HTTPS)
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Using a Load Balancer from a different Stack</h3>
 * <p>
 * If you want to put your Load Balancer and the Service it is load balancing to in
 * different stacks, you may not be able to use the convenience methods
 * <code>loadBalancer.addListener()</code> and <code>listener.addTargets()</code>.
 * <p>
 * The reason is that these methods will create resources in the same Stack as the
 * object they're called on, which may lead to cyclic references between stacks.
 * Instead, you will have to create an <code>ApplicationListener</code> in the service stack,
 * or an empty <code>TargetGroup</code> in the load balancer stack that you attach your
 * service to.
 * <p>
 * See the <a href="https://github.com/aws-samples/aws-cdk-examples/tree/master/typescript/ecs/cross-stack-load-balancer/">ecs/cross-stack-load-balancer example</a>
 * for the alternatives.
 * <p>
 * <h3>Include a classic load balancer</h3>
 * <p>
 * <code>Services</code> can also be directly attached to a classic load balancer as targets:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * Vpc vpc;
 * 
 * Ec2Service service = Ec2Service.Builder.create(this, "Service").cluster(cluster).taskDefinition(taskDefinition).minHealthyPercent(100).build();
 * 
 * LoadBalancer lb = LoadBalancer.Builder.create(this, "LB").vpc(vpc).build();
 * lb.addListener(LoadBalancerListener.builder().externalPort(80).build());
 * lb.addTarget(service);
 * </pre></blockquote>
 * <p>
 * Similarly, if you want to have more control over load balancer targeting:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * Vpc vpc;
 * 
 * Ec2Service service = Ec2Service.Builder.create(this, "Service").cluster(cluster).taskDefinition(taskDefinition).minHealthyPercent(100).build();
 * 
 * LoadBalancer lb = LoadBalancer.Builder.create(this, "LB").vpc(vpc).build();
 * lb.addListener(LoadBalancerListener.builder().externalPort(80).build());
 * lb.addTarget(service.loadBalancerTarget(LoadBalancerTargetOptions.builder()
 *         .containerName("MyContainer")
 *         .containerPort(80)
 *         .build()));
 * </pre></blockquote>
 * <p>
 * There are two higher-level constructs available which include a load balancer for you that can be found in the aws-ecs-patterns module:
 * <p>
 * <ul>
 * <li><code>LoadBalancedFargateService</code></li>
 * <li><code>LoadBalancedEc2Service</code></li>
 * </ul>
 * <p>
 * <h3>Import existing services</h3>
 * <p>
 * <code>Ec2Service</code> and <code>FargateService</code> provide methods to import existing EC2/Fargate services.
 * The ARN of the existing service has to be specified to import the service.
 * <p>
 * Since AWS has changed the <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-account-settings.html#ecs-resource-ids">ARN format for ECS</a>,
 * feature flag <code>&#64;aws-cdk/aws-ecs:arnFormatIncludesClusterName</code> must be enabled to use the new ARN format.
 * The feature flag changes behavior for the entire CDK project. Therefore it is not possible to mix the old and the new format in one CDK project.
 * <p>
 * <blockquote><pre>
 * declare const cluster: ecs.Cluster;
 * 
 * // Import service from EC2 service attributes
 * const service = ecs.Ec2Service.fromEc2ServiceAttributes(this, 'EcsService', {
 *   serviceArn: 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service',
 *   cluster,
 * });
 * 
 * // Import service from EC2 service ARN
 * const service = ecs.Ec2Service.fromEc2ServiceArn(this, 'EcsService', 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service');
 * 
 * // Import service from Fargate service attributes
 * const service = ecs.FargateService.fromFargateServiceAttributes(this, 'EcsService', {
 *   serviceArn: 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service',
 *   cluster,
 * });
 * 
 * // Import service from Fargate service ARN
 * const service = ecs.FargateService.fromFargateServiceArn(this, 'EcsService', 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service');
 * </pre></blockquote>
 * <p>
 * <h3>Availability Zone rebalancing</h3>
 * <p>
 * ECS services running in AWS can be launched in multiple VPC subnets that are
 * each in different Availability Zones (AZs) to achieve high availability. Fargate
 * services launched this way will automatically try to achieve an even spread of
 * service tasks across AZs, and EC2 services can be instructed to do the same with
 * placement strategies. This ensures that the service has equal availability in
 * each AZ.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         // Fargate will try to ensure an even spread of newly launched tasks across
 *         // all AZs corresponding to the public subnets of the VPC.
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * However, those approaches only affect how newly launched tasks are placed.
 * Service tasks can still become unevenly spread across AZs if there is an
 * infrastructure event, like an AZ outage or a lack of available compute capacity
 * in an AZ. During such events, newly launched tasks may be placed in AZs in such
 * a way that tasks are not evenly spread across all AZs. After the infrastructure
 * event is over, the service will remain imbalanced until new tasks are launched
 * for some other reason, such as a service deployment.
 * <p>
 * Availability Zone rebalancing is a feature whereby ECS actively tries to correct
 * service AZ imbalances whenever they exist, by moving service tasks from
 * overbalanced AZs to underbalanced AZs. When an imbalance is detected, ECS will
 * launch new tasks in underbalanced AZs, then stop existing tasks in overbalanced
 * AZs, to ensure an even spread.
 * <p>
 * You can enabled Availability Zone rebalancing when creating your service:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .availabilityZoneRebalancing(AvailabilityZoneRebalancing.ENABLED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Availability Zone rebalancing works in the following configurations:
 * <p>
 * <ul>
 * <li>Services that use the Replica strategy.</li>
 * <li>Services that specify Availability Zone spread as the first task placement
 * strategy, or do not specify a placement strategy.</li>
 * </ul>
 * <p>
 * You can't use Availability Zone rebalancing with services that meet any of the
 * following criteria:
 * <p>
 * <ul>
 * <li>Uses the Daemon strategy.</li>
 * <li>Uses the EXTERNAL launch type (ECSAnywhere).</li>
 * <li>Uses 100% for the maximumPercent value.</li>
 * <li>Uses a Classic Load Balancer.</li>
 * <li>Uses the <code>attribute:ecs.availability-zone</code> as a task placement constraint.</li>
 * </ul>
 * <p>
 * <h2>Task Auto-Scaling</h2>
 * <p>
 * You can configure the task count of a service to match demand. Task auto-scaling is
 * configured by calling <code>autoScaleTaskCount()</code>:
 * <p>
 * <blockquote><pre>
 * ApplicationTargetGroup target;
 * BaseService service;
 * 
 * ScalableTaskCount scaling = service.autoScaleTaskCount(EnableScalingProps.builder().maxCapacity(10).build());
 * scaling.scaleOnCpuUtilization("CpuScaling", CpuUtilizationScalingProps.builder()
 *         .targetUtilizationPercent(50)
 *         .build());
 * 
 * scaling.scaleOnRequestCount("RequestScaling", RequestCountScalingProps.builder()
 *         .requestsPerTarget(10000)
 *         .targetGroup(target)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Task auto-scaling is powered by <em>Application Auto-Scaling</em>.
 * See that section for details.
 * <p>
 * <h2>Integration with CloudWatch Events</h2>
 * <p>
 * To start an Amazon ECS task on an Amazon EC2-backed Cluster, instantiate an
 * <code>aws-cdk-lib/aws-events-targets.EcsTask</code> instead of an <code>Ec2Service</code>:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromAsset(resolve(__dirname, "..", "eventhandler-image")))
 *         .memoryLimitMiB(256)
 *         .logging(AwsLogDriver.Builder.create().streamPrefix("EventDemo").mode(AwsLogDriverMode.NON_BLOCKING).build())
 *         .build());
 * 
 * // An Rule that describes the event trigger (in this case a scheduled run)
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .build();
 * 
 * // Pass an environment variable to the container 'TheContainer' in the task
 * rule.addTarget(EcsTask.Builder.create()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .taskCount(1)
 *         .containerOverrides(List.of(ContainerOverride.builder()
 *                 .containerName("TheContainer")
 *                 .environment(List.of(TaskEnvironmentVariable.builder()
 *                         .name("I_WAS_TRIGGERED")
 *                         .value("From CloudWatch Events")
 *                         .build()))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Log Drivers</h2>
 * <p>
 * Currently Supported Log Drivers:
 * <p>
 * <ul>
 * <li>awslogs</li>
 * <li>fluentd</li>
 * <li>gelf</li>
 * <li>journald</li>
 * <li>json-file</li>
 * <li>splunk</li>
 * <li>syslog</li>
 * <li>awsfirelens</li>
 * <li>Generic</li>
 * </ul>
 * <p>
 * <h3>awslogs Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.awsLogs(AwsLogDriverProps.builder()
 *                 .streamPrefix("EventDemo")
 *                 .mode(AwsLogDriverMode.NON_BLOCKING)
 *                 .maxBufferSize(Size.mebibytes(25))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>fluentd Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.fluentd())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>gelf Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.gelf(GelfLogDriverProps.builder().address("my-gelf-address").build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>journald Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.journald())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>json-file Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.jsonFile())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>splunk Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * Secret secret;
 * 
 * 
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.splunk(SplunkLogDriverProps.builder()
 *                 .secretToken(secret)
 *                 .url("my-splunk-url")
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>syslog Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.syslog())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>firelens Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.firelens(FireLensLogDriverProps.builder()
 *                 .options(Map.of(
 *                         "Name", "firehose",
 *                         "region", "us-west-2",
 *                         "delivery_stream", "my-stream"))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To pass secrets to the log configuration, use the <code>secretOptions</code> property of the log configuration. The task execution role is automatically granted read permissions on the secrets/parameters.
 * <p>
 * <blockquote><pre>
 * Secret secret;
 * StringParameter parameter;
 * 
 * 
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.firelens(FireLensLogDriverProps.builder()
 *                 .options(Map.of())
 *                 .secretOptions(Map.of( // Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store
 *                         "apikey", Secret.fromSecretsManager(secret),
 *                         "host", Secret.fromSsmParameter(parameter)))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * When forwarding logs to CloudWatch Logs using Fluent Bit, you can set the retention period for the newly created Log Group by specifying the <code>log_retention_days</code> parameter.
 * If a Fluent Bit container has not been added, CDK will automatically add it to the task definition, and the necessary IAM permissions will be added to the task role.
 * If you are adding the Fluent Bit container manually, ensure to add the <code>logs:PutRetentionPolicy</code> policy to the task role.
 * <p>
 * <blockquote><pre>
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.firelens(FireLensLogDriverProps.builder()
 *                 .options(Map.of(
 *                         "Name", "cloudwatch",
 *                         "region", "us-west-2",
 *                         "log_group_name", "firelens-fluent-bit",
 *                         "log_stream_prefix", "from-fluent-bit",
 *                         "auto_create_group", "true",
 *                         "log_retention_days", "1"))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.fluentbit.io/manual/pipeline/outputs/cloudwatch#configuration-parameters">Fluent Bit CloudWatch Configuration Parameters</a>
 * for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Generic Log Driver</h3>
 * <p>
 * A generic log driver object exists to provide a lower level abstraction of the log driver configuration.
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(GenericLogDriver.Builder.create()
 *                 .logDriver("fluentd")
 *                 .options(Map.of(
 *                         "tag", "example-tag"))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>CloudMap Service Discovery</h2>
 * <p>
 * To register your ECS service with a CloudMap Service Registry, you may add the
 * <code>cloudMapOptions</code> property to your service:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * Cluster cluster;
 * 
 * 
 * Ec2Service service = Ec2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .cloudMapOptions(CloudMapOptions.builder()
 *                 // Create A records - useful for AWSVPC network mode.
 *                 .dnsRecordType(DnsRecordType.A)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * With <code>bridge</code> or <code>host</code> network modes, only <code>SRV</code> DNS record types are supported.
 * By default, <code>SRV</code> DNS record types will target the default container and default
 * port. However, you may target a different container and port on the same ECS task:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * Cluster cluster;
 * 
 * 
 * // Add a container to the task definition
 * ContainerDefinition specificContainer = taskDefinition.addContainer("Container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("/aws/aws-example-app"))
 *         .memoryLimitMiB(2048)
 *         .build());
 * 
 * // Add a port mapping
 * specificContainer.addPortMappings(PortMapping.builder()
 *         .containerPort(7600)
 *         .protocol(Protocol.TCP)
 *         .build());
 * 
 * Ec2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .cloudMapOptions(CloudMapOptions.builder()
 *                 // Create SRV records - useful for bridge networking
 *                 .dnsRecordType(DnsRecordType.SRV)
 *                 // Targets port TCP port 7600 `specificContainer`
 *                 .container(specificContainer)
 *                 .containerPort(7600)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Associate With a Specific CloudMap Service</h3>
 * <p>
 * You may associate an ECS service with a specific CloudMap service. To do
 * this, use the service's <code>associateCloudMapService</code> method:
 * <p>
 * <blockquote><pre>
 * Service cloudMapService;
 * FargateService ecsService;
 * 
 * 
 * ecsService.associateCloudMapService(AssociateCloudMapServiceOptions.builder()
 *         .service(cloudMapService)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Capacity Providers</h2>
 * <p>
 * There are two major families of Capacity Providers: <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-capacity-providers.html">AWS
 * Fargate</a>
 * (including Fargate Spot) and EC2 <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html">Auto Scaling
 * Group</a>
 * Capacity Providers. Both are supported.
 * <p>
 * <h3>Fargate Capacity Providers</h3>
 * <p>
 * To enable Fargate capacity providers, you can either set
 * <code>enableFargateCapacityProviders</code> to <code>true</code> when creating your cluster, or by
 * invoking the <code>enableFargateCapacityProviders()</code> method after creating your
 * cluster. This will add both <code>FARGATE</code> and <code>FARGATE_SPOT</code> as available capacity
 * providers on your cluster.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "FargateCPCluster")
 *         .vpc(vpc)
 *         .enableFargateCapacityProviders(true)
 *         .build();
 * 
 * FargateTaskDefinition taskDefinition = new FargateTaskDefinition(this, "TaskDef");
 * 
 * taskDefinition.addContainer("web", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .build());
 * 
 * FargateService.Builder.create(this, "FargateService")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE_SPOT")
 *                 .weight(2)
 *                 .build(), CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE")
 *                 .weight(1)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Auto Scaling Group Capacity Providers</h3>
 * <p>
 * To add an Auto Scaling Group Capacity Provider, first create an EC2 Auto Scaling
 * Group. Then, create an <code>AsgCapacityProvider</code> and pass the Auto Scaling Group to
 * it in the constructor. Then add the Capacity Provider to the cluster. Finally,
 * you can refer to the Provider by its name in your service's or task's Capacity
 * Provider strategy.
 * <p>
 * By default, Auto Scaling Group Capacity Providers will manage the scale-in and
 * scale-out behavior of the auto scaling group based on the load your tasks put on
 * the cluster, this is called <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html#asg-capacity-providers-managed-scaling">Managed Scaling</a>. If you'd
 * rather manage scaling behavior yourself set <code>enableManagedScaling</code> to <code>false</code>.
 * <p>
 * Additionally <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-auto-scaling.html#managed-termination-protection">Managed Termination Protection</a> is enabled by default to
 * prevent scale-in behavior from terminating instances that have non-daemon tasks
 * running on them. This is ideal for tasks that can be run to completion. If your
 * tasks are safe to interrupt then this protection can be disabled by setting
 * <code>enableManagedTerminationProtection</code> to <code>false</code>. Managed Scaling must be enabled for
 * Managed Termination Protection to work.
 * <p>
 * <blockquote>
 * <p>
 * Currently there is a known <a href="https://github.com/aws/containers-roadmap/issues/631">CloudFormation issue</a>
 * that prevents CloudFormation from automatically deleting Auto Scaling Groups that
 * have Managed Termination Protection enabled. To work around this issue you could set
 * <code>enableManagedTerminationProtection</code> to <code>false</code> on the Auto Scaling Group Capacity
 * Provider. If you'd rather not disable Managed Termination Protection, you can <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-process-shutdown.html">manually
 * delete the Auto Scaling Group</a>.
 * For other workarounds, see <a href="https://github.com/aws/aws-cdk/issues/18179">this GitHub issue</a>.
 * <p>
 * </blockquote>
 * <p>
 * Managed instance draining facilitates graceful termination of Amazon ECS instances.
 * This allows your service workloads to stop safely and be rescheduled to non-terminating instances.
 * Infrastructure maintenance and updates are preformed without disruptions to workloads.
 * To use managed instance draining, set enableManagedDraining to true.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * 
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(new InstanceType("t2.micro"))
 *         .machineImage(EcsOptimizedImage.amazonLinux2())
 *         .minCapacity(0)
 *         .maxCapacity(100)
 *         .build();
 * 
 * AsgCapacityProvider capacityProvider = AsgCapacityProvider.Builder.create(this, "AsgCapacityProvider")
 *         .autoScalingGroup(autoScalingGroup)
 *         .instanceWarmupPeriod(300)
 *         .build();
 * cluster.addAsgCapacityProvider(capacityProvider);
 * 
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * 
 * taskDefinition.addContainer("web", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryReservationMiB(256)
 *         .build());
 * 
 * Ec2Service.Builder.create(this, "EC2Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider(capacityProvider.getCapacityProviderName())
 *                 .weight(1)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cluster Default Provider Strategy</h3>
 * <p>
 * A capacity provider strategy determines whether ECS tasks are launched on EC2 instances or Fargate/Fargate Spot. It can be specified at the cluster, service, or task level, and consists of one or more capacity providers. You can specify an optional base and weight value for finer control of how tasks are launched. The <code>base</code> specifies a minimum number of tasks on one capacity provider, and the <code>weight</code>s of each capacity provider determine how tasks are distributed after <code>base</code> is satisfied.
 * <p>
 * You can associate a default capacity provider strategy with an Amazon ECS cluster. After you do this, a default capacity provider strategy is used when creating a service or running a standalone task in the cluster and whenever a custom capacity provider strategy or a launch type isn't specified. We recommend that you define a default capacity provider strategy for each cluster.
 * <p>
 * For more information visit https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-capacity-providers.html
 * <p>
 * When the service does not have a capacity provider strategy, the cluster's default capacity provider strategy will be used. Default Capacity Provider Strategy can be added by using the method <code>addDefaultCapacityProviderStrategy</code>. A capacity provider strategy cannot contain a mix of EC2 Autoscaling Group capacity providers and Fargate providers.
 * <p>
 * <blockquote><pre>
 * AsgCapacityProvider capacityProvider;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster")
 *         .enableFargateCapacityProviders(true)
 *         .build();
 * cluster.addAsgCapacityProvider(capacityProvider);
 * 
 * cluster.addDefaultCapacityProviderStrategy(List.of(CapacityProviderStrategy.builder().capacityProvider("FARGATE").base(10).weight(50).build(), CapacityProviderStrategy.builder().capacityProvider("FARGATE_SPOT").weight(50).build()));
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * AsgCapacityProvider capacityProvider;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster")
 *         .enableFargateCapacityProviders(true)
 *         .build();
 * cluster.addAsgCapacityProvider(capacityProvider);
 * 
 * cluster.addDefaultCapacityProviderStrategy(List.of(CapacityProviderStrategy.builder().capacityProvider(capacityProvider.getCapacityProviderName()).build()));
 * </pre></blockquote>
 * <p>
 * <h2>Elastic Inference Accelerators</h2>
 * <p>
 * Currently, this feature is only supported for services with EC2 launch types.
 * <p>
 * To add elastic inference accelerators to your EC2 instance, first add
 * <code>inferenceAccelerators</code> field to the Ec2TaskDefinition and set the <code>deviceName</code>
 * and <code>deviceType</code> properties.
 * <p>
 * <blockquote><pre>
 * Map&lt;String, String&gt;[] inferenceAccelerators = List.of(Map.of(
 *         "deviceName", "device1",
 *         "deviceType", "eia2.medium"));
 * 
 * Ec2TaskDefinition taskDefinition = Ec2TaskDefinition.Builder.create(this, "Ec2TaskDef")
 *         .inferenceAccelerators(inferenceAccelerators)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To enable using the inference accelerators in the containers, add <code>inferenceAcceleratorResources</code>
 * field and set it to a list of device names used for the inference accelerators. Each value in the
 * list should match a <code>DeviceName</code> for an <code>InferenceAccelerator</code> specified in the task definition.
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * String[] inferenceAcceleratorResources = List.of("device1");
 * 
 * taskDefinition.addContainer("cont", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("test"))
 *         .memoryLimitMiB(1024)
 *         .inferenceAcceleratorResources(inferenceAcceleratorResources)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>ECS Exec command</h2>
 * <p>
 * Please note, ECS Exec leverages AWS Systems Manager (SSM). So as a prerequisite for the exec command
 * to work, you need to have the SSM plugin for the AWS CLI installed locally. For more information, see
 * <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-working-with-install-plugin.html">Install Session Manager plugin for AWS CLI</a>.
 * <p>
 * To enable the ECS Exec feature for your containers, set the boolean flag <code>enableExecuteCommand</code> to <code>true</code> in
 * your <code>Ec2Service</code>, <code>FargateService</code> or <code>ExternalService</code>.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * Ec2Service service = Ec2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .enableExecuteCommand(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Enabling logging</h3>
 * <p>
 * You can enable sending logs of your execute session commands to a CloudWatch log group or S3 bucket by configuring
 * the <code>executeCommandConfiguration</code> property for your cluster. The default configuration will send the
 * logs to the CloudWatch Logs using the <code>awslogs</code> log driver that is configured in your task definition. Please note,
 * when using your own <code>logConfiguration</code> the log group or S3 Bucket specified must already be created.
 * <p>
 * To encrypt data using your own KMS Customer Key (CMK), you must create a CMK and provide the key in the <code>kmsKey</code> field
 * of the <code>executeCommandConfiguration</code>. To use this key for encrypting CloudWatch log data or S3 bucket, make sure to associate the key
 * to these resources on creation.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * Key kmsKey = new Key(this, "KmsKey");
 * 
 * // Pass the KMS key in the `encryptionKey` field to associate the key to the log group
 * LogGroup logGroup = LogGroup.Builder.create(this, "LogGroup")
 *         .encryptionKey(kmsKey)
 *         .build();
 * 
 * // Pass the KMS key in the `encryptionKey` field to associate the key to the S3 bucket
 * Bucket execBucket = Bucket.Builder.create(this, "EcsExecBucket")
 *         .encryptionKey(kmsKey)
 *         .build();
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .executeCommandConfiguration(ExecuteCommandConfiguration.builder()
 *                 .kmsKey(kmsKey)
 *                 .logConfiguration(ExecuteCommandLogConfiguration.builder()
 *                         .cloudWatchLogGroup(logGroup)
 *                         .cloudWatchEncryptionEnabled(true)
 *                         .s3Bucket(execBucket)
 *                         .s3EncryptionEnabled(true)
 *                         .s3KeyPrefix("exec-command-output")
 *                         .build())
 *                 .logging(ExecuteCommandLogging.OVERRIDE)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Amazon ECS Service Connect</h2>
 * <p>
 * Service Connect is a managed AWS mesh network offering. It simplifies DNS queries and inter-service communication for
 * ECS Services by allowing customers to set up simple DNS aliases for their services, which are accessible to all
 * services that have enabled Service Connect.
 * <p>
 * To enable Service Connect, you must have created a CloudMap namespace. The CDK can infer your cluster's default CloudMap namespace,
 * or you can specify a custom namespace. You must also have created a named port mapping on at least one container in your Task Definition.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * ContainerDefinitionOptions containerOptions;
 * 
 * 
 * ContainerDefinition container = taskDefinition.addContainer("MyContainer", containerOptions);
 * 
 * container.addPortMappings(PortMapping.builder()
 *         .name("api")
 *         .containerPort(8080)
 *         .build());
 * 
 * cluster.addDefaultCloudMapNamespace(CloudMapNamespaceOptions.builder()
 *         .name("local")
 *         .build());
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .serviceConnectConfiguration(ServiceConnectProps.builder()
 *                 .services(List.of(ServiceConnectService.builder()
 *                         .portMappingName("api")
 *                         .dnsName("http-api")
 *                         .port(80)
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Service Connect-enabled services may now reach this service at <code>http-api:80</code>. Traffic to this endpoint will
 * be routed to the container's port 8080.
 * <p>
 * To opt a service into using service connect without advertising a port, simply call the 'enableServiceConnect' method on an initialized service.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .build();
 * service.enableServiceConnect();
 * </pre></blockquote>
 * <p>
 * Service Connect also allows custom logging, Service Discovery name, and configuration of the port where service connect traffic is received.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * FargateService customService = FargateService.Builder.create(this, "CustomizedService")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .serviceConnectConfiguration(ServiceConnectProps.builder()
 *                 .logDriver(LogDrivers.awsLogs(AwsLogDriverProps.builder()
 *                         .streamPrefix("sc-traffic")
 *                         .build()))
 *                 .services(List.of(ServiceConnectService.builder()
 *                         .portMappingName("api")
 *                         .dnsName("customized-api")
 *                         .port(80)
 *                         .ingressPortOverride(20040)
 *                         .discoveryName("custom")
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * To set a timeout for service connect, use <code>idleTimeout</code> and <code>perRequestTimeout</code>.
 * <p>
 * <strong>Note</strong>: If <code>idleTimeout</code> is set to a time that is less than <code>perRequestTimeout</code>, the connection will close when
 * the <code>idleTimeout</code> is reached and not the <code>perRequestTimeout</code>.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .serviceConnectConfiguration(ServiceConnectProps.builder()
 *                 .services(List.of(ServiceConnectService.builder()
 *                         .portMappingName("api")
 *                         .idleTimeout(Duration.minutes(5))
 *                         .perRequestTimeout(Duration.minutes(5))
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://aws.amazon.com/about-aws/whats-new/2024/01/amazon-ecs-configurable-timeout-service-connect/">Amazon ECS support for configurable timeout for services running with Service Connect</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h2>ServiceManagedVolume</h2>
 * <p>
 * Amazon ECS now supports the attachment of Amazon Elastic Block Store (EBS) volumes to ECS tasks,
 * allowing you to utilize persistent, high-performance block storage with your ECS services.
 * This feature supports various use cases, such as using EBS volumes as extended ephemeral storage or
 * loading data from EBS snapshots.
 * You can also specify <code>encrypted: true</code> so that ECS will manage the KMS key. If you want to use your own KMS key, you may do so by providing both <code>encrypted: true</code> and <code>kmsKeyId</code>.
 * <p>
 * You can only attach a single volume for each task in the ECS Service.
 * <p>
 * To add an empty EBS Volume to an ECS Service, call service.addVolume().
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * FargateTaskDefinition taskDefinition = new FargateTaskDefinition(this, "TaskDef");
 * 
 * ContainerDefinition container = taskDefinition.addContainer("web", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .portMappings(List.of(PortMapping.builder()
 *                 .containerPort(80)
 *                 .protocol(Protocol.TCP)
 *                 .build()))
 *         .build());
 * 
 * ServiceManagedVolume volume = ServiceManagedVolume.Builder.create(this, "EBSVolume")
 *         .name("ebs1")
 *         .managedEBSVolume(ServiceManagedEBSVolumeConfiguration.builder()
 *                 .size(Size.gibibytes(15))
 *                 .volumeType(EbsDeviceVolumeType.GP3)
 *                 .fileSystemType(FileSystemType.XFS)
 *                 .tagSpecifications(List.of(EBSTagSpecification.builder()
 *                         .tags(Map.of(
 *                                 "purpose", "production"))
 *                         .propagateTags(EbsPropagatedTagSource.SERVICE)
 *                         .build()))
 *                 .build())
 *         .build();
 * 
 * volume.mountIn(container, ContainerMountPoint.builder()
 *         .containerPath("/var/lib")
 *         .readOnly(false)
 *         .build());
 * 
 * taskDefinition.addVolume(volume);
 * 
 * FargateService service = FargateService.Builder.create(this, "FargateService")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .build();
 * 
 * service.addVolume(volume);
 * </pre></blockquote>
 * <p>
 * To create an EBS volume from an existing snapshot by specifying the <code>snapShotId</code> while adding a volume to the service.
 * <p>
 * <blockquote><pre>
 * ContainerDefinition container;
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * ServiceManagedVolume volumeFromSnapshot = ServiceManagedVolume.Builder.create(this, "EBSVolume")
 *         .name("nginx-vol")
 *         .managedEBSVolume(ServiceManagedEBSVolumeConfiguration.builder()
 *                 .snapShotId("snap-066877671789bd71b")
 *                 .volumeType(EbsDeviceVolumeType.GP3)
 *                 .fileSystemType(FileSystemType.XFS)
 *                 .build())
 *         .build();
 * 
 * volumeFromSnapshot.mountIn(container, ContainerMountPoint.builder()
 *         .containerPath("/var/lib")
 *         .readOnly(false)
 *         .build());
 * taskDefinition.addVolume(volumeFromSnapshot);
 * FargateService service = FargateService.Builder.create(this, "FargateService")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .minHealthyPercent(100)
 *         .build();
 * 
 * service.addVolume(volumeFromSnapshot);
 * </pre></blockquote>
 * <p>
 * <h2>Enable pseudo-terminal (TTY) allocation</h2>
 * <p>
 * You can allocate a pseudo-terminal (TTY) for a container passing <code>pseudoTerminal</code> option while adding the container
 * to the task definition.
 * This maps to Tty option in the <a href="https://docs.docker.com/engine/api/v1.38/#operation/ContainerCreate">"Create a container section"</a>
 * of the <a href="https://docs.docker.com/engine/api/v1.38/">Docker Remote API</a> and the --tty option to <a href="https://docs.docker.com/engine/reference/commandline/run/"><code>docker run</code></a>.
 * <p>
 * <blockquote><pre>
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .pseudoTerminal(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Disable service container image version consistency</h2>
 * <p>
 * You can disable the
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html#deployment-container-image-stability">container image "version consistency"</a>
 * feature of ECS service deployments on a per-container basis.
 * <p>
 * <blockquote><pre>
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .versionConsistency(VersionConsistency.DISABLED)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Specify a container ulimit</h2>
 * <p>
 * You can specify a container <code>ulimits</code> by specifying them in the <code>ulimits</code> option while adding the container
 * to the task definition.
 * <p>
 * <blockquote><pre>
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .ulimits(List.of(Ulimit.builder()
 *                 .hardLimit(128)
 *                 .name(UlimitName.RSS)
 *                 .softLimit(128)
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Service Connect TLS</h2>
 * <p>
 * Service Connect TLS is a feature that allows you to secure the communication between services using TLS.
 * <p>
 * You can specify the <code>tls</code> option in the <code>services</code> array of the <code>serviceConnectConfiguration</code> property.
 * <p>
 * The <code>tls</code> property is an object with the following properties:
 * <p>
 * <ul>
 * <li><code>role</code>: The IAM role that's associated with the Service Connect TLS.</li>
 * <li><code>awsPcaAuthorityArn</code>: The ARN of the certificate root authority that secures your service.</li>
 * <li><code>kmsKey</code>: The KMS key used for encryption and decryption.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * IKey kmsKey;
 * IRole role;
 * 
 * 
 * FargateService service = FargateService.Builder.create(this, "FargateService")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .serviceConnectConfiguration(ServiceConnectProps.builder()
 *                 .services(List.of(ServiceConnectService.builder()
 *                         .tls(ServiceConnectTlsConfiguration.builder()
 *                                 .role(role)
 *                                 .kmsKey(kmsKey)
 *                                 .awsPcaAuthorityArn("arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/123456789012")
 *                                 .build())
 *                         .portMappingName("api")
 *                         .build()))
 *                 .namespace("sample namespace")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Daemon scheduling strategy</h2>
 * <p>
 * You can specify whether service use Daemon scheduling strategy by specifying <code>daemon</code> option in Service constructs. See <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">differences between Daemon and Replica scheduling strategy</a>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * Ec2Service.Builder.create(this, "Ec2Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .daemon(true)
 *         .build();
 * 
 * ExternalService.Builder.create(this, "ExternalService")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .daemon(true)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ecs;
