/**
 * <h1>CDK Pipelines</h1>
 * <p>
 * A construct library for painless Continuous Delivery of CDK applications.
 * <p>
 * CDK Pipelines is an <em>opinionated construct library</em>. It is purpose-built to
 * deploy one or more copies of your CDK applications using CloudFormation with a
 * minimal amount of effort on your part. It is <em>not</em> intended to support arbitrary
 * deployment pipelines, and very specifically it is not built to use CodeDeploy to
 * deploy applications to instances, or deploy your custom-built ECR images to an ECS
 * cluster directly: use CDK file assets with CloudFormation Init for instances, or
 * CDK container assets for ECS clusters instead.
 * <p>
 * Give the CDK Pipelines way of doing things a shot first: you might find it does
 * everything you need. If you need more control, or if you need <code>v2</code> support from
 * <code>aws-codepipeline</code>, we recommend you drop down to using the <code>aws-codepipeline</code>
 * construct library directly.
 * <p>
 * <blockquote>
 * <p>
 * This module contains two sets of APIs: an <strong>original</strong> and a <strong>modern</strong> version of
 * CDK Pipelines. The <em>modern</em> API has been updated to be easier to work with and
 * customize, and will be the preferred API going forward. The <em>original</em> version
 * of the API is still available for backwards compatibility, but we recommend migrating
 * to the new version if possible.
 * <p>
 * Compared to the original API, the modern API: has more sensible defaults; is
 * more flexible; supports parallel deployments; supports multiple synth inputs;
 * allows more control of CodeBuild project generation; supports deployment
 * engines other than CodePipeline.
 * <p>
 * The README for the original API, as well as a migration guide, can be found in
 * <a href="https://github.com/aws/aws-cdk/blob/main/packages/aws-cdk-lib/pipelines/ORIGINAL_API.md">our GitHub repository</a>.
 * <p>
 * </blockquote>
 * <p>
 * <h2>At a glance</h2>
 * <p>
 * Deploying your application continuously starts by defining a
 * <code>MyApplicationStage</code>, a subclass of <code>Stage</code> that contains the stacks that make
 * up a single copy of your application.
 * <p>
 * You then define a <code>Pipeline</code>, instantiate as many instances of
 * <code>MyApplicationStage</code> as you want for your test and production environments, with
 * different parameters for each, and calling <code>pipeline.addStage()</code> for each of
 * them. You can deploy to the same account and Region, or to a different one,
 * with the same amount of code. The <em>CDK Pipelines</em> library takes care of the
 * details.
 * <p>
 * CDK Pipelines supports multiple <em>deployment engines</em> (see
 * <a href="#using-a-different-deployment-engine">Using a different deployment engine</a>),
 * and comes with a deployment engine that deploys CDK apps using AWS CodePipeline.
 * To use the CodePipeline engine, define a <code>CodePipeline</code> construct. The following
 * example creates a CodePipeline that deploys an application from GitHub:
 * <p>
 * <blockquote><pre>
 * /** The stacks for our app are minimally defined here.  The internals of these
 *  * stacks aren't important, except that DatabaseStack exposes an attribute
 *  * "table" for a database table it defines, and ComputeStack accepts a reference
 *  * to this table in its properties.
 *  *&#47;
 * public class DatabaseStack extends Stack {
 *     public final TableV2 table;
 * 
 *     public DatabaseStack(Construct scope, String id) {
 *         super(scope, id);
 *         this.table = TableV2.Builder.create(this, "Table")
 *                 .partitionKey(Attribute.builder().name("id").type(AttributeType.STRING).build())
 *                 .build();
 *     }
 * }
 * 
 * public class ComputeProps {
 *     private TableV2 table;
 *     public TableV2 getTable() {
 *         return this.table;
 *     }
 *     public ComputeProps table(TableV2 table) {
 *         this.table = table;
 *         return this;
 *     }
 * }
 * 
 * public class ComputeStack extends Stack {
 *     public ComputeStack(Construct scope, String id, ComputeProps props) {
 *         super(scope, id);
 *     }
 * }
 * 
 * /**
 *  * Stack to hold the pipeline
 *  *&#47;
 * public class MyPipelineStack extends Stack {
 *     public MyPipelineStack(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public MyPipelineStack(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *                 .synth(ShellStep.Builder.create("Synth")
 *                         // Use a connection created using the AWS console to authenticate to GitHub
 *                         // Other sources are available.
 *                         .input(CodePipelineSource.connection("my-org/my-app", "main", ConnectionSourceOptions.builder()
 *                                 .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *                                 .build()))
 *                         .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                         .build())
 *                 .build();
 * 
 *         // 'MyApplication' is defined below. Call `addStage` as many times as
 *         // necessary with any account and region (may be different from the
 *         // pipeline's).
 *         pipeline.addStage(
 *         MyApplication.Builder.create(this, "Prod")
 *                 .env(Environment.builder()
 *                         .account("123456789012")
 *                         .region("eu-west-1")
 *                         .build())
 *                 .build());
 *     }
 * }
 * 
 * /**
 *  * Your application
 *  *
 *  * May consist of one or more Stacks (here, two)
 *  *
 *  * By declaring our DatabaseStack and our ComputeStack inside a Stage,
 *  * we make sure they are deployed together, or not at all.
 *  *&#47;
 * public class MyApplication extends Stage {
 *     public MyApplication(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public MyApplication(Construct scope, String id, StageProps props) {
 *         super(scope, id, props);
 * 
 *         DatabaseStack dbStack = new DatabaseStack(this, "Database");
 *         new ComputeStack(this, "Compute", new ComputeProps()
 *                 .table(dbStack.getTable())
 *                 );
 *     }
 * }
 * 
 * // In your main file
 * // In your main file
 * MyPipelineStack.Builder.create(this, "PipelineStack")
 *         .env(Environment.builder()
 *                 .account("123456789012")
 *                 .region("eu-west-1")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The pipeline is <strong>self-mutating</strong>, which means that if you add new
 * application stages in the source code, or new stacks to <code>MyApplication</code>, the
 * pipeline will automatically reconfigure itself to deploy those new stages and
 * stacks.
 * <p>
 * (Note that you have to <em>bootstrap</em> all environments before the above code
 * will work, and switch on "Modern synthesis" if you are using
 * CDKv1. See the section <strong>CDK Environment Bootstrapping</strong> below for
 * more information).
 * <p>
 * <h2>Provisioning the pipeline</h2>
 * <p>
 * To provision the pipeline you have defined, make sure the target environment
 * has been bootstrapped (see below), and then execute deploying the
 * <code>PipelineStack</code> <em>once</em>. Afterwards, the pipeline will keep itself up-to-date.
 * <p>
 * <blockquote>
 * <p>
 * <strong>Important</strong>: be sure to <code>git commit</code> and <code>git push</code> before deploying the
 * Pipeline stack using <code>cdk deploy</code>!
 * <p>
 * The reason is that the pipeline will start deploying and self-mutating
 * right away based on the sources in the repository, so the sources it finds
 * in there should be the ones you want it to find.
 * <p>
 * </blockquote>
 * <p>
 * Run the following commands to get the pipeline going:
 * <p>
 * <blockquote><pre>
 * $ git commit -a
 * $ git push
 * $ cdk deploy PipelineStack
 * </pre></blockquote>
 * <p>
 * Administrative permissions to the account are only necessary up until
 * this point. We recommend you remove access to these credentials after doing this.
 * <p>
 * <h3>Working on the pipeline</h3>
 * <p>
 * The self-mutation feature of the Pipeline might at times get in the way
 * of the pipeline development workflow. Each change to the pipeline must be pushed
 * to git, otherwise, after the pipeline was updated using <code>cdk deploy</code>, it will
 * automatically revert to the state found in git.
 * <p>
 * To make the development more convenient, the self-mutation feature can be turned
 * off temporarily, by passing <code>selfMutation: false</code> property, example:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .selfMutation(false)
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(CodePipelineSource.connection("my-org/my-app", "main", ConnectionSourceOptions.builder()
 *                         .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *                         .build()))
 *                 .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Defining the pipeline</h2>
 * <p>
 * This section of the documentation describes the AWS CodePipeline engine,
 * which comes with this library. If you want to use a different deployment
 * engine, read the section
 * <a href="#using-a-different-deployment-engine">Using a different deployment engine</a> below.
 * <p>
 * <h3>Synth and sources</h3>
 * <p>
 * To define a pipeline, instantiate a <code>CodePipeline</code> construct from the
 * <code>aws-cdk-lib/pipelines</code> module. It takes one argument, a <code>synth</code> step, which is
 * expected to produce the CDK Cloud Assembly as its single output (the contents of
 * the <code>cdk.out</code> directory after running <code>cdk synth</code>). "Steps" are arbitrary
 * actions in the pipeline, typically used to run scripts or commands.
 * <p>
 * For the synth, use a <code>ShellStep</code> and specify the commands necessary to install
 * dependencies, the CDK CLI, build your project and run <code>cdk synth</code>; the specific
 * commands required will depend on the programming language you are using. For a
 * typical NPM-based project, the synth will look like this:
 * <p>
 * <blockquote><pre>
 * IFileSetProducer source;
 * // the repository source
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(source)
 *                 .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The pipeline assumes that your <code>ShellStep</code> will produce a <code>cdk.out</code>
 * directory in the root, containing the CDK cloud assembly. If your
 * CDK project lives in a subdirectory, be sure to adjust the
 * <code>primaryOutputDirectory</code> to match:
 * <p>
 * <blockquote><pre>
 * IFileSetProducer source;
 * // the repository source
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(source)
 *                 .commands(List.of("cd mysubdir", "npm ci", "npm run build", "npx cdk synth"))
 *                 .primaryOutputDirectory("mysubdir/cdk.out")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The underlying <code>aws-cdk-lib/aws-codepipeline.Pipeline</code> construct will be produced
 * when <code>app.synth()</code> is called. You can also force it to be produced
 * earlier by calling <code>pipeline.buildPipeline()</code>. After you've called
 * that method, you can inspect the constructs that were produced by
 * accessing the properties of the <code>pipeline</code> object.
 * <p>
 * <h4>Commands for other languages and package managers</h4>
 * <p>
 * The commands you pass to <code>new ShellStep</code> will be very similar to the commands
 * you run on your own workstation to install dependencies and synth your CDK
 * project. Here are some (non-exhaustive) examples for what those commands might
 * look like in a number of different situations.
 * <p>
 * For Yarn, the install commands are different:
 * <p>
 * <blockquote><pre>
 * IFileSetProducer source;
 * // the repository source
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(source)
 *                 .commands(List.of("yarn install --frozen-lockfile", "yarn build", "npx cdk synth"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * For Python projects, remember to install the CDK CLI globally (as
 * there is no <code>package.json</code> to automatically install it for you):
 * <p>
 * <blockquote><pre>
 * IFileSetProducer source;
 * // the repository source
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(source)
 *                 .commands(List.of("pip install -r requirements.txt", "npm install -g aws-cdk", "cdk synth"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * For Java projects, remember to install the CDK CLI globally (as
 * there is no <code>package.json</code> to automatically install it for you),
 * and the Maven compilation step is automatically executed for you
 * as you run <code>cdk synth</code>:
 * <p>
 * <blockquote><pre>
 * IFileSetProducer source;
 * // the repository source
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(source)
 *                 .commands(List.of("npm install -g aws-cdk", "cdk synth"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can adapt these examples to your own situation.
 * <p>
 * <h4>Migrating from buildspec.yml files</h4>
 * <p>
 * You may currently have the build instructions for your CodeBuild Projects in a
 * <code>buildspec.yml</code> file in your source repository. In addition to your build
 * commands, the CodeBuild Project's buildspec also controls some information that
 * CDK Pipelines manages for you, like artifact identifiers, input artifact
 * locations, Docker authorization, and exported variables.
 * <p>
 * Since there is no way in general for CDK Pipelines to modify the file in your
 * resource repository, CDK Pipelines configures the BuildSpec directly on the
 * CodeBuild Project, instead of loading it from the <code>buildspec.yml</code> file.
 * This requires a pipeline self-mutation to update.
 * <p>
 * To avoid this, put your build instructions in a separate script, for example
 * <code>build.sh</code>, and call that script from the build <code>commands</code> array:
 * <p>
 * <blockquote><pre>
 * IFileSetProducer source;
 * 
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(source)
 *                 .commands(List.of("./build.sh"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Doing so keeps your exact build instructions in sync with your source code in
 * the source repository where it belongs, and provides a convenient build script
 * for developers at the same time.
 * <p>
 * <h4>CodePipeline Sources</h4>
 * <p>
 * In CodePipeline, <em>Sources</em> define where the source of your application lives.
 * When a change to the source is detected, the pipeline will start executing.
 * Source objects can be created by factory methods on the <code>CodePipelineSource</code> class:
 * <p>
 * <h5>GitHub, GitHub Enterprise, BitBucket using a connection</h5>
 * <p>
 * The recommended way of connecting to GitHub or BitBucket is by using a <em>connection</em>.
 * You will first use the AWS Console to authenticate to the source control
 * provider, and then use the connection ARN in your pipeline definition:
 * <p>
 * <blockquote><pre>
 * CodePipelineSource.connection("org/repo", "branch", ConnectionSourceOptions.builder()
 *         .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h5>GitHub using OAuth</h5>
 * <p>
 * You can also authenticate to GitHub using a personal access token. This expects
 * that you've created a personal access token and stored it in Secrets Manager.
 * By default, the source object will look for a secret named <strong>github-token</strong>, but
 * you can change the name. The token should have the <strong>repo</strong> and <strong>admin:repo_hook</strong>
 * scopes.
 * <p>
 * <blockquote><pre>
 * CodePipelineSource.gitHub("org/repo", "branch", GitHubSourceOptions.builder()
 *         // This is optional
 *         .authentication(SecretValue.secretsManager("my-token"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h5>CodeCommit</h5>
 * <p>
 * You can use a CodeCommit repository as the source. Either create or import
 * that the CodeCommit repository and then use <code>CodePipelineSource.codeCommit</code>
 * to reference it:
 * <p>
 * <blockquote><pre>
 * IRepository repository = Repository.fromRepositoryName(this, "Repository", "my-repository");
 * CodePipelineSource.codeCommit(repository, "main");
 * </pre></blockquote>
 * <p>
 * <h5>S3</h5>
 * <p>
 * You can use a zip file in S3 as the source of the pipeline. The pipeline will be
 * triggered every time the file in S3 is changed:
 * <p>
 * <blockquote><pre>
 * IBucket bucket = Bucket.fromBucketName(this, "Bucket", "amzn-s3-demo-bucket");
 * CodePipelineSource.s3(bucket, "my/source.zip");
 * </pre></blockquote>
 * <p>
 * <h5>ECR</h5>
 * <p>
 * You can use a Docker image in ECR as the source of the pipeline. The pipeline will be
 * triggered every time an image is pushed to ECR:
 * <p>
 * <blockquote><pre>
 * Repository repository = new Repository(this, "Repository");
 * CodePipelineSource.ecr(repository);
 * </pre></blockquote>
 * <p>
 * <h4>Additional inputs</h4>
 * <p>
 * <code>ShellStep</code> allows passing in more than one input: additional
 * inputs will be placed in the directories you specify. Any step that produces an
 * output file set can be used as an input, such as a <code>CodePipelineSource</code>, but
 * also other <code>ShellStep</code>:
 * <p>
 * <blockquote><pre>
 * ShellStep prebuild = ShellStep.Builder.create("Prebuild")
 *         .input(CodePipelineSource.gitHub("myorg/repo1", "main"))
 *         .primaryOutputDirectory("./build")
 *         .commands(List.of("./build.sh"))
 *         .build();
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(CodePipelineSource.gitHub("myorg/repo2", "main"))
 *                 .additionalInputs(Map.of(
 *                         "subdir", CodePipelineSource.gitHub("myorg/repo3", "main"),
 *                         "../siblingdir", prebuild))
 * 
 *                 .commands(List.of("./build.sh"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>CDK application deployments</h3>
 * <p>
 * After you have defined the pipeline and the <code>synth</code> step, you can add one or
 * more CDK <code>Stages</code> which will be deployed to their target environments. To do
 * so, call <code>pipeline.addStage()</code> on the Stage object:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline;
 * 
 * // Do this as many times as necessary with any account and region
 * // Account and region may different from the pipeline's.
 * pipeline.addStage(
 * MyApplicationStage.Builder.create(this, "Prod")
 *         .env(Environment.builder()
 *                 .account("123456789012")
 *                 .region("eu-west-1")
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * CDK Pipelines will automatically discover all <code>Stacks</code> in the given <code>Stage</code>
 * object, determine their dependency order, and add appropriate actions to the
 * pipeline to publish the assets referenced in those stacks and deploy the stacks
 * in the right order.
 * <p>
 * If the <code>Stacks</code> are targeted at an environment in a different AWS account or
 * Region and that environment has been
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html">bootstrapped</a>
 * , CDK Pipelines will transparently make sure the IAM roles are set up
 * correctly and any requisite replication Buckets are created.
 * <p>
 * <h4>Deploying in parallel</h4>
 * <p>
 * By default, all applications added to CDK Pipelines by calling <code>addStage()</code> will
 * be deployed in sequence, one after the other. If you have a lot of stages, you can
 * speed up the pipeline by choosing to deploy some stages in parallel. You do this
 * by calling <code>addWave()</code> instead of <code>addStage()</code>: a <em>wave</em> is a set of stages that
 * are all deployed in parallel instead of sequentially. Waves themselves are still
 * deployed in sequence. For example, the following will deploy two copies of your
 * application to <code>eu-west-1</code> and <code>eu-central-1</code> in parallel:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline;
 * 
 * Wave europeWave = pipeline.addWave("Europe");
 * europeWave.addStage(
 * MyApplicationStage.Builder.create(this, "Ireland")
 *         .env(Environment.builder().region("eu-west-1").build())
 *         .build());
 * europeWave.addStage(
 * MyApplicationStage.Builder.create(this, "Germany")
 *         .env(Environment.builder().region("eu-central-1").build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Deploying to other accounts / encrypting the Artifact Bucket</h4>
 * <p>
 * CDK Pipelines can transparently deploy to other Regions and other accounts
 * (provided those target environments have been
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html"><em>bootstrapped</em></a>).
 * However, deploying to another account requires one additional piece of
 * configuration: you need to enable <code>crossAccountKeys: true</code> when creating the
 * pipeline.
 * <p>
 * This will encrypt the artifact bucket(s), but incurs a cost for maintaining the
 * KMS key.
 * <p>
 * You may also wish to enable automatic key rotation for the created KMS key.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         // Encrypt artifacts, required for cross-account deployments
 *         .crossAccountKeys(true)
 *         .enableKeyRotation(true) // optional
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(CodePipelineSource.connection("my-org/my-app", "main", ConnectionSourceOptions.builder()
 *                         .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *                         .build()))
 *                 .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Deploying without change sets</h4>
 * <p>
 * Deployment is done by default with <code>CodePipeline</code> engine using change sets,
 * i.e. to first create a change set and then execute it. This allows you to inject
 * steps that inspect the change set and approve or reject it, but failed deployments
 * are not retryable and creation of the change set costs time.
 * <p>
 * The creation of change sets can be switched off by setting <code>useChangeSets: false</code>:
 * <p>
 * <blockquote><pre>
 * ShellStep synth;
 * 
 * 
 * public class PipelineStack extends Stack {
 *     public PipelineStack(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public PipelineStack(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *                 .synth(synth)
 * 
 *                 // Disable change set creation and make deployments in pipeline as single step
 *                 .useChangeSets(false)
 *                 .build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Validation</h3>
 * <p>
 * Every <code>addStage()</code> and <code>addWave()</code> command takes additional options. As part of these options,
 * you can specify <code>pre</code> and <code>post</code> steps, which are arbitrary steps that run before or after
 * the contents of the stage or wave, respectively. You can use these to add validations like
 * manual or automated gates to your pipeline. We recommend putting manual approval gates in the set of <code>pre</code> steps, and automated approval gates in
 * the set of <code>post</code> steps.
 * <p>
 * The following example shows both an automated approval in the form of a <code>ShellStep</code>, and
 * a manual approval in the form of a <code>ManualApprovalStep</code> added to the pipeline. Both must
 * pass in order to promote from the <code>PreProd</code> to the <code>Prod</code> environment:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline;
 * 
 * MyApplicationStage preprod = new MyApplicationStage(this, "PreProd");
 * MyApplicationStage prod = new MyApplicationStage(this, "Prod");
 * 
 * pipeline.addStage(preprod, AddStageOpts.builder()
 *         .post(List.of(
 *             ShellStep.Builder.create("Validate Endpoint")
 *                     .commands(List.of("curl -Ssf https://my.webservice.com/"))
 *                     .build()))
 *         .build());
 * pipeline.addStage(prod, AddStageOpts.builder()
 *         .pre(List.of(new ManualApprovalStep("PromoteToProd")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also specify steps to be executed at the stack level. To achieve this, you can specify the stack and step via the <code>stackSteps</code> property:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline;
 * public class MyStacksStage extends Stage {
 *     public final Stack stack1;
 *     public final Stack stack2;
 * 
 *     public MyStacksStage(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public MyStacksStage(Construct scope, String id, StageProps props) {
 *         super(scope, id, props);
 *         this.stack1 = new Stack(this, "stack1");
 *         this.stack2 = new Stack(this, "stack2");
 *     }
 * }
 * MyStacksStage prod = new MyStacksStage(this, "Prod");
 * 
 * pipeline.addStage(prod, AddStageOpts.builder()
 *         .stackSteps(List.of(StackSteps.builder()
 *                 .stack(prod.getStack1())
 *                 .pre(List.of(new ManualApprovalStep("Pre-Stack Check"))) // Executed before stack is prepared
 *                 .changeSet(List.of(new ManualApprovalStep("ChangeSet Approval"))) // Executed after stack is prepared but before the stack is deployed
 *                 .post(List.of(new ManualApprovalStep("Post-Deploy Check")))
 *                 .build(), StackSteps.builder()
 *                 .stack(prod.getStack2())
 *                 .post(List.of(new ManualApprovalStep("Post-Deploy Check")))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you specify multiple steps, they will execute in parallel by default. You can add dependencies between them
 * to if you wish to specify an order. To add a dependency, call <code>step.addStepDependency()</code>:
 * <p>
 * <blockquote><pre>
 * ManualApprovalStep firstStep = new ManualApprovalStep("A");
 * ManualApprovalStep secondStep = new ManualApprovalStep("B");
 * secondStep.addStepDependency(firstStep);
 * </pre></blockquote>
 * <p>
 * For convenience, <code>Step.sequence()</code> will take an array of steps and dependencies between adjacent steps,
 * so that the whole list executes in order:
 * <p>
 * <blockquote><pre>
 * // Step A will depend on step B and step B will depend on step C
 * Step[] orderedSteps = Step.sequence(List.of(
 *     new ManualApprovalStep("A"),
 *     new ManualApprovalStep("B"),
 *     new ManualApprovalStep("C")));
 * </pre></blockquote>
 * <p>
 * <h4>Using CloudFormation Stack Outputs in approvals</h4>
 * <p>
 * Because many CloudFormation deployments result in the generation of resources with unpredictable
 * names, validations have support for reading back CloudFormation Outputs after a deployment. This
 * makes it possible to pass (for example) the generated URL of a load balancer to the test set.
 * <p>
 * To use Stack Outputs, expose the <code>CfnOutput</code> object you're interested in, and
 * pass it to <code>envFromCfnOutputs</code> of the <code>ShellStep</code>:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline;
 * public class MyOutputStage extends Stage {
 *     public final CfnOutput loadBalancerAddress;
 * 
 *     public MyOutputStage(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public MyOutputStage(Construct scope, String id, StageProps props) {
 *         super(scope, id, props);
 *         this.loadBalancerAddress = CfnOutput.Builder.create(this, "Output")
 *                 .value("value")
 *                 .build();
 *     }
 * }
 * 
 * MyOutputStage lbApp = new MyOutputStage(this, "MyApp");
 * pipeline.addStage(lbApp, AddStageOpts.builder()
 *         .post(List.of(
 *             ShellStep.Builder.create("HitEndpoint")
 *                     .envFromCfnOutputs(Map.of(
 *                             // Make the load balancer address available as $URL inside the commands
 *                             "URL", lbApp.getLoadBalancerAddress()))
 *                     .commands(List.of("curl -Ssf $URL"))
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Running scripts compiled during the synth step</h4>
 * <p>
 * As part of a validation, you probably want to run a test suite that's more
 * elaborate than what can be expressed in a couple of lines of shell script.
 * You can bring additional files into the shell script validation by supplying
 * the <code>input</code> or <code>additionalInputs</code> property of <code>ShellStep</code>. The input can
 * be produced by the <code>Synth</code> step, or come from a source or any other build
 * step.
 * <p>
 * Here's an example that captures an additional output directory in the synth
 * step and runs tests from there:
 * <p>
 * <blockquote><pre>
 * ShellStep synth;
 * 
 * MyApplicationStage stage = new MyApplicationStage(this, "MyApplication");
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline").synth(synth).build();
 * 
 * pipeline.addStage(stage, AddStageOpts.builder()
 *         .post(List.of(
 *             ShellStep.Builder.create("Approve")
 *                     // Use the contents of the 'integ' directory from the synth step as the input
 *                     .input(synth.addOutputDirectory("integ"))
 *                     .commands(List.of("cd integ &amp;&amp; ./run.sh"))
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Customizing CodeBuild Projects</h3>
 * <p>
 * CDK pipelines will generate CodeBuild projects for each <code>ShellStep</code> you use, and it
 * will also generate CodeBuild projects to publish assets and perform the self-mutation
 * of the pipeline. To control the various aspects of the CodeBuild projects that get
 * generated, use a <code>CodeBuildStep</code> instead of a <code>ShellStep</code>. This class has a number
 * of properties that allow you to customize various aspects of the projects:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup mySecurityGroup;
 * 
 * CodeBuildStep.Builder.create("Synth")
 *         // ...standard ShellStep props...
 *         .commands(List.of())
 *         .env(Map.of())
 * 
 *         // If you are using a CodeBuildStep explicitly, set the 'cdk.out' directory
 *         // to be the synth step's output.
 *         .primaryOutputDirectory("cdk.out")
 * 
 *         // Control the name of the project
 *         .projectName("MyProject")
 * 
 *         // Control parts of the BuildSpec other than the regular 'build' and 'install' commands
 *         .partialBuildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2")))
 * 
 *         // Control the build environment
 *         .buildEnvironment(BuildEnvironment.builder()
 *                 .computeType(ComputeType.LARGE)
 *                 .privileged(true)
 *                 .build())
 *         .timeout(Duration.minutes(90))
 *         .fileSystemLocations(List.of(FileSystemLocation.efs(EfsFileSystemLocationProps.builder()
 *                 .identifier("myidentifier2")
 *                 .location("myclodation.mydnsroot.com:/loc")
 *                 .mountPoint("/media")
 *                 .mountOptions("opts")
 *                 .build())))
 * 
 *         // Control Elastic Network Interface creation
 *         .vpc(vpc)
 *         .subnetSelection(SubnetSelection.builder().subnetType(SubnetType.PRIVATE_WITH_EGRESS).build())
 *         .securityGroups(List.of(mySecurityGroup))
 * 
 *         // Control caching
 *         .cache(Cache.bucket(new Bucket(this, "Cache")))
 * 
 *         // Additional policy statements for the execution role
 *         .rolePolicyStatements(List.of(
 *             PolicyStatement.Builder.create().build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also configure defaults for <em>all</em> CodeBuild projects by passing <code>codeBuildDefaults</code>,
 * or just for the synth, asset publishing, and self-mutation projects by passing <code>synthCodeBuildDefaults</code>,
 * <code>assetPublishingCodeBuildDefaults</code>, or <code>selfMutationCodeBuildDefaults</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * 
 * Vpc vpc;
 * SecurityGroup mySecurityGroup;
 * 
 * 
 * CodePipeline.Builder.create(this, "Pipeline")
 *         // Standard CodePipeline properties
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(CodePipelineSource.connection("my-org/my-app", "main", ConnectionSourceOptions.builder()
 *                         .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *                         .build()))
 *                 .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                 .build())
 * 
 *         // Defaults for all CodeBuild projects
 *         .codeBuildDefaults(CodeBuildOptions.builder()
 *                 // Prepend commands and configuration to all projects
 *                 .partialBuildSpec(BuildSpec.fromObject(Map.of(
 *                         "version", "0.2")))
 * 
 *                 // Control the build environment
 *                 .buildEnvironment(BuildEnvironment.builder()
 *                         .computeType(ComputeType.LARGE)
 *                         .build())
 * 
 *                 // Control Elastic Network Interface creation
 *                 .vpc(vpc)
 *                 .subnetSelection(SubnetSelection.builder().subnetType(SubnetType.PRIVATE_WITH_EGRESS).build())
 *                 .securityGroups(List.of(mySecurityGroup))
 * 
 *                 // Additional policy statements for the execution role
 *                 .rolePolicy(List.of(
 *                     PolicyStatement.Builder.create().build()))
 * 
 *                 // Information about logs
 *                 .logging(LoggingOptions.builder()
 *                         .cloudWatch(CloudWatchLoggingOptions.builder()
 *                                 .logGroup(new LogGroup(this, "MyLogGroup"))
 *                                 .build())
 *                         .s3(S3LoggingOptions.builder()
 *                                 .bucket(new Bucket(this, "LogBucket"))
 *                                 .build())
 *                         .build())
 *                 .build())
 * 
 *         .synthCodeBuildDefaults(CodeBuildOptions.builder().build())
 *         .assetPublishingCodeBuildDefaults(CodeBuildOptions.builder().build())
 *         .selfMutationCodeBuildDefaults(CodeBuildOptions.builder().build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Arbitrary CodePipeline actions</h3>
 * <p>
 * If you want to add a type of CodePipeline action to the CDK Pipeline that
 * doesn't have a matching class yet, you can define your own step class that extends
 * <code>Step</code> and implements <code>ICodePipelineActionFactory</code>.
 * <p>
 * Here's an example that adds a Jenkins step:
 * <p>
 * <blockquote><pre>
 * public class MyJenkinsStep extends Step implements ICodePipelineActionFactory {
 *     public MyJenkinsStep(JenkinsProvider provider, FileSet input) {
 *         super("MyJenkinsStep");
 * 
 *         // This is necessary if your step accepts parameters, like environment variables,
 *         // that may contain outputs from other steps. It doesn't matter what the
 *         // structure is, as long as it contains the values that may contain outputs.
 *         this.discoverReferencedOutputs(Map.of(
 *                 "env", Map.of()));
 *     }
 * 
 *     public CodePipelineActionFactoryResult produceAction(IStage stage, ProduceActionOptions options) {
 *         // This is where you control what type of Action gets added to the
 *         // CodePipeline
 *         stage.addAction(
 *         JenkinsAction.Builder.create()
 *                 // Copy 'actionName' and 'runOrder' from the options
 *                 .actionName(options.getActionName())
 *                 .runOrder(options.getRunOrder())
 * 
 *                 // Jenkins-specific configuration
 *                 .type(JenkinsActionType.TEST)
 *                 .jenkinsProvider(this.provider)
 *                 .projectName("MyJenkinsProject")
 * 
 *                 // Translate the FileSet into a codepipeline.Artifact
 *                 .inputs(List.of(options.artifacts.toCodePipeline(this.input)))
 *                 .build());
 * 
 *         return CodePipelineActionFactoryResult.builder().runOrdersConsumed(1).build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Another example, adding a lambda step referencing outputs from a stack:
 * <p>
 * <blockquote><pre>
 * public class MyLambdaStep extends Step implements ICodePipelineActionFactory {
 *     private StackOutputReference stackOutputReference;
 * 
 *     public MyLambdaStep(Function fn, CfnOutput stackOutput) {
 *         super("MyLambdaStep");
 *         this.stackOutputReference = StackOutputReference.fromCfnOutput(stackOutput);
 *     }
 * 
 *     public CodePipelineActionFactoryResult produceAction(IStage stage, ProduceActionOptions options) {
 *         stage.addAction(
 *         LambdaInvokeAction.Builder.create()
 *                 .actionName(options.getActionName())
 *                 .runOrder(options.getRunOrder())
 *                 // Map the reference to the variable name the CDK has generated for you.
 *                 .userParameters(Map.of(
 *                         "stackOutput", options.stackOutputsMap.toCodePipeline(this.stackOutputReference)))
 *                 .lambda(this.fn)
 *                 .build());
 * 
 *         return CodePipelineActionFactoryResult.builder().runOrdersConsumed(1).build();
 *     }public get consumedStackOutputs(): pipelines.StackOutputReference[] {
 *         return [this.stackOutputReference];
 *       }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Using an existing AWS Codepipeline</h3>
 * <p>
 * If you wish to use an existing <code>CodePipeline.Pipeline</code> while using the modern API's
 * methods and classes, you can pass in the existing <code>CodePipeline.Pipeline</code> to be built upon
 * instead of having the <code>pipelines.CodePipeline</code> construct create a new <code>CodePipeline.Pipeline</code>.
 * This also gives you more direct control over the underlying <code>CodePipeline.Pipeline</code> construct
 * if the way the modern API creates it doesn't allow for desired configurations. Use <code>CodePipelineFileset</code> to convert CodePipeline <strong>artifacts</strong> into CDK Pipelines <strong>file sets</strong>,
 * that can be used everywhere a file set or file set producer is expected.
 * <p>
 * Here's an example of passing in an existing pipeline and using a <em>source</em> that's already
 * in the pipeline:
 * <p>
 * <blockquote><pre>
 * Pipeline codePipeline;
 * 
 * 
 * Artifact sourceArtifact = new Artifact("MySourceArtifact");
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .codePipeline(codePipeline)
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(CodePipelineFileSet.fromArtifact(sourceArtifact))
 *                 .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If your existing pipeline already provides a synth step, pass the existing
 * artifact in place of the <code>synth</code> step:
 * <p>
 * <blockquote><pre>
 * Pipeline codePipeline;
 * 
 * 
 * Artifact buildArtifact = new Artifact("MyBuildArtifact");
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .codePipeline(codePipeline)
 *         .synth(CodePipelineFileSet.fromArtifact(buildArtifact))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that if you provide an existing pipeline, you cannot provide values for
 * <code>pipelineName</code>, <code>crossAccountKeys</code>, <code>reuseCrossRegionSupportStacks</code>, or <code>role</code>
 * because those values are passed in directly to the underlying <code>codepipeline.Pipeline</code>.
 * <p>
 * <h2>Using Docker in the pipeline</h2>
 * <p>
 * Docker can be used in 3 different places in the pipeline:
 * <p>
 * <ul>
 * <li>If you are using Docker image assets in your application stages: Docker will
 * run in the asset publishing projects.</li>
 * <li>If you are using Docker image assets in your stack (for example as
 * images for your CodeBuild projects): Docker will run in the self-mutate project.</li>
 * <li>If you are using Docker to bundle file assets anywhere in your project (for
 * example, if you are using such construct libraries as
 * <code>aws-cdk-lib/aws-lambda-nodejs</code>): Docker will run in the
 * <em>synth</em> project.</li>
 * </ul>
 * <p>
 * For the first case, you don't need to do anything special. For the other two cases,
 * you need to make sure that <strong>privileged mode</strong> is enabled on the correct CodeBuild
 * projects, so that Docker can run correctly. The follow sections describe how to do
 * that.
 * <p>
 * You may also need to authenticate to Docker registries to avoid being throttled.
 * See the section <strong>Authenticating to Docker registries</strong> below for information on how to do
 * that.
 * <p>
 * <h3>Using Docker image assets in the pipeline</h3>
 * <p>
 * If your <code>PipelineStack</code> is using Docker image assets (as opposed to the application
 * stacks the pipeline is deploying), for example by the use of <code>LinuxBuildImage.fromAsset()</code>,
 * you need to pass <code>dockerEnabledForSelfMutation: true</code> to the pipeline. For example:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(CodePipelineSource.connection("my-org/my-app", "main", ConnectionSourceOptions.builder()
 *                         .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *                         .build()))
 *                 .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                 .build())
 * 
 *         // Turn this on because the pipeline uses Docker image assets
 *         .dockerEnabledForSelfMutation(true)
 *         .build();
 * 
 * pipeline.addWave("MyWave", WaveOptions.builder()
 *         .post(List.of(
 *             CodeBuildStep.Builder.create("RunApproval")
 *                     .commands(List.of("command-from-image"))
 *                     .buildEnvironment(BuildEnvironment.builder()
 *                             // The user of a Docker image asset in the pipeline requires turning on
 *                             // 'dockerEnabledForSelfMutation'.
 *                             .buildImage(LinuxBuildImage.fromAsset(this, "Image", DockerImageAssetProps.builder()
 *                                     .directory("./docker-image")
 *                                     .build()))
 *                             .build())
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Important</strong>: You must turn on the <code>dockerEnabledForSelfMutation</code> flag,
 * commit and allow the pipeline to self-update <em>before</em> adding the actual
 * Docker asset.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Using bundled file assets</h3>
 * <p>
 * If you are using asset bundling anywhere (such as automatically done for you
 * if you add a construct like <code>aws-cdk-lib/aws-lambda-nodejs</code>), you need to pass
 * <code>dockerEnabledForSynth: true</code> to the pipeline. For example:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(CodePipelineSource.connection("my-org/my-app", "main", ConnectionSourceOptions.builder()
 *                         .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *                         .build()))
 *                 .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                 .build())
 * 
 *         // Turn this on because the application uses bundled file assets
 *         .dockerEnabledForSynth(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Important</strong>: You must turn on the <code>dockerEnabledForSynth</code> flag,
 * commit and allow the pipeline to self-update <em>before</em> adding the actual
 * Docker asset.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Authenticating to Docker registries</h3>
 * <p>
 * You can specify credentials to use for authenticating to Docker registries as part of the
 * pipeline definition. This can be useful if any Docker image assets — in the pipeline or
 * any of the application stages — require authentication, either due to being in a
 * different environment (e.g., ECR repo) or to avoid throttling (e.g., DockerHub).
 * <p>
 * <blockquote><pre>
 * ISecret dockerHubSecret = Secret.fromSecretCompleteArn(this, "DHSecret", "arn:aws:...");
 * ISecret customRegSecret = Secret.fromSecretCompleteArn(this, "CRSecret", "arn:aws:...");
 * IRepository repo1 = Repository.fromRepositoryArn(this, "Repo", "arn:aws:ecr:eu-west-1:0123456789012:repository/Repo1");
 * IRepository repo2 = Repository.fromRepositoryArn(this, "Repo", "arn:aws:ecr:eu-west-1:0123456789012:repository/Repo2");
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .dockerCredentials(List.of(DockerCredential.dockerHub(dockerHubSecret), DockerCredential.customRegistry("dockerregistry.example.com", customRegSecret), DockerCredential.ecr(List.of(repo1, repo2))))
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(CodePipelineSource.connection("my-org/my-app", "main", ConnectionSourceOptions.builder()
 *                         .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *                         .build()))
 *                 .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * For authenticating to Docker registries that require a username and password combination
 * (like DockerHub), create a Secrets Manager Secret with fields named <code>username</code>
 * and <code>secret</code>, and import it (the field names change be customized).
 * <p>
 * Authentication to ECR repositories is done using the execution role of the
 * relevant CodeBuild job. Both types of credentials can be provided with an
 * optional role to assume before requesting the credentials.
 * <p>
 * By default, the Docker credentials provided to the pipeline will be available to
 * the <strong>Synth</strong>, <strong>Self-Update</strong>, and <strong>Asset Publishing</strong> actions within the
 * *pipeline. The scope of the credentials can be limited via the <code>DockerCredentialUsage</code> option.
 * <p>
 * <blockquote><pre>
 * ISecret dockerHubSecret = Secret.fromSecretCompleteArn(this, "DHSecret", "arn:aws:...");
 * // Only the image asset publishing actions will be granted read access to the secret.
 * DockerCredential creds = DockerCredential.dockerHub(dockerHubSecret, ExternalDockerCredentialOptions.builder()
 *         .usages(List.of(DockerCredentialUsage.ASSET_PUBLISHING))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>CDK Environment Bootstrapping</h2>
 * <p>
 * An <em>environment</em> is an <em>(account, region)</em> pair where you want to deploy a
 * CDK stack (see
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/environments.html">Environments</a>
 * in the CDK Developer Guide). In a Continuous Deployment pipeline, there are
 * at least two environments involved: the environment where the pipeline is
 * provisioned, and the environment where you want to deploy the application (or
 * different stages of the application). These can be the same, though best
 * practices recommend you isolate your different application stages from each
 * other in different AWS accounts or regions.
 * <p>
 * Before you can provision the pipeline, you have to <em>bootstrap</em> the environment you want
 * to create it in. If you are deploying your application to different environments, you
 * also have to bootstrap those and be sure to add a <em>trust</em> relationship.
 * <p>
 * After you have bootstrapped an environment and created a pipeline that deploys
 * to it, it's important that you don't delete the stack or change its <em>Qualifier</em>,
 * or future deployments to this environment will fail. If you want to upgrade
 * the bootstrap stack to a newer version, do that by updating it in-place.
 * <p>
 * <blockquote>
 * <p>
 * This library requires the <em>modern</em> bootstrapping stack which has
 * been updated specifically to support cross-account continuous delivery.
 * <p>
 * If you are using CDKv2, you do not need to do anything else. Modern
 * bootstrapping and modern stack synthesis (also known as "default stack
 * synthesis") is the default.
 * <p>
 * If you are using CDKv1, you need to opt in to modern bootstrapping and
 * modern stack synthesis using a feature flag. Make sure <code>cdk.json</code> includes:
 * <p>
 * <blockquote><pre>
 * {
 *   "context": {
 *     "&#64;aws-cdk/core:newStyleStackSynthesis": true
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * And be sure to run <code>cdk bootstrap</code> in the same directory as the <code>cdk.json</code>
 * file.
 * <p>
 * </blockquote>
 * <p>
 * To bootstrap an environment for provisioning the pipeline:
 * <p>
 * <blockquote><pre>
 * $ npx cdk bootstrap \
 *     [--profile admin-profile-1] \
 *     --cloudformation-execution-policies arn:aws:iam::aws:policy/AdministratorAccess \
 *     aws://111111111111/us-east-1
 * </pre></blockquote>
 * <p>
 * To bootstrap a different environment for deploying CDK applications into using
 * a pipeline in account <code>111111111111</code>:
 * <p>
 * <blockquote><pre>
 * $ npx cdk bootstrap \
 *     [--profile admin-profile-2] \
 *     --cloudformation-execution-policies arn:aws:iam::aws:policy/AdministratorAccess \
 *     --trust 11111111111 \
 *     aws://222222222222/us-east-2
 * </pre></blockquote>
 * <p>
 * If you only want to trust an account to do lookups (e.g, when your CDK application has a
 * <code>Vpc.fromLookup()</code> call), use the option <code>--trust-for-lookup</code>:
 * <p>
 * <blockquote><pre>
 * $ npx cdk bootstrap \
 *     [--profile admin-profile-2] \
 *     --cloudformation-execution-policies arn:aws:iam::aws:policy/AdministratorAccess \
 *     --trust-for-lookup 11111111111 \
 *     aws://222222222222/us-east-2
 * </pre></blockquote>
 * <p>
 * These command lines explained:
 * <p>
 * <ul>
 * <li><code>npx</code>: means to use the CDK CLI from the current NPM install. If you are using
 * a global install of the CDK CLI, leave this out.</li>
 * <li><code>--profile</code>: should indicate a profile with administrator privileges that has
 * permissions to provision a pipeline in the indicated account. You can leave this
 * flag out if either the AWS default credentials or the <code>AWS_*</code> environment
 * variables confer these permissions.</li>
 * <li><code>--cloudformation-execution-policies</code>: ARN of the managed policy that future CDK
 * deployments should execute with. By default this is <code>AdministratorAccess</code>, but
 * if you also specify the <code>--trust</code> flag to give another Account permissions to
 * deploy into the current account, you must specify a value here.</li>
 * <li><code>--trust</code>: indicates which other account(s) should have permissions to deploy
 * CDK applications into this account. In this case we indicate the Pipeline's account,
 * but you could also use this for developer accounts (don't do that for production
 * application accounts though!).</li>
 * <li><code>--trust-for-lookup</code>: gives a more limited set of permissions to the
 * trusted account, only allowing it to look up values such as availability zones, EC2 images and
 * VPCs. <code>--trust-for-lookup</code> does not give permissions to modify anything in the account.
 * Note that <code>--trust</code> implies <code>--trust-for-lookup</code>, so you don't need to specify
 * the same account twice.</li>
 * <li><code>aws://222222222222/us-east-2</code>: the account and region we're bootstrapping.</li>
 * </ul>
 * <p>
 * <blockquote>
 * <p>
 * Be aware that anyone who has access to the trusted Accounts <strong>effectively has all
 * permissions conferred by the configured CloudFormation execution policies</strong>,
 * allowing them to do things like read arbitrary S3 buckets and create arbitrary
 * infrastructure in the bootstrapped account. Restrict the list of <code>--trust</code>ed Accounts,
 * or restrict the policies configured by <code>--cloudformation-execution-policies</code>.
 * <p>
 * </blockquote><br><blockquote>
 * <p>
 * <strong>Security tip</strong>: we recommend that you use administrative credentials to an
 * account only to bootstrap it and provision the initial pipeline. Otherwise,
 * access to administrative credentials should be dropped as soon as possible.
 * <p>
 * </blockquote><br><blockquote>
 * <p>
 * <strong>On the use of AdministratorAccess</strong>: The use of the <code>AdministratorAccess</code> policy
 * ensures that your pipeline can deploy every type of AWS resource to your account.
 * Make sure you trust all the code and dependencies that make up your CDK app.
 * Check with the appropriate department within your organization to decide on the
 * proper policy to use.
 * <p>
 * If your policy includes permissions to create on attach permission to a role,
 * developers can escalate their privilege with more permissive permission.
 * Thus, we recommend implementing <a href="https://aws.amazon.com/premiumsupport/knowledge-center/iam-permission-boundaries/">permissions boundary</a>
 * in the CDK Execution role. To do this, you can bootstrap with the <code>--template</code> option with
 * <a href="https://github.com/aws-samples/aws-bootstrap-kit-examples/blob/ba28a97d289128281bc9483bcba12c1793f2c27a/source/1-SDLC-organization/lib/cdk-bootstrap-template.yml#L395">a customized template</a> that contains a permission boundary.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Migrating from old bootstrap stack</h3>
 * <p>
 * The bootstrap stack is a CloudFormation stack in your account named
 * <strong>CDKToolkit</strong> that provisions a set of resources required for the CDK
 * to deploy into that environment.
 * <p>
 * The "new" bootstrap stack (obtained by running <code>cdk bootstrap</code> with
 * <code>CDK_NEW_BOOTSTRAP=1</code>) is slightly more elaborate than the "old" stack. It
 * contains:
 * <p>
 * <ul>
 * <li>An S3 bucket and ECR repository with predictable names, so that we can reference
 * assets in these storage locations <em>without</em> the use of CloudFormation template
 * parameters.</li>
 * <li>A set of roles with permissions to access these asset locations and to execute
 * CloudFormation, assumable from whatever accounts you specify under <code>--trust</code>.</li>
 * </ul>
 * <p>
 * It is possible and safe to migrate from the old bootstrap stack to the new
 * bootstrap stack. This will create a new S3 file asset bucket in your account
 * and orphan the old bucket. You should manually delete the orphaned bucket
 * after you are sure you have redeployed all CDK applications and there are no
 * more references to the old asset bucket.
 * <p>
 * <h2>Considerations around Running at Scale</h2>
 * <p>
 * If you are planning to run pipelines for more than a hundred repos
 * deploying across multiple regions, then you will want to consider reusing
 * both artifacts buckets and cross-region replication buckets.
 * <p>
 * In a situation like this, you will want to have a separate CDK app / dedicated repo which creates
 * and managed the buckets which will be shared by the pipelines of all your other apps.
 * Note that this app must NOT be using the shared buckets because of chicken &amp; egg issues.
 * <p>
 * The following code assumes you have created and are managing your buckets in the aforementioned
 * separate cdk repo and are just importing them for use in one of your (many) pipelines.
 * <p>
 * <blockquote><pre>
 * String sharedXRegionUsWest1BucketArn;
 * String sharedXRegionUsWest1KeyArn;
 * 
 * String sharedXRegionUsWest2BucketArn;
 * String sharedXRegionUsWest2KeyArn;
 * 
 * 
 * IBucket usWest1Bucket = Bucket.fromBucketAttributes(scope, "UsWest1Bucket", BucketAttributes.builder()
 *         .bucketArn(sharedXRegionUsWest1BucketArn)
 *         .encryptionKey(Key.fromKeyArn(scope, "UsWest1BucketKeyArn", sharedXRegionUsWest1BucketArn))
 *         .build());
 * 
 * IBucket usWest2Bucket = Bucket.fromBucketAttributes(scope, "UsWest2Bucket", BucketAttributes.builder()
 *         .bucketArn(sharedXRegionUsWest2BucketArn)
 *         .encryptionKey(Key.fromKeyArn(scope, "UsWest2BucketKeyArn", sharedXRegionUsWest2KeyArn))
 *         .build());
 * 
 * Map&lt;String, IBucket&gt; crossRegionReplicationBuckets = Map.of(
 *         "us-west-1", usWest1Bucket,
 *         "us-west-2", usWest2Bucket);
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(CodePipelineSource.connection("my-org/my-app", "main", ConnectionSourceOptions.builder()
 *                         .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *                         .build()))
 *                 .commands(List.of("npm ci", "npm run build", "npx cdk synth"))
 *                 .build()) // Use shared buckets.
 *         .crossRegionReplicationBuckets(crossRegionReplicationBuckets)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Context Lookups</h2>
 * <p>
 * You might be using CDK constructs that need to look up <a href="https://docs.aws.amazon.com/cdk/latest/guide/context.html#context_methods">runtime
 * context</a>,
 * which is information from the target AWS Account and Region the CDK needs to
 * synthesize CloudFormation templates appropriate for that environment. Examples
 * of this kind of context lookups are the number of Availability Zones available
 * to you, a Route53 Hosted Zone ID, or the ID of an AMI in a given region. This
 * information is automatically looked up when you run <code>cdk synth</code>.
 * <p>
 * By default, a <code>cdk synth</code> performed in a pipeline will not have permissions
 * to perform these lookups, and the lookups will fail. This is by design.
 * <p>
 * <strong>Our recommended way of using lookups</strong> is by running <code>cdk synth</code> on the
 * developer workstation and checking in the <code>cdk.context.json</code> file, which
 * contains the results of the context lookups. This will make sure your
 * synthesized infrastructure is consistent and repeatable. If you do not commit
 * <code>cdk.context.json</code>, the results of the lookups may suddenly be different in
 * unexpected ways, and even produce results that cannot be deployed or will cause
 * data loss. To give an account permissions to perform lookups against an
 * environment, without being able to deploy to it and make changes, run
 * <code>cdk bootstrap --trust-for-lookup=&lt;account&gt;</code>.
 * <p>
 * If you want to use lookups directly from the pipeline, you either need to accept
 * the risk of nondeterminism, or make sure you save and load the
 * <code>cdk.context.json</code> file somewhere between synth runs. Finally, you should
 * give the synth CodeBuild execution role permissions to assume the bootstrapped
 * lookup roles. As an example, doing so would look like this:
 * <p>
 * <blockquote><pre>
 * CodePipeline.Builder.create(this, "Pipeline")
 *         .synth(CodeBuildStep.Builder.create("Synth")
 *                 .input(CodePipelineSource.connection("my-org/my-app", "main", ConnectionSourceOptions.builder()
 *                         .connectionArn("arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41")
 *                         .build()))
 *                 .commands(List.of("...", "npm ci", "npm run build", "npx cdk synth", "..."))
 *                 .rolePolicyStatements(List.of(
 *                     PolicyStatement.Builder.create()
 *                             .actions(List.of("sts:AssumeRole"))
 *                             .resources(List.of("*"))
 *                             .conditions(Map.of(
 *                                     "StringEquals", Map.of(
 *                                             "iam:ResourceTag/aws-cdk:bootstrap-role", "lookup")))
 *                             .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The above example requires that the target environments have all
 * been bootstrapped with bootstrap stack version <code>8</code>, released with
 * CDK CLI <code>1.114.0</code>.
 * <p>
 * <h2>Security Considerations</h2>
 * <p>
 * It's important to stay safe while employing Continuous Delivery. The CDK Pipelines
 * library comes with secure defaults to the best of our ability, but by its
 * very nature the library cannot take care of everything.
 * <p>
 * We therefore expect you to mind the following:
 * <p>
 * <ul>
 * <li>Maintain dependency hygiene and vet 3rd-party software you use. Any software you
 * run on your build machine has the ability to change the infrastructure that gets
 * deployed. Be careful with the software you depend on.</li>
 * <li>Use dependency locking to prevent accidental upgrades! The default <code>CdkSynths</code> that
 * come with CDK Pipelines will expect <code>package-lock.json</code> and <code>yarn.lock</code> to
 * ensure your dependencies are the ones you expect.</li>
 * <li>CDK Pipelines runs on resources created in your own account, and the configuration
 * of those resources is controlled by developers submitting code through the pipeline.
 * Therefore, CDK Pipelines by itself cannot protect against malicious
 * developers trying to bypass compliance checks. If your threat model includes
 * developers writing CDK code, you should have external compliance mechanisms in place like
 * <a href="https://aws.amazon.com/blogs/mt/proactively-keep-resources-secure-and-compliant-with-aws-cloudformation-hooks/">AWS CloudFormation Hooks</a>
 * (preventive) or <a href="https://aws.amazon.com/config/">AWS Config</a> (reactive) that
 * the CloudFormation Execution Role does not have permissions to disable.</li>
 * <li>Credentials to production environments should be short-lived. After
 * bootstrapping and the initial pipeline provisioning, there is no more need for
 * developers to have access to any of the account credentials; all further
 * changes can be deployed through git. Avoid the chances of credentials leaking
 * by not having them in the first place!</li>
 * </ul>
 * <p>
 * <h3>Confirm permissions broadening</h3>
 * <p>
 * To keep tabs on the security impact of changes going out through your pipeline,
 * you can insert a security check before any stage deployment. This security check
 * will check if the upcoming deployment would add any new IAM permissions or
 * security group rules, and if so pause the pipeline and require you to confirm
 * the changes.
 * <p>
 * The security check will appear as two distinct actions in your pipeline: first
 * a CodeBuild project that runs <code>cdk diff</code> on the stage that's about to be deployed,
 * followed by a Manual Approval action that pauses the pipeline. If it so happens
 * that there no new IAM permissions or security group rules will be added by the deployment,
 * the manual approval step is automatically satisfied. The pipeline will look like this:
 * <p>
 * <blockquote><pre>
 * Pipeline
 * ├── ...
 * ├── MyApplicationStage
 * │    ├── MyApplicationSecurityCheck       // Security Diff Action
 * │    ├── MyApplicationManualApproval      // Manual Approval Action
 * │    ├── Stack.Prepare
 * │    └── Stack.Deploy
 * └── ...
 * </pre></blockquote>
 * <p>
 * You can insert the security check by using a <code>ConfirmPermissionsBroadening</code> step:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline;
 * 
 * MyApplicationStage stage = new MyApplicationStage(this, "MyApplication");
 * pipeline.addStage(stage, AddStageOpts.builder()
 *         .pre(List.of(ConfirmPermissionsBroadening.Builder.create("Check").stage(stage).build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To get notified when there is a change that needs your manual approval,
 * create an SNS Topic, subscribe your own email address, and pass it in as
 * as the <code>notificationTopic</code> property:
 * <p>
 * <blockquote><pre>
 * CodePipeline pipeline;
 * 
 * Topic topic = new Topic(this, "SecurityChangesTopic");
 * topic.addSubscription(new EmailSubscription("test&#64;email.com"));
 * 
 * MyApplicationStage stage = new MyApplicationStage(this, "MyApplication");
 * pipeline.addStage(stage, AddStageOpts.builder()
 *         .pre(List.of(
 *             ConfirmPermissionsBroadening.Builder.create("Check")
 *                     .stage(stage)
 *                     .notificationTopic(topic)
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: Manual Approvals notifications only apply when an application has security
 * check enabled.
 * <p>
 * <h2>Using a different deployment engine</h2>
 * <p>
 * CDK Pipelines supports multiple <em>deployment engines</em>, but this module vends a
 * construct for only one such engine: AWS CodePipeline. It is also possible to
 * use CDK Pipelines to build pipelines backed by other deployment engines.
 * <p>
 * Here is a list of CDK Libraries that integrate CDK Pipelines with
 * alternative deployment engines:
 * <p>
 * <ul>
 * <li>GitHub Workflows: <a href="https://github.com/cdklabs/cdk-pipelines-github"><code>cdk-pipelines-github</code></a></li>
 * </ul>
 * <p>
 * <h2>Troubleshooting</h2>
 * <p>
 * Here are some common errors you may encounter while using this library.
 * <p>
 * <h3>Pipeline: Internal Failure</h3>
 * <p>
 * If you see the following error during deployment of your pipeline:
 * <p>
 * <blockquote><pre>
 * CREATE_FAILED  | AWS::CodePipeline::Pipeline | Pipeline/Pipeline
 * Internal Failure
 * </pre></blockquote>
 * <p>
 * There's something wrong with your GitHub access token. It might be missing, or not have the
 * right permissions to access the repository you're trying to access.
 * <p>
 * <h3>Key: Policy contains a statement with one or more invalid principals</h3>
 * <p>
 * If you see the following error during deployment of your pipeline:
 * <p>
 * <blockquote><pre>
 * CREATE_FAILED | AWS::KMS::Key | Pipeline/Pipeline/ArtifactsBucketEncryptionKey
 * Policy contains a statement with one or more invalid principals.
 * </pre></blockquote>
 * <p>
 * One of the target (account, region) environments has not been bootstrapped
 * with the new bootstrap stack. Check your target environments and make sure
 * they are all bootstrapped.
 * <p>
 * <h3>Message: no matching base directory path found for cdk.out</h3>
 * <p>
 * If you see this error during the <strong>Synth</strong> step, it means that CodeBuild
 * is expecting to find a <code>cdk.out</code> directory in the root of your CodeBuild project,
 * but the directory wasn't there. There are two common causes for this:
 * <p>
 * <ul>
 * <li><code>cdk synth</code> is not being executed: <code>cdk synth</code> used to be run
 * implicitly for you, but you now have to explicitly include the command.
 * For NPM-based projects, add <code>npx cdk synth</code> to the end of the <code>commands</code>
 * property, for other languages add <code>npm install -g aws-cdk</code> and <code>cdk synth</code>.</li>
 * <li>Your CDK project lives in a subdirectory: you added a <code>cd &lt;somedirectory&gt;</code> command
 * to the list of commands; don't forget to tell the <code>ScriptStep</code> about the
 * different location of <code>cdk.out</code>, by passing <code>primaryOutputDirectory: '&lt;somedirectory&gt;/cdk.out'</code>.</li>
 * </ul>
 * <p>
 * <h3><Stack> is in ROLLBACK_COMPLETE state and can not be updated</h3>
 * <p>
 * If you see the following error during execution of your pipeline:
 * <p>
 * <blockquote><pre>
 * Stack ... is in ROLLBACK_COMPLETE state and can not be updated. (Service:
 * AmazonCloudFormation; Status Code: 400; Error Code: ValidationError; Request
 * ID: ...)
 * </pre></blockquote>
 * <p>
 * The stack failed its previous deployment, and is in a non-retryable state.
 * Go into the CloudFormation console, delete the stack, and retry the deployment.
 * <p>
 * <h3>Cannot find module 'xxxx' or its corresponding type declarations</h3>
 * <p>
 * You may see this if you are using TypeScript or other NPM-based languages,
 * when using NPM 7 on your workstation (where you generate <code>package-lock.json</code>)
 * and NPM 6 on the CodeBuild image used for synthesizing.
 * <p>
 * It looks like NPM 7 has started writing less information to <code>package-lock.json</code>,
 * leading NPM 6 reading that same file to not install all required packages anymore.
 * <p>
 * Make sure you are using the same NPM version everywhere, either downgrade your
 * workstation's version or upgrade the CodeBuild version.
 * <p>
 * <h3>Cannot find module '.../check-node-version.js' (MODULE_NOT_FOUND)</h3>
 * <p>
 * The above error may be produced by <code>npx</code> when executing the CDK CLI, or any
 * project that uses the AWS SDK for JavaScript, without the target application
 * having been installed yet. For example, it can be triggered by <code>npx cdk synth</code>
 * if <code>aws-cdk</code> is not in your <code>package.json</code>.
 * <p>
 * Work around this by either installing the target application using NPM <em>before</em>
 * running <code>npx</code>, or set the environment variable <code>NPM_CONFIG_UNSAFE_PERM=true</code>.
 * <p>
 * <h3>Cannot connect to the Docker daemon at unix:///var/run/docker.sock</h3>
 * <p>
 * If, in the 'Synth' action (inside the 'Build' stage) of your pipeline, you get an error like this:
 * <p>
 * <blockquote><pre>
 * stderr: docker: Cannot connect to the Docker daemon at unix:///var/run/docker.sock. Is the docker daemon running?.
 * See 'docker run --help'.
 * </pre></blockquote>
 * <p>
 * It means that the AWS CodeBuild project for 'Synth' is not configured to run in privileged mode,
 * which prevents Docker builds from happening. This typically happens if you use a CDK construct
 * that bundles asset using tools run via Docker, like <code>aws-lambda-nodejs</code>, <code>aws-lambda-python</code>,
 * <code>aws-lambda-go</code> and others.
 * <p>
 * Make sure you set the <code>privileged</code> environment variable to <code>true</code> in the synth definition:
 * <p>
 * After turning on <code>privilegedMode: true</code>, you will need to do a one-time manual cdk deploy of your
 * pipeline to get it going again (as with a broken 'synth' the pipeline will not be able to self
 * update to the right state).
 * <p>
 * <h3>Not authorized to perform sts:AssumeRole on arn:aws:iam::<em>:role/</em>-lookup-role-*</h3>
 * <p>
 * You may get an error like the following in the <strong>Synth</strong> step:
 * <p>
 * <blockquote><pre>
 * Could not assume role in target account using current credentials (which are for account 111111111111). User:
 * arn:aws:sts::111111111111:assumed-role/PipelineStack-PipelineBuildSynthCdkBuildProje-..../AWSCodeBuild-....
 * is not authorized to perform: sts:AssumeRole on resource:
 * arn:aws:iam::222222222222:role/cdk-hnb659fds-lookup-role-222222222222-us-east-1.
 * Please make sure that this role exists in the account. If it doesn't exist, (re)-bootstrap the environment with
 * the right '--trust', using the latest version of the CDK CLI.
 * </pre></blockquote>
 * <p>
 * This is a sign that the CLI is trying to do Context Lookups during the <strong>Synth</strong> step, which are failing
 * because it cannot assume the right role. We recommend you don't rely on Context Lookups in the pipeline at
 * all, and commit a file called <code>cdk.context.json</code> with the right lookup values in it to source control.
 * <p>
 * If you do want to do lookups in the pipeline, the cause is one of the following:
 * <p>
 * <ul>
 * <li>The target environment has not been bootstrapped; OR</li>
 * <li>The target environment has been bootstrapped without the right <code>--trust</code> relationship; OR</li>
 * <li>The CodeBuild execution role does not have permissions to call <code>sts:AssumeRole</code>.</li>
 * </ul>
 * <p>
 * See the section called <strong>Context Lookups</strong> for more information on using this feature.
 * <p>
 * <h3>IAM policies: Cannot exceed quota for PoliciesPerRole / Maximum policy size exceeded</h3>
 * <p>
 * This happens as a result of having a lot of targets in the Pipeline: the IAM policies that
 * get generated enumerate all required roles and grow too large.
 * <p>
 * Make sure you are on version <code>2.26.0</code> or higher, and that your <code>cdk.json</code> contains the
 * following:
 * <p>
 * <blockquote><pre>
 * {
 *   "context": {
 *     "aws-cdk-lib/aws-iam:minimizePolicies": true
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>S3 error: Access Denied</h3>
 * <p>
 * An "S3 Access Denied" error can have two causes:
 * <p>
 * <ul>
 * <li>Asset hashes have changed, but self-mutation has been disabled in the pipeline.</li>
 * <li>You have deleted and recreated the bootstrap stack, or changed its qualifier.</li>
 * </ul>
 * <p>
 * <h4>Bootstrap roles have been renamed or recreated</h4>
 * <p>
 * While attempting to deploy an application stage, the "Prepare" or "Deploy" stage may fail with a cryptic error like:
 * <p>
 * <code>Action execution failed Access Denied (Service: Amazon S3; Status Code: 403; Error Code: AccessDenied; Request ID: 0123456ABCDEFGH; S3 Extended Request ID: 3hWcrVkhFGxfiMb/rTJO0Bk7Qn95x5ll4gyHiFsX6Pmk/NT+uX9+Z1moEcfkL7H3cjH7sWZfeD0=; Proxy: null)</code>
 * <p>
 * This generally indicates that the roles necessary to deploy have been deleted (or deleted and re-created);
 * for example, if the bootstrap stack has been deleted and re-created, this scenario will happen. Under the hood,
 * the resources that rely on these roles (e.g., <code>cdk-$qualifier-deploy-role-$account-$region</code>) point to different
 * canonical IDs than the recreated versions of these roles, which causes the errors. There are no simple solutions
 * to this issue, and for that reason we <strong>strongly recommend</strong> that bootstrap stacks not be deleted and re-created
 * once created.
 * <p>
 * The most automated way to solve the issue is to introduce a secondary bootstrap stack. By changing the qualifier
 * that the pipeline stack looks for, a change will be detected and the impacted policies and resources will be updated.
 * A hypothetical recovery workflow would look something like this:
 * <p>
 * <ul>
 * <li>First, for all impacted environments, create a secondary bootstrap stack:</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * $ env CDK_NEW_BOOTSTRAP=1 npx cdk bootstrap \
 *     --qualifier random1234 \
 *     --toolkit-stack-name CDKToolkitTemp \
 *     aws://111111111111/us-east-1
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>Update all impacted stacks in the pipeline to use this new qualifier.
 * See https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html for more info.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Stack.Builder.create(this, "MyStack")
 *         // Update this qualifier to match the one used above.
 *         .synthesizer(DefaultStackSynthesizer.Builder.create()
 *                 .qualifier("randchars1234")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>Deploy the updated stacks. This will update the stacks to use the roles created in the new bootstrap stack.</li>
 * <li>(Optional) Restore back to the original state:
 * <p>
 * <ul>
 * <li>Revert the change made in step #2 above</li>
 * <li>Re-deploy the pipeline to use the original qualifier.</li>
 * <li>Delete the temporary bootstrap stack(s)</li>
 * </ul></li>
 * </ul>
 * <p>
 * <h5>Manual Alternative</h5>
 * <p>
 * Alternatively, the errors can be resolved by finding each impacted resource and policy, and correcting the policies
 * by replacing the canonical IDs (e.g., <code>AROAYBRETNYCYV6ZF2R93</code>) with the appropriate ARNs. As an example, the KMS
 * encryption key policy for the artifacts bucket may have a statement that looks like the following:
 * <p>
 * <blockquote><pre>
 * {
 *   "Effect": "Allow",
 *   "Principal": {
 *     // "AWS" : "AROAYBRETNYCYV6ZF2R93"  // Indicates this issue; replace this value
 *     "AWS": "arn:aws:iam::0123456789012:role/cdk-hnb659fds-deploy-role-0123456789012-eu-west-1" // Correct value
 *   },
 *   "Action": ["kms:Decrypt", "kms:DescribeKey"],
 *   "Resource": "*"
 * }
 * </pre></blockquote>
 * <p>
 * Any resource or policy that references the qualifier (<code>hnb659fds</code> by default) will need to be updated.
 * <p>
 * <h3>This CDK CLI is not compatible with the CDK library used by your application</h3>
 * <p>
 * The CDK CLI version used in your pipeline is too old to read the Cloud Assembly
 * produced by your CDK app.
 * <p>
 * Most likely this happens in the <code>SelfMutate</code> action, you are passing the <code>cliVersion</code>
 * parameter to control the version of the CDK CLI, and you just updated the CDK
 * framework version that your application uses. You either forgot to change the
 * <code>cliVersion</code> parameter, or changed the <code>cliVersion</code> in the same commit in which
 * you changed the framework version. Because a change to the pipeline settings needs
 * a successful run of the <code>SelfMutate</code> step to be applied, the next iteration of the
 * <code>SelfMutate</code> step still executes with the <em>old</em> CLI version, and that old CLI version
 * is not able to read the cloud assembly produced by the new framework version.
 * <p>
 * Solution: change the <code>cliVersion</code> first, commit, push and deploy, and only then
 * change the framework version.
 * <p>
 * We recommend you avoid specifying the <code>cliVersion</code> parameter at all. By default
 * the pipeline will use the latest CLI version, which will support all cloud assembly
 * versions.
 * <p>
 * <h2>Using Drop-in Docker Replacements</h2>
 * <p>
 * By default, the AWS CDK will build and publish Docker image assets using the
 * <code>docker</code> command. However, by specifying the <code>CDK_DOCKER</code> environment variable,
 * you can override the command that will be used to build and publish your
 * assets.
 * <p>
 * In CDK Pipelines, the drop-in replacement for the <code>docker</code> command must be
 * included in the CodeBuild environment and configured for your pipeline.
 * <p>
 * <h3>Adding to the default CodeBuild image</h3>
 * <p>
 * You can add a drop-in Docker replacement command to the default CodeBuild
 * environment by adding install-phase commands that encode how to install
 * your tooling and by adding the <code>CDK_DOCKER</code> environment variable to your
 * build environment.
 * <p>
 * <blockquote><pre>
 * IFileSetProducer source; // the repository source
 * String[] synthCommands; // Commands to synthesize your app
 * String[] installCommands;
 * // Commands to install your toolchain
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         // Standard CodePipeline properties...
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(source)
 *                 .commands(synthCommands)
 *                 .build())
 * 
 *         // Configure CodeBuild to use a drop-in Docker replacement.
 *         .codeBuildDefaults(CodeBuildOptions.builder()
 *                 .partialBuildSpec(BuildSpec.fromObject(Map.of(
 *                         "phases", Map.of(
 *                                 "install", Map.of(
 *                                         // Add the shell commands to install your drop-in Docker
 *                                         // replacement to the CodeBuild enviromment.
 *                                         "commands", installCommands)))))
 *                 .buildEnvironment(BuildEnvironment.builder()
 *                         .environmentVariables(Map.of(
 *                                 // Instruct the AWS CDK to use `drop-in-replacement` instead of
 *                                 // `docker` when building / publishing docker images.
 *                                 // e.g., `drop-in-replacement build . -f path/to/Dockerfile`
 *                                 "CDK_DOCKER", BuildEnvironmentVariable.builder().value("drop-in-replacement").build()))
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Using a custom build image</h3>
 * <p>
 * If you're using a custom build image in CodeBuild, you can override the
 * command the AWS CDK uses to build Docker images by providing <code>CDK_DOCKER</code> as
 * an <code>ENV</code> in your <code>Dockerfile</code> or by providing the environment variable in the
 * pipeline as shown below.
 * <p>
 * <blockquote><pre>
 * IFileSetProducer source; // the repository source
 * String[] synthCommands;
 * // Commands to synthesize your app
 * 
 * CodePipeline pipeline = CodePipeline.Builder.create(this, "Pipeline")
 *         // Standard CodePipeline properties...
 *         .synth(ShellStep.Builder.create("Synth")
 *                 .input(source)
 *                 .commands(synthCommands)
 *                 .build())
 * 
 *         // Configure CodeBuild to use a drop-in Docker replacement.
 *         .codeBuildDefaults(CodeBuildOptions.builder()
 *                 .buildEnvironment(BuildEnvironment.builder()
 *                         // Provide a custom build image containing your toolchain and the
 *                         // pre-installed replacement for the `docker` command.
 *                         .buildImage(LinuxBuildImage.fromDockerRegistry("your-docker-registry"))
 *                         .environmentVariables(Map.of(
 *                                 // If you haven't provided an `ENV` in your Dockerfile that overrides
 *                                 // `CDK_DOCKER`, then you must provide the name of the command that
 *                                 // the AWS CDK should run instead of `docker` here.
 *                                 "CDK_DOCKER", BuildEnvironmentVariable.builder().value("drop-in-replacement").build()))
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Known Issues</h2>
 * <p>
 * There are some usability issues that are caused by underlying technology, and
 * cannot be remedied by CDK at this point. They are reproduced here for completeness.
 * <p>
 * <ul>
 * <li><strong>Console links to other accounts will not work</strong>: the AWS CodePipeline
 * console will assume all links are relative to the current account. You will
 * not be able to use the pipeline console to click through to a CloudFormation
 * stack in a different account.</li>
 * <li><strong>If a change set failed to apply the pipeline must be restarted</strong>: if a change
 * set failed to apply, it cannot be retried. The pipeline must be restarted from
 * the top by clicking <strong>Release Change</strong>.</li>
 * <li><strong>A stack that failed to create must be deleted manually</strong>: if a stack
 * failed to create on the first attempt, you must delete it using the
 * CloudFormation console before starting the pipeline again by clicking
 * <strong>Release Change</strong>.</li>
 * </ul>
 */
package software.amazon.awscdk.pipelines;
