/**
 * <h1>AWS Lambda Event Sources</h1>
 * <p>
 * An event source mapping is an AWS Lambda resource that reads from an event source and invokes a Lambda function.
 * You can use event source mappings to process items from a stream or queue in services that don't invoke Lambda
 * functions directly. Lambda provides event source mappings for the following services. Read more about lambda
 * event sources <a href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html">here</a>.
 * <p>
 * This module includes classes that allow using various AWS services as event
 * sources for AWS Lambda via the high-level <code>lambda.addEventSource(source)</code> API.
 * <p>
 * NOTE: In most cases, it is also possible to use the resource APIs to invoke an
 * AWS Lambda function. This library provides a uniform API for all Lambda event
 * sources regardless of the underlying mechanism they use.
 * <p>
 * The following code sets up a lambda function with an SQS queue event source -
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.SqsEventSource;
 * 
 * Function fn;
 * 
 * Queue queue = new Queue(this, "MyQueue");
 * SqsEventSource eventSource = new SqsEventSource(queue);
 * fn.addEventSource(eventSource);
 * 
 * String eventSourceId = eventSource.getEventSourceMappingId();
 * String eventSourceMappingArn = eventSource.getEventSourceMappingArn();
 * </pre></blockquote>
 * <p>
 * The <code>eventSourceId</code> property contains the event source id. This will be a
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/tokens.html">token</a> that will resolve to the final value at the time of
 * deployment.
 * <p>
 * The <code>eventSourceMappingArn</code> property contains the event source mapping ARN. This will be a
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/tokens.html">token</a> that will resolve to the final value at the time of
 * deployment.
 * <p>
 * <h2>SQS</h2>
 * <p>
 * Amazon Simple Queue Service (Amazon SQS) allows you to build asynchronous
 * workflows. For more information about Amazon SQS, see Amazon Simple Queue
 * Service. You can configure AWS Lambda to poll for these messages as they arrive
 * and then pass the event to a Lambda function invocation. To view a sample event,
 * see <a href="https://docs.aws.amazon.com/lambda/latest/dg/eventsources.html#eventsources-sqs">Amazon SQS Event</a>.
 * <p>
 * To set up Amazon Simple Queue Service as an event source for AWS Lambda, you
 * first create or update an Amazon SQS queue and select custom values for the
 * queue parameters. The following parameters will impact Amazon SQS's polling
 * behavior:
 * <p>
 * <ul>
 * <li><strong>visibilityTimeout</strong>: May impact the period between retries.</li>
 * <li><strong>batchSize</strong>: Determines how many records are buffered before invoking your lambda function.</li>
 * <li><strong>maxBatchingWindow</strong>: The maximum amount of time to gather records before invoking the lambda. This increases the likelihood of a full batch at the cost of delayed processing.</li>
 * <li><strong>maxConcurrency</strong>: The maximum concurrency setting limits the number of concurrent instances of the function that an Amazon SQS event source can invoke.</li>
 * <li><strong>enabled</strong>: If the SQS event source mapping should be enabled. The default is true.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.SqsEventSource;
 * Function fn;
 * 
 * 
 * Queue queue = Queue.Builder.create(this, "MyQueue")
 *         .visibilityTimeout(Duration.seconds(30))
 *         .build();
 * 
 * fn.addEventSource(SqsEventSource.Builder.create(queue)
 *         .batchSize(10) // default
 *         .maxBatchingWindow(Duration.minutes(5))
 *         .reportBatchItemFailures(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>S3</h2>
 * <p>
 * You can write Lambda functions to process S3 bucket events, such as the
 * object-created or object-deleted events. For example, when a user uploads a
 * photo to a bucket, you might want Amazon S3 to invoke your Lambda function so
 * that it reads the image and creates a thumbnail for the photo.
 * <p>
 * You can use the bucket notification configuration feature in Amazon S3 to
 * configure the event source mapping, identifying the bucket events that you want
 * Amazon S3 to publish and which Lambda function to invoke.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.services.lambda.eventsources.S3EventSource;
 * Function fn;
 * 
 * 
 * Bucket bucket = new Bucket(this, "mybucket");
 * 
 * fn.addEventSource(S3EventSource.Builder.create(bucket)
 *         .events(List.of(EventType.OBJECT_CREATED, EventType.OBJECT_REMOVED))
 *         .filters(List.of(NotificationKeyFilter.builder().prefix("subdir/").build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * In the example above, <code>S3EventSource</code> is accepting <code>Bucket</code> type as parameter.
 * However, Functions like <code>from_bucket_name</code> and <code>from_bucket_arn</code> will return <code>IBucket</code>
 * and is not compliant with <code>S3EventSource</code>. If this is the case, please consider using
 * <code>S3EventSourceV2</code> instead, this class accepts <code>IBucket</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.services.lambda.eventsources.S3EventSourceV2;
 * Function fn;
 * 
 * 
 * IBucket bucket = Bucket.fromBucketName(this, "Bucket", "amzn-s3-demo-bucket");
 * 
 * fn.addEventSource(S3EventSourceV2.Builder.create(bucket)
 *         .events(List.of(EventType.OBJECT_CREATED, EventType.OBJECT_REMOVED))
 *         .filters(List.of(NotificationKeyFilter.builder().prefix("subdir/").build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>SNS</h2>
 * <p>
 * You can write Lambda functions to process Amazon Simple Notification Service
 * notifications. When a message is published to an Amazon SNS topic, the service
 * can invoke your Lambda function by passing the message payload as a parameter.
 * Your Lambda function code can then process the event, for example publish the
 * message to other Amazon SNS topics, or send the message to other AWS services.
 * <p>
 * This also enables you to trigger a Lambda function in response to Amazon
 * CloudWatch alarms and other AWS services that use Amazon SNS.
 * <p>
 * For an example event, see <a href="https://docs.aws.amazon.com/sns/latest/dg/json-formats.html">Appendix: Message and JSON
 * Formats</a> and
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/eventsources.html#eventsources-sns">Amazon SNS Sample
 * Event</a>.
 * For an example use case, see <a href="https://docs.aws.amazon.com/lambda/latest/dg/with-sns.html">Using AWS Lambda with Amazon SNS from Different
 * Accounts</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.sns.*;
 * import software.amazon.awscdk.services.lambda.eventsources.SnsEventSource;
 * 
 * Topic topic;
 * 
 * Function fn;
 * 
 * Queue deadLetterQueue = new Queue(this, "deadLetterQueue");
 * fn.addEventSource(SnsEventSource.Builder.create(topic)
 *         .filterPolicy(Map.of())
 *         .deadLetterQueue(deadLetterQueue)
 *         .build());
 * </pre></blockquote>
 * <p>
 * When a user calls the SNS Publish API on a topic that your Lambda function is
 * subscribed to, Amazon SNS will call Lambda to invoke your function
 * asynchronously. Lambda will then return a delivery status. If there was an error
 * calling Lambda, Amazon SNS will retry invoking the Lambda function up to three
 * times. After three tries, if Amazon SNS still could not successfully invoke the
 * Lambda function, then Amazon SNS will send a delivery status failure message to
 * CloudWatch.
 * <p>
 * <h2>DynamoDB Streams</h2>
 * <p>
 * You can write Lambda functions to process change events from a DynamoDB Table. An event is emitted to a DynamoDB stream (if configured) whenever a write (Put, Delete, Update)
 * operation is performed against the table. See <a href="https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html">Using AWS Lambda with Amazon DynamoDB</a> for more information about configuring Lambda function event sources with DynamoDB.
 * <p>
 * To process events with a Lambda function, first create or update a DynamoDB table and enable a <code>stream</code> specification. Then, create a <code>DynamoEventSource</code>
 * and add it to your Lambda function. The following parameters will impact Amazon DynamoDB's polling behavior:
 * <p>
 * <ul>
 * <li><strong>batchSize</strong>: Determines how many records are buffered before invoking your lambda function - could impact your function's memory usage (if too high) and ability to keep up with incoming data velocity (if too low).</li>
 * <li><strong>bisectBatchOnError</strong>: If a batch encounters an error, this will cause the batch to be split in two and have each new smaller batch retried, allowing the records in error to be isolated.</li>
 * <li><strong>reportBatchItemFailures</strong>: Allow functions to return partially successful responses for a batch of records.</li>
 * <li><strong>maxBatchingWindow</strong>: The maximum amount of time to gather records before invoking the lambda. This increases the likelihood of a full batch at the cost of delayed processing.</li>
 * <li><strong>maxRecordAge</strong>: The maximum age of a record that will be sent to the function for processing. Records that exceed the max age will be treated as failures.</li>
 * <li><strong>onFailure</strong>: In the event a record fails after all retries or if the record age has exceeded the configured value, the record will be sent to S3 bucket, SQS queue or SNS topic that is specified here</li>
 * <li><strong>parallelizationFactor</strong>: The number of batches to concurrently process on each shard.</li>
 * <li><strong>retryAttempts</strong>: The maximum number of times a record should be retried in the event of failure.</li>
 * <li><strong>startingPosition</strong>: Will determine where to being consumption, either at the most recent ('LATEST') record or the oldest record ('TRIM_HORIZON'). 'TRIM_HORIZON' will ensure you process all available data, while 'LATEST' will ignore all records that arrived prior to attaching the event source.</li>
 * <li><strong>tumblingWindow</strong>: The duration in seconds of a processing window when using streams.</li>
 * <li><strong>enabled</strong>: If the DynamoDB Streams event source mapping should be enabled. The default is true.</li>
 * <li><strong>filters</strong>: Filters to apply before sending a change event from a DynamoDB table to a Lambda function. Events that are filtered out are not sent to the Lambda function.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.dynamodb.*;
 * import software.amazon.awscdk.services.lambda.eventsources.DynamoEventSource;
 * import software.amazon.awscdk.services.lambda.eventsources.SqsDlq;
 * 
 * Table table;
 * 
 * Function fn;
 * 
 * 
 * Queue deadLetterQueue = new Queue(this, "deadLetterQueue");
 * fn.addEventSource(DynamoEventSource.Builder.create(table)
 *         .startingPosition(StartingPosition.TRIM_HORIZON)
 *         .batchSize(5)
 *         .bisectBatchOnError(true)
 *         .onFailure(new SqsDlq(deadLetterQueue))
 *         .retryAttempts(10)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The following code sets up a Lambda function with a DynamoDB event source. A filter is applied to only send DynamoDB events to
 * the Lambda function when the <code>id</code> column is a boolean that equals <code>true</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.dynamodb.*;
 * import software.amazon.awscdk.services.lambda.eventsources.DynamoEventSource;
 * 
 * Table table;
 * 
 * Function fn;
 * 
 * fn.addEventSource(DynamoEventSource.Builder.create(table)
 *         .startingPosition(StartingPosition.LATEST)
 *         .filters(List.of(FilterCriteria.filter(Map.of(
 *                 "eventName", FilterRule.isEqual("INSERT"),
 *                 "dynamodb", Map.of(
 *                         "NewImage", Map.of(
 *                                 "id", Map.of("BOOL", FilterRule.isEqual(true))))))))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Kinesis</h2>
 * <p>
 * You can write Lambda functions to process streaming data in Amazon Kinesis Streams. For more information about Amazon Kinesis, see <a href="https://aws.amazon.com/kinesis/data-streams/">Amazon Kinesis
 * Service</a>. To learn more about configuring Lambda function event sources with kinesis and view a sample event,
 * see <a href="https://docs.aws.amazon.com/lambda/latest/dg/with-kinesis.html">Amazon Kinesis Event</a>.
 * <p>
 * To set up Amazon Kinesis as an event source for AWS Lambda, you
 * first create or update an Amazon Kinesis stream and select custom values for the
 * event source parameters. The following parameters will impact Amazon Kinesis's polling
 * behavior:
 * <p>
 * <ul>
 * <li><strong>batchSize</strong>: Determines how many records are buffered before invoking your lambda function - could impact your function's memory usage (if too high) and ability to keep up with incoming data velocity (if too low).</li>
 * <li><strong>bisectBatchOnError</strong>: If a batch encounters an error, this will cause the batch to be split in two and have each new smaller batch retried, allowing the records in error to be isolated.</li>
 * <li><strong>reportBatchItemFailures</strong>: Allow functions to return partially successful responses for a batch of records.</li>
 * <li><strong>maxBatchingWindow</strong>: The maximum amount of time to gather records before invoking the lambda. This increases the likelihood of a full batch at the cost of possibly delaying processing.</li>
 * <li><strong>maxRecordAge</strong>: The maximum age of a record that will be sent to the function for processing. Records that exceed the max age will be treated as failures.</li>
 * <li><strong>onFailure</strong>: In the event a record fails and consumes all retries, the record will be sent to S3 bucket, SQS queue or SNS topic that is specified here</li>
 * <li><strong>parallelizationFactor</strong>: The number of batches to concurrently process on each shard.</li>
 * <li><strong>retryAttempts</strong>: The maximum number of times a record should be retried in the event of failure.</li>
 * <li><strong>startingPosition</strong>: Will determine where to begin consumption. 'LATEST' will start at the most recent record and ignore all records that arrived prior to attaching the event source, 'TRIM_HORIZON' will start at the oldest record and ensure you process all available data, while 'AT_TIMESTAMP' will start reading records from a specified time stamp. Note that 'AT_TIMESTAMP' is only supported for Amazon Kinesis streams.</li>
 * <li><strong>startingPositionTimestamp</strong>: The time stamp from which to start reading. Used in conjunction with <strong>startingPosition</strong> when set to 'AT_TIMESTAMP'.</li>
 * <li><strong>tumblingWindow</strong>: The duration in seconds of a processing window when using streams.</li>
 * <li><strong>enabled</strong>: If the event source mapping should be enabled. The default is true.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kinesis.*;
 * import software.amazon.awscdk.services.lambda.eventsources.KinesisEventSource;
 * 
 * Function myFunction;
 * 
 * 
 * Stream stream = new Stream(this, "MyStream");
 * myFunction.addEventSource(KinesisEventSource.Builder.create(stream)
 *         .batchSize(100) // default
 *         .startingPosition(StartingPosition.TRIM_HORIZON)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To use a dedicated-throughput consumer with enhanced fan-out
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kinesis.*;
 * import software.amazon.awscdk.services.lambda.eventsources.KinesisConsumerEventSource;
 * 
 * Function myFunction;
 * 
 * 
 * Stream stream = new Stream(this, "MyStream");
 * StreamConsumer streamConsumer = StreamConsumer.Builder.create(this, "MyStreamConsumer")
 *         .stream(stream)
 *         .streamConsumerName("MyStreamConsumer")
 *         .build();
 * myFunction.addEventSource(KinesisConsumerEventSource.Builder.create(streamConsumer)
 *         .batchSize(100) // default
 *         .startingPosition(StartingPosition.TRIM_HORIZON)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Kafka</h2>
 * <p>
 * You can write Lambda functions to process data either from <a href="https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html">Amazon MSK</a> or a <a href="https://docs.aws.amazon.com/lambda/latest/dg/kafka-smaa.html">self managed Kafka</a> cluster.
 * <p>
 * The following code sets up Amazon MSK as an event source for a lambda function. Credentials will need to be configured to access the
 * MSK cluster, as described in <a href="https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html">Username/Password authentication</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.secretsmanager.Secret;
 * import software.amazon.awscdk.services.lambda.eventsources.ManagedKafkaEventSource;
 * 
 * Function myFunction;
 * 
 * 
 * // Your MSK cluster arn
 * String clusterArn = "arn:aws:kafka:us-east-1:0123456789019:cluster/SalesCluster/abcd1234-abcd-cafe-abab-9876543210ab-4";
 * 
 * // The Kafka topic you want to subscribe to
 * String topic = "some-cool-topic";
 * 
 * // The secret that allows access to your MSK cluster
 * // You still have to make sure that it is associated with your cluster as described in the documentation
 * Secret secret = Secret.Builder.create(this, "Secret").secretName("AmazonMSK_KafkaSecret").build();
 * myFunction.addEventSource(ManagedKafkaEventSource.Builder.create()
 *         .clusterArn(clusterArn)
 *         .topic(topic)
 *         .secret(secret)
 *         .batchSize(100) // default
 *         .startingPosition(StartingPosition.TRIM_HORIZON)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The following code sets up a self managed Kafka cluster as an event source. Username and password based authentication
 * will need to be set up as described in <a href="https://docs.aws.amazon.com/lambda/latest/dg/smaa-permissions.html#smaa-permissions-add-secret">Managing access and permissions</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.secretsmanager.Secret;
 * import software.amazon.awscdk.services.lambda.eventsources.SelfManagedKafkaEventSource;
 * 
 * // The secret that allows access to your self hosted Kafka cluster
 * Secret secret;
 * 
 * Function myFunction;
 * 
 * 
 * // The list of Kafka brokers
 * String[] bootstrapServers = List.of("kafka-broker:9092");
 * 
 * // The Kafka topic you want to subscribe to
 * String topic = "some-cool-topic";
 * 
 * // (Optional) The consumer group id to use when connecting to the Kafka broker. If omitted the UUID of the event source mapping will be used.
 * String consumerGroupId = "my-consumer-group-id";
 * myFunction.addEventSource(SelfManagedKafkaEventSource.Builder.create()
 *         .bootstrapServers(bootstrapServers)
 *         .topic(topic)
 *         .consumerGroupId(consumerGroupId)
 *         .secret(secret)
 *         .batchSize(100) // default
 *         .startingPosition(StartingPosition.TRIM_HORIZON)
 *         .build());
 * </pre></blockquote>
 * <p>
 * If your self managed Kafka cluster is only reachable via VPC also configure <code>vpc</code> <code>vpcSubnets</code> and <code>securityGroup</code>.
 * <p>
 * You can specify <a href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html#filtering-msk-smak">event filtering</a>
 * for managed and self managed Kafka clusters using the <code>filters</code> property:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.ManagedKafkaEventSource;
 * 
 * Function myFunction;
 * 
 * 
 * // Your MSK cluster arn
 * String clusterArn = "arn:aws:kafka:us-east-1:0123456789019:cluster/SalesCluster/abcd1234-abcd-cafe-abab-9876543210ab-4";
 * 
 * // The Kafka topic you want to subscribe to
 * String topic = "some-cool-topic";
 * myFunction.addEventSource(ManagedKafkaEventSource.Builder.create()
 *         .clusterArn(clusterArn)
 *         .topic(topic)
 *         .startingPosition(StartingPosition.TRIM_HORIZON)
 *         .filters(List.of(FilterCriteria.filter(Map.of(
 *                 "stringEquals", FilterRule.isEqual("test")))))
 *         .build());
 * </pre></blockquote>
 * <p>
 * By default, Lambda will encrypt Filter Criteria using AWS managed keys. But if you want to use a self managed KMS key to encrypt the filters, You can specify the self managed key using the <code>filterEncryption</code> property.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.ManagedKafkaEventSource;
 * import software.amazon.awscdk.services.kms.Key;
 * 
 * Function myFunction;
 * 
 * 
 * // Your MSK cluster arn
 * String clusterArn = "arn:aws:kafka:us-east-1:0123456789019:cluster/SalesCluster/abcd1234-abcd-cafe-abab-9876543210ab-4";
 * 
 * // The Kafka topic you want to subscribe to
 * String topic = "some-cool-topic";
 * 
 * // Your self managed KMS key
 * IKey myKey = Key.fromKeyArn(this, "SourceBucketEncryptionKey", "arn:aws:kms:us-east-1:123456789012:key/&lt;key-id&gt;");
 * myFunction.addEventSource(ManagedKafkaEventSource.Builder.create()
 *         .clusterArn(clusterArn)
 *         .topic(topic)
 *         .startingPosition(StartingPosition.TRIM_HORIZON)
 *         .filters(List.of(FilterCriteria.filter(Map.of(
 *                 "stringEquals", FilterRule.isEqual("test")))))
 *         .filterEncryption(myKey)
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also specify an S3 bucket as an "on failure" destination:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.ManagedKafkaEventSource;
 * import software.amazon.awscdk.services.lambda.eventsources.S3OnFailureDestination;
 * import software.amazon.awscdk.services.s3.IBucket;
 * 
 * IBucket bucket;
 * Function myFunction;
 * 
 * 
 * // Your MSK cluster arn
 * String clusterArn = "arn:aws:kafka:us-east-1:0123456789019:cluster/SalesCluster/abcd1234-abcd-cafe-abab-9876543210ab-4";
 * 
 * // The Kafka topic you want to subscribe to
 * String topic = "some-cool-topic";
 * 
 * S3OnFailureDestination s3OnFailureDestination = new S3OnFailureDestination(bucket);
 * 
 * myFunction.addEventSource(ManagedKafkaEventSource.Builder.create()
 *         .clusterArn(clusterArn)
 *         .topic(topic)
 *         .startingPosition(StartingPosition.TRIM_HORIZON)
 *         .onFailure(s3OnFailureDestination)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Set configuration for provisioned pollers that read from the event source.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.ManagedKafkaEventSource;
 * 
 * // Your MSK cluster arn
 * String clusterArn;
 * 
 * Function myFunction;
 * 
 * 
 * // The Kafka topic you want to subscribe to
 * String topic = "some-cool-topic";
 * myFunction.addEventSource(ManagedKafkaEventSource.Builder.create()
 *         .clusterArn(clusterArn)
 *         .topic(topic)
 *         .startingPosition(StartingPosition.TRIM_HORIZON)
 *         .provisionedPollerConfig(ProvisionedPollerConfig.builder()
 *                 .minimumPollers(1)
 *                 .maximumPollers(3)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Roadmap</h2>
 * <p>
 * Eventually, this module will support all the event sources described under
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/invoking-lambda-function.html">Supported Event
 * Sources</a>
 * in the AWS Lambda Developer Guide.
 */
package software.amazon.awscdk.services.lambda.eventsources;
