package software.amazon.awscdk;

/**
 * Possible values for a resource's Removal Policy.
 * <p>
 * The removal policy controls what happens to the resource if it stops being
 * managed by CloudFormation. This can happen in one of three situations:
 * <p>
 * <ul>
 * <li>The resource is removed from the template, so CloudFormation stops managing it;</li>
 * <li>A change to the resource is made that requires it to be replaced, so CloudFormation stops
 * managing it;</li>
 * <li>The stack is deleted, so CloudFormation stops managing all resources in it.</li>
 * </ul>
 * <p>
 * The Removal Policy applies to all above cases.
 * <p>
 * Many stateful resources in the AWS Construct Library will accept a
 * <code>removalPolicy</code> as a property, typically defaulting it to <code>RETAIN</code>.
 * <p>
 * If the AWS Construct Library resource does not accept a <code>removalPolicy</code>
 * argument, you can always configure it by using the escape hatch mechanism,
 * as shown in the following example:
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * CfnResource cfnBucket = (CfnResource)bucket.node.findChild("Resource");
 * cfnBucket.applyRemovalPolicy(RemovalPolicy.DESTROY);
 * </pre></blockquote>
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Role myRole;
 * AwsCustomResource.Builder.create(this, "Customized")
 *         .role(myRole) // must be assumable by the `lambda.amazonaws.com` service principal
 *         .timeout(Duration.minutes(10)) // defaults to 2 minutes
 *         .memorySize(1025) // defaults to 512 if installLatestAwsSdk is true
 *         .logGroup(LogGroup.Builder.create(this, "AwsCustomResourceLogs")
 *                 .retention(RetentionDays.ONE_DAY)
 *                 .build())
 *         .functionName("my-custom-name") // defaults to a CloudFormation generated name
 *         .removalPolicy(RemovalPolicy.RETAIN) // defaults to `RemovalPolicy.DESTROY`
 *         .policy(AwsCustomResourcePolicy.fromSdkCalls(SdkCallsPolicyOptions.builder()
 *                 .resources(AwsCustomResourcePolicy.ANY_RESOURCE)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 */
@javax.annotation.Generated(value = "jsii-pacmak/1.110.0 (build 336b265)", date = "2025-04-14T15:41:11.581Z")
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
@software.amazon.jsii.Jsii(module = software.amazon.awscdk.$Module.class, fqn = "aws-cdk-lib.RemovalPolicy")
public enum RemovalPolicy {
    /**
     * This is the default removal policy.
     * <p>
     * It means that when the resource is
     * removed from the app, it will be physically destroyed.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    DESTROY,
    /**
     * This uses the 'Retain' DeletionPolicy, which will cause the resource to be retained in the account, but orphaned from the stack.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    RETAIN,
    /**
     * This retention policy deletes the resource, but saves a snapshot of its data before deleting, so that it can be re-created later.
     * <p>
     * Only available for some stateful resources,
     * like databases, EC2 volumes, etc.
     * <p>
     * @see <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options">https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options</a>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    SNAPSHOT,
    /**
     * Resource will be retained when they are requested to be deleted during a stack delete request or need to be replaced due to a stack update request.
     * <p>
     * Resource are not retained, if the creation is rolled back.
     * <p>
     * The result is that new, empty, and unused resources are deleted,
     * while in-use resources and their data are retained.
     * <p>
     * This uses the 'RetainExceptOnCreate' DeletionPolicy,
     * and the 'Retain' UpdateReplacePolicy, when <code>applyToUpdateReplacePolicy</code> is set.
     * <p>
     * @see <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options">https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options</a>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    RETAIN_ON_UPDATE_OR_DELETE,
}
