/**
 * <h1>AWS Identity and Access Management Construct Library</h1>
 * <p>
 * <h2>Security and Safety Dev Guide</h2>
 * <p>
 * For a detailed guide on CDK security and safety please see the <a href="https://github.com/aws/aws-cdk/wiki/Security-And-Safety-Dev-Guide">CDK Security And
 * Safety Dev Guide</a>
 * <p>
 * The guide will cover topics like:
 * <p>
 * <ul>
 * <li>What permissions to extend to CDK deployments</li>
 * <li>How to control the permissions of CDK deployments via IAM identities and policies</li>
 * <li>How to use CDK to configure the IAM identities and policies of deployed applications</li>
 * <li>Using Permissions Boundaries with CDK</li>
 * </ul>
 * <p>
 * <h2>Overview</h2>
 * <p>
 * Define a role and add permissions to it. This will automatically create and
 * attach an IAM policy to the role:
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "MyRole")
 *         .assumedBy(new ServicePrincipal("sns.amazonaws.com"))
 *         .build();
 * 
 * role.addToPolicy(PolicyStatement.Builder.create()
 *         .resources(List.of("*"))
 *         .actions(List.of("lambda:InvokeFunction"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Define a policy and attach it to groups, users and roles. Note that it is possible to attach
 * the policy either by calling <code>xxx.attachInlinePolicy(policy)</code> or <code>policy.attachToXxx(xxx)</code>.
 * <p>
 * <blockquote><pre>
 * User user = User.Builder.create(this, "MyUser").password(SecretValue.plainText("1234")).build();
 * Group group = new Group(this, "MyGroup");
 * 
 * Policy policy = new Policy(this, "MyPolicy");
 * policy.attachToUser(user);
 * group.attachInlinePolicy(policy);
 * </pre></blockquote>
 * <p>
 * Managed policies can be attached using <code>xxx.addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName(policyName))</code>:
 * <p>
 * <blockquote><pre>
 * Group group = new Group(this, "MyGroup");
 * group.addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName("AdministratorAccess"));
 * </pre></blockquote>
 * <p>
 * <h2>Granting permissions to resources</h2>
 * <p>
 * Many of the AWS CDK resources have <code>grant*</code> methods that allow you to grant other resources access to that resource. As an example, the following code gives a Lambda function write permissions (Put, Update, Delete) to a DynamoDB table.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * Table table;
 * 
 * 
 * table.grantWriteData(fn);
 * </pre></blockquote>
 * <p>
 * The more generic <code>grant</code> method allows you to give specific permissions to a resource:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * Table table;
 * 
 * 
 * table.grant(fn, "dynamodb:PutItem");
 * </pre></blockquote>
 * <p>
 * The <code>grant*</code> methods accept an <code>IGrantable</code> object. This interface is implemented by IAM principal resources (groups, users and roles), policies, managed policies and resources that assume a role such as a Lambda function, EC2 instance or a Codebuild project.
 * <p>
 * You can find which <code>grant*</code> methods exist for a resource in the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-construct-library.html">AWS CDK API Reference</a>.
 * <p>
 * <h2>Roles</h2>
 * <p>
 * Many AWS resources require <em>Roles</em> to operate. These Roles define the AWS API
 * calls an instance or other AWS service is allowed to make.
 * <p>
 * Creating Roles and populating them with the right permissions <em>Statements</em> is
 * a necessary but tedious part of setting up AWS infrastructure. In order to
 * help you focus on your business logic, CDK will take care of creating
 * roles and populating them with least-privilege permissions automatically.
 * <p>
 * All constructs that require Roles will create one for you if don't specify
 * one at construction time. Permissions will be added to that role
 * automatically if you associate the construct with other constructs from the
 * AWS Construct Library (for example, if you tell an <em>AWS CodePipeline</em> to trigger
 * an <em>AWS Lambda Function</em>, the Pipeline's Role will automatically get
 * <code>lambda:InvokeFunction</code> permissions on that particular Lambda Function),
 * or if you explicitly grant permissions using <code>grant</code> functions (see the
 * previous section).
 * <p>
 * <h3>Opting out of automatic permissions management</h3>
 * <p>
 * You may prefer to manage a Role's permissions yourself instead of having the
 * CDK automatically manage them for you. This may happen in one of the
 * following cases:
 * <p>
 * <ul>
 * <li>You don't like the permissions that CDK automatically generates and
 * want to substitute your own set.</li>
 * <li>The least-permissions policy that the CDK generates is becoming too
 * big for IAM to store, and you need to add some wildcards to keep the
 * policy size down.</li>
 * </ul>
 * <p>
 * To prevent constructs from updating your Role's policy, pass the object
 * returned by <code>myRole.withoutPolicyUpdates()</code> instead of <code>myRole</code> itself.
 * <p>
 * For example, to have an AWS CodePipeline <em>not</em> automatically add the required
 * permissions to trigger the expected targets, do the following:
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("codepipeline.amazonaws.com"))
 *         // custom description if desired
 *         .description("This is a custom role...")
 *         .build();
 * 
 * Pipeline.Builder.create(this, "Pipeline")
 *         // Give the Pipeline an immutable view of the Role
 *         .role(role.withoutPolicyUpdates())
 *         .build();
 * 
 * // You now have to manage the Role policies yourself
 * role.addToPolicy(PolicyStatement.Builder.create()
 *         .actions(List.of())
 *         .resources(List.of())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Using existing roles</h3>
 * <p>
 * If there are Roles in your account that have already been created which you
 * would like to use in your CDK application, you can use <code>Role.fromRoleArn</code> to
 * import them, as follows:
 * <p>
 * <blockquote><pre>
 * IRole role = Role.fromRoleArn(this, "Role", "arn:aws:iam::123456789012:role/MyExistingRole", FromRoleArnOptions.builder()
 *         // Set 'mutable' to 'false' to use the role as-is and prevent adding new
 *         // policies to it. The default is 'true', which means the role may be
 *         // modified as part of the deployment.
 *         .mutable(false)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Customizing role creation</h3>
 * <p>
 * It is best practice to allow CDK to manage IAM roles and permissions. You can prevent CDK from
 * creating roles by using the <code>customizeRoles</code> method for special cases. One such case is using CDK in
 * an environment where role creation is not allowed or needs to be managed through a process outside
 * of the CDK application.
 * <p>
 * An example of how to opt in to this behavior is below:
 * <p>
 * <blockquote><pre>
 * Stack stack;
 * 
 * Role.customizeRoles(stack);
 * </pre></blockquote>
 * <p>
 * CDK will not create any IAM roles or policies with the <code>stack</code> scope. <code>cdk synth</code> will fail and
 * it will generate a policy report to the cloud assembly (i.e. cdk.out). The <code>iam-policy-report.txt</code>
 * report will contain a list of IAM roles and associated permissions that would have been created.
 * This report can be used to create the roles with the appropriate permissions outside of
 * the CDK application.
 * <p>
 * Once the missing roles have been created, their names can be added to the <code>usePrecreatedRoles</code>
 * property, like shown below:
 * <p>
 * <blockquote><pre>
 * App app;
 * 
 * Stack stack = new Stack(app, "MyStack");
 * Role.customizeRoles(this, CustomizeRolesOptions.builder()
 *         .usePrecreatedRoles(Map.of(
 *                 "MyStack/MyRole", "my-precreated-role-name"))
 *         .build());
 * 
 * Role.Builder.create(this, "MyRole")
 *         .assumedBy(new ServicePrincipal("sns.amazonaws.com"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If any IAM policies reference deploy time values (i.e. ARN of a resource that hasn't been created
 * yet) you will have to modify the generated report to be more generic. For example, given the
 * following CDK code:
 * <p>
 * <blockquote><pre>
 * App app;
 * 
 * Stack stack = new Stack(app, "MyStack");
 * Role.customizeRoles(stack);
 * 
 * Function fn = Function.Builder.create(this, "MyLambda")
 *         .code(new InlineCode("foo"))
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .build();
 * 
 * Bucket bucket = new Bucket(this, "Bucket");
 * bucket.grantRead(fn);
 * </pre></blockquote>
 * <p>
 * The following report will be generated.
 * <p>
 * <blockquote><pre>
 * &lt;missing role&gt; (MyStack/MyLambda/ServiceRole)
 * 
 * AssumeRole Policy:
 * [
 *   {
 *     "Action": "sts:AssumeRole",
 *     "Effect": "Allow",
 *     "Principal": {
 *       "Service": "lambda.amazonaws.com"
 *     }
 *   }
 * ]
 * 
 * Managed Policy ARNs:
 * [
 *   "arn:(PARTITION):iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
 * ]
 * 
 * Managed Policies Statements:
 * NONE
 * 
 * Identity Policy Statements:
 * [
 *   {
 *     "Action": [
 *       "s3:GetObject*",
 *       "s3:GetBucket*",
 *       "s3:List*"
 *     ],
 *     "Effect": "Allow",
 *     "Resource": [
 *       "(MyStack/Bucket/Resource.Arn)",
 *       "(MyStack/Bucket/Resource.Arn)/*"
 *     ]
 *   }
 * ]
 * </pre></blockquote>
 * <p>
 * You would then need to create the role with the inline &amp; managed policies in the report and then
 * come back and update the <code>customizeRoles</code> with the role name.
 * <p>
 * <blockquote><pre>
 * App app;
 * 
 * Stack stack = new Stack(app, "MyStack");
 * Role.customizeRoles(this, CustomizeRolesOptions.builder()
 *         .usePrecreatedRoles(Map.of(
 *                 "MyStack/MyLambda/ServiceRole", "my-role-name"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * For more information on configuring permissions see the <a href="https://github.com/aws/aws-cdk/wiki/Security-And-Safety-Dev-Guide">Security And Safety Dev
 * Guide</a>
 * <p>
 * <h4>Policy report generation</h4>
 * <p>
 * When <code>customizeRoles</code> is used, the <code>iam-policy-report.txt</code> report will contain a list
 * of IAM roles and associated permissions that would have been created. This report is
 * generated in an attempt to resolve and replace any references with a more user-friendly
 * value.
 * <p>
 * The following are some examples of the value that will appear in the report:
 * <p>
 * <blockquote><pre>
 * "Resource": {
 *   "Fn::Join": [
 *     "",
 *     [
 *       "arn:",
 *       {
 *         "Ref": "AWS::Partition"
 *       },
 *       ":iam::",
 *       {
 *         "Ref": "AWS::AccountId"
 *       },
 *       ":role/Role"
 *     ]
 *   ]
 * }
 * </pre></blockquote>
 * <p>
 * The policy report will instead get:
 * <p>
 * <blockquote><pre>
 * "Resource": "arn:(PARTITION):iam::(ACCOUNT):role/Role"
 * </pre></blockquote>
 * <p>
 * If IAM policy is referencing a resource attribute:
 * <p>
 * <blockquote><pre>
 * "Resource": [
 *   {
 *     "Fn::GetAtt": [
 *       "SomeResource",
 *       "Arn"
 *     ]
 *   },
 *   {
 *     "Ref": "AWS::NoValue",
 *   }
 * ]
 * </pre></blockquote>
 * <p>
 * The policy report will instead get:
 * <p>
 * <blockquote><pre>
 * "Resource": [
 *   "(Path/To/SomeResource.Arn)"
 *   "(NOVALUE)"
 * ]
 * </pre></blockquote>
 * <p>
 * The following pseudo parameters will be converted:
 * <p>
 * <ol>
 * <li><code>{ 'Ref': 'AWS::AccountId' }</code> -&gt; `(ACCOUNT)</li>
 * <li><code>{ 'Ref': 'AWS::Partition' }</code> -&gt; `(PARTITION)</li>
 * <li><code>{ 'Ref': 'AWS::Region' }</code> -&gt; `(REGION)</li>
 * <li><code>{ 'Ref': 'AWS::NoValue' }</code> -&gt; `(NOVALUE)</li>
 * </ol>
 * <p>
 * <h4>Generating a permissions report</h4>
 * <p>
 * It is also possible to generate the report <em>without</em> preventing the role/policy creation.
 * <p>
 * <blockquote><pre>
 * Stack stack;
 * 
 * Role.customizeRoles(this, CustomizeRolesOptions.builder()
 *         .preventSynthesis(false)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Configuring an ExternalId</h2>
 * <p>
 * If you need to create Roles that will be assumed by third parties, it is generally a good idea to <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html">require an <code>ExternalId</code>
 * to assume them</a>.  Configuring
 * an <code>ExternalId</code> works like this:
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "MyRole")
 *         .assumedBy(new AccountPrincipal("123456789012"))
 *         .externalIds(List.of("SUPPLY-ME"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>SourceArn and SourceAccount</h2>
 * <p>
 * If you need to create resource policies using <code>aws:SourceArn</code> and <code>aws:SourceAccount</code> for cross-service resource access,
 * use <code>addSourceArnCondition</code> and <code>addSourceAccountCondition</code> to create the conditions.
 * <p>
 * See <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/confused-deputy.html#cross-service-confused-deputy-prevention">Cross-service confused deputy prevention for more details</a>.
 * <p>
 * <h2>Principals vs Identities</h2>
 * <p>
 * When we say <em>Principal</em>, we mean an entity you grant permissions to. This
 * entity can be an AWS Service, a Role, or something more abstract such as "all
 * users in this account" or even "all users in this organization". An
 * <em>Identity</em> is an IAM representing a single IAM entity that can have
 * a policy attached, one of <code>Role</code>, <code>User</code>, or <code>Group</code>.
 * <p>
 * <h2>IAM Principals</h2>
 * <p>
 * When defining policy statements as part of an AssumeRole policy or as part of a
 * resource policy, statements would usually refer to a specific IAM principal
 * under <code>Principal</code>.
 * <p>
 * IAM principals are modeled as classes that derive from the <code>iam.PolicyPrincipal</code>
 * abstract class. Principal objects include principal type (string) and value
 * (array of string), optional set of conditions and the action that this principal
 * requires when it is used in an assume role policy document.
 * <p>
 * To add a principal to a policy statement you can either use the abstract
 * <code>statement.addPrincipal</code>, one of the concrete <code>addXxxPrincipal</code> methods:
 * <p>
 * <ul>
 * <li><code>addAwsPrincipal</code>, <code>addArnPrincipal</code> or <code>new ArnPrincipal(arn)</code> for <code>{ "AWS": arn }</code></li>
 * <li><code>addAwsAccountPrincipal</code> or <code>new AccountPrincipal(accountId)</code> for <code>{ "AWS": account-arn }</code></li>
 * <li><code>addServicePrincipal</code> or <code>new ServicePrincipal(service)</code> for <code>{ "Service": service }</code></li>
 * <li><code>addAccountRootPrincipal</code> or <code>new AccountRootPrincipal()</code> for <code>{ "AWS": { "Ref: "AWS::AccountId" } }</code></li>
 * <li><code>addCanonicalUserPrincipal</code> or <code>new CanonicalUserPrincipal(id)</code> for <code>{ "CanonicalUser": id }</code></li>
 * <li><code>addFederatedPrincipal</code> or <code>new FederatedPrincipal(federated, conditions, assumeAction)</code> for
 * <code>{ "Federated": arn }</code> and a set of optional conditions and the assume role action to use.</li>
 * <li><code>addAnyPrincipal</code> or <code>new AnyPrincipal</code> for <code>{ "AWS": "*" }</code></li>
 * </ul>
 * <p>
 * If multiple principals are added to the policy statement, they will be merged together:
 * <p>
 * <blockquote><pre>
 * PolicyStatement statement = new PolicyStatement();
 * statement.addServicePrincipal("cloudwatch.amazonaws.com");
 * statement.addServicePrincipal("ec2.amazonaws.com");
 * statement.addArnPrincipal("arn:aws:boom:boom");
 * </pre></blockquote>
 * <p>
 * Will result in:
 * <p>
 * <blockquote><pre>
 * {
 *   "Principal": {
 *     "Service": [ "cloudwatch.amazonaws.com", "ec2.amazonaws.com" ],
 *     "AWS": "arn:aws:boom:boom"
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * The <code>CompositePrincipal</code> class can also be used to define complex principals, for example:
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "MyRole")
 *         .assumedBy(new CompositePrincipal(
 *         new ServicePrincipal("ec2.amazonaws.com"),
 *         new AccountPrincipal("1818188181818187272")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>PrincipalWithConditions</code> class can be used to add conditions to a
 * principal, especially those that don't take a <code>conditions</code> parameter in their
 * constructor. The <code>principal.withConditions()</code> method can be used to create a
 * <code>PrincipalWithConditions</code> from an existing principal, for example:
 * <p>
 * <blockquote><pre>
 * PrincipalBase principal = new AccountPrincipal("123456789000").withConditions(Map.of("StringEquals", Map.of("foo", "baz")));
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * NOTE: If you need to define an IAM condition that uses a token (such as a
 * deploy-time attribute of another resource) in a JSON map key, use <code>CfnJson</code> to
 * render this condition. See <a href="./test/integ.condition-with-ref.ts">this test</a> for
 * an example.
 * <p>
 * </blockquote>
 * <p>
 * The <code>WebIdentityPrincipal</code> class can be used as a principal for web identities like
 * Cognito, Amazon, Google or Facebook, for example:
 * <p>
 * <blockquote><pre>
 * WebIdentityPrincipal principal = new WebIdentityPrincipal("cognito-identity.amazonaws.com", Map.of(
 *         "StringEquals", Map.of("cognito-identity.amazonaws.com:aud", "us-east-2:12345678-abcd-abcd-abcd-123456"),
 *         "ForAnyValue:StringLike", Map.of("cognito-identity.amazonaws.com:amr", "unauthenticated")));
 * </pre></blockquote>
 * <p>
 * If your identity provider is configured to assume a Role with <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html">session
 * tags</a>, you
 * need to call <code>.withSessionTags()</code> to add the required permissions to the Role's
 * policy document:
 * <p>
 * <blockquote><pre>
 * Role.Builder.create(this, "Role")
 *         .assumedBy(new WebIdentityPrincipal("cognito-identity.amazonaws.com", Map.of(
 *                 "StringEquals", Map.of(
 *                         "cognito-identity.amazonaws.com:aud", "us-east-2:12345678-abcd-abcd-abcd-123456"),
 *                 "ForAnyValue:StringLike", Map.of(
 *                         "cognito-identity.amazonaws.com:amr", "unauthenticated"))).withSessionTags())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Granting a principal permission to assume a role</h3>
 * <p>
 * A principal can be granted permission to assume a role using <code>grantAssumeRole</code>.
 * <p>
 * Note that this does not apply to service principals or account principals as they must be added to the role trust policy via <code>assumeRolePolicy</code>.
 * <p>
 * <blockquote><pre>
 * User user = new User(this, "user");
 * Role role = Role.Builder.create(this, "role")
 *         .assumedBy(new AccountPrincipal(this.account))
 *         .build();
 * 
 * role.grantAssumeRole(user);
 * </pre></blockquote>
 * <p>
 * <h3>Granting service and account principals permission to assume a role</h3>
 * <p>
 * Service principals and account principals can be granted permission to assume a role using <code>assumeRolePolicy</code> which modifies the role trust policy.
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "role")
 *         .assumedBy(new AccountPrincipal(this.account))
 *         .build();
 * 
 * role.assumeRolePolicy.addStatements(PolicyStatement.Builder.create()
 *         .actions(List.of("sts:AssumeRole"))
 *         .principals(List.of(
 *             new AccountPrincipal("123456789"),
 *             new ServicePrincipal("beep-boop.amazonaws.com")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Fixing the synthesized service principle for services that do not follow the IAM Pattern</h3>
 * <p>
 * In some cases, certain AWS services may not use the standard <code>&lt;service&gt;.amazonaws.com</code> pattern for their service principals. For these services, you can define the ServicePrincipal as following where the provided service principle name will be used as is without any changing.
 * <p>
 * <blockquote><pre>
 * ServicePrincipal sp = ServicePrincipal.fromStaticServicePrincipleName("elasticmapreduce.amazonaws.com.cn");
 * </pre></blockquote>
 * <p>
 * This principle can use as normal in defining any role, for example:
 * <p>
 * <blockquote><pre>
 * Role emrServiceRole = Role.Builder.create(this, "EMRServiceRole")
 *         .assumedBy(ServicePrincipal.fromStaticServicePrincipleName("elasticmapreduce.amazonaws.com.cn"))
 *         .managedPolicies(List.of(ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonElasticMapReduceRole")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Parsing JSON Policy Documents</h2>
 * <p>
 * The <code>PolicyDocument.fromJson</code> and <code>PolicyStatement.fromJson</code> static methods can be used to parse JSON objects. For example:
 * <p>
 * <blockquote><pre>
 * Map&lt;String, Object&gt; policyDocument = Map.of(
 *         "Version", "2012-10-17",
 *         "Statement", List.of(Map.of(
 *                 "Sid", "FirstStatement",
 *                 "Effect", "Allow",
 *                 "Action", List.of("iam:ChangePassword"),
 *                 "Resource", List.of("*")), Map.of(
 *                 "Sid", "SecondStatement",
 *                 "Effect", "Allow",
 *                 "Action", List.of("s3:ListAllMyBuckets"),
 *                 "Resource", List.of("*")), Map.of(
 *                 "Sid", "ThirdStatement",
 *                 "Effect", "Allow",
 *                 "Action", List.of("s3:List*", "s3:Get*"),
 *                 "Resource", List.of("arn:aws:s3:::confidential-data", "arn:aws:s3:::confidential-data/*"),
 *                 "Condition", Map.of("Bool", Map.of("aws:MultiFactorAuthPresent", "true")))));
 * 
 * PolicyDocument customPolicyDocument = PolicyDocument.fromJson(policyDocument);
 * 
 * // You can pass this document as an initial document to a ManagedPolicy
 * // or inline Policy.
 * ManagedPolicy newManagedPolicy = ManagedPolicy.Builder.create(this, "MyNewManagedPolicy")
 *         .document(customPolicyDocument)
 *         .build();
 * Policy newPolicy = Policy.Builder.create(this, "MyNewPolicy")
 *         .document(customPolicyDocument)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Permissions Boundaries</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html">Permissions
 * Boundaries</a>
 * can be used as a mechanism to prevent privilege escalation by creating new
 * <code>Role</code>s. Permissions Boundaries are a Managed Policy, attached to Roles or
 * Users, that represent the <em>maximum</em> set of permissions they can have. The
 * effective set of permissions of a Role (or User) will be the intersection of
 * the Identity Policy and the Permissions Boundary attached to the Role (or
 * User). Permissions Boundaries are typically created by account
 * Administrators, and their use on newly created <code>Role</code>s will be enforced by
 * IAM policies.
 * <p>
 * <h3>Bootstrap Permissions Boundary</h3>
 * <p>
 * If a permissions boundary has been enforced as part of CDK bootstrap, all IAM
 * Roles and Users that are created as part of the CDK application must be created
 * with the permissions boundary attached. The most common scenario will be to
 * apply the enforced permissions boundary to the entire CDK app. This can be done
 * either by adding the value to <code>cdk.json</code> or directly in the <code>App</code> constructor.
 * <p>
 * For example if your organization has created and is enforcing a permissions
 * boundary with the name
 * <code>cdk-${Qualifier}-PermissionsBoundary</code>
 * <p>
 * <blockquote><pre>
 * {
 *   "context": {
 *      "&#64;aws-cdk/core:permissionsBoundary": {
 * 	   "name": "cdk-${Qualifier}-PermissionsBoundary"
 * 	 }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * OR
 * <p>
 * <blockquote><pre>
 * App.Builder.create()
 *         .context(Map.of(
 *                 PERMISSIONS_BOUNDARY_CONTEXT_KEY, Map.of(
 *                         "name", "cdk-${Qualifier}-PermissionsBoundary")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Another scenario might be if your organization enforces different permissions
 * boundaries for different environments. For example your CDK application may have
 * <p>
 * <ul>
 * <li><code>DevStage</code> that deploys to a personal dev environment where you have elevated
 * privileges</li>
 * <li><code>BetaStage</code> that deploys to a beta environment which and has a relaxed
 * permissions boundary</li>
 * <li><code>GammaStage</code> that deploys to a gamma environment which has the prod
 * permissions boundary</li>
 * <li><code>ProdStage</code> that deploys to the prod environment and has the prod permissions
 * boundary</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * App app;
 * 
 * 
 * new Stage(app, "DevStage");
 * 
 * Stage.Builder.create(app, "BetaStage")
 *         .permissionsBoundary(PermissionsBoundary.fromName("beta-permissions-boundary"))
 *         .build();
 * 
 * Stage.Builder.create(app, "GammaStage")
 *         .permissionsBoundary(PermissionsBoundary.fromName("prod-permissions-boundary"))
 *         .build();
 * 
 * Stage.Builder.create(app, "ProdStage")
 *         .permissionsBoundary(PermissionsBoundary.fromName("prod-permissions-boundary"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The provided name can include placeholders for the partition, region, qualifier, and account
 * These placeholders will be replaced with the actual values if available. This requires
 * that the Stack has the environment specified, it does not work with environment.
 * <p>
 * <ul>
 * <li>'${AWS::Partition}'</li>
 * <li>'${AWS::Region}'</li>
 * <li>'${AWS::AccountId}'</li>
 * <li>'${Qualifier}'</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * App app;
 * 
 * 
 * Stage prodStage = Stage.Builder.create(app, "ProdStage")
 *         .permissionsBoundary(PermissionsBoundary.fromName("cdk-${Qualifier}-PermissionsBoundary-${AWS::AccountId}-${AWS::Region}"))
 *         .build();
 * 
 * Stack.Builder.create(prodStage, "ProdStack")
 *         .synthesizer(DefaultStackSynthesizer.Builder.create()
 *                 .qualifier("custom")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more information on configuring permissions see the <a href="https://github.com/aws/aws-cdk/wiki/Security-And-Safety-Dev-Guide">Security And Safety Dev
 * Guide</a>
 * <p>
 * <h3>Custom Permissions Boundary</h3>
 * <p>
 * It is possible to attach Permissions Boundaries to all Roles created in a construct
 * tree all at once:
 * <p>
 * <blockquote><pre>
 * // Directly apply the boundary to a Role you create
 * Role role;
 * 
 * // Apply the boundary to an Role that was implicitly created for you
 * Function fn;
 * 
 * // Remove a Permissions Boundary that is inherited, for example from the Stack level
 * CustomResource customResource;
 * // This imports an existing policy.
 * IManagedPolicy boundary = ManagedPolicy.fromManagedPolicyArn(this, "Boundary", "arn:aws:iam::123456789012:policy/boundary");
 * 
 * // This creates a new boundary
 * ManagedPolicy boundary2 = ManagedPolicy.Builder.create(this, "Boundary2")
 *         .statements(List.of(
 *             PolicyStatement.Builder.create()
 *                     .effect(Effect.DENY)
 *                     .actions(List.of("iam:*"))
 *                     .resources(List.of("*"))
 *                     .build()))
 *         .build();
 * PermissionsBoundary.of(role).apply(boundary);
 * PermissionsBoundary.of(fn).apply(boundary);
 * 
 * // Apply the boundary to all Roles in a stack
 * PermissionsBoundary.of(this).apply(boundary);
 * PermissionsBoundary.of(customResource).clear();
 * </pre></blockquote>
 * <p>
 * <h2>OpenID Connect Providers</h2>
 * <p>
 * OIDC identity providers are entities in IAM that describe an external identity
 * provider (IdP) service that supports the <a href="http://openid.net/connect">OpenID Connect</a> (OIDC) standard, such
 * as Google or Salesforce. You use an IAM OIDC identity provider when you want to
 * establish trust between an OIDC-compatible IdP and your AWS account. This is
 * useful when creating a mobile app or web application that requires access to AWS
 * resources, but you don't want to create custom sign-in code or manage your own
 * user identities. For more information about this scenario, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc.html">About Web
 * Identity Federation</a> and the relevant documentation in the <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/open-id.html">Amazon Cognito
 * Identity Pools Developer Guide</a>.
 * <p>
 * The following examples defines an OpenID Connect provider. Two client IDs
 * (audiences) are will be able to send authentication requests to
 * <a href="https://openid/connect">https://openid/connect</a>.
 * <p>
 * <blockquote><pre>
 * OpenIdConnectProvider provider = OpenIdConnectProvider.Builder.create(this, "MyProvider")
 *         .url("https://openid/connect")
 *         .clientIds(List.of("myclient1", "myclient2"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can specify an optional list of <code>thumbprints</code>. If not specified, the
 * thumbprint of the root certificate authority (CA) will automatically be obtained
 * from the host as described
 * <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc_verify-thumbprint.html">here</a>.
 * <p>
 * Byy default, the custom resource enforces strict security practices by rejecting
 * any unauthorized connections when downloading CA thumbprints from the issuer URL.
 * If you need to connect to an unauthorized OIDC identity provider and understand the
 * implications, you can disable this behavior by setting the feature flag
 * <code>IAM_OIDC_REJECT_UNAUTHORIZED_CONNECTIONS</code> to <code>false</code> in your <code>cdk.context.json</code>
 * or <code>cdk.json</code>. Visit <a href="https://docs.aws.amazon.com/cdk/v2/guide/featureflags.html">CDK Feature Flag</a>
 * for more information on how to configure feature flags.
 * <p>
 * Once you define an OpenID connect provider, you can use it with AWS services
 * that expect an IAM OIDC provider. For example, when you define an <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/open-id.html">Amazon
 * Cognito identity
 * pool</a>
 * you can reference the provider's ARN as follows:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cognito.*;
 * 
 * OpenIdConnectProvider myProvider;
 * 
 * CfnIdentityPool.Builder.create(this, "IdentityPool")
 *         .openIdConnectProviderArns(List.of(myProvider.getOpenIdConnectProviderArn()))
 *         // And the other properties for your identity pool
 *         .allowUnauthenticatedIdentities(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>OpenIdConnectPrincipal</code> class can be used as a principal used with a <code>OpenIdConnectProvider</code>, for example:
 * <p>
 * <blockquote><pre>
 * OpenIdConnectProvider provider = OpenIdConnectProvider.Builder.create(this, "MyProvider")
 *         .url("https://openid/connect")
 *         .clientIds(List.of("myclient1", "myclient2"))
 *         .build();
 * OpenIdConnectPrincipal principal = new OpenIdConnectPrincipal(provider);
 * </pre></blockquote>
 * <p>
 * <h2>SAML provider</h2>
 * <p>
 * An IAM SAML 2.0 identity provider is an entity in IAM that describes an external
 * identity provider (IdP) service that supports the SAML 2.0 (Security Assertion
 * Markup Language 2.0) standard. You use an IAM identity provider when you want
 * to establish trust between a SAML-compatible IdP such as Shibboleth or Active
 * Directory Federation Services and AWS, so that users in your organization can
 * access AWS resources. IAM SAML identity providers are used as principals in an
 * IAM trust policy.
 * <p>
 * <blockquote><pre>
 * SamlProvider.Builder.create(this, "Provider")
 *         .metadataDocument(SamlMetadataDocument.fromFile("/path/to/saml-metadata-document.xml"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>SamlPrincipal</code> class can be used as a principal with a <code>SamlProvider</code>:
 * <p>
 * <blockquote><pre>
 * SamlProvider provider = SamlProvider.Builder.create(this, "Provider")
 *         .metadataDocument(SamlMetadataDocument.fromFile("/path/to/saml-metadata-document.xml"))
 *         .build();
 * SamlPrincipal principal = new SamlPrincipal(provider, Map.of(
 *         "StringEquals", Map.of(
 *                 "SAML:iss", "issuer")));
 * </pre></blockquote>
 * <p>
 * When creating a role for programmatic and AWS Management Console access, use the <code>SamlConsolePrincipal</code>
 * class:
 * <p>
 * <blockquote><pre>
 * SamlProvider provider = SamlProvider.Builder.create(this, "Provider")
 *         .metadataDocument(SamlMetadataDocument.fromFile("/path/to/saml-metadata-document.xml"))
 *         .build();
 * Role.Builder.create(this, "Role")
 *         .assumedBy(new SamlConsolePrincipal(provider))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Users</h2>
 * <p>
 * IAM manages users for your AWS account. To create a new user:
 * <p>
 * <blockquote><pre>
 * User user = new User(this, "MyUser");
 * </pre></blockquote>
 * <p>
 * To import an existing user by name <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names">with path</a>:
 * <p>
 * <blockquote><pre>
 * IUser user = User.fromUserName(this, "MyImportedUserByName", "johnsmith");
 * </pre></blockquote>
 * <p>
 * To import an existing user by ARN:
 * <p>
 * <blockquote><pre>
 * IUser user = User.fromUserArn(this, "MyImportedUserByArn", "arn:aws:iam::123456789012:user/johnsmith");
 * </pre></blockquote>
 * <p>
 * To import an existing user by attributes:
 * <p>
 * <blockquote><pre>
 * IUser user = User.fromUserAttributes(this, "MyImportedUserByAttributes", UserAttributes.builder()
 *         .userArn("arn:aws:iam::123456789012:user/johnsmith")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Access Keys</h3>
 * <p>
 * The ability for a user to make API calls via the CLI or an SDK is enabled by the user having an
 * access key pair. To create an access key:
 * <p>
 * <blockquote><pre>
 * User user = new User(this, "MyUser");
 * AccessKey accessKey = AccessKey.Builder.create(this, "MyAccessKey").user(user).build();
 * </pre></blockquote>
 * <p>
 * You can force CloudFormation to rotate the access key by providing a monotonically increasing <code>serial</code>
 * property. Simply provide a higher serial value than any number used previously:
 * <p>
 * <blockquote><pre>
 * User user = new User(this, "MyUser");
 * AccessKey accessKey = AccessKey.Builder.create(this, "MyAccessKey").user(user).serial(1).build();
 * </pre></blockquote>
 * <p>
 * An access key may only be associated with a single user and cannot be "moved" between users. Changing
 * the user associated with an access key replaces the access key (and its ID and secret value).
 * <p>
 * <h2>Groups</h2>
 * <p>
 * An IAM user group is a collection of IAM users. User groups let you specify permissions for multiple users.
 * <p>
 * <blockquote><pre>
 * Group group = new Group(this, "MyGroup");
 * </pre></blockquote>
 * <p>
 * To import an existing group by ARN:
 * <p>
 * <blockquote><pre>
 * IGroup group = Group.fromGroupArn(this, "MyImportedGroupByArn", "arn:aws:iam::account-id:group/group-name");
 * </pre></blockquote>
 * <p>
 * To import an existing group by name <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names">with path</a>:
 * <p>
 * <blockquote><pre>
 * IGroup group = Group.fromGroupName(this, "MyImportedGroupByName", "group-name");
 * </pre></blockquote>
 * <p>
 * To add a user to a group (both for a new and imported user/group):
 * <p>
 * <blockquote><pre>
 * User user = new User(this, "MyUser"); // or User.fromUserName(this, 'User', 'johnsmith');
 * Group group = new Group(this, "MyGroup"); // or Group.fromGroupArn(this, 'Group', 'arn:aws:iam::account-id:group/group-name');
 * 
 * user.addToGroup(group);
 * // or
 * group.addUser(user);
 * </pre></blockquote>
 * <p>
 * <h2>Instance Profiles</h2>
 * <p>
 * An IAM instance profile is a container for an IAM role that you can use to pass role information to an EC2 instance when the instance starts. By default, an instance profile must be created with a role:
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("ec2.amazonaws.com"))
 *         .build();
 * 
 * InstanceProfile instanceProfile = InstanceProfile.Builder.create(this, "InstanceProfile")
 *         .role(role)
 *         .build();
 * </pre></blockquote>
 * <p>
 * An instance profile can also optionally be created with an instance profile name and/or a <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names">path</a> to the instance profile:
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("ec2.amazonaws.com"))
 *         .build();
 * 
 * InstanceProfile instanceProfile = InstanceProfile.Builder.create(this, "InstanceProfile")
 *         .role(role)
 *         .instanceProfileName("MyInstanceProfile")
 *         .path("/sample/path/")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To import an existing instance profile by name:
 * <p>
 * <blockquote><pre>
 * IInstanceProfile instanceProfile = InstanceProfile.fromInstanceProfileName(this, "ImportedInstanceProfile", "MyInstanceProfile");
 * </pre></blockquote>
 * <p>
 * To import an existing instance profile by ARN:
 * <p>
 * <blockquote><pre>
 * IInstanceProfile instanceProfile = InstanceProfile.fromInstanceProfileArn(this, "ImportedInstanceProfile", "arn:aws:iam::account-id:instance-profile/MyInstanceProfile");
 * </pre></blockquote>
 * <p>
 * To import an existing instance profile with an associated role:
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("ec2.amazonaws.com"))
 *         .build();
 * 
 * IInstanceProfile instanceProfile = InstanceProfile.fromInstanceProfileAttributes(this, "ImportedInstanceProfile", InstanceProfileAttributes.builder()
 *         .instanceProfileArn("arn:aws:iam::account-id:instance-profile/MyInstanceProfile")
 *         .role(role)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Features</h2>
 * <p>
 * <ul>
 * <li>Policy name uniqueness is enforced. If two policies by the same name are attached to the same
 * principal, the attachment will fail.</li>
 * <li>Policy names are not required - the CDK logical ID will be used and ensured to be unique.</li>
 * <li>Policies are validated during synthesis to ensure that they have actions, and that policies
 * attached to IAM principals specify relevant resources, while policies attached to resources
 * specify which IAM principals they apply to.</li>
 * </ul>
 */
package software.amazon.awscdk.services.iam;
