/**
 * <h1>AWS CDK Assets</h1>
 * <p>
 * Assets are local files or directories which are needed by a CDK app. A common
 * example is a directory which contains the handler code for a Lambda function,
 * but assets can represent any artifact that is needed for the app's operation.
 * <p>
 * When deploying a CDK app that includes constructs with assets, the CDK toolkit
 * will first upload all the assets to S3, and only then deploy the stacks. The S3
 * locations of the uploaded assets will be passed in as CloudFormation Parameters
 * to the relevant stacks.
 * <p>
 * The following JavaScript example defines a directory asset which is archived as
 * a .zip file and uploaded to S3 during deployment.
 * <p>
 * <blockquote><pre>
 * Asset asset = Asset.Builder.create(this, "SampleAsset")
 *         .path(join(__dirname, "sample-asset-directory"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following JavaScript example defines a file asset, which is uploaded as-is
 * to an S3 bucket during deployment.
 * <p>
 * <blockquote><pre>
 * Asset asset = Asset.Builder.create(this, "SampleAsset")
 *         .path(join(__dirname, "file-asset.txt"))
 * 
 *         // Optional: describe the purpose of the asset with a human-readable string
 *         .displayName("My file")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Attributes</h2>
 * <p>
 * <code>Asset</code> constructs expose the following deploy-time attributes:
 * <p>
 * <ul>
 * <li><code>s3BucketName</code> - the name of the assets S3 bucket.</li>
 * <li><code>s3ObjectKey</code> - the S3 object key of the asset file (whether it's a file or a zip archive)</li>
 * <li><code>s3ObjectUrl</code> - the S3 object URL of the asset (i.e. s3://amzn-s3-demo-bucket/mykey.zip)</li>
 * <li><code>httpUrl</code> - the S3 HTTP URL of the asset (i.e. https://s3.us-east-1.amazonaws.com/amzn-s3-demo-bucket/mykey.zip)</li>
 * </ul>
 * <p>
 * In the following example, the various asset attributes are exported as stack outputs:
 * <p>
 * <blockquote><pre>
 * Asset asset = Asset.Builder.create(this, "SampleAsset")
 *         .path(join(__dirname, "sample-asset-directory"))
 *         .build();
 * 
 * CfnOutput.Builder.create(this, "S3BucketName").value(asset.getS3BucketName()).build();
 * CfnOutput.Builder.create(this, "S3ObjectKey").value(asset.getS3ObjectKey()).build();
 * CfnOutput.Builder.create(this, "S3HttpURL").value(asset.getHttpUrl()).build();
 * CfnOutput.Builder.create(this, "S3ObjectURL").value(asset.getS3ObjectUrl()).build();
 * </pre></blockquote>
 * <p>
 * <h2>Permissions</h2>
 * <p>
 * IAM roles, users or groups which need to be able to read assets in runtime will should be
 * granted IAM permissions. To do that use the <code>asset.grantRead(principal)</code> method:
 * <p>
 * The following example grants an IAM group read permissions on an asset:
 * <p>
 * <blockquote><pre>
 * Group group = new Group(this, "MyUserGroup");
 * asset.grantRead(group);
 * </pre></blockquote>
 * <p>
 * <h2>How does it work</h2>
 * <p>
 * When an asset is defined in a construct, a construct metadata entry
 * <code>aws:cdk:asset</code> is emitted with instructions on where to find the asset and what
 * type of packaging to perform (<code>zip</code> or <code>file</code>). Furthermore, the synthesized
 * CloudFormation template will also include two CloudFormation parameters: one for
 * the asset's bucket and one for the asset S3 key. Those parameters are used to
 * reference the deploy-time values of the asset (using <code>{ Ref: "Param" }</code>).
 * <p>
 * Then, when the stack is deployed, the toolkit will package the asset (i.e. zip
 * the directory), calculate an MD5 hash of the contents and will render an S3 key
 * for this asset within the toolkit's asset store. If the file doesn't exist in
 * the asset store, it is uploaded during deployment.
 * <p>
 * <blockquote>
 * <p>
 * The toolkit's asset store is an S3 bucket created by the toolkit for each
 * environment the toolkit operates in (environment = account + region).
 * <p>
 * </blockquote>
 * <p>
 * Now, when the toolkit deploys the stack, it will set the relevant CloudFormation
 * Parameters to point to the actual bucket and key for each asset.
 * <p>
 * <h2>Asset Bundling</h2>
 * <p>
 * When defining an asset, you can use the <code>bundling</code> option to specify a command
 * to run inside a docker container. The command can read the contents of the asset
 * source from <code>/asset-input</code> and is expected to write files under <code>/asset-output</code>
 * (directories mapped inside the container). The files under <code>/asset-output</code> will
 * be zipped and uploaded to S3 as the asset.
 * <p>
 * The following example uses custom asset bundling to convert a markdown file to html:
 * <p>
 * <blockquote><pre>
 * Asset asset = Asset.Builder.create(this, "BundledAsset")
 *         .path(join(__dirname, "markdown-asset")) // /asset-input and working directory in the container
 *         .bundling(BundlingOptions.builder()
 *                 .image(DockerImage.fromBuild(join(__dirname, "alpine-markdown"))) // Build an image
 *                 .command(List.of("sh", "-c", "\n            markdown index.md &gt; /asset-output/index.html\n          "))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The bundling docker image (<code>image</code>) can either come from a registry (<code>DockerImage.fromRegistry</code>)
 * or it can be built from a <code>Dockerfile</code> located inside your project (<code>DockerImage.fromBuild</code>).
 * <p>
 * You can set the <code>CDK_DOCKER</code> environment variable in order to provide a custom
 * docker program to execute. This may sometime be needed when building in
 * environments where the standard docker cannot be executed (see
 * https://github.com/aws/aws-cdk/issues/8460 for details).
 * <p>
 * Use <code>local</code> to specify a local bundling provider. The provider implements a
 * method <code>tryBundle()</code> which should return <code>true</code> if local bundling was performed.
 * If <code>false</code> is returned, docker bundling will be done:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * public class MyBundle implements ILocalBundling {
 *     public boolean tryBundle(String outputDir, BundlingOptions options) {
 *         boolean canRunLocally = true; // replace with actual logic
 *         if (canRunLocally) {
 *             // perform local bundling here
 *             return true;
 *         }
 *         return false;
 *     }
 * }
 * 
 * Asset.Builder.create(this, "BundledAsset")
 *         .path("/path/to/asset")
 *         .bundling(BundlingOptions.builder()
 *                 .local(new MyBundle())
 *                 // Docker bundling fallback
 *                 .image(DockerImage.fromRegistry("alpine"))
 *                 .entrypoint(List.of("/bin/sh", "-c"))
 *                 .command(List.of("bundle"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Although optional, it's recommended to provide a local bundling method which can
 * greatly improve performance.
 * <p>
 * If the bundling output contains a single archive file (zip or jar) it will be
 * uploaded to S3 as-is and will not be zipped. Otherwise the contents of the
 * output directory will be zipped and the zip file will be uploaded to S3. This
 * is the default behavior for <code>bundling.outputType</code> (<code>BundlingOutput.AUTO_DISCOVER</code>).
 * <p>
 * Use <code>BundlingOutput.NOT_ARCHIVED</code> if the bundling output must always be zipped:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * Asset asset = Asset.Builder.create(this, "BundledAsset")
 *         .path("/path/to/asset")
 *         .bundling(BundlingOptions.builder()
 *                 .image(DockerImage.fromRegistry("alpine"))
 *                 .command(List.of("command-that-produces-an-archive.sh"))
 *                 .outputType(BundlingOutput.NOT_ARCHIVED)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>BundlingOutput.ARCHIVED</code> if the bundling output contains a single archive file and
 * you don't want it to be zipped.
 * <p>
 * <h3>Docker options</h3>
 * <p>
 * Depending on your build environment, you may need to pass certain docker options to the <code>docker run</code> command that bundles assets.
 * This can be done using <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.BundlingOptions.html">BundlingOptions</a> properties.
 * <p>
 * Some optional properties to pass to the docker bundling
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * 
 * Asset asset = Asset.Builder.create(this, "BundledAsset")
 *         .path("/path/to/asset")
 *         .bundling(BundlingOptions.builder()
 *                 .image(Runtime.PYTHON_3_9.getBundlingImage())
 *                 .command(List.of("bash", "-c", "pip install -r requirements.txt -t /asset-output &amp;&amp; cp -au . /asset-output"))
 *                 .securityOpt("no-new-privileges:true") // https://docs.docker.com/engine/reference/commandline/run/#optional-security-options---security-opt
 *                 .network("host")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>CloudFormation Resource Metadata</h2>
 * <p>
 * <blockquote>
 * <p>
 * NOTE: This section is relevant for authors of AWS Resource Constructs.
 * <p>
 * </blockquote>
 * <p>
 * In certain situations, it is desirable for tools to be able to know that a certain CloudFormation
 * resource is using a local asset. For example, SAM CLI can be used to invoke AWS Lambda functions
 * locally for debugging purposes.
 * <p>
 * To enable such use cases, external tools will consult a set of metadata entries on AWS CloudFormation
 * resources:
 * <p>
 * <ul>
 * <li><code>aws:asset:path</code> points to the local path of the asset.</li>
 * <li><code>aws:asset:property</code> is the name of the resource property where the asset is used</li>
 * </ul>
 * <p>
 * Using these two metadata entries, tools will be able to identify that assets are used
 * by a certain resource, and enable advanced local experiences.
 * <p>
 * To add these metadata entries to a resource, use the
 * <code>asset.addResourceMetadata(resource, property)</code> method.
 * <p>
 * See https://github.com/aws/aws-cdk/issues/1432 for more details
 */
package software.amazon.awscdk.services.s3.assets;
