/**
 * <h1>Amazon S3 Construct Library</h1>
 * <p>
 * Define an S3 bucket.
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyFirstBucket");
 * </pre></blockquote>
 * <p>
 * <code>Bucket</code> constructs expose the following deploy-time attributes:
 * <p>
 * <ul>
 * <li><code>bucketArn</code> - the ARN of the bucket (i.e. <code>arn:aws:s3:::amzn-s3-demo-bucket</code>)</li>
 * <li><code>bucketName</code> - the name of the bucket (i.e. <code>amzn-s3-demo-bucket</code>)</li>
 * <li><code>bucketWebsiteUrl</code> - the Website URL of the bucket (i.e.
 * <code>http://amzn-s3-demo-bucket.s3-website-us-west-1.amazonaws.com</code>)</li>
 * <li><code>bucketDomainName</code> - the URL of the bucket (i.e. <code>amzn-s3-demo-bucket.s3.amazonaws.com</code>)</li>
 * <li><code>bucketDualStackDomainName</code> - the dual-stack URL of the bucket (i.e.
 * <code>amzn-s3-demo-bucket.s3.dualstack.eu-west-1.amazonaws.com</code>)</li>
 * <li><code>bucketRegionalDomainName</code> - the regional URL of the bucket (i.e.
 * <code>amzn-s3-demo-bucket.s3.eu-west-1.amazonaws.com</code>)</li>
 * <li><code>arnForObjects(pattern)</code> - the ARN of an object or objects within the bucket (i.e.
 * <code>arn:aws:s3:::amzn-s3-demo-bucket/exampleobject.png</code> or
 * <code>arn:aws:s3:::amzn-s3-demo-bucket/Development/*</code>)</li>
 * <li><code>urlForObject(key)</code> - the HTTP URL of an object within the bucket (i.e.
 * <code>https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey</code>)</li>
 * <li><code>virtualHostedUrlForObject(key)</code> - the virtual-hosted style HTTP URL of an object
 * within the bucket (i.e. <code>https://china-bucket-s3.cn-north-1.amazonaws.com.cn/mykey</code>)</li>
 * <li><code>s3UrlForObject(key)</code> - the S3 URL of an object within the bucket (i.e.
 * <code>s3://bucket/mykey</code>)</li>
 * </ul>
 * <p>
 * <h2>Encryption</h2>
 * <p>
 * Define a KMS-encrypted bucket:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyEncryptedBucket")
 *         .encryption(BucketEncryption.KMS)
 *         .build();
 * 
 * // you can access the encryption key:
 * assert(bucket.getEncryptionKey() instanceof Key);
 * </pre></blockquote>
 * <p>
 * You can also supply your own key:
 * <p>
 * <blockquote><pre>
 * Key myKmsKey = new Key(this, "MyKey");
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyEncryptedBucket")
 *         .encryption(BucketEncryption.KMS)
 *         .encryptionKey(myKmsKey)
 *         .build();
 * 
 * assert(bucket.getEncryptionKey() == myKmsKey);
 * </pre></blockquote>
 * <p>
 * Enable KMS-SSE encryption via <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-key.html">S3 Bucket Keys</a>:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyEncryptedBucket")
 *         .encryption(BucketEncryption.KMS)
 *         .bucketKeyEnabled(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>BucketEncryption.ManagedKms</code> to use the S3 master KMS key:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "Buck")
 *         .encryption(BucketEncryption.KMS_MANAGED)
 *         .build();
 * 
 * assert(bucket.getEncryptionKey() == null);
 * </pre></blockquote>
 * <p>
 * Enable DSSE encryption:
 * <p>
 * <blockquote><pre>
 * const bucket = new s3.Bucket(stack, 'MyDSSEBucket', {
 *   encryption: s3.BucketEncryption.DSSE_MANAGED,
 *   bucketKeyEnabled: true,
 * });
 * </pre></blockquote>
 * <p>
 * <h2>Permissions</h2>
 * <p>
 * A bucket policy will be automatically created for the bucket upon the first call to
 * <code>addToResourcePolicy(statement)</code>:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyBucket");
 * AddToResourcePolicyResult result = bucket.addToResourcePolicy(
 * PolicyStatement.Builder.create()
 *         .actions(List.of("s3:GetObject"))
 *         .resources(List.of(bucket.arnForObjects("file.txt")))
 *         .principals(List.of(new AccountRootPrincipal()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you try to add a policy statement to an existing bucket, this method will
 * not do anything:
 * <p>
 * <blockquote><pre>
 * IBucket bucket = Bucket.fromBucketName(this, "existingBucket", "amzn-s3-demo-bucket");
 * 
 * // No policy statement will be added to the resource
 * AddToResourcePolicyResult result = bucket.addToResourcePolicy(
 * PolicyStatement.Builder.create()
 *         .actions(List.of("s3:GetObject"))
 *         .resources(List.of(bucket.arnForObjects("file.txt")))
 *         .principals(List.of(new AccountRootPrincipal()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * That's because it's not possible to tell whether the bucket
 * already has a policy attached, let alone to re-use that policy to add more
 * statements to it. We recommend that you always check the result of the call:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyBucket");
 * AddToResourcePolicyResult result = bucket.addToResourcePolicy(
 * PolicyStatement.Builder.create()
 *         .actions(List.of("s3:GetObject"))
 *         .resources(List.of(bucket.arnForObjects("file.txt")))
 *         .principals(List.of(new AccountRootPrincipal()))
 *         .build());
 * 
 * if (!result.getStatementAdded()) {
 * }
 * </pre></blockquote>
 * <p>
 * The bucket policy can be directly accessed after creation to add statements or
 * adjust the removal policy.
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyBucket");
 * bucket.policy.applyRemovalPolicy(RemovalPolicy.RETAIN);
 * </pre></blockquote>
 * <p>
 * Most of the time, you won't have to manipulate the bucket policy directly.
 * Instead, buckets have "grant" methods called to give prepackaged sets of permissions
 * to other resources. For example:
 * <p>
 * <blockquote><pre>
 * Function myLambda;
 * 
 * 
 * Bucket bucket = new Bucket(this, "MyBucket");
 * bucket.grantReadWrite(myLambda);
 * </pre></blockquote>
 * <p>
 * Will give the Lambda's execution role permissions to read and write
 * from the bucket.
 * <p>
 * <h2>AWS Foundational Security Best Practices</h2>
 * <p>
 * <h3>Enforcing SSL</h3>
 * <p>
 * To require all requests use Secure Socket Layer (SSL):
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "Bucket")
 *         .enforceSSL(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To require a minimum TLS version for all requests:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "Bucket")
 *         .enforceSSL(true)
 *         .minimumTLSVersion(1.2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Sharing buckets between stacks</h2>
 * <p>
 * To use a bucket in a different stack in the same CDK application, pass the object to the other stack:
 * <p>
 * <blockquote><pre>
 * /**
 *  * Stack that defines the bucket
 *  *&#47;
 * public class Producer extends Stack {
 *     public final Bucket myBucket;
 * 
 *     public Producer(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public Producer(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *                 .removalPolicy(RemovalPolicy.DESTROY)
 *                 .build();
 *         this.myBucket = bucket;
 *     }
 * }
 * 
 * public class ConsumerProps extends StackProps {
 *     private IBucket userBucket;
 *     public IBucket getUserBucket() {
 *         return this.userBucket;
 *     }
 *     public ConsumerProps userBucket(IBucket userBucket) {
 *         this.userBucket = userBucket;
 *         return this;
 *     }
 * }
 * 
 * /**
 *  * Stack that consumes the bucket
 *  *&#47;
 * public class Consumer extends Stack {
 *     public Consumer(Construct scope, String id, ConsumerProps props) {
 *         super(scope, id, props);
 * 
 *         User user = new User(this, "MyUser");
 *         props.userBucket.grantReadWrite(user);
 *     }
 * }
 * 
 * App app = new App();
 * Producer producer = new Producer(app, "ProducerStack");
 * new Consumer(app, "ConsumerStack", new ConsumerProps().userBucket(producer.getMyBucket()));
 * </pre></blockquote>
 * <p>
 * <h2>Importing existing buckets</h2>
 * <p>
 * To import an existing bucket into your CDK application, use the <code>Bucket.fromBucketAttributes</code>
 * factory method. This method accepts <code>BucketAttributes</code> which describes the properties of an already
 * existing bucket:
 * <p>
 * Note that this method allows importing buckets with legacy names containing uppercase letters (<code>A-Z</code>) or underscores (<code>_</code>), which were
 * permitted for buckets created before March 1, 2018. For buckets created after this date, uppercase letters and underscores
 * are not allowed in the bucket name.
 * <p>
 * <blockquote><pre>
 * Function myLambda;
 * 
 * IBucket bucket = Bucket.fromBucketAttributes(this, "ImportedBucket", BucketAttributes.builder()
 *         .bucketArn("arn:aws:s3:::amzn-s3-demo-bucket")
 *         .build());
 * 
 * // now you can just call methods on the bucket
 * bucket.addEventNotification(EventType.OBJECT_CREATED, new LambdaDestination(myLambda), NotificationKeyFilter.builder()
 *         .prefix("home/myusername/*")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Alternatively, short-hand factories are available as <code>Bucket.fromBucketName</code> and
 * <code>Bucket.fromBucketArn</code>, which will derive all bucket attributes from the bucket
 * name or ARN respectively:
 * <p>
 * <blockquote><pre>
 * IBucket byName = Bucket.fromBucketName(this, "BucketByName", "amzn-s3-demo-bucket");
 * IBucket byArn = Bucket.fromBucketArn(this, "BucketByArn", "arn:aws:s3:::amzn-s3-demo-bucket");
 * </pre></blockquote>
 * <p>
 * The bucket's region defaults to the current stack's region, but can also be explicitly set in cases where one of the bucket's
 * regional properties needs to contain the correct values.
 * <p>
 * <blockquote><pre>
 * IBucket myCrossRegionBucket = Bucket.fromBucketAttributes(this, "CrossRegionImport", BucketAttributes.builder()
 *         .bucketArn("arn:aws:s3:::amzn-s3-demo-bucket")
 *         .region("us-east-1")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Bucket Notifications</h2>
 * <p>
 * The Amazon S3 notification feature enables you to receive notifications when
 * certain events happen in your bucket as described under <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html">S3 Bucket
 * Notifications</a> of the S3 Developer Guide.
 * <p>
 * To subscribe for bucket notifications, use the <code>bucket.addEventNotification</code> method. The
 * <code>bucket.addObjectCreatedNotification</code> and <code>bucket.addObjectRemovedNotification</code> can also be used for
 * these common use cases.
 * <p>
 * The following example will subscribe an SNS topic to be notified of all <code>s3:ObjectCreated:*</code> events:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyBucket");
 * Topic topic = new Topic(this, "MyTopic");
 * bucket.addEventNotification(EventType.OBJECT_CREATED, new SnsDestination(topic));
 * </pre></blockquote>
 * <p>
 * This call will also ensure that the topic policy can accept notifications for
 * this specific bucket.
 * <p>
 * Supported S3 notification targets are exposed by the <code>aws-cdk-lib/aws-s3-notifications</code> package.
 * <p>
 * It is also possible to specify S3 object key filters when subscribing. The
 * following example will notify <code>myQueue</code> when objects prefixed with <code>foo/</code> and
 * have the <code>.jpg</code> suffix are removed from the bucket.
 * <p>
 * <blockquote><pre>
 * Queue myQueue;
 * 
 * Bucket bucket = new Bucket(this, "MyBucket");
 * bucket.addEventNotification(EventType.OBJECT_REMOVED, new SqsDestination(myQueue), NotificationKeyFilter.builder()
 *         .prefix("foo/")
 *         .suffix(".jpg")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Adding notifications on existing buckets:
 * <p>
 * <blockquote><pre>
 * Topic topic;
 * 
 * IBucket bucket = Bucket.fromBucketAttributes(this, "ImportedBucket", BucketAttributes.builder()
 *         .bucketArn("arn:aws:s3:::amzn-s3-demo-bucket")
 *         .build());
 * bucket.addEventNotification(EventType.OBJECT_CREATED, new SnsDestination(topic));
 * </pre></blockquote>
 * <p>
 * If you do not want for S3 to validate permissions of Amazon SQS, Amazon SNS, and Lambda destinations you can use the <code>notificationsSkipDestinationValidation</code> flag:
 * <p>
 * <blockquote><pre>
 * Queue myQueue;
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .notificationsSkipDestinationValidation(true)
 *         .build();
 * bucket.addEventNotification(EventType.OBJECT_REMOVED, new SqsDestination(myQueue));
 * </pre></blockquote>
 * <p>
 * When you add an event notification to a bucket, a custom resource is created to
 * manage the notifications. By default, a new role is created for the Lambda
 * function that implements this feature. If you want to use your own role instead,
 * you should provide it in the <code>Bucket</code> constructor:
 * <p>
 * <blockquote><pre>
 * IRole myRole;
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .notificationsHandlerRole(myRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Whatever role you provide, the CDK will try to modify it by adding the
 * permissions from <code>AWSLambdaBasicExecutionRole</code> (an AWS managed policy) as well
 * as the permissions <code>s3:PutBucketNotification</code> and <code>s3:GetBucketNotification</code>.
 * If you’re passing an imported role, and you don’t want this to happen, configure
 * it to be immutable:
 * <p>
 * <blockquote><pre>
 * IRole importedRole = Role.fromRoleArn(this, "role", "arn:aws:iam::123456789012:role/RoleName", FromRoleArnOptions.builder()
 *         .mutable(false)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * If you provide an imported immutable role, make sure that it has at least all
 * the permissions mentioned above. Otherwise, the deployment will fail!
 * <p>
 * </blockquote>
 * <p>
 * <h3>EventBridge notifications</h3>
 * <p>
 * Amazon S3 can send events to Amazon EventBridge whenever certain events happen in your bucket.
 * Unlike other destinations, you don't need to select which event types you want to deliver.
 * <p>
 * The following example will enable EventBridge notifications:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyEventBridgeBucket")
 *         .eventBridgeEnabled(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Block Public Access</h2>
 * <p>
 * Use <code>blockPublicAccess</code> to specify <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html">block public access settings</a> on the bucket.
 * <p>
 * Enable all block public access settings:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyBlockedBucket")
 *         .blockPublicAccess(BlockPublicAccess.BLOCK_ALL)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Block and ignore public ACLs (other options remain unblocked):
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyBlockedBucket")
 *         .blockPublicAccess(BlockPublicAccess.BLOCK_ACLS_ONLY)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, specify the settings manually (unspecified options will remain blocked):
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyBlockedBucket")
 *         .blockPublicAccess(BlockPublicAccess.Builder.create().blockPublicPolicy(false).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * When <code>blockPublicPolicy</code> is set to <code>true</code>, <code>grantPublicRead()</code> throws an error.
 * <p>
 * <h2>Public Read Access</h2>
 * <p>
 * Use <code>publicReadAccess</code> to allow public read access to the bucket.
 * <p>
 * Note that to enable <code>publicReadAccess</code>, make sure both bucket-level and account-level block public access control is disabled.
 * <p>
 * Bucket-level block public access control can be configured through <code>blockPublicAccess</code> property. Account-level block public
 * access control can be configured on AWS Console -&gt; S3 -&gt; Block Public Access settings for this account (Navigation Panel).
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "Bucket")
 *         .publicReadAccess(true)
 *         .blockPublicAccess(Map.of(
 *                 "blockPublicPolicy", false,
 *                 "blockPublicAcls", false,
 *                 "ignorePublicAcls", false,
 *                 "restrictPublicBuckets", false))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Logging configuration</h2>
 * <p>
 * Use <code>serverAccessLogsBucket</code> to describe where server access logs are to be stored.
 * <p>
 * <blockquote><pre>
 * Bucket accessLogsBucket = new Bucket(this, "AccessLogsBucket");
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .build();
 * </pre></blockquote>
 * <p>
 * It's also possible to specify a prefix for Amazon S3 to assign to all log object keys.
 * <p>
 * <blockquote><pre>
 * Bucket accessLogsBucket = new Bucket(this, "AccessLogsBucket");
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .serverAccessLogsPrefix("logs")
 *         .build();
 * </pre></blockquote>
 * <p>
 * You have two options for the log object key format.
 * <code>Non-date-based partitioning</code> is the default log object key format and appears as follows:
 * <p>
 * <blockquote><pre>
 * [DestinationPrefix][YYYY]-[MM]-[DD]-[hh]-[mm]-[ss]-[UniqueString]
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * Bucket accessLogsBucket = new Bucket(this, "AccessLogsBucket");
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .serverAccessLogsPrefix("logs")
 *         // You can use a simple prefix with `TargetObjectKeyFormat.simplePrefix()`, but it is the same even if you do not specify `targetObjectKeyFormat` property.
 *         .targetObjectKeyFormat(TargetObjectKeyFormat.simplePrefix())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Another option is <code>Date-based partitioning</code>.
 * If you choose this format, you can select either the event time or the delivery time of the log file as the date source used in the log format.
 * This format appears as follows:
 * <p>
 * <blockquote><pre>
 * [DestinationPrefix][SourceAccountId]/[SourceRegion]/[SourceBucket]/[YYYY]/[MM]/[DD]/[YYYY]-[MM]-[DD]-[hh]-[mm]-[ss]-[UniqueString]
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * Bucket accessLogsBucket = new Bucket(this, "AccessLogsBucket");
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .serverAccessLogsPrefix("logs")
 *         .targetObjectKeyFormat(TargetObjectKeyFormat.partitionedPrefix(PartitionDateSource.EVENT_TIME))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Allowing access log delivery using a Bucket Policy (recommended)</h3>
 * <p>
 * When possible, it is recommended to use a bucket policy to grant access instead of
 * using ACLs. When the <code>&#64;aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy</code> feature flag
 * is enabled, this is done by default for server access logs. If S3 Server Access Logs
 * are the only logs delivered to your bucket (or if all other services logging to the
 * bucket support using bucket policy instead of ACLs), you can set object ownership
 * to <a href="#bucket-owner-enforced-recommended">bucket owner enforced</a>, as is recommended.
 * <p>
 * <blockquote><pre>
 * Bucket accessLogsBucket = Bucket.Builder.create(this, "AccessLogsBucket")
 *         .objectOwnership(ObjectOwnership.BUCKET_OWNER_ENFORCED)
 *         .build();
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .serverAccessLogsPrefix("logs")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The above code will create a new bucket policy if none exists or update the
 * existing bucket policy to allow access log delivery.
 * <p>
 * However, there could be an edge case if the <code>accessLogsBucket</code> also defines a bucket
 * policy resource using the L1 Construct. Although the mixing of L1 and L2 Constructs is not
 * recommended, there are no mechanisms in place to prevent users from doing this at the moment.
 * <p>
 * <blockquote><pre>
 * String bucketName = "amzn-s3-demo-bucket";
 * Bucket accessLogsBucket = Bucket.Builder.create(this, "AccessLogsBucket")
 *         .objectOwnership(ObjectOwnership.BUCKET_OWNER_ENFORCED)
 *         .bucketName(bucketName)
 *         .build();
 * 
 * // Creating a bucket policy using L1
 * CfnBucketPolicy bucketPolicy = CfnBucketPolicy.Builder.create(this, "BucketPolicy")
 *         .bucket(bucketName)
 *         .policyDocument(Map.of(
 *                 "Statement", List.of(Map.of(
 *                         "Action", "s3:*",
 *                         "Effect", "Deny",
 *                         "Principal", Map.of(
 *                                 "AWS", "*"),
 *                         "Resource", List.of(accessLogsBucket.getBucketArn(), String.format("%s/*", accessLogsBucket.getBucketArn())))),
 *                 "Version", "2012-10-17"))
 *         .build();
 * 
 * // 'serverAccessLogsBucket' will create a new L2 bucket policy
 * // to allow log delivery and overwrite the L1 bucket policy.
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .serverAccessLogsPrefix("logs")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The above example uses the L2 Bucket Construct with the L1 CfnBucketPolicy Construct. However,
 * when <code>serverAccessLogsBucket</code> is set, a new L2 Bucket Policy resource will be created
 * which overwrites the permissions defined in the L1 Bucket Policy causing unintended
 * behaviours.
 * <p>
 * As noted above, we highly discourage the mixed usage of L1 and L2 Constructs. The recommended
 * approach would to define the bucket policy using <code>addToResourcePolicy</code> method.
 * <p>
 * <blockquote><pre>
 * Bucket accessLogsBucket = Bucket.Builder.create(this, "AccessLogsBucket")
 *         .objectOwnership(ObjectOwnership.BUCKET_OWNER_ENFORCED)
 *         .build();
 * 
 * accessLogsBucket.addToResourcePolicy(
 * PolicyStatement.Builder.create()
 *         .actions(List.of("s3:*"))
 *         .resources(List.of(accessLogsBucket.getBucketArn(), accessLogsBucket.arnForObjects("*")))
 *         .principals(List.of(new AnyPrincipal()))
 *         .build());
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .serverAccessLogsPrefix("logs")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, users can use the L2 Bucket Policy Construct
 * <code>BucketPolicy.fromCfnBucketPolicy</code> to wrap around <code>CfnBucketPolicy</code> Construct. This will allow the subsequent bucket policy generated by <code>serverAccessLogsBucket</code> usage to append to the existing bucket policy instead of overwriting.
 * <p>
 * <blockquote><pre>
 * String bucketName = "amzn-s3-demo-bucket";
 * Bucket accessLogsBucket = Bucket.Builder.create(this, "AccessLogsBucket")
 *         .objectOwnership(ObjectOwnership.BUCKET_OWNER_ENFORCED)
 *         .bucketName(bucketName)
 *         .build();
 * 
 * CfnBucketPolicy bucketPolicy = CfnBucketPolicy.Builder.create(this, "BucketPolicy")
 *         .bucket(bucketName)
 *         .policyDocument(Map.of(
 *                 "Statement", List.of(Map.of(
 *                         "Action", "s3:*",
 *                         "Effect", "Deny",
 *                         "Principal", Map.of(
 *                                 "AWS", "*"),
 *                         "Resource", List.of(accessLogsBucket.getBucketArn(), String.format("%s/*", accessLogsBucket.getBucketArn())))),
 *                 "Version", "2012-10-17"))
 *         .build();
 * 
 * // Wrap L1 Construct with L2 Bucket Policy Construct. Subsequent
 * // generated bucket policy to allow access log delivery would append
 * // to the current policy.
 * BucketPolicy.fromCfnBucketPolicy(bucketPolicy);
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .serverAccessLogsBucket(accessLogsBucket)
 *         .serverAccessLogsPrefix("logs")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>S3 Inventory</h2>
 * <p>
 * An <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-inventory.html">inventory</a> contains a list of the objects in the source bucket and metadata for each object. The inventory lists are stored in the destination bucket as a CSV file compressed with GZIP, as an Apache optimized row columnar (ORC) file compressed with ZLIB, or as an Apache Parquet (Parquet) file compressed with Snappy.
 * <p>
 * You can configure multiple inventory lists for a bucket. You can configure what object metadata to include in the inventory, whether to list all object versions or only current versions, where to store the inventory list file output, and whether to generate the inventory on a daily or weekly basis.
 * <p>
 * <blockquote><pre>
 * Bucket inventoryBucket = new Bucket(this, "InventoryBucket");
 * 
 * Bucket dataBucket = Bucket.Builder.create(this, "DataBucket")
 *         .inventories(List.of(Inventory.builder()
 *                 .frequency(InventoryFrequency.DAILY)
 *                 .includeObjectVersions(InventoryObjectVersion.CURRENT)
 *                 .destination(InventoryDestination.builder()
 *                         .bucket(inventoryBucket)
 *                         .build())
 *                 .build(), Inventory.builder()
 *                 .frequency(InventoryFrequency.WEEKLY)
 *                 .includeObjectVersions(InventoryObjectVersion.ALL)
 *                 .destination(InventoryDestination.builder()
 *                         .bucket(inventoryBucket)
 *                         .prefix("with-all-versions")
 *                         .build())
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If the destination bucket is created as part of the same CDK application, the necessary permissions will be automatically added to the bucket policy.
 * However, if you use an imported bucket (i.e <code>Bucket.fromXXX()</code>), you'll have to make sure it contains the following policy document:
 * <p>
 * <blockquote><pre>
 * {
 *   "Version": "2012-10-17",
 *   "Statement": [
 *     {
 *       "Sid": "InventoryAndAnalyticsExamplePolicy",
 *       "Effect": "Allow",
 *       "Principal": { "Service": "s3.amazonaws.com" },
 *       "Action": "s3:PutObject",
 *       "Resource": ["arn:aws:s3:::amzn-s3-demo-destination-bucket/*"]
 *     }
 *   ]
 * }
 * </pre></blockquote>
 * <p>
 * <h2>Website redirection</h2>
 * <p>
 * You can use the two following properties to specify the bucket <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html#advanced-conditional-redirects">redirection policy</a>. Please note that these methods cannot both be applied to the same bucket.
 * <p>
 * <h3>Static redirection</h3>
 * <p>
 * You can statically redirect a to a given Bucket URL or any other host name with <code>websiteRedirect</code>:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyRedirectedBucket")
 *         .websiteRedirect(RedirectTarget.builder().hostName("www.example.com").build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Routing rules</h3>
 * <p>
 * Alternatively, you can also define multiple <code>websiteRoutingRules</code>, to define complex, conditional redirections:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyRedirectedBucket")
 *         .websiteRoutingRules(List.of(RoutingRule.builder()
 *                 .hostName("www.example.com")
 *                 .httpRedirectCode("302")
 *                 .protocol(RedirectProtocol.HTTPS)
 *                 .replaceKey(ReplaceKey.prefixWith("test/"))
 *                 .condition(RoutingRuleCondition.builder()
 *                         .httpErrorCodeReturnedEquals("200")
 *                         .keyPrefixEquals("prefix")
 *                         .build())
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Filling the bucket as part of deployment</h2>
 * <p>
 * To put files into a bucket as part of a deployment (for example, to host a
 * website), see the <code>aws-cdk-lib/aws-s3-deployment</code> package, which provides a
 * resource that can do just that.
 * <p>
 * <h2>The URL for objects</h2>
 * <p>
 * S3 provides two types of URLs for accessing objects via HTTP(S). Path-Style and
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/VirtualHosting.html">Virtual Hosted-Style</a>
 * URL. Path-Style is a classic way and will be
 * <a href="https://aws.amazon.com/jp/blogs/aws/amazon-s3-path-deprecation-plan-the-rest-of-the-story">deprecated</a>.
 * We recommend to use Virtual Hosted-Style URL for newly made bucket.
 * <p>
 * You can generate both of them.
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyBucket");
 * bucket.urlForObject("objectname"); // Path-Style URL
 * bucket.virtualHostedUrlForObject("objectname"); // Virtual Hosted-Style URL
 * bucket.virtualHostedUrlForObject("objectname", VirtualHostedStyleUrlOptions.builder().regional(false).build());
 * </pre></blockquote>
 * <p>
 * <h2>Object Ownership</h2>
 * <p>
 * You can use one of following properties to specify the bucket <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/about-object-ownership.html">object Ownership</a>.
 * <p>
 * <h3>Object writer</h3>
 * <p>
 * The Uploading account will own the object.
 * <p>
 * <blockquote><pre>
 * Bucket.Builder.create(this, "MyBucket")
 *         .objectOwnership(ObjectOwnership.OBJECT_WRITER)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Bucket owner preferred</h3>
 * <p>
 * The bucket owner will own the object if the object is uploaded with the bucket-owner-full-control canned ACL. Without this setting and canned ACL, the object is uploaded and remains owned by the uploading account.
 * <p>
 * <blockquote><pre>
 * Bucket.Builder.create(this, "MyBucket")
 *         .objectOwnership(ObjectOwnership.BUCKET_OWNER_PREFERRED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Bucket owner enforced (recommended)</h3>
 * <p>
 * ACLs are disabled, and the bucket owner automatically owns and has full control
 * over every object in the bucket. ACLs no longer affect permissions to data in the
 * S3 bucket. The bucket uses policies to define access control.
 * <p>
 * <blockquote><pre>
 * Bucket.Builder.create(this, "MyBucket")
 *         .objectOwnership(ObjectOwnership.BUCKET_OWNER_ENFORCED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Some services may not not support log delivery to buckets that have object ownership
 * set to bucket owner enforced, such as
 * <a href="#allowing-access-log-delivery-using-a-bucket-policy-recommended">S3 buckets using ACLs</a>
 * or <a href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/AccessLogs.html#AccessLogsBucketAndFileOwnership">CloudFront Distributions</a>.
 * <p>
 * <h2>Bucket deletion</h2>
 * <p>
 * When a bucket is removed from a stack (or the stack is deleted), the S3
 * bucket will be removed according to its removal policy (which by default will
 * simply orphan the bucket and leave it in your AWS account). If the removal
 * policy is set to <code>RemovalPolicy.DESTROY</code>, the bucket will be deleted as long
 * as it does not contain any objects.
 * <p>
 * To override this and force all objects to get deleted during bucket deletion,
 * enable the<code>autoDeleteObjects</code> option.
 * <p>
 * When <code>autoDeleteObjects</code> is enabled, <code>s3:PutBucketPolicy</code> is added to the bucket policy. This is done to allow the custom resource this feature is built on to add a deny policy for <code>s3:PutObject</code> to the bucket policy when a delete stack event occurs. Adding this deny policy prevents new objects from being written to the bucket. Doing this prevents race conditions with external bucket writers during the deletion process.
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyTempFileBucket")
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .autoDeleteObjects(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Warning</strong> if you have deployed a bucket with <code>autoDeleteObjects: true</code>,
 * switching this to <code>false</code> in a CDK version <em>before</em> <code>1.126.0</code> will lead to
 * all objects in the bucket being deleted. Be sure to update your bucket resources
 * by deploying with CDK version <code>1.126.0</code> or later <strong>before</strong> switching this value to <code>false</code>.
 * <p>
 * <h2>Transfer Acceleration</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/transfer-acceleration.html">Transfer Acceleration</a> can be configured to enable fast, easy, and secure transfers of files over long distances:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .transferAcceleration(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To access the bucket that is enabled for Transfer Acceleration, you must use a special endpoint. The URL can be generated using method <code>transferAccelerationUrlForObject</code>:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .transferAcceleration(true)
 *         .build();
 * bucket.transferAccelerationUrlForObject("objectname");
 * </pre></blockquote>
 * <p>
 * <h2>Intelligent Tiering</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/intelligent-tiering.html">Intelligent Tiering</a> can be configured to automatically move files to glacier:
 * <p>
 * <blockquote><pre>
 * Bucket.Builder.create(this, "MyBucket")
 *         .intelligentTieringConfigurations(List.of(IntelligentTieringConfiguration.builder()
 *                 .name("foo")
 *                 .prefix("folder/name")
 *                 .archiveAccessTierTime(Duration.days(90))
 *                 .deepArchiveAccessTierTime(Duration.days(180))
 *                 .tags(List.of(Tag.builder().key("tagname").value("tagvalue").build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Lifecycle Rule</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lifecycle-mgmt.html">Managing lifecycle</a> can be configured transition or expiration actions.
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .lifecycleRules(List.of(LifecycleRule.builder()
 *                 .abortIncompleteMultipartUploadAfter(Duration.minutes(30))
 *                 .enabled(false)
 *                 .expiration(Duration.days(30))
 *                 .expirationDate(new Date())
 *                 .expiredObjectDeleteMarker(false)
 *                 .id("id")
 *                 .noncurrentVersionExpiration(Duration.days(30))
 * 
 *                 // the properties below are optional
 *                 .noncurrentVersionsToRetain(123)
 *                 .noncurrentVersionTransitions(List.of(NoncurrentVersionTransition.builder()
 *                         .storageClass(StorageClass.GLACIER)
 *                         .transitionAfter(Duration.days(30))
 * 
 *                         // the properties below are optional
 *                         .noncurrentVersionsToRetain(123)
 *                         .build()))
 *                 .objectSizeGreaterThan(500)
 *                 .prefix("prefix")
 *                 .objectSizeLessThan(10000)
 *                 .transitions(List.of(Transition.builder()
 *                         .storageClass(StorageClass.GLACIER)
 * 
 *                         // exactly one of transitionAfter or transitionDate must be specified
 *                         .transitionAfter(Duration.days(30))
 *                         .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To indicate which default minimum object size behavior is applied to the lifecycle configuration, use the
 * <code>transitionDefaultMinimumObjectSize</code> property.
 * <p>
 * The default value of the property before September 2024 is <code>TransitionDefaultMinimumObjectSize.VARIES_BY_STORAGE_CLASS</code>
 * that allows objects smaller than 128 KB to be transitioned only to the S3 Glacier and S3 Glacier Deep Archive storage classes,
 * otherwise <code>TransitionDefaultMinimumObjectSize.ALL_STORAGE_CLASSES_128_K</code> that prevents objects smaller than 128 KB from being
 * transitioned to any storage class.
 * <p>
 * To customize the minimum object size for any transition you
 * can add a filter that specifies a custom <code>objectSizeGreaterThan</code> or <code>objectSizeLessThan</code> for <code>lifecycleRules</code>
 * property. Custom filters always take precedence over the default transition behavior.
 * <p>
 * <blockquote><pre>
 * Bucket.Builder.create(this, "MyBucket")
 *         .transitionDefaultMinimumObjectSize(TransitionDefaultMinimumObjectSize.VARIES_BY_STORAGE_CLASS)
 *         .lifecycleRules(List.of(LifecycleRule.builder()
 *                 .transitions(List.of(Transition.builder()
 *                         .storageClass(StorageClass.DEEP_ARCHIVE)
 *                         .transitionAfter(Duration.days(30))
 *                         .build()))
 *                 .build(), LifecycleRule.builder()
 *                 .objectSizeLessThan(300000)
 *                 .objectSizeGreaterThan(200000)
 *                 .transitions(List.of(Transition.builder()
 *                         .storageClass(StorageClass.ONE_ZONE_INFREQUENT_ACCESS)
 *                         .transitionAfter(Duration.days(30))
 *                         .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Object Lock Configuration</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html">Object Lock</a>
 * can be configured to enable a write-once-read-many model for an S3 bucket. Object Lock must be
 * configured when a bucket is created; if a bucket is created without Object Lock, it cannot be
 * enabled later via the CDK.
 * <p>
 * Object Lock can be enabled on an S3 bucket by specifying:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .objectLockEnabled(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Usually, it is desired to not just enable Object Lock for a bucket but to also configure a
 * <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html#object-lock-retention-modes">retention mode</a>
 * and a <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html#object-lock-retention-periods">retention period</a>.
 * These can be specified by providing <code>objectLockDefaultRetention</code>:
 * <p>
 * <blockquote><pre>
 * // Configure for governance mode with a duration of 7 years
 * // Configure for governance mode with a duration of 7 years
 * Bucket.Builder.create(this, "Bucket1")
 *         .objectLockDefaultRetention(ObjectLockRetention.governance(Duration.days(7 * 365)))
 *         .build();
 * 
 * // Configure for compliance mode with a duration of 1 year
 * // Configure for compliance mode with a duration of 1 year
 * Bucket.Builder.create(this, "Bucket2")
 *         .objectLockDefaultRetention(ObjectLockRetention.compliance(Duration.days(365)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Replicating Objects</h2>
 * <p>
 * You can use <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/replication.html">replicating objects</a> to enable automatic, asynchronous copying of objects across Amazon S3 buckets.
 * Buckets that are configured for object replication can be owned by the same AWS account or by different accounts.
 * You can replicate objects to a single destination bucket or to multiple destination buckets.
 * The destination buckets can be in different AWS Regions or within the same Region as the source bucket.
 * <p>
 * To replicate objects to a destination bucket, you can specify the <code>replicationRules</code> property:
 * <p>
 * <blockquote><pre>
 * IBucket destinationBucket1;
 * IBucket destinationBucket2;
 * IRole replicationRole;
 * IKey kmsKey;
 * 
 * 
 * Bucket sourceBucket = Bucket.Builder.create(this, "SourceBucket")
 *         // Versioning must be enabled on both the source and destination bucket
 *         .versioned(true)
 *         // Optional. If not specified, a new role will be created.
 *         .replicationRole(replicationRole)
 *         .replicationRules(List.of(ReplicationRule.builder()
 *                 // The destination bucket for the replication rule.
 *                 .destination(destinationBucket1)
 *                 // The priority of the rule.
 *                 // Amazon S3 will attempt to replicate objects according to all replication rules.
 *                 // However, if there are two or more rules with the same destination bucket, then objects will be replicated according to the rule with the highest priority.
 *                 // The higher the number, the higher the priority.
 *                 // It is essential to specify priority explicitly when the replication configuration has multiple rules.
 *                 .priority(1)
 *                 .build(), ReplicationRule.builder()
 *                 .destination(destinationBucket2)
 *                 .priority(2)
 *                 // Whether to specify S3 Replication Time Control (S3 RTC).
 *                 // S3 RTC replicates most objects that you upload to Amazon S3 in seconds,
 *                 // and 99.99 percent of those objects within specified time.
 *                 .replicationTimeControl(ReplicationTimeValue.FIFTEEN_MINUTES)
 *                 // Whether to enable replication metrics about S3 RTC.
 *                 // If set, metrics will be output to indicate whether replication by S3 RTC took longer than the configured time.
 *                 .metrics(ReplicationTimeValue.FIFTEEN_MINUTES)
 *                 // The kms key to use for the destination bucket.
 *                 .kmsKey(kmsKey)
 *                 // The storage class to use for the destination bucket.
 *                 .storageClass(StorageClass.INFREQUENT_ACCESS)
 *                 // Whether to replicate objects with SSE-KMS encryption.
 *                 .sseKmsEncryptedObjects(false)
 *                 // Whether to replicate modifications on replicas.
 *                 .replicaModifications(true)
 *                 // Whether to replicate delete markers.
 *                 // This property cannot be enabled if the replication rule has a tag filter.
 *                 .deleteMarkerReplication(false)
 *                 // The ID of the rule.
 *                 .id("full-settings-rule")
 *                 // The object filter for the rule.
 *                 .filter(Filter.builder()
 *                         // The prefix filter for the rule.
 *                         .prefix("prefix")
 *                         // The tag filter for the rule.
 *                         .tags(List.of(Tag.builder()
 *                                 .key("tagKey")
 *                                 .value("tagValue")
 *                                 .build()))
 *                         .build())
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cross Account Replication</h3>
 * <p>
 * You can also set a destination bucket from a different account as the replication destination.
 * <p>
 * In this case, the bucket policy for the destination bucket is required, to configure it through CDK use  <code>addReplicationPolicy()</code> method to add bucket policy on destination bucket.
 * In a cross-account scenario, where the source and destination buckets are owned by different AWS accounts, you can use a KMS key to encrypt object replicas. However, the KMS key owner must grant the source bucket owner permission to use the KMS key.
 * For more information, please refer to https://docs.aws.amazon.com/AmazonS3/latest/userguide/replication-walkthrough-2.html .
 * <p>
 * <blockquote>
 * <p>
 * <strong>NOTE:</strong> AWS managed keys don't allow cross-account use, and therefore can't be used to perform cross-account replication.
 * <p>
 * </blockquote>
 * <p>
 * If you need to override the bucket ownership to destination account pass the account value to the method to provide permissions to override bucket owner.
 * <code>addReplicationPolicy(bucket.replicationRoleArn, true, '11111111111')</code>;
 * <p>
 * However, if the destination bucket is a referenced bucket, CDK cannot set the bucket policy,
 * so you will need to <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/replication-walkthrough-2.html">configure the necessary bucket policy</a> separately.
 * <p>
 * <blockquote><pre>
 * // The destination bucket in a different account.
 * IBucket destinationBucket;
 * IRole replicationRole;
 * 
 * Bucket sourceBucket = Bucket.Builder.create(this, "SourceBucket")
 *         .versioned(true)
 *         // Optional. If not specified, a new role will be created.
 *         .replicationRole(replicationRole)
 *         .replicationRules(List.of(ReplicationRule.builder()
 *                 .destination(destinationBucket)
 *                 .priority(1)
 *                 // Whether to want to change replica ownership to the AWS account that owns the destination bucket.
 *                 // The replicas are owned by same AWS account that owns the source object by default.
 *                 .accessControlTransition(true)
 *                 .build()))
 *         .build();
 * 
 * //Add permissions to the destination after replication role is created
 * if (sourceBucket.getReplicationRoleArn()) {
 *     destinationBucket.addReplicationPolicy(sourceBucket.getReplicationRoleArn(), true, "111111111111");
 * }
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.s3;
