/**
 * <h1>AWS Systems Manager Construct Library</h1>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Using existing SSM Parameters in your CDK app</h2>
 * <p>
 * You can reference existing SSM Parameter Store values that you want to use in
 * your CDK app by using <code>ssm.StringParameter.fromStringParameterAttributes</code>:
 * <p>
 * <blockquote><pre>
 * Number parameterVersion = Token.asNumber(Map.of("Ref", "MyParameter"));
 * 
 * // Retrieve the latest value of the non-secret parameter
 * // with name "/My/String/Parameter".
 * String stringValue = StringParameter.fromStringParameterAttributes(this, "MyValue", StringParameterAttributes.builder()
 *         .parameterName("/My/Public/Parameter")
 *         .build()).getStringValue();
 * String stringValueVersionFromToken = StringParameter.fromStringParameterAttributes(this, "MyValueVersionFromToken", StringParameterAttributes.builder()
 *         .parameterName("/My/Public/Parameter")
 *         // parameter version from token
 *         .version(parameterVersion)
 *         .build()).getStringValue();
 * 
 * // Retrieve a specific version of the secret (SecureString) parameter.
 * // 'version' is always required.
 * IStringParameter secretValue = StringParameter.fromSecureStringParameterAttributes(this, "MySecureValue", SecureStringParameterAttributes.builder()
 *         .parameterName("/My/Secret/Parameter")
 *         .version(5)
 *         .build());
 * IStringParameter secretValueVersionFromToken = StringParameter.fromSecureStringParameterAttributes(this, "MySecureValueVersionFromToken", SecureStringParameterAttributes.builder()
 *         .parameterName("/My/Secret/Parameter")
 *         // parameter version from token
 *         .version(parameterVersion)
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also reference an existing SSM Parameter Store value that matches an
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html#aws-specific-parameter-types">AWS specific parameter type</a>:
 * <p>
 * <blockquote><pre>
 * StringParameter.valueForTypedStringParameterV2(this, "/My/Public/Parameter", ParameterValueType.AWS_EC2_IMAGE_ID);
 * </pre></blockquote>
 * <p>
 * To do the same for a SSM Parameter Store value that is stored as a list:
 * <p>
 * <blockquote><pre>
 * StringListParameter.valueForTypedListParameter(this, "/My/Public/Parameter", ParameterValueType.AWS_EC2_IMAGE_ID);
 * </pre></blockquote>
 * <p>
 * <h3>Lookup existing parameters</h3>
 * <p>
 * You can also use an existing parameter by looking up the parameter from the AWS environment.
 * This method uses AWS API calls to lookup the value from SSM during synthesis.
 * <p>
 * <blockquote><pre>
 * String stringValue = StringParameter.valueFromLookup(this, "/My/Public/Parameter");
 * </pre></blockquote>
 * <p>
 * When using <code>valueFromLookup</code> an initial value of 'dummy-value-for-${parameterName}'
 * (<code>dummy-value-for-/My/Public/Parameter</code> in the above example)
 * is returned prior to the lookup being performed. This can lead to errors if you are using this
 * value in places that require a certain format. For example if you have stored the ARN for a SNS
 * topic in a SSM Parameter which you want to lookup and provide to <code>Topic.fromTopicArn()</code>
 * <p>
 * <blockquote><pre>
 * String arnLookup = StringParameter.valueFromLookup(this, "/my/topic/arn");
 * Topic.fromTopicArn(this, "Topic", arnLookup);
 * </pre></blockquote>
 * <p>
 * Initially <code>arnLookup</code> will be equal to <code>dummy-value-for-/my/topic/arn</code> which will cause
 * <code>Topic.fromTopicArn</code> to throw an error indicating that the value is not in <code>arn</code> format.
 * <p>
 * For these use cases you need to handle the <code>dummy-value</code> in your code. For example:
 * <p>
 * <blockquote><pre>
 * String arnLookup = StringParameter.valueFromLookup(this, "/my/topic/arn");
 * String arnLookupValue;
 * if (arnLookup.includes("dummy-value")) {
 *     arnLookupValue = this.formatArn(ArnComponents.builder()
 *             .service("sns")
 *             .resource("topic")
 *             .resourceName(arnLookup)
 *             .build());
 * } else {
 *     arnLookupValue = arnLookup;
 * }
 * 
 * Topic.fromTopicArn(this, "Topic", arnLookupValue);
 * </pre></blockquote>
 * <p>
 * Alternatively, if the property supports tokens you can convert the parameter value into a token
 * to be resolved <em>after</em> the lookup has been completed.
 * <p>
 * <blockquote><pre>
 * String arnLookup = StringParameter.valueFromLookup(this, "/my/role/arn");
 * Role.fromRoleArn(this, "role", Lazy.string(Map.of("produce", () =&gt; arnLookup)));
 * </pre></blockquote>
 * <p>
 * <h3>cross-account SSM Parameters sharing</h3>
 * <p>
 * AWS Systems Manager (SSM) Parameter Store supports cross-account sharing of parameters using the AWS Resource Access Manager (AWS RAM)
 * service. In a multi-account environment, this feature enables accounts (referred to as "consuming accounts") to access and retrieve
 * parameter values that are shared by other accounts (referred to as "sharing accounts"). To reference and use a shared SSM parameter
 * in a consuming account, the <code>fromStringParameterArn()</code> method can be employed.
 * <p>
 * The <code>fromStringParameterArn()</code> method provides a way for consuming accounts to create an instance of the StringParameter
 * class from the Amazon Resource Name (ARN) of a shared SSM parameter. This allows the consuming account to retrieve and utilize the
 * parameter value, even though the parameter itself is owned and managed by a different sharing account.
 * <p>
 * <blockquote><pre>
 * String sharingParameterArn = "arn:aws:ssm:us-east-1:1234567890:parameter/dummyName";
 * IStringParameter sharedParam = StringParameter.fromStringParameterArn(this, "SharedParam", sharingParameterArn);
 * </pre></blockquote>
 * <p>
 * Things to note:
 * <p>
 * <ul>
 * <li>The account that owns the AWS Systems Manager (SSM) parameter and wants to share it with other accounts (referred to as the "sharing account") must create the parameter in the advanced tier. This is a prerequisite for sharing SSM parameters across accounts.</li>
 * <li>After creating the parameter in the advanced tier, the sharing account needs to set up a resource share using AWS Resource Access Manager (RAM). This resource share will specify the SSM parameter(s) to be shared and the accounts (referred to as "consuming accounts") with which the parameter(s) should be shared.</li>
 * <li>Once the resource share is created by the sharing account, the consuming account(s) will receive an invitation to join the resource share. For the consuming account(s) to access and use the shared SSM parameter(s), they must accept the resource share invitation from the sharing account.</li>
 * <li>The AWS Systems Manager Parameter Store parameter being referenced must be located in the same AWS region as the AWS CDK stack that is consuming or using the parameter.</li>
 * </ul>
 * <p>
 * In summary, the process involves three main steps:
 * <p>
 * <ol>
 * <li>The sharing account creates the SSM parameter(s) in the advanced tier.</li>
 * <li>The sharing account creates a resource share using AWS RAM, specifying the SSM parameter(s) and the consuming account(s).</li>
 * <li>The consuming account(s) accept the resource share invitation to gain access to the shared SSM parameter(s).</li>
 * </ol>
 * <p>
 * This cross-account sharing mechanism allows for centralized management and distribution of configuration data (stored as SSM parameters) across multiple AWS accounts within an organization or between different organizations.
 * <p>
 * Read <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-shared-parameters.html">Working with shared parameters</a> for more details.
 * <p>
 * <h2>Creating new SSM Parameters in your CDK app</h2>
 * <p>
 * You can create either <code>ssm.StringParameter</code> or <code>ssm.StringListParameter</code>s in
 * a CDK app. These are public (not secret) values. Parameters of type
 * <em>SecureString</em> cannot be created directly from a CDK application; if you want
 * to provision secrets automatically, use Secrets Manager Secrets (see the
 * <code>aws-cdk-lib/aws-secretsmanager</code> package).
 * <p>
 * <blockquote><pre>
 * StringParameter.Builder.create(this, "Parameter")
 *         .allowedPattern(".*")
 *         .description("The value Foo")
 *         .parameterName("FooParameter")
 *         .stringValue("Foo")
 *         .tier(ParameterTier.ADVANCED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * // Grant read access to some Role
 * IRole role;
 * // Create a new SSM Parameter holding a String
 * StringParameter param = StringParameter.Builder.create(this, "StringParameter")
 *         // description: 'Some user-friendly description',
 *         // name: 'ParameterName',
 *         .stringValue("Initial parameter value")
 *         .build();
 * param.grantRead(role);
 * 
 * // Create a new SSM Parameter holding a StringList
 * StringListParameter listParameter = StringListParameter.Builder.create(this, "StringListParameter")
 *         // description: 'Some user-friendly description',
 *         // name: 'ParameterName',
 *         .stringListValue(List.of("Initial parameter value A", "Initial parameter value B"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When specifying an <code>allowedPattern</code>, the values provided as string literals
 * are validated against the pattern and an exception is raised if a value
 * provided does not comply.
 * <p>
 * <h2>Using Tokens in parameter name</h2>
 * <p>
 * When using <a href="https://docs.aws.amazon.com/cdk/v2/guide/tokens.html">CDK Tokens</a> in parameter name,
 * you need to explicitly set the <code>simpleName</code> property. Setting <code>simpleName</code> to an incorrect boolean
 * value may result in unexpected behaviours, such as having duplicate '/' in the parameter ARN
 * or missing a '/' in the parameter ARN.
 * <p>
 * <code>simpleName</code> is used to indicates whether the parameter name is a simple name. A parameter name
 * without any '/' is considered a simple name, thus you should set <code>simpleName</code> to <code>true</code>.
 * If the parameter name includes '/', set <code>simpleName</code> to <code>false</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * IFunction func;
 * 
 * 
 * StringParameter simpleParameter = StringParameter.Builder.create(this, "StringParameter")
 *         // the parameter name doesn't contain any '/'
 *         .parameterName("parameter")
 *         .stringValue("SOME_VALUE")
 *         .simpleName(true)
 *         .build();
 * StringParameter nonSimpleParameter = StringParameter.Builder.create(this, "StringParameter")
 *         // the parameter name contains '/'
 *         .parameterName(String.format("/%s/my/app/param", func.getFunctionName()))
 *         .stringValue("SOME_VALUE")
 *         .simpleName(false)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ssm;
