/**
 * <h1>Amazon Data Firehose Construct Library</h1>
 * <p>
 * <a href="https://docs.aws.amazon.com/firehose/latest/dev/what-is-this-service.html">Amazon Data Firehose</a>, <a href="https://aws.amazon.com/about-aws/whats-new/2024/02/amazon-data-firehose-formerly-kinesis-data-firehose/">formerly known as Amazon Kinesis Data Firehose</a>,
 * is a service for fully-managed delivery of real-time streaming data to storage services
 * such as Amazon S3, Amazon Redshift, Amazon Elasticsearch, Splunk, or any custom HTTP
 * endpoint or third-party services such as Datadog, Dynatrace, LogicMonitor, MongoDB, New
 * Relic, and Sumo Logic.
 * <p>
 * Amazon Data Firehose delivery streams are distinguished from Kinesis data streams in
 * their models of consumption. Whereas consumers read from a data stream by actively pulling
 * data from the stream, a delivery stream pushes data to its destination on a regular
 * cadence. This means that data streams are intended to have consumers that do on-demand
 * processing, like AWS Lambda or Amazon EC2. On the other hand, delivery streams are
 * intended to have destinations that are sources for offline processing and analytics, such
 * as Amazon S3 and Amazon Redshift.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a>
 * project. It allows you to define Amazon Data Firehose delivery streams.
 * <p>
 * <h2>Defining a Delivery Stream</h2>
 * <p>
 * In order to define a Delivery Stream, you must specify a destination. An S3 bucket can be
 * used as a destination. Currently the CDK supports only S3 as a destination which is covered <a href="#destinations">below</a>.
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "Bucket");
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(new S3Bucket(bucket))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The above example defines the following resources:
 * <p>
 * <ul>
 * <li>An S3 bucket</li>
 * <li>An Amazon Data Firehose delivery stream with Direct PUT as the source and CloudWatch
 * error logging turned on.</li>
 * <li>An IAM role which gives the delivery stream permission to write to the S3 bucket.</li>
 * </ul>
 * <p>
 * <h2>Sources</h2>
 * <p>
 * An Amazon Data Firehose delivery stream can accept data from three main sources: Kinesis Data Streams, Managed Streaming for Apache Kafka (MSK), or via a "direct put" (API calls). Currently only Kinesis Data Streams and direct put are supported in the CDK.
 * <p>
 * See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/basic-write.html">Sending Data to a Delivery Stream</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.
 * <p>
 * <h3>Kinesis Data Stream</h3>
 * <p>
 * A delivery stream can read directly from a Kinesis data stream as a consumer of the data
 * stream. Configure this behaviour by passing in a data stream in the <code>source</code>
 * property via the <code>KinesisStreamSource</code> class when constructing a delivery stream:
 * <p>
 * <blockquote><pre>
 * IDestination destination;
 * 
 * Stream sourceStream = new Stream(this, "Source Stream");
 * 
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .source(new KinesisStreamSource(sourceStream))
 *         .destination(destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Direct Put</h3>
 * <p>
 * Data must be provided via "direct put", ie., by using a <code>PutRecord</code> or
 * <code>PutRecordBatch</code> API call. There are a number of ways of doing so, such as:
 * <p>
 * <ul>
 * <li>Kinesis Agent: a standalone Java application that monitors and delivers files while
 * handling file rotation, checkpointing, and retries. See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/writing-with-agents.html">Writing to Amazon Data Firehose Using Kinesis Agent</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.</li>
 * <li>AWS SDK: a general purpose solution that allows you to deliver data to a delivery stream
 * from anywhere using Java, .NET, Node.js, Python, or Ruby. See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/writing-with-sdk.html">Writing to Amazon Data Firehose Using the AWS SDK</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.</li>
 * <li>CloudWatch Logs: subscribe to a log group and receive filtered log events directly into
 * a delivery stream. See: <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-logs-destinations-readme.html">logs-destinations</a>.</li>
 * <li>Eventbridge: add an event rule target to send events to a delivery stream based on the
 * rule filtering. See: <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-events-targets-readme.html">events-targets</a>.</li>
 * <li>SNS: add a subscription to send all notifications from the topic to a delivery
 * stream. See: <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-sns-subscriptions-readme.html">sns-subscriptions</a>.</li>
 * <li>IoT: add an action to an IoT rule to send various IoT information to a delivery stream</li>
 * </ul>
 * <p>
 * <h2>Destinations</h2>
 * <p>
 * Amazon Data Firehose supports multiple AWS and third-party services as destinations, including Amazon S3, Amazon Redshift, and more. You can find the full list of supported destination <a href="https://docs.aws.amazon.com/firehose/latest/dev/create-destination.html">here</a>.
 * <p>
 * Currently in the AWS CDK, only S3 is implemented as an L2 construct destination. Other destinations can still be configured using L1 constructs.
 * <p>
 * <h3>S3</h3>
 * <p>
 * Defining a delivery stream with an S3 bucket destination:
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * 
 * S3Bucket s3Destination = new S3Bucket(bucket);
 * 
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(s3Destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The S3 destination also supports custom dynamic prefixes. <code>dataOutputPrefix</code>
 * will be used for files successfully delivered to S3. <code>errorOutputPrefix</code> will be added to
 * failed records before writing them to S3.
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * 
 * S3Bucket s3Destination = S3Bucket.Builder.create(bucket)
 *         .dataOutputPrefix("myFirehose/DeliveredYear=!{timestamp:yyyy}/anyMonth/rand=!{firehose:random-string}")
 *         .errorOutputPrefix("myFirehoseFailures/!{firehose:error-output-type}/!{timestamp:yyyy}/anyMonth/!{timestamp:dd}")
 *         .build();
 * </pre></blockquote>
 * <p>
 * See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/s3-prefixes.html">Custom S3 Prefixes</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.
 * <p>
 * To override default file extension appended by Data Format Conversion or S3 compression features, specify <code>fileExtension</code>.
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * 
 * S3Bucket s3Destination = S3Bucket.Builder.create(bucket)
 *         .compression(Compression.GZIP)
 *         .fileExtension(".json.gz")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Server-side Encryption</h2>
 * <p>
 * Enabling server-side encryption (SSE) requires Amazon Data Firehose to encrypt all data
 * sent to delivery stream when it is stored at rest. This means that data is encrypted
 * before being written to the service's internal storage layer and decrypted after it is
 * received from the internal storage layer. The service manages keys and cryptographic
 * operations so that sources and destinations do not need to, as the data is encrypted and
 * decrypted at the boundaries of the service (i.e., before the data is delivered to a
 * destination). By default, delivery streams do not have SSE enabled.
 * <p>
 * The Key Management Service keys (KMS keys) used for SSE can either be AWS-owned or
 * customer-managed. AWS-owned KMS keys are created, owned and managed by AWS for use in
 * multiple AWS accounts. As a customer, you cannot view, use, track, or manage these keys,
 * and you are not charged for their use. On the other hand, customer-managed KMS keys are
 * created and owned within your account and managed entirely by you. As a customer, you are
 * responsible for managing access, rotation, aliases, and deletion for these keys, and you
 * are changed for their use.
 * <p>
 * See: <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#kms_keys">AWS KMS keys</a>
 * in the <em>KMS Developer Guide</em>.
 * <p>
 * <blockquote><pre>
 * IDestination destination;
 * // SSE with an customer-managed key that is explicitly specified
 * Key key;
 * 
 * 
 * // SSE with an AWS-owned key
 * // SSE with an AWS-owned key
 * DeliveryStream.Builder.create(this, "Delivery Stream with AWS Owned Key")
 *         .encryption(StreamEncryption.awsOwnedKey())
 *         .destination(destination)
 *         .build();
 * // SSE with an customer-managed key that is created automatically by the CDK
 * // SSE with an customer-managed key that is created automatically by the CDK
 * DeliveryStream.Builder.create(this, "Delivery Stream with Customer Managed Key")
 *         .encryption(StreamEncryption.customerManagedKey())
 *         .destination(destination)
 *         .build();
 * DeliveryStream.Builder.create(this, "Delivery Stream with Customer Managed and Provided Key")
 *         .encryption(StreamEncryption.customerManagedKey(key))
 *         .destination(destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/encryption.html">Data Protection</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.
 * <p>
 * <h2>Monitoring</h2>
 * <p>
 * Amazon Data Firehose is integrated with CloudWatch, so you can monitor the performance of
 * your delivery streams via logs and metrics.
 * <p>
 * <h3>Logs</h3>
 * <p>
 * Amazon Data Firehose will send logs to CloudWatch when data transformation or data
 * delivery fails. The CDK will enable logging by default and create a CloudWatch LogGroup
 * and LogStream with default settings for your Delivery Stream.
 * <p>
 * When creating a destination, you can provide an <code>ILoggingConfig</code>, which can either be an <code>EnableLogging</code> or <code>DisableLogging</code> instance.
 * If you use <code>EnableLogging</code>, the CDK will create a CloudWatch LogGroup and LogStream with all CloudFormation default settings for you, or you can optionally
 * specify your own log group to be used for capturing and storing log events. For example:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * Bucket bucket;
 * 
 * 
 * LogGroup logGroup = new LogGroup(this, "Log Group");
 * S3Bucket destination = S3Bucket.Builder.create(bucket)
 *         .loggingConfig(new EnableLogging(logGroup))
 *         .build();
 * 
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Logging can also be disabled:
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * 
 * S3Bucket destination = S3Bucket.Builder.create(bucket)
 *         .loggingConfig(new DisableLogging())
 *         .build();
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/monitoring-with-cloudwatch-logs.html">Monitoring using CloudWatch Logs</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.
 * <p>
 * <h3>Metrics</h3>
 * <p>
 * Amazon Data Firehose sends metrics to CloudWatch so that you can collect and analyze the
 * performance of the delivery stream, including data delivery, data ingestion, data
 * transformation, format conversion, API usage, encryption, and resource usage. You can then
 * use CloudWatch alarms to alert you, for example, when data freshness (the age of the
 * oldest record in the delivery stream) exceeds the buffering limit (indicating that data is
 * not being delivered to your destination), or when the rate of incoming records exceeds the
 * limit of records per second (indicating data is flowing into your delivery stream faster
 * than it is configured to process).
 * <p>
 * CDK provides methods for accessing delivery stream metrics with default configuration,
 * such as <code>metricIncomingBytes</code>, and <code>metricIncomingRecords</code> (see <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-cdk-lib.aws_kinesisfirehose.IDeliveryStream.html"><code>IDeliveryStream</code></a>
 * for a full list). CDK also provides a generic <code>metric</code> method that can be used to produce
 * metric configurations for any metric provided by Amazon Data Firehose; the configurations
 * are pre-populated with the correct dimensions for the delivery stream.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * DeliveryStream deliveryStream;
 * 
 * 
 * // Alarm that triggers when the per-second average of incoming bytes exceeds 90% of the current service limit
 * MathExpression incomingBytesPercentOfLimit = MathExpression.Builder.create()
 *         .expression("incomingBytes / 300 / bytePerSecLimit")
 *         .usingMetrics(Map.of(
 *                 "incomingBytes", deliveryStream.metricIncomingBytes(MetricOptions.builder().statistic(Statistic.SUM).build()),
 *                 "bytePerSecLimit", deliveryStream.metric("BytesPerSecondLimit")))
 *         .build();
 * 
 * Alarm.Builder.create(this, "Alarm")
 *         .metric(incomingBytesPercentOfLimit)
 *         .threshold(0.9)
 *         .evaluationPeriods(3)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/monitoring-with-cloudwatch-metrics.html">Monitoring Using CloudWatch Metrics</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.
 * <p>
 * <h2>Compression</h2>
 * <p>
 * Your data can automatically be compressed when it is delivered to S3 as either a final or
 * an intermediary/backup destination. Supported compression formats are: gzip, Snappy,
 * Hadoop-compatible Snappy, and ZIP, except for Redshift destinations, where Snappy
 * (regardless of Hadoop-compatibility) and ZIP are not supported. By default, data is
 * delivered to S3 without compression.
 * <p>
 * <blockquote><pre>
 * // Compress data delivered to S3 using Snappy
 * Bucket bucket;
 * 
 * S3Bucket s3Destination = S3Bucket.Builder.create(bucket)
 *         .compression(Compression.SNAPPY)
 *         .build();
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(s3Destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Buffering</h2>
 * <p>
 * Incoming data is buffered before it is delivered to the specified destination. The
 * delivery stream will wait until the amount of incoming data has exceeded some threshold
 * (the "buffer size") or until the time since the last data delivery occurred exceeds some
 * threshold (the "buffer interval"), whichever happens first. You can configure these
 * thresholds based on the capabilities of the destination and your use-case. By default, the
 * buffer size is 5 MiB and the buffer interval is 5 minutes.
 * <p>
 * <blockquote><pre>
 * // Increase the buffer interval and size to 10 minutes and 8 MiB, respectively
 * Bucket bucket;
 * 
 * S3Bucket destination = S3Bucket.Builder.create(bucket)
 *         .bufferingInterval(Duration.minutes(10))
 *         .bufferingSize(Size.mebibytes(8))
 *         .build();
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/basic-deliver.html#frequency">Data Delivery Frequency</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.
 * <p>
 * Zero buffering, where Amazon Data Firehose stream can be configured to not buffer data before delivery, is supported by
 * setting the "buffer interval" to 0.
 * <p>
 * <blockquote><pre>
 * // Setup zero buffering
 * Bucket bucket;
 * 
 * S3Bucket destination = S3Bucket.Builder.create(bucket)
 *         .bufferingInterval(Duration.seconds(0))
 *         .build();
 * DeliveryStream.Builder.create(this, "ZeroBufferDeliveryStream")
 *         .destination(destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/buffering-hints.html">Buffering Hints</a>.
 * <p>
 * <h2>Destination Encryption</h2>
 * <p>
 * Your data can be automatically encrypted when it is delivered to S3 as a final or an
 * intermediary/backup destination. Amazon Data Firehose supports Amazon S3 server-side
 * encryption with AWS Key Management Service (AWS KMS) for encrypting delivered data in
 * Amazon S3. You can choose to not encrypt the data or to encrypt with a key from the list
 * of AWS KMS keys that you own. For more information,
 * see <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html">Protecting Data Using Server-Side Encryption with AWS KMS–Managed Keys (SSE-KMS)</a>.
 * By default, encryption isn’t directly enabled on the delivery stream; instead, it uses the default encryption settings of the destination S3 bucket.
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * Key key;
 * 
 * S3Bucket destination = S3Bucket.Builder.create(bucket)
 *         .encryptionKey(key)
 *         .build();
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Backup</h2>
 * <p>
 * A delivery stream can be configured to back up data to S3 that it attempted to deliver to
 * the configured destination. Backed up data can be all the data that the delivery stream
 * attempted to deliver or just data that it failed to deliver (Redshift and S3 destinations
 * can only back up all data). CDK can create a new S3 bucket where it will back up data, or
 * you can provide a bucket where data will be backed up. You can also provide a prefix under
 * which your backed-up data will be placed within the bucket. By default, source data is not
 * backed up to S3.
 * <p>
 * <blockquote><pre>
 * // Enable backup of all source records (to an S3 bucket created by CDK).
 * Bucket bucket;
 * // Explicitly provide an S3 bucket to which all source records will be backed up.
 * Bucket backupBucket;
 * 
 * DeliveryStream.Builder.create(this, "Delivery Stream Backup All")
 *         .destination(
 *         S3Bucket.Builder.create(bucket)
 *                 .s3Backup(DestinationS3BackupProps.builder()
 *                         .mode(BackupMode.ALL)
 *                         .build())
 *                 .build())
 *         .build();
 * DeliveryStream.Builder.create(this, "Delivery Stream Backup All Explicit Bucket")
 *         .destination(
 *         S3Bucket.Builder.create(bucket)
 *                 .s3Backup(DestinationS3BackupProps.builder()
 *                         .bucket(backupBucket)
 *                         .build())
 *                 .build())
 *         .build();
 * // Explicitly provide an S3 prefix under which all source records will be backed up.
 * // Explicitly provide an S3 prefix under which all source records will be backed up.
 * DeliveryStream.Builder.create(this, "Delivery Stream Backup All Explicit Prefix")
 *         .destination(
 *         S3Bucket.Builder.create(bucket)
 *                 .s3Backup(DestinationS3BackupProps.builder()
 *                         .mode(BackupMode.ALL)
 *                         .dataOutputPrefix("mybackup")
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If any Data Processing or Transformation is configured on your Delivery Stream, the source
 * records will be backed up in their original format.
 * <p>
 * <h2>Data Processing/Transformation</h2>
 * <p>
 * Data can be transformed before being delivered to destinations. There are two types of
 * data processing for delivery streams: record transformation with AWS Lambda, and record
 * format conversion using a schema stored in an AWS Glue table. If both types of data
 * processing are configured, then the Lambda transformation is performed first. By default,
 * no data processing occurs. This construct library currently only supports data
 * transformation with AWS Lambda. See <a href="https://github.com/aws/aws-cdk/issues/15501">#15501</a>
 * to track the status of adding support for record format conversion.
 * <p>
 * <h3>Data transformation with AWS Lambda</h3>
 * <p>
 * To transform the data, Amazon Data Firehose will call a Lambda function that you provide
 * and deliver the data returned in place of the source record. The function must return a
 * result that contains records in a specific format, including the following fields:
 * <p>
 * <ul>
 * <li><code>recordId</code> -- the ID of the input record that corresponds the results.</li>
 * <li><code>result</code> -- the status of the transformation of the record: "Ok" (success), "Dropped"
 * (not processed intentionally), or "ProcessingFailed" (not processed due to an error).</li>
 * <li><code>data</code> -- the transformed data, Base64-encoded.</li>
 * </ul>
 * <p>
 * The data is buffered up to 1 minute and up to 3 MiB by default before being sent to the
 * function, but can be configured using <code>bufferInterval</code> and <code>bufferSize</code>
 * in the processor configuration (see: <a href="#buffering">Buffering</a>). If the function invocation
 * fails due to a network timeout or because of hitting an invocation limit, the invocation
 * is retried 3 times by default, but can be configured using <code>retries</code> in the processor
 * configuration.
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * // Provide a Lambda function that will transform records before delivery, with custom
 * // buffering and retry configuration
 * Function lambdaFunction = Function.Builder.create(this, "Processor")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "process-records")))
 *         .build();
 * LambdaFunctionProcessor lambdaProcessor = LambdaFunctionProcessor.Builder.create(lambdaFunction)
 *         .bufferInterval(Duration.minutes(5))
 *         .bufferSize(Size.mebibytes(5))
 *         .retries(5)
 *         .build();
 * S3Bucket s3Destination = S3Bucket.Builder.create(bucket)
 *         .processor(lambdaProcessor)
 *         .build();
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(s3Destination)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * import path.*;
 * import software.amazon.awscdk.services.kinesisfirehose.*;
 * import software.amazon.awscdk.services.kms.*;
 * import software.amazon.awscdk.services.lambda.nodejs.*;
 * import software.amazon.awscdk.services.logs.*;
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.integtests.alpha.AwsApiCall;
 * import software.amazon.awscdk.integtests.alpha.ExpectedResult;
 * import software.amazon.awscdk.integtests.alpha.IntegTest;
 * 
 * App app = App.Builder.create()
 *         .postCliContext(Map.of(
 *                 "&#64;aws-cdk/aws-lambda:useCdkManagedLogGroup", false))
 *         .build();
 * 
 * Stack stack = new Stack(app, "aws-cdk-firehose-delivery-stream-s3-all-properties");
 * 
 * Bucket bucket = Bucket.Builder.create(stack, "FirehoseDeliveryStreamS3AllPropertiesBucket")
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .autoDeleteObjects(true)
 *         .build();
 * 
 * Bucket backupBucket = Bucket.Builder.create(stack, "FirehoseDeliveryStreamS3AllPropertiesBackupBucket")
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .autoDeleteObjects(true)
 *         .build();
 * LogGroup logGroup = LogGroup.Builder.create(stack, "LogGroup")
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .build();
 * 
 * NodejsFunction dataProcessorFunction = NodejsFunction.Builder.create(stack, "DataProcessorFunction")
 *         .entry(join(__dirname, "lambda-data-processor.js"))
 *         .timeout(Duration.minutes(1))
 *         .build();
 * 
 * LambdaFunctionProcessor processor = LambdaFunctionProcessor.Builder.create(dataProcessorFunction)
 *         .bufferInterval(Duration.seconds(60))
 *         .bufferSize(Size.mebibytes(1))
 *         .retries(1)
 *         .build();
 * 
 * Key key = Key.Builder.create(stack, "Key")
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .build();
 * 
 * Key backupKey = Key.Builder.create(stack, "BackupKey")
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .build();
 * 
 * DeliveryStream deliveryStream = DeliveryStream.Builder.create(stack, "DeliveryStream")
 *         .destination(S3Bucket.Builder.create(bucket)
 *                 .loggingConfig(new EnableLogging(logGroup))
 *                 .processor(processor)
 *                 .compression(Compression.GZIP)
 *                 .dataOutputPrefix("regularPrefix")
 *                 .errorOutputPrefix("errorPrefix")
 *                 .fileExtension(".log.gz")
 *                 .bufferingInterval(Duration.seconds(60))
 *                 .bufferingSize(Size.mebibytes(1))
 *                 .encryptionKey(key)
 *                 .s3Backup(DestinationS3BackupProps.builder()
 *                         .mode(BackupMode.ALL)
 *                         .bucket(backupBucket)
 *                         .compression(Compression.ZIP)
 *                         .dataOutputPrefix("backupPrefix")
 *                         .errorOutputPrefix("backupErrorPrefix")
 *                         .bufferingInterval(Duration.seconds(60))
 *                         .bufferingSize(Size.mebibytes(1))
 *                         .encryptionKey(backupKey)
 *                         .build())
 *                 .build())
 *         .build();
 * 
 * DeliveryStream.Builder.create(stack, "ZeroBufferingDeliveryStream")
 *         .destination(S3Bucket.Builder.create(bucket)
 *                 .compression(Compression.GZIP)
 *                 .dataOutputPrefix("regularPrefix")
 *                 .errorOutputPrefix("errorPrefix")
 *                 .bufferingInterval(Duration.seconds(0))
 *                 .build())
 *         .build();
 * 
 * IntegTest testCase = IntegTest.Builder.create(app, "integ-tests")
 *         .testCases(List.of(stack))
 *         .regions(List.of("us-east-1"))
 *         .build();
 * 
 * testCase.assertions.awsApiCall("Firehose", "putRecord", Map.of(
 *         "DeliveryStreamName", deliveryStream.getDeliveryStreamName(),
 *         "Record", Map.of(
 *                 "Data", "testData123")));
 * 
 * IApiCall s3ApiCall = testCase.assertions.awsApiCall("S3", "listObjectsV2", Map.of(
 *         "Bucket", bucket.bucketName,
 *         "MaxKeys", 1)).expect(ExpectedResult.objectLike(Map.of(
 *         "KeyCount", 1))).waitForAssertions(WaiterStateMachineOptions.builder()
 *         .interval(Duration.seconds(30))
 *         .totalTimeout(Duration.minutes(10))
 *         .build());
 * 
 * if (s3ApiCall instanceof AwsApiCall &amp;&amp; s3ApiCall.getWaiterProvider()) {
 *     s3ApiCall.waiterProvider.addToRolePolicy(Map.of(
 *             "Effect", "Allow",
 *             "Action", List.of("s3:GetObject", "s3:ListBucket"),
 *             "Resource", List.of("*")));
 * }
 * </pre></blockquote>
 * <p>
 * See: <a href="https://docs.aws.amazon.com/firehose/latest/dev/data-transformation.html">Data Transformation</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.
 * <p>
 * <h2>Specifying an IAM role</h2>
 * <p>
 * The DeliveryStream class automatically creates IAM service roles with all the minimum
 * necessary permissions for Amazon Data Firehose to access the resources referenced by your
 * delivery stream. One service role is created for the delivery stream that allows Amazon
 * Data Firehose to read from a Kinesis data stream (if one is configured as the delivery
 * stream source) and for server-side encryption. Note that if the DeliveryStream is created
 * without specifying a <code>source</code> or <code>encryptionKey</code>, this role is not created as it is not needed.
 * <p>
 * Another service role is created for each destination, which gives Amazon Data Firehose write
 * access to the destination resource, as well as the ability to invoke data transformers and
 * read schemas for record format conversion. If you wish, you may specify your own IAM role for
 * either the delivery stream or the destination service role, or both. It must have the correct
 * trust policy (it must allow Amazon Data Firehose to assume it) or delivery stream creation or
 * data delivery will fail. Other required permissions to destination resources, encryption keys, etc.,
 * will be provided automatically.
 * <p>
 * <blockquote><pre>
 * // Specify the roles created above when defining the destination and delivery stream.
 * Bucket bucket;
 * // Create service roles for the delivery stream and destination.
 * // These can be used for other purposes and granted access to different resources.
 * // They must include the Amazon Data Firehose service principal in their trust policies.
 * // Two separate roles are shown below, but the same role can be used for both purposes.
 * Role deliveryStreamRole = Role.Builder.create(this, "Delivery Stream Role")
 *         .assumedBy(new ServicePrincipal("firehose.amazonaws.com"))
 *         .build();
 * Role destinationRole = Role.Builder.create(this, "Destination Role")
 *         .assumedBy(new ServicePrincipal("firehose.amazonaws.com"))
 *         .build();
 * S3Bucket destination = S3Bucket.Builder.create(bucket).role(destinationRole).build();
 * DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(destination)
 *         .role(deliveryStreamRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/firehose/latest/dev/controlling-access.html">Controlling Access</a>
 * in the <em>Amazon Data Firehose Developer Guide</em>.
 * <p>
 * <h2>Granting application access to a delivery stream</h2>
 * <p>
 * IAM roles, users or groups which need to be able to work with delivery streams should be
 * granted IAM permissions.
 * <p>
 * Any object that implements the <code>IGrantable</code> interface (i.e., has an associated principal)
 * can be granted permissions to a delivery stream by calling:
 * <p>
 * <ul>
 * <li><code>grantPutRecords(principal)</code> - grants the principal the ability to put records onto the
 * delivery stream</li>
 * <li><code>grant(principal, ...actions)</code> - grants the principal permission to a custom set of
 * actions</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Give the role permissions to write data to the delivery stream
 * DeliveryStream deliveryStream;
 * Role lambdaRole = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * deliveryStream.grantPutRecords(lambdaRole);
 * </pre></blockquote>
 * <p>
 * The following write permissions are provided to a service principal by the
 * <code>grantPutRecords()</code> method:
 * <p>
 * <ul>
 * <li><code>firehose:PutRecord</code></li>
 * <li><code>firehose:PutRecordBatch</code></li>
 * </ul>
 * <p>
 * <h2>Granting a delivery stream access to a resource</h2>
 * <p>
 * Conversely to the above, Amazon Data Firehose requires permissions in order for delivery
 * streams to interact with resources that you own. For example, if an S3 bucket is specified
 * as a destination of a delivery stream, the delivery stream must be granted permissions to
 * put and get objects from the bucket. When using the built-in AWS service destinations, the CDK grants the
 * permissions automatically. However, custom or third-party destinations may require custom
 * permissions. In this case, use the delivery stream as an <code>IGrantable</code>, as follows:
 * <p>
 * <blockquote><pre>
 * DeliveryStream deliveryStream;
 * Function fn = Function.Builder.create(this, "Function")
 *         .code(Code.fromInline("exports.handler = (event) =&gt; {}"))
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .build();
 * fn.grantInvoke(deliveryStream);
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.kinesisfirehose;
