/**
 * <h1>AWS Lambda Construct Library</h1>
 * <p>
 * This construct library allows you to define AWS Lambda Functions.
 * <p>
 * <blockquote><pre>
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When deployed, this construct creates or updates an existing
 * <code>AWS::Lambda::Function</code> resource. When updating, AWS CloudFormation calls the
 * <a href="https://docs.aws.amazon.com/lambda/latest/api/API_UpdateFunctionConfiguration.html">UpdateFunctionConfiguration</a>
 * and <a href="https://docs.aws.amazon.com/lambda/latest/api/API_UpdateFunctionCode.html">UpdateFunctionCode</a>
 * Lambda APIs under the hood. Because these calls happen sequentially, and
 * invocations can happen between these calls, your function may encounter errors
 * in the time between the calls. For example, if you update an existing Lambda
 * function by removing an environment variable and the code that references that
 * environment variable in the same CDK deployment, you may see invocation errors
 * related to a missing environment variable. To work around this, you can invoke
 * your function against a version or alias by default, rather than the <code>$LATEST</code>
 * version.
 * <p>
 * To further mitigate these issues, you can ensure consistency between your function code and infrastructure configuration by defining environment variables as a single source of truth in your CDK stack. You can define them in a separate <code>env.ts</code> file and reference them in both your handler and CDK configuration. This approach allows you to catch errors at compile time, benefit from improved IDE support, minimize the risk of mismatched configurations, and enhance maintainability.
 * <p>
 * <h2>Handler Code</h2>
 * <p>
 * The <code>lambda.Code</code> class includes static convenience methods for various types of
 * runtime code.
 * <p>
 * <ul>
 * <li><code>lambda.Code.fromBucket(bucket, key, objectVersion)</code> - specify an S3 object
 * that contains the archive of your runtime code.</li>
 * <li><code>lambda.Code.fromBucketV2(bucket, key, {objectVersion: version, sourceKMSKey: key})</code> - specify an S3 object
 * that contains the archive of your runtime code.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.Key;
 * import software.amazon.awscdk.services.s3.*;
 * Key key;
 * 
 * 
 * Bucket bucket = new Bucket(this, "Bucket");
 * 
 * Map&lt;String, Key&gt; options = Map.of(
 *         "sourceKMSKey", key);
 * Function fnBucket = Function.Builder.create(this, "myFunction2")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromBucketV2(bucket, "python-lambda-handler.zip", options))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>lambda.Code.fromInline(code)</code> - inline the handle code as a string. This is
 * limited to supported runtimes.</li>
 * <li><code>lambda.Code.fromAsset(path)</code> - specify a directory or a .zip file in the local
 * filesystem which will be zipped and uploaded to S3 before deployment. See also
 * <a href="#bundling-asset-code">bundling asset code</a>.</li>
 * <li><code>lambda.Code.fromDockerBuild(path, options)</code> - use the result of a Docker
 * build as code. The runtime code is expected to be located at <code>/asset</code> in the
 * image and will be zipped and uploaded to S3 as an asset.</li>
 * <li><code>lambda.Code.fromCustomCommand(output, command, customCommandOptions)</code> -
 * supply a command that is invoked during cdk synth. That command is meant to direct
 * the generated code to output (a zip file or a directory), which is then used as the
 * code for the created AWS Lambda.</li>
 * </ul>
 * <p>
 * The following example shows how to define a Python function and deploy the code
 * from the local directory <code>my-lambda-handler</code> to it:
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "MyLambda")
 *         .code(Code.fromAsset(join(__dirname, "my-lambda-handler")))
 *         .handler("index.main")
 *         .runtime(Runtime.PYTHON_3_9)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When deploying a stack that contains this code, the directory will be zip
 * archived and then uploaded to an S3 bucket, then the exact location of the S3
 * objects will be passed when the stack is deployed.
 * <p>
 * During synthesis, the CDK expects to find a directory on disk at the asset
 * directory specified. Note that we are referencing the asset directory relatively
 * to our CDK project directory. This is especially important when we want to share
 * this construct through a library. Different programming languages will have
 * different techniques for bundling resources into libraries.
 * <p>
 * <h2>Docker Images</h2>
 * <p>
 * Lambda functions allow specifying their handlers within docker images. The docker
 * image can be an image from ECR or a local asset that the CDK will package and load
 * into ECR.
 * <p>
 * The following <code>DockerImageFunction</code> construct uses a local folder with a
 * Dockerfile as the asset that will be used as the function handler.
 * <p>
 * <blockquote><pre>
 * DockerImageFunction.Builder.create(this, "AssetFunction")
 *         .code(DockerImageCode.fromImageAsset(join(__dirname, "docker-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also specify an image that already exists in ECR as the function handler.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.*;
 * 
 * Repository repo = new Repository(this, "Repository");
 * 
 * DockerImageFunction.Builder.create(this, "ECRFunction")
 *         .code(DockerImageCode.fromEcr(repo))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The props for these docker image resources allow overriding the image's <code>CMD</code>, <code>ENTRYPOINT</code>, and <code>WORKDIR</code>
 * configurations as well as choosing a specific tag or digest. See their docs for more information.
 * <p>
 * To deploy a <code>DockerImageFunction</code> on Lambda <code>arm64</code> architecture, specify <code>Architecture.ARM_64</code> in <code>architecture</code>.
 * This will bundle docker image assets for <code>arm64</code> architecture with <code>--platform linux/arm64</code> even if build within an <code>x86_64</code> host.
 * <p>
 * With that being said, if you are bundling <code>DockerImageFunction</code> for Lambda <code>amd64</code> architecture from a <code>arm64</code> machine like a Macbook with <code>arm64</code> CPU, you would
 * need to specify <code>architecture: lambda.Architecture.X86_64</code> as well. This ensures the <code>--platform</code> argument is passed to the image assets
 * bundling process so you can bundle up <code>X86_64</code> images from the <code>arm64</code> machine.
 * <p>
 * <blockquote><pre>
 * DockerImageFunction.Builder.create(this, "AssetFunction")
 *         .code(DockerImageCode.fromImageAsset(join(__dirname, "docker-arm64-handler")))
 *         .architecture(Architecture.ARM_64)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Execution Role</h2>
 * <p>
 * Lambda functions assume an IAM role during execution. In CDK by default, Lambda
 * functions will use an autogenerated Role if one is not provided.
 * <p>
 * The autogenerated Role is automatically given permissions to execute the Lambda
 * function. To reference the autogenerated Role:
 * <p>
 * <blockquote><pre>
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * 
 * IRole role = fn.getRole();
 * </pre></blockquote>
 * <p>
 * You can also provide your own IAM role. Provided IAM roles will not automatically
 * be given permissions to execute the Lambda function. To provide a role and grant
 * it appropriate permissions:
 * <p>
 * <blockquote><pre>
 * Role myRole = Role.Builder.create(this, "My Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * 
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .role(myRole)
 *         .build();
 * 
 * myRole.addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaBasicExecutionRole"));
 * myRole.addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaVPCAccessExecutionRole"));
 * </pre></blockquote>
 * <p>
 * <h2>Function Timeout</h2>
 * <p>
 * AWS Lambda functions have a default timeout of 3 seconds, but this can be increased
 * up to 15 minutes. The timeout is available as a property of <code>Function</code> so that
 * you can reference it elsewhere in your stack. For instance, you could use it to create
 * a CloudWatch alarm to report when your function timed out:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * 
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .timeout(Duration.minutes(5))
 *         .build();
 * 
 * if (fn.getTimeout()) {
 *     Alarm.Builder.create(this, "MyAlarm")
 *             .metric(fn.metricDuration().with(MetricOptions.builder()
 *                     .statistic("Maximum")
 *                     .build()))
 *             .evaluationPeriods(1)
 *             .datapointsToAlarm(1)
 *             .threshold(fn.timeout.toMilliseconds())
 *             .treatMissingData(TreatMissingData.IGNORE)
 *             .alarmName("My Lambda Timeout")
 *             .build();
 * }
 * </pre></blockquote>
 * <p>
 * <h2>Advanced Logging</h2>
 * <p>
 * You can have more control over your function logs, by specifying the log format
 * (Json or plain text), the system log level, the application log level, as well
 * as choosing the log group:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.ILogGroup;
 * 
 * ILogGroup logGroup;
 * 
 * 
 * Function.Builder.create(this, "Lambda")
 *         .code(new InlineCode("foo"))
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .loggingFormat(LoggingFormat.JSON)
 *         .systemLogLevelV2(SystemLogLevel.INFO)
 *         .applicationLogLevelV2(ApplicationLogLevel.INFO)
 *         .logGroup(logGroup)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To use <code>applicationLogLevelV2</code> and/or <code>systemLogLevelV2</code> you must set <code>loggingFormat</code> to <code>LoggingFormat.JSON</code>.
 * <p>
 * <h2>Customizing Log Group Creation</h2>
 * <p>
 * <h3>Log Group Creation Methods</h3>
 * <p>
 * | Method | Description | Tag Propagation | Prop Injection | Aspects | Notes |
 * |--------|-------------|-----------------|----------------|---------|-------|
 * | <strong>logRetention prop</strong> | Legacy approach using Custom Resource | False | False | False | Does not support TPA |
 * | <strong>logGroup prop</strong> | Explicitly supplied by user in CDK app | True | True | True | Full support for TPA |
 * | <strong>Lazy creation</strong> | Lambda service creates logGroup on first invocation | False | False | False | Occurs when both logRetention and logGroup are undefined and USE_CDK_MANAGED_LAMBDA_LOGGROUP is false |
 * | <strong>USE_CDK_MANAGED_LAMBDA_LOGGROUP</strong> | CDK Lambda function construct creates log group with default props | True | True | True | Feature flag must be enabled |
 * <p>
 * <em>TPA: Tag propagation, Prop Injection, Aspects</em>
 * <p>
 * <h4>Order of precedence</h4>
 * <p>
 * <blockquote><pre>
 *                        Highest Precedence
 *                              |
 *              +---------------+---------------+
 *              |                               |
 *   +-------------------------+      +------------------------+
 *   |   logRetention is set   |      |     logGroup is set    |
 *   +-----------+-------------+      +----------+-------------+
 *               |                               |
 *               v                               v
 *       Create LogGroup via            Use the provided LogGroup
 *   Custom Resource (retention       instance (CDK-managed,
 *   managed, logGroup disallowed)    logRetention disallowed)
 *               |                               |
 *               +---------------+---------------+
 *                               |
 *                               v
 *           +-----------------------------------------------+
 *           |         Feature flag enabled:                 |
 *           | aws-cdk:aws-lambda:useCdkManagedLogGroup      |
 *           +------------------ +---------------------------+
 *                               |
 *                               v
 *               Create LogGroup at synth time
 *           (CDK-managed, default settings for logGroup)
 *                               |
 *                               v
 *                   +---------------------------+
 *                   | Default (no config set)   |
 *                   +------------+--------------+
 *                               |
 *                               v
 *      Lambda service creates log group on first invocation runtime
 *             (CDK does not manage the log group resource)
 * </pre></blockquote>
 * <p>
 * <h3>Tag Propagation</h3>
 * <p>
 * Refer section <code>Log Group Creation Methods</code> to find out which modes support tag propagation.
 * As an example, adding the following line in your cdk app will also propagate to the logGroup.
 * <p>
 * <blockquote><pre>
 * const fn = new lambda.Function(this, 'MyFunctionWithFFTrue', {
 *   runtime: lambda.Runtime.NODEJS_20_X,
 *   handler: 'handler.main',
 *   code: lambda.Code.fromAsset('lambda'),
 * });
 * cdk.Tags.of(fn).add('env', 'dev'); // the tag is also added to the log group
 * </pre></blockquote>
 * <p>
 * <h2>Resource-based Policies</h2>
 * <p>
 * AWS Lambda supports resource-based policies for controlling access to Lambda
 * functions and layers on a per-resource basis. In particular, this allows you to
 * give permission to AWS services, AWS Organizations, or other AWS accounts to
 * modify and invoke your functions.
 * <p>
 * <h3>Grant function access to AWS services</h3>
 * <p>
 * <blockquote><pre>
 * // Grant permissions to a service
 * Function fn;
 * 
 * ServicePrincipal principal = new ServicePrincipal("my-service");
 * 
 * fn.grantInvoke(principal);
 * 
 * // Equivalent to:
 * fn.addPermission("my-service Invocation", Permission.builder()
 *         .principal(principal)
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can also restrict permissions given to AWS services by providing
 * a source account or ARN (representing the account and identifier of the resource
 * that accesses the function or layer).
 * <p>
 * <strong>Important</strong>:
 * <p>
 * <blockquote>
 * <p>
 * By default <code>fn.grantInvoke()</code> grants permission to the principal to invoke any version of the function, including all past ones. If you only want the principal to be granted permission to invoke the latest version or the unqualified Lambda ARN, use <code>grantInvokeLatestVersion(grantee)</code>.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * ServicePrincipal principal = new ServicePrincipal("my-service");
 * // Grant invoke only to latest version and unqualified lambda arn
 * fn.grantInvokeLatestVersion(principal);
 * </pre></blockquote>
 * <p>
 * If you want to grant access for invoking a specific version of Lambda function, you can use <code>fn.grantInvokeVersion(grantee, version)</code>
 * <p>
 * <blockquote><pre>
 * Function fn;
 * IVersion version;
 * 
 * ServicePrincipal principal = new ServicePrincipal("my-service");
 * // Grant invoke only to the specific version
 * fn.grantInvokeVersion(principal, version);
 * </pre></blockquote>
 * <p>
 * For more information, see
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html#permissions-resource-serviceinvoke">Granting function access to AWS services</a>
 * in the AWS Lambda Developer Guide.
 * <p>
 * <h3>Grant function access to an AWS Organization</h3>
 * <p>
 * <blockquote><pre>
 * // Grant permissions to an entire AWS organization
 * Function fn;
 * 
 * OrganizationPrincipal org = new OrganizationPrincipal("o-xxxxxxxxxx");
 * 
 * fn.grantInvoke(org);
 * </pre></blockquote>
 * <p>
 * In the above example, the <code>principal</code> will be <code>*</code> and all users in the
 * organization <code>o-xxxxxxxxxx</code> will get function invocation permissions.
 * <p>
 * You can restrict permissions given to the organization by specifying an
 * AWS account or role as the <code>principal</code>:
 * <p>
 * <blockquote><pre>
 * // Grant permission to an account ONLY IF they are part of the organization
 * Function fn;
 * 
 * AccountPrincipal account = new AccountPrincipal("123456789012");
 * 
 * fn.grantInvoke(account.inOrganization("o-xxxxxxxxxx"));
 * </pre></blockquote>
 * <p>
 * For more information, see
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html#permissions-resource-xorginvoke">Granting function access to an organization</a>
 * in the AWS Lambda Developer Guide.
 * <p>
 * <h3>Grant function access to other AWS accounts</h3>
 * <p>
 * <blockquote><pre>
 * // Grant permission to other AWS account
 * Function fn;
 * 
 * AccountPrincipal account = new AccountPrincipal("123456789012");
 * 
 * fn.grantInvoke(account);
 * </pre></blockquote>
 * <p>
 * For more information, see
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html#permissions-resource-xaccountinvoke">Granting function access to other accounts</a>
 * in the AWS Lambda Developer Guide.
 * <p>
 * <h3>Grant function access to unowned principals</h3>
 * <p>
 * Providing an unowned principal (such as account principals, generic ARN
 * principals, service principals, and principals in other accounts) to a call to
 * <code>fn.grantInvoke</code> will result in a resource-based policy being created. If the
 * principal in question has conditions limiting the source account or ARN of the
 * operation (see above), these conditions will be automatically added to the
 * resource policy.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * ServicePrincipal servicePrincipal = new ServicePrincipal("my-service");
 * String sourceArn = "arn:aws:s3:::amzn-s3-demo-bucket";
 * String sourceAccount = "111122223333";
 * PrincipalBase servicePrincipalWithConditions = servicePrincipal.withConditions(Map.of(
 *         "ArnLike", Map.of(
 *                 "aws:SourceArn", sourceArn),
 *         "StringEquals", Map.of(
 *                 "aws:SourceAccount", sourceAccount)));
 * 
 * fn.grantInvoke(servicePrincipalWithConditions);
 * </pre></blockquote>
 * <p>
 * <h3>Grant function access to a CompositePrincipal</h3>
 * <p>
 * To grant invoke permissions to a <code>CompositePrincipal</code> use the <code>grantInvokeCompositePrincipal</code> method:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * CompositePrincipal compositePrincipal = new CompositePrincipal(
 * new OrganizationPrincipal("o-zzzzzzzzzz"),
 * new ServicePrincipal("apigateway.amazonaws.com"));
 * 
 * fn.grantInvokeCompositePrincipal(compositePrincipal);
 * </pre></blockquote>
 * <p>
 * <h2>Versions</h2>
 * <p>
 * You can use
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html">versions</a>
 * to manage the deployment of your AWS Lambda functions. For example, you can
 * publish a new version of a function for beta testing without affecting users of
 * the stable production version.
 * <p>
 * The function version includes the following information:
 * <p>
 * <ul>
 * <li>The function code and all associated dependencies.</li>
 * <li>The Lambda runtime that executes the function.</li>
 * <li>All of the function settings, including the environment variables.</li>
 * <li>A unique Amazon Resource Name (ARN) to identify this version of the function.</li>
 * </ul>
 * <p>
 * You could create a version to your lambda function using the <code>Version</code> construct.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * Version version = Version.Builder.create(this, "MyVersion")
 *         .lambda(fn)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The major caveat to know here is that a function version must always point to a
 * specific 'version' of the function. When the function is modified, the version
 * will continue to point to the 'then version' of the function.
 * <p>
 * One way to ensure that the <code>lambda.Version</code> always points to the latest version
 * of your <code>lambda.Function</code> is to set an environment variable which changes at
 * least as often as your code does. This makes sure the function always has the
 * latest code. For instance -
 * <p>
 * <blockquote><pre>
 * String codeVersion = "stringOrMethodToGetCodeVersion";
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .environment(Map.of(
 *                 "CodeVersionString", codeVersion))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>fn.latestVersion</code> property returns a <code>lambda.IVersion</code> which represents
 * the <code>$LATEST</code> pseudo-version.
 * <p>
 * However, most AWS services require a specific AWS Lambda version,
 * and won't allow you to use <code>$LATEST</code>. Therefore, you would normally want
 * to use <code>lambda.currentVersion</code>.
 * <p>
 * The <code>fn.currentVersion</code> property can be used to obtain a <code>lambda.Version</code>
 * resource that represents the AWS Lambda function defined in your application.
 * Any change to your function's code or configuration will result in the creation
 * of a new version resource. You can specify options for this version through the
 * <code>currentVersionOptions</code> property.
 * <p>
 * NOTE: The <code>currentVersion</code> property is only supported when your AWS Lambda function
 * uses either <code>lambda.Code.fromAsset</code> or <code>lambda.Code.fromInline</code>. Other types
 * of code providers (such as <code>lambda.Code.fromBucket</code>) require that you define a
 * <code>lambda.Version</code> resource directly since the CDK is unable to determine if
 * their contents had changed.
 * <p>
 * <h3><code>currentVersion</code>: Updated hashing logic</h3>
 * <p>
 * To produce a new lambda version each time the lambda function is modified, the
 * <code>currentVersion</code> property under the hood, computes a new logical id based on the
 * properties of the function. This informs CloudFormation that a new
 * <code>AWS::Lambda::Version</code> resource should be created pointing to the updated Lambda
 * function.
 * <p>
 * However, a bug was introduced in this calculation that caused the logical id to
 * change when it was not required (ex: when the Function's <code>Tags</code> property, or
 * when the <code>DependsOn</code> clause was modified). This caused the deployment to fail
 * since the Lambda service does not allow creating duplicate versions.
 * <p>
 * This has been fixed in the AWS CDK but <em>existing</em> users need to opt-in via a
 * <a href="https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html">feature flag</a>. Users who have run <code>cdk init</code> since this fix will be opted in,
 * by default.
 * <p>
 * Otherwise, you will need to enable the <a href="https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html">feature flag</a>
 * <code>&#64;aws-cdk/aws-lambda:recognizeVersionProps</code>. Since CloudFormation does not
 * allow duplicate versions, you will also need to make some modification to
 * your function so that a new version can be created. To efficiently and trivially
 * modify all your lambda functions at once, you can attach the
 * <code>FunctionVersionUpgrade</code> aspect to the stack, which slightly alters the
 * function description. This aspect is intended for one-time use to upgrade the
 * version of all your functions at the same time, and can safely be removed after
 * deploying once.
 * <p>
 * <blockquote><pre>
 * Stack stack = new Stack();
 * Aspects.of(stack).add(new FunctionVersionUpgrade(LAMBDA_RECOGNIZE_VERSION_PROPS));
 * </pre></blockquote>
 * <p>
 * When the new logic is in effect, you may rarely come across the following error:
 * <code>The following properties are not recognized as version properties</code>. This will
 * occur, typically when <a href="https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html#cfn_layer_raw">property overrides</a> are used, when a new property
 * introduced in <code>AWS::Lambda::Function</code> is used that CDK is still unaware of.
 * <p>
 * To overcome this error, use the API <code>Function.classifyVersionProperty()</code> to
 * record whether a new version should be generated when this property is changed.
 * This can be typically determined by checking whether the property can be
 * modified using the <em><a href="https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html">UpdateFunctionConfiguration</a></em> API or not.
 * <p>
 * <h3><code>currentVersion</code>: Updated hashing logic for layer versions</h3>
 * <p>
 * An additional update to the hashing logic fixes two issues surrounding layers.
 * Prior to this change, updating the lambda layer version would have no effect on
 * the function version. Also, the order of lambda layers provided to the function
 * was unnecessarily baked into the hash.
 * <p>
 * This has been fixed in the AWS CDK starting with version 2.27. If you ran
 * <code>cdk init</code> with an earlier version, you will need to opt-in via a <a href="https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html">feature flag</a>.
 * If you run <code>cdk init</code> with v2.27 or later, this fix will be opted in, by default.
 * <p>
 * Existing users will need to enable the <a href="https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html">feature flag</a>
 * <code>&#64;aws-cdk/aws-lambda:recognizeLayerVersion</code>. Since CloudFormation does not
 * allow duplicate versions, they will also need to make some modification to
 * their function so that a new version can be created. To efficiently and trivially
 * modify all your lambda functions at once, users can attach the
 * <code>FunctionVersionUpgrade</code> aspect to the stack, which slightly alters the
 * function description. This aspect is intended for one-time use to upgrade the
 * version of all your functions at the same time, and can safely be removed after
 * deploying once.
 * <p>
 * <blockquote><pre>
 * Stack stack = new Stack();
 * Aspects.of(stack).add(new FunctionVersionUpgrade(LAMBDA_RECOGNIZE_LAYER_VERSION));
 * </pre></blockquote>
 * <p>
 * <h2>Aliases</h2>
 * <p>
 * You can define one or more
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/configuration-aliases.html">aliases</a>
 * for your AWS Lambda function. A Lambda alias is like a pointer to a specific
 * Lambda function version. Users can access the function version using the alias
 * ARN.
 * <p>
 * The <code>version.addAlias()</code> method can be used to define an AWS Lambda alias that
 * points to a specific version.
 * <p>
 * The following example defines an alias named <code>live</code> which will always point to a
 * version that represents the function as defined in your CDK app. When you change
 * your lambda code or configuration, a new resource will be created. You can
 * specify options for the current version through the <code>currentVersionOptions</code>
 * property.
 * <p>
 * <blockquote><pre>
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .currentVersionOptions(VersionOptions.builder()
 *                 .removalPolicy(RemovalPolicy.RETAIN) // retain old versions
 *                 .retryAttempts(1)
 *                 .build())
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * 
 * fn.addAlias("live");
 * </pre></blockquote>
 * <p>
 * <h2>Function URL</h2>
 * <p>
 * A function URL is a dedicated HTTP(S) endpoint for your Lambda function. When you create a function URL, Lambda automatically generates a unique URL endpoint for you. Function URLs can be created for the latest version Lambda Functions, or Function Aliases (but not for Versions).
 * <p>
 * Function URLs are dual stack-enabled, supporting IPv4 and IPv6, and cross-origin resource sharing (CORS) configuration. After you configure a function URL for your function, you can invoke your function through its HTTP(S) endpoint via a web browser, curl, Postman, or any HTTP client. To invoke a function using IAM authentication your HTTP client must support SigV4 signing.
 * <p>
 * See the <a href="https://docs.aws.amazon.com/lambda/latest/dg/urls-invocation.html">Invoking Function URLs</a> section of the AWS Lambda Developer Guide
 * for more information on the input and output payloads of Functions invoked in this way.
 * <p>
 * <h3>IAM-authenticated Function URLs</h3>
 * <p>
 * To create a Function URL which can be called by an IAM identity, call <code>addFunctionUrl()</code>, followed by <code>grantInvokeFunctionUrl()</code>:
 * <p>
 * <blockquote><pre>
 * // Can be a Function or an Alias
 * Function fn;
 * Role myRole;
 * 
 * 
 * FunctionUrl fnUrl = fn.addFunctionUrl();
 * fnUrl.grantInvokeUrl(myRole);
 * 
 * CfnOutput.Builder.create(this, "TheUrl")
 *         // The .url attributes will return the unique Function URL
 *         .value(fnUrl.getUrl())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Calls to this URL need to be signed with SigV4.
 * <p>
 * <h3>Anonymous Function URLs</h3>
 * <p>
 * To create a Function URL which can be called anonymously, pass <code>authType: FunctionUrlAuthType.NONE</code> to <code>addFunctionUrl()</code>:
 * <p>
 * <blockquote><pre>
 * // Can be a Function or an Alias
 * Function fn;
 * 
 * 
 * FunctionUrl fnUrl = fn.addFunctionUrl(FunctionUrlOptions.builder()
 *         .authType(FunctionUrlAuthType.NONE)
 *         .build());
 * 
 * CfnOutput.Builder.create(this, "TheUrl")
 *         .value(fnUrl.getUrl())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>CORS configuration for Function URLs</h3>
 * <p>
 * If you want your Function URLs to be invokable from a web page in browser, you
 * will need to configure cross-origin resource sharing to allow the call (if you do
 * not do this, your browser will refuse to make the call):
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * fn.addFunctionUrl(FunctionUrlOptions.builder()
 *         .authType(FunctionUrlAuthType.NONE)
 *         .cors(FunctionUrlCorsOptions.builder()
 *                 // Allow this to be called from websites on https://example.com.
 *                 // Can also be ['*'] to allow all domain.
 *                 .allowedOrigins(List.of("https://example.com"))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Invoke Mode for Function URLs</h3>
 * <p>
 * Invoke mode determines how AWS Lambda invokes your function. You can configure the invoke mode when creating a Function URL using the invokeMode property
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * fn.addFunctionUrl(FunctionUrlOptions.builder()
 *         .authType(FunctionUrlAuthType.NONE)
 *         .invokeMode(InvokeMode.RESPONSE_STREAM)
 *         .build());
 * </pre></blockquote>
 * <p>
 * If the invokeMode property is not specified, the default BUFFERED mode will be used.
 * <p>
 * <h2>Layers</h2>
 * <p>
 * The <code>lambda.LayerVersion</code> class can be used to define Lambda layers and manage
 * granting permissions to other AWS accounts or organizations.
 * <p>
 * <blockquote><pre>
 * LayerVersion layer = LayerVersion.Builder.create(stack, "MyLayer")
 *         .code(Code.fromAsset(join(__dirname, "layer-code")))
 *         .compatibleRuntimes(List.of(Runtime.NODEJS_LATEST))
 *         .license("Apache-2.0")
 *         .description("A layer to test the L2 construct")
 *         .build();
 * 
 * // To grant usage by other AWS accounts
 * layer.addPermission("remote-account-grant", LayerVersionPermission.builder().accountId(awsAccountId).build());
 * 
 * // To grant usage to all accounts in some AWS Ogranization
 * // layer.grantUsage({ accountId: '*', organizationId });
 * 
 * // To grant usage to all accounts in some AWS Ogranization
 * // layer.grantUsage({ accountId: '*', organizationId });
 * Function.Builder.create(stack, "MyLayeredLambda")
 *         .code(new InlineCode("foo"))
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .layers(List.of(layer))
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, updating a layer creates a new layer version, and CloudFormation will delete the old version as part of the stack update.
 * <p>
 * Alternatively, a removal policy can be used to retain the old version:
 * <p>
 * <blockquote><pre>
 * LayerVersion.Builder.create(this, "MyLayer")
 *         .removalPolicy(RemovalPolicy.RETAIN)
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Architecture</h2>
 * <p>
 * Lambda functions, by default, run on compute systems that have the 64 bit x86 architecture.
 * <p>
 * The AWS Lambda service also runs compute on the ARM architecture, which can reduce cost
 * for some workloads.
 * <p>
 * A lambda function can be configured to be run on one of these platforms:
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .architecture(Architecture.ARM_64)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Similarly, lambda layer versions can also be tagged with architectures it is compatible with.
 * <p>
 * <blockquote><pre>
 * LayerVersion.Builder.create(this, "MyLayer")
 *         .removalPolicy(RemovalPolicy.RETAIN)
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .compatibleArchitectures(List.of(Architecture.X86_64, Architecture.ARM_64))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Lambda Insights</h2>
 * <p>
 * Lambda functions can be configured to use CloudWatch <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights.html">Lambda Insights</a>
 * which provides low-level runtime metrics for a Lambda functions.
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .insightsVersion(LambdaInsightsVersion.VERSION_1_0_98_0)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If the version of insights is not yet available in the CDK, you can also provide the ARN directly as so -
 * <p>
 * <blockquote><pre>
 * String layerArn = "arn:aws:lambda:us-east-1:580247275435:layer:LambdaInsightsExtension:14";
 * Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .insightsVersion(LambdaInsightsVersion.fromInsightVersionArn(layerArn))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you are deploying an ARM_64 Lambda Function, you must specify a
 * Lambda Insights Version &gt;= <code>1_0_119_0</code>.
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .architecture(Architecture.ARM_64)
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .insightsVersion(LambdaInsightsVersion.VERSION_1_0_119_0)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Parameters and Secrets Extension</h3>
 * <p>
 * Lambda functions can be configured to use the Parameters and Secrets Extension. The Parameters and Secrets Extension can be used to retrieve and cache <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/retrieving-secrets_lambda.html">secrets</a> from Secrets Manager or <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-integration-lambda-extensions.html">parameters</a> from Parameter Store in Lambda functions without using an SDK.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.secretsmanager.*;
 * import software.amazon.awscdk.services.ssm.*;
 * 
 * 
 * Secret secret = new Secret(this, "Secret");
 * StringParameter parameter = StringParameter.Builder.create(this, "Parameter")
 *         .parameterName("mySsmParameterName")
 *         .stringValue("mySsmParameterValue")
 *         .build();
 * 
 * ParamsAndSecretsLayerVersion paramsAndSecrets = ParamsAndSecretsLayerVersion.fromVersion(ParamsAndSecretsVersions.V1_0_103, ParamsAndSecretsOptions.builder()
 *         .cacheSize(500)
 *         .logLevel(ParamsAndSecretsLogLevel.DEBUG)
 *         .build());
 * 
 * Function lambdaFunction = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .architecture(Architecture.ARM_64)
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .paramsAndSecrets(paramsAndSecrets)
 *         .build();
 * 
 * secret.grantRead(lambdaFunction);
 * parameter.grantRead(lambdaFunction);
 * </pre></blockquote>
 * <p>
 * If the version of Parameters and Secrets Extension is not yet available in the CDK, you can also provide the ARN directly as so:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.secretsmanager.*;
 * import software.amazon.awscdk.services.ssm.*;
 * 
 * 
 * Secret secret = new Secret(this, "Secret");
 * StringParameter parameter = StringParameter.Builder.create(this, "Parameter")
 *         .parameterName("mySsmParameterName")
 *         .stringValue("mySsmParameterValue")
 *         .build();
 * 
 * String layerArn = "arn:aws:lambda:us-east-1:177933569100:layer:AWS-Parameters-and-Secrets-Lambda-Extension:4";
 * ParamsAndSecretsLayerVersion paramsAndSecrets = ParamsAndSecretsLayerVersion.fromVersionArn(layerArn, ParamsAndSecretsOptions.builder()
 *         .cacheSize(500)
 *         .build());
 * 
 * Function lambdaFunction = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .architecture(Architecture.ARM_64)
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .paramsAndSecrets(paramsAndSecrets)
 *         .build();
 * 
 * secret.grantRead(lambdaFunction);
 * parameter.grantRead(lambdaFunction);
 * </pre></blockquote>
 * <p>
 * <h2>Event Rule Target</h2>
 * <p>
 * You can use an AWS Lambda function as a target for an Amazon CloudWatch event
 * rule:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.events.*;
 * import software.amazon.awscdk.services.events.targets.*;
 * 
 * Function fn;
 * 
 * Rule rule = Rule.Builder.create(this, "Schedule Rule")
 *         .schedule(Schedule.cron(CronOptions.builder().minute("0").hour("4").build()))
 *         .build();
 * rule.addTarget(new LambdaFunction(fn));
 * </pre></blockquote>
 * <p>
 * <h2>Event Sources</h2>
 * <p>
 * AWS Lambda supports a <a href="https://docs.aws.amazon.com/lambda/latest/dg/invoking-lambda-function.html">variety of event sources</a>.
 * <p>
 * In most cases, it is possible to trigger a function as a result of an event by
 * using one of the <code>add&lt;Event&gt;Notification</code> methods on the source construct. For
 * example, the <code>s3.Bucket</code> construct has an <code>onEvent</code> method which can be used to
 * trigger a Lambda when an event, such as PutObject occurs on an S3 bucket.
 * <p>
 * An alternative way to add event sources to a function is to use <code>function.addEventSource(source)</code>.
 * This method accepts an <code>IEventSource</code> object. The module <strong>&#64;aws-cdk/aws-lambda-event-sources</strong>
 * includes classes for the various event sources supported by AWS Lambda.
 * <p>
 * For example, the following code adds an SQS queue as an event source for a function:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.*;
 * import software.amazon.awscdk.services.sqs.*;
 * 
 * Function fn;
 * 
 * Queue queue = new Queue(this, "Queue");
 * fn.addEventSource(new SqsEventSource(queue));
 * </pre></blockquote>
 * <p>
 * The following code adds an S3 bucket notification as an event source:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.*;
 * import software.amazon.awscdk.services.s3.*;
 * 
 * Function fn;
 * 
 * Bucket bucket = new Bucket(this, "Bucket");
 * fn.addEventSource(S3EventSource.Builder.create(bucket)
 *         .events(List.of(EventType.OBJECT_CREATED, EventType.OBJECT_REMOVED))
 *         .filters(List.of(NotificationKeyFilter.builder().prefix("subdir/").build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The following code adds an DynamoDB notification as an event source filtering insert events:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.*;
 * import software.amazon.awscdk.services.dynamodb.*;
 * 
 * Function fn;
 * 
 * Table table = Table.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .stream(StreamViewType.NEW_IMAGE)
 *         .build();
 * fn.addEventSource(DynamoEventSource.Builder.create(table)
 *         .startingPosition(StartingPosition.LATEST)
 *         .filters(List.of(FilterCriteria.filter(Map.of("eventName", FilterRule.isEqual("INSERT")))))
 *         .build());
 * </pre></blockquote>
 * <p>
 * By default, Lambda will encrypt Filter Criteria using AWS managed keys. But if you want to use a self managed KMS key to encrypt the filters, You can specify the self managed key using the <code>filterEncryption</code> property.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.*;
 * import software.amazon.awscdk.services.dynamodb.*;
 * import software.amazon.awscdk.services.kms.Key;
 * 
 * Function fn;
 * 
 * Table table = Table.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .stream(StreamViewType.NEW_IMAGE)
 *         .build();
 * // Your self managed KMS key
 * IKey myKey = Key.fromKeyArn(this, "SourceBucketEncryptionKey", "arn:aws:kms:us-east-1:123456789012:key/&lt;key-id&gt;");
 * 
 * fn.addEventSource(DynamoEventSource.Builder.create(table)
 *         .startingPosition(StartingPosition.LATEST)
 *         .filters(List.of(FilterCriteria.filter(Map.of("eventName", FilterRule.isEqual("INSERT")))))
 *         .filterEncryption(myKey)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Lambda requires allow <code>kms:Decrypt</code> on Lambda principal <code>lambda.amazonaws.com</code> to use the key for Filter Criteria Encryption. If you create the KMS key in the stack, CDK will automatically add this permission to the Key when you creates eventSourceMapping. However, if you import the key using function like <code>Key.fromKeyArn</code> then you need to add the following permission to the KMS key before using it to encrypt Filter Criteria
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * {
 *     "Version": "2012-10-17",
 *     "Statement": [
 *         {
 *             "Effect": "Allow",
 *             "Principal": {
 *                 "Service": "lambda.amazonaws.com"
 *             },
 *             "Action": "kms:Decrypt",
 *             "Resource": "*"
 *         }
 *     ]
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Observability</h3>
 * <p>
 * Customers can now opt-in to get enhanced metrics for their event source mapping that capture each stage of processing using the <code>MetricsConfig</code> property.
 * <p>
 * The following code shows how to opt in for the enhanced metrics.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.eventsources.*;
 * import software.amazon.awscdk.services.dynamodb.*;
 * 
 * Function fn;
 * 
 * Table table = Table.Builder.create(this, "Table")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .stream(StreamViewType.NEW_IMAGE)
 *         .build();
 * 
 * fn.addEventSource(DynamoEventSource.Builder.create(table)
 *         .startingPosition(StartingPosition.LATEST)
 *         .metricsConfig(MetricsConfig.builder()
 *                 .metrics(List.of(MetricType.EVENT_COUNT))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * See the documentation for the <strong>&#64;aws-cdk/aws-lambda-event-sources</strong> module for more details.
 * <p>
 * <h2>Imported Lambdas</h2>
 * <p>
 * When referencing an imported lambda in the CDK, use <code>fromFunctionArn()</code> for most use cases:
 * <p>
 * <blockquote><pre>
 * IFunction fn = Function.fromFunctionArn(this, "Function", "arn:aws:lambda:us-east-1:123456789012:function:MyFn");
 * </pre></blockquote>
 * <p>
 * The <code>fromFunctionAttributes()</code> API is available for more specific use cases:
 * <p>
 * <blockquote><pre>
 * IFunction fn = Function.fromFunctionAttributes(this, "Function", FunctionAttributes.builder()
 *         .functionArn("arn:aws:lambda:us-east-1:123456789012:function:MyFn")
 *         // The following are optional properties for specific use cases and should be used with caution:
 * 
 *         // Use Case: imported function is in the same account as the stack. This tells the CDK that it
 *         // can modify the function's permissions.
 *         .sameEnvironment(true)
 * 
 *         // Use Case: imported function is in a different account and user commits to ensuring that the
 *         // imported function has the correct permissions outside the CDK.
 *         .skipPermissions(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <code>Function.fromFunctionArn()</code> and <code>Function.fromFunctionAttributes()</code> will attempt to parse the Function's Region and Account ID from the ARN. <code>addPermissions</code> will only work on the <code>Function</code> object if the Region and Account ID are deterministically the same as the scope of the Stack the referenced <code>Function</code> object is created in.
 * If the containing Stack is environment-agnostic or the Function ARN is a Token, this comparison will fail, and calls to <code>Function.addPermission</code> will do nothing.
 * If you know Function permissions can safely be added, you can use <code>Function.fromFunctionName()</code> instead, or pass <code>sameEnvironment: true</code> to <code>Function.fromFunctionAttributes()</code>.
 * <p>
 * <blockquote><pre>
 * IFunction fn = Function.fromFunctionName(this, "Function", "MyFn");
 * </pre></blockquote>
 * <p>
 * <h2>Lambda with DLQ</h2>
 * <p>
 * A dead-letter queue can be automatically created for a Lambda function by
 * setting the <code>deadLetterQueueEnabled: true</code> configuration. In such case CDK creates
 * a <code>sqs.Queue</code> as <code>deadLetterQueue</code>.
 * <p>
 * <blockquote><pre>
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"))
 *         .deadLetterQueueEnabled(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * It is also possible to provide a dead-letter queue instead of getting a new queue created:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.sqs.*;
 * 
 * 
 * Queue dlq = new Queue(this, "DLQ");
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"))
 *         .deadLetterQueue(dlq)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also use a <code>sns.Topic</code> instead of an <code>sqs.Queue</code> as dead-letter queue:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.sns.*;
 * 
 * 
 * Topic dlt = new Topic(this, "DLQ");
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromInline("// your code here"))
 *         .deadLetterTopic(dlt)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/lambda/latest/dg/dlq.html">the AWS documentation</a>
 * to learn more about AWS Lambdas and DLQs.
 * <p>
 * <h2>Lambda with X-Ray Tracing</h2>
 * <p>
 * <blockquote><pre>
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"))
 *         .tracing(Tracing.ACTIVE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/lambda/latest/dg/lambda-x-ray.html">the AWS documentation</a>
 * to learn more about AWS Lambda's X-Ray support.
 * <p>
 * <h2>Lambda with AWS Distro for OpenTelemetry layer</h2>
 * <p>
 * To have automatic integration with XRay without having to add dependencies or change your code, you can use the
 * <a href="https://aws-otel.github.io/docs/getting-started/lambda">AWS Distro for OpenTelemetry Lambda (ADOT) layer</a>.
 * Consuming the latest ADOT layer can be done with the following snippet:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.AdotLambdaExecWrapper;
 * import software.amazon.awscdk.services.lambda.AdotLayerVersion;
 * import software.amazon.awscdk.services.lambda.AdotLambdaLayerJavaScriptSdkVersion;
 * 
 * 
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"))
 *         .adotInstrumentation(AdotInstrumentationConfig.builder()
 *                 .layerVersion(AdotLayerVersion.fromJavaScriptSdkLayerVersion(AdotLambdaLayerJavaScriptSdkVersion.LATEST))
 *                 .execWrapper(AdotLambdaExecWrapper.REGULAR_HANDLER)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * To use a different layer version, use one of the following helper functions for the <code>layerVersion</code> prop:
 * <p>
 * <ul>
 * <li><code>AdotLayerVersion.fromJavaScriptSdkLayerVersion</code></li>
 * <li><code>AdotLayerVersion.fromPythonSdkLayerVersion</code></li>
 * <li><code>AdotLayerVersion.fromJavaSdkLayerVersion</code></li>
 * <li><code>AdotLayerVersion.fromJavaAutoInstrumentationSdkLayerVersion</code></li>
 * <li><code>AdotLayerVersion.fromGenericSdkLayerVersion</code></li>
 * </ul>
 * <p>
 * Each helper function expects a version value from a corresponding enum-like class as below:
 * <p>
 * <ul>
 * <li><code>AdotLambdaLayerJavaScriptSdkVersion</code></li>
 * <li><code>AdotLambdaLayerPythonSdkVersion</code></li>
 * <li><code>AdotLambdaLayerJavaSdkVersion</code></li>
 * <li><code>AdotLambdaLayerJavaAutoInstrumentationSdkVersion</code></li>
 * <li><code>AdotLambdaLayerGenericSdkVersion</code></li>
 * </ul>
 * <p>
 * For more examples, see our <a href="test/integ.lambda-adot.ts">the integration test</a>.
 * <p>
 * If you want to retrieve the ARN of the ADOT Lambda layer without enabling ADOT in a Lambda function:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * String layerArn = AdotLambdaLayerJavaSdkVersion.V1_19_0.layerArn(fn.getStack(), fn.getArchitecture());
 * </pre></blockquote>
 * <p>
 * When using the <code>AdotLambdaLayerPythonSdkVersion</code> the <code>AdotLambdaExecWrapper</code> needs to be <code>AdotLambdaExecWrapper.INSTRUMENT_HANDLER</code> as per <a href="https://aws-otel.github.io/docs/getting-started/lambda/lambda-python">AWS Distro for OpenTelemetry Lambda Support For Python</a>
 * <p>
 * <h2>Lambda with Profiling</h2>
 * <p>
 * The following code configures the lambda function with CodeGuru profiling. By default, this creates a new CodeGuru
 * profiling group -
 * <p>
 * <blockquote><pre>
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.PYTHON_3_9)
 *         .handler("index.handler")
 *         .code(Code.fromAsset("lambda-handler"))
 *         .profiling(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>profilingGroup</code> property can be used to configure an existing CodeGuru profiler group.
 * <p>
 * CodeGuru profiling is supported for all Java runtimes and Python3.6+ runtimes.
 * <p>
 * See <a href="https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html">the AWS documentation</a>
 * to learn more about AWS Lambda's Profiling support.
 * <p>
 * <h2>Lambda with Reserved Concurrent Executions</h2>
 * <p>
 * <blockquote><pre>
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"))
 *         .reservedConcurrentExecutions(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * https://docs.aws.amazon.com/lambda/latest/dg/invocation-recursion.html
 * <p>
 * <h2>Lambda with SnapStart</h2>
 * <p>
 * SnapStart is currently supported on Python 3.12, Python 3.13, .NET 8, and Java 11 and later <a href="https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html">Java managed runtimes</a>. SnapStart does not support provisioned concurrency, Amazon Elastic File System (Amazon EFS), or ephemeral storage greater than 512 MB. After you enable Lambda SnapStart for a particular Lambda function, publishing a new version of the function will trigger an optimization process.
 * <p>
 * See <a href="https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html">the AWS documentation</a> to learn more about AWS Lambda SnapStart
 * <p>
 * <blockquote><pre>
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .code(Code.fromAsset(join(__dirname, "handler.zip")))
 *         .runtime(Runtime.JAVA_11)
 *         .handler("example.Handler::handleRequest")
 *         .snapStart(SnapStartConf.ON_PUBLISHED_VERSIONS)
 *         .build();
 * 
 * Version version = fn.getCurrentVersion();
 * </pre></blockquote>
 * <p>
 * <h2>AutoScaling</h2>
 * <p>
 * You can use Application AutoScaling to automatically configure the provisioned concurrency for your functions. AutoScaling can be set to track utilization or be based on a schedule. To configure AutoScaling on a function alias:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.autoscaling.*;
 * 
 * Function fn;
 * 
 * Alias alias = fn.addAlias("prod");
 * 
 * // Create AutoScaling target
 * IScalableFunctionAttribute as = alias.addAutoScaling(AutoScalingOptions.builder().maxCapacity(50).build());
 * 
 * // Configure Target Tracking
 * as.scaleOnUtilization(UtilizationScalingOptions.builder()
 *         .utilizationTarget(0.5)
 *         .build());
 * 
 * // Configure Scheduled Scaling
 * as.scaleOnSchedule("ScaleUpInTheMorning", ScalingSchedule.builder()
 *         .schedule(Schedule.cron(CronOptions.builder().hour("8").minute("0").build()))
 *         .minCapacity(20)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.applicationautoscaling.*;
 * import software.amazon.awscdk.*;
 * import cx.api.LAMBDA_RECOGNIZE_LAYER_VERSION;
 * import software.amazon.awscdk.*;
 * 
 * /**
 *  * Stack verification steps:
 *  * aws application-autoscaling describe-scalable-targets --service-namespace lambda --resource-ids function:&lt;function name&gt;:prod
 *  * has a minCapacity of 3 and maxCapacity of 50
 *  *&#47;
 * public class TestStack extends Stack {
 *     public TestStack(App scope, String id) {
 *         super(scope, id);
 * 
 *         Function fn = Function.Builder.create(this, "MyLambda")
 *                 .code(new InlineCode("exports.handler = async () =&gt; { console.log('hello world'); };"))
 *                 .handler("index.handler")
 *                 .runtime(Runtime.NODEJS_LATEST)
 *                 .build();
 * 
 *         Version version = fn.getCurrentVersion();
 * 
 *         Alias alias = Alias.Builder.create(this, "Alias")
 *                 .aliasName("prod")
 *                 .version(version)
 *                 .build();
 * 
 *         IScalableFunctionAttribute scalingTarget = alias.addAutoScaling(AutoScalingOptions.builder().minCapacity(3).maxCapacity(50).build());
 * 
 *         scalingTarget.scaleOnUtilization(UtilizationScalingOptions.builder()
 *                 .utilizationTarget(0.5)
 *                 .build());
 * 
 *         scalingTarget.scaleOnSchedule("ScaleUpInTheMorning", ScalingSchedule.builder()
 *                 .schedule(Schedule.cron(CronOptions.builder().hour("8").minute("0").build()))
 *                 .minCapacity(20)
 *                 .build());
 * 
 *         scalingTarget.scaleOnSchedule("ScaleDownAtNight", ScalingSchedule.builder()
 *                 .schedule(Schedule.cron(CronOptions.builder().hour("20").minute("0").build()))
 *                 .maxCapacity(20)
 *                 .build());
 * 
 *         CfnOutput.Builder.create(this, "FunctionName")
 *                 .value(fn.getFunctionName())
 *                 .build();
 *     }
 * }
 * 
 * App app = new App();
 * 
 * TestStack stack = new TestStack(app, "aws-lambda-autoscaling");
 * 
 * // Changes the function description when the feature flag is present
 * // to validate the changed function hash.
 * Aspects.of(stack).add(new FunctionVersionUpgrade(LAMBDA_RECOGNIZE_LAYER_VERSION));
 * 
 * app.synth();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-scaling.html">the AWS documentation</a> on autoscaling lambda functions.
 * <p>
 * <h2>Log Group</h2>
 * <p>
 * By default, Lambda functions automatically create a log group with the name <code>/aws/lambda/&lt;function-name&gt;</code> upon first execution with
 * log data set to never expire.
 * This is convenient, but prevents you from changing any of the properties of this auto-created log group using the AWS CDK.
 * For example you cannot set log retention or assign a data protection policy.
 * <p>
 * To fully customize the logging behavior of your Lambda function, create a <code>logs.LogGroup</code> ahead of time and use the <code>logGroup</code> property to instruct the Lambda function to send logs to it.
 * This way you can use the full features set supported by Amazon CloudWatch Logs.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.LogGroup;
 * 
 * 
 * LogGroup myLogGroup = LogGroup.Builder.create(this, "MyLogGroupWithLogGroupName")
 *         .logGroupName("customLogGroup")
 *         .build();
 * 
 * Function.Builder.create(this, "Lambda")
 *         .code(new InlineCode("foo"))
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .logGroup(myLogGroup)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Providing a user-controlled log group was rolled out to commercial regions on 2023-11-16.
 * If you are deploying to another type of region, please check regional availability first.
 * <p>
 * <h2>Lambda with Recursive Loop protection</h2>
 * <p>
 * Recursive loop protection is to stop unintended loops. The customers are opted in by default for Lambda to detect and terminate unintended loops between Lambda and other AWS Services.
 * The property can be assigned two values here, "Allow" and "Terminate".
 * <p>
 * The default value is set to "Terminate", which lets the Lambda to detect and terminate the recursive loops.
 * <p>
 * When the value is set to "Allow", the customers opt out of recursive loop detection and Lambda does not terminate recursive loops if any.
 * <p>
 * See <a href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-recursion.html">the AWS documentation</a> to learn more about AWS Lambda Recusrive Loop Detection
 * <p>
 * <blockquote><pre>
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .code(Code.fromAsset(join(__dirname, "handler.zip")))
 *         .runtime(Runtime.JAVA_11)
 *         .handler("example.Handler::handleRequest")
 *         .recursiveLoop(RecursiveLoop.TERMINATE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Legacy Log Retention</h3>
 * <p>
 * As an alternative to providing a custom, user controlled log group, the legacy <code>logRetention</code> property can be used to set a different expiration period.
 * This feature uses a Custom Resource to change the log retention of the automatically created log group.
 * <p>
 * By default, CDK uses the AWS SDK retry options when creating a log group. The <code>logRetentionRetryOptions</code> property
 * allows you to customize the maximum number of retries and base backoff duration.
 * <p>
 * <em>Note</em> that a <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html">CloudFormation custom
 * resource</a> is added
 * to the stack that pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the
 * correct log retention period (never expire, by default). This Custom Resource will also create a log group to log events of the custom resource. The log retention period for this addtional log group is hard-coded to 1 day.
 * <p>
 * <em>Further note</em> that, if the log group already exists and the <code>logRetention</code> is not set, the custom resource will reset
 * the log retention to never expire even if it was configured with a different value.
 * <p>
 * <h2>FileSystem Access</h2>
 * <p>
 * You can configure a function to mount an Amazon Elastic File System (Amazon EFS) to a
 * directory in your runtime environment with the <code>filesystem</code> property. To access Amazon EFS
 * from lambda function, the Amazon EFS access point will be required.
 * <p>
 * The following sample allows the lambda function to mount the Amazon EFS access point to <code>/mnt/msg</code> in the runtime environment and access the filesystem with the POSIX identity defined in <code>posixUser</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.efs.*;
 * 
 * 
 * // create a new VPC
 * Vpc vpc = new Vpc(this, "VPC");
 * 
 * // create a new Amazon EFS filesystem
 * FileSystem fileSystem = FileSystem.Builder.create(this, "Efs").vpc(vpc).build();
 * 
 * // create a new access point from the filesystem
 * AccessPoint accessPoint = fileSystem.addAccessPoint("AccessPoint", AccessPointOptions.builder()
 *         // set /export/lambda as the root of the access point
 *         .path("/export/lambda")
 *         // as /export/lambda does not exist in a new efs filesystem, the efs will create the directory with the following createAcl
 *         .createAcl(Acl.builder()
 *                 .ownerUid("1001")
 *                 .ownerGid("1001")
 *                 .permissions("750")
 *                 .build())
 *         // enforce the POSIX identity so lambda function will access with this identity
 *         .posixUser(PosixUser.builder()
 *                 .uid("1001")
 *                 .gid("1001")
 *                 .build())
 *         .build());
 * 
 * Function fn = Function.Builder.create(this, "MyLambda")
 *         // mount the access point to /mnt/msg in the lambda runtime environment
 *         .filesystem(FileSystem.fromEfsAccessPoint(accessPoint, "/mnt/msg"))
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>IPv6 support</h2>
 * <p>
 * You can configure IPv6 connectivity for lambda function by setting <code>Ipv6AllowedForDualStack</code> to true.
 * It allows Lambda functions to specify whether the IPv6 traffic should be allowed when using dual-stack VPCs.
 * To access IPv6 network using Lambda, Dual-stack VPC is required. Using dual-stack VPC a function communicates with subnet over either of IPv4 or IPv6.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * 
 * 
 * NatProvider natProvider = NatProvider.gateway();
 * 
 * // create dual-stack VPC
 * Vpc vpc = Vpc.Builder.create(this, "DualStackVpc")
 *         .ipProtocol(IpProtocol.DUAL_STACK)
 *         .subnetConfiguration(List.of(SubnetConfiguration.builder()
 *                 .name("Ipv6Public1")
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build(), SubnetConfiguration.builder()
 *                 .name("Ipv6Public2")
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build(), SubnetConfiguration.builder()
 *                 .name("Ipv6Private1")
 *                 .subnetType(SubnetType.PRIVATE_WITH_EGRESS)
 *                 .build()))
 *         .natGatewayProvider(natProvider)
 *         .build();
 * 
 * String natGatewayId = natProvider.getConfiguredGateways()[0].getGatewayId();
 * ((PrivateSubnet)vpc.privateSubnets[0]).addIpv6Nat64Route(natGatewayId);
 * 
 * Function fn = Function.Builder.create(this, "Lambda_with_IPv6_VPC")
 *         .code(new InlineCode("def main(event, context): pass"))
 *         .handler("index.main")
 *         .runtime(Runtime.PYTHON_3_9)
 *         .vpc(vpc)
 *         .ipv6AllowedForDualStack(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Outbound traffic</h2>
 * <p>
 * By default, when creating a Lambda function, it would add a security group outbound rule to allow sending all network traffic (except IPv6). This is controlled by <code>allowAllOutbound</code> in function properties, which has a default value of <code>true</code>.
 * <p>
 * To allow outbound IPv6 traffic by default, explicitly set <code>allowAllIpv6Outbound</code> to <code>true</code> in function properties as shown below (the default value for <code>allowAllIpv6Outbound</code> is <code>false</code>):
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * 
 * 
 * Vpc vpc = new Vpc(this, "Vpc");
 * 
 * Function fn = Function.Builder.create(this, "LambdaWithIpv6Outbound")
 *         .code(new InlineCode("def main(event, context): pass"))
 *         .handler("index.main")
 *         .runtime(Runtime.PYTHON_3_9)
 *         .vpc(vpc)
 *         .allowAllIpv6Outbound(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Do not specify <code>allowAllOutbound</code> or <code>allowAllIpv6Outbound</code> property if the <code>securityGroups</code> or <code>securityGroup</code> property is set. Instead, configure these properties directly on the security group.
 * <p>
 * <h2>Ephemeral Storage</h2>
 * <p>
 * You can configure ephemeral storage on a function to control the amount of storage it gets for reading
 * or writing data, allowing you to use AWS Lambda for ETL jobs, ML inference, or other data-intensive workloads.
 * The ephemeral storage will be accessible in the functions' <code>/tmp</code> directory.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.Size;
 * 
 * 
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .ephemeralStorageSize(Size.mebibytes(1024))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Read more about using this feature in <a href="https://aws.amazon.com/blogs/aws/aws-lambda-now-supports-up-to-10-gb-ephemeral-storage/">this AWS blog post</a>.
 * <p>
 * <h2>Singleton Function</h2>
 * <p>
 * The <code>SingletonFunction</code> construct is a way to guarantee that a lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack. This is guaranteed
 * as long as the <code>uuid</code> property and the optional <code>lambdaPurpose</code> property stay the same whenever they're declared into the
 * stack.
 * <p>
 * A typical use case of this function is when a higher level construct needs to declare a Lambda function as part of it but
 * needs to guarantee that the function is declared once. However, a user of this higher level construct can declare it any
 * number of times and with different properties. Using <code>SingletonFunction</code> here with a fixed <code>uuid</code> will guarantee this.
 * <p>
 * For example, the <code>AwsCustomResource</code> construct requires only one single lambda function for all api calls that are made.
 * <p>
 * <h2>Bundling Asset Code</h2>
 * <p>
 * When using <code>lambda.Code.fromAsset(path)</code> it is possible to bundle the code by running a
 * command in a Docker container. The asset path will be mounted at <code>/asset-input</code>. The
 * Docker container is responsible for putting content at <code>/asset-output</code>. The content at
 * <code>/asset-output</code> will be zipped and used as Lambda code.
 * <p>
 * Example with Python:
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "Function")
 *         .code(Code.fromAsset(join(__dirname, "my-python-handler"), AssetOptions.builder()
 *                 .bundling(BundlingOptions.builder()
 *                         .image(Runtime.PYTHON_3_9.getBundlingImage())
 *                         .command(List.of("bash", "-c", "pip install -r requirements.txt -t /asset-output &amp;&amp; cp -au . /asset-output"))
 *                         .build())
 *                 .build()))
 *         .runtime(Runtime.PYTHON_3_9)
 *         .handler("index.handler")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Runtimes expose a <code>bundlingImage</code> property that points to the <a href="https://github.com/awslabs/aws-sam-cli">AWS SAM</a> build image.
 * <p>
 * Use <code>cdk.DockerImage.fromRegistry(image)</code> to use an existing image or
 * <code>cdk.DockerImage.fromBuild(path)</code> to build a specific image:
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "Function")
 *         .code(Code.fromAsset("/path/to/handler", AssetOptions.builder()
 *                 .bundling(BundlingOptions.builder()
 *                         .image(DockerImage.fromBuild("/path/to/dir/with/DockerFile", DockerBuildOptions.builder()
 *                                 .buildArgs(Map.of(
 *                                         "ARG1", "value1"))
 *                                 .build()))
 *                         .command(List.of("my", "cool", "command"))
 *                         .build())
 *                 .build()))
 *         .runtime(Runtime.PYTHON_3_9)
 *         .handler("index.handler")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Language-specific APIs</h2>
 * <p>
 * Language-specific higher level constructs are provided in separate modules:
 * <p>
 * <ul>
 * <li><code>aws-cdk-lib/aws-lambda-nodejs</code>: <a href="https://github.com/aws/aws-cdk/tree/main/packages/aws-cdk-lib/aws-lambda-nodejs">Github</a> &amp; <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_lambda_nodejs-readme.html">CDK Docs</a></li>
 * <li><code>&#64;aws-cdk/aws-lambda-python-alpha</code>: <a href="https://github.com/aws/aws-cdk/tree/main/packages/%40aws-cdk/aws-lambda-python-alpha">Github</a> &amp; <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-lambda-python-alpha-readme.html">CDK Docs</a></li>
 * </ul>
 * <p>
 * <h2>Code Signing</h2>
 * <p>
 * Code signing for AWS Lambda helps to ensure that only trusted code runs in your Lambda functions.
 * When enabled, AWS Lambda checks every code deployment and verifies that the code package is signed by a trusted source.
 * For more information, see <a href="https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html">Configuring code signing for AWS Lambda</a>.
 * The following code configures a function with code signing.
 * <p>
 * Please note the code will not be automatically signed before deployment. To ensure your code is properly signed, you'll need to conduct the code signing process either through the AWS CLI (Command Line Interface) <a href="https://awscli.amazonaws.com/v2/documentation/api/latest/reference/signer/start-signing-job.html">start-signing-job</a> or by accessing the AWS Signer console.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.signer.*;
 * 
 * 
 * SigningProfile signingProfile = SigningProfile.Builder.create(this, "SigningProfile")
 *         .platform(Platform.AWS_LAMBDA_SHA384_ECDSA)
 *         .build();
 * 
 * CodeSigningConfig codeSigningConfig = CodeSigningConfig.Builder.create(this, "CodeSigningConfig")
 *         .signingProfiles(List.of(signingProfile))
 *         .build();
 * 
 * Function.Builder.create(this, "Function")
 *         .codeSigningConfig(codeSigningConfig)
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Runtime updates</h2>
 * <p>
 * Lambda runtime management controls help reduce the risk of impact to your workloads in the rare event of a runtime version incompatibility.
 * For more information, see <a href="https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html#runtime-management-controls">Runtime management controls</a>
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "Lambda")
 *         .runtimeManagementMode(RuntimeManagementMode.AUTO)
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to set the "Manual" setting, using the ARN of the runtime version as the argument.
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "Lambda")
 *         .runtimeManagementMode(RuntimeManagementMode.manual("runtimeVersion-arn"))
 *         .runtime(Runtime.NODEJS_18_X)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Exclude Patterns for Assets</h2>
 * <p>
 * When using <code>lambda.Code.fromAsset(path)</code> an <code>exclude</code> property allows you to ignore particular files for assets by providing patterns for file paths to exclude. Note that this has no effect on <code>Assets</code> bundled using the <code>bundling</code> property.
 * <p>
 * The <code>ignoreMode</code> property can be used with the <code>exclude</code> property to specify the file paths to ignore based on the <a href="https://git-scm.com/docs/gitignore">.gitignore specification</a> or the <a href="https://docs.docker.com/engine/reference/builder/#dockerignore-file">.dockerignore specification</a>. The default behavior is to ignore file paths based on simple glob patterns.
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "Function")
 *         .code(Code.fromAsset(join(__dirname, "my-python-handler"), AssetOptions.builder()
 *                 .exclude(List.of("*.ignore"))
 *                 .ignoreMode(IgnoreMode.DOCKER)
 *                 .build()))
 *         .runtime(Runtime.PYTHON_3_9)
 *         .handler("index.handler")
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also write to include only certain files by using a negation.
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "Function")
 *         .code(Code.fromAsset(join(__dirname, "my-python-handler"), AssetOptions.builder()
 *                 .exclude(List.of("*", "!index.py"))
 *                 .build()))
 *         .runtime(Runtime.PYTHON_3_9)
 *         .handler("index.handler")
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.lambda;
