/**
 * <h1>Amazon CloudWatch Construct Library</h1>
 * <p>
 * <h2>Metric objects</h2>
 * <p>
 * Metric objects represent a metric that is emitted by AWS services or your own
 * application, such as <code>CPUUsage</code>, <code>FailureCount</code> or <code>Bandwidth</code>.
 * <p>
 * Metric objects can be constructed directly or are exposed by resources as
 * attributes. Resources that expose metrics will have functions that look
 * like <code>metricXxx()</code> which will return a Metric object, initialized with defaults
 * that make sense.
 * <p>
 * For example, <code>lambda.Function</code> objects have the <code>fn.metricErrors()</code> method, which
 * represents the amount of errors reported by that Lambda function:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Metric errors = fn.metricErrors();
 * </pre></blockquote>
 * <p>
 * <code>Metric</code> objects can be account and region aware. You can specify <code>account</code> and <code>region</code> as properties of the metric, or use the <code>metric.attachTo(Construct)</code> method. <code>metric.attachTo()</code> will automatically copy the <code>region</code> and <code>account</code> fields of the <code>Construct</code>, which can come from anywhere in the Construct tree.
 * <p>
 * You can also instantiate <code>Metric</code> objects to reference any
 * <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html">published metric</a>
 * that's not exposed using a convenience method on the CDK construct.
 * For example:
 * <p>
 * <blockquote><pre>
 * HostedZone hostedZone = HostedZone.Builder.create(this, "MyHostedZone").zoneName("example.org").build();
 * Metric metric = Metric.Builder.create()
 *         .namespace("AWS/Route53")
 *         .metricName("DNSQueries")
 *         .dimensionsMap(Map.of(
 *                 "HostedZoneId", hostedZone.getHostedZoneId()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Instantiating a new Metric object</h3>
 * <p>
 * If you want to reference a metric that is not yet exposed by an existing construct,
 * you can instantiate a <code>Metric</code> object to represent it. For example:
 * <p>
 * <blockquote><pre>
 * Metric metric = Metric.Builder.create()
 *         .namespace("MyNamespace")
 *         .metricName("MyMetric")
 *         .dimensionsMap(Map.of(
 *                 "ProcessingStep", "Download"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Metric ID</h3>
 * <p>
 * Metrics can be assigned a unique identifier using the <code>id</code> property. This is
 * useful when referencing metrics in math expressions:
 * <p>
 * <blockquote><pre>
 * Metric metric = Metric.Builder.create()
 *         .namespace("AWS/Lambda")
 *         .metricName("Invocations")
 *         .dimensionsMap(Map.of(
 *                 "FunctionName", "MyFunction"))
 *         .id("invocations")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>id</code> must start with a lowercase letter and can only contain letters, numbers, and underscores.
 * <p>
 * <h3>Metric Visible</h3>
 * <p>
 * Metrics can be hidden from dashboard graphs using the <code>visible</code> property:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Metric metric = fn.metricErrors(MetricOptions.builder()
 *         .visible(false)
 *         .build());
 * </pre></blockquote>
 * <p>
 * By default, all metrics are visible (<code>visible: true</code>). Setting <code>visible: false</code>
 * hides the metric from dashboard visualizations while still allowing it to be
 * used in math expressions given that it has an <code>id</code> set to it.
 * <p>
 * <h3>Metric Math</h3>
 * <p>
 * Math expressions are supported by instantiating the <code>MathExpression</code> class.
 * For example, a math expression that sums two other metrics looks like this:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * MathExpression allProblems = MathExpression.Builder.create()
 *         .expression("errors + throttles")
 *         .usingMetrics(Map.of(
 *                 "errors", fn.metricErrors(),
 *                 "throttles", fn.metricThrottles()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can use <code>MathExpression</code> objects like any other metric, including using
 * them in other math expressions:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * MathExpression allProblems;
 * 
 * 
 * MathExpression problemPercentage = MathExpression.Builder.create()
 *         .expression("(problems / invocations) * 100")
 *         .usingMetrics(Map.of(
 *                 "problems", allProblems,
 *                 "invocations", fn.metricInvocations()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Metric ID Usage in Math Expressions</h3>
 * <p>
 * When metrics have custom IDs, you can reference them directly in math expressions.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Metric invocations = fn.metricInvocations(MetricOptions.builder()
 *         .id("lambda_invocations")
 *         .build());
 * 
 * Metric errors = fn.metricErrors(MetricOptions.builder()
 *         .id("lambda_errors")
 *         .build());
 * </pre></blockquote>
 * <p>
 * When metrics have predefined IDs, they can be referenced directly in math expressions by their ID without requiring the <code>usingMetrics</code> property.
 * <p>
 * <blockquote><pre>
 * MathExpression errorRate = MathExpression.Builder.create()
 *         .expression("lambda_errors / lambda_invocations * 100")
 *         .label("Error Rate (%)")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Search Expressions</h3>
 * <p>
 * Math expressions also support search expressions. For example, the following
 * search expression returns all CPUUtilization metrics that it finds, with the
 * graph showing the Average statistic with an aggregation period of 5 minutes:
 * <p>
 * <blockquote><pre>
 * MathExpression cpuUtilization = MathExpression.Builder.create()
 *         .expression("SEARCH('{AWS/EC2,InstanceId} MetricName=\"CPUUtilization\"', 'Average', 300)")
 * 
 *         // Specifying '' as the label suppresses the default behavior
 *         // of using the expression as metric label. This is especially appropriate
 *         // when using expressions that return multiple time series (like SEARCH()
 *         // or METRICS()), to show the labels of the retrieved metrics only.
 *         .label("")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Cross-account and cross-region search expressions are also supported. Use
 * the <code>searchAccount</code> and <code>searchRegion</code> properties to specify the account
 * and/or region to evaluate the search expression against.
 * <p>
 * <h3>Aggregation</h3>
 * <p>
 * To graph or alarm on metrics you must aggregate them first, using a function
 * like <code>Average</code> or a percentile function like <code>P99</code>. By default, most Metric objects
 * returned by CDK libraries will be configured as <code>Average</code> over <code>300 seconds</code> (5 minutes).
 * The exception is if the metric represents a count of discrete events, such as
 * failures. In that case, the Metric object will be configured as <code>Sum</code> over <code>300 seconds</code>, i.e. it represents the number of times that event occurred over the
 * time period.
 * <p>
 * If you want to change the default aggregation of the Metric object (for example,
 * the function or the period), you can do so by passing additional parameters
 * to the metric function call:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Metric minuteErrorRate = fn.metricErrors(MetricOptions.builder()
 *         .statistic(Stats.AVERAGE)
 *         .period(Duration.minutes(1))
 *         .label("Lambda failure rate")
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>statistic</code> field accepts a <code>string</code>; the <code>cloudwatch.Stats</code> object has a
 * number of predefined factory functions that help you constructs strings that are
 * appropriate for CloudWatch. The <code>metricErrors</code> function also allows changing the
 * metric label or color, which will be useful when embedding them in graphs (see
 * below).
 * <p>
 * <blockquote>
 * <p>
 * Rates versus Sums
 * <p>
 * The reason for using <code>Sum</code> to count discrete events is that <em>some</em> events are
 * emitted as either <code>0</code> or <code>1</code> (for example <code>Errors</code> for a Lambda) and some are
 * only emitted as <code>1</code> (for example <code>NumberOfMessagesPublished</code> for an SNS
 * topic).
 * <p>
 * In case <code>0</code>-metrics are emitted, it makes sense to take the <code>Average</code> of this
 * metric: the result will be the fraction of errors over all executions.
 * <p>
 * If <code>0</code>-metrics are not emitted, the <code>Average</code> will always be equal to <code>1</code>,
 * and not be very useful.
 * <p>
 * In order to simplify the mental model of <code>Metric</code> objects, we default to
 * aggregating using <code>Sum</code>, which will be the same for both metrics types. If you
 * happen to know the Metric you want to alarm on makes sense as a rate
 * (<code>Average</code>) you can always choose to change the statistic.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Available Aggregation Statistics</h3>
 * <p>
 * For your metrics aggregation, you can use the following statistics:
 * <p>
 * | Statistic                |    Short format     |                 Long format                  | Factory name         |
 * | ------------------------ | :-----------------: | :------------------------------------------: | -------------------- |
 * | SampleCount (n)          |         ❌          |                      ❌                      | <code>Stats.SAMPLE_COUNT</code> |
 * | Average (avg)            |         ❌          |                      ❌                      | <code>Stats.AVERAGE</code>      |
 * | Sum                      |         ❌          |                      ❌                      | <code>Stats.SUM</code>          |
 * | Minimum (min)            |         ❌          |                      ❌                      | <code>Stats.MINIMUM</code>      |
 * | Maximum (max)            |         ❌          |                      ❌                      | <code>Stats.MAXIMUM</code>      |
 * | Interquartile mean (IQM) |         ❌          |                      ❌                      | <code>Stats.IQM</code>          |
 * | Percentile (p)           |        <code>p99</code>        |                      ❌                      | <code>Stats.p(99)</code>        |
 * | Winsorized mean (WM)     | <code>wm99</code> = <code>WM(:99%)</code> | <code>WM(x:y) \| WM(x%:y%) \| WM(x%:) \| WM(:y%)</code> | <code>Stats.wm(10, 90)</code>   |
 * | Trimmed count (TC)       | <code>tc99</code> = <code>TC(:99%)</code> | <code>TC(x:y) \| TC(x%:y%) \| TC(x%:) \| TC(:y%)</code> | <code>Stats.tc(10, 90)</code>   |
 * | Trimmed sum (TS)         | <code>ts99</code> = <code>TS(:99%)</code> | <code>TS(x:y) \| TS(x%:y%) \| TS(x%:) \| TS(:y%)</code> | <code>Stats.ts(10, 90)</code>   |
 * | Percentile rank (PR)     |         ❌          |        <code>PR(x:y) \| PR(x:) \| PR(:y)</code>         | <code>Stats.pr(10, 5000)</code> |
 * <p>
 * The most common values are provided in the <code>cloudwatch.Stats</code> class. You can provide any string if your statistic is not in the class.
 * <p>
 * Read more at <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Statistics-definitions.html">CloudWatch statistics definitions</a>.
 * <p>
 * <blockquote><pre>
 * HostedZone hostedZone;
 * 
 * 
 * Metric.Builder.create()
 *         .namespace("AWS/Route53")
 *         .metricName("DNSQueries")
 *         .dimensionsMap(Map.of(
 *                 "HostedZoneId", hostedZone.getHostedZoneId()))
 *         .statistic(Stats.SAMPLE_COUNT)
 *         .period(Duration.minutes(5))
 *         .build();
 * 
 * Metric.Builder.create()
 *         .namespace("AWS/Route53")
 *         .metricName("DNSQueries")
 *         .dimensionsMap(Map.of(
 *                 "HostedZoneId", hostedZone.getHostedZoneId()))
 *         .statistic(Stats.p(99))
 *         .period(Duration.minutes(5))
 *         .build();
 * 
 * Metric.Builder.create()
 *         .namespace("AWS/Route53")
 *         .metricName("DNSQueries")
 *         .dimensionsMap(Map.of(
 *                 "HostedZoneId", hostedZone.getHostedZoneId()))
 *         .statistic("TS(7.5%:90%)")
 *         .period(Duration.minutes(5))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Labels</h3>
 * <p>
 * Metric labels are displayed in the legend of graphs that include the metrics.
 * <p>
 * You can use <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html">dynamic labels</a>
 * to show summary information about the displayed time series
 * in the legend. For example, if you use:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Metric minuteErrorRate = fn.metricErrors(MetricOptions.builder()
 *         .statistic(Stats.SUM)
 *         .period(Duration.hours(1))
 * 
 *         // Show the maximum hourly error count in the legend
 *         .label("[max: ${MAX}] Lambda failure rate")
 *         .build());
 * </pre></blockquote>
 * <p>
 * As the metric label, the maximum value in the visible range will
 * be shown next to the time series name in the graph's legend.
 * <p>
 * If the metric is a math expression producing more than one time series, the
 * maximum will be individually calculated and shown for each time series produce
 * by the math expression.
 * <p>
 * <h2>Alarms</h2>
 * <p>
 * Alarms can be created on metrics in one of two ways. Either create an <code>Alarm</code>
 * object, passing the <code>Metric</code> object to set the alarm on:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Alarm.Builder.create(this, "Alarm")
 *         .metric(fn.metricErrors())
 *         .threshold(100)
 *         .evaluationPeriods(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can call <code>metric.createAlarm()</code>:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * fn.metricErrors().createAlarm(this, "Alarm", CreateAlarmOptions.builder()
 *         .threshold(100)
 *         .evaluationPeriods(2)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The most important properties to set while creating an Alarms are:
 * <p>
 * <ul>
 * <li><code>threshold</code>: the value to compare the metric against.</li>
 * <li><code>comparisonOperator</code>: the comparison operation to use, defaults to <code>metric &gt;= threshold</code>.</li>
 * <li><code>evaluationPeriods</code>: how many consecutive periods the metric has to be
 * breaching the threshold for the alarm to trigger.</li>
 * </ul>
 * <p>
 * To create a cross-account alarm, make sure you have enabled <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Cross-Account-Cross-Region.html">cross-account functionality</a> in CloudWatch. Then, set the <code>account</code> property in the <code>Metric</code> object either manually or via the <code>metric.attachTo()</code> method.
 * <p>
 * Please note that it is <strong>not possible</strong> to:
 * <p>
 * <ul>
 * <li>Create a cross-Account alarm that has <code>evaluateLowSampleCountPercentile: "ignore"</code>. The reason is that the only
 * way to pass an AccountID is to use the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudwatch-alarm.html#cfn-cloudwatch-alarm-metrics"><code>Metrics</code> field of the Alarm resource</a>. If we use the <code>Metrics</code> field, the CloudWatch event that is
 * used to evaluate the Alarm doesn't have a <code>SampleCount</code> field anymore ("<a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Create-alarm-on-metric-math-expression.html">When CloudWatch evaluates alarms, periods are aggregated into single data points</a>"). The result is that the Alarm cannot evaluate at all.</li>
 * <li>Create a cross-Region alarm (<a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Cross-Account-Cross-Region.html">source</a>).</li>
 * </ul>
 * <p>
 * <h3>Alarm Actions</h3>
 * <p>
 * To add actions to an alarm, use the integration classes from the
 * <code>aws-cdk-lib/aws-cloudwatch-actions</code> package. For example, to post a message to
 * an SNS topic when an alarm breaches, do the following:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.actions.*;
 * Alarm alarm;
 * 
 * 
 * Topic topic = new Topic(this, "Topic");
 * alarm.addAlarmAction(new SnsAction(topic));
 * </pre></blockquote>
 * <p>
 * <h4>Notification formats</h4>
 * <p>
 * Alarms can be created in one of two "formats":
 * <p>
 * <ul>
 * <li>With "top-level parameters" (these are the classic style of CloudWatch Alarms).</li>
 * <li>With a list of metrics specifications (these are the modern style of CloudWatch Alarms).</li>
 * </ul>
 * <p>
 * For backwards compatibility, CDK will try to create classic, top-level CloudWatch alarms
 * as much as possible, unless you are using features that cannot be expressed in that format.
 * Features that require the new-style alarm format are:
 * <p>
 * <ul>
 * <li>Metric math</li>
 * <li>Cross-account metrics</li>
 * <li>Labels</li>
 * </ul>
 * <p>
 * The difference between these two does not impact the functionality of the alarm
 * in any way, <em>except</em> that the format of the notifications the Alarm generates is
 * different between them. This affects both the notifications sent out over SNS,
 * as well as the EventBridge events generated by this Alarm. If you are writing
 * code to consume these notifications, be sure to handle both formats.
 * <p>
 * <h3>Composite Alarms</h3>
 * <p>
 * <a href="https://aws.amazon.com/about-aws/whats-new/2020/03/amazon-cloudwatch-now-allows-you-to-combine-multiple-alarms/">Composite Alarms</a>
 * can be created from existing Alarm resources.
 * <p>
 * <blockquote><pre>
 * Alarm alarm1;
 * Alarm alarm2;
 * Alarm alarm3;
 * Alarm alarm4;
 * 
 * 
 * IAlarmRule alarmRule = AlarmRule.anyOf(AlarmRule.allOf(AlarmRule.anyOf(alarm1, AlarmRule.fromAlarm(alarm2, AlarmState.OK), alarm3), AlarmRule.not(AlarmRule.fromAlarm(alarm4, AlarmState.INSUFFICIENT_DATA))), AlarmRule.fromBoolean(false));
 * 
 * CompositeAlarm.Builder.create(this, "MyAwesomeCompositeAlarm")
 *         .alarmRule(alarmRule)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Actions Suppressor</h4>
 * <p>
 * If you want to disable actions of a Composite Alarm based on a certain condition, you can use <a href="https://www.amazonaws.cn/en/new/2022/amazon-cloudwatch-supports-composite-alarm-actions-suppression/">Actions Suppression</a>.
 * <p>
 * <blockquote><pre>
 * Alarm alarm1;
 * Alarm alarm2;
 * IAlarmAction onAlarmAction;
 * IAlarmAction onOkAction;
 * Alarm actionsSuppressor;
 * 
 * 
 * IAlarmRule alarmRule = AlarmRule.anyOf(alarm1, alarm2);
 * 
 * CompositeAlarm myCompositeAlarm = CompositeAlarm.Builder.create(this, "MyAwesomeCompositeAlarm")
 *         .alarmRule(alarmRule)
 *         .actionsSuppressor(actionsSuppressor)
 *         .build();
 * myCompositeAlarm.addAlarmAction(onAlarmAction);
 * myCompositeAlarm.addOkAction(onOkAction);
 * </pre></blockquote>
 * <p>
 * In the provided example, if <code>actionsSuppressor</code> is in <code>ALARM</code> state, <code>onAlarmAction</code> won't be triggered even if <code>myCompositeAlarm</code> goes into <code>ALARM</code> state.
 * Similar, if <code>actionsSuppressor</code> is in <code>ALARM</code> state and <code>myCompositeAlarm</code> goes from <code>ALARM</code> into <code>OK</code> state, <code>onOkAction</code> won't be triggered.
 * <p>
 * <h3>A note on units</h3>
 * <p>
 * In CloudWatch, Metrics datums are emitted with units, such as <code>seconds</code> or
 * <code>bytes</code>. When <code>Metric</code> objects are given a <code>unit</code> attribute, it will be used to
 * <em>filter</em> the stream of metric datums for datums emitted using the same <code>unit</code>
 * attribute.
 * <p>
 * In particular, the <code>unit</code> field is <em>not</em> used to rescale datums or alarm threshold
 * values (for example, it cannot be used to specify an alarm threshold in
 * <em>Megabytes</em> if the metric stream is being emitted as <em>bytes</em>).
 * <p>
 * You almost certainly don't want to specify the <code>unit</code> property when creating
 * <code>Metric</code> objects (which will retrieve all datums regardless of their unit),
 * unless you have very specific requirements. Note that in any case, CloudWatch
 * only supports filtering by <code>unit</code> for Alarms, not in Dashboard graphs.
 * <p>
 * Please see the following GitHub issue for a discussion on real unit
 * calculations in CDK: https://github.com/aws/aws-cdk/issues/5595
 * <p>
 * <h2>Anomaly Detection Alarms</h2>
 * <p>
 * CloudWatch anomaly detection applies machine learning algorithms to create a model of expected metric behavior. You can use anomaly detection to:
 * <p>
 * <ul>
 * <li>Detect anomalies with minimal configuration</li>
 * <li>Visualize expected metric behavior</li>
 * <li>Create alarms that trigger when metrics deviate from expected patterns</li>
 * </ul>
 * <p>
 * <h3>Creating an Anomaly Detection Alarm</h3>
 * <p>
 * To build an Anomaly Detection Alarm, you should create a MathExpression that
 * uses an <code>ANOMALY_DETECTION_BAND()</code> function, and use one of the band comparison
 * operators (see the next section). Anomaly Detection Alarms have a dynamic
 * threshold, not a fixed one, so the value for <code>threshold</code> is ignored. Specify the
 * value <code>0</code> or use the symbolic <code>Alarm.ANOMALY_DETECTION_NO_THRESHOLD</code> value.
 * <p>
 * You can use the <code>AnomalyDetectionAlarm</code> class for convenience, which takes care
 * of building the right metric math expression and passing in a magic value for
 * the treshold for you:
 * <p>
 * <blockquote><pre>
 * // Create a metric
 * Metric metric = Metric.Builder.create()
 *         .namespace("AWS/EC2")
 *         .metricName("CPUUtilization")
 *         .statistic("Average")
 *         .period(Duration.minutes(5))
 *         .build();
 * 
 * // Create an anomaly detection alarm
 * AnomalyDetectionAlarm alarm = AnomalyDetectionAlarm.Builder.create(this, "AnomalyAlarm")
 *         .metric(metric)
 *         .evaluationPeriods(1)
 * 
 *         // Number of standard deviations for the band (default: 2)
 *         .stdDevs(2)
 *         // Alarm outside on either side of the band, or just below or above it (default: outside)
 *         .comparisonOperator(ComparisonOperator.LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD)
 *         .alarmDescription("Alarm when metric is outside the expected band")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Comparison Operators for Anomaly Detection</h3>
 * <p>
 * When creating an anomaly detection alarm, you must use one of the following comparison operators:
 * <p>
 * <ul>
 * <li><code>LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD</code>: Alarm when the metric is outside the band, on either side of it</li>
 * <li><code>GREATER_THAN_UPPER_THRESHOLD</code>: Alarm only when the metric is above the band</li>
 * <li><code>LESS_THAN_LOWER_THRESHOLD</code>: Alarm only when the metric is below the band</li>
 * </ul>
 * <p>
 * For more information on anomaly detection in CloudWatch, see the <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Anomaly_Detection.html">AWS documentation</a>.
 * <p>
 * <h2>Dashboards</h2>
 * <p>
 * Dashboards are set of Widgets stored server-side which can be accessed quickly
 * from the AWS console. Available widgets are graphs of a metric over time, the
 * current value of a metric, or a static piece of Markdown which explains what the
 * graphs mean.
 * <p>
 * The following widgets are available:
 * <p>
 * <ul>
 * <li><code>GraphWidget</code> -- shows any number of metrics on both the left and right
 * vertical axes.</li>
 * <li><code>AlarmWidget</code> -- shows the graph and alarm line for a single alarm.</li>
 * <li><code>SingleValueWidget</code> -- shows the current value of a set of metrics.</li>
 * <li><code>TextWidget</code> -- shows some static Markdown.</li>
 * <li><code>AlarmStatusWidget</code> -- shows the status of your alarms in a grid view.</li>
 * </ul>
 * <p>
 * <h3>Graph widget</h3>
 * <p>
 * A graph widget can display any number of metrics on either the <code>left</code> or
 * <code>right</code> vertical axis:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Metric executionCountMetric;
 * Metric errorCountMetric;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         .title("Executions vs error rate")
 * 
 *         .left(List.of(executionCountMetric))
 * 
 *         .right(List.of(errorCountMetric.with(MetricOptions.builder()
 *                 .statistic(Stats.AVERAGE)
 *                 .label("Error rate")
 *                 .color(Color.GREEN)
 *                 .build())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Using the methods <code>addLeftMetric()</code> and <code>addRightMetric()</code> you can add metrics to a graph widget later on.
 * <p>
 * Graph widgets can also display annotations attached to the left or right y-axis or the x-axis.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 * 
 *         .leftAnnotations(List.of(HorizontalAnnotation.builder().value(1800).label(Duration.minutes(30).toHumanString()).color(Color.RED).build(), HorizontalAnnotation.builder().value(3600).label("1 hour").color("#2ca02c").build()))
 *         .verticalAnnotations(List.of(VerticalAnnotation.builder().date("2022-10-19T00:00:00Z").label("Deployment").color(Color.RED).build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The graph legend can be adjusted from the default position at bottom of the widget.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 * 
 *         .legendPosition(LegendPosition.RIGHT)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The graph can publish live data within the last minute that has not been fully aggregated.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 * 
 *         .liveData(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The graph view can be changed from default 'timeSeries' to 'bar' or 'pie'.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 * 
 *         .view(GraphWidgetView.BAR)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>start</code> and <code>end</code> properties can be used to specify the time range for each graph widget independently from those of the dashboard.
 * The parameters can be specified at <code>GraphWidget</code>, <code>GaugeWidget</code>, and <code>SingleValueWidget</code>.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 * 
 *         .start("-P7D")
 *         .end("2018-12-17T06:00:00.000Z")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Table Widget</h3>
 * <p>
 * A <code>TableWidget</code> can display any number of metrics in tabular form.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Metric executionCountMetric;
 * 
 * 
 * dashboard.addWidgets(TableWidget.Builder.create()
 *         .title("Executions")
 *         .metrics(List.of(executionCountMetric))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>layout</code> property can be used to invert the rows and columns of the table.
 * The default <code>cloudwatch.TableLayout.HORIZONTAL</code> means that metrics are shown in rows and datapoints in columns.
 * <code>cloudwatch.TableLayout.VERTICAL</code> means that metrics are shown in columns and datapoints in rows.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(TableWidget.Builder.create()
 *         // ...
 * 
 *         .layout(TableLayout.VERTICAL)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>summary</code> property allows customizing the table to show summary columns (<code>columns</code> sub property),
 * whether to make the summary columns sticky remaining in view while scrolling (<code>sticky</code> sub property),
 * and to optionally only present summary columns (<code>hideNonSummaryColumns</code> sub property).
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(TableWidget.Builder.create()
 *         // ...
 * 
 *         .summary(TableSummaryProps.builder()
 *                 .columns(List.of(TableSummaryColumn.AVERAGE))
 *                 .hideNonSummaryColumns(true)
 *                 .sticky(true)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>thresholds</code> property can be used to highlight cells with a color when the datapoint value falls within the threshold.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(TableWidget.Builder.create()
 *         // ...
 * 
 *         .thresholds(List.of(TableThreshold.above(1000, Color.RED), TableThreshold.between(500, 1000, Color.ORANGE), TableThreshold.below(500, Color.GREEN)))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>showUnitsInLabel</code> property can be used to display what unit is associated with a metric in the label column.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(TableWidget.Builder.create()
 *         // ...
 * 
 *         .showUnitsInLabel(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>fullPrecision</code> property can be used to show as many digits as can fit in a cell, before rounding.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(TableWidget.Builder.create()
 *         // ...
 * 
 *         .fullPrecision(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Gauge widget</h3>
 * <p>
 * Gauge graph requires the max and min value of the left Y axis, if no value is informed the limits will be from 0 to 100.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Alarm errorAlarm;
 * Metric gaugeMetric;
 * 
 * 
 * dashboard.addWidgets(GaugeWidget.Builder.create()
 *         .metrics(List.of(gaugeMetric))
 *         .leftYAxis(YAxisProps.builder()
 *                 .min(0)
 *                 .max(1000)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Alarm widget</h3>
 * <p>
 * An alarm widget shows the graph and the alarm line of a single alarm:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Alarm errorAlarm;
 * 
 * 
 * dashboard.addWidgets(AlarmWidget.Builder.create()
 *         .title("Errors")
 *         .alarm(errorAlarm)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Single value widget</h3>
 * <p>
 * A single-value widget shows the latest value of a set of metrics (as opposed
 * to a graph of the value over time):
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Metric visitorCount;
 * Metric purchaseCount;
 * 
 * 
 * dashboard.addWidgets(SingleValueWidget.Builder.create()
 *         .metrics(List.of(visitorCount, purchaseCount))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Show as many digits as can fit, before rounding.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(SingleValueWidget.Builder.create()
 *         .metrics(List.of())
 * 
 *         .fullPrecision(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Sparkline allows you to glance the trend of a metric by displaying a simplified linegraph below the value. You can't use <code>sparkline: true</code> together with <code>setPeriodToTimeRange: true</code>
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(SingleValueWidget.Builder.create()
 *         .metrics(List.of())
 * 
 *         .sparkline(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Period allows you to set the default period for the widget:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(SingleValueWidget.Builder.create()
 *         .metrics(List.of())
 * 
 *         .period(Duration.minutes(15))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Text widget</h3>
 * <p>
 * A text widget shows an arbitrary piece of MarkDown. Use this to add explanations
 * to your dashboard:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(TextWidget.Builder.create()
 *         .markdown("# Key Performance Indicators")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Optionally set the TextWidget background to be transparent
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(TextWidget.Builder.create()
 *         .markdown("# Key Performance Indicators")
 *         .background(TextWidgetBackground.TRANSPARENT)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Alarm Status widget</h3>
 * <p>
 * An alarm status widget displays instantly the status of any type of alarms and gives the
 * ability to aggregate one or more alarms together in a small surface.
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Alarm errorAlarm;
 * 
 * 
 * dashboard.addWidgets(
 * AlarmStatusWidget.Builder.create()
 *         .alarms(List.of(errorAlarm))
 *         .build());
 * </pre></blockquote>
 * <p>
 * An alarm status widget only showing firing alarms, sorted by state and timestamp:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * Alarm errorAlarm;
 * 
 * 
 * dashboard.addWidgets(AlarmStatusWidget.Builder.create()
 *         .title("Errors")
 *         .alarms(List.of(errorAlarm))
 *         .sortBy(AlarmStatusWidgetSortBy.STATE_UPDATED_TIMESTAMP)
 *         .states(List.of(AlarmState.ALARM))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Query results widget</h3>
 * <p>
 * A <code>LogQueryWidget</code> shows the results of a query from Logs Insights:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(LogQueryWidget.Builder.create()
 *         .logGroupNames(List.of("my-log-group"))
 *         .view(LogQueryVisualizationType.TABLE)
 *         // The lines will be automatically combined using '\n|'.
 *         .queryLines(List.of("fields &#64;message", "filter &#64;message like /Error/"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Custom widget</h3>
 * <p>
 * A <code>CustomWidget</code> shows the result of an AWS Lambda function:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * // Import or create a lambda function
 * IFunction fn = Function.fromFunctionArn(dashboard, "Function", "arn:aws:lambda:us-east-1:123456789012:function:MyFn");
 * 
 * dashboard.addWidgets(CustomWidget.Builder.create()
 *         .functionArn(fn.getFunctionArn())
 *         .title("My lambda baked widget")
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can learn more about custom widgets in the <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/add_custom_widget_dashboard.html">Amazon Cloudwatch User Guide</a>.
 * <p>
 * <h3>Dashboard Layout</h3>
 * <p>
 * The widgets on a dashboard are visually laid out in a grid that is 24 columns
 * wide. Normally you specify X and Y coordinates for the widgets on a Dashboard,
 * but because this is inconvenient to do manually, the library contains a simple
 * layout system to help you lay out your dashboards the way you want them to.
 * <p>
 * Widgets have a <code>width</code> and <code>height</code> property, and they will be automatically
 * laid out either horizontally or vertically stacked to fill out the available
 * space.
 * <p>
 * Widgets are added to a Dashboard by calling <code>add(widget1, widget2, ...)</code>.
 * Widgets given in the same call will be laid out horizontally. Widgets given
 * in different calls will be laid out vertically. To make more complex layouts,
 * you can use the following widgets to pack widgets together in different ways:
 * <p>
 * <ul>
 * <li><code>Column</code>: stack two or more widgets vertically.</li>
 * <li><code>Row</code>: lay out two or more widgets horizontally.</li>
 * <li><code>Spacer</code>: take up empty space</li>
 * </ul>
 * <p>
 * <h3>Column widget</h3>
 * <p>
 * A column widget contains other widgets and they will be laid out in a
 * vertical column. Widgets will be put one after another in order.
 * <p>
 * <blockquote><pre>
 * IWidget widgetA;
 * IWidget widgetB;
 * 
 * 
 * new Column(widgetA, widgetB);
 * </pre></blockquote>
 * <p>
 * You can add a widget after object instantiation with the method
 * <code>addWidget()</code>. Each new widget will be put at the bottom of the column.
 * <p>
 * <h3>Row widget</h3>
 * <p>
 * A row widget contains other widgets and they will be laid out in a
 * horizontal row. Widgets will be put one after another in order.
 * If the total width of the row exceeds the max width of the grid of 24
 * columns, the row will wrap automatically and adapt its height.
 * <p>
 * <blockquote><pre>
 * IWidget widgetA;
 * IWidget widgetB;
 * 
 * 
 * new Row(widgetA, widgetB);
 * </pre></blockquote>
 * <p>
 * You can add a widget after object instantiation with the method
 * <code>addWidget()</code>.
 * <p>
 * <h3>Interval duration for dashboard</h3>
 * <p>
 * Interval duration for metrics in dashboard. You can specify <code>defaultInterval</code> with
 * the relative time(eg. 7 days) as <code>Duration.days(7)</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * 
 * Dashboard dashboard = Dashboard.Builder.create(this, "Dash")
 *         .defaultInterval(Duration.days(7))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Here, the dashboard would show the metrics for the last 7 days.
 * <p>
 * <h3>Dashboard variables</h3>
 * <p>
 * Dashboard variables are a convenient way to create flexible dashboards that display different content depending
 * on the value of an input field within a dashboard. They create a dashboard on which it's possible to quickly switch between
 * different Lambda functions, Amazon EC2 instances, etc.
 * <p>
 * You can learn more about Dashboard variables in the <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_dashboard_variables.html">Amazon Cloudwatch User Guide</a>
 * <p>
 * There are two types of dashboard variables available: a property variable and a pattern variable.
 * <p>
 * <ul>
 * <li>Property variables can change any JSON property in the JSON source of a dashboard like <code>region</code>. It can also change the dimension name for a metric.</li>
 * <li>Pattern variables use a regular expression pattern to change all or part of a JSON property.</li>
 * </ul>
 * <p>
 * A use case of a <strong>property variable</strong> is a dashboard with the ability to toggle the <code>region</code> property to see the same dashboard in different regions:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * 
 * Dashboard dashboard = Dashboard.Builder.create(this, "Dash")
 *         .defaultInterval(Duration.days(7))
 *         .variables(List.of(DashboardVariable.Builder.create()
 *                 .id("region")
 *                 .type(VariableType.PROPERTY)
 *                 .label("Region")
 *                 .inputType(VariableInputType.RADIO)
 *                 .value("region")
 *                 .values(Values.fromValues(VariableValue.builder().label("IAD").value("us-east-1").build(), VariableValue.builder().label("DUB").value("us-west-2").build()))
 *                 .defaultValue(DefaultValue.value("us-east-1"))
 *                 .visible(true)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This example shows how to change <code>region</code> everywhere, assuming the current dashboard is showing region <code>us-east-1</code> already, by using <strong>pattern variable</strong>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * 
 * Dashboard dashboard = Dashboard.Builder.create(this, "Dash")
 *         .defaultInterval(Duration.days(7))
 *         .variables(List.of(DashboardVariable.Builder.create()
 *                 .id("region2")
 *                 .type(VariableType.PATTERN)
 *                 .label("RegionPattern")
 *                 .inputType(VariableInputType.INPUT)
 *                 .value("us-east-1")
 *                 .defaultValue(DefaultValue.value("us-east-1"))
 *                 .visible(true)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following example generates a Lambda function variable, with a radio button for each function. Functions are discovered by a metric query search.
 * The <code>values</code> with <code>cw.Values.fromSearchComponents</code> indicates that the values will be populated from <code>FunctionName</code> values retrieved from the search expression <code>{AWS/Lambda,FunctionName} MetricName=\"Duration\"</code>.
 * The <code>defaultValue</code> with <code>cw.DefaultValue.FIRST</code> indicates that the default value will be the first value returned from the search.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * 
 * Dashboard dashboard = Dashboard.Builder.create(this, "Dash")
 *         .defaultInterval(Duration.days(7))
 *         .variables(List.of(DashboardVariable.Builder.create()
 *                 .id("functionName")
 *                 .type(VariableType.PATTERN)
 *                 .label("Function")
 *                 .inputType(VariableInputType.RADIO)
 *                 .value("originalFuncNameInDashboard")
 *                 // equivalent to cw.Values.fromSearch('{AWS/Lambda,FunctionName} MetricName=\"Duration\"', 'FunctionName')
 *                 .values(Values.fromSearchComponents(SearchComponents.builder()
 *                         .namespace("AWS/Lambda")
 *                         .dimensions(List.of("FunctionName"))
 *                         .metricName("Duration")
 *                         .populateFrom("FunctionName")
 *                         .build()))
 *                 .defaultValue(DefaultValue.FIRST)
 *                 .visible(true)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can add a variable after object instantiation with the method <code>dashboard.addVariable()</code>.
 * <p>
 * <h3>Cross-Account Visibility</h3>
 * <p>
 * Both Log and Metric Widget objects support cross-account visibility by allowing you to specify the AWS Account ID that the data (logs or metrics) originates from.
 * <p>
 * <strong>Prerequisites:</strong>
 * <p>
 * <ol>
 * <li>The monitoring account must be set up as a monitoring account</li>
 * <li>The source account must grant permissions to the monitoring account</li>
 * <li>Appropriate IAM roles and policies must be configured</li>
 * </ol>
 * <p>
 * For detailed setup instructions, see <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Cross-Account-Cross-Region.html">Cross-Account Cross-Region CloudWatch Console</a>.
 * <p>
 * To use this feature, you can set the <code>accountId</code> property on <code>LogQueryWidget</code>, <code>GraphWidget</code>, <code>AlarmWidget</code>, <code>SingleValueWidget</code>, and <code>GaugeWidget</code> constructs:
 * <p>
 * <blockquote><pre>
 * Dashboard dashboard;
 * 
 * 
 * dashboard.addWidgets(GraphWidget.Builder.create()
 *         // ...
 *         .accountId("123456789012")
 *         .build());
 * 
 * dashboard.addWidgets(LogQueryWidget.Builder.create()
 *         .logGroupNames(List.of("my-log-group"))
 *         // ...
 *         .accountId("123456789012")
 *         .queryLines(List.of("fields &#64;message"))
 *         .build());
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.cloudwatch;
