/**
 * <h1>AWS Step Functions Construct Library</h1>
 * <p>
 * The <code>aws-cdk-lib/aws-stepfunctions</code> package contains constructs for building
 * serverless workflows using objects. Use this in conjunction with the
 * <code>aws-cdk-lib/aws-stepfunctions-tasks</code> package, which contains classes used
 * to call other AWS services.
 * <p>
 * Defining a workflow looks like this (for the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/job-status-poller-sample.html">Step Functions Job Poller
 * example</a>):
 * <p>
 * <h2>State Machine</h2>
 * <p>
 * A <code>stepfunctions.StateMachine</code> is a resource that takes a state machine
 * definition. The definition is specified by its start state, and encompasses
 * all states reachable from the start state:
 * <p>
 * <blockquote><pre>
 * Pass startState = Pass.jsonata(this, "StartState");
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(startState))
 *         .build();
 * </pre></blockquote>
 * <p>
 * State machines are made up of a sequence of <strong>Steps</strong>, which represent different actions
 * taken in sequence. Some of these steps represent <em>control flow</em> (like <code>Choice</code>, <code>Map</code> and <code>Wait</code>)
 * while others represent calls made against other AWS services (like <code>LambdaInvoke</code>).
 * The second category are called <code>Task</code>s and they can all be found in the module <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks-readme.html"><code>aws-stepfunctions-tasks</code></a>.
 * <p>
 * State machines execute using an IAM Role, which will automatically have all
 * permissions added that are required to make all state machine tasks execute
 * properly (for example, permissions to invoke any Lambda functions you add to
 * your workflow). A role will be created by default, but you can supply an
 * existing one as well.
 * <p>
 * Set the <code>removalPolicy</code> prop to <code>RemovalPolicy.RETAIN</code> if you want to retain the execution
 * history when CloudFormation deletes your state machine.
 * <p>
 * Alternatively you can specify an existing step functions definition by providing a string or a file that contains the ASL JSON.
 * <p>
 * <blockquote><pre>
 * StateMachine.Builder.create(this, "StateMachineFromString")
 *         .definitionBody(DefinitionBody.fromString("{\"StartAt\":\"Pass\",\"States\":{\"Pass\":{\"Type\":\"Pass\",\"End\":true}}}"))
 *         .build();
 * 
 * StateMachine.Builder.create(this, "StateMachineFromFile")
 *         .definitionBody(DefinitionBody.fromFile("./asl.json"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Query Language</h2>
 * <p>
 * Step Functions now provides the ability to select the <code>QueryLanguage</code> for the state machine or its states: <code>JSONata</code> or <code>JSONPath</code>.
 * <p>
 * For new state machines, we recommend using <code>JSONata</code> by specifying it at the state machine level.
 * If you do not specify a <code>QueryLanguage</code>, the state machine will default to using <code>JSONPath</code>.
 * <p>
 * If a state contains a specified <code>QueryLanguage</code>, Step Functions will use the specified query language for that state.
 * If the state does not specify a <code>QueryLanguage</code>, then it will use the query language specified in the state machine level, or <code>JSONPath</code> if none.
 * <p>
 * If the state machine level <code>QueryLanguage</code>  is set to <code>JSONPath</code>, then any individual state-level <code>QueryLanguage</code> can be set to either <code>JSONPath</code> or <code>JSONata</code> to support incremental upgrades.
 * If the state-level <code>QueryLanguage</code> is set to <code>JSONata</code>, then any individual state-level <code>QueryLanguage</code> can either be <code>JSONata</code> or not set.
 * <p>
 * <blockquote><pre>
 * Pass jsonata = Pass.jsonata(this, "JSONata");
 * Pass jsonPath = Pass.jsonPath(this, "JSONPath");
 * Chain definition = jsonata.next(jsonPath);
 * 
 * StateMachine.Builder.create(this, "MixedStateMachine")
 *         // queryLanguage: sfn.QueryLanguage.JSON_PATH, // default
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .build();
 * 
 * // This throws an error. If JSONata is specified at the top level, JSONPath cannot be used in the state machine definition.
 * // This throws an error. If JSONata is specified at the top level, JSONPath cannot be used in the state machine definition.
 * StateMachine.Builder.create(this, "JSONataOnlyStateMachine")
 *         .queryLanguage(QueryLanguage.JSONATA)
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The AWS CDK defines state constructs, and there are 3 ways to initialize them.
 * <p>
 * | Method | Query Language | Description |
 * | ------ | ------- | ------- |
 * | <code>State.jsonata()</code> | <code>JSONata</code> | Use this method to specify a state definition using JSONata only fields.  |
 * | <code>State.jsonPath()</code> | <code>JSONPath</code> | Use this method to specify a state definition using JSONPath only fields. |
 * | <code>new State()</code> | <code>JSONata</code> or <code>JSONPath</code> | This is a legacy pattern. Since fields for both JSONata and JSONPath can be used, it is recommended to use <code>State.jsonata()</code> or <code>State.jsonPath()</code> for better type safety and clarity. |
 * <p>
 * Code examples for initializing a <code>Pass</code> State with each pattern are shown below.
 * <p>
 * <blockquote><pre>
 * // JSONata Pattern
 * Pass.jsonata(this, "JSONata Pattern", PassJsonataProps.builder()
 *         .outputs(Map.of("foo", "bar"))
 *         .build());
 * 
 * // JSONPath Pattern
 * Pass.jsonPath(this, "JSONPath Pattern", PassJsonPathProps.builder()
 *         // The outputs does not exist in the props type
 *         // outputs: { foo: 'bar' },
 *         .outputPath("$.status")
 *         .build());
 * 
 * // Constructor (Legacy) Pattern
 * // Constructor (Legacy) Pattern
 * Pass.Builder.create(this, "Constructor Pattern")
 *         .queryLanguage(QueryLanguage.JSONATA) // or JSON_PATH
 *         // Both outputs and outputPath exist as prop types.
 *         .outputs(Map.of("foo", "bar")) // For JSONata
 *         // or
 *         .outputPath("$.status")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Learn more in the blog post <a href="https://aws.amazon.com/jp/blogs/compute/simplifying-developer-experience-with-variables-and-jsonata-in-aws-step-functions/">Simplifying developer experience with variables and JSONata in AWS Step Functions</a>.
 * <p>
 * <h3>JSONata example</h3>
 * <p>
 * The following example defines a state machine in <code>JSONata</code> that calls a fictional service API to update issue labels.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.events.*;
 * Connection connection;
 * 
 * 
 * HttpInvoke getIssue = HttpInvoke.jsonata(this, "Get Issue", HttpInvokeJsonataProps.builder()
 *         .connection(connection)
 *         .apiRoot("{% 'https://' &amp; $states.input.hostname %}")
 *         .apiEndpoint(TaskInput.fromText("{% 'issues/' &amp; $states.input.issue.id %}"))
 *         .method(TaskInput.fromText("GET"))
 *         // Parse the API call result to object and set to the variables
 *         .assign(Map.of(
 *                 "hostname", "{% $states.input.hostname %}",
 *                 "issue", "{% $parse($states.result.ResponseBody) %}"))
 *         .build());
 * 
 * HttpInvoke updateLabels = HttpInvoke.jsonata(this, "Update Issue Labels", HttpInvokeJsonataProps.builder()
 *         .connection(connection)
 *         .apiRoot("{% 'https://' &amp; $states.input.hostname %}")
 *         .apiEndpoint(TaskInput.fromText("{% 'issues/' &amp; $states.input.issue.id &amp; 'labels' %}"))
 *         .method(TaskInput.fromText("POST"))
 *         .body(TaskInput.fromObject(Map.of(
 *                 "labels", "{% [$type, $component] %}")))
 *         .build());
 * Pass notMatchTitleTemplate = Pass.jsonata(this, "Not Match Title Template");
 * 
 * Chain definition = getIssue.next(Choice.jsonata(this, "Match Title Template?").when(Condition.jsonata("{% $contains($issue.title, /(feat)|(fix)|(chore)(w*):.*&#47;) %}"), updateLabels, ChoiceTransitionOptions.builder()
 *         .assign(Map.of(
 *                 "type", "{% $match($states.input.title, /(w*)((.*))/).groups[0] %}",
 *                 "component", "{% $match($states.input.title, /(w*)((.*))/).groups[1] %}"))
 *         .build()).otherwise(notMatchTitleTemplate));
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .timeout(Duration.minutes(5))
 *         .comment("automate issue labeling state machine")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>JSONPath (Legacy pattern) example</h3>
 * <p>
 * Defining a workflow looks like this (for the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/sample-project-job-poller.html">Step Functions Job Poller
 * example</a>):
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function submitLambda;
 * Function getStatusLambda;
 * 
 * 
 * LambdaInvoke submitJob = LambdaInvoke.Builder.create(this, "Submit Job")
 *         .lambdaFunction(submitLambda)
 *         // Lambda's result is in the attribute `guid`
 *         .outputPath("$.guid")
 *         .build();
 * 
 * Wait waitX = Wait.Builder.create(this, "Wait X Seconds")
 *         .time(WaitTime.secondsPath("$.waitSeconds"))
 *         .build();
 * 
 * LambdaInvoke getStatus = LambdaInvoke.Builder.create(this, "Get Job Status")
 *         .lambdaFunction(getStatusLambda)
 *         // Pass just the field named "guid" into the Lambda, put the
 *         // Lambda's result in a field called "status" in the response
 *         .inputPath("$.guid")
 *         .outputPath("$.status")
 *         .build();
 * 
 * Fail jobFailed = Fail.Builder.create(this, "Job Failed")
 *         .cause("AWS Batch Job Failed")
 *         .error("DescribeJob returned FAILED")
 *         .build();
 * 
 * LambdaInvoke finalStatus = LambdaInvoke.Builder.create(this, "Get Final Job Status")
 *         .lambdaFunction(getStatusLambda)
 *         // Use "guid" field as input
 *         .inputPath("$.guid")
 *         .outputPath("$.Payload")
 *         .build();
 * 
 * Chain definition = submitJob.next(waitX).next(getStatus).next(new Choice(this, "Job Complete?").when(Condition.stringEquals("$.status", "FAILED"), jobFailed).when(Condition.stringEquals("$.status", "SUCCEEDED"), finalStatus).otherwise(waitX));
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .timeout(Duration.minutes(5))
 *         .comment("a super cool state machine")
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can find more sample snippets and learn more about the service integrations
 * in the <code>aws-cdk-lib/aws-stepfunctions-tasks</code> package.
 * <p>
 * <h2>State Machine Data</h2>
 * <p>
 * With variables and state output, you can pass data between the steps of your workflow.
 * <p>
 * Using workflow variables, you can store data in a step and retrieve that data in future steps. For example, you could store an API response that contains data you might need later. Conversely, state output can only be used as input to the very next step.
 * <p>
 * <h3>Variable</h3>
 * <p>
 * With workflow variables, you can store data to reference later. For example, Step 1 might store the result from an API request so a part of that request can be re-used later in Step 5.
 * <p>
 * In the following scenario, the state machine fetches data from an API once. In Step 1, the workflow stores the returned API data (up to 256 KiB per state) in a variable ‘x’ to use in later steps.
 * <p>
 * Without variables, you would need to pass the data through output from Step 1 to Step 2 to Step 3 to Step 4 to use it in Step 5. What if those intermediate steps do not need the data? Passing data from state to state through outputs and input would be unnecessary effort.
 * <p>
 * With variables, you can store data and use it in any future step. You can also modify, rearrange, or add steps without disrupting the flow of your data. Given the flexibility of variables, you might only need to use output to return data from <code>Parallel</code> and <code>Map</code> sub-workflows, and at the end of your state machine execution.
 * <p>
 * <blockquote><pre>
 * (Start)
 *    ↓
 * [Step 1]→[Return from API]
 *    ↓             ↓
 * [Step 2] [Assign data to $x]
 *    ↓             │
 * [Step 3]         │
 *    ↓             │
 * [Step 4]         │
 *    ↓             │
 * [Step 5]←────────┘
 *    ↓     Use variable $x
 *  (End)
 * </pre></blockquote>
 * <p>
 * Use <code>assign</code> to express the above example in AWS CDK. You can use both JSONata and JSONPath to assign.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function callApiFunc;
 * Function useVariableFunc;
 * 
 * LambdaInvoke step1 = LambdaInvoke.jsonata(this, "Step 1", LambdaInvokeJsonataProps.builder()
 *         .lambdaFunction(callApiFunc)
 *         .assign(Map.of(
 *                 "x", "{% $states.result.Payload.x %}"))
 *         .build());
 * Pass step2 = Pass.jsonata(this, "Step 2");
 * Pass step3 = Pass.jsonata(this, "Step 3");
 * Pass step4 = Pass.jsonata(this, "Step 4");
 * LambdaInvoke step5 = LambdaInvoke.jsonata(this, "Step 5", LambdaInvokeJsonataProps.builder()
 *         .lambdaFunction(useVariableFunc)
 *         .payload(TaskInput.fromObject(Map.of(
 *                 "x", "{% $x %}")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * For more details, see the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html">official documentation</a>
 * <p>
 * <h3>State Output</h3>
 * <p>
 * An Execution represents each time the State Machine is run. Every Execution has <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-data.html">State Machine
 * Data</a>:
 * a JSON document containing keys and values that is fed into the state machine,
 * gets modified by individual steps as the state machine progresses, and finally
 * is produced as output.
 * <p>
 * By default, the entire Data object is passed into every state, and the return data of the step
 * becomes new the new Data object. You can change this behavior, but the way you specify it differs depending on the query language you use.
 * <p>
 * <h4>JSONata</h4>
 * <p>
 * To change the default behavior of using a JSONata, supplying values for <code>outputs</code>. When a string in the value of an ASL field, a JSON object field, or a JSON array element is surrounded by <code>{% %}</code> characters, that string will be evaluated as JSONata . Note, the string must start with <code>{%</code> with no leading spaces, and must end with <code>%}</code> with no trailing spaces. Improperly opening or closing the expression will result in a validation error.
 * <p>
 * The following example uses JSONata expressions for <code>outputs</code> and <code>time</code>.
 * <p>
 * <blockquote><pre>
 * Wait.jsonata(this, "Wait", WaitJsonataProps.builder()
 *         .time(WaitTime.timestamp("{% $timestamp %}"))
 *         .outputs(Map.of(
 *                 "stringArgument", "inital-task",
 *                 "numberArgument", 123,
 *                 "booleanArgument", true,
 *                 "arrayArgument", List.of(1, "{% $number %}", 3),
 *                 "intrinsicFunctionsArgument", "{% $join($each($obj, function($v) { $v }), ', ') %}"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * For a brief introduction and complete JSONata reference, see <a href="https://docs.jsonata.org/overview.html">JSONata.org documentation</a>.
 * <p>
 * <h5>Reserved variable : $states</h5>
 * <p>
 * Step Functions defines a single reserved variable called $states. In JSONata states, the following structures are assigned to $states for use in JSONata expressions:
 * <p>
 * <blockquote><pre>
 * // Reserved $states variable in JSONata states
 * const $states = {
 *   "input":      // Original input to the state
 *   "result":     // API or sub-workflow's result (if successful)
 *   "errorOutput":// Error Output in a Catch
 *   "context":    // Context object
 * }
 * </pre></blockquote>
 * <p>
 * The structure is as follows when expressed as a type.
 * <p>
 * <blockquote><pre>
 * public class JsonataStates {
 *     private Object input;
 *     public Object getInput() {
 *         return this.input;
 *     }
 *     public JsonataStates input(Object input) {
 *         this.input = input;
 *         return this;
 *     }
 *     private Object result;
 *     public Object getResult() {
 *         return this.result;
 *     }
 *     public JsonataStates result(Object result) {
 *         this.result = result;
 *         return this;
 *     }
 *     private Map&lt;String, Object&gt; errorOutput;
 *     public Map&lt;String, Object&gt; getErrorOutput() {
 *         return this.errorOutput;
 *     }
 *     public JsonataStates errorOutput(Map&lt;String, Object&gt; errorOutput) {
 *         this.errorOutput = errorOutput;
 *         return this;
 *     }
 *     private Map&lt;String, Object&gt; context;
 *     public Map&lt;String, Object&gt; getContext() {
 *         return this.context;
 *     }
 *     public JsonataStates context(Map&lt;String, Object&gt; context) {
 *         this.context = context;
 *         return this;
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * You can access reserved variables as follows:
 * <p>
 * <blockquote><pre>
 * Pass.jsonata(this, "Pass", PassJsonataProps.builder()
 *         .outputs(Map.of(
 *                 "foo", "{% $states.input.foo %}"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>JSONPath</h4>
 * <p>
 * To change the default behavior of using a JSON path, supplying values for <code>inputPath</code>, <code>resultSelector</code>, <code>resultPath</code>, and <code>outputPath</code>.
 * <p>
 * <h5>Manipulating state machine data using inputPath, resultSelector, resultPath and outputPath</h5>
 * <p>
 * These properties impact how each individual step interacts with the state machine data:
 * <p>
 * <ul>
 * <li><code>stateName</code>: the name of the state in the state machine definition. If not supplied, defaults to the construct id.</li>
 * <li><code>inputPath</code>: the part of the data object that gets passed to the step (<code>itemsPath</code> for <code>Map</code> states)</li>
 * <li><code>resultSelector</code>: the part of the step result that should be added to the state machine data</li>
 * <li><code>resultPath</code>: where in the state machine data the step result should be inserted</li>
 * <li><code>outputPath</code>: what part of the state machine data should be retained</li>
 * <li><code>errorPath</code>: the part of the data object that gets returned as the step error</li>
 * <li><code>causePath</code>: the part of the data object that gets returned as the step cause</li>
 * </ul>
 * <p>
 * Their values should be a string indicating a <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html">JSON path</a> into the State Machine Data object (like <code>"$.MyKey"</code>). If absent, the values are treated as if they were <code>"$"</code>, which means the entire object.
 * <p>
 * The following pseudocode shows how AWS Step Functions uses these parameters when executing a step:
 * <p>
 * <blockquote><pre>
 * // Schematically show how Step Functions evaluates functions.
 * // [] represents indexing into an object by a using JSON path.
 * 
 * input = state[inputPath]
 * 
 * result = invoke_step(select_parameters(input))
 * 
 * state[resultPath] = result[resultSelector]
 * 
 * state = state[outputPath]
 * </pre></blockquote>
 * <p>
 * Instead of a JSON path string, each of these paths can also have the special value <code>JsonPath.DISCARD</code>, which causes the corresponding indexing expression to return an empty object (<code>{}</code>). Effectively, that means there will be an empty input object, an empty result object, no effect on the state, or an empty state, respectively.
 * <p>
 * Some steps (mostly Tasks) have <em>Parameters</em>, which are selected differently. See the next section.
 * <p>
 * See the official documentation on <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html">input and output processing in Step Functions</a>.
 * <p>
 * <h3>Passing Parameters to Tasks</h3>
 * <p>
 * Tasks take parameters, whose values can be taken from the State Machine Data object. For example, your
 * workflow may want to start a CodeBuild with an environment variable that is taken from the State Machine data, or pass part of the State Machine Data into an AWS Lambda Function.
 * <p>
 * In the original JSON-based states language used by AWS Step Functions, you would
 * add <code>.$</code> to the end of a key to indicate that a value needs to be interpreted as
 * a JSON path. In the CDK API you do not change the names of any keys. Instead, you
 * pass special values. There are 3 types of task inputs to consider:
 * <p>
 * <ul>
 * <li>Tasks that accept a "payload" type of input (like AWS Lambda invocations, or posting messages to SNS topics or SQS queues), will take an object of type <code>TaskInput</code>, like <code>TaskInput.fromObject()</code> or <code>TaskInput.fromJsonPathAt()</code>.</li>
 * <li>When tasks expect individual string or number values to customize their behavior, you can also pass a value constructed by <code>JsonPath.stringAt()</code> or <code>JsonPath.numberAt()</code>.</li>
 * <li>When tasks expect strongly-typed resources and you want to vary the resource that is referenced based on a name from the State Machine Data, reference the resource as if it was external (using <code>JsonPath.stringAt()</code>). For example, for a Lambda function: <code>Function.fromFunctionName(this, 'ReferencedFunction', JsonPath.stringAt('$.MyFunctionName'))</code>.</li>
 * </ul>
 * <p>
 * For example, to pass the value that's in the data key of <code>OrderId</code> to a Lambda
 * function as you invoke it, use <code>JsonPath.stringAt('$.OrderId')</code>, like so:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function orderFn;
 * 
 * 
 * LambdaInvoke submitJob = LambdaInvoke.Builder.create(this, "InvokeOrderProcessor")
 *         .lambdaFunction(orderFn)
 *         .payload(TaskInput.fromObject(Map.of(
 *                 "OrderId", JsonPath.stringAt("$.OrderId"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following methods are available:
 * <p>
 * | Method | Purpose |
 * |--------|---------|
 * | <code>JsonPath.stringAt('$.Field')</code> | reference a field, return the type as a <code>string</code>. |
 * | <code>JsonPath.listAt('$.Field')</code> | reference a field, return the type as a list of strings. |
 * | <code>JsonPath.numberAt('$.Field')</code> | reference a field, return the type as a number. Use this for functions that expect a number argument. |
 * | <code>JsonPath.objectAt('$.Field')</code> | reference a field, return the type as an <code>IResolvable</code>. Use this for functions that expect an object argument. |
 * | <code>JsonPath.entirePayload</code> | reference the entire data object (equivalent to a path of <code>$</code>). |
 * | <code>JsonPath.taskToken</code> | reference the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token">Task Token</a>, used for integration patterns that need to run for a long time. |
 * | <code>JsonPath.executionId</code> | reference the Execution Id field of the context object. |
 * | <code>JsonPath.executionInput</code> | reference the Execution Input object of the context object. |
 * | <code>JsonPath.executionName</code> | reference the Execution Name field of the context object. |
 * | <code>JsonPath.executionRoleArn</code> | reference the Execution RoleArn field of the context object. |
 * | <code>JsonPath.executionStartTime</code> | reference the Execution StartTime field of the context object. |
 * | <code>JsonPath.stateEnteredTime</code> | reference the State EnteredTime field of the context object. |
 * | <code>JsonPath.stateName</code> | reference the State Name field of the context object. |
 * | <code>JsonPath.stateRetryCount</code> | reference the State RetryCount field of the context object. |
 * | <code>JsonPath.stateMachineId</code> | reference the StateMachine Id field of the context object. |
 * | <code>JsonPath.stateMachineName</code> | reference the StateMachine Name field of the context object. |
 * <p>
 * You can also call <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html">intrinsic functions</a> using the methods on <code>JsonPath</code>:
 * <p>
 * | Method | Purpose |
 * |--------|---------|
 * | <code>JsonPath.array(JsonPath.stringAt('$.Field'), ...)</code> | make an array from other elements. |
 * | <code>JsonPath.arrayPartition(JsonPath.listAt('$.inputArray'), 4)</code> | partition an array. |
 * | <code>JsonPath.arrayContains(JsonPath.listAt('$.inputArray'), 5)</code> | determine if a specific value is present in an array. |
 * | <code>JsonPath.arrayRange(1, 9, 2)</code> | create a new array containing a specific range of elements. |
 * | <code>JsonPath.arrayGetItem(JsonPath.listAt('$.inputArray'), 5)</code> | get a specified index's value in an array. |
 * | <code>JsonPath.arrayLength(JsonPath.listAt('$.inputArray'))</code> | get the length of an array. |
 * | <code>JsonPath.arrayUnique(JsonPath.listAt('$.inputArray'))</code> | remove duplicate values from an array. |
 * | <code>JsonPath.base64Encode(JsonPath.stringAt('$.input'))</code> | encode data based on MIME Base64 encoding scheme. |
 * | <code>JsonPath.base64Decode(JsonPath.stringAt('$.base64'))</code> | decode data based on MIME Base64 decoding scheme. |
 * | <code>JsonPath.hash(JsonPath.objectAt('$.Data'), JsonPath.stringAt('$.Algorithm'))</code> | calculate the hash value of a given input. |
 * | <code>JsonPath.jsonMerge(JsonPath.objectAt('$.Obj1'), JsonPath.objectAt('$.Obj2'))</code> | merge two JSON objects into a single object. |
 * | <code>JsonPath.stringToJson(JsonPath.stringAt('$.ObjStr'))</code> | parse a JSON string to an object |
 * | <code>JsonPath.jsonToString(JsonPath.objectAt('$.Obj'))</code> | stringify an object to a JSON string |
 * | <code>JsonPath.mathRandom(1, 999)</code> | return a random number. |
 * | <code>JsonPath.mathAdd(JsonPath.numberAt('$.value1'), JsonPath.numberAt('$.step'))</code> | return the sum of two numbers. |
 * | <code>JsonPath.stringSplit(JsonPath.stringAt('$.inputString'), JsonPath.stringAt('$.splitter'))</code> | split a string into an array of values. |
 * | <code>JsonPath.uuid()</code> | return a version 4 universally unique identifier (v4 UUID). |
 * | <code>JsonPath.format('The value is {}.', JsonPath.stringAt('$.Value'))</code> | insert elements into a format string. |
 * <p>
 * <h2>Amazon States Language</h2>
 * <p>
 * This library comes with a set of classes that model the <a href="https://states-language.net/spec.html">Amazon States
 * Language</a>. The following State classes
 * are supported:
 * <p>
 * <ul>
 * <li><a href="#task"><code>Task</code></a></li>
 * <li><a href="#pass"><code>Pass</code></a></li>
 * <li><a href="#wait"><code>Wait</code></a></li>
 * <li><a href="#choice"><code>Choice</code></a></li>
 * <li><a href="#parallel"><code>Parallel</code></a></li>
 * <li><a href="#succeed"><code>Succeed</code></a></li>
 * <li><a href="#fail"><code>Fail</code></a></li>
 * <li><a href="#map"><code>Map</code></a></li>
 * <li><a href="#distributed-map"><code>Distributed Map</code></a></li>
 * <li><a href="#custom-state"><code>Custom State</code></a></li>
 * </ul>
 * <p>
 * An arbitrary JSON object (specified at execution start) is passed from state to
 * state and transformed during the execution of the workflow. For more
 * information, see the States Language spec.
 * <p>
 * <h3>Task</h3>
 * <p>
 * A <code>Task</code> represents some work that needs to be done. Do not use the <code>Task</code> class directly.
 * <p>
 * Instead, use one of the classes in the <code>aws-cdk-lib/aws-stepfunctions-tasks</code> module,
 * which provide a much more ergonomic way to integrate with various AWS services.
 * <p>
 * <h3>Pass</h3>
 * <p>
 * A <code>Pass</code> state passes its input to its output, without performing work.
 * Pass states are useful when constructing and debugging state machines.
 * <p>
 * The following example injects some fixed data into the state machine through
 * the <code>result</code> field. The <code>result</code> field will be added to the input and the result
 * will be passed as the state's output.
 * <p>
 * <blockquote><pre>
 * // Makes the current JSON state { ..., "subObject": { "hello": "world" } }
 * Pass pass = Pass.Builder.create(this, "Add Hello World")
 *         .result(Result.fromObject(Map.of("hello", "world")))
 *         .resultPath("$.subObject")
 *         .build();
 * 
 * // Set the next state
 * Pass nextState = new Pass(this, "NextState");
 * pass.next(nextState);
 * </pre></blockquote>
 * <p>
 * When using JSONata, you can use only <code>outputs</code>.
 * <p>
 * <blockquote><pre>
 * Pass pass = Pass.Builder.create(this, "Add Hello World")
 *         .outputs(Map.of(
 *                 "subObject", Map.of("hello", "world")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>Pass</code> state also supports passing key-value pairs as input. Values can
 * be static, or selected from the input with a path.
 * <p>
 * The following example filters the <code>greeting</code> field from the state input
 * and also injects a field called <code>otherData</code>.
 * <p>
 * <blockquote><pre>
 * Pass pass = Pass.Builder.create(this, "Filter input and inject data")
 *         .stateName("my-pass-state") // the custom state name for the Pass state, defaults to 'Filter input and inject data' as the state name
 *         .parameters(Map.of( // input to the pass state
 *                 "input", JsonPath.stringAt("$.input.greeting"),
 *                 "otherData", "some-extra-stuff"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The object specified in <code>parameters</code> will be the input of the <code>Pass</code> state.
 * Since neither <code>Result</code> nor <code>ResultPath</code> are supplied, the <code>Pass</code> state copies
 * its input through to its output.
 * <p>
 * Learn more about the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-pass-state.html">Pass state</a>
 * <p>
 * <h3>Wait</h3>
 * <p>
 * A <code>Wait</code> state waits for a given number of seconds, or until the current time
 * hits a particular time. The time to wait may be taken from the execution's JSON
 * state.
 * <p>
 * <blockquote><pre>
 * // Wait until it's the time mentioned in the the state object's "triggerTime"
 * // field.
 * Wait wait = Wait.Builder.create(this, "Wait For Trigger Time")
 *         .time(WaitTime.timestampPath("$.triggerTime"))
 *         .build();
 * 
 * // When using JSONata
 * // const wait = sfn.Wait.jsonata(this, 'Wait For Trigger Time', {
 * //   time: sfn.WaitTime.timestamp('{% $triggerTime %}'),
 * // });
 * 
 * // Set the next state
 * Pass startTheWork = new Pass(this, "StartTheWork");
 * wait.next(startTheWork);
 * </pre></blockquote>
 * <p>
 * <h3>Choice</h3>
 * <p>
 * A <code>Choice</code> state can take a different path through the workflow based on the
 * values in the execution's JSON state:
 * <p>
 * <blockquote><pre>
 * Choice choice = new Choice(this, "Did it work?");
 * 
 * // Add conditions with .when()
 * Pass successState = new Pass(this, "SuccessState");
 * Pass failureState = new Pass(this, "FailureState");
 * choice.when(Condition.stringEquals("$.status", "SUCCESS"), successState);
 * choice.when(Condition.numberGreaterThan("$.attempts", 5), failureState);
 * 
 * // When using JSONata
 * // choice.when(sfn.Condition.jsonata("{% $status = 'SUCCESS'"), successState);
 * // choice.when(sfn.Condition.jsonata('{% $attempts &gt; 5 %}'), failureState);
 * 
 * // Use .otherwise() to indicate what should be done if none of the conditions match
 * Pass tryAgainState = new Pass(this, "TryAgainState");
 * choice.otherwise(tryAgainState);
 * </pre></blockquote>
 * <p>
 * If you want to temporarily branch your workflow based on a condition, but have
 * all branches come together and continuing as one (similar to how an <code>if ... then ... else</code> works in a programming language), use the <code>.afterwards()</code> method:
 * <p>
 * <blockquote><pre>
 * Choice choice = new Choice(this, "What color is it?");
 * Pass handleBlueItem = new Pass(this, "HandleBlueItem");
 * Pass handleRedItem = new Pass(this, "HandleRedItem");
 * Pass handleOtherItemColor = new Pass(this, "HanldeOtherItemColor");
 * choice.when(Condition.stringEquals("$.color", "BLUE"), handleBlueItem);
 * choice.when(Condition.stringEquals("$.color", "RED"), handleRedItem);
 * choice.otherwise(handleOtherItemColor);
 * 
 * // Use .afterwards() to join all possible paths back together and continue
 * Pass shipTheItem = new Pass(this, "ShipTheItem");
 * choice.afterwards().next(shipTheItem);
 * </pre></blockquote>
 * <p>
 * You can add comments to <code>Choice</code> states as well as conditions that use <code>choice.when</code>.
 * <p>
 * <blockquote><pre>
 * Choice choice = Choice.Builder.create(this, "What color is it?")
 *         .comment("color comment")
 *         .build();
 * Pass handleBlueItem = new Pass(this, "HandleBlueItem");
 * Pass handleOtherItemColor = new Pass(this, "HanldeOtherItemColor");
 * choice.when(Condition.stringEquals("$.color", "BLUE"), handleBlueItem, ChoiceTransitionOptions.builder()
 *         .comment("blue item comment")
 *         .build());
 * choice.otherwise(handleOtherItemColor);
 * </pre></blockquote>
 * <p>
 * If your <code>Choice</code> doesn't have an <code>otherwise()</code> and none of the conditions match
 * the JSON state, a <code>NoChoiceMatched</code> error will be thrown. Wrap the state machine
 * in a <code>Parallel</code> state if you want to catch and recover from this.
 * <p>
 * <h4>Available Conditions</h4>
 * <p>
 * <h4>JSONata</h4>
 * <p>
 * When you're using JSONata, use the <code>jsonata</code> function to specify the condition using a JSONata expression:
 * <p>
 * <blockquote><pre>
 * Condition.jsonata("{% 1+1 = 2 %}"); // true
 * Condition.jsonata("{% 1+1 != 3 %}"); // true
 * Condition.jsonata("{% 'world' in ['hello', 'world'] %}"); // true
 * Condition.jsonata("{% $contains('abracadabra', /a.*a/) %}");
 * </pre></blockquote>
 * <p>
 * See the <a href="https://docs.jsonata.org/comparison-operators">JSONata comparison operators</a> to find more operators.
 * <p>
 * <h5>JSONPath</h5>
 * <p>
 * see <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-choice-state.html#amazon-states-language-choice-state-rules">step function comparison operators</a>
 * <p>
 * <ul>
 * <li><code>Condition.isPresent</code> - matches if a json path is present</li>
 * <li><code>Condition.isNotPresent</code> - matches if a json path is not present</li>
 * <li><code>Condition.isString</code> - matches if a json path contains a string</li>
 * <li><code>Condition.isNotString</code> - matches if a json path is not a string</li>
 * <li><code>Condition.isNumeric</code> - matches if a json path is numeric</li>
 * <li><code>Condition.isNotNumeric</code> - matches if a json path is not numeric</li>
 * <li><code>Condition.isBoolean</code> - matches if a json path is boolean</li>
 * <li><code>Condition.isNotBoolean</code> - matches if a json path is not boolean</li>
 * <li><code>Condition.isTimestamp</code> - matches if a json path is a timestamp</li>
 * <li><code>Condition.isNotTimestamp</code> - matches if a json path is not a timestamp</li>
 * <li><code>Condition.isNotNull</code> - matches if a json path is not null</li>
 * <li><code>Condition.isNull</code> - matches if a json path is null</li>
 * <li><code>Condition.booleanEquals</code> - matches if a boolean field has a given value</li>
 * <li><code>Condition.booleanEqualsJsonPath</code> - matches if a boolean field equals a value in a given mapping path</li>
 * <li><code>Condition.stringEqualsJsonPath</code> - matches if a string field equals a given mapping path</li>
 * <li><code>Condition.stringEquals</code> - matches if a field equals a string value</li>
 * <li><code>Condition.stringLessThan</code> - matches if a string field sorts before a given value</li>
 * <li><code>Condition.stringLessThanJsonPath</code> - matches if a string field sorts before a value at given mapping path</li>
 * <li><code>Condition.stringLessThanEquals</code> - matches if a string field sorts equal to or before a given value</li>
 * <li><code>Condition.stringLessThanEqualsJsonPath</code> - matches if a string field sorts equal to or before a given mapping</li>
 * <li><code>Condition.stringGreaterThan</code> - matches if a string field sorts after a given value</li>
 * <li><code>Condition.stringGreaterThanJsonPath</code> - matches if a string field sorts after a value at a given mapping path</li>
 * <li><code>Condition.stringGreaterThanEqualsJsonPath</code> - matches if a string field sorts after or equal to value at a given mapping path</li>
 * <li><code>Condition.stringGreaterThanEquals</code> - matches if a string field sorts after or equal to a given value</li>
 * <li><code>Condition.numberEquals</code> - matches if a numeric field has the given value</li>
 * <li><code>Condition.numberEqualsJsonPath</code> - matches if a numeric field has the value in a given mapping path</li>
 * <li><code>Condition.numberLessThan</code> - matches if a numeric field is less than the given value</li>
 * <li><code>Condition.numberLessThanJsonPath</code> - matches if a numeric field is less than the value at the given mapping path</li>
 * <li><code>Condition.numberLessThanEquals</code> - matches if a numeric field is less than or equal to the given value</li>
 * <li><code>Condition.numberLessThanEqualsJsonPath</code> - matches if a numeric field is less than or equal to the numeric value at given mapping path</li>
 * <li><code>Condition.numberGreaterThan</code> - matches if a numeric field is greater than the given value</li>
 * <li><code>Condition.numberGreaterThanJsonPath</code> - matches if a numeric field is greater than the value at a given mapping path</li>
 * <li><code>Condition.numberGreaterThanEquals</code> - matches if a numeric field is greater than or equal to the given value</li>
 * <li><code>Condition.numberGreaterThanEqualsJsonPath</code> - matches if a numeric field is greater than or equal to the value at a given mapping path</li>
 * <li><code>Condition.timestampEquals</code> - matches if a timestamp field is the same time as the given timestamp</li>
 * <li><code>Condition.timestampEqualsJsonPath</code> - matches if a timestamp field is the same time as the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampLessThan</code> - matches if a timestamp field is before the given timestamp</li>
 * <li><code>Condition.timestampLessThanJsonPath</code> - matches if a timestamp field is before the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampLessThanEquals</code> - matches if a timestamp field is before or equal to the given timestamp</li>
 * <li><code>Condition.timestampLessThanEqualsJsonPath</code> - matches if a timestamp field is before or equal to the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampGreaterThan</code> - matches if a timestamp field is after the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampGreaterThanJsonPath</code> - matches if a timestamp field is after the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampGreaterThanEquals</code> - matches if a timestamp field is after or equal to the given timestamp</li>
 * <li><code>Condition.timestampGreaterThanEqualsJsonPath</code> - matches if a timestamp field is after or equal to the timestamp at a given mapping path</li>
 * <li><code>Condition.stringMatches</code> - matches if a field matches a string pattern that can contain a wild card (<em>) e.g: log-</em>.txt or <em>LATEST</em>. No other characters other than "<em>" have any special meaning - * can be escaped: \</em></li>
 * </ul>
 * <p>
 * <h3>Parallel</h3>
 * <p>
 * A <code>Parallel</code> state executes one or more subworkflows in parallel. It can also
 * be used to catch and recover from errors in subworkflows. The <code>parameters</code> property
 * can be used to transform the input that is passed to each branch of the parallel execution.
 * <p>
 * <blockquote><pre>
 * Parallel parallel = new Parallel(this, "Do the work in parallel");
 * 
 * // Add branches to be executed in parallel
 * Pass shipItem = new Pass(this, "ShipItem");
 * Pass sendInvoice = new Pass(this, "SendInvoice");
 * Pass restock = new Pass(this, "Restock");
 * parallel.branch(shipItem);
 * parallel.branch(sendInvoice);
 * parallel.branch(restock);
 * 
 * // Retry the whole workflow if something goes wrong with exponential backoff
 * parallel.addRetry(RetryProps.builder()
 *         .maxAttempts(1)
 *         .maxDelay(Duration.seconds(5))
 *         .jitterStrategy(JitterType.FULL)
 *         .build());
 * 
 * // How to recover from errors
 * Pass sendFailureNotification = new Pass(this, "SendFailureNotification");
 * parallel.addCatch(sendFailureNotification);
 * 
 * // What to do in case everything succeeded
 * Pass closeOrder = new Pass(this, "CloseOrder");
 * parallel.next(closeOrder);
 * </pre></blockquote>
 * <p>
 * <h3>Succeed</h3>
 * <p>
 * Reaching a <code>Succeed</code> state terminates the state machine execution with a
 * successful status.
 * <p>
 * <blockquote><pre>
 * Succeed success = new Succeed(this, "We did it!");
 * </pre></blockquote>
 * <p>
 * <h3>Fail</h3>
 * <p>
 * Reaching a <code>Fail</code> state terminates the state machine execution with a
 * failure status. The fail state should report the reason for the failure.
 * Failures can be caught by encompassing <code>Parallel</code> states.
 * <p>
 * <blockquote><pre>
 * Fail fail = Fail.Builder.create(this, "Fail")
 *         .error("WorkflowFailure")
 *         .cause("Something went wrong")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>Fail</code> state also supports returning dynamic values as the error and cause that are selected from the input with a path.
 * <p>
 * <blockquote><pre>
 * Fail fail = Fail.Builder.create(this, "Fail")
 *         .errorPath(JsonPath.stringAt("$.someError"))
 *         .causePath(JsonPath.stringAt("$.someCause"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also use an intrinsic function that returns a string to specify CausePath and ErrorPath.
 * The available functions include States.Format, States.JsonToString, States.ArrayGetItem, States.Base64Encode, States.Base64Decode, States.Hash, and States.UUID.
 * <p>
 * <blockquote><pre>
 * Fail fail = Fail.Builder.create(this, "Fail")
 *         .errorPath(JsonPath.format("error: {}.", JsonPath.stringAt("$.someError")))
 *         .causePath("States.Format('cause: {}.', $.someCause)")
 *         .build();
 * </pre></blockquote>
 * <p>
 * When you use JSONata, you can use JSONata expression in the <code>error</code> or <code>cause</code> properties.
 * <p>
 * <blockquote><pre>
 * Fail fail = Fail.Builder.create(this, "Fail")
 *         .error("{% 'error:' &amp; $someError &amp; '.' %}")
 *         .cause("{% 'cause:' &amp; $someCause &amp; '.' %}")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Map</h3>
 * <p>
 * A <code>Map</code> state can be used to run a set of steps for each element of an input array.
 * A <code>Map</code> state will execute the same steps for multiple entries of an array in the state input.
 * <p>
 * While the <code>Parallel</code> state executes multiple branches of steps using the same input, a <code>Map</code> state will
 * execute the same steps for multiple entries of an array in the state input.
 * <p>
 * <blockquote><pre>
 * Map map = Map.Builder.create(this, "Map State")
 *         .maxConcurrency(1)
 *         .itemsPath(JsonPath.stringAt("$.inputForMap"))
 *         .itemSelector(Map.of(
 *                 "item", JsonPath.stringAt("$.Map.Item.Value")))
 *         .resultPath("$.mapOutput")
 *         .build();
 * 
 * // The Map iterator can contain a IChainable, which can be an individual or multiple steps chained together.
 * // Below example is with a Choice and Pass step
 * Choice choice = new Choice(this, "Choice");
 * Condition condition1 = Condition.stringEquals("$.item.status", "SUCCESS");
 * Pass step1 = new Pass(this, "Step1");
 * Pass step2 = new Pass(this, "Step2");
 * Pass finish = new Pass(this, "Finish");
 * 
 * Chain definition = choice.when(condition1, step1).otherwise(step2).afterwards().next(finish);
 * 
 * map.itemProcessor(definition);
 * </pre></blockquote>
 * <p>
 * When using <code>JSONata</code>, the <code>itemSelector</code> property in a Map state can be specified in one of two ways. You can provide a valid JSON object containing JSONata expressions for each value:
 * <p>
 * <blockquote><pre>
 * Map map = Map.Builder.create(this, "Map State")
 *         .maxConcurrency(1)
 *         .itemSelector(Map.of(
 *                 "id", "{% $states.context.Map.Item.Value.id %}",
 *                 "status", "{% $states.context.Map.Item.Value.status %}"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can use the <code>jsonataItemSelector</code> field to directly supply a JSONata string that evaluates to a complete JSON object:
 * <p>
 * <blockquote><pre>
 * Map map = Map.Builder.create(this, "Map State")
 *         .maxConcurrency(1)
 *         .jsonataItemSelector("{% {\"id\": $states.input.id, \"status\": $states.input.status} %}")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To define a distributed <code>Map</code> state set <code>itemProcessors</code> mode to <code>ProcessorMode.DISTRIBUTED</code>.
 * An <code>executionType</code> must be specified for the distributed <code>Map</code> workflow.
 * <p>
 * <blockquote><pre>
 * Map map = Map.Builder.create(this, "Map State")
 *         .maxConcurrency(1)
 *         .itemsPath(JsonPath.stringAt("$.inputForMap"))
 *         .itemSelector(Map.of(
 *                 "item", JsonPath.stringAt("$.Map.Item.Value")))
 *         .resultPath("$.mapOutput")
 *         .build();
 * 
 * map.itemProcessor(new Pass(this, "Pass State"), ProcessorConfig.builder()
 *         .mode(ProcessorMode.DISTRIBUTED)
 *         .executionType(ProcessorType.STANDARD)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.aws.amazon.com/step-functions/latest/dg/use-dist-map-orchestrate-large-scale-parallel-workloads.html">Using Map state in Distributed mode to orchestrate large-scale parallel workloads</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Distributed Map</h3>
 * <p>
 * Step Functions provides a high-concurrency mode for the Map state known as Distributed mode. In this mode, the Map state can accept input from large-scale Amazon S3 data sources. For example, your input can be a JSON or CSV file stored in an Amazon S3 bucket, or a JSON array passed from a previous step in the workflow. A Map state set to Distributed is known as a Distributed Map state. In this mode, the Map state runs each iteration as a child workflow execution, which enables high concurrency of up to 10,000 parallel child workflow executions. Each child workflow execution has its own, separate execution history from that of the parent workflow.
 * <p>
 * Use the Map state in Distributed mode when you need to orchestrate large-scale parallel workloads that meet any combination of the following conditions:
 * <p>
 * <ul>
 * <li>The size of your dataset exceeds 256 KB.</li>
 * <li>The workflow's execution event history exceeds 25,000 entries.</li>
 * <li>You need a concurrency of more than 40 parallel iterations.</li>
 * </ul>
 * <p>
 * A <code>DistributedMap</code> state can be used to run a set of steps for each element of an input array with high concurrency.
 * A <code>DistributedMap</code> state will execute the same steps for multiple entries of an array in the state input or from S3 objects.
 * <p>
 * <blockquote><pre>
 * DistributedMap distributedMap = DistributedMap.Builder.create(this, "Distributed Map State")
 *         .maxConcurrency(1)
 *         .itemsPath(JsonPath.stringAt("$.inputForMap"))
 *         .build();
 * distributedMap.itemProcessor(new Pass(this, "Pass State"));
 * </pre></blockquote>
 * <p>
 * <code>DistributedMap</code> supports various input source types to determine a list of objects to iterate over:
 * <p>
 * <ul>
 * <li>JSON array from the JSON state input
 * <p>
 * <ul>
 * <li>By default, <code>DistributedMap</code> assumes whole JSON state input is an JSON array and iterates over it:</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * /**
 *  * JSON state input:
 *  *  [
 *  *    "item1",
 *  *    "item2"
 *  *  ]
 *  *&#47;
 * DistributedMap distributedMap = new DistributedMap(this, "DistributedMap");
 * distributedMap.itemProcessor(new Pass(this, "Pass"));
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>When input source is present at a specific path in JSON state input, then <code>itemsPath</code> can be utilised to configure the iterator source.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * /**
 *  * JSON state input:
 *  *  {
 *  *    "distributedMapItemList": [
 *  *      "item1",
 *  *      "item2"
 *  *    ]
 *  *  }
 *  *&#47;
 * DistributedMap distributedMap = DistributedMap.Builder.create(this, "DistributedMap")
 *         .itemsPath("$.distributedMapItemList")
 *         .build();
 * distributedMap.itemProcessor(new Pass(this, "Pass"));
 * </pre></blockquote></li>
 * <li>Objects in a S3 bucket with an optional prefix.
 * <p>
 * <ul>
 * <li>When <code>DistributedMap</code> is required to iterate over objects stored in a S3 bucket, then an object of <code>S3ObjectsItemReader</code> can be passed to <code>itemReader</code> to configure the iterator source. Note that <code>S3ObjectsItemReader</code> will default to use Distributed map's query language. If the
 * map does not specify a query language, then it falls back to the State machine's query language. An exmaple of using <code>S3ObjectsItemReader</code>
 * is as follows:</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * 
 * 
 * /**
 *  * Tree view of bucket:
 *  *  my-bucket
 *  *  |
 *  *  +--item1
 *  *  |
 *  *  +--otherItem
 *  *  |
 *  *  +--item2
 *  *  |
 *  *  ...
 *  *&#47;
 * Bucket bucket = Bucket.Builder.create(this, "Bucket")
 *         .bucketName("my-bucket")
 *         .build();
 * 
 * DistributedMap distributedMap = DistributedMap.Builder.create(this, "DistributedMap")
 *         .itemReader(S3ObjectsItemReader.Builder.create()
 *                 .bucket(bucket)
 *                 .prefix("item")
 *                 .build())
 *         .build();
 * distributedMap.itemProcessor(new Pass(this, "Pass"));
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>If information about <code>bucket</code> is only known while starting execution of <code>StateMachine</code> (dynamically or at run-time) via JSON state input:</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * /**
 *  * JSON state input:
 *  *  {
 *  *    "bucketName": "my-bucket",
 *  *    "prefix": "item"
 *  *  }
 *  *&#47;
 * DistributedMap distributedMap = DistributedMap.Builder.create(this, "DistributedMap")
 *         .itemReader(S3ObjectsItemReader.Builder.create()
 *                 .bucketNamePath(JsonPath.stringAt("$.bucketName"))
 *                 .prefix(JsonPath.stringAt("$.prefix"))
 *                 .build())
 *         .build();
 * distributedMap.itemProcessor(new Pass(this, "Pass"));
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>Both <code>bucket</code> and <code>bucketNamePath</code> are mutually exclusive.</li>
 * </ul></li>
 * <li>JSON array in a JSON file stored in S3
 * <p>
 * <ul>
 * <li>When <code>DistributedMap</code> is required to iterate over a JSON array stored in a JSON file in a S3 bucket, then an object of <code>S3JsonItemReader</code> can be passed to <code>itemReader</code> to configure the iterator source as follows:</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * 
 * 
 * /**
 *  * Tree view of bucket:
 *  *  my-bucket
 *  *  |
 *  *  +--input.json
 *  *  |
 *  *  ...
 *  *
 *  * File content of input.json:
 *  *  [
 *  *    "item1",
 *  *    "item2"
 *  *  ]
 *  *&#47;
 * Bucket bucket = Bucket.Builder.create(this, "Bucket")
 *         .bucketName("my-bucket")
 *         .build();
 * 
 * DistributedMap distributedMap = DistributedMap.Builder.create(this, "DistributedMap")
 *         .itemReader(S3JsonItemReader.Builder.create()
 *                 .bucket(bucket)
 *                 .key("input.json")
 *                 .build())
 *         .build();
 * distributedMap.itemProcessor(new Pass(this, "Pass"));
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>If information about <code>bucket</code> is only known while starting execution of <code>StateMachine</code> (dynamically or at run-time) via state input:</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * /**
 *  * JSON state input:
 *  *  {
 *  *    "bucketName": "my-bucket",
 *  *    "key": "input.json"
 *  *  }
 *  *&#47;
 * DistributedMap distributedMap = DistributedMap.Builder.create(this, "DistributedMap")
 *         .itemReader(S3JsonItemReader.Builder.create()
 *                 .bucketNamePath(JsonPath.stringAt("$.bucketName"))
 *                 .key(JsonPath.stringAt("$.key"))
 *                 .build())
 *         .build();
 * distributedMap.itemProcessor(new Pass(this, "Pass"));
 * </pre></blockquote></li>
 * <li>CSV file stored in S3</li>
 * <li>S3 inventory manifest stored in S3</li>
 * </ul>
 * <p>
 * Map states in Distributed mode also support writing results of the iterator to an S3 bucket and optional prefix.  Use a <code>ResultWriterV2</code> object provided via the optional <code>resultWriter</code> property to configure which S3 location iterator results will be written. The default behavior id <code>resultWriter</code> is omitted is to use the state output payload. However, if the iterator results are larger than the 256 kb limit for Step Functions payloads then the State Machine will fail.
 * <p>
 * ResultWriterV2 object will default to use the Distributed map's query language. If the Distributed map's does not specify a query language, then it
 * will fall back to the State machine's query langauge.
 * <p>
 * Note: <code>ResultWriter</code> has been deprecated, use <code>ResultWriterV2</code> instead. To enable <code>ResultWriterV2</code>,
 * you will have to set the value for <code>'&#64;aws-cdk/aws-stepfunctions:useDistributedMapResultWriterV2'</code> to true in the  CDK context
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * 
 * 
 * // create a bucket
 * Bucket bucket = new Bucket(this, "Bucket");
 * 
 * // create a WriterConfig
 * 
 * DistributedMap distributedMap = DistributedMap.Builder.create(this, "Distributed Map State")
 *         .resultWriterV2(ResultWriterV2.Builder.create()
 *                 .bucket(bucket)
 *                 .prefix("my-prefix")
 *                 .writerConfig(Map.of(
 *                         "outputType", OutputType.JSONL,
 *                         "transformation", Transformation.NONE))
 *                 .build())
 *         .build();
 * distributedMap.itemProcessor(new Pass(this, "Pass State"));
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>If information about <code>bucket</code> is only known while starting execution of <code>StateMachine</code> (dynamically or at run-time) via JSON state input:</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * /**
 *  * JSON state input:
 *  *  {
 *  *    "bucketName": "my-bucket"
 *  *  }
 *  *&#47;
 * DistributedMap distributedMap = DistributedMap.Builder.create(this, "DistributedMap")
 *         .resultWriterV2(ResultWriterV2.Builder.create()
 *                 .bucketNamePath(JsonPath.stringAt("$.bucketName"))
 *                 .build())
 *         .build();
 * distributedMap.itemProcessor(new Pass(this, "Pass"));
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>Both <code>bucket</code> and <code>bucketNamePath</code> are mutually exclusive.</li>
 * </ul>
 * <p>
 * If you want to specify the execution type for the ItemProcessor in the DistributedMap, you must set the <code>mapExecutionType</code> property in the <code>DistributedMap</code> class. When using the <code>DistributedMap</code> class, the <code>ProcessorConfig.executionType</code> property is ignored.
 * <p>
 * In the following example, the execution type for the ItemProcessor in the DistributedMap is set to <code>EXPRESS</code> based on the value specified for <code>mapExecutionType</code>.
 * <p>
 * <blockquote><pre>
 * DistributedMap distributedMap = DistributedMap.Builder.create(this, "DistributedMap")
 *         .mapExecutionType(StateMachineType.EXPRESS)
 *         .build();
 * 
 * distributedMap.itemProcessor(new Pass(this, "Pass"), ProcessorConfig.builder()
 *         .mode(ProcessorMode.DISTRIBUTED)
 *         .executionType(ProcessorType.STANDARD)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Custom State</h3>
 * <p>
 * It's possible that the high-level constructs for the states or <code>stepfunctions-tasks</code> do not have
 * the states or service integrations you are looking for. The primary reasons for this lack of
 * functionality are:
 * <p>
 * <ul>
 * <li>A <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-service-integrations.html">service integration</a> is available through Amazon States Language, but not available as construct
 * classes in the CDK.</li>
 * <li>The state or state properties are available through Step Functions, but are not configurable
 * through constructs</li>
 * </ul>
 * <p>
 * If a feature is not available, a <code>CustomState</code> can be used to supply any Amazon States Language
 * JSON-based object as the state definition.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/tutorial-code-snippet.html#tutorial-code-snippet-1">Code Snippets</a> are available and can be plugged in as the state definition.
 * <p>
 * Custom states can be chained together with any of the other states to create your state machine
 * definition. You will also need to provide any permissions that are required to the <code>role</code> that
 * the State Machine uses.
 * <p>
 * The Retry and Catch fields are available for error handling.
 * You can configure the Retry field by defining it in the JSON object or by adding it using the <code>addRetry</code> method.
 * However, the Catch field cannot be configured by defining it in the JSON object, so it must be added using the <code>addCatch</code> method.
 * <p>
 * The following example uses the <code>DynamoDB</code> service integration to insert data into a DynamoDB table.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.dynamodb.*;
 * 
 * 
 * // create a table
 * Table table = Table.Builder.create(this, "montable")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .build();
 * 
 * Pass finalStatus = new Pass(this, "final step");
 * 
 * // States language JSON to put an item into DynamoDB
 * // snippet generated from https://docs.aws.amazon.com/step-functions/latest/dg/tutorial-code-snippet.html#tutorial-code-snippet-1
 * Map&lt;String, Object&gt; stateJson = Map.of(
 *         "Type", "Task",
 *         "Resource", "arn:aws:states:::dynamodb:putItem",
 *         "Parameters", Map.of(
 *                 "TableName", table.getTableName(),
 *                 "Item", Map.of(
 *                         "id", Map.of(
 *                                 "S", "MyEntry"))),
 *         "ResultPath", null);
 * 
 * // custom state which represents a task to insert data into DynamoDB
 * CustomState custom = CustomState.Builder.create(this, "my custom task")
 *         .stateJson(stateJson)
 *         .build();
 * 
 * // catch errors with addCatch
 * Pass errorHandler = new Pass(this, "handle failure");
 * custom.addCatch(errorHandler);
 * 
 * // retry the task if something goes wrong
 * custom.addRetry(RetryProps.builder()
 *         .errors(List.of(Errors.ALL))
 *         .interval(Duration.seconds(10))
 *         .maxAttempts(5)
 *         .build());
 * 
 * Chain chain = Chain.start(custom).next(finalStatus);
 * 
 * StateMachine sm = StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(chain))
 *         .timeout(Duration.seconds(30))
 *         .comment("a super cool state machine")
 *         .build();
 * 
 * // don't forget permissions. You need to assign them
 * table.grantWriteData(sm);
 * </pre></blockquote>
 * <p>
 * <h2>Task Chaining</h2>
 * <p>
 * To make defining work flows as convenient (and readable in a top-to-bottom way)
 * as writing regular programs, it is possible to chain most methods invocations.
 * In particular, the <code>.next()</code> method can be repeated. The result of a series of
 * <code>.next()</code> calls is called a <strong>Chain</strong>, and can be used when defining the jump
 * targets of <code>Choice.on</code> or <code>Parallel.branch</code>:
 * <p>
 * <blockquote><pre>
 * Pass step1 = new Pass(this, "Step1");
 * Pass step2 = new Pass(this, "Step2");
 * Pass step3 = new Pass(this, "Step3");
 * Pass step4 = new Pass(this, "Step4");
 * Pass step5 = new Pass(this, "Step5");
 * Pass step6 = new Pass(this, "Step6");
 * Pass step7 = new Pass(this, "Step7");
 * Pass step8 = new Pass(this, "Step8");
 * Pass step9 = new Pass(this, "Step9");
 * Pass step10 = new Pass(this, "Step10");
 * Choice choice = new Choice(this, "Choice");
 * Condition condition1 = Condition.stringEquals("$.status", "SUCCESS");
 * Parallel parallel = new Parallel(this, "Parallel");
 * Pass finish = new Pass(this, "Finish");
 * 
 * Chain definition = step1.next(step2).next(choice.when(condition1, step3.next(step4).next(step5)).otherwise(step6).afterwards()).next(parallel.branch(step7.next(step8)).branch(step9.next(step10))).next(finish);
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you don't like the visual look of starting a chain directly off the first
 * step, you can use <code>Chain.start</code>:
 * <p>
 * <blockquote><pre>
 * Pass step1 = new Pass(this, "Step1");
 * Pass step2 = new Pass(this, "Step2");
 * Pass step3 = new Pass(this, "Step3");
 * 
 * Chain definition = Chain.start(step1).next(step2).next(step3);
 * </pre></blockquote>
 * <p>
 * <h2>Task Credentials</h2>
 * <p>
 * Tasks are executed using the State Machine's execution role. In some cases, e.g. cross-account access, an IAM role can be assumed by the State Machine's execution role to provide access to the resource.
 * This can be achieved by providing the optional <code>credentials</code> property which allows using a fixed role or a json expression to resolve the role at runtime from the task's inputs.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function submitLambda;
 * Role iamRole;
 * 
 * 
 * // use a fixed role for all task invocations
 * TaskRole role = TaskRole.fromRole(iamRole);
 * // or use a json expression to resolve the role at runtime based on task inputs
 * //const role = sfn.TaskRole.fromRoleArnJsonPath('$.RoleArn');
 * 
 * LambdaInvoke submitJob = LambdaInvoke.Builder.create(this, "Submit Job")
 *         .lambdaFunction(submitLambda)
 *         .outputPath("$.Payload")
 *         // use credentials
 *         .credentials(Credentials.builder().role(role).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html">the AWS documentation</a>
 * to learn more about AWS Step Functions support for accessing resources in other AWS accounts.
 * <p>
 * <h2>Service Integration Patterns</h2>
 * <p>
 * AWS Step functions integrate directly with other services, either through an optimised integration pattern, or through the AWS SDK.
 * Therefore, it is possible to change the <code>integrationPattern</code> of services, to enable additional functionality of the said AWS Service:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.glue.alpha.*;
 * 
 * Job submitGlue;
 * 
 * 
 * GlueStartJobRun submitJob = GlueStartJobRun.Builder.create(this, "Submit Job")
 *         .glueJobName(submitGlue.getJobName())
 *         .integrationPattern(IntegrationPattern.RUN_JOB)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>State Machine Fragments</h2>
 * <p>
 * It is possible to define reusable (or abstracted) mini-state machines by
 * defining a construct that implements <code>IChainable</code>, which requires you to define
 * two fields:
 * <p>
 * <ul>
 * <li><code>startState: State</code>, representing the entry point into this state machine.</li>
 * <li><code>endStates: INextable[]</code>, representing the (one or more) states that outgoing
 * transitions will be added to if you chain onto the fragment.</li>
 * </ul>
 * <p>
 * Since states will be named after their construct IDs, you may need to prefix the
 * IDs of states if you plan to instantiate the same state machine fragment
 * multiples times (otherwise all states in every instantiation would have the same
 * name).
 * <p>
 * The class <code>StateMachineFragment</code> contains some helper functions (like
 * <code>prefixStates()</code>) to make it easier for you to do this. If you define your state
 * machine as a subclass of this, it will be convenient to use:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.Stack;
 * import software.constructs.Construct;
 * import software.amazon.awscdk.services.stepfunctions.*;
 * 
 * public class MyJobProps {
 *     private String jobFlavor;
 *     public String getJobFlavor() {
 *         return this.jobFlavor;
 *     }
 *     public MyJobProps jobFlavor(String jobFlavor) {
 *         this.jobFlavor = jobFlavor;
 *         return this;
 *     }
 * }
 * 
 * public class MyJob extends StateMachineFragment {
 *     public final State startState;
 *     public final INextable[] endStates;
 * 
 *     public MyJob(Construct parent, String id, MyJobProps props) {
 *         super(parent, id);
 * 
 *         Choice choice = new Choice(this, "Choice").when(Condition.stringEquals("$.branch", "left"), new Pass(this, "Left Branch")).when(Condition.stringEquals("$.branch", "right"), new Pass(this, "Right Branch"));
 * 
 *         // ...
 * 
 *         this.startState = choice;
 *         this.endStates = choice.afterwards().getEndStates();
 *     }
 * }
 * 
 * public class MyStack extends Stack {
 *     public MyStack(Construct scope, String id) {
 *         super(scope, id);
 *         // Do 3 different variants of MyJob in parallel
 *         Parallel parallel = new Parallel(this, "All jobs").branch(new MyJob(this, "Quick", new MyJobProps().jobFlavor("quick")).prefixStates()).branch(new MyJob(this, "Medium", new MyJobProps().jobFlavor("medium")).prefixStates()).branch(new MyJob(this, "Slow", new MyJobProps().jobFlavor("slow")).prefixStates());
 * 
 *         StateMachine.Builder.create(this, "MyStateMachine")
 *                 .definitionBody(DefinitionBody.fromChainable(parallel))
 *                 .build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * A few utility functions are available to parse state machine fragments.
 * <p>
 * <ul>
 * <li><code>State.findReachableStates</code>: Retrieve the list of states reachable from a given state.</li>
 * <li><code>State.findReachableEndStates</code>: Retrieve the list of end or terminal states reachable from a given state.</li>
 * </ul>
 * <p>
 * <h2>Activity</h2>
 * <p>
 * <strong>Activities</strong> represent work that is done on some non-Lambda worker pool. The
 * Step Functions workflow will submit work to this Activity, and a worker pool
 * that you run yourself, probably on EC2, will pull jobs from the Activity and
 * submit the results of individual jobs back.
 * <p>
 * You need the ARN to do so, so if you use Activities be sure to pass the Activity
 * ARN into your worker pool:
 * <p>
 * <blockquote><pre>
 * Activity activity = new Activity(this, "Activity");
 * 
 * // Read this CloudFormation Output from your application and use it to poll for work on
 * // the activity.
 * // Read this CloudFormation Output from your application and use it to poll for work on
 * // the activity.
 * CfnOutput.Builder.create(this, "ActivityArn").value(activity.getActivityArn()).build();
 * </pre></blockquote>
 * <p>
 * <h3>Activity-Level Permissions</h3>
 * <p>
 * Granting IAM permissions to an activity can be achieved by calling the <code>grant(principal, actions)</code> API:
 * <p>
 * <blockquote><pre>
 * Activity activity = new Activity(this, "Activity");
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * 
 * activity.grant(role, "states:SendTaskSuccess");
 * </pre></blockquote>
 * <p>
 * This will grant the IAM principal the specified actions onto the activity.
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * <code>Task</code> object expose various metrics on the execution of that particular task. For example,
 * to create an alarm on a particular task failing:
 * <p>
 * <blockquote><pre>
 * Task task;
 * 
 * Alarm.Builder.create(this, "TaskAlarm")
 *         .metric(task.metricFailed())
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * There are also metrics on the complete state machine:
 * <p>
 * <blockquote><pre>
 * StateMachine stateMachine;
 * 
 * Alarm.Builder.create(this, "StateMachineAlarm")
 *         .metric(stateMachine.metricFailed())
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * And there are metrics on the capacity of all state machines in your account:
 * <p>
 * <blockquote><pre>
 * Alarm.Builder.create(this, "ThrottledAlarm")
 *         .metric(StateTransitionMetric.metricThrottledEvents())
 *         .threshold(10)
 *         .evaluationPeriods(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Error names</h2>
 * <p>
 * Step Functions identifies errors in the Amazon States Language using case-sensitive strings, known as error names.
 * The Amazon States Language defines a set of built-in strings that name well-known errors, all beginning with the <code>States.</code> prefix.
 * <p>
 * <ul>
 * <li><code>States.ALL</code> - A wildcard that matches any known error name.</li>
 * <li><code>States.Runtime</code> - An execution failed due to some exception that could not be processed. Often these are caused by errors at runtime, such as attempting to apply InputPath or OutputPath on a null JSON payload. A <code>States.Runtime</code> error is not retriable, and will always cause the execution to fail. A retry or catch on <code>States.ALL</code> will NOT catch States.Runtime errors.</li>
 * <li><code>States.DataLimitExceeded</code> - A States.DataLimitExceeded exception will be thrown for the following:
 * <p>
 * <ul>
 * <li>When the output of a connector is larger than payload size quota.</li>
 * <li>When the output of a state is larger than payload size quota.</li>
 * <li>When, after Parameters processing, the input of a state is larger than the payload size quota.</li>
 * <li>See <a href="https://docs.aws.amazon.com/step-functions/latest/dg/limits-overview.html">the AWS documentation</a> to learn more about AWS Step Functions Quotas.</li>
 * </ul></li>
 * <li><code>States.HeartbeatTimeout</code> - A Task state failed to send a heartbeat for a period longer than the HeartbeatSeconds value.</li>
 * <li><code>States.Timeout</code> - A Task state either ran longer than the TimeoutSeconds value, or failed to send a heartbeat for a period longer than the HeartbeatSeconds value.</li>
 * <li><code>States.TaskFailed</code>- A Task state failed during the execution. When used in a retry or catch, <code>States.TaskFailed</code> acts as a wildcard that matches any known error name except for <code>States.Timeout</code>.</li>
 * </ul>
 * <p>
 * <h2>Logging</h2>
 * <p>
 * Enable logging to CloudWatch by passing a logging configuration with a
 * destination LogGroup:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * 
 * 
 * LogGroup logGroup = new LogGroup(this, "MyLogGroup");
 * 
 * Chain definition = Chain.start(new Pass(this, "Pass"));
 * 
 * StateMachine.Builder.create(this, "MyStateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .logs(LogOptions.builder()
 *                 .destination(logGroup)
 *                 .level(LogLevel.ALL)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Encryption</h2>
 * <p>
 * You can encrypt your data using a customer managed key for AWS Step Functions state machines and activities. You can configure a symmetric AWS KMS key and data key reuse period when creating or updating a State Machine or when creating an Activity. The execution history and state machine definition will be encrypted with the key applied to the State Machine. Activity inputs will be encrypted with the key applied to the Activity.
 * <p>
 * <h3>Encrypting state machines</h3>
 * <p>
 * You can provide a symmetric KMS key to encrypt the state machine definition and execution history:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * import software.amazon.awscdk.*;
 * 
 * 
 * Key kmsKey = new Key(this, "Key");
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachineWithCMKEncryptionConfiguration")
 *         .stateMachineName("StateMachineWithCMKEncryptionConfiguration")
 *         .definitionBody(DefinitionBody.fromChainable(Chain.start(new Pass(this, "Pass"))))
 *         .stateMachineType(StateMachineType.STANDARD)
 *         .encryptionConfiguration(new CustomerManagedEncryptionConfiguration(kmsKey, Duration.seconds(60)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Encrypting state machine logs in Cloud Watch Logs</h3>
 * <p>
 * If a state machine is encrypted with a customer managed key and has logging enabled, its decrypted execution history will be stored in CloudWatch Logs. If you want to encrypt the logs from the state machine using your own KMS key, you can do so by configuring the <code>LogGroup</code> associated with the state machine to use a KMS key.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * import software.amazon.awscdk.services.iam.*;
 * import software.amazon.awscdk.services.logs.*;
 * 
 * 
 * Key stateMachineKmsKey = new Key(this, "StateMachine Key");
 * Key logGroupKey = new Key(this, "LogGroup Key");
 * 
 * /*
 *   Required KMS key policy which allows the CloudWatchLogs service principal to encrypt the entire log group using the
 *   customer managed kms key. See: https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/encrypt-log-data-kms.html#cmk-permissions
 * *&#47;
 * logGroupKey.addToResourcePolicy(PolicyStatement.Builder.create()
 *         .resources(List.of("*"))
 *         .actions(List.of("kms:Encrypt*", "kms:Decrypt*", "kms:ReEncrypt*", "kms:GenerateDataKey*", "kms:Describe*"))
 *         .principals(List.of(new ServicePrincipal(String.format("logs.%s.amazonaws.com", cdk.Stack.of(this).getRegion()))))
 *         .conditions(Map.of(
 *                 "ArnEquals", Map.of(
 *                         "kms:EncryptionContext:aws:logs:arn", cdk.Stack.of(this).formatArn(ArnComponents.builder()
 *                                 .service("logs")
 *                                 .resource("log-group")
 *                                 .sep(":")
 *                                 .resourceName("/aws/vendedlogs/states/MyLogGroup")
 *                                 .build()))))
 *         .build());
 * 
 * // Create logGroup and provding encryptionKey which will be used to encrypt the log group
 * LogGroup logGroup = LogGroup.Builder.create(this, "MyLogGroup")
 *         .logGroupName("/aws/vendedlogs/states/MyLogGroup")
 *         .encryptionKey(logGroupKey)
 *         .build();
 * 
 * // Create state machine with CustomerManagedEncryptionConfiguration
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachineWithCMKWithCWLEncryption")
 *         .stateMachineName("StateMachineWithCMKWithCWLEncryption")
 *         .definitionBody(DefinitionBody.fromChainable(Chain.start(Pass.Builder.create(this, "PassState")
 *                 .result(Result.fromString("Hello World"))
 *                 .build())))
 *         .stateMachineType(StateMachineType.STANDARD)
 *         .encryptionConfiguration(new CustomerManagedEncryptionConfiguration(stateMachineKmsKey))
 *         .logs(LogOptions.builder()
 *                 .destination(logGroup)
 *                 .level(LogLevel.ALL)
 *                 .includeExecutionData(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Encrypting activity inputs</h3>
 * <p>
 * When you provide a symmetric KMS key, all inputs from the Step Functions Activity will be encrypted using the provided KMS key:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * import software.amazon.awscdk.*;
 * 
 * 
 * Key kmsKey = new Key(this, "Key");
 * Activity activity = Activity.Builder.create(this, "ActivityWithCMKEncryptionConfiguration")
 *         .activityName("ActivityWithCMKEncryptionConfiguration")
 *         .encryptionConfiguration(new CustomerManagedEncryptionConfiguration(kmsKey, Duration.seconds(75)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Changing Encryption</h3>
 * <p>
 * If you want to switch encryption from a customer provided key to a Step Functions owned key or vice-versa you must explicitly provide <code>encryptionConfiguration?</code>
 * <p>
 * <h4>Example: Switching from a customer managed key to a Step Functions owned key for StateMachine</h4>
 * <p>
 * <h4>Before</h4>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * import software.amazon.awscdk.*;
 * 
 * 
 * Key kmsKey = new Key(this, "Key");
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .stateMachineName("StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(Chain.start(new Pass(this, "Pass"))))
 *         .stateMachineType(StateMachineType.STANDARD)
 *         .encryptionConfiguration(new CustomerManagedEncryptionConfiguration(kmsKey, Duration.seconds(60)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>After</h4>
 * <p>
 * <blockquote><pre>
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .stateMachineName("StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(Chain.start(new Pass(this, "Pass"))))
 *         .stateMachineType(StateMachineType.STANDARD)
 *         .encryptionConfiguration(new AwsOwnedEncryptionConfiguration())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>X-Ray tracing</h2>
 * <p>
 * Enable X-Ray tracing for StateMachine:
 * <p>
 * <blockquote><pre>
 * Chain definition = Chain.start(new Pass(this, "Pass"));
 * 
 * StateMachine.Builder.create(this, "MyStateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .tracingEnabled(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-xray-tracing.html">the AWS documentation</a>
 * to learn more about AWS Step Functions's X-Ray support.
 * <p>
 * <h2>State Machine Permission Grants</h2>
 * <p>
 * IAM roles, users, or groups which need to be able to work with a State Machine should be granted IAM permissions.
 * <p>
 * Any object that implements the <code>IGrantable</code> interface (has an associated principal) can be granted permissions by calling:
 * <p>
 * <ul>
 * <li><code>stateMachine.grantStartExecution(principal)</code> - grants the principal the ability to execute the state machine</li>
 * <li><code>stateMachine.grantRead(principal)</code> - grants the principal read access</li>
 * <li><code>stateMachine.grantTaskResponse(principal)</code> - grants the principal the ability to send task tokens to the state machine</li>
 * <li><code>stateMachine.grantExecution(principal, actions)</code> - grants the principal execution-level permissions for the IAM actions specified</li>
 * <li><code>stateMachine.grantRedriveExecution(principal)</code> - grants the principal permission to redrive the executions of the state machine</li>
 * <li><code>stateMachine.grant(principal, actions)</code> - grants the principal state-machine-level permissions for the IAM actions specified</li>
 * </ul>
 * <p>
 * <h3>Start Execution Permission</h3>
 * <p>
 * Grant permission to start an execution of a state machine by calling the <code>grantStartExecution()</code> API.
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .build();
 * 
 * // Give role permission to start execution of state machine
 * stateMachine.grantStartExecution(role);
 * </pre></blockquote>
 * <p>
 * The following permission is provided to a service principal by the <code>grantStartExecution()</code> API:
 * <p>
 * <ul>
 * <li><code>states:StartExecution</code> - to state machine</li>
 * </ul>
 * <p>
 * <h3>Read Permissions</h3>
 * <p>
 * Grant <code>read</code> access to a state machine by calling the <code>grantRead()</code> API.
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .build();
 * 
 * // Give role read access to state machine
 * stateMachine.grantRead(role);
 * </pre></blockquote>
 * <p>
 * The following read permissions are provided to a service principal by the <code>grantRead()</code> API:
 * <p>
 * <ul>
 * <li><code>states:ListExecutions</code> - to state machine</li>
 * <li><code>states:ListStateMachines</code> - to state machine</li>
 * <li><code>states:DescribeExecution</code> - to executions</li>
 * <li><code>states:DescribeStateMachineForExecution</code> - to executions</li>
 * <li><code>states:GetExecutionHistory</code> - to executions</li>
 * <li><code>states:ListActivities</code> - to <code>*</code></li>
 * <li><code>states:DescribeStateMachine</code> - to <code>*</code></li>
 * <li><code>states:DescribeActivity</code> - to <code>*</code></li>
 * </ul>
 * <p>
 * <h3>Task Response Permissions</h3>
 * <p>
 * Grant permission to allow task responses to a state machine by calling the <code>grantTaskResponse()</code> API:
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .build();
 * 
 * // Give role task response permissions to the state machine
 * stateMachine.grantTaskResponse(role);
 * </pre></blockquote>
 * <p>
 * The following read permissions are provided to a service principal by the <code>grantRead()</code> API:
 * <p>
 * <ul>
 * <li><code>states:SendTaskSuccess</code> - to state machine</li>
 * <li><code>states:SendTaskFailure</code> - to state machine</li>
 * <li><code>states:SendTaskHeartbeat</code> - to state machine</li>
 * </ul>
 * <p>
 * <h3>Execution-level Permissions</h3>
 * <p>
 * Grant execution-level permissions to a state machine by calling the <code>grantExecution()</code> API:
 * <p>
 * <h3>Redrive Execution Permission</h3>
 * <p>
 * Grant the given identity permission to redrive the execution of the state machine:
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .build();
 * 
 * // Give role permission to start execution of state machine
 * stateMachine.grantStartExecution(role);
 * // Give role permission to redrive any executions of the state machine
 * stateMachine.grantRedriveExecution(role);
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .build();
 * 
 * // Give role permission to get execution history of ALL executions for the state machine
 * stateMachine.grantExecution(role, "states:GetExecutionHistory");
 * </pre></blockquote>
 * <p>
 * <h3>Custom Permissions</h3>
 * <p>
 * You can add any set of permissions to a state machine by calling the <code>grant()</code> API.
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * User user = new User(this, "MyUser");
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(definition))
 *         .build();
 * 
 * //give user permission to send task success to the state machine
 * stateMachine.grant(user, "states:SendTaskSuccess");
 * </pre></blockquote>
 * <p>
 * <h2>Import</h2>
 * <p>
 * Any Step Functions state machine that has been created outside the stack can be imported
 * into your CDK stack.
 * <p>
 * State machines can be imported by their ARN via the <code>StateMachine.fromStateMachineArn()</code> API.
 * In addition, the StateMachine can be imported via the <code>StateMachine.fromStateMachineName()</code> method, as long as they are in the same account/region as the current construct.
 * <p>
 * <blockquote><pre>
 * App app = new App();
 * Stack stack = new Stack(app, "MyStack");
 * StateMachine.fromStateMachineArn(this, "ViaArnImportedStateMachine", "arn:aws:states:us-east-1:123456789012:stateMachine:StateMachine2E01A3A5-N5TJppzoevKQ");
 * 
 * StateMachine.fromStateMachineName(this, "ViaResourceNameImportedStateMachine", "StateMachine2E01A3A5-N5TJppzoevKQ");
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.stepfunctions;
