/**
 * <h1>Amazon Relational Database Service Construct Library</h1>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.rds.*;
 * </pre></blockquote>
 * <p>
 * <h2>Starting a clustered database</h2>
 * <p>
 * To set up a clustered database (like Aurora), define a <code>DatabaseCluster</code>. You must
 * always launch a database in a VPC. Use the <code>vpcSubnets</code> attribute to control whether
 * your instances will be launched privately or publicly:
 * <p>
 * You must specify the instance to use as the writer, along with an optional list
 * of readers (up to 15).
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_01_0).build()))
 *         .credentials(Credentials.fromGeneratedSecret("clusteradmin")) // Optional - will default to 'admin' username and generated password
 *         .writer(ClusterInstance.provisioned("writer", ProvisionedClusterInstanceProps.builder()
 *                 .publiclyAccessible(false)
 *                 .build()))
 *         .readers(List.of(ClusterInstance.provisioned("reader1", ProvisionedClusterInstanceProps.builder().promotionTier(1).build()), ClusterInstance.serverlessV2("reader2")))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PRIVATE_WITH_EGRESS)
 *                 .build())
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To adopt Aurora I/O-Optimized, specify <code>DBClusterStorageType.AURORA_IOPT1</code> on the <code>storageType</code> property.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraPostgres(AuroraPostgresClusterEngineProps.builder().version(AuroraPostgresEngineVersion.VER_15_2).build()))
 *         .credentials(Credentials.fromUsername("adminuser", CredentialsFromUsernameOptions.builder().password(SecretValue.unsafePlainText("7959866cacc02c2d243ecfe177464fe6")).build()))
 *         .writer(ClusterInstance.provisioned("writer", ProvisionedClusterInstanceProps.builder()
 *                 .publiclyAccessible(false)
 *                 .build()))
 *         .readers(List.of(ClusterInstance.provisioned("reader")))
 *         .storageType(DBClusterStorageType.AURORA_IOPT1)
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PRIVATE_WITH_EGRESS)
 *                 .build())
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If there isn't a constant for the exact version you want to use,
 * all of the <code>Version</code> classes have a static <code>of</code> method that can be used to create an arbitrary version.
 * <p>
 * <blockquote><pre>
 * AuroraMysqlEngineVersion customEngineVersion = AuroraMysqlEngineVersion.of("5.7.mysql_aurora.2.08.1");
 * </pre></blockquote>
 * <p>
 * By default, the master password will be generated and stored in AWS Secrets Manager with auto-generated description.
 * <p>
 * Your cluster will be empty by default. To add a default database upon construction, specify the
 * <code>defaultDatabaseName</code> attribute.
 * <p>
 * When you create a DB instance in your cluster, Aurora automatically chooses an appropriate AZ for that instance if you don't specify an AZ.
 * You can place each instance in fixed availability zone by specifying <code>availabilityZone</code> property.
 * For details, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html">Regions and Availability Zones</a>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_02_1).build()))
 *         .writer(ClusterInstance.provisioned("writer", ProvisionedClusterInstanceProps.builder()
 *                 .availabilityZone("us-east-1a")
 *                 .build()))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To use dual-stack mode, specify <code>NetworkType.DUAL</code> on the <code>networkType</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * // VPC and subnets must have IPv6 CIDR blocks
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_02_1).build()))
 *         .writer(ClusterInstance.provisioned("writer", ProvisionedClusterInstanceProps.builder()
 *                 .publiclyAccessible(false)
 *                 .build()))
 *         .vpc(vpc)
 *         .networkType(NetworkType.DUAL)
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more information about dual-stack mode, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html">Working with a DB cluster in a VPC</a>.
 * <p>
 * If you want to issue read/write transactions directly on an Aurora Replica, you can use local write forwarding on <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-write-forwarding.html">Aurora MySQL</a>
 * and <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-postgresql-write-forwarding.html">Aurora PostgreSQL</a>.
 * Local write forwarding allows read replicas to accept write transactions and forward them to the writer DB instance to be committed.
 * <p>
 * To enable local write forwarding, set the <code>enableLocalWriteForwarding</code> property to <code>true</code>:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * DatabaseCluster.Builder.create(this, "DatabaseCluster")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_07_0).build()))
 *         .writer(ClusterInstance.serverlessV2("writerInstance"))
 *         .readers(List.of(ClusterInstance.serverlessV2("readerInstance1")))
 *         .vpc(vpc)
 *         .enableLocalWriteForwarding(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: Local write forwarding is supported only for Aurora MySQL 3.04 or higher, and for Aurora PostgreSQL
 * 16.4 or higher (for version 16), 15.8 or higher (for version 15), and 14.13 or higher (for version 14).
 * <p>
 * Use <code>DatabaseClusterFromSnapshot</code> to create a cluster from a snapshot:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseClusterFromSnapshot.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.aurora(AuroraClusterEngineProps.builder().version(AuroraEngineVersion.VER_1_22_2).build()))
 *         .writer(ClusterInstance.provisioned("writer"))
 *         .vpc(vpc)
 *         .snapshotIdentifier("mySnapshot")
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, automatic minor version upgrades for the engine type are enabled in a cluster, but you can also disable this.
 * To do so, set <code>autoMinorVersionUpgrade</code> to <code>false</code>.
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * DatabaseCluster.Builder.create(this, "DatabaseCluster")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_07_0).build()))
 *         .writer(ClusterInstance.serverlessV2("writerInstance"))
 *         .vpc(vpc)
 *         .autoMinorVersionUpgrade(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Updating the database instances in a cluster</h3>
 * <p>
 * Database cluster instances may be updated in bulk or on a rolling basis.
 * <p>
 * An update to all instances in a cluster may cause significant downtime. To reduce the downtime, set the
 * <code>instanceUpdateBehavior</code> property in <code>DatabaseClusterBaseProps</code> to <code>InstanceUpdateBehavior.ROLLING</code>.
 * This adds a dependency between each instance so the update is performed on only one instance at a time.
 * <p>
 * Use <code>InstanceUpdateBehavior.BULK</code> to update all instances at once.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_01_0).build()))
 *         .writer(ClusterInstance.provisioned("Instance", ProvisionedClusterInstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.SMALL))
 *                 .build()))
 *         .readers(List.of(ClusterInstance.provisioned("reader")))
 *         .instanceUpdateBehaviour(InstanceUpdateBehaviour.ROLLING) // Optional - defaults to rds.InstanceUpdateBehaviour.BULK
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Serverless V2 instances in a Cluster</h3>
 * <p>
 * It is possible to create an RDS cluster with <em>both</em> serverlessV2 and provisioned
 * instances. For example, this will create a cluster with a provisioned writer and
 * a serverless v2 reader.
 * <p>
 * <blockquote>
 * <p>
 * <em>Note</em> Before getting starting with this type of cluster it is
 * highly recommended that you read through the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html">Developer Guide</a>
 * which goes into much more detail on the things you need to take into
 * consideration.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_01_0).build()))
 *         .writer(ClusterInstance.provisioned("writer"))
 *         .readers(List.of(ClusterInstance.serverlessV2("reader")))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Monitoring</h3>
 * <p>
 * There are some CloudWatch metrics that are <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html#aurora-serverless-v2.viewing.monitoring">important for Aurora Serverless
 * v2</a>.
 * <p>
 * <ul>
 * <li><code>ServerlessDatabaseCapacity</code>: An instance-level metric that can also be
 * evaluated at the cluster level. At the cluster-level it represents the average
 * capacity of all the instances in the cluster.</li>
 * <li><code>ACUUtilization</code>: Value of the <code>ServerlessDatabaseCapacity</code>/ max ACU of the
 * cluster.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_01_0).build()))
 *         .writer(ClusterInstance.provisioned("writer"))
 *         .readers(List.of(ClusterInstance.serverlessV2("reader")))
 *         .vpc(vpc)
 *         .build();
 * 
 * cluster.metricServerlessDatabaseCapacity(MetricOptions.builder()
 *         .period(Duration.minutes(10))
 *         .build()).createAlarm(this, "capacity", CreateAlarmOptions.builder()
 *         .threshold(1.5)
 *         .evaluationPeriods(3)
 *         .build());
 * cluster.metricACUUtilization(MetricOptions.builder()
 *         .period(Duration.minutes(10))
 *         .build()).createAlarm(this, "alarm", CreateAlarmOptions.builder()
 *         .evaluationPeriods(3)
 *         .threshold(90)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Capacity &amp; Scaling</h4>
 * <p>
 * There are some things to take into consideration with Aurora Serverless v2.
 * <p>
 * To create a cluster that can support serverless v2 instances you configure a
 * minimum and maximum capacity range on the cluster. This is an example showing
 * the default values:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_01_0).build()))
 *         .writer(ClusterInstance.serverlessV2("writer"))
 *         .serverlessV2MinCapacity(0.5)
 *         .serverlessV2MaxCapacity(2)
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The capacity is defined as a number of Aurora capacity units (ACUs). You can
 * specify in half-step increments (40, 40.5, 41, etc). Each serverless instance in
 * the cluster inherits the capacity that is defined on the cluster. It is not
 * possible to configure separate capacity at the instance level.
 * <p>
 * The maximum capacity is mainly used for budget control since it allows you to
 * set a cap on how high your instance can scale.
 * <p>
 * The minimum capacity is a little more involved. This controls a couple different
 * things.
 * <p>
 * <ul>
 * <li>The scale-up rate is proportional to the current capacity (larger instances
 * scale up faster)
 * <p>
 * <ul>
 * <li>Adjust the minimum capacity to obtain a suitable scaling rate</li>
 * </ul></li>
 * <li>Network throughput is proportional to capacity</li>
 * </ul>
 * <p>
 * <blockquote>
 * <p>
 * <em>Info</em> More complete details can be found <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html#aurora-serverless-v2-examples-setting-capacity-range-for-cluster">in the docs</a>
 * <p>
 * </blockquote>
 * <p>
 * You can also set minimum capacity to zero ACUs and automatically pause,
 * if they don't have any connections initiated by user activity within a time period specified by <code>serverlessV2AutoPauseDuration</code> (300 seconds by default).
 * For more information, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2-auto-pause.html">Scaling to Zero ACUs with automatic pause and resume for Aurora Serverless v2</a>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_08_0).build()))
 *         .writer(ClusterInstance.serverlessV2("writer"))
 *         .serverlessV2MinCapacity(0)
 *         .serverlessV2AutoPauseDuration(Duration.hours(1))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Another way that you control the capacity/scaling of your serverless v2 reader
 * instances is based on the <a href="https://aws.amazon.com/blogs/aws/additional-failover-control-for-amazon-aurora/">promotion tier</a>
 * which can be between 0-15. Any serverless v2 instance in the 0-1 tiers will scale alongside the
 * writer even if the current read load does not require the capacity. This is
 * because instances in the 0-1 tier are first priority for failover and Aurora
 * wants to ensure that in the event of a failover the reader that gets promoted is
 * scaled to handle the write load.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_01_0).build()))
 *         .writer(ClusterInstance.serverlessV2("writer"))
 *         .readers(List.of(ClusterInstance.serverlessV2("reader1", ServerlessV2ClusterInstanceProps.builder().scaleWithWriter(true).build()), ClusterInstance.serverlessV2("reader2")))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>When the writer scales up, any readers in tier 0-1 will scale up to match</li>
 * <li>Scaling for tier 2-15 is independent of what is happening on the writer</li>
 * <li>Readers in tier 2-15 scale up based on read load against the individual reader</li>
 * </ul>
 * <p>
 * When configuring your cluster it is important to take this into consideration
 * and ensure that in the event of a failover there is an instance that is scaled
 * up to take over.
 * <p>
 * <h3>Mixing Serverless v2 and Provisioned instances</h3>
 * <p>
 * You are able to create a cluster that has both provisioned and serverless
 * instances. <a href="https://aws.amazon.com/blogs/database/evaluate-amazon-aurora-serverless-v2-for-your-provisioned-aurora-clusters/">This blog post</a>
 * has an excellent guide on choosing between serverless and provisioned instances
 * based on use case.
 * <p>
 * There are a couple of high level differences:
 * <p>
 * <ul>
 * <li>Engine Version (serverless only supports MySQL 8+ &amp; PostgreSQL 13+)</li>
 * <li>Memory up to 256GB can be supported by serverless</li>
 * </ul>
 * <p>
 * <h4>Provisioned writer</h4>
 * <p>
 * With a provisioned writer and serverless v2 readers, some of the serverless
 * readers will need to be configured to scale with the writer so they can act as
 * failover targets. You will need to determine the correct capacity based on the
 * provisioned instance type and it's utilization.
 * <p>
 * As an example, if the CPU utilization for a db.r6g.4xlarge (128 GB) instance
 * stays at 10% most times, then the minimum ACUs may be set at 6.5 ACUs
 * (10% of 128 GB) and maximum may be set at 64 ACUs (64x2GB=128GB). Keep in mind
 * that the speed at which the serverless instance can scale up is determined by
 * the minimum capacity so if your cluster has spiky workloads you may need to set
 * a higher minimum capacity.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_01_0).build()))
 *         .writer(ClusterInstance.provisioned("writer", ProvisionedClusterInstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.R6G, InstanceSize.XLARGE4))
 *                 .build()))
 *         .serverlessV2MinCapacity(6.5)
 *         .serverlessV2MaxCapacity(64)
 *         .readers(List.of(ClusterInstance.serverlessV2("reader1", ServerlessV2ClusterInstanceProps.builder().scaleWithWriter(true).build()), ClusterInstance.serverlessV2("reader2")))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * In the above example <code>reader1</code> will scale with the writer based on the writer's
 * utilization. So if the writer were to go to <code>50%</code> utilization then <code>reader1</code>
 * would scale up to use <code>32</code> ACUs. If the read load stayed consistent then
 * <code>reader2</code> may remain at <code>6.5</code> since it is not configured to scale with the
 * writer.
 * <p>
 * If one of your Aurora Serverless v2 DB instances consistently reaches the
 * limit of its maximum capacity, Aurora indicates this condition by setting the
 * DB instance to a status of <code>incompatible-parameters</code>. While the DB instance has
 * the incompatible-parameters status, some operations are blocked. For example,
 * you can't upgrade the engine version.
 * <p>
 * <h3>CA certificate</h3>
 * <p>
 * Use the <code>caCertificate</code> property to specify the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html">CA certificates</a>
 * to use for a cluster instances:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_01_0).build()))
 *         .writer(ClusterInstance.provisioned("writer", ProvisionedClusterInstanceProps.builder()
 *                 .caCertificate(CaCertificate.RDS_CA_RSA2048_G1)
 *                 .build()))
 *         .readers(List.of(ClusterInstance.serverlessV2("reader", ServerlessV2ClusterInstanceProps.builder()
 *                 .caCertificate(CaCertificate.of("custom-ca"))
 *                 .build())))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Scheduling modifications</h3>
 * <p>
 * To schedule modifications to database instances in the next scheduled maintenance window, specify <code>applyImmediately</code> to <code>false</code> in the instance props:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_01_0).build()))
 *         .writer(ClusterInstance.provisioned("writer", ProvisionedClusterInstanceProps.builder()
 *                 .applyImmediately(false)
 *                 .build()))
 *         .readers(List.of(ClusterInstance.serverlessV2("reader", ServerlessV2ClusterInstanceProps.builder()
 *                 .applyImmediately(false)
 *                 .build())))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Until RDS applies the changes, the DB instance remains in a drift state.
 * As a result, the configuration doesn't fully reflect the requested modifications and temporarily diverges from the intended state.
 * <p>
 * Currently, CloudFormation does not support to schedule modifications of the cluster configurations.
 * To apply changes of the cluster, such as engine version, in the next scheduled maintenance window, you should use <code>modify-db-cluster</code> CLI command or management console.
 * <p>
 * For details, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Modifying.html">Modifying an Amazon Aurora DB cluster</a>.
 * <p>
 * <h3>Migrating from instanceProps</h3>
 * <p>
 * Creating instances in a <code>DatabaseCluster</code> using <code>instanceProps</code> &amp; <code>instances</code> is
 * deprecated. To migrate to the new properties you can provide the
 * <code>isFromLegacyInstanceProps</code> property.
 * <p>
 * For example, in order to migrate from this deprecated config:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder()
 *                 .version(AuroraMysqlEngineVersion.VER_3_03_0)
 *                 .build()))
 *         .instances(2)
 *         .instanceProps(InstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.SMALL))
 *                 .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build())
 *                 .vpc(vpc)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You would need to migrate to this. The old method of providing <code>instanceProps</code>
 * and <code>instances</code> will create the number of <code>instances</code> that you provide. The
 * first instance will be the writer and the rest will be the readers. It's
 * important that the <code>id</code> that you provide is <code>Instance{NUMBER}</code>. The writer
 * should always be <code>Instance1</code> and the readers will increment from there.
 * <p>
 * Make sure to run a <code>cdk diff</code> before deploying to make sure that all changes are
 * expected. <strong>Always test the migration in a non-production environment first.</strong>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Map&lt;String, Object&gt; instanceProps = Map.of(
 *         "instanceType", InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.SMALL),
 *         "isFromLegacyInstanceProps", true);
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder()
 *                 .version(AuroraMysqlEngineVersion.VER_3_03_0)
 *                 .build()))
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build())
 *         .vpc(vpc)
 *         .writer(ClusterInstance.provisioned("Instance1", ProvisionedClusterInstanceProps.builder()
 *                 .instanceType(instanceProps.getInstanceType())
 *                 .isFromLegacyInstanceProps(instanceProps.getIsFromLegacyInstanceProps())
 *                 .build()))
 *         .readers(List.of(ClusterInstance.provisioned("Instance2", ProvisionedClusterInstanceProps.builder()
 *                 .instanceType(instanceProps.getInstanceType())
 *                 .isFromLegacyInstanceProps(instanceProps.getIsFromLegacyInstanceProps())
 *                 .build())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Creating a read replica cluster</h3>
 * <p>
 * Use <code>replicationSourceIdentifier</code> to create a read replica cluster:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * DatabaseCluster primaryCluster;
 * 
 * 
 * DatabaseCluster.Builder.create(this, "DatabaseCluster")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder()
 *                 .version(AuroraMysqlEngineVersion.VER_3_03_0)
 *                 .build()))
 *         .writer(ClusterInstance.serverlessV2("Writer"))
 *         .vpc(vpc)
 *         .replicationSourceIdentifier(primaryCluster.getClusterArn())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: Cannot create a read replica cluster with <code>credentials</code> as the value is inherited from the source DB cluster.
 * <p>
 * <h2>Starting an instance database</h2>
 * <p>
 * To set up an instance database, define a <code>DatabaseInstance</code>. You must
 * always launch a database in a VPC. Use the <code>vpcSubnets</code> attribute to control whether
 * your instances will be launched privately or publicly:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder().version(OracleEngineVersion.VER_19_0_0_0_2020_04_R1).build()))
 *         // optional, defaults to m5.large
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.SMALL))
 *         .credentials(Credentials.fromGeneratedSecret("syscdk")) // Optional - will default to 'admin' username and generated password
 *         .vpc(vpc)
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PRIVATE_WITH_EGRESS)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If there isn't a constant for the exact engine version you want to use,
 * all of the <code>Version</code> classes have a static <code>of</code> method that can be used to create an arbitrary version.
 * <p>
 * <blockquote><pre>
 * OracleEngineVersion customEngineVersion = OracleEngineVersion.of("19.0.0.0.ru-2020-04.rur-2020-04.r1", "19");
 * </pre></blockquote>
 * <p>
 * By default, the master password will be generated and stored in AWS Secrets Manager.
 * <p>
 * To use the storage auto scaling option of RDS you can specify the maximum allocated storage.
 * This is the upper limit to which RDS can automatically scale the storage. More info can be found
 * <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling">here</a>
 * Example for max storage configuration:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_16_3).build()))
 *         // optional, defaults to m5.large
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.SMALL))
 *         .vpc(vpc)
 *         .maxAllocatedStorage(200)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When you create a DB instance, you can choose an Availability Zone or have Amazon RDS choose one for you randomly.
 * For details, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.RegionsAndAvailabilityZones.html">Regions, Availability Zones, and Local Zones</a>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_16_3).build()))
 *         .vpc(vpc)
 *         .availabilityZone("us-east-1a")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To use dual-stack mode, specify <code>NetworkType.DUAL</code> on the <code>networkType</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * // VPC and subnets must have IPv6 CIDR blocks
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_16_3).build()))
 *         .vpc(vpc)
 *         .networkType(NetworkType.DUAL)
 *         .publiclyAccessible(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more information about dual-stack mode, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html">Working with a DB instance in a VPC</a>.
 * <p>
 * Use <code>DatabaseInstanceFromSnapshot</code> and <code>DatabaseInstanceReadReplica</code> to create an instance from snapshot or
 * a source database respectively:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance sourceInstance;
 * 
 * DatabaseInstanceFromSnapshot.Builder.create(this, "Instance")
 *         .snapshotIdentifier("my-snapshot")
 *         .engine(DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_16_3).build()))
 *         // optional, defaults to m5.large
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.LARGE))
 *         .vpc(vpc)
 *         .build();
 * DatabaseInstanceReadReplica.Builder.create(this, "ReadReplica")
 *         .sourceDatabaseInstance(sourceInstance)
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.LARGE))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Or you can <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_RestoreFromMultiAZDBClusterSnapshot.html">restore a DB instance from a Multi-AZ DB cluster snapshot</a>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseInstanceFromSnapshot.Builder.create(this, "Instance")
 *         .clusterSnapshotIdentifier("my-cluster-snapshot")
 *         .engine(DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_16_3).build()))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Automatic backups of read replica instances are only supported for MySQL and MariaDB. By default,
 * automatic backups are disabled for read replicas and can only be enabled (using <code>backupRetention</code>)
 * if also enabled on the source instance.
 * <p>
 * Creating a "production" Oracle database instance with option and parameter groups:
 * <p>
 * <blockquote><pre>
 * // Set open cursors with parameter group
 * ParameterGroup parameterGroup = ParameterGroup.Builder.create(this, "ParameterGroup")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder().version(OracleEngineVersion.VER_19_0_0_0_2020_04_R1).build()))
 *         .parameters(Map.of(
 *                 "open_cursors", "2500"))
 *         .build();
 * 
 * OptionGroup optionGroup = OptionGroup.Builder.create(this, "OptionGroup")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder().version(OracleEngineVersion.VER_19_0_0_0_2020_04_R1).build()))
 *         .configurations(List.of(OptionConfiguration.builder()
 *                 .name("LOCATOR")
 *                 .build(), OptionConfiguration.builder()
 *                 .name("OEM")
 *                 .port(1158)
 *                 .vpc(vpc)
 *                 .build()))
 *         .build();
 * 
 * // Allow connections to OEM
 * optionGroup.optionConnections.OEM.connections.allowDefaultPortFromAnyIpv4();
 * 
 * // Database instance with production values
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder().version(OracleEngineVersion.VER_19_0_0_0_2020_04_R1).build()))
 *         .licenseModel(LicenseModel.BRING_YOUR_OWN_LICENSE)
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.MEDIUM))
 *         .multiAz(true)
 *         .storageType(StorageType.IO1)
 *         .credentials(Credentials.fromUsername("syscdk"))
 *         .vpc(vpc)
 *         .databaseName("ORCL")
 *         .storageEncrypted(true)
 *         .backupRetention(Duration.days(7))
 *         .monitoringInterval(Duration.seconds(60))
 *         .enablePerformanceInsights(true)
 *         .cloudwatchLogsExports(List.of("trace", "audit", "alert", "listener"))
 *         .cloudwatchLogsRetention(RetentionDays.ONE_MONTH)
 *         .autoMinorVersionUpgrade(true) // required to be true if LOCATOR is used in the option group
 *         .optionGroup(optionGroup)
 *         .parameterGroup(parameterGroup)
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .build();
 * 
 * // Allow connections on default port from any IPV4
 * instance.connections.allowDefaultPortFromAnyIpv4();
 * 
 * // Rotate the master user password every 30 days
 * instance.addRotationSingleUser();
 * 
 * // Add alarm for high CPU
 * // Add alarm for high CPU
 * Alarm.Builder.create(this, "HighCPU")
 *         .metric(instance.metricCPUUtilization())
 *         .threshold(90)
 *         .evaluationPeriods(1)
 *         .build();
 * 
 * // Trigger Lambda function on instance availability events
 * Function fn = Function.Builder.create(this, "Function")
 *         .code(Code.fromInline("exports.handler = (event) =&gt; console.log(event);"))
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .build();
 * 
 * Rule availabilityRule = instance.onEvent("Availability", OnEventOptions.builder().target(new LambdaFunction(fn)).build());
 * availabilityRule.addEventPattern(EventPattern.builder()
 *         .detail(Map.of(
 *                 "EventCategories", List.of("availability")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Add XMLDB and OEM with option group
 * <p>
 * <blockquote><pre>
 * // Set open cursors with parameter group
 * ParameterGroup parameterGroup = ParameterGroup.Builder.create(this, "ParameterGroup")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder().version(OracleEngineVersion.VER_19_0_0_0_2020_04_R1).build()))
 *         .parameters(Map.of(
 *                 "open_cursors", "2500"))
 *         .build();
 * 
 * OptionGroup optionGroup = OptionGroup.Builder.create(this, "OptionGroup")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder().version(OracleEngineVersion.VER_19_0_0_0_2020_04_R1).build()))
 *         .configurations(List.of(OptionConfiguration.builder()
 *                 .name("LOCATOR")
 *                 .build(), OptionConfiguration.builder()
 *                 .name("OEM")
 *                 .port(1158)
 *                 .vpc(vpc)
 *                 .build()))
 *         .build();
 * 
 * // Allow connections to OEM
 * optionGroup.optionConnections.OEM.connections.allowDefaultPortFromAnyIpv4();
 * 
 * // Database instance with production values
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder().version(OracleEngineVersion.VER_19_0_0_0_2020_04_R1).build()))
 *         .licenseModel(LicenseModel.BRING_YOUR_OWN_LICENSE)
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.MEDIUM))
 *         .multiAz(true)
 *         .storageType(StorageType.IO1)
 *         .credentials(Credentials.fromUsername("syscdk"))
 *         .vpc(vpc)
 *         .databaseName("ORCL")
 *         .storageEncrypted(true)
 *         .backupRetention(Duration.days(7))
 *         .monitoringInterval(Duration.seconds(60))
 *         .enablePerformanceInsights(true)
 *         .cloudwatchLogsExports(List.of("trace", "audit", "alert", "listener"))
 *         .cloudwatchLogsRetention(RetentionDays.ONE_MONTH)
 *         .autoMinorVersionUpgrade(true) // required to be true if LOCATOR is used in the option group
 *         .optionGroup(optionGroup)
 *         .parameterGroup(parameterGroup)
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .build();
 * 
 * // Allow connections on default port from any IPV4
 * instance.connections.allowDefaultPortFromAnyIpv4();
 * 
 * // Rotate the master user password every 30 days
 * instance.addRotationSingleUser();
 * 
 * // Add alarm for high CPU
 * // Add alarm for high CPU
 * Alarm.Builder.create(this, "HighCPU")
 *         .metric(instance.metricCPUUtilization())
 *         .threshold(90)
 *         .evaluationPeriods(1)
 *         .build();
 * 
 * // Trigger Lambda function on instance availability events
 * Function fn = Function.Builder.create(this, "Function")
 *         .code(Code.fromInline("exports.handler = (event) =&gt; console.log(event);"))
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .build();
 * 
 * Rule availabilityRule = instance.onEvent("Availability", OnEventOptions.builder().target(new LambdaFunction(fn)).build());
 * availabilityRule.addEventPattern(EventPattern.builder()
 *         .detail(Map.of(
 *                 "EventCategories", List.of("availability")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Use the <code>storageType</code> property to specify the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html">type of storage</a>
 * to use for the instance:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseInstance iopsInstance = DatabaseInstance.Builder.create(this, "IopsInstance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_39).build()))
 *         .vpc(vpc)
 *         .storageType(StorageType.IO1)
 *         .iops(5000)
 *         .build();
 * 
 * DatabaseInstance gp3Instance = DatabaseInstance.Builder.create(this, "Gp3Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_39).build()))
 *         .vpc(vpc)
 *         .allocatedStorage(500)
 *         .storageType(StorageType.GP3)
 *         .storageThroughput(500)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use the <code>allocatedStorage</code> property to specify the amount of storage (in gigabytes) that is initially allocated for the instance
 * to use for the instance:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * // Setting allocatedStorage for DatabaseInstance replica
 * // Note: If allocatedStorage isn't set here, the replica instance will inherit the allocatedStorage of the source instance
 * DatabaseInstance sourceInstance;
 * 
 * 
 * // Setting allocatedStorage for DatabaseInstance
 * DatabaseInstance iopsInstance = DatabaseInstance.Builder.create(this, "IopsInstance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_39).build()))
 *         .vpc(vpc)
 *         .storageType(StorageType.IO1)
 *         .iops(5000)
 *         .allocatedStorage(500)
 *         .build();
 * DatabaseInstanceReadReplica.Builder.create(this, "ReadReplica")
 *         .sourceDatabaseInstance(sourceInstance)
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.LARGE))
 *         .vpc(vpc)
 *         .allocatedStorage(500)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use the <code>caCertificate</code> property to specify the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html">CA certificates</a>
 * to use for the instance:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_39).build()))
 *         .vpc(vpc)
 *         .caCertificate(CaCertificate.RDS_CA_RSA2048_G1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can specify a custom CA certificate with:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_39).build()))
 *         .vpc(vpc)
 *         .caCertificate(CaCertificate.of("future-rds-ca"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Scheduling modifications</h3>
 * <p>
 * To schedule modifications in the next scheduled maintenance window, specify <code>applyImmediately</code> to <code>false</code>:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_39).build()))
 *         .vpc(vpc)
 *         .applyImmediately(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Until RDS applies the changes, the DB instance remains in a drift state.
 * As a result, the configuration doesn't fully reflect the requested modifications and temporarily diverges from the intended state.
 * <p>
 * For details, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_ModifyInstance.ApplyImmediately.html">Using the schedule modifications setting</a>.
 * <p>
 * <h2>Setting Public Accessibility</h2>
 * <p>
 * You can set public accessibility for the <code>DatabaseInstance</code> or the <code>ClusterInstance</code> using the <code>publiclyAccessible</code> property.
 * If you specify <code>true</code>, it creates an instance with a publicly resolvable DNS name, which resolves to a public IP address.
 * If you specify <code>false</code>, it creates an internal instance with a DNS name that resolves to a private IP address.
 * <p>
 * The default value will be <code>true</code> if <code>vpcSubnets</code> is <code>subnetType: SubnetType.PUBLIC</code>, <code>false</code> otherwise. In the case of a
 * cluster, the default value will be determined on the vpc placement of the <code>DatabaseCluster</code> otherwise it will be determined
 * based on the vpc placement of standalone <code>DatabaseInstance</code>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * // Setting public accessibility for DB instance
 * // Setting public accessibility for DB instance
 * DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder()
 *                 .version(MysqlEngineVersion.VER_8_0_19)
 *                 .build()))
 *         .vpc(vpc)
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PRIVATE_WITH_EGRESS)
 *                 .build())
 *         .publiclyAccessible(true)
 *         .build();
 * 
 * // Setting public accessibility for DB cluster instance
 * // Setting public accessibility for DB cluster instance
 * DatabaseCluster.Builder.create(this, "DatabaseCluster")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder()
 *                 .version(AuroraMysqlEngineVersion.VER_3_03_0)
 *                 .build()))
 *         .writer(ClusterInstance.serverlessV2("Writer", ServerlessV2ClusterInstanceProps.builder()
 *                 .publiclyAccessible(true)
 *                 .build()))
 *         .vpc(vpc)
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PRIVATE_WITH_EGRESS)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Instance events</h2>
 * <p>
 * To define Amazon CloudWatch event rules for database instances, use the <code>onEvent</code>
 * method:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * Function fn;
 * 
 * Rule rule = instance.onEvent("InstanceEvent", OnEventOptions.builder().target(new LambdaFunction(fn)).build());
 * </pre></blockquote>
 * <p>
 * <h2>Login credentials</h2>
 * <p>
 * By default, database instances and clusters (with the exception of <code>DatabaseInstanceFromSnapshot</code> and <code>ServerlessClusterFromSnapshot</code>) will have <code>admin</code> user with an auto-generated password.
 * An alternative username (and password) may be specified for the admin user instead of the default.
 * <p>
 * The following examples use a <code>DatabaseInstance</code>, but the same usage is applicable to <code>DatabaseCluster</code>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * IInstanceEngine engine = DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_16_3).build());
 * DatabaseInstance.Builder.create(this, "InstanceWithUsername")
 *         .engine(engine)
 *         .vpc(vpc)
 *         .credentials(Credentials.fromGeneratedSecret("postgres"))
 *         .build();
 * 
 * DatabaseInstance.Builder.create(this, "InstanceWithUsernameAndPassword")
 *         .engine(engine)
 *         .vpc(vpc)
 *         .credentials(Credentials.fromPassword("postgres", SecretValue.ssmSecure("/dbPassword", "1")))
 *         .build();
 * 
 * ISecret mySecret = Secret.fromSecretName(this, "DBSecret", "myDBLoginInfo");
 * DatabaseInstance.Builder.create(this, "InstanceWithSecretLogin")
 *         .engine(engine)
 *         .vpc(vpc)
 *         .credentials(Credentials.fromSecret(mySecret))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Secrets generated by <code>fromGeneratedSecret()</code> can be customized:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * IInstanceEngine engine = DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_16_3).build());
 * Key myKey = new Key(this, "MyKey");
 * 
 * DatabaseInstance.Builder.create(this, "InstanceWithCustomizedSecret")
 *         .engine(engine)
 *         .vpc(vpc)
 *         .credentials(Credentials.fromGeneratedSecret("postgres", CredentialsBaseOptions.builder()
 *                 .secretName("my-cool-name")
 *                 .encryptionKey(myKey)
 *                 .excludeCharacters("!&amp;*^#&#64;()")
 *                 .replicaRegions(List.of(ReplicaRegion.builder().region("eu-west-1").build(), ReplicaRegion.builder().region("eu-west-2").build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Snapshot credentials</h3>
 * <p>
 * As noted above, Databases created with <code>DatabaseInstanceFromSnapshot</code> or <code>ServerlessClusterFromSnapshot</code> will not create user and auto-generated password by default because it's not possible to change the master username for a snapshot. Instead, they will use the existing username and password from the snapshot. You can still generate a new password - to generate a secret similarly to the other constructs, pass in credentials with <code>fromGeneratedSecret()</code> or <code>fromGeneratedPassword()</code>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * IInstanceEngine engine = DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_16_3).build());
 * Key myKey = new Key(this, "MyKey");
 * 
 * DatabaseInstanceFromSnapshot.Builder.create(this, "InstanceFromSnapshotWithCustomizedSecret")
 *         .engine(engine)
 *         .vpc(vpc)
 *         .snapshotIdentifier("mySnapshot")
 *         .credentials(SnapshotCredentials.fromGeneratedSecret("username", SnapshotCredentialsFromGeneratedPasswordOptions.builder()
 *                 .encryptionKey(myKey)
 *                 .excludeCharacters("!&amp;*^#&#64;()")
 *                 .replicaRegions(List.of(ReplicaRegion.builder().region("eu-west-1").build(), ReplicaRegion.builder().region("eu-west-2").build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Connecting</h2>
 * <p>
 * To control who can access the cluster or instance, use the <code>.connections</code> attribute. RDS databases have
 * a default port, so you don't need to specify the port:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * cluster.connections.allowFromAnyIpv4(Port.allTraffic(), "Open to the world");
 * </pre></blockquote>
 * <p>
 * The endpoints to access your database cluster will be available as the <code>.clusterEndpoint</code> and <code>.readerEndpoint</code>
 * attributes:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * String writeAddress = cluster.getClusterEndpoint().getSocketAddress();
 * </pre></blockquote>
 * <p>
 * For an instance database:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * 
 * String address = instance.getInstanceEndpoint().getSocketAddress();
 * </pre></blockquote>
 * <p>
 * <h2>Rotating credentials</h2>
 * <p>
 * When the master password is generated and stored in AWS Secrets Manager, it can be rotated automatically:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * SecurityGroup mySecurityGroup;
 * 
 * 
 * instance.addRotationSingleUser(RotationSingleUserOptions.builder()
 *         .automaticallyAfter(Duration.days(7)) // defaults to 30 days
 *         .excludeCharacters("!&#64;#$%^&amp;*") // defaults to the set " %+~`#/// here*()|[]{}:;&lt;&gt;?!'/&#64;\"\\"
 *         .securityGroup(mySecurityGroup)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(stack, "Database")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .instanceProps(InstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.SMALL))
 *                 .vpc(vpc)
 *                 .build())
 *         .build();
 * 
 * cluster.addRotationSingleUser();
 * 
 * DatabaseCluster clusterWithCustomRotationOptions = DatabaseCluster.Builder.create(stack, "CustomRotationOptions")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .instanceProps(InstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.SMALL))
 *                 .vpc(vpc)
 *                 .build())
 *         .build();
 * clusterWithCustomRotationOptions.addRotationSingleUser(RotationSingleUserOptions.builder()
 *         .automaticallyAfter(Duration.days(7))
 *         .excludeCharacters("!&#64;#$%^&amp;*")
 *         .securityGroup(securityGroup)
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PRIVATE_WITH_EGRESS).build())
 *         .endpoint(endpoint)
 *         .build());
 * </pre></blockquote>
 * <p>
 * The multi user rotation scheme is also available:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * DatabaseSecret myImportedSecret;
 * 
 * instance.addRotationMultiUser("MyUser", RotationMultiUserOptions.builder()
 *         .secret(myImportedSecret)
 *         .build());
 * </pre></blockquote>
 * <p>
 * It's also possible to create user credentials together with the instance/cluster and add rotation:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * 
 * DatabaseSecret myUserSecret = DatabaseSecret.Builder.create(this, "MyUserSecret")
 *         .username("myuser")
 *         .secretName("my-user-secret") // optional, defaults to a CloudFormation-generated name
 *         .dbname("mydb") //optional, defaults to the main database of the RDS cluster this secret gets attached to
 *         .masterSecret(instance.getSecret())
 *         .excludeCharacters("{}[]()'\"/\\")
 *         .build();
 * ISecret myUserSecretAttached = myUserSecret.attach(instance); // Adds DB connections information in the secret
 * 
 * instance.addRotationMultiUser("MyUser", RotationMultiUserOptions.builder() // Add rotation using the multi user scheme
 *         .secret(myUserSecretAttached).build());
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: This user must be created manually in the database using the master credentials.
 * The rotation will start as soon as this user exists.
 * <p>
 * Access to the Secrets Manager API is required for the secret rotation. This can be achieved either with
 * internet connectivity (through NAT) or with a VPC interface endpoint. By default, the rotation Lambda function
 * is deployed in the same subnets as the instance/cluster. If access to the Secrets Manager API is not possible from
 * those subnets or using the default API endpoint, use the <code>vpcSubnets</code> and/or <code>endpoint</code> options:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * InterfaceVpcEndpoint myEndpoint;
 * 
 * 
 * instance.addRotationSingleUser(RotationSingleUserOptions.builder()
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PRIVATE_WITH_EGRESS).build()) // Place rotation Lambda in private subnets
 *         .endpoint(myEndpoint)
 *         .build());
 * </pre></blockquote>
 * <p>
 * See also <a href="https://github.com/aws/aws-cdk/blob/main/packages/aws-cdk-lib/aws-secretsmanager/README.md">aws-cdk-lib/aws-secretsmanager</a> for credentials rotation of existing clusters/instances.
 * <p>
 * By default, any stack updates will cause AWS Secrets Manager to rotate a secret immediately. To prevent this behavior and wait until the next scheduled rotation window specified via the <code>automaticallyAfter</code> property, set the <code>rotateImmediatelyOnUpdate</code> property to false:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * SecurityGroup mySecurityGroup;
 * 
 * 
 * instance.addRotationSingleUser(RotationSingleUserOptions.builder()
 *         .automaticallyAfter(Duration.days(7)) // defaults to 30 days
 *         .excludeCharacters("!&#64;#$%^&amp;*") // defaults to the set " %+~`#/// here*()|[]{}:;&lt;&gt;?!'/&#64;\"\\"
 *         .securityGroup(mySecurityGroup) // defaults to an auto-created security group
 *         .rotateImmediatelyOnUpdate(false)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>IAM Authentication</h2>
 * <p>
 * You can also authenticate to a database instance using AWS Identity and Access Management (IAM) database authentication;
 * See <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html">https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html</a> for more information
 * and a list of supported versions and limitations.
 * <p>
 * The following example shows enabling IAM authentication for a database instance and granting connection access to an IAM role.
 * <p>
 * <h3>Instance</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_19).build()))
 *         .vpc(vpc)
 *         .iamAuthentication(true)
 *         .build();
 * Role role = Role.Builder.create(this, "DBRole").assumedBy(new AccountPrincipal(this.account)).build();
 * instance.grantConnect(role);
 * </pre></blockquote>
 * <p>
 * <h3>Proxy</h3>
 * <p>
 * The following example shows granting connection access for RDS Proxy to an IAM role.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder()
 *                 .version(AuroraMysqlEngineVersion.VER_3_03_0)
 *                 .build()))
 *         .writer(ClusterInstance.provisioned("writer"))
 *         .vpc(vpc)
 *         .build();
 * 
 * DatabaseProxy proxy = DatabaseProxy.Builder.create(this, "Proxy")
 *         .proxyTarget(ProxyTarget.fromCluster(cluster))
 *         .secrets(List.of(cluster.getSecret()))
 *         .vpc(vpc)
 *         .build();
 * 
 * Role role = Role.Builder.create(this, "DBProxyRole").assumedBy(new AccountPrincipal(this.account)).build();
 * proxy.grantConnect(role, "admin");
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: In addition to the setup above, a database user will need to be created to support IAM auth.
 * See <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html">https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html</a> for setup instructions.
 * <p>
 * To specify the details of authentication used by a proxy to log in as a specific database
 * user use the <code>clientPasswordAuthType</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder()
 *                 .version(AuroraMysqlEngineVersion.VER_3_03_0)
 *                 .build()))
 *         .writer(ClusterInstance.provisioned("writer"))
 *         .vpc(vpc)
 *         .build();
 * 
 * DatabaseProxy proxy = DatabaseProxy.Builder.create(this, "Proxy")
 *         .proxyTarget(ProxyTarget.fromCluster(cluster))
 *         .secrets(List.of(cluster.getSecret()))
 *         .vpc(vpc)
 *         .clientPasswordAuthType(ClientPasswordAuthType.MYSQL_NATIVE_PASSWORD)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cluster</h3>
 * <p>
 * The following example shows granting connection access for an IAM role to an Aurora Cluster.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder()
 *                 .version(AuroraMysqlEngineVersion.VER_3_03_0)
 *                 .build()))
 *         .writer(ClusterInstance.provisioned("writer"))
 *         .vpc(vpc)
 *         .build();
 * Role role = Role.Builder.create(this, "AppRole").assumedBy(new ServicePrincipal("someservice.amazonaws.com")).build();
 * cluster.grantConnect(role, "somedbuser");
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: In addition to the setup above, a database user will need to be created to support IAM auth.
 * See <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html">https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html</a> for setup instructions.
 * <p>
 * <h2>Kerberos Authentication</h2>
 * <p>
 * You can also authenticate using Kerberos to a database instance using AWS Managed Microsoft AD for authentication;
 * See <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html">https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html</a> for more information
 * and a list of supported versions and limitations.
 * <p>
 * The following example shows enabling domain support for a database instance and creating an IAM role to access
 * Directory Services.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * Role role = Role.Builder.create(this, "RDSDirectoryServicesRole")
 *         .assumedBy(new CompositePrincipal(
 *         new ServicePrincipal("rds.amazonaws.com"),
 *         new ServicePrincipal("directoryservice.rds.amazonaws.com")))
 *         .managedPolicies(List.of(ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonRDSDirectoryServiceAccess")))
 *         .build();
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_19).build()))
 *         .vpc(vpc)
 *         .domain("d-????????") // The ID of the domain for the instance to join.
 *         .domainRole(role)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also use the Kerberos authentication for an Aurora database cluster.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * Role iamRole = Role.Builder.create(this, "Role")
 *         .assumedBy(new CompositePrincipal(
 *         new ServicePrincipal("rds.amazonaws.com"),
 *         new ServicePrincipal("directoryservice.rds.amazonaws.com")))
 *         .managedPolicies(List.of(ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonRDSDirectoryServiceAccess")))
 *         .build();
 * 
 * DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_05_1).build()))
 *         .writer(ClusterInstance.provisioned("Instance", ProvisionedClusterInstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.MEDIUM))
 *                 .build()))
 *         .vpc(vpc)
 *         .domain("d-????????") // The ID of the domain for the cluster to join.
 *         .domainRole(iamRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: In addition to the setup above, you need to make sure that the database instance or cluster has network connectivity
 * to the domain controllers. This includes enabling cross-VPC traffic if in a different VPC and setting up the
 * appropriate security groups/network ACL to allow traffic between the database instance and domain controllers.
 * Once configured, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html">https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html</a> for details
 * on configuring users for each available database engine.
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * Database instances and clusters both expose metrics (<code>cloudwatch.Metric</code>):
 * <p>
 * <blockquote><pre>
 * // The number of database connections in use (average over 5 minutes)
 * DatabaseInstance instance;
 * 
 * // Average CPU utilization over 5 minutes
 * DatabaseCluster cluster;
 * 
 * Metric dbConnections = instance.metricDatabaseConnections();
 * Metric cpuUtilization = cluster.metricCPUUtilization();
 * 
 * // The average amount of time taken per disk I/O operation (average over 1 minute)
 * Metric readLatency = instance.metric("ReadLatency", MetricOptions.builder().statistic("Average").period(Duration.seconds(60)).build());
 * </pre></blockquote>
 * <p>
 * <h2>Enabling S3 integration</h2>
 * <p>
 * Data in S3 buckets can be imported to and exported from certain database engines using SQL queries. To enable this
 * functionality, set the <code>s3ImportBuckets</code> and <code>s3ExportBuckets</code> properties for import and export respectively. When
 * configured, the CDK automatically creates and configures IAM roles as required.
 * Additionally, the <code>s3ImportRole</code> and <code>s3ExportRole</code> properties can be used to set this role directly.
 * Note: To use <code>s3ImportRole</code> and <code>s3ExportRole</code> with Aurora PostgreSQL, you must also enable the S3 import and export features when you create the DatabaseClusterEngine.
 * <p>
 * You can read more about loading data to (or from) S3 here:
 * <p>
 * <ul>
 * <li>Aurora MySQL - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.LoadFromS3.html">import</a>
 * and <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.SaveIntoS3.html">export</a>.</li>
 * <li>Aurora PostgreSQL - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html#USER_PostgreSQL.S3Import">import</a>
 * and <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html">export</a>.</li>
 * <li>Microsoft SQL Server - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/SQLServer.Procedural.Importing.html">import and export</a></li>
 * <li>PostgreSQL - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html">import</a>
 * and <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/postgresql-s3-export.html">export</a></li>
 * <li>Oracle - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html">import and export</a></li>
 * </ul>
 * <p>
 * The following snippet sets up a database cluster with different S3 buckets where the data is imported and exported -
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * 
 * Vpc vpc;
 * 
 * Bucket importBucket = new Bucket(this, "importbucket");
 * Bucket exportBucket = new Bucket(this, "exportbucket");
 * DatabaseCluster.Builder.create(this, "dbcluster")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder()
 *                 .version(AuroraMysqlEngineVersion.VER_3_03_0)
 *                 .build()))
 *         .writer(ClusterInstance.provisioned("writer"))
 *         .vpc(vpc)
 *         .s3ImportBuckets(List.of(importBucket))
 *         .s3ExportBuckets(List.of(exportBucket))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Creating a Database Proxy</h2>
 * <p>
 * Amazon RDS Proxy sits between your application and your relational database to efficiently manage
 * connections to the database and improve scalability of the application. Learn more about at <a href="https://aws.amazon.com/rds/proxy/">Amazon RDS Proxy</a>.
 * <p>
 * RDS Proxy is supported for MySQL, MariaDB, Postgres, and SQL Server.
 * <p>
 * The following code configures an RDS Proxy for a <code>DatabaseInstance</code>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup securityGroup;
 * Secret[] secrets;
 * DatabaseInstance dbInstance;
 * 
 * 
 * DatabaseProxy proxy = dbInstance.addProxy("proxy", DatabaseProxyOptions.builder()
 *         .borrowTimeout(Duration.seconds(30))
 *         .maxConnectionsPercent(50)
 *         .secrets(secrets)
 *         .vpc(vpc)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Exporting Logs</h2>
 * <p>
 * You can publish database logs to Amazon CloudWatch Logs. With CloudWatch Logs, you can perform real-time analysis of the log data,
 * store the data in highly durable storage, and manage the data with the CloudWatch Logs Agent. This is available for both database
 * instances and clusters; the types of logs available depend on the database type and engine being used.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * Role myLogsPublishingRole;
 * Vpc vpc;
 * 
 * 
 * // Exporting logs from a cluster
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.aurora(AuroraClusterEngineProps.builder()
 *                 .version(AuroraEngineVersion.VER_1_17_9)
 *                 .build()))
 *         .writer(ClusterInstance.provisioned("writer"))
 *         .vpc(vpc)
 *         .cloudwatchLogsExports(List.of("error", "general", "slowquery", "audit")) // Export all available MySQL-based logs
 *         .cloudwatchLogsRetention(RetentionDays.THREE_MONTHS) // Optional - default is to never expire logs
 *         .cloudwatchLogsRetentionRole(myLogsPublishingRole)
 *         .build();
 * 
 * // When 'cloudwatchLogsExports' is set, each export value creates its own log group in DB cluster.
 * // Specify an export value to access its log group.
 * ILogGroup errorLogGroup = cluster.getCloudwatchLogGroups()["error"];
 * ILogGroup auditLogGroup = cluster.getCloudwatchLogGroups().getAudit();
 * 
 * // Exporting logs from an instance
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder()
 *                 .version(PostgresEngineVersion.VER_16_3)
 *                 .build()))
 *         .vpc(vpc)
 *         .cloudwatchLogsExports(List.of("postgresql")) // Export the PostgreSQL logs
 *         .cloudwatchLogsRetention(RetentionDays.THREE_MONTHS)
 *         .build();
 * 
 * // When 'cloudwatchLogsExports' is set, each export value creates its own log group in DB instance.
 * // Specify an export value to access its log group.
 * ILogGroup postgresqlLogGroup = instance.getCloudwatchLogGroups()["postgresql"];
 * </pre></blockquote>
 * <p>
 * <h2>Option Groups</h2>
 * <p>
 * Some DB engines offer additional features that make it easier to manage data and databases, and to provide additional security for your database.
 * Amazon RDS uses option groups to enable and configure these features. An option group can specify features, called options,
 * that are available for a particular Amazon RDS DB instance.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup securityGroup;
 * 
 * 
 * OptionGroup.Builder.create(this, "Options")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder()
 *                 .version(OracleEngineVersion.VER_19)
 *                 .build()))
 *         .configurations(List.of(OptionConfiguration.builder()
 *                 .name("OEM")
 *                 .port(5500)
 *                 .vpc(vpc)
 *                 .securityGroups(List.of(securityGroup))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Parameter Groups</h2>
 * <p>
 * Database parameters specify how the database is configured.
 * For example, database parameters can specify the amount of resources, such as memory, to allocate to a database.
 * You manage your database configuration by associating your DB instances with parameter groups.
 * Amazon RDS defines parameter groups with default settings.
 * <p>
 * You can create your own parameter group for your cluster or instance and associate it with your database:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * ParameterGroup parameterGroup = ParameterGroup.Builder.create(this, "ParameterGroup")
 *         .engine(DatabaseInstanceEngine.sqlServerEe(SqlServerEeInstanceEngineProps.builder()
 *                 .version(SqlServerEngineVersion.VER_11)
 *                 .build()))
 *         .name("my-parameter-group")
 *         .parameters(Map.of(
 *                 "locks", "100"))
 *         .build();
 * 
 * DatabaseInstance.Builder.create(this, "Database")
 *         .engine(DatabaseInstanceEngine.SQL_SERVER_EE)
 *         .vpc(vpc)
 *         .parameterGroup(parameterGroup)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Another way to specify parameters is to use the inline field <code>parameters</code> that creates an RDS parameter group for you.
 * You can use this if you do not want to reuse the parameter group instance for different instances:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseInstance.Builder.create(this, "Database")
 *         .engine(DatabaseInstanceEngine.sqlServerEe(SqlServerEeInstanceEngineProps.builder().version(SqlServerEngineVersion.VER_11).build()))
 *         .vpc(vpc)
 *         .parameters(Map.of(
 *                 "locks", "100"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You cannot specify a parameter map and a parameter group at the same time.
 * <p>
 * <h2>Serverless v1</h2>
 * <p>
 * <a href="https://aws.amazon.com/rds/aurora/serverless/">Amazon Aurora Serverless v1</a> is an on-demand, auto-scaling configuration for Amazon
 * Aurora. The database will automatically start up, shut down, and scale capacity
 * up or down based on your application's needs. It enables you to run your database
 * in the cloud without managing any database instances.
 * <p>
 * The following example initializes an Aurora Serverless v1 PostgreSql cluster.
 * Aurora Serverless clusters can specify scaling properties which will be used to
 * automatically scale the database cluster seamlessly based on the workload.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * ServerlessCluster cluster = ServerlessCluster.Builder.create(this, "AnotherCluster")
 *         .engine(DatabaseClusterEngine.AURORA_POSTGRESQL)
 *         .copyTagsToSnapshot(true) // whether to save the cluster tags when creating the snapshot. Default is 'true'
 *         .parameterGroup(ParameterGroup.fromParameterGroupName(this, "ParameterGroup", "default.aurora-postgresql11"))
 *         .vpc(vpc)
 *         .scaling(ServerlessScalingOptions.builder()
 *                 .autoPause(Duration.minutes(10)) // default is to pause after 5 minutes of idle time
 *                 .minCapacity(AuroraCapacityUnit.ACU_8) // default is 2 Aurora capacity units (ACUs)
 *                 .maxCapacity(AuroraCapacityUnit.ACU_32) // default is 16 Aurora capacity units (ACUs)
 *                 .timeout(Duration.seconds(100)) // default is 5 minutes
 *                 .timeoutAction(TimeoutAction.FORCE_APPLY_CAPACITY_CHANGE)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: The <code>rds.ServerlessCluster</code> class is for Aurora Serverless v1. If you want to use Aurora Serverless v2, use the <code>rds.DatabaseCluster</code> class.
 * <p>
 * Aurora Serverless v1 Clusters do not support the following features:
 * <p>
 * <ul>
 * <li>Loading data from an Amazon S3 bucket</li>
 * <li>Saving data to an Amazon S3 bucket</li>
 * <li>Invoking an AWS Lambda function with an Aurora MySQL native function</li>
 * <li>Aurora replicas</li>
 * <li>Backtracking</li>
 * <li>Multi-master clusters</li>
 * <li>Database cloning</li>
 * <li>IAM database cloning</li>
 * <li>IAM database authentication</li>
 * <li>Restoring a snapshot from MySQL DB instance</li>
 * <li>Performance Insights</li>
 * <li>RDS Proxy</li>
 * </ul>
 * <p>
 * Read more about the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations">limitations of Aurora Serverless v1</a>
 * <p>
 * Learn more about using Amazon Aurora Serverless v1 by reading the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html">documentation</a>
 * <p>
 * Use <code>ServerlessClusterFromSnapshot</code> to create a serverless cluster from a snapshot:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * ServerlessClusterFromSnapshot.Builder.create(this, "Cluster")
 *         .engine(DatabaseClusterEngine.AURORA_MYSQL)
 *         .vpc(vpc)
 *         .snapshotIdentifier("mySnapshot")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Data API</h2>
 * <p>
 * You can access your Aurora DB cluster using the built-in Data API. The Data API doesn't require a persistent connection to the DB cluster. Instead, it provides a secure HTTP endpoint and integration with AWS SDKs.
 * <p>
 * The following example shows granting Data API access to a Lambda function.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Function fn;
 * Secret secret;
 * 
 * 
 * // Create a serverless V1 cluster
 * ServerlessCluster serverlessV1Cluster = ServerlessCluster.Builder.create(this, "AnotherCluster")
 *         .engine(DatabaseClusterEngine.AURORA_MYSQL)
 *         .vpc(vpc) // this parameter is optional for serverless Clusters
 *         .enableDataApi(true)
 *         .build();
 * serverlessV1Cluster.grantDataApiAccess(fn);
 * 
 * // Create an Aurora cluster
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Cluster")
 *         .engine(DatabaseClusterEngine.AURORA_MYSQL)
 *         .vpc(vpc)
 *         .enableDataApi(true)
 *         .build();
 * cluster.grantDataApiAccess(fn);
 * 
 * // Import an Aurora cluster
 * IDatabaseCluster importedCluster = DatabaseCluster.fromDatabaseClusterAttributes(this, "ImportedCluster", DatabaseClusterAttributes.builder()
 *         .clusterIdentifier("clusterIdentifier")
 *         .secret(secret)
 *         .dataApiEnabled(true)
 *         .build());
 * importedCluster.grantDataApiAccess(fn);
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: To invoke the Data API, the resource will need to read the secret associated with the cluster.
 * <p>
 * To learn more about using the Data API, see the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html">documentation</a>.
 * <p>
 * <h2>Default VPC</h2>
 * <p>
 * The <code>vpc</code> parameter is optional.
 * <p>
 * If not provided, the cluster will be created in the default VPC of the account and region.
 * As this VPC is not deployed with AWS CDK, you can't configure the <code>vpcSubnets</code>, <code>subnetGroup</code> or <code>securityGroups</code> of the Aurora Serverless Cluster.
 * If you want to provide one of <code>vpcSubnets</code>, <code>subnetGroup</code> or <code>securityGroups</code> parameter, please provide a <code>vpc</code>.
 * <p>
 * <h2>Preferred Maintenance Window</h2>
 * <p>
 * When creating an RDS cluster, it is possible to (optionally) specify a preferred maintenance window for the cluster as well as the instances under the cluster.
 * See <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance">AWS docs</a> for more information regarding maintenance windows.
 * <p>
 * The following code snippet shows an example of setting the cluster's maintenance window to 22:15-22:45 (UTC) on Saturdays, but setting the instances' maintenance window
 * to 23:15-23:45 on Sundays
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster.Builder.create(this, "DatabaseCluster")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .instanceProps(InstanceProps.builder()
 *                 .vpc(vpc)
 *                 .preferredMaintenanceWindow("Sun:23:15-Sun:23:45")
 *                 .build())
 *         .preferredMaintenanceWindow("Sat:22:15-Sat:22:45")
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also set the preferred maintenance window via reader and writer props:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster.Builder.create(this, "DatabaseCluster")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .vpc(vpc)
 *         .writer(ClusterInstance.provisioned("WriterInstance", ProvisionedClusterInstanceProps.builder()
 *                 .preferredMaintenanceWindow("Sat:22:15-Sat:22:45")
 *                 .build()))
 *         .preferredMaintenanceWindow("Sat:22:15-Sat:22:45")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Performance Insights</h2>
 * <p>
 * You can enable Performance Insights for a clustered database or an instance database.
 * <p>
 * <h3>Clustered Database</h3>
 * <p>
 * You can enable Performance Insights at cluster level or instance level.
 * <p>
 * To enable Performance Insights at the cluster level, set the <code>enablePerformanceInsights</code> property for the <code>DatabaseCluster</code> to <code>true</code>.
 * If you want to specify the detailed settings, you can use the <code>performanceInsightRetention</code> and <code>performanceInsightEncryptionKey</code> properties.
 * <p>
 * The settings are then applied to all instances in the cluster.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Key kmsKey;
 * 
 * DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .vpc(vpc)
 *         .enablePerformanceInsights(true)
 *         .performanceInsightRetention(PerformanceInsightRetention.LONG_TERM)
 *         .performanceInsightEncryptionKey(kmsKey)
 *         .writer(ClusterInstance.provisioned("Writer", ProvisionedClusterInstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.R7G, InstanceSize.LARGE))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To enable Performance Insights at the instance level, set the same properties for each instance of the <code>writer</code> and the <code>readers</code>.
 * <p>
 * In this way, different settings can be applied to different instances in a cluster.
 * <p>
 * <strong>Note:</strong> If Performance Insights is enabled at the cluster level, it is also automatically enabled for each instance. If specified, Performance Insights
 * for each instance require the same retention period and encryption key as the cluster level.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Key kmsKey;
 * 
 * DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .vpc(vpc)
 *         .writer(ClusterInstance.provisioned("Writer", ProvisionedClusterInstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.R7G, InstanceSize.LARGE))
 *                 .enablePerformanceInsights(true)
 *                 .performanceInsightRetention(PerformanceInsightRetention.LONG_TERM)
 *                 .performanceInsightEncryptionKey(kmsKey)
 *                 .build()))
 *         .readers(List.of(ClusterInstance.provisioned("Reader", ProvisionedClusterInstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.R7G, InstanceSize.LARGE))
 *                 .enablePerformanceInsights(false)
 *                 .build())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Instance Database</h3>
 * <p>
 * To enable Performance Insights for an instance database, set the <code>enablePerformanceInsights</code> property for the <code>DatabaseInstance</code> to <code>true</code>.
 * If you want to specify the detailed settings, you can use the <code>performanceInsightRetention</code> and <code>performanceInsightEncryptionKey</code> properties.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Key kmsKey;
 * 
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_39).build()))
 *         .instanceType(InstanceType.of(InstanceClass.R7G, InstanceSize.LARGE))
 *         .vpc(vpc)
 *         .enablePerformanceInsights(true)
 *         .performanceInsightRetention(PerformanceInsightRetention.LONG_TERM)
 *         .performanceInsightEncryptionKey(kmsKey)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Supported Engines</h3>
 * <p>
 * Performance Insights supports a limited number of engines.
 * <p>
 * To see Amazon RDS DB engines that support Performance Insights, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.Overview.Engines.html">Amazon RDS DB engine, Region, and instance class support for Performance Insights</a>.
 * <p>
 * To see Amazon Aurora DB engines that support Performance Insights, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_PerfInsights.Overview.Engines.html">Amazon Aurora DB engine, Region, and instance class support for Performance Insights</a>.
 * <p>
 * For more information about Performance Insights, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html">Monitoring DB load with Performance Insights on Amazon RDS</a>.
 * <p>
 * <h2>Database Insights</h2>
 * <p>
 * The standard mode of Database Insights is enabled by default for Aurora databases.
 * <p>
 * You can enhance the monitoring of your Aurora databases by enabling the advanced mode of Database Insights.
 * <p>
 * To control Database Insights mode, use the <code>databaseInsightsMode</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .vpc(vpc)
 *         // If you enable the advanced mode of Database Insights,
 *         // Performance Insights is enabled and you must set the `performanceInsightRetention` to 465(15 months).
 *         .databaseInsightsMode(DatabaseInsightsMode.ADVANCED)
 *         .performanceInsightRetention(PerformanceInsightRetention.MONTHS_15)
 *         .writer(ClusterInstance.provisioned("Writer", ProvisionedClusterInstanceProps.builder()
 *                 .instanceType(InstanceType.of(InstanceClass.R7G, InstanceSize.LARGE))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note: Database Insights are only supported for Amazon Aurora MySQL and Amazon Aurora PostgreSQL clusters.
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Database-Insights.html">CloudWatch Database Insights</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Enhanced Monitoring</h2>
 * <p>
 * With <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling">Enhanced Monitoring</a>, you can monitor the operating system of your DB instance in real time.
 * <p>
 * To enable Enhanced Monitoring for a clustered database, set the <code>monitoringInterval</code> property.
 * This value is applied at instance level to all instances in the cluster by default.
 * <p>
 * If you want to enable enhanced monitoring at the cluster level, you can set the <code>enableClusterLevelEnhancedMonitoring</code> property to <code>true</code>. Note that you must set <code>monitoringInterval</code> when using <code>enableClusterLevelEnhancedMonitoring</code>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * // Enable Enhanced Monitoring at instance level to all instances in the cluster
 * // Enable Enhanced Monitoring at instance level to all instances in the cluster
 * DatabaseCluster.Builder.create(this, "Cluster")
 *         .engine(DatabaseClusterEngine.auroraPostgres(AuroraPostgresClusterEngineProps.builder().version(AuroraPostgresEngineVersion.VER_16_1).build()))
 *         .writer(ClusterInstance.serverlessV2("writerInstance"))
 *         .vpc(vpc)
 *         .monitoringInterval(Duration.seconds(5))
 *         .build();
 * 
 * // Enable Enhanced Monitoring at the cluster level
 * // Enable Enhanced Monitoring at the cluster level
 * DatabaseCluster.Builder.create(this, "Cluster")
 *         .engine(DatabaseClusterEngine.auroraPostgres(AuroraPostgresClusterEngineProps.builder().version(AuroraPostgresEngineVersion.VER_16_1).build()))
 *         .writer(ClusterInstance.serverlessV2("writerInstance"))
 *         .vpc(vpc)
 *         .monitoringInterval(Duration.seconds(5))
 *         .enableClusterLevelEnhancedMonitoring(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * AWS CDK automatically generate the IAM role for Enhanced Monitoring.
 * If you want to create the IAM role manually, you can use the <code>monitoringRole</code> property.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Role monitoringRole;
 * 
 * 
 * DatabaseCluster.Builder.create(this, "Cluster")
 *         .engine(DatabaseClusterEngine.auroraPostgres(AuroraPostgresClusterEngineProps.builder().version(AuroraPostgresEngineVersion.VER_16_1).build()))
 *         .writer(ClusterInstance.serverlessV2("writerInstance"))
 *         .vpc(vpc)
 *         .monitoringInterval(Duration.seconds(5))
 *         .monitoringRole(monitoringRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Extended Support</h2>
 * <p>
 * With Amazon RDS Extended Support, you can continue running your database on a major engine version past the RDS end of
 * standard support date for an additional cost. To configure the life cycle type, use the <code>engineLifecycleSupport</code> property:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * DatabaseCluster.Builder.create(this, "DatabaseCluster")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_3_07_0).build()))
 *         .writer(ClusterInstance.serverlessV2("writerInstance"))
 *         .vpc(vpc)
 *         .engineLifecycleSupport(EngineLifecycleSupport.OPEN_SOURCE_RDS_EXTENDED_SUPPORT)
 *         .build();
 * 
 * DatabaseInstance.Builder.create(this, "DatabaseInstance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_39).build()))
 *         .instanceType(InstanceType.of(InstanceClass.R7G, InstanceSize.LARGE))
 *         .vpc(vpc)
 *         .engineLifecycleSupport(EngineLifecycleSupport.OPEN_SOURCE_RDS_EXTENDED_SUPPORT_DISABLED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Importing existing DatabaseInstance</h2>
 * <p>
 * <h3>Lookup DatabaseInstance by instanceIdentifier</h3>
 * <p>
 * You can lookup an existing DatabaseInstance by its instanceIdentifier using <code>DatabaseInstance.fromLookup()</code>.  This method returns an <code>IDatabaseInstance</code>.
 * <p>
 * Here's how <code>DatabaseInstance.fromLookup()</code> can be used:
 * <p>
 * <blockquote><pre>
 * Role myUserRole;
 * 
 * 
 * IDatabaseInstance dbFromLookup = DatabaseInstance.fromLookup(this, "dbFromLookup", DatabaseInstanceLookupOptions.builder()
 *         .instanceIdentifier("instanceId")
 *         .build());
 * 
 * // Grant a connection
 * dbFromLookup.grantConnect(myUserRole, "my-user-id");
 * </pre></blockquote>
 * <p>
 * <h2>Importing existing DatabaseCluster</h2>
 * <p>
 * <h3>Lookup DatabaseCluster by clusterIdentifier</h3>
 * <p>
 * You can lookup an existing DatabaseCluster by its clusterIdentifier using <code>DatabaseCluster.fromLookup()</code>. This method returns an <code>IDatabaseCluster</code>.
 * <p>
 * Here's how <code>DatabaseCluster.fromLookup()</code> can be used:
 * <p>
 * <blockquote><pre>
 * Role myUserRole;
 * 
 * 
 * IDatabaseCluster clusterFromLookup = DatabaseCluster.fromLookup(this, "ClusterFromLookup", DatabaseClusterLookupOptions.builder()
 *         .clusterIdentifier("my-cluster-id")
 *         .build());
 * 
 * // Grant a connection
 * clusterFromLookup.grantConnect(myUserRole, "my-user-id");
 * </pre></blockquote>
 * <p>
 * <h2>Limitless Database Cluster</h2>
 * <p>
 * Amazon Aurora <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/limitless.html">PostgreSQL Limitless Database</a> provides automated horizontal scaling to process millions of write transactions per second and manages petabytes of data while maintaining the simplicity of operating inside a single database.
 * <p>
 * The following example shows creating an Aurora PostgreSQL Limitless Database cluster:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * DatabaseCluster.Builder.create(this, "LimitlessDatabaseCluster")
 *         .engine(DatabaseClusterEngine.auroraPostgres(AuroraPostgresClusterEngineProps.builder()
 *                 .version(AuroraPostgresEngineVersion.VER_16_4_LIMITLESS)
 *                 .build()))
 *         .vpc(vpc)
 *         .clusterScalabilityType(ClusterScalabilityType.LIMITLESS)
 *         // Requires enabling Performance Insights
 *         .enablePerformanceInsights(true)
 *         .performanceInsightRetention(PerformanceInsightRetention.MONTHS_1)
 *         // Requires enabling Enhanced Monitoring at the cluster level
 *         .monitoringInterval(Duration.minutes(1))
 *         .enableClusterLevelEnhancedMonitoring(true)
 *         // Requires I/O optimized storage type
 *         .storageType(DBClusterStorageType.AURORA_IOPT1)
 *         // Requires exporting the PostgreSQL log to Amazon CloudWatch Logs.
 *         .cloudwatchLogsExports(List.of("postgresql"))
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.rds;
