/**
 * <h1>Tasks for AWS Step Functions</h1>
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/welcome.html">AWS Step Functions</a> is a web service that enables you to coordinate the
 * components of distributed applications and microservices using visual workflows.
 * You build applications from individual components that each perform a discrete
 * function, or task, allowing you to scale and change applications quickly.
 * <p>
 * A <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-task-state.html">Task</a> state represents a single unit of work performed by a state machine.
 * All work in your state machine is performed by tasks.  This module contains a collection of classes that allow you to call various AWS services
 * from your Step Functions state machine.
 * <p>
 * Be sure to familiarize yourself with the <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions-readme.html"><code>aws-stepfunctions</code> module documentation</a> first.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Table Of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#tasks-for-aws-step-functions">Tasks for AWS Step Functions</a>
 * <p>
 * <ul>
 * <li><a href="#table-of-contents">Table Of Contents</a></li>
 * <li><a href="#paths">Paths</a></li>
 * <li><a href="#evaluate-expression">Evaluate Expression</a></li>
 * <li><a href="#api-gateway">API Gateway</a>
 * <p>
 * <ul>
 * <li><a href="#call-rest-api-endpoint">Call REST API Endpoint</a></li>
 * <li><a href="#call-http-api-endpoint">Call HTTP API Endpoint</a></li>
 * </ul></li>
 * <li><a href="#aws-sdk">AWS SDK</a>
 * <p>
 * <ul>
 * <li><a href="#cross-region-aws-api-call">Cross-region AWS API call</a></li>
 * </ul></li>
 * <li><a href="#athena">Athena</a>
 * <p>
 * <ul>
 * <li><a href="#startqueryexecution">StartQueryExecution</a></li>
 * <li><a href="#getqueryexecution">GetQueryExecution</a></li>
 * <li><a href="#getqueryresults">GetQueryResults</a></li>
 * <li><a href="#stopqueryexecution">StopQueryExecution</a></li>
 * </ul></li>
 * <li><a href="#batch">Batch</a>
 * <p>
 * <ul>
 * <li><a href="#submitjob">SubmitJob</a></li>
 * </ul></li>
 * <li><a href="#bedrock">Bedrock</a>
 * <p>
 * <ul>
 * <li><a href="#invokemodel">InvokeModel</a></li>
 * <li><a href="#createmodelcustomizationjob">createModelCustomizationJob</a></li>
 * </ul></li>
 * <li><a href="#codebuild">CodeBuild</a>
 * <p>
 * <ul>
 * <li><a href="#startbuild">StartBuild</a></li>
 * <li><a href="#startbuildbatch">StartBuildBatch</a></li>
 * </ul></li>
 * <li><a href="#dynamodb">DynamoDB</a>
 * <p>
 * <ul>
 * <li><a href="#getitem">GetItem</a></li>
 * <li><a href="#putitem">PutItem</a></li>
 * <li><a href="#deleteitem">DeleteItem</a></li>
 * <li><a href="#updateitem">UpdateItem</a></li>
 * </ul></li>
 * <li><a href="#ecs">ECS</a>
 * <p>
 * <ul>
 * <li><a href="#runtask">RunTask</a>
 * <p>
 * <ul>
 * <li><a href="#ec2">EC2</a></li>
 * <li><a href="#fargate">Fargate</a></li>
 * <li><a href="#ecs-enable-exec">ECS enable Exec</a></li>
 * </ul></li>
 * </ul></li>
 * <li><a href="#emr">EMR</a>
 * <p>
 * <ul>
 * <li><a href="#create-cluster">Create Cluster</a></li>
 * <li><a href="#termination-protection">Termination Protection</a></li>
 * <li><a href="#terminate-cluster">Terminate Cluster</a></li>
 * <li><a href="#add-step">Add Step</a></li>
 * <li><a href="#cancel-step">Cancel Step</a></li>
 * <li><a href="#modify-instance-fleet">Modify Instance Fleet</a></li>
 * <li><a href="#modify-instance-group">Modify Instance Group</a></li>
 * </ul></li>
 * <li><a href="#emr-on-eks">EMR on EKS</a>
 * <p>
 * <ul>
 * <li><a href="#create-virtual-cluster">Create Virtual Cluster</a></li>
 * <li><a href="#delete-virtual-cluster">Delete Virtual Cluster</a></li>
 * <li><a href="#start-job-run">Start Job Run</a></li>
 * </ul></li>
 * <li><a href="#eks">EKS</a>
 * <p>
 * <ul>
 * <li><a href="#call">Call</a></li>
 * </ul></li>
 * <li><a href="#eventbridge">EventBridge</a>
 * <p>
 * <ul>
 * <li><a href="#put-events">Put Events</a></li>
 * </ul></li>
 * <li><a href="#glue">Glue</a>
 * <p>
 * <ul>
 * <li><a href="#startjobrun">StartJobRun</a></li>
 * <li><a href="#startcrawlerrun">StartCrawlerRun</a></li>
 * </ul></li>
 * <li><a href="#glue-databrew">Glue DataBrew</a>
 * <p>
 * <ul>
 * <li><a href="#start-job-run-1">Start Job Run</a></li>
 * </ul></li>
 * <li><a href="#lambda">Lambda</a>
 * <p>
 * <ul>
 * <li><a href="#invoke">Invoke</a></li>
 * </ul></li>
 * <li><a href="#mediaconvert">MediaConvert</a>
 * <p>
 * <ul>
 * <li><a href="#create-job">Create Job</a></li>
 * </ul></li>
 * <li><a href="#sagemaker">SageMaker</a>
 * <p>
 * <ul>
 * <li><a href="#create-training-job">Create Training Job</a></li>
 * <li><a href="#create-transform-job">Create Transform Job</a></li>
 * <li><a href="#create-endpoint">Create Endpoint</a></li>
 * <li><a href="#create-endpoint-config">Create Endpoint Config</a></li>
 * <li><a href="#create-model">Create Model</a></li>
 * <li><a href="#update-endpoint">Update Endpoint</a></li>
 * </ul></li>
 * <li><a href="#sns">SNS</a>
 * <p>
 * <ul>
 * <li><a href="#publish">Publish</a></li>
 * </ul></li>
 * <li><a href="#step-functions">Step Functions</a>
 * <p>
 * <ul>
 * <li><a href="#start-execution">Start Execution</a></li>
 * <li><a href="#invoke-activity">Invoke Activity</a></li>
 * </ul></li>
 * <li><a href="#sqs">SQS</a>
 * <p>
 * <ul>
 * <li><a href="#send-message">Send Message</a></li>
 * </ul></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>Paths</h2>
 * <p>
 * Learn more about input and output processing in Step Functions <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html">here</a>
 * <p>
 * <h2>Evaluate Expression</h2>
 * <p>
 * Use the <code>EvaluateExpression</code> to perform simple operations referencing state paths. The
 * <code>expression</code> referenced in the task will be evaluated in a Lambda function
 * (<code>eval()</code>). This allows you to not have to write Lambda code for simple operations.
 * <p>
 * Example: convert a wait time from milliseconds to seconds, concat this in a message and wait:
 * <p>
 * <blockquote><pre>
 * EvaluateExpression convertToSeconds = EvaluateExpression.Builder.create(this, "Convert to seconds")
 *         .expression("$.waitMilliseconds / 1000")
 *         .resultPath("$.waitSeconds")
 *         .build();
 * 
 * EvaluateExpression createMessage = EvaluateExpression.Builder.create(this, "Create message")
 *         // Note: this is a string inside a string.
 *         .expression("`Now waiting ${$.waitSeconds} seconds...`")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .resultPath("$.message")
 *         .build();
 * 
 * SnsPublish publishMessage = SnsPublish.Builder.create(this, "Publish message")
 *         .topic(new Topic(this, "cool-topic"))
 *         .message(TaskInput.fromJsonPathAt("$.message"))
 *         .resultPath("$.sns")
 *         .build();
 * 
 * Wait wait = Wait.Builder.create(this, "Wait")
 *         .time(WaitTime.secondsPath("$.waitSeconds"))
 *         .build();
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definition(convertToSeconds.next(createMessage).next(publishMessage).next(wait))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>EvaluateExpression</code> supports a <code>runtime</code> prop to specify the Lambda
 * runtime to use to evaluate the expression. Currently, only runtimes
 * of the Node.js family are supported.
 * <p>
 * <h2>API Gateway</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html">API Gateway</a> through the service integration pattern.
 * <p>
 * HTTP APIs are designed for low-latency, cost-effective integrations with AWS services, including AWS Lambda, and HTTP endpoints.
 * HTTP APIs support OIDC and OAuth 2.0 authorization, and come with built-in support for CORS and automatic deployments.
 * Previous-generation REST APIs currently offer more features. More details can be found <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-vs-rest.html">here</a>.
 * <p>
 * <h3>Call REST API Endpoint</h3>
 * <p>
 * The <code>CallApiGatewayRestApiEndpoint</code> calls the REST API endpoint.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigateway.*;
 * 
 * RestApi restApi = new RestApi(this, "MyRestApi");
 * 
 * CallApiGatewayRestApiEndpoint invokeTask = CallApiGatewayRestApiEndpoint.Builder.create(this, "Call REST API")
 *         .api(restApi)
 *         .stageName("prod")
 *         .method(HttpMethod.GET)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, the API endpoint URI will be constructed using the AWS region of
 * the stack in which the provided <code>api</code> is created.
 * <p>
 * To construct the endpoint with a different region, use the <code>region</code> parameter:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigateway.*;
 * 
 * RestApi restApi = new RestApi(this, "MyRestApi");
 * CallApiGatewayRestApiEndpoint invokeTask = CallApiGatewayRestApiEndpoint.Builder.create(this, "Call REST API")
 *         .api(restApi)
 *         .stageName("prod")
 *         .method(HttpMethod.GET)
 *         .region("us-west-2")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Be aware that the header values must be arrays. When passing the Task Token
 * in the headers field <code>WAIT_FOR_TASK_TOKEN</code> integration, use
 * <code>JsonPath.array()</code> to wrap the token in an array:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigateway.*;
 * RestApi api;
 * 
 * 
 * CallApiGatewayRestApiEndpoint.Builder.create(this, "Endpoint")
 *         .api(api)
 *         .stageName("Stage")
 *         .method(HttpMethod.PUT)
 *         .integrationPattern(IntegrationPattern.WAIT_FOR_TASK_TOKEN)
 *         .headers(TaskInput.fromObject(Map.of(
 *                 "TaskToken", JsonPath.array(JsonPath.getTaskToken()))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Call HTTP API Endpoint</h3>
 * <p>
 * The <code>CallApiGatewayHttpApiEndpoint</code> calls the HTTP API endpoint.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.*;
 * 
 * HttpApi httpApi = new HttpApi(this, "MyHttpApi");
 * 
 * CallApiGatewayHttpApiEndpoint invokeTask = CallApiGatewayHttpApiEndpoint.Builder.create(this, "Call HTTP API")
 *         .apiId(httpApi.getApiId())
 *         .apiStack(Stack.of(httpApi))
 *         .method(HttpMethod.GET)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>AWS SDK</h2>
 * <p>
 * Step Functions supports calling <a href="https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html">AWS service's API actions</a>
 * through the service integration pattern.
 * <p>
 * You can use Step Functions' AWS SDK integrations to call any of the over two hundred AWS services
 * directly from your state machine, giving you access to over nine thousand API actions.
 * <p>
 * <blockquote><pre>
 * Bucket myBucket;
 * 
 * CallAwsService getObject = CallAwsService.Builder.create(this, "GetObject")
 *         .service("s3")
 *         .action("getObject")
 *         .parameters(Map.of(
 *                 "Bucket", myBucket.getBucketName(),
 *                 "Key", JsonPath.stringAt("$.key")))
 *         .iamResources(List.of(myBucket.arnForObjects("*")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use camelCase for actions and PascalCase for parameter names.
 * <p>
 * The task automatically adds an IAM statement to the state machine role's policy based on the
 * service and action called. The resources for this statement must be specified in <code>iamResources</code>.
 * <p>
 * Use the <code>iamAction</code> prop to manually specify the IAM action name in the case where the IAM
 * action name does not match with the API service/action name:
 * <p>
 * <blockquote><pre>
 * CallAwsService listBuckets = CallAwsService.Builder.create(this, "ListBuckets")
 *         .service("s3")
 *         .action("listBuckets")
 *         .iamResources(List.of("*"))
 *         .iamAction("s3:ListAllMyBuckets")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use the <code>additionalIamStatements</code> prop to pass additional IAM statements that will be added to the
 * state machine role's policy. Use it in the case where the call requires more than a single statement
 * to be executed:
 * <p>
 * <blockquote><pre>
 * CallAwsService detectLabels = CallAwsService.Builder.create(this, "DetectLabels")
 *         .service("rekognition")
 *         .action("detectLabels")
 *         .iamResources(List.of("*"))
 *         .additionalIamStatements(List.of(
 *             PolicyStatement.Builder.create()
 *                     .actions(List.of("s3:getObject"))
 *                     .resources(List.of("arn:aws:s3:::amzn-s3-demo-bucket/*"))
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cross-region AWS API call</h3>
 * <p>
 * You can call AWS API in a different region from your state machine by using the <code>CallAwsServiceCrossRegion</code> construct. In addition to the properties for <code>CallAwsService</code> construct, you have to set <code>region</code> property to call the API.
 * <p>
 * <blockquote><pre>
 * Bucket myBucket;
 * 
 * CallAwsServiceCrossRegion getObject = CallAwsServiceCrossRegion.Builder.create(this, "GetObject")
 *         .region("ap-northeast-1")
 *         .service("s3")
 *         .action("getObject")
 *         .parameters(Map.of(
 *                 "Bucket", myBucket.getBucketName(),
 *                 "Key", JsonPath.stringAt("$.key")))
 *         .iamResources(List.of(myBucket.arnForObjects("*")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Other properties such as <code>additionalIamStatements</code> can be used in the same way as the <code>CallAwsService</code> task.
 * <p>
 * Note that when you use <code>integrationPattern.WAIT_FOR_TASK_TOKEN</code>, the output path changes under <code>Payload</code> property.
 * <p>
 * <h2>Athena</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html">Athena</a> through the service integration pattern.
 * <p>
 * <h3>StartQueryExecution</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html">StartQueryExecution</a> API runs the SQL query statement.
 * <p>
 * <blockquote><pre>
 * AthenaStartQueryExecution startQueryExecutionJob = AthenaStartQueryExecution.Builder.create(this, "Start Athena Query")
 *         .queryString(JsonPath.stringAt("$.queryString"))
 *         .queryExecutionContext(QueryExecutionContext.builder()
 *                 .databaseName("mydatabase")
 *                 .build())
 *         .resultConfiguration(ResultConfiguration.builder()
 *                 .encryptionConfiguration(EncryptionConfiguration.builder()
 *                         .encryptionOption(EncryptionOption.S3_MANAGED)
 *                         .build())
 *                 .outputLocation(Location.builder()
 *                         .bucketName("amzn-s3-demo-bucket")
 *                         .objectKey("folder")
 *                         .build())
 *                 .build())
 *         .executionParameters(List.of("param1", "param2"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can reuse the query results by setting the <code>resultReuseConfigurationMaxAge</code> property.
 * <p>
 * <blockquote><pre>
 * AthenaStartQueryExecution startQueryExecutionJob = AthenaStartQueryExecution.Builder.create(this, "Start Athena Query")
 *         .queryString(JsonPath.stringAt("$.queryString"))
 *         .queryExecutionContext(QueryExecutionContext.builder()
 *                 .databaseName("mydatabase")
 *                 .build())
 *         .resultConfiguration(ResultConfiguration.builder()
 *                 .encryptionConfiguration(EncryptionConfiguration.builder()
 *                         .encryptionOption(EncryptionOption.S3_MANAGED)
 *                         .build())
 *                 .outputLocation(Location.builder()
 *                         .bucketName("query-results-bucket")
 *                         .objectKey("folder")
 *                         .build())
 *                 .build())
 *         .executionParameters(List.of("param1", "param2"))
 *         .resultReuseConfigurationMaxAge(Duration.minutes(100))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>GetQueryExecution</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryExecution.html">GetQueryExecution</a> API gets information about a single execution of a query.
 * <p>
 * <blockquote><pre>
 * AthenaGetQueryExecution getQueryExecutionJob = AthenaGetQueryExecution.Builder.create(this, "Get Query Execution")
 *         .queryExecutionId(JsonPath.stringAt("$.QueryExecutionId"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>GetQueryResults</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryResults.html">GetQueryResults</a> API that streams the results of a single query execution specified by QueryExecutionId from S3.
 * <p>
 * <blockquote><pre>
 * AthenaGetQueryResults getQueryResultsJob = AthenaGetQueryResults.Builder.create(this, "Get Query Results")
 *         .queryExecutionId(JsonPath.stringAt("$.QueryExecutionId"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>StopQueryExecution</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html">StopQueryExecution</a> API that stops a query execution.
 * <p>
 * <blockquote><pre>
 * AthenaStopQueryExecution stopQueryExecutionJob = AthenaStopQueryExecution.Builder.create(this, "Stop Query Execution")
 *         .queryExecutionId(JsonPath.stringAt("$.QueryExecutionId"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Batch</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html">Batch</a> through the service integration pattern.
 * <p>
 * <h3>SubmitJob</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html">SubmitJob</a> API submits an AWS Batch job from a job definition.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.batch.*;
 * EcsJobDefinition batchJobDefinition;
 * JobQueue batchQueue;
 * 
 * 
 * BatchSubmitJob task = BatchSubmitJob.Builder.create(this, "Submit Job")
 *         .jobDefinitionArn(batchJobDefinition.getJobDefinitionArn())
 *         .jobName("MyJob")
 *         .jobQueueArn(batchQueue.getJobQueueArn())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Bedrock</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-bedrock.html">Bedrock</a> through the service integration pattern.
 * <p>
 * <h3>InvokeModel</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html">InvokeModel</a> API
 * invokes the specified Bedrock model to run inference using the input provided.
 * The format of the input body and the response body depend on the model selected.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.bedrock.*;
 * 
 * 
 * FoundationModel model = FoundationModel.fromFoundationModelId(this, "Model", FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1);
 * 
 * BedrockInvokeModel task = BedrockInvokeModel.Builder.create(this, "Prompt Model")
 *         .model(model)
 *         .body(TaskInput.fromObject(Map.of(
 *                 "inputText", "Generate a list of five first names.",
 *                 "textGenerationConfig", Map.of(
 *                         "maxTokenCount", 100,
 *                         "temperature", 1))))
 *         .resultSelector(Map.of(
 *                 "names", JsonPath.stringAt("$.Body.results[0].outputText")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Using Input Path for S3 URI</h3>
 * <p>
 * Provide S3 URI as an input or output path to invoke a model
 * <p>
 * To specify the S3 URI as JSON path to your input or output fields, use props <code>s3InputUri</code> and <code>s3OutputUri</code> under BedrockInvokeModelProps and set
 * feature flag <code>&#64;aws-cdk/aws-stepfunctions-tasks:useNewS3UriParametersForBedrockInvokeModelTask</code> to true.
 * <p>
 * If this flag is not enabled, the code will populate the S3Uri using <code>InputPath</code> and <code>OutputPath</code> fields, which is not recommended.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.bedrock.*;
 * 
 * 
 * FoundationModel model = FoundationModel.fromFoundationModelId(this, "Model", FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1);
 * 
 * BedrockInvokeModel task = BedrockInvokeModel.Builder.create(this, "Prompt Model")
 *         .model(model)
 *         .input(BedrockInvokeModelInputProps.builder().s3InputUri(JsonPath.stringAt("$.prompt")).build())
 *         .output(BedrockInvokeModelOutputProps.builder().s3OutputUri(JsonPath.stringAt("$.prompt")).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Using Input Path</h3>
 * <p>
 * Provide S3 URI as an input or output path to invoke a model
 * <p>
 * Currently, input and output Path provided in the BedrockInvokeModelProps input is defined as S3URI field under task definition of state machine.
 * To modify the existing behaviour, set <code>&#64;aws-cdk/aws-stepfunctions-tasks:useNewS3UriParametersForBedrockInvokeModelTask</code> to true.
 * <p>
 * If this feature flag is enabled, S3URI fields will be generated from other Props(<code>s3InputUri</code> and <code>s3OutputUri</code>), and the given inputPath, OutputPath will be rendered as
 * it is in the JSON task definition.
 * <p>
 * If the feature flag is set to <code>false</code>, the behavior will be to populate the S3Uri using the <code>InputPath</code> and <code>OutputPath</code> fields, which is not recommended.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.bedrock.*;
 * 
 * 
 * FoundationModel model = FoundationModel.fromFoundationModelId(this, "Model", FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1);
 * 
 * BedrockInvokeModel task = BedrockInvokeModel.Builder.create(this, "Prompt Model")
 *         .model(model)
 *         .inputPath(JsonPath.stringAt("$.prompt"))
 *         .outputPath(JsonPath.stringAt("$.prompt"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can apply a guardrail to the invocation by setting <code>guardrail</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.bedrock.*;
 * 
 * 
 * FoundationModel model = FoundationModel.fromFoundationModelId(this, "Model", FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1);
 * 
 * BedrockInvokeModel task = BedrockInvokeModel.Builder.create(this, "Prompt Model with guardrail")
 *         .model(model)
 *         .body(TaskInput.fromObject(Map.of(
 *                 "inputText", "Generate a list of five first names.",
 *                 "textGenerationConfig", Map.of(
 *                         "maxTokenCount", 100,
 *                         "temperature", 1))))
 *         .guardrail(Guardrail.enable("guardrailId", 1))
 *         .resultSelector(Map.of(
 *                 "names", JsonPath.stringAt("$.Body.results[0].outputText")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>createModelCustomizationJob</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_CreateModelCustomizationJob.html">CreateModelCustomizationJob</a> API creates a fine-tuning job to customize a base model.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.bedrock.*;
 * import software.amazon.awscdk.services.kms.*;
 * 
 * IBucket outputBucket;
 * IBucket trainingBucket;
 * IBucket validationBucket;
 * IKey kmsKey;
 * IVpc vpc;
 * 
 * 
 * FoundationModel model = FoundationModel.fromFoundationModelId(this, "Model", FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1);
 * 
 * BedrockCreateModelCustomizationJob task = BedrockCreateModelCustomizationJob.Builder.create(this, "CreateModelCustomizationJob")
 *         .baseModel(model)
 *         .clientRequestToken("MyToken")
 *         .customizationType(CustomizationType.FINE_TUNING)
 *         .customModelKmsKey(kmsKey)
 *         .customModelName("MyCustomModel") // required
 *         .customModelTags(List.of(CustomModelTag.builder().key("key1").value("value1").build()))
 *         .hyperParameters(Map.of(
 *                 "batchSize", "10"))
 *         .jobName("MyCustomizationJob") // required
 *         .jobTags(List.of(CustomModelTag.builder().key("key2").value("value2").build()))
 *         .outputData(OutputBucketConfiguration.builder()
 *                 .bucket(outputBucket) // required
 *                 .path("output-data/")
 *                 .build())
 *         .trainingData(TrainingBucketConfiguration.builder()
 *                 .bucket(trainingBucket)
 *                 .path("training-data/data.json")
 *                 .build()) // required
 *         // If you don't provide validation data, you have to specify `Evaluation percentage` hyperparameter.
 *         .validationData(List.of(ValidationBucketConfiguration.builder()
 *                 .bucket(validationBucket)
 *                 .path("validation-data/data.json")
 *                 .build()))
 *         .vpcConfig(Map.of(
 *                 "securityGroups", List.of(SecurityGroup.Builder.create(this, "SecurityGroup").vpc(vpc).build()),
 *                 "subnets", vpc.getPrivateSubnets()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>CodeBuild</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html">CodeBuild</a> through the service integration pattern.
 * <p>
 * <h3>StartBuild</h3>
 * <p>
 * <a href="https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html">StartBuild</a> starts a CodeBuild Project by Project Name.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codebuild.*;
 * 
 * 
 * Project codebuildProject = Project.Builder.create(this, "Project")
 *         .projectName("MyTestProject")
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "build", Map.of(
 *                                 "commands", List.of("echo \"Hello, CodeBuild!\""))))))
 *         .build();
 * 
 * CodeBuildStartBuild task = CodeBuildStartBuild.Builder.create(this, "Task")
 *         .project(codebuildProject)
 *         .integrationPattern(IntegrationPattern.RUN_JOB)
 *         .environmentVariablesOverride(Map.of(
 *                 "ZONE", BuildEnvironmentVariable.builder()
 *                         .type(BuildEnvironmentVariableType.PLAINTEXT)
 *                         .value(JsonPath.stringAt("$.envVariables.zone"))
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>StartBuildBatch</h3>
 * <p>
 * <a href="https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuildBatch.html">StartBuildBatch</a> starts a batch CodeBuild for a project by project name.
 * It is necessary to enable the batch build feature in the CodeBuild project.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codebuild.*;
 * 
 * 
 * Project project = Project.Builder.create(this, "Project")
 *         .projectName("MyTestProject")
 *         .buildSpec(BuildSpec.fromObjectToYaml(Map.of(
 *                 "version", 0.2,
 *                 "batch", Map.of(
 *                         "build-list", List.of(Map.of(
 *                                 "identifier", "id",
 *                                 "buildspec", "version: 0.2\nphases:\n  build:\n    commands:\n      - echo \"Hello, from small!\""))))))
 *         .build();
 * project.enableBatchBuilds();
 * 
 * CodeBuildStartBuildBatch task = CodeBuildStartBuildBatch.Builder.create(this, "buildBatchTask")
 *         .project(project)
 *         .integrationPattern(IntegrationPattern.REQUEST_RESPONSE)
 *         .environmentVariablesOverride(Map.of(
 *                 "test", BuildEnvironmentVariable.builder()
 *                         .type(BuildEnvironmentVariableType.PLAINTEXT)
 *                         .value("testValue")
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: <code>enableBatchBuilds()</code> will do nothing for imported projects.
 * If you are using an imported project, you must ensure that the project is already configured for batch builds.
 * <p>
 * <h2>DynamoDB</h2>
 * <p>
 * You can call DynamoDB APIs from a <code>Task</code> state.
 * Read more about calling DynamoDB APIs <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-ddb.html">here</a>
 * <p>
 * <h3>GetItem</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html">GetItem</a> operation returns a set of attributes for the item with the given primary key.
 * <p>
 * <blockquote><pre>
 * Table myTable;
 * 
 * DynamoGetItem.Builder.create(this, "Get Item")
 *         .key(Map.of("messageId", DynamoAttributeValue.fromString("message-007")))
 *         .table(myTable)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>PutItem</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html">PutItem</a> operation creates a new item, or replaces an old item with a new item.
 * <p>
 * <blockquote><pre>
 * Table myTable;
 * 
 * DynamoPutItem.Builder.create(this, "PutItem")
 *         .item(Map.of(
 *                 "MessageId", DynamoAttributeValue.fromString("message-007"),
 *                 "Text", DynamoAttributeValue.fromString(JsonPath.stringAt("$.bar")),
 *                 "TotalCount", DynamoAttributeValue.fromNumber(10)))
 *         .table(myTable)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>DeleteItem</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html">DeleteItem</a> operation deletes a single item in a table by primary key.
 * <p>
 * <blockquote><pre>
 * Table myTable;
 * 
 * DynamoDeleteItem.Builder.create(this, "DeleteItem")
 *         .key(Map.of("MessageId", DynamoAttributeValue.fromString("message-007")))
 *         .table(myTable)
 *         .resultPath(JsonPath.DISCARD)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>UpdateItem</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html">UpdateItem</a> operation edits an existing item's attributes, or adds a new item
 * to the table if it does not already exist.
 * <p>
 * <blockquote><pre>
 * Table myTable;
 * 
 * DynamoUpdateItem.Builder.create(this, "UpdateItem")
 *         .key(Map.of(
 *                 "MessageId", DynamoAttributeValue.fromString("message-007")))
 *         .table(myTable)
 *         .expressionAttributeValues(Map.of(
 *                 ":val", DynamoAttributeValue.numberFromString(JsonPath.stringAt("$.Item.TotalCount.N")),
 *                 ":rand", DynamoAttributeValue.fromNumber(20)))
 *         .updateExpression("SET TotalCount = :val + :rand")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>ECS</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html">ECS/Fargate</a> through the service integration pattern.
 * <p>
 * <h3>RunTask</h3>
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html">RunTask</a> starts a new task using the specified task definition.
 * <p>
 * <h4>EC2</h4>
 * <p>
 * The EC2 launch type allows you to run your containerized applications on a cluster
 * of Amazon EC2 instances that you manage.
 * <p>
 * When a task that uses the EC2 launch type is launched, Amazon ECS must determine where
 * to place the task based on the requirements specified in the task definition, such as
 * CPU and memory. Similarly, when you scale down the task count, Amazon ECS must determine
 * which tasks to terminate. You can apply task placement strategies and constraints to
 * customize how Amazon ECS places and terminates tasks. Learn more about <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement.html">task placement</a>
 * <p>
 * The latest ACTIVE revision of the passed task definition is used for running the task.
 * <p>
 * The following example runs a job from a task definition on EC2
 * <p>
 * <blockquote><pre>
 * IVpc vpc = Vpc.fromLookup(this, "Vpc", VpcLookupOptions.builder()
 *         .isDefault(true)
 *         .build());
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Ec2Cluster").vpc(vpc).build();
 * cluster.addCapacity("DefaultAutoScalingGroup", AddCapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.micro"))
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build())
 *         .build());
 * 
 * TaskDefinition taskDefinition = TaskDefinition.Builder.create(this, "TD")
 *         .compatibility(Compatibility.EC2)
 *         .build();
 * 
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("foo/bar"))
 *         .memoryLimitMiB(256)
 *         .build());
 * 
 * EcsRunTask runTask = EcsRunTask.Builder.create(this, "Run")
 *         .integrationPattern(IntegrationPattern.RUN_JOB)
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .launchTarget(EcsEc2LaunchTarget.Builder.create()
 *                 .placementStrategies(List.of(PlacementStrategy.spreadAcrossInstances(), PlacementStrategy.packedByCpu(), PlacementStrategy.randomly()))
 *                 .placementConstraints(List.of(PlacementConstraint.memberOf("blieptuut")))
 *                 .build())
 *         .propagatedTagSource(PropagatedTagSource.TASK_DEFINITION)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Fargate</h4>
 * <p>
 * AWS Fargate is a serverless compute engine for containers that works with Amazon
 * Elastic Container Service (ECS). Fargate makes it easy for you to focus on building
 * your applications. Fargate removes the need to provision and manage servers, lets you
 * specify and pay for resources per application, and improves security through application
 * isolation by design. Learn more about <a href="https://aws.amazon.com/fargate/">Fargate</a>
 * <p>
 * The Fargate launch type allows you to run your containerized applications without the need
 * to provision and manage the backend infrastructure. Just register your task definition and
 * Fargate launches the container for you. The latest ACTIVE revision of the passed
 * task definition is used for running the task. Learn more about
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskDefinition.html">Fargate Versioning</a>
 * <p>
 * The following example runs a job from a task definition on Fargate
 * <p>
 * <blockquote><pre>
 * IVpc vpc = Vpc.fromLookup(this, "Vpc", VpcLookupOptions.builder()
 *         .isDefault(true)
 *         .build());
 * 
 * Cluster cluster = Cluster.Builder.create(this, "FargateCluster").vpc(vpc).build();
 * 
 * TaskDefinition taskDefinition = TaskDefinition.Builder.create(this, "TD")
 *         .memoryMiB("512")
 *         .cpu("256")
 *         .compatibility(Compatibility.FARGATE)
 *         .build();
 * 
 * ContainerDefinition containerDefinition = taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("foo/bar"))
 *         .memoryLimitMiB(256)
 *         .build());
 * 
 * EcsRunTask runTask = EcsRunTask.Builder.create(this, "RunFargate")
 *         .integrationPattern(IntegrationPattern.RUN_JOB)
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .assignPublicIp(true)
 *         .containerOverrides(List.of(ContainerOverride.builder()
 *                 .containerDefinition(containerDefinition)
 *                 .environment(List.of(TaskEnvironmentVariable.builder().name("SOME_KEY").value(JsonPath.stringAt("$.SomeKey")).build()))
 *                 .build()))
 *         .launchTarget(new EcsFargateLaunchTarget())
 *         .propagatedTagSource(PropagatedTagSource.TASK_DEFINITION)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Override CPU and Memory Parameter</h4>
 * <p>
 * By setting the property cpu or memoryMiB, you can override the Fargate or EC2 task instance size at runtime.
 * <p>
 * see: https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TaskOverride.html
 * <p>
 * <blockquote><pre>
 * IVpc vpc = Vpc.fromLookup(this, "Vpc", VpcLookupOptions.builder()
 *         .isDefault(true)
 *         .build());
 * Cluster cluster = Cluster.Builder.create(this, "ECSCluster").vpc(vpc).build();
 * 
 * TaskDefinition taskDefinition = TaskDefinition.Builder.create(this, "TD")
 *         .compatibility(Compatibility.FARGATE)
 *         .cpu("256")
 *         .memoryMiB("512")
 *         .build();
 * 
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("foo/bar"))
 *         .build());
 * 
 * EcsRunTask runTask = EcsRunTask.Builder.create(this, "Run")
 *         .integrationPattern(IntegrationPattern.RUN_JOB)
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .launchTarget(new EcsFargateLaunchTarget())
 *         .cpu("1024")
 *         .memoryMiB("1048")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>ECS enable Exec</h4>
 * <p>
 * By setting the property <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html#ECS-RunTask-request-enableExecuteCommand"><code>enableExecuteCommand</code></a> to <code>true</code>, you can enable the <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-exec.html">ECS Exec feature</a> for the task for either Fargate or EC2 launch types.
 * <p>
 * <blockquote><pre>
 * IVpc vpc = Vpc.fromLookup(this, "Vpc", VpcLookupOptions.builder()
 *         .isDefault(true)
 *         .build());
 * Cluster cluster = Cluster.Builder.create(this, "ECSCluster").vpc(vpc).build();
 * 
 * TaskDefinition taskDefinition = TaskDefinition.Builder.create(this, "TD")
 *         .compatibility(Compatibility.EC2)
 *         .build();
 * 
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("foo/bar"))
 *         .memoryLimitMiB(256)
 *         .build());
 * 
 * EcsRunTask runTask = EcsRunTask.Builder.create(this, "Run")
 *         .integrationPattern(IntegrationPattern.RUN_JOB)
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .launchTarget(new EcsEc2LaunchTarget())
 *         .enableExecuteCommand(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>EMR</h2>
 * <p>
 * Step Functions supports Amazon EMR through the service integration pattern.
 * The service integration APIs correspond to Amazon EMR APIs but differ in the
 * parameters that are used.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html">Read more</a> about the differences when using these service integrations.
 * <p>
 * <h3>Create Cluster</h3>
 * <p>
 * Creates and starts running a cluster (job flow).
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html"><code>runJobFlow</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * Role clusterRole = Role.Builder.create(this, "ClusterRole")
 *         .assumedBy(new ServicePrincipal("ec2.amazonaws.com"))
 *         .build();
 * 
 * Role serviceRole = Role.Builder.create(this, "ServiceRole")
 *         .assumedBy(new ServicePrincipal("elasticmapreduce.amazonaws.com"))
 *         .build();
 * 
 * Role autoScalingRole = Role.Builder.create(this, "AutoScalingRole")
 *         .assumedBy(new ServicePrincipal("elasticmapreduce.amazonaws.com"))
 *         .build();
 * 
 * autoScalingRole.assumeRolePolicy.addStatements(
 * PolicyStatement.Builder.create()
 *         .effect(Effect.ALLOW)
 *         .principals(List.of(
 *             new ServicePrincipal("application-autoscaling.amazonaws.com")))
 *         .actions(List.of("sts:AssumeRole"))
 *         .build());
 * 
 * EmrCreateCluster.Builder.create(this, "Create Cluster")
 *         .instances(InstancesConfigProperty.builder().build())
 *         .clusterRole(clusterRole)
 *         .name(TaskInput.fromJsonPathAt("$.ClusterName").getValue())
 *         .serviceRole(serviceRole)
 *         .autoScalingRole(autoScalingRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can use the launch specification for On-Demand and Spot instances in the fleet.
 * <p>
 * <blockquote><pre>
 * EmrCreateCluster.Builder.create(this, "OnDemandSpecification")
 *         .instances(InstancesConfigProperty.builder()
 *                 .instanceFleets(List.of(InstanceFleetConfigProperty.builder()
 *                         .instanceFleetType(EmrCreateCluster.getInstanceRoleType().MASTER)
 *                         .launchSpecifications(InstanceFleetProvisioningSpecificationsProperty.builder()
 *                                 .onDemandSpecification(OnDemandProvisioningSpecificationProperty.builder()
 *                                         .allocationStrategy(EmrCreateCluster.getOnDemandAllocationStrategy().LOWEST_PRICE)
 *                                         .build())
 *                                 .build())
 *                         .build()))
 *                 .build())
 *         .name("OnDemandCluster")
 *         .integrationPattern(IntegrationPattern.RUN_JOB)
 *         .build();
 * 
 * EmrCreateCluster.Builder.create(this, "SpotSpecification")
 *         .instances(InstancesConfigProperty.builder()
 *                 .instanceFleets(List.of(InstanceFleetConfigProperty.builder()
 *                         .instanceFleetType(EmrCreateCluster.getInstanceRoleType().MASTER)
 *                         .launchSpecifications(InstanceFleetProvisioningSpecificationsProperty.builder()
 *                                 .spotSpecification(SpotProvisioningSpecificationProperty.builder()
 *                                         .allocationStrategy(EmrCreateCluster.getSpotAllocationStrategy().CAPACITY_OPTIMIZED)
 *                                         .timeoutAction(EmrCreateCluster.getSpotTimeoutAction().TERMINATE_CLUSTER)
 *                                         .timeout(Duration.minutes(5))
 *                                         .build())
 *                                 .build())
 *                         .build()))
 *                 .build())
 *         .name("SpotCluster")
 *         .integrationPattern(IntegrationPattern.RUN_JOB)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to run multiple steps in <a href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-concurrent-steps.html">parallel</a>,
 * you can specify the <code>stepConcurrencyLevel</code> property. The concurrency range is between 1
 * and 256 inclusive, where the default concurrency of 1 means no step concurrency is allowed.
 * <code>stepConcurrencyLevel</code> requires the EMR release label to be 5.28.0 or above.
 * <p>
 * <blockquote><pre>
 * EmrCreateCluster.Builder.create(this, "Create Cluster")
 *         .instances(InstancesConfigProperty.builder().build())
 *         .name(TaskInput.fromJsonPathAt("$.ClusterName").getValue())
 *         .stepConcurrencyLevel(10)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to use an <a href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-auto-termination-policy.html">auto-termination policy</a>,
 * you can specify the <code>autoTerminationPolicyIdleTimeout</code> property.
 * Specifies the amount of idle time after which the cluster automatically terminates. You can specify a minimum of 60 seconds and a maximum of 604800 seconds (seven days).
 * <p>
 * <blockquote><pre>
 * EmrCreateCluster.Builder.create(this, "Create Cluster")
 *         .instances(InstancesConfigProperty.builder().build())
 *         .name("ClusterName")
 *         .autoTerminationPolicyIdleTimeout(Duration.seconds(100))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Termination Protection</h3>
 * <p>
 * Locks a cluster (job flow) so the EC2 instances in the cluster cannot be
 * terminated by user intervention, an API call, or a job-flow error.
 * <p>
 * Corresponds to the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html"><code>setTerminationProtection</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * EmrSetClusterTerminationProtection.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .terminationProtected(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Terminate Cluster</h3>
 * <p>
 * Shuts down a cluster (job flow).
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_TerminateJobFlows.html"><code>terminateJobFlows</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * EmrTerminateCluster.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Add Step</h3>
 * <p>
 * Adds a new step to a running cluster.
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_AddJobFlowSteps.html"><code>addJobFlowSteps</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * EmrAddStep.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .name("StepName")
 *         .jar("Jar")
 *         .actionOnFailure(ActionOnFailure.CONTINUE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To specify a custom runtime role use the <code>executionRoleArn</code> property.
 * <p>
 * <strong>Note:</strong> The EMR cluster must be created with a security configuration and the runtime role must have a specific trust policy.
 * See this <a href="https://aws.amazon.com/blogs/big-data/introducing-runtime-roles-for-amazon-emr-steps-use-iam-roles-and-aws-lake-formation-for-access-control-with-amazon-emr/">blog post</a> for more details.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.emr.*;
 * 
 * 
 * CfnSecurityConfiguration cfnSecurityConfiguration = CfnSecurityConfiguration.Builder.create(this, "EmrSecurityConfiguration")
 *         .name("AddStepRuntimeRoleSecConfig")
 *         .securityConfiguration(JSON.parse("\n    {\n      \"AuthorizationConfiguration\": {\n          \"IAMConfiguration\": {\n              \"EnableApplicationScopedIAMRole\": true,\n              \"ApplicationScopedIAMRoleConfiguration\":\n                  {\n                      \"PropagateSourceIdentity\": true\n                  }\n          },\n          \"LakeFormationConfiguration\": {\n              \"AuthorizedSessionTagValue\": \"Amazon EMR\"\n          }\n      }\n    }"))
 *         .build();
 * 
 * EmrCreateCluster task = EmrCreateCluster.Builder.create(this, "Create Cluster")
 *         .instances(InstancesConfigProperty.builder().build())
 *         .name(TaskInput.fromJsonPathAt("$.ClusterName").getValue())
 *         .securityConfiguration(cfnSecurityConfiguration.getName())
 *         .build();
 * 
 * Role executionRole = Role.Builder.create(this, "Role")
 *         .assumedBy(new ArnPrincipal(task.getClusterRole().getRoleArn()))
 *         .build();
 * 
 * executionRole.assumeRolePolicy.addStatements(
 * PolicyStatement.Builder.create()
 *         .effect(Effect.ALLOW)
 *         .principals(List.of(task.getClusterRole()))
 *         .actions(List.of("sts:SetSourceIdentity"))
 *         .build(),
 * PolicyStatement.Builder.create()
 *         .effect(Effect.ALLOW)
 *         .principals(List.of(task.getClusterRole()))
 *         .actions(List.of("sts:TagSession"))
 *         .conditions(Map.of(
 *                 "StringEquals", Map.of(
 *                         "aws:RequestTag/LakeFormationAuthorizedCaller", "Amazon EMR")))
 *         .build());
 * 
 * EmrAddStep.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .executionRoleArn(executionRole.getRoleArn())
 *         .name("StepName")
 *         .jar("Jar")
 *         .actionOnFailure(ActionOnFailure.CONTINUE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cancel Step</h3>
 * <p>
 * Cancels a pending step in a running cluster.
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_CancelSteps.html"><code>cancelSteps</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * EmrCancelStep.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .stepId("StepId")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Modify Instance Fleet</h3>
 * <p>
 * Modifies the target On-Demand and target Spot capacities for the instance
 * fleet with the specified InstanceFleetName.
 * <p>
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceFleet.html"><code>modifyInstanceFleet</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * EmrModifyInstanceFleetByName.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .instanceFleetName("InstanceFleetName")
 *         .targetOnDemandCapacity(2)
 *         .targetSpotCapacity(0)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Modify Instance Group</h3>
 * <p>
 * Modifies the number of nodes and configuration settings of an instance group.
 * <p>
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html"><code>modifyInstanceGroups</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * EmrModifyInstanceGroupByName.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .instanceGroupName(JsonPath.stringAt("$.InstanceGroupName"))
 *         .instanceGroup(InstanceGroupModifyConfigProperty.builder()
 *                 .instanceCount(1)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>EMR on EKS</h2>
 * <p>
 * Step Functions supports Amazon EMR on EKS through the service integration pattern.
 * The service integration APIs correspond to Amazon EMR on EKS APIs, but differ in the parameters that are used.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html">Read more</a> about the differences when using these service integrations.
 * <p>
 * <a href="https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up.html">Setting up</a> the EKS cluster is required.
 * <p>
 * <h3>Create Virtual Cluster</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_CreateVirtualCluster.html">CreateVirtualCluster</a> API creates a single virtual cluster that's mapped to a single Kubernetes namespace.
 * <p>
 * The EKS cluster containing the Kubernetes namespace where the virtual cluster will be mapped can be passed in from the task input.
 * <p>
 * <blockquote><pre>
 * EmrContainersCreateVirtualCluster.Builder.create(this, "Create a Virtual Cluster")
 *         .eksCluster(EksClusterInput.fromTaskInput(TaskInput.fromText("clusterId")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The EKS cluster can also be passed in directly.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.eks.*;
 * 
 * Cluster eksCluster;
 * 
 * 
 * EmrContainersCreateVirtualCluster.Builder.create(this, "Create a Virtual Cluster")
 *         .eksCluster(EksClusterInput.fromCluster(eksCluster))
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, the Kubernetes namespace that a virtual cluster maps to is "default", but a specific namespace within an EKS cluster can be selected.
 * <p>
 * <blockquote><pre>
 * EmrContainersCreateVirtualCluster.Builder.create(this, "Create a Virtual Cluster")
 *         .eksCluster(EksClusterInput.fromTaskInput(TaskInput.fromText("clusterId")))
 *         .eksNamespace("specified-namespace")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Delete Virtual Cluster</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_DeleteVirtualCluster.html">DeleteVirtualCluster</a> API deletes a virtual cluster.
 * <p>
 * <blockquote><pre>
 * EmrContainersDeleteVirtualCluster.Builder.create(this, "Delete a Virtual Cluster")
 *         .virtualClusterId(TaskInput.fromJsonPathAt("$.virtualCluster"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Start Job Run</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_StartJobRun.html">StartJobRun</a> API starts a job run. A job is a unit of work that you submit to Amazon EMR on EKS for execution. The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query. A job run is an execution of the job on the virtual cluster.
 * <p>
 * Required setup:
 * <p>
 * <ul>
 * <li>If not done already, follow the <a href="https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up.html">steps</a> to setup EMR on EKS and <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-eks-readme.html#quick-start">create an EKS Cluster</a>.</li>
 * <li>Enable <a href="https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-cluster-access.html">Cluster access</a></li>
 * <li>Enable <a href="https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-enable-IAM.html">IAM Role access</a></li>
 * </ul>
 * <p>
 * The following actions must be performed if the virtual cluster ID is supplied from the task input. Otherwise, if it is supplied statically in the state machine definition, these actions will be done automatically.
 * <p>
 * <ul>
 * <li>Create an <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-iam.Role.html">IAM role</a></li>
 * <li>Update the <a href="https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-trust-policy.html">Role Trust Policy</a> of the Job Execution Role.</li>
 * </ul>
 * <p>
 * The job can be configured with spark submit parameters:
 * <p>
 * <blockquote><pre>
 * EmrContainersStartJobRun.Builder.create(this, "EMR Containers Start Job Run")
 *         .virtualCluster(VirtualClusterInput.fromVirtualClusterId("de92jdei2910fwedz"))
 *         .releaseLabel(ReleaseLabel.EMR_6_2_0)
 *         .jobDriver(JobDriver.builder()
 *                 .sparkSubmitJobDriver(SparkSubmitJobDriver.builder()
 *                         .entryPoint(TaskInput.fromText("local:///usr/lib/spark/examples/src/main/python/pi.py"))
 *                         .sparkSubmitParameters("--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1")
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Configuring the job can also be done via application configuration:
 * <p>
 * <blockquote><pre>
 * EmrContainersStartJobRun.Builder.create(this, "EMR Containers Start Job Run")
 *         .virtualCluster(VirtualClusterInput.fromVirtualClusterId("de92jdei2910fwedz"))
 *         .releaseLabel(ReleaseLabel.EMR_6_2_0)
 *         .jobName("EMR-Containers-Job")
 *         .jobDriver(JobDriver.builder()
 *                 .sparkSubmitJobDriver(SparkSubmitJobDriver.builder()
 *                         .entryPoint(TaskInput.fromText("local:///usr/lib/spark/examples/src/main/python/pi.py"))
 *                         .build())
 *                 .build())
 *         .applicationConfig(List.of(ApplicationConfiguration.builder()
 *                 .classification(Classification.SPARK_DEFAULTS)
 *                 .properties(Map.of(
 *                         "spark.executor.instances", "1",
 *                         "spark.executor.memory", "512M"))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Job monitoring can be enabled if <code>monitoring.logging</code> is set true. This automatically generates an S3 bucket and CloudWatch logs.
 * <p>
 * <blockquote><pre>
 * EmrContainersStartJobRun.Builder.create(this, "EMR Containers Start Job Run")
 *         .virtualCluster(VirtualClusterInput.fromVirtualClusterId("de92jdei2910fwedz"))
 *         .releaseLabel(ReleaseLabel.EMR_6_2_0)
 *         .jobDriver(JobDriver.builder()
 *                 .sparkSubmitJobDriver(SparkSubmitJobDriver.builder()
 *                         .entryPoint(TaskInput.fromText("local:///usr/lib/spark/examples/src/main/python/pi.py"))
 *                         .sparkSubmitParameters("--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1")
 *                         .build())
 *                 .build())
 *         .monitoring(Monitoring.builder()
 *                 .logging(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Otherwise, providing monitoring for jobs with existing log groups and log buckets is also available.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * 
 * 
 * LogGroup logGroup = new LogGroup(this, "Log Group");
 * Bucket logBucket = new Bucket(this, "S3 Bucket");
 * 
 * EmrContainersStartJobRun.Builder.create(this, "EMR Containers Start Job Run")
 *         .virtualCluster(VirtualClusterInput.fromVirtualClusterId("de92jdei2910fwedz"))
 *         .releaseLabel(ReleaseLabel.EMR_6_2_0)
 *         .jobDriver(JobDriver.builder()
 *                 .sparkSubmitJobDriver(SparkSubmitJobDriver.builder()
 *                         .entryPoint(TaskInput.fromText("local:///usr/lib/spark/examples/src/main/python/pi.py"))
 *                         .sparkSubmitParameters("--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1")
 *                         .build())
 *                 .build())
 *         .monitoring(Monitoring.builder()
 *                 .logGroup(logGroup)
 *                 .logBucket(logBucket)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Users can provide their own existing Job Execution Role.
 * <p>
 * <blockquote><pre>
 * EmrContainersStartJobRun.Builder.create(this, "EMR Containers Start Job Run")
 *         .virtualCluster(VirtualClusterInput.fromTaskInput(TaskInput.fromJsonPathAt("$.VirtualClusterId")))
 *         .releaseLabel(ReleaseLabel.EMR_6_2_0)
 *         .jobName("EMR-Containers-Job")
 *         .executionRole(Role.fromRoleArn(this, "Job-Execution-Role", "arn:aws:iam::xxxxxxxxxxxx:role/JobExecutionRole"))
 *         .jobDriver(JobDriver.builder()
 *                 .sparkSubmitJobDriver(SparkSubmitJobDriver.builder()
 *                         .entryPoint(TaskInput.fromText("local:///usr/lib/spark/examples/src/main/python/pi.py"))
 *                         .sparkSubmitParameters("--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1")
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>EKS</h2>
 * <p>
 * Step Functions supports Amazon EKS through the service integration pattern.
 * The service integration APIs correspond to Amazon EKS APIs.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html">Read more</a> about the differences when using these service integrations.
 * <p>
 * <h3>Call</h3>
 * <p>
 * Read and write Kubernetes resource objects via a Kubernetes API endpoint.
 * Corresponds to the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html"><code>call</code></a> API in Step Functions Connector.
 * <p>
 * The following code snippet includes a Task state that uses eks:call to list the pods.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.eks.*;
 * import software.amazon.awscdk.cdk.lambdalayer.kubectl.v32.KubectlV32Layer;
 * 
 * 
 * Cluster myEksCluster = Cluster.Builder.create(this, "my sample cluster")
 *         .version(KubernetesVersion.V1_32)
 *         .clusterName("myEksCluster")
 *         .kubectlLayer(new KubectlV32Layer(this, "kubectl"))
 *         .build();
 * 
 * EksCall.Builder.create(this, "Call a EKS Endpoint")
 *         .cluster(myEksCluster)
 *         .httpMethod(HttpMethods.GET)
 *         .httpPath("/api/v1/namespaces/default/pods")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>EventBridge</h2>
 * <p>
 * Step Functions supports Amazon EventBridge through the service integration pattern.
 * The service integration APIs correspond to Amazon EventBridge APIs.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html">Read more</a> about the differences when using these service integrations.
 * <p>
 * <h3>Put Events</h3>
 * <p>
 * Send events to an EventBridge bus.
 * Corresponds to the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html"><code>put-events</code></a> API in Step Functions Connector.
 * <p>
 * The following code snippet includes a Task state that uses events:putevents to send an event to the default bus.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.events.*;
 * 
 * 
 * EventBus myEventBus = EventBus.Builder.create(this, "EventBus")
 *         .eventBusName("MyEventBus1")
 *         .build();
 * 
 * EventBridgePutEvents.Builder.create(this, "Send an event to EventBridge")
 *         .entries(List.of(EventBridgePutEventsEntry.builder()
 *                 .detail(TaskInput.fromObject(Map.of(
 *                         "Message", "Hello from Step Functions!")))
 *                 .eventBus(myEventBus)
 *                 .detailType("MessageFromStepFunctions")
 *                 .source("step.functions")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>EventBridge Scheduler</h2>
 * <p>
 * You can call EventBridge Scheduler APIs from a <code>Task</code> state.
 * Read more about calling Scheduler APIs <a href="https://docs.aws.amazon.com/scheduler/latest/APIReference/API_Operations.html">here</a>
 * <p>
 * <h3>Create Scheduler</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/scheduler/latest/APIReference/API_CreateSchedule.html">CreateSchedule</a> API creates a new schedule.
 * <p>
 * Here is an example of how to create a schedule that puts an event to SQS queue every 5 minutes:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.scheduler.*;
 * import software.amazon.awscdk.services.kms.*;
 * 
 * Key key;
 * CfnScheduleGroup scheduleGroup;
 * Queue targetQueue;
 * Queue deadLetterQueue;
 * 
 * 
 * Role schedulerRole = Role.Builder.create(this, "SchedulerRole")
 *         .assumedBy(new ServicePrincipal("scheduler.amazonaws.com"))
 *         .build();
 * // To send the message to the queue
 * // This policy changes depending on the type of target.
 * schedulerRole.addToPrincipalPolicy(PolicyStatement.Builder.create()
 *         .actions(List.of("sqs:SendMessage"))
 *         .resources(List.of(targetQueue.getQueueArn()))
 *         .build());
 * 
 * EventBridgeSchedulerCreateScheduleTask createScheduleTask1 = EventBridgeSchedulerCreateScheduleTask.Builder.create(this, "createSchedule")
 *         .scheduleName("TestSchedule")
 *         .actionAfterCompletion(ActionAfterCompletion.NONE)
 *         .clientToken("testToken")
 *         .description("TestDescription")
 *         .startDate(new Date())
 *         .endDate(new Date(new Date().getTime() + 1000 * 60 * 60))
 *         .flexibleTimeWindow(Duration.minutes(5))
 *         .groupName(scheduleGroup.getRef())
 *         .kmsKey(key)
 *         .schedule(Schedule.rate(Duration.minutes(5)))
 *         .timezone("UTC")
 *         .enabled(true)
 *         .target(EventBridgeSchedulerTarget.Builder.create()
 *                 .arn(targetQueue.getQueueArn())
 *                 .role(schedulerRole)
 *                 .retryPolicy(RetryPolicy.builder()
 *                         .maximumRetryAttempts(2)
 *                         .maximumEventAge(Duration.minutes(5))
 *                         .build())
 *                 .deadLetterQueue(deadLetterQueue)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Glue</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html">AWS Glue</a> through the service integration pattern.
 * <p>
 * <h3>StartJobRun</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun"><code>StartJobRun</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * GlueStartJobRun.Builder.create(this, "Task")
 *         .glueJobName("my-glue-job")
 *         .arguments(TaskInput.fromObject(Map.of(
 *                 "key", "value")))
 *         .taskTimeout(Timeout.duration(Duration.minutes(30)))
 *         .notifyDelayAfter(Duration.minutes(5))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can configure workers by setting the <code>workerTypeV2</code> and <code>numberOfWorkers</code> properties.
 * <code>workerType</code> is deprecated and no longer recommended. Use <code>workerTypeV2</code> which is
 * a ENUM-like class for more powerful worker configuration around using pre-defined values or
 * dynamic values.
 * <p>
 * <blockquote><pre>
 * GlueStartJobRun.Builder.create(this, "Task")
 *         .glueJobName("my-glue-job")
 *         .workerConfiguration(WorkerConfigurationProperty.builder()
 *                 .workerTypeV2(WorkerTypeV2.G_1X) // Worker type
 *                 .numberOfWorkers(2)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * To configure the worker type or number of workers dynamically from StateMachine's input,
 * you can configure it using JSON Path values using <code>workerTypeV2</code> like this:
 * <p>
 * <blockquote><pre>
 * GlueStartJobRun.Builder.create(this, "Glue Job Task")
 *         .glueJobName("my-glue-job")
 *         .workerConfiguration(WorkerConfigurationProperty.builder()
 *                 .workerTypeV2(WorkerTypeV2.of(JsonPath.stringAt("$.glue_jobs_configs.executor_type")))
 *                 .numberOfWorkers(JsonPath.numberAt("$.glue_jobs_configs.max_number_workers"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can choose the execution class by setting the <code>executionClass</code> property.
 * <p>
 * <blockquote><pre>
 * GlueStartJobRun.Builder.create(this, "Task")
 *         .glueJobName("my-glue-job")
 *         .executionClass(ExecutionClass.FLEX)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>StartCrawlerRun</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StartCrawler"><code>StartCrawler</code></a> API from a <code>Task</code> state through AWS SDK service integrations.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.glue.*;
 * 
 * CfnCrawler myCrawler;
 * 
 * 
 * // You can get the crawler name from `crawler.ref`
 * // You can get the crawler name from `crawler.ref`
 * GlueStartCrawlerRun.Builder.create(this, "Task1")
 *         .crawlerName(myCrawler.getRef())
 *         .build();
 * 
 * // Of course, you can also specify the crawler name directly.
 * // Of course, you can also specify the crawler name directly.
 * GlueStartCrawlerRun.Builder.create(this, "Task2")
 *         .crawlerName("my-crawler-job")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Glue DataBrew</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-databrew.html">AWS Glue DataBrew</a> through the service integration pattern.
 * <p>
 * <h3>Start Job Run</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/databrew/latest/dg/API_StartJobRun.html"><code>StartJobRun</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * GlueDataBrewStartJobRun.Builder.create(this, "Task")
 *         .name("databrew-job")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Invoke HTTP API</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-third-party-apis.html">calling third-party APIs</a> with credentials managed by Amazon EventBridge <a href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_Connection.html">Connections</a>.
 * <p>
 * The following snippet creates a new API destination connection, and uses it to make a POST request to the specified URL. The endpoint response is available at the <code>$.ResponseBody</code> path.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.events.*;
 * 
 * 
 * Connection connection = Connection.Builder.create(this, "Connection")
 *         .authorization(Authorization.basic("username", SecretValue.unsafePlainText("password")))
 *         .build();
 * 
 * HttpInvoke.Builder.create(this, "Invoke HTTP API")
 *         .apiRoot("https://api.example.com")
 *         .apiEndpoint(TaskInput.fromText("path/to/resource"))
 *         .body(TaskInput.fromObject(Map.of("foo", "bar")))
 *         .connection(connection)
 *         .headers(TaskInput.fromObject(Map.of("Content-Type", "application/json")))
 *         .method(TaskInput.fromText("POST"))
 *         .queryStringParameters(TaskInput.fromObject(Map.of("id", "123")))
 *         .urlEncodingFormat(URLEncodingFormat.BRACKETS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Lambda</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html">AWS Lambda</a> through the service integration pattern.
 * <p>
 * <h3>Invoke</h3>
 * <p>
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">Invoke</a> a Lambda function.
 * <p>
 * You can specify the input to your Lambda function through the <code>payload</code> attribute.
 * By default, Step Functions invokes Lambda function with the state input (JSON path '$')
 * as the input.
 * <p>
 * The following snippet invokes a Lambda Function with the state input as the payload
 * by referencing the <code>$</code> path.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * LambdaInvoke.Builder.create(this, "Invoke with state input")
 *         .lambdaFunction(fn)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When a function is invoked, the Lambda service sends  <a href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html#API_Invoke_ResponseElements">these response
 * elements</a>
 * back.
 * <p>
 * ⚠️ The response from the Lambda function is in an attribute called <code>Payload</code>
 * <p>
 * The following snippet invokes a Lambda Function by referencing the <code>$.Payload</code> path
 * to reference the output of a Lambda executed before it.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * LambdaInvoke.Builder.create(this, "Invoke with empty object as payload")
 *         .lambdaFunction(fn)
 *         .payload(TaskInput.fromObject(Map.of()))
 *         .build();
 * 
 * // use the output of fn as input
 * // use the output of fn as input
 * LambdaInvoke.Builder.create(this, "Invoke with payload field in the state input")
 *         .lambdaFunction(fn)
 *         .payload(TaskInput.fromJsonPathAt("$.Payload"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following snippet invokes a Lambda and sets the task output to only include
 * the Lambda function response.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * LambdaInvoke.Builder.create(this, "Invoke and set function response as task output")
 *         .lambdaFunction(fn)
 *         .outputPath("$.Payload")
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to combine the input and the Lambda function response you can use
 * the <code>payloadResponseOnly</code> property and specify the <code>resultPath</code>. This will put the
 * Lambda function ARN directly in the "Resource" string, but it conflicts with the
 * integrationPattern, invocationType, clientContext, and qualifier properties.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * LambdaInvoke.Builder.create(this, "Invoke and combine function response with task input")
 *         .lambdaFunction(fn)
 *         .payloadResponseOnly(true)
 *         .resultPath("$.fn")
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can have Step Functions pause a task, and wait for an external process to
 * return a task token. Read more about the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/callback-task-sample-sqs.html#call-back-lambda-example">callback pattern</a>
 * <p>
 * To use the callback pattern, set the <code>token</code> property on the task. Call the Step
 * Functions <code>SendTaskSuccess</code> or <code>SendTaskFailure</code> APIs with the token to
 * indicate that the task has completed and the state machine should resume execution.
 * <p>
 * The following snippet invokes a Lambda with the task token as part of the input
 * to the Lambda.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * LambdaInvoke.Builder.create(this, "Invoke with callback")
 *         .lambdaFunction(fn)
 *         .integrationPattern(IntegrationPattern.WAIT_FOR_TASK_TOKEN)
 *         .payload(TaskInput.fromObject(Map.of(
 *                 "token", JsonPath.getTaskToken(),
 *                 "input", JsonPath.stringAt("$.someField"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * ⚠️ The task will pause until it receives that task token back with a <code>SendTaskSuccess</code> or <code>SendTaskFailure</code>
 * call. Learn more about <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token">Callback with the Task
 * Token</a>.
 * <p>
 * AWS Lambda can occasionally experience transient service errors. In this case, invoking Lambda
 * results in a 500 error, such as <code>ClientExecutionTimeoutException</code>, <code>ServiceException</code>, <code>AWSLambdaException</code>, or <code>SdkClientException</code>.
 * As a best practice, the <code>LambdaInvoke</code> task will retry on those errors with an interval of 2 seconds,
 * a back-off rate of 2 and 6 maximum attempts. Set the <code>retryOnServiceExceptions</code> prop to <code>false</code> to
 * disable this behavior.
 * <p>
 * <h2>MediaConvert</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-mediaconvert.html">AWS MediaConvert</a> through the Optimized integration pattern.
 * <p>
 * <h3>CreateJob</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html#jobspost">CreateJob</a> API creates a new transcoding job.
 * For information about jobs and job settings, see the User Guide at http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html
 * <p>
 * You can call the <code>CreateJob</code> API from a <code>Task</code> state. Optionally you can specify the <code>integrationPattern</code>.
 * <p>
 * Make sure you update the required fields - <a href="https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html#jobs-prop-createjobrequest-role">Role</a> &amp;
 * <a href="https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html#jobs-prop-createjobrequest-settings">Settings</a> and refer
 * <a href="https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html#jobs-model-createjobrequest">CreateJobRequest</a> for all other optional parameters.
 * <p>
 * <blockquote><pre>
 * MediaConvertCreateJob.Builder.create(this, "CreateJob")
 *         .createJobRequest(Map.of(
 *                 "Role", "arn:aws:iam::123456789012:role/MediaConvertRole",
 *                 "Settings", Map.of(
 *                         "OutputGroups", List.of(Map.of(
 *                                 "Outputs", List.of(Map.of(
 *                                         "ContainerSettings", Map.of(
 *                                                 "Container", "MP4"),
 *                                         "VideoDescription", Map.of(
 *                                                 "CodecSettings", Map.of(
 *                                                         "Codec", "H_264",
 *                                                         "H264Settings", Map.of(
 *                                                                 "MaxBitrate", 1000,
 *                                                                 "RateControlMode", "QVBR",
 *                                                                 "SceneChangeDetect", "TRANSITION_DETECTION"))),
 *                                         "AudioDescriptions", List.of(Map.of(
 *                                                 "CodecSettings", Map.of(
 *                                                         "Codec", "AAC",
 *                                                         "AacSettings", Map.of(
 *                                                                 "Bitrate", 96000,
 *                                                                 "CodingMode", "CODING_MODE_2_0",
 *                                                                 "SampleRate", 48000)))))),
 *                                 "OutputGroupSettings", Map.of(
 *                                         "Type", "FILE_GROUP_SETTINGS",
 *                                         "FileGroupSettings", Map.of(
 *                                                 "Destination", "s3://EXAMPLE-DESTINATION-BUCKET/")))),
 *                         "Inputs", List.of(Map.of(
 *                                 "AudioSelectors", Map.of(
 *                                         "Audio Selector 1", Map.of(
 *                                                 "DefaultSelection", "DEFAULT")),
 *                                 "FileInput", "s3://EXAMPLE-SOURCE-BUCKET/EXAMPLE-SOURCE_FILE")))))
 *         .integrationPattern(IntegrationPattern.RUN_JOB)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>SageMaker</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html">AWS SageMaker</a> through the service integration pattern.
 * <p>
 * If your training job or model uses resources from AWS Marketplace,
 * <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/mkt-algo-model-internet-free.html">network isolation is required</a>.
 * To do so, set the <code>enableNetworkIsolation</code> property to <code>true</code> for <code>SageMakerCreateModel</code> or <code>SageMakerCreateTrainingJob</code>.
 * <p>
 * To set environment variables for the Docker container use the <code>environment</code> property.
 * <p>
 * <h3>Create Training Job</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrainingJob.html"><code>CreateTrainingJob</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * SageMakerCreateTrainingJob.Builder.create(this, "TrainSagemaker")
 *         .trainingJobName(JsonPath.stringAt("$.JobName"))
 *         .algorithmSpecification(AlgorithmSpecification.builder()
 *                 .algorithmName("BlazingText")
 *                 .trainingInputMode(InputMode.FILE)
 *                 .build())
 *         .inputDataConfig(List.of(Channel.builder()
 *                 .channelName("train")
 *                 .dataSource(DataSource.builder()
 *                         .s3DataSource(S3DataSource.builder()
 *                                 .s3DataType(S3DataType.S3_PREFIX)
 *                                 .s3Location(S3Location.fromJsonExpression("$.S3Bucket"))
 *                                 .build())
 *                         .build())
 *                 .build()))
 *         .outputDataConfig(OutputDataConfig.builder()
 *                 .s3OutputLocation(S3Location.fromBucket(Bucket.fromBucketName(this, "Bucket", "amzn-s3-demo-bucket"), "myoutputpath"))
 *                 .build())
 *         .resourceConfig(ResourceConfig.builder()
 *                 .instanceCount(1)
 *                 .instanceType(new InstanceType(JsonPath.stringAt("$.InstanceType")))
 *                 .volumeSize(Size.gibibytes(50))
 *                 .build()) // optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
 *         .stoppingCondition(StoppingCondition.builder()
 *                 .maxRuntime(Duration.hours(2))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can specify <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AlgorithmSpecification.html#API_AlgorithmSpecification_Contents">TrainingInputMode</a> via the trainingInputMode property.
 * <p>
 * <ul>
 * <li>To download the data from Amazon Simple Storage Service (Amazon S3) to the provisioned ML storage volume, and mount the directory to a Docker volume, choose <code>InputMode.FILE</code> if an algorithm supports it.</li>
 * <li>To stream data directly from Amazon S3 to the container, choose <code>InputMode.PIPE</code> if an algorithm supports it.</li>
 * <li>To stream data directly from Amazon S3 to the container with no code changes and to provide file system access to the data, choose <code>InputMode.FAST_FILE</code> if an algorithm supports it.</li>
 * </ul>
 * <p>
 * <h3>Create Transform Job</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTransformJob.html"><code>CreateTransformJob</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * SageMakerCreateTransformJob.Builder.create(this, "Batch Inference")
 *         .transformJobName("MyTransformJob")
 *         .modelName("MyModelName")
 *         .modelClientOptions(ModelClientOptions.builder()
 *                 .invocationsMaxRetries(3) // default is 0
 *                 .invocationsTimeout(Duration.minutes(5))
 *                 .build())
 *         .transformInput(TransformInput.builder()
 *                 .transformDataSource(TransformDataSource.builder()
 *                         .s3DataSource(TransformS3DataSource.builder()
 *                                 .s3Uri("s3://inputbucket/train")
 *                                 .s3DataType(S3DataType.S3_PREFIX)
 *                                 .build())
 *                         .build())
 *                 .build())
 *         .transformOutput(TransformOutput.builder()
 *                 .s3OutputPath("s3://outputbucket/TransformJobOutputPath")
 *                 .build())
 *         .transformResources(TransformResources.builder()
 *                 .instanceCount(1)
 *                 .instanceType(InstanceType.of(InstanceClass.M4, InstanceSize.XLARGE))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Create Endpoint</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpoint.html"><code>CreateEndpoint</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * SageMakerCreateEndpoint.Builder.create(this, "SagemakerEndpoint")
 *         .endpointName(JsonPath.stringAt("$.EndpointName"))
 *         .endpointConfigName(JsonPath.stringAt("$.EndpointConfigName"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Create Endpoint Config</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpointConfig.html"><code>CreateEndpointConfig</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * SageMakerCreateEndpointConfig.Builder.create(this, "SagemakerEndpointConfig")
 *         .endpointConfigName("MyEndpointConfig")
 *         .productionVariants(List.of(ProductionVariant.builder()
 *                 .initialInstanceCount(2)
 *                 .instanceType(InstanceType.of(InstanceClass.M5, InstanceSize.XLARGE))
 *                 .modelName("MyModel")
 *                 .variantName("awesome-variant")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Create Model</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModel.html"><code>CreateModel</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * SageMakerCreateModel.Builder.create(this, "Sagemaker")
 *         .modelName("MyModel")
 *         .primaryContainer(ContainerDefinition.Builder.create()
 *                 .image(DockerImage.fromJsonExpression(JsonPath.stringAt("$.Model.imageName")))
 *                 .mode(Mode.SINGLE_MODEL)
 *                 .modelS3Location(S3Location.fromJsonExpression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Update Endpoint</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpoint.html"><code>UpdateEndpoint</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * SageMakerUpdateEndpoint.Builder.create(this, "SagemakerEndpoint")
 *         .endpointName(JsonPath.stringAt("$.Endpoint.Name"))
 *         .endpointConfigName(JsonPath.stringAt("$.Endpoint.EndpointConfig"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>SNS</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-sns.html">Amazon SNS</a> through the service integration pattern.
 * <p>
 * <h3>Publish</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sns/latest/api/API_Publish.html"><code>Publish</code></a> API from a <code>Task</code> state to publish to an SNS topic.
 * <p>
 * <blockquote><pre>
 * Topic topic = new Topic(this, "Topic");
 * 
 * // Use a field from the execution data as message.
 * SnsPublish task1 = SnsPublish.Builder.create(this, "Publish1")
 *         .topic(topic)
 *         .integrationPattern(IntegrationPattern.REQUEST_RESPONSE)
 *         .message(TaskInput.fromDataAt("$.state.message"))
 *         .messageAttributes(Map.of(
 *                 "place", MessageAttribute.builder()
 *                         .value(JsonPath.stringAt("$.place"))
 *                         .build(),
 *                 "pic", MessageAttribute.builder()
 *                         // BINARY must be explicitly set
 *                         .dataType(MessageAttributeDataType.BINARY)
 *                         .value(JsonPath.stringAt("$.pic"))
 *                         .build(),
 *                 "people", MessageAttribute.builder()
 *                         .value(4)
 *                         .build(),
 *                 "handles", MessageAttribute.builder()
 *                         .value(List.of("&#64;kslater", "&#64;jjf", null, "&#64;mfanning"))
 *                         .build()))
 *         .build();
 * 
 * // Combine a field from the execution data with
 * // a literal object.
 * SnsPublish task2 = SnsPublish.Builder.create(this, "Publish2")
 *         .topic(topic)
 *         .message(TaskInput.fromObject(Map.of(
 *                 "field1", "somedata",
 *                 "field2", JsonPath.stringAt("$.field2"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Step Functions</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html">AWS Step Functions</a> through the service integration pattern.
 * <p>
 * <h3>Start Execution</h3>
 * <p>
 * You can manage <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html">AWS Step Functions</a> executions.
 * <p>
 * AWS Step Functions supports it's own <a href="https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartExecution.html"><code>StartExecution</code></a> API as a service integration.
 * <p>
 * <blockquote><pre>
 * // Define a state machine with one Pass state
 * StateMachine child = StateMachine.Builder.create(this, "ChildStateMachine")
 *         .definition(Chain.start(new Pass(this, "PassState")))
 *         .build();
 * 
 * // Include the state machine in a Task state with callback pattern
 * StepFunctionsStartExecution task = StepFunctionsStartExecution.Builder.create(this, "ChildTask")
 *         .stateMachine(child)
 *         .integrationPattern(IntegrationPattern.WAIT_FOR_TASK_TOKEN)
 *         .input(TaskInput.fromObject(Map.of(
 *                 "token", JsonPath.getTaskToken(),
 *                 "foo", "bar")))
 *         .name("MyExecutionName")
 *         .build();
 * 
 * // Define a second state machine with the Task state above
 * // Define a second state machine with the Task state above
 * StateMachine.Builder.create(this, "ParentStateMachine")
 *         .definition(task)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can utilize <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-nested-workflows.html#nested-execution-startid">Associate Workflow Executions</a>
 * via the <code>associateWithParent</code> property. This allows the Step Functions UI to link child
 * executions from parent executions, making it easier to trace execution flow across state machines.
 * <p>
 * <blockquote><pre>
 * StateMachine child;
 * 
 * StepFunctionsStartExecution task = StepFunctionsStartExecution.Builder.create(this, "ChildTask")
 *         .stateMachine(child)
 *         .associateWithParent(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will add the payload <code>AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID.$: $$.Execution.Id</code> to the
 * <code>input</code>property for you, which will pass the execution ID from the context object to the
 * execution input. It requires <code>input</code> to be an object or not be set at all.
 * <p>
 * <h3>Invoke Activity</h3>
 * <p>
 * You can invoke a <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-activities.html">Step Functions Activity</a> which enables you to have
 * a task in your state machine where the work is performed by a <em>worker</em> that can
 * be hosted on Amazon EC2, Amazon ECS, AWS Lambda, basically anywhere. Activities
 * are a way to associate code running somewhere (known as an activity worker) with
 * a specific task in a state machine.
 * <p>
 * When Step Functions reaches an activity task state, the workflow waits for an
 * activity worker to poll for a task. An activity worker polls Step Functions by
 * using GetActivityTask, and sending the ARN for the related activity.
 * <p>
 * After the activity worker completes its work, it can provide a report of its
 * success or failure by using <code>SendTaskSuccess</code> or <code>SendTaskFailure</code>. These two
 * calls use the taskToken provided by GetActivityTask to associate the result
 * with that task.
 * <p>
 * The following example creates an activity and creates a task that invokes the activity.
 * <p>
 * <blockquote><pre>
 * Activity submitJobActivity = new Activity(this, "SubmitJob");
 * 
 * StepFunctionsInvokeActivity.Builder.create(this, "Submit Job")
 *         .activity(submitJobActivity)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-parameters">Parameters</a> field to create a collection of key-value pairs that are passed as input.
 * The values of each can either be static values that you include in your state machine definition, or selected from either the input or the context object with a path.
 * <p>
 * <blockquote><pre>
 * Activity submitJobActivity = new Activity(this, "SubmitJob");
 * 
 * StepFunctionsInvokeActivity.Builder.create(this, "Submit Job")
 *         .activity(submitJobActivity)
 *         .parameters(Map.of(
 *                 "comment", "Selecting what I care about.",
 *                 "MyDetails", Map.of(
 *                         "size", JsonPath.stringAt("$.product.details.size"),
 *                         "exists", JsonPath.stringAt("$.product.availability"),
 *                         "StaticValue", "foo")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>SQS</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-sqs.html">Amazon SQS</a>
 * <p>
 * <h3>Send Message</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessage.html"><code>SendMessage</code></a> API from a <code>Task</code> state
 * to send a message to an SQS queue.
 * <p>
 * <blockquote><pre>
 * Queue queue = new Queue(this, "Queue");
 * 
 * // Use a field from the execution data as message.
 * SqsSendMessage task1 = SqsSendMessage.Builder.create(this, "Send1")
 *         .queue(queue)
 *         .messageBody(TaskInput.fromJsonPathAt("$.message"))
 *         .build();
 * 
 * // Combine a field from the execution data with
 * // a literal object.
 * SqsSendMessage task2 = SqsSendMessage.Builder.create(this, "Send2")
 *         .queue(queue)
 *         .messageBody(TaskInput.fromObject(Map.of(
 *                 "field1", "somedata",
 *                 "field2", JsonPath.stringAt("$.field2"))))
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.stepfunctions.tasks;
