/**
 * <h1>CDK Construct library for higher-level ECS Constructs</h1>
 * <p>
 * This library provides higher-level Amazon ECS constructs which follow common architectural patterns. It contains:
 * <p>
 * <ul>
 * <li>Application Load Balanced Services</li>
 * <li>Network Load Balanced Services</li>
 * <li>Queue Processing Services</li>
 * <li>Scheduled Tasks (cron jobs)</li>
 * <li>Additional Examples</li>
 * </ul>
 * <p>
 * <h2>Application Load Balanced Services</h2>
 * <p>
 * To define an Amazon ECS service that is behind an application load balancer, instantiate one of the following:
 * <p>
 * <ul>
 * <li><code>ApplicationLoadBalancedEc2Service</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedEc2Service loadBalancedEcsService = ApplicationLoadBalancedEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("test"))
 *                 .environment(Map.of(
 *                         "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                         "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *                 .command(List.of("command"))
 *                 .entryPoint(List.of("entry", "point"))
 *                 .build())
 *         .desiredCount(2)
 *         .minHealthyPercent(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>ApplicationLoadBalancedFargateService</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .command(List.of("command"))
 *                 .entryPoint(List.of("entry", "point"))
 *                 .build())
 *         .containerCpu(256)
 *         .containerMemoryLimitMiB(512)
 *         .minHealthyPercent(100)
 *         .build();
 * 
 * loadBalancedFargateService.targetGroup.configureHealthCheck(HealthCheck.builder()
 *         .path("/custom-health-path")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Instead of providing a cluster you can specify a VPC and CDK will create a new ECS cluster.
 * If you deploy multiple services CDK will only create one cluster per VPC.
 * <p>
 * You can omit <code>cluster</code> and <code>vpc</code> to let CDK create a new VPC with two AZs and create a cluster inside this VPC.
 * <p>
 * You can customize the health check for your target group; otherwise it defaults to <code>HTTP</code> over port <code>80</code> hitting path <code>/</code>.
 * <p>
 * You can customize the health check configuration of the container via the <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ecs.HealthCheck.html"><code>healthCheck</code></a> property; otherwise it defaults to the health check configuration from the container.
 * <p>
 * Fargate services will use the <code>LATEST</code> platform version by default, but you can override by providing a value for the <code>platformVersion</code> property in the constructor.
 * <p>
 * Fargate services use the default VPC Security Group unless one or more are provided using the <code>securityGroups</code> property in the constructor.
 * <p>
 * By setting <code>redirectHTTP</code> to true, CDK will automatically create a listener on port 80 that redirects HTTP traffic to the HTTPS port.
 * <p>
 * If you specify the option <code>recordType</code> you can decide if you want the construct to use CNAME or Route53-Aliases as record sets.
 * <p>
 * To set the minimum number of CPU units to reserve for the container, you can use the <code>containerCpu</code> property.
 * <p>
 * To set the amount of memory (in MiB) to provide to the container, you can use the <code>containerMemoryLimitMiB</code> property.
 * <p>
 * If you need to encrypt the traffic between the load balancer and the ECS tasks, you can set the <code>targetProtocol</code> to <code>HTTPS</code>.
 * <p>
 * Additionally, if more than one application target group are needed, instantiate one of the following:
 * <p>
 * <ul>
 * <li><code>ApplicationMultipleTargetGroupsEc2Service</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // One application load balancer with one listener and two target groups.
 * Cluster cluster;
 * 
 * ApplicationMultipleTargetGroupsEc2Service loadBalancedEc2Service = ApplicationMultipleTargetGroupsEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(256)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .targetGroups(List.of(ApplicationTargetProps.builder()
 *                 .containerPort(80)
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(90)
 *                 .pathPattern("a/b/c")
 *                 .priority(10)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>ApplicationMultipleTargetGroupsFargateService</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // One application load balancer with one listener and two target groups.
 * Cluster cluster;
 * 
 * ApplicationMultipleTargetGroupsFargateService loadBalancedFargateService = ApplicationMultipleTargetGroupsFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .targetGroups(List.of(ApplicationTargetProps.builder()
 *                 .containerPort(80)
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(90)
 *                 .pathPattern("a/b/c")
 *                 .priority(10)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Network Load Balanced Services</h2>
 * <p>
 * To define an Amazon ECS service that is behind a network load balancer, instantiate one of the following:
 * <p>
 * <ul>
 * <li><code>NetworkLoadBalancedEc2Service</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * NetworkLoadBalancedEc2Service loadBalancedEcsService = NetworkLoadBalancedEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("test"))
 *                 .environment(Map.of(
 *                         "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                         "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *                 .build())
 *         .desiredCount(2)
 *         .minHealthyPercent(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>NetworkLoadBalancedFargateService</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * NetworkLoadBalancedFargateService loadBalancedFargateService = NetworkLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .cpu(512)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The CDK will create a new Amazon ECS cluster if you specify a VPC and omit <code>cluster</code>. If you deploy multiple services the CDK will only create one cluster per VPC.
 * <p>
 * If <code>cluster</code> and <code>vpc</code> are omitted, the CDK creates a new VPC with subnets in two Availability Zones and a cluster within this VPC.
 * <p>
 * If you specify the option <code>recordType</code> you can decide if you want the construct to use CNAME or Route53-Aliases as record sets.
 * <p>
 * Additionally, if more than one network target group is needed, instantiate one of the following:
 * <p>
 * <ul>
 * <li>NetworkMultipleTargetGroupsEc2Service</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Two network load balancers, each with their own listener and target group.
 * Cluster cluster;
 * 
 * NetworkMultipleTargetGroupsEc2Service loadBalancedEc2Service = NetworkMultipleTargetGroupsEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(256)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .loadBalancers(List.of(NetworkLoadBalancerProps.builder()
 *                 .name("lb1")
 *                 .listeners(List.of(NetworkListenerProps.builder()
 *                         .name("listener1")
 *                         .build()))
 *                 .build(), NetworkLoadBalancerProps.builder()
 *                 .name("lb2")
 *                 .listeners(List.of(NetworkListenerProps.builder()
 *                         .name("listener2")
 *                         .build()))
 *                 .build()))
 *         .targetGroups(List.of(NetworkTargetProps.builder()
 *                 .containerPort(80)
 *                 .listener("listener1")
 *                 .build(), NetworkTargetProps.builder()
 *                 .containerPort(90)
 *                 .listener("listener2")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li>NetworkMultipleTargetGroupsFargateService</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Two network load balancers, each with their own listener and target group.
 * Cluster cluster;
 * 
 * NetworkMultipleTargetGroupsFargateService loadBalancedFargateService = NetworkMultipleTargetGroupsFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .loadBalancers(List.of(NetworkLoadBalancerProps.builder()
 *                 .name("lb1")
 *                 .listeners(List.of(NetworkListenerProps.builder()
 *                         .name("listener1")
 *                         .build()))
 *                 .build(), NetworkLoadBalancerProps.builder()
 *                 .name("lb2")
 *                 .listeners(List.of(NetworkListenerProps.builder()
 *                         .name("listener2")
 *                         .build()))
 *                 .build()))
 *         .targetGroups(List.of(NetworkTargetProps.builder()
 *                 .containerPort(80)
 *                 .listener("listener1")
 *                 .build(), NetworkTargetProps.builder()
 *                 .containerPort(90)
 *                 .listener("listener2")
 *                 .build()))
 *         .minHealthyPercent(100)
 *         .maxHealthyPercent(200)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Queue Processing Services</h2>
 * <p>
 * To define a service that creates a queue and reads from that queue, instantiate one of the following:
 * <p>
 * <ul>
 * <li><code>QueueProcessingEc2Service</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * QueueProcessingEc2Service queueProcessingEc2Service = QueueProcessingEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .command(List.of("-c", "4", "amazon.com"))
 *         .enableLogging(false)
 *         .desiredTaskCount(2)
 *         .environment(Map.of(
 *                 "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                 "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *         .maxScalingCapacity(5)
 *         .containerName("test")
 *         .minHealthyPercent(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>QueueProcessingFargateService</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .command(List.of("-c", "4", "amazon.com"))
 *         .enableLogging(false)
 *         .desiredTaskCount(2)
 *         .environment(Map.of(
 *                 "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                 "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *         .maxScalingCapacity(5)
 *         .containerName("test")
 *         .minHealthyPercent(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * when queue not provided by user, CDK will create a primary queue and a dead letter queue with default redrive policy and attach permission to the task to be able to access the primary queue.
 * <p>
 * NOTE: <code>QueueProcessingFargateService</code> adds a CPU Based scaling strategy by default. You can turn this off by setting <code>disableCpuBasedScaling: true</code>.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .command(List.of("-c", "4", "amazon.com"))
 *         .enableLogging(false)
 *         .desiredTaskCount(2)
 *         .environment(Map.of(
 *                 "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                 "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *         .maxScalingCapacity(5)
 *         .containerName("test")
 *         .minHealthyPercent(100)
 *         .disableCpuBasedScaling(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To specify a custom target CPU utilization percentage for the scaling strategy use the  <code>cpuTargetUtilizationPercent</code> property:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .command(List.of("-c", "4", "amazon.com"))
 *         .enableLogging(false)
 *         .desiredTaskCount(2)
 *         .environment(Map.of())
 *         .maxScalingCapacity(5)
 *         .containerName("test")
 *         .minHealthyPercent(100)
 *         .cpuTargetUtilizationPercent(90)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Scheduled Tasks</h2>
 * <p>
 * To define a task that runs periodically, there are 2 options:
 * <p>
 * <ul>
 * <li><code>ScheduledEc2Task</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Instantiate an Amazon EC2 Task to run at a scheduled interval
 * Cluster cluster;
 * 
 * ScheduledEc2Task ecsScheduledTask = ScheduledEc2Task.Builder.create(this, "ScheduledTask")
 *         .cluster(cluster)
 *         .scheduledEc2TaskImageOptions(ScheduledEc2TaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .memoryLimitMiB(256)
 *                 .environment(Map.of("name", "TRIGGER", "value", "CloudWatch Events"))
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .enabled(true)
 *         .ruleName("sample-scheduled-task-rule")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>ScheduledFargateTask</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ScheduledFargateTask scheduledFargateTask = ScheduledFargateTask.Builder.create(this, "ScheduledFargateTask")
 *         .cluster(cluster)
 *         .scheduledFargateTaskImageOptions(ScheduledFargateTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .memoryLimitMiB(512)
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .platformVersion(FargatePlatformVersion.LATEST)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Additional Examples</h2>
 * <p>
 * In addition to using the constructs, users can also add logic to customize these constructs:
 * <p>
 * <h3>Configure HTTPS on an ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.route53.HostedZone;
 * import software.amazon.awscdk.services.certificatemanager.Certificate;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.SslPolicy;
 * 
 * Vpc vpc;
 * Cluster cluster;
 * 
 * 
 * IHostedZone domainZone = HostedZone.fromLookup(this, "Zone", HostedZoneProviderProps.builder().domainName("example.com").build());
 * ICertificate certificate = Certificate.fromCertificateArn(this, "Cert", "arn:aws:acm:us-east-1:123456:certificate/abcdefg");
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .cluster(cluster)
 *         .minHealthyPercent(100)
 *         .certificate(certificate)
 *         .sslPolicy(SslPolicy.RECOMMENDED)
 *         .domainName("api.example.com")
 *         .domainZone(domainZone)
 *         .redirectHTTP(true)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set capacityProviderStrategies for ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.enableFargateCapacityProviders();
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE_SPOT")
 *                 .weight(2)
 *                 .base(0)
 *                 .build(), CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE")
 *                 .weight(1)
 *                 .base(1)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Add Schedule-Based Auto-Scaling to an ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .build();
 * 
 * ScalableTaskCount scalableTarget = loadBalancedFargateService.service.autoScaleTaskCount(EnableScalingProps.builder()
 *         .minCapacity(5)
 *         .maxCapacity(20)
 *         .build());
 * 
 * scalableTarget.scaleOnSchedule("DaytimeScaleDown", ScalingSchedule.builder()
 *         .schedule(Schedule.cron(CronOptions.builder().hour("8").minute("0").build()))
 *         .minCapacity(1)
 *         .build());
 * 
 * scalableTarget.scaleOnSchedule("EveningRushScaleUp", ScalingSchedule.builder()
 *         .schedule(Schedule.cron(CronOptions.builder().hour("20").minute("0").build()))
 *         .minCapacity(10)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Add Metric-Based Auto-Scaling to an ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .build();
 * 
 * ScalableTaskCount scalableTarget = loadBalancedFargateService.service.autoScaleTaskCount(EnableScalingProps.builder()
 *         .minCapacity(1)
 *         .maxCapacity(20)
 *         .build());
 * 
 * scalableTarget.scaleOnCpuUtilization("CpuScaling", CpuUtilizationScalingProps.builder()
 *         .targetUtilizationPercent(50)
 *         .build());
 * 
 * scalableTarget.scaleOnMemoryUtilization("MemoryScaling", MemoryUtilizationScalingProps.builder()
 *         .targetUtilizationPercent(50)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Change the default Deployment Controller</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .deploymentController(DeploymentController.builder()
 *                 .type(DeploymentControllerType.CODE_DEPLOY)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deployment circuit breaker and rollback</h3>
 * <p>
 * Amazon ECS <a href="https://aws.amazon.com/tw/blogs/containers/announcing-amazon-ecs-deployment-circuit-breaker/">deployment circuit breaker</a>
 * automatically rolls back unhealthy service deployments without the need for manual intervention. Use <code>circuitBreaker</code> to enable
 * deployment circuit breaker and optionally enable <code>rollback</code> for automatic rollback. See <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Using the deployment circuit breaker</a>
 * for more details.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService service = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .circuitBreaker(DeploymentCircuitBreaker.builder().rollback(true).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set deployment configuration on QueueProcessingService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .command(List.of("-c", "4", "amazon.com"))
 *         .enableLogging(false)
 *         .desiredTaskCount(2)
 *         .environment(Map.of())
 *         .maxScalingCapacity(5)
 *         .maxHealthyPercent(200)
 *         .minHealthyPercent(66)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set taskSubnets and securityGroups for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup securityGroup;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .minHealthyPercent(100)
 *         .securityGroups(List.of(securityGroup))
 *         .taskSubnets(SubnetSelection.builder().subnetType(SubnetType.PRIVATE_ISOLATED).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Define tasks with public IPs for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .minHealthyPercent(100)
 *         .assignPublicIp(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Define tasks with custom queue parameters for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .minHealthyPercent(100)
 *         .maxReceiveCount(42)
 *         .retentionPeriod(Duration.days(7))
 *         .visibilityTimeout(Duration.minutes(5))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set cooldown for QueueProcessingFargateService</h3>
 * <p>
 * The cooldown period is the amount of time to wait for a previous scaling activity to take effect.
 * To specify something other than the default cooldown period of 300 seconds, use the <code>cooldown</code> parameter:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .minHealthyPercent(100)
 *         .assignPublicIp(true)
 *         .cooldown(Duration.seconds(500))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set capacityProviderStrategies for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * cluster.enableFargateCapacityProviders();
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .minHealthyPercent(100)
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE_SPOT")
 *                 .weight(2)
 *                 .build(), CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE")
 *                 .weight(1)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set a custom container-level Healthcheck for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup securityGroup;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .minHealthyPercent(100)
 *         .healthCheck(HealthCheck.builder()
 *                 .command(List.of("CMD-SHELL", "curl -f http://localhost/ || exit 1"))
 *                 // the properties below are optional
 *                 .interval(Duration.minutes(30))
 *                 .retries(123)
 *                 .startPeriod(Duration.minutes(30))
 *                 .timeout(Duration.minutes(30))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set capacityProviderStrategies for QueueProcessingEc2Service</h3>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.autoscaling.*;
 * 
 * 
 * Vpc vpc = Vpc.Builder.create(this, "Vpc").maxAzs(1).build();
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build();
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "asg")
 *         .vpc(vpc)
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.MICRO))
 *         .machineImage(EcsOptimizedImage.amazonLinux2())
 *         .build();
 * AsgCapacityProvider capacityProvider = AsgCapacityProvider.Builder.create(this, "provider")
 *         .autoScalingGroup(autoScalingGroup)
 *         .build();
 * cluster.addAsgCapacityProvider(capacityProvider);
 * 
 * QueueProcessingEc2Service queueProcessingEc2Service = QueueProcessingEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .minHealthyPercent(100)
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider(capacityProvider.getCapacityProviderName())
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Select specific vpc subnets for ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .taskSubnets(SubnetSelection.builder()
 *                 .subnets(List.of(Subnet.fromSubnetId(this, "subnet", "VpcISOLATEDSubnet1Subnet80F07FA0")))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Select idleTimeout for ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .idleTimeout(Duration.seconds(120))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Select idleTimeout for ApplicationMultipleTargetGroupsFargateService</h3>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.Certificate;
 * import software.amazon.awscdk.services.ec2.InstanceType;
 * import software.amazon.awscdk.services.ecs.Cluster;
 * import software.amazon.awscdk.services.ecs.ContainerImage;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.ApplicationProtocol;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.SslPolicy;
 * import software.amazon.awscdk.services.route53.PublicHostedZone;
 * 
 * Vpc vpc = Vpc.Builder.create(this, "Vpc").maxAzs(1).build();
 * ApplicationMultipleTargetGroupsFargateService loadBalancedFargateService = ApplicationMultipleTargetGroupsFargateService.Builder.create(this, "myService")
 *         .cluster(Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build())
 *         .memoryLimitMiB(256)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .enableExecuteCommand(true)
 *         .loadBalancers(List.of(ApplicationLoadBalancerProps.builder()
 *                 .name("lb")
 *                 .idleTimeout(Duration.seconds(400))
 *                 .domainName("api.example.com")
 *                 .domainZone(PublicHostedZone.Builder.create(this, "HostedZone").zoneName("example.com").build())
 *                 .listeners(List.of(ApplicationListenerProps.builder()
 *                         .name("listener")
 *                         .protocol(ApplicationProtocol.HTTPS)
 *                         .certificate(Certificate.fromCertificateArn(this, "Cert", "helloworld"))
 *                         .sslPolicy(SslPolicy.TLS12_EXT)
 *                         .build()))
 *                 .build(), ApplicationLoadBalancerProps.builder()
 *                 .name("lb2")
 *                 .idleTimeout(Duration.seconds(120))
 *                 .domainName("frontend.com")
 *                 .domainZone(PublicHostedZone.Builder.create(this, "HostedZone").zoneName("frontend.com").build())
 *                 .listeners(List.of(ApplicationListenerProps.builder()
 *                         .name("listener2")
 *                         .protocol(ApplicationProtocol.HTTPS)
 *                         .certificate(Certificate.fromCertificateArn(this, "Cert2", "helloworld"))
 *                         .sslPolicy(SslPolicy.TLS12_EXT)
 *                         .build()))
 *                 .build()))
 *         .targetGroups(List.of(ApplicationTargetProps.builder()
 *                 .containerPort(80)
 *                 .listener("listener")
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(90)
 *                 .pathPattern("a/b/c")
 *                 .priority(10)
 *                 .listener("listener")
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(443)
 *                 .listener("listener2")
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(80)
 *                 .pathPattern("a/b/c")
 *                 .priority(10)
 *                 .listener("listener2")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set health checks for ApplicationMultipleTargetGroupsFargateService</h3>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.Certificate;
 * import software.amazon.awscdk.services.ec2.InstanceType;
 * import software.amazon.awscdk.services.ecs.Cluster;
 * import software.amazon.awscdk.services.ecs.ContainerImage;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.ApplicationProtocol;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.Protocol;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.SslPolicy;
 * import software.amazon.awscdk.services.route53.PublicHostedZone;
 * 
 * Vpc vpc = Vpc.Builder.create(this, "Vpc").maxAzs(1).build();
 * 
 * ApplicationMultipleTargetGroupsFargateService loadBalancedFargateService = ApplicationMultipleTargetGroupsFargateService.Builder.create(this, "myService")
 *         .cluster(Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build())
 *         .memoryLimitMiB(256)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageProps.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .enableExecuteCommand(true)
 *         .loadBalancers(List.of(ApplicationLoadBalancerProps.builder()
 *                 .name("lb")
 *                 .idleTimeout(Duration.seconds(400))
 *                 .domainName("api.example.com")
 *                 .domainZone(PublicHostedZone.Builder.create(this, "HostedZone").zoneName("example.com").build())
 *                 .listeners(List.of(ApplicationListenerProps.builder()
 *                         .name("listener")
 *                         .protocol(ApplicationProtocol.HTTPS)
 *                         .certificate(Certificate.fromCertificateArn(this, "Cert", "helloworld"))
 *                         .sslPolicy(SslPolicy.TLS12_EXT)
 *                         .build()))
 *                 .build(), ApplicationLoadBalancerProps.builder()
 *                 .name("lb2")
 *                 .idleTimeout(Duration.seconds(120))
 *                 .domainName("frontend.com")
 *                 .domainZone(PublicHostedZone.Builder.create(this, "HostedZone").zoneName("frontend.com").build())
 *                 .listeners(List.of(ApplicationListenerProps.builder()
 *                         .name("listener2")
 *                         .protocol(ApplicationProtocol.HTTPS)
 *                         .certificate(Certificate.fromCertificateArn(this, "Cert2", "helloworld"))
 *                         .sslPolicy(SslPolicy.TLS12_EXT)
 *                         .build()))
 *                 .build()))
 *         .targetGroups(List.of(ApplicationTargetProps.builder()
 *                 .containerPort(80)
 *                 .listener("listener")
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(90)
 *                 .pathPattern("a/b/c")
 *                 .priority(10)
 *                 .listener("listener")
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(443)
 *                 .listener("listener2")
 *                 .build(), ApplicationTargetProps.builder()
 *                 .containerPort(80)
 *                 .pathPattern("a/b/c")
 *                 .priority(10)
 *                 .listener("listener2")
 *                 .build()))
 *         .build();
 * 
 * loadBalancedFargateService.targetGroups[0].configureHealthCheck(HealthCheck.builder()
 *         .port("8050")
 *         .protocol(Protocol.HTTP)
 *         .healthyThresholdCount(2)
 *         .unhealthyThresholdCount(2)
 *         .timeout(Duration.seconds(10))
 *         .interval(Duration.seconds(30))
 *         .healthyHttpCodes("200")
 *         .build());
 * 
 * loadBalancedFargateService.targetGroups[1].configureHealthCheck(HealthCheck.builder()
 *         .port("8050")
 *         .protocol(Protocol.HTTP)
 *         .healthyThresholdCount(2)
 *         .unhealthyThresholdCount(2)
 *         .timeout(Duration.seconds(10))
 *         .interval(Duration.seconds(30))
 *         .healthyHttpCodes("200")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Set runtimePlatform for ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService applicationLoadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .runtimePlatform(RuntimePlatform.builder()
 *                 .cpuArchitecture(CpuArchitecture.ARM64)
 *                 .operatingSystemFamily(OperatingSystemFamily.LINUX)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Customize Container Name for ScheduledFargateTask</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ScheduledFargateTask scheduledFargateTask = ScheduledFargateTask.Builder.create(this, "ScheduledFargateTask")
 *         .cluster(cluster)
 *         .scheduledFargateTaskImageOptions(ScheduledFargateTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .containerName("customContainerName")
 *                 .memoryLimitMiB(512)
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .platformVersion(FargatePlatformVersion.LATEST)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Customize Container Name for ScheduledEc2Task</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ScheduledEc2Task ecsScheduledTask = ScheduledEc2Task.Builder.create(this, "ScheduledTask")
 *         .cluster(cluster)
 *         .scheduledEc2TaskImageOptions(ScheduledEc2TaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .containerName("customContainerName")
 *                 .memoryLimitMiB(256)
 *                 .environment(Map.of("name", "TRIGGER", "value", "CloudWatch Events"))
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .enabled(true)
 *         .ruleName("sample-scheduled-task-rule")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set PlatformVersion for ScheduledFargateTask</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ScheduledFargateTask scheduledFargateTask = ScheduledFargateTask.Builder.create(this, "ScheduledFargateTask")
 *         .cluster(cluster)
 *         .scheduledFargateTaskImageOptions(ScheduledFargateTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .memoryLimitMiB(512)
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .platformVersion(FargatePlatformVersion.VERSION1_4)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set SecurityGroups for ScheduledFargateTask</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc = Vpc.Builder.create(this, "Vpc").maxAzs(1).build();
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build();
 * SecurityGroup securityGroup = SecurityGroup.Builder.create(this, "SG").vpc(vpc).build();
 * 
 * ScheduledFargateTask scheduledFargateTask = ScheduledFargateTask.Builder.create(this, "ScheduledFargateTask")
 *         .cluster(cluster)
 *         .scheduledFargateTaskImageOptions(ScheduledFargateTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .memoryLimitMiB(512)
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .securityGroups(List.of(securityGroup))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deploy application and metrics sidecar</h3>
 * <p>
 * The following is an example of deploying an application along with a metrics sidecar container that utilizes <code>dockerLabels</code> for discovery:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * Vpc vpc;
 * 
 * ApplicationLoadBalancedFargateService service = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .vpc(vpc)
 *         .desiredCount(1)
 *         .minHealthyPercent(100)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .dockerLabels(Map.of(
 *                         "application.label.one", "first_label",
 *                         "application.label.two", "second_label"))
 *                 .build())
 *         .build();
 * 
 * service.taskDefinition.addContainer("Sidecar", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example/metrics-sidecar"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Select specific load balancer name ApplicationLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .taskSubnets(SubnetSelection.builder()
 *                 .subnets(List.of(Subnet.fromSubnetId(this, "subnet", "VpcISOLATEDSubnet1Subnet80F07FA0")))
 *                 .build())
 *         .loadBalancerName("application-lb-name")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>ECS Exec</h3>
 * <p>
 * You can use ECS Exec to run commands in or get a shell to a container running on an Amazon EC2 instance or on
 * AWS Fargate. Enable ECS Exec, by setting <code>enableExecuteCommand</code> to <code>true</code>.
 * <p>
 * ECS Exec is supported by all Services i.e. <code>ApplicationLoadBalanced(Fargate|Ec2)Service</code>, <code>ApplicationMultipleTargetGroups(Fargate|Ec2)Service</code>, <code>NetworkLoadBalanced(Fargate|Ec2)Service</code>, <code>NetworkMultipleTargetGroups(Fargate|Ec2)Service</code>, <code>QueueProcessing(Fargate|Ec2)Service</code>. It is not supported for <code>ScheduledTask</code>s.
 * <p>
 * Read more about ECS Exec in the <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-exec.html">ECS Developer Guide</a>.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * ApplicationLoadBalancedFargateService loadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .desiredCount(1)
 *         .cpu(512)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .enableExecuteCommand(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Please note, ECS Exec leverages AWS Systems Manager (SSM). So as a prerequisite for the exec command
 * to work, you need to have the SSM plugin for the AWS CLI installed locally. For more information, see
 * <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-working-with-install-plugin.html">Install Session Manager plugin for AWS CLI</a>.
 * <p>
 * <h3>Propagate Tags from task definition for ScheduledFargateTask</h3>
 * <p>
 * For tasks that are defined by a Task Definition, tags applied to the definition will not be applied
 * to the running task by default. To get this behavior, set <code>propagateTags</code> to <code>ecs.PropagatedTagSource.TASK_DEFINITION</code> as
 * shown below:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.Tags;
 * 
 * 
 * Vpc vpc = Vpc.Builder.create(this, "Vpc").maxAzs(1).build();
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build();
 * FargateTaskDefinition taskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .build();
 * taskDefinition.addContainer("WebContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .build());
 * Tags.of(taskDefinition).add("my-tag", "my-tag-value");
 * ScheduledFargateTask scheduledFargateTask = ScheduledFargateTask.Builder.create(this, "ScheduledFargateTask")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .propagateTags(PropagatedTagSource.TASK_DEFINITION)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Pass a list of tags for ScheduledFargateTask</h3>
 * <p>
 * You can pass a list of tags to be applied to a Fargate task directly. These tags are in addition to any tags
 * that could be applied to the task definition and propagated using the <code>propagateTags</code> attribute.
 * <p>
 * <blockquote><pre>
 * Vpc vpc = Vpc.Builder.create(this, "Vpc").maxAzs(1).build();
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build();
 * ScheduledFargateTask scheduledFargateTask = ScheduledFargateTask.Builder.create(this, "ScheduledFargateTask")
 *         .cluster(cluster)
 *         .scheduledFargateTaskImageOptions(ScheduledFargateTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .memoryLimitMiB(512)
 *                 .build())
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .tags(List.of(Tag.builder()
 *                 .key("my-tag")
 *                 .value("my-tag-value")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Use custom ephemeral storage for ECS Fargate tasks</h3>
 * <p>
 * You can pass a custom ephemeral storage (21GiB - 200GiB) to ECS Fargate tasks on Fargate Platform Version 1.4.0 or later.
 * <p>
 * <blockquote><pre>
 * Vpc vpc = Vpc.Builder.create(this, "Vpc").maxAzs(2).restrictDefaultSecurityGroup(false).build();
 * Cluster cluster = Cluster.Builder.create(this, "FargateCluster").vpc(vpc).build();
 * 
 * ApplicationLoadBalancedFargateService applicationLoadBalancedFargateService = ApplicationLoadBalancedFargateService.Builder.create(this, "ALBFargateServiceWithCustomEphemeralStorage")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .cpu(512)
 *         .ephemeralStorageGiB(21)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .build();
 * 
 * NetworkLoadBalancedFargateService networkLoadBalancedFargateService = NetworkLoadBalancedFargateService.Builder.create(this, "NLBFargateServiceWithCustomEphemeralStorage")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         .cpu(512)
 *         .ephemeralStorageGiB(200)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set healthCheckGracePeriod for QueueProcessingFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * QueueProcessingFargateService queueProcessingFargateService = QueueProcessingFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .image(ContainerImage.fromRegistry("test"))
 *         .minHealthyPercent(100)
 *         .healthCheckGracePeriod(Duration.seconds(120))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set securityGroups for NetworkLoadBalancedFargateService</h3>
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup securityGroup;
 * 
 * NetworkLoadBalancedFargateService queueProcessingFargateService = NetworkLoadBalancedFargateService.Builder.create(this, "Service")
 *         .vpc(vpc)
 *         .memoryLimitMiB(512)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .securityGroups(List.of(securityGroup))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Set TLS for NetworkLoadBalancedFargateService / NetworkLoadBalancedEc2Service</h3>
 * <p>
 * To set up TLS listener in Network Load Balancer, you need to pass extactly one ACM certificate into the option <code>listenerCertificate</code>. The listener port and the target group port will also become 443 by default. You can override the listener's port with <code>listenerPort</code> and the target group's port with <code>taskImageOptions.containerPort</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.Certificate;
 * 
 * 
 * ICertificate certificate = Certificate.fromCertificateArn(this, "Cert", "arn:aws:acm:us-east-1:123456:certificate/abcdefg");
 * NetworkLoadBalancedFargateService loadBalancedFargateService = NetworkLoadBalancedFargateService.Builder.create(this, "Service")
 *         // The default value of listenerPort is 443 if you pass in listenerCertificate
 *         // It is configured to port 4443 here
 *         .listenerPort(4443)
 *         .listenerCertificate(certificate)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 // The default value of containerPort is 443 if you pass in listenerCertificate
 *                 // It is configured to port 8443 here
 *                 .containerPort(8443)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.certificatemanager.Certificate;
 * 
 * Cluster cluster;
 * 
 * ICertificate certificate = Certificate.fromCertificateArn(this, "Cert", "arn:aws:acm:us-east-1:123456:certificate/abcdefg");
 * NetworkLoadBalancedEc2Service loadBalancedEcsService = NetworkLoadBalancedEc2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .memoryLimitMiB(1024)
 *         // The default value of listenerPort is 443 if you pass in listenerCertificate
 *         // It is configured to port 4443 here
 *         .listenerPort(4443)
 *         .listenerCertificate(certificate)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("test"))
 *                 // The default value of containerPort is 443 if you pass in listenerCertificate
 *                 // It is configured to port 8443 here
 *                 .containerPort(8443)
 *                 .environment(Map.of(
 *                         "TEST_ENVIRONMENT_VARIABLE1", "test environment variable 1 value",
 *                         "TEST_ENVIRONMENT_VARIABLE2", "test environment variable 2 value"))
 *                 .build())
 *         .desiredCount(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Use dualstack Load Balancer</h3>
 * <p>
 * You can use dualstack IP address type for Application Load Balancer and Network Load Balancer.
 * <p>
 * To use dualstack IP address type, you must have associated IPv6 CIDR blocks with the VPC and subnets and set the <code>ipAddressType</code> to <code>IpAddressType.DUAL_STACK</code> when creating the load balancer.
 * <p>
 * <h3>Application Load Balancer</h3>
 * <p>
 * You can use dualstack Application Load Balancer for Fargate and EC2 services.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * 
 * // The VPC and subnet must have associated IPv6 CIDR blocks.
 * Vpc vpc = Vpc.Builder.create(this, "Vpc")
 *         .ipProtocol(IpProtocol.DUAL_STACK)
 *         .build();
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build();
 * 
 * ApplicationLoadBalancedFargateService service = ApplicationLoadBalancedFargateService.Builder.create(this, "myService")
 *         .cluster(cluster)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .ipAddressType(IpAddressType.DUAL_STACK)
 *         .build();
 * 
 * ApplicationLoadBalancedEc2Service applicationLoadBalancedEc2Service = ApplicationLoadBalancedEc2Service.Builder.create(this, "myService")
 *         .cluster(cluster)
 *         .taskImageOptions(ApplicationLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .ipAddressType(IpAddressType.DUAL_STACK)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Network Load Balancer</h3>
 * <p>
 * You can use dualstack Network Load Balancer for Fargate and EC2 services.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * 
 * // The VPC and subnet must have associated IPv6 CIDR blocks.
 * Vpc vpc = Vpc.Builder.create(this, "Vpc")
 *         .ipProtocol(IpProtocol.DUAL_STACK)
 *         .build();
 * Cluster cluster = Cluster.Builder.create(this, "EcsCluster").vpc(vpc).build();
 * 
 * NetworkLoadBalancedFargateService networkLoadbalancedFargateService = NetworkLoadBalancedFargateService.Builder.create(this, "NlbFargateService")
 *         .cluster(cluster)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .ipAddressType(IpAddressType.DUAL_STACK)
 *         .build();
 * 
 * NetworkLoadBalancedEc2Service networkLoadbalancedEc2Service = NetworkLoadBalancedEc2Service.Builder.create(this, "NlbEc2Service")
 *         .cluster(cluster)
 *         .taskImageOptions(NetworkLoadBalancedTaskImageOptions.builder()
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .build())
 *         .minHealthyPercent(100)
 *         .ipAddressType(IpAddressType.DUAL_STACK)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ecs.patterns;
