/**
 * <h1>CDK Construct Library for Amazon Simple Notification Service Subscriptions</h1>
 * <p>
 * This library provides constructs for adding subscriptions to an Amazon SNS topic.
 * Subscriptions can be added by calling the <code>.addSubscription(...)</code> method on the topic.
 * <p>
 * <h2>Subscriptions</h2>
 * <p>
 * Subscriptions can be added to the following endpoints:
 * <p>
 * <ul>
 * <li>HTTPS</li>
 * <li>Amazon SQS</li>
 * <li>AWS Lambda</li>
 * <li>Email</li>
 * <li>SMS</li>
 * <li>Amazon Data Firehose</li>
 * </ul>
 * <p>
 * Subscriptions to Amazon SQS, AWS Lambda and Amazon Data Firehose can be added on topics across regions.
 * <p>
 * Create an Amazon SNS Topic to add subscriptions.
 * <p>
 * <blockquote><pre>
 * Topic myTopic = new Topic(this, "MyTopic");
 * </pre></blockquote>
 * <p>
 * <h3>HTTPS</h3>
 * <p>
 * Add an HTTP or HTTPS Subscription to your topic:
 * <p>
 * <blockquote><pre>
 * Topic myTopic = new Topic(this, "MyTopic");
 * 
 * myTopic.addSubscription(new UrlSubscription("https://foobar.com/"));
 * </pre></blockquote>
 * <p>
 * The URL being subscribed can also be <a href="https://docs.aws.amazon.com/cdk/latest/guide/tokens.html">tokens</a>, that resolve
 * to a URL during deployment. A typical use case is when the URL is passed in as a <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html">CloudFormation
 * parameter</a>. The
 * following code defines a CloudFormation parameter and uses it in a URL subscription.
 * <p>
 * <blockquote><pre>
 * Topic myTopic = new Topic(this, "MyTopic");
 * CfnParameter url = new CfnParameter(this, "url-param");
 * 
 * myTopic.addSubscription(new UrlSubscription(url.getValueAsString()));
 * </pre></blockquote>
 * <p>
 * The <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-message-delivery-retries.html">delivery policy</a> can also be set like so:
 * <p>
 * <blockquote><pre>
 * Topic myTopic = new Topic(this, "MyTopic");
 * 
 * myTopic.addSubscription(
 * UrlSubscription.Builder.create("https://foobar.com/")
 *         .deliveryPolicy(DeliveryPolicy.builder()
 *                 .healthyRetryPolicy(HealthyRetryPolicy.builder()
 *                         .minDelayTarget(Duration.seconds(5))
 *                         .maxDelayTarget(Duration.seconds(10))
 *                         .numRetries(6)
 *                         .backoffFunction(BackoffFunction.EXPONENTIAL)
 *                         .build())
 *                 .throttlePolicy(ThrottlePolicy.builder()
 *                         .maxReceivesPerSecond(10)
 *                         .build())
 *                 .requestPolicy(RequestPolicy.builder()
 *                         .headerContentType("application/json")
 *                         .build())
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Amazon SQS</h3>
 * <p>
 * Subscribe a queue to your topic:
 * <p>
 * <blockquote><pre>
 * Queue myQueue = new Queue(this, "MyQueue");
 * Topic myTopic = new Topic(this, "MyTopic");
 * 
 * myTopic.addSubscription(new SqsSubscription(myQueue));
 * </pre></blockquote>
 * <p>
 * KMS key permissions will automatically be granted to SNS when a subscription is made to
 * an encrypted queue.
 * <p>
 * Note that subscriptions of queues in different accounts need to be manually confirmed by
 * reading the initial message from the queue and visiting the link found in it.
 * <p>
 * <h3>AWS Lambda</h3>
 * <p>
 * Subscribe an AWS Lambda function to your topic:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * Function myFunction;
 * 
 * 
 * Topic myTopic = new Topic(this, "myTopic");
 * myTopic.addSubscription(new LambdaSubscription(myFunction));
 * </pre></blockquote>
 * <p>
 * <h3>Email</h3>
 * <p>
 * Subscribe an email address to your topic:
 * <p>
 * <blockquote><pre>
 * Topic myTopic = new Topic(this, "MyTopic");
 * myTopic.addSubscription(new EmailSubscription("foo&#64;bar.com"));
 * </pre></blockquote>
 * <p>
 * The email being subscribed can also be <a href="https://docs.aws.amazon.com/cdk/latest/guide/tokens.html">tokens</a>, that resolve
 * to an email address during deployment. A typical use case is when the email address is passed in as a <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html">CloudFormation
 * parameter</a>. The
 * following code defines a CloudFormation parameter and uses it in an email subscription.
 * <p>
 * <blockquote><pre>
 * Topic myTopic = new Topic(this, "Topic");
 * CfnParameter emailAddress = new CfnParameter(this, "email-param");
 * 
 * myTopic.addSubscription(new EmailSubscription(emailAddress.getValueAsString()));
 * </pre></blockquote>
 * <p>
 * Note that email subscriptions require confirmation by visiting the link sent to the
 * email address.
 * <p>
 * <h3>SMS</h3>
 * <p>
 * Subscribe an sms number to your topic:
 * <p>
 * <blockquote><pre>
 * Topic myTopic = new Topic(this, "Topic");
 * 
 * myTopic.addSubscription(new SmsSubscription("+15551231234"));
 * </pre></blockquote>
 * <p>
 * The number being subscribed can also be <a href="https://docs.aws.amazon.com/cdk/latest/guide/tokens.html">tokens</a>, that resolve
 * to a number during deployment. A typical use case is when the number is passed in as a <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html">CloudFormation
 * parameter</a>. The
 * following code defines a CloudFormation parameter and uses it in an sms subscription.
 * <p>
 * <blockquote><pre>
 * Topic myTopic = new Topic(this, "Topic");
 * CfnParameter smsNumber = new CfnParameter(this, "sms-param");
 * 
 * myTopic.addSubscription(new SmsSubscription(smsNumber.getValueAsString()));
 * </pre></blockquote>
 * <p>
 * <h3>Amazon Data Firehose</h3>
 * <p>
 * Subscribe an Amazon Data Firehose delivery stream to your topic:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kinesisfirehose.*;
 * DeliveryStream stream;
 * 
 * 
 * Topic myTopic = new Topic(this, "Topic");
 * 
 * myTopic.addSubscription(new FirehoseSubscription(stream));
 * </pre></blockquote>
 * <p>
 * To remove any Amazon SNS metadata from published messages, specify <code>rawMessageDelivery</code> to true.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kinesisfirehose.*;
 * DeliveryStream stream;
 * 
 * 
 * Topic myTopic = new Topic(this, "Topic");
 * 
 * myTopic.addSubscription(FirehoseSubscription.Builder.create(stream)
 *         .rawMessageDelivery(true)
 *         .build());
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.sns.subscriptions;
