/**
 * <h1>AWS Backup Construct Library</h1>
 * <p>
 * AWS Backup is a fully managed backup service that makes it easy to centralize and automate the
 * backup of data across AWS services in the cloud and on premises. Using AWS Backup, you can
 * configure backup policies and monitor backup activity for your AWS resources in one place.
 * <p>
 * <h2>Backup plan and selection</h2>
 * <p>
 * In AWS Backup, a <em>backup plan</em> is a policy expression that defines when and how you want to back up
 * your AWS resources, such as Amazon DynamoDB tables or Amazon Elastic File System (Amazon EFS) file
 * systems. You can assign resources to backup plans, and AWS Backup automatically backs up and retains
 * backups for those resources according to the backup plan. You can create multiple backup plans if you
 * have workloads with different backup requirements.
 * <p>
 * This module provides ready-made backup plans (similar to the console experience):
 * <p>
 * <blockquote><pre>
 * // Daily, weekly and monthly with 5 year retention
 * BackupPlan plan = BackupPlan.dailyWeeklyMonthly5YearRetention(this, "Plan");
 * </pre></blockquote>
 * <p>
 * Assigning resources to a plan can be done with <code>addSelection()</code>:
 * <p>
 * <blockquote><pre>
 * BackupPlan plan;
 * Vpc vpc;
 * 
 * ITable myTable = Table.fromTableName(this, "Table", "myTableName");
 * DatabaseInstance myDatabaseInstance = DatabaseInstance.Builder.create(this, "DatabaseInstance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_26).build()))
 *         .vpc(vpc)
 *         .build();
 * DatabaseCluster myDatabaseCluster = DatabaseCluster.Builder.create(this, "DatabaseCluster")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_2_08_1).build()))
 *         .credentials(Credentials.fromGeneratedSecret("clusteradmin"))
 *         .instanceProps(InstanceProps.builder()
 *                 .vpc(vpc)
 *                 .build())
 *         .build();
 * ServerlessCluster myServerlessCluster = ServerlessCluster.Builder.create(this, "ServerlessCluster")
 *         .engine(DatabaseClusterEngine.AURORA_POSTGRESQL)
 *         .parameterGroup(ParameterGroup.fromParameterGroupName(this, "ParameterGroup", "default.aurora-postgresql11"))
 *         .vpc(vpc)
 *         .build();
 * Construct myCoolConstruct = new Construct(this, "MyCoolConstruct");
 * 
 * plan.addSelection("Selection", BackupSelectionOptions.builder()
 *         .resources(List.of(BackupResource.fromDynamoDbTable(myTable), BackupResource.fromRdsDatabaseInstance(myDatabaseInstance), BackupResource.fromRdsDatabaseCluster(myDatabaseCluster), BackupResource.fromRdsServerlessCluster(myServerlessCluster), BackupResource.fromTag("stage", "prod"), BackupResource.fromConstruct(myCoolConstruct)))
 *         .build());
 * </pre></blockquote>
 * <p>
 * If not specified, a new IAM role with a managed policy for backup will be
 * created for the selection. The <code>BackupSelection</code> implements <code>IGrantable</code>.
 * <p>
 * To disable the plan from assigning the default <code>AWSBackupServiceRolePolicyForBackup</code> backup policy use the <code>disableDefaultBackupPolicy</code> property.
 * <p>
 * This is useful if you want to avoid granting unnecessary permissions to the role.
 * <p>
 * <blockquote><pre>
 * BackupPlan plan;
 * 
 * 
 * Role role = Role.Builder.create(this, "BackupRole")
 *         .assumedBy(new ServicePrincipal("backup.amazonaws.com"))
 *         .build();
 * // Assign S3-specific backup policy
 * role.addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName("AWSBackupServiceRolePolicyForS3Backup"));
 * 
 * plan.addSelection("Selection", BackupSelectionOptions.builder()
 *         .resources(List.of(BackupResource.fromTag("stage", "prod")))
 *         .role(role)
 *         .disableDefaultBackupPolicy(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To add rules to a plan, use <code>addRule()</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.TimeZone;
 * BackupPlan plan;
 * 
 * plan.addRule(BackupPlanRule.Builder.create()
 *         .completionWindow(Duration.hours(2))
 *         .startWindow(Duration.hours(1))
 *         .scheduleExpression(Schedule.cron(CronOptions.builder() // Only cron expressions are supported
 *                 .day("15")
 *                 .hour("3")
 *                 .minute("30").build()))
 *         .scheduleExpressionTimezone(TimeZone.ETC_UTC)
 *         .moveToColdStorageAfter(Duration.days(30))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Continuous backup and point-in-time restores (PITR) can be configured.
 * Property <code>deleteAfter</code> defines the retention period for the backup. It is mandatory if PITR is enabled.
 * If no value is specified, the retention period is set to 35 days which is the maximum retention period supported by PITR.
 * Property <code>moveToColdStorageAfter</code> must not be specified because PITR does not support this option.
 * This example defines an AWS Backup rule with PITR and a retention period set to 14 days:
 * <p>
 * <blockquote><pre>
 * BackupPlan plan;
 * 
 * plan.addRule(BackupPlanRule.Builder.create()
 *         .enableContinuousBackup(true)
 *         .deleteAfter(Duration.days(14))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Rules can also specify to copy recovery points to another Backup Vault using <code>copyActions</code>. Copied recovery points can
 * optionally have <code>moveToColdStorageAfter</code> and <code>deleteAfter</code> configured.
 * <p>
 * <blockquote><pre>
 * BackupPlan plan;
 * BackupVault secondaryVault;
 * 
 * plan.addRule(BackupPlanRule.Builder.create()
 *         .copyActions(List.of(BackupPlanCopyActionProps.builder()
 *                 .destinationBackupVault(secondaryVault)
 *                 .moveToColdStorageAfter(Duration.days(30))
 *                 .deleteAfter(Duration.days(120))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can assign your own metadata to the resources that are associated with the rule when restored from backup using <code>recoveryPointTags</code>. Each tag is a key-value pair.
 * <p>
 * <blockquote><pre>
 * BackupPlan plan;
 * 
 * plan.addRule(BackupPlanRule.Builder.create()
 *         .recoveryPointTags(Map.of(
 *                 "key", "value"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Ready-made rules are also available:
 * <p>
 * <blockquote><pre>
 * BackupPlan plan;
 * 
 * plan.addRule(BackupPlanRule.daily());
 * plan.addRule(BackupPlanRule.weekly());
 * </pre></blockquote>
 * <p>
 * By default a new <a href="#Backup-vault">vault</a> is created when creating a plan.
 * It is also possible to specify a vault either at the plan level or at the
 * rule level.
 * <p>
 * <blockquote><pre>
 * IBackupVault myVault = BackupVault.fromBackupVaultName(this, "Vault1", "myVault");
 * IBackupVault otherVault = BackupVault.fromBackupVaultName(this, "Vault2", "otherVault");
 * 
 * BackupPlan plan = BackupPlan.daily35DayRetention(this, "Plan", myVault); // Use `myVault` for all plan rules
 * plan.addRule(BackupPlanRule.monthly1Year(otherVault));
 * </pre></blockquote>
 * <p>
 * You can <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/windows-backups.html">backup</a>
 * VSS-enabled Windows applications running on Amazon EC2 instances by setting the <code>windowsVss</code>
 * parameter to <code>true</code>. If the application has VSS writer registered with Windows VSS,
 * then AWS Backup creates a snapshot that will be consistent for that application.
 * <p>
 * <blockquote><pre>
 * BackupPlan plan = BackupPlan.Builder.create(this, "Plan")
 *         .windowsVss(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Backup vault</h2>
 * <p>
 * In AWS Backup, a <em>backup vault</em> is a container that you organize your backups in. You can use backup
 * vaults to set the AWS Key Management Service (AWS KMS) encryption key that is used to encrypt backups
 * in the backup vault and to control access to the backups in the backup vault. If you require different
 * encryption keys or access policies for different groups of backups, you can optionally create multiple
 * backup vaults.
 * <p>
 * <blockquote><pre>
 * IKey myKey = Key.fromKeyArn(this, "MyKey", "aaa");
 * ITopic myTopic = Topic.fromTopicArn(this, "MyTopic", "bbb");
 * 
 * BackupVault vault = BackupVault.Builder.create(this, "Vault")
 *         .encryptionKey(myKey) // Custom encryption key
 *         .notificationTopic(myTopic)
 *         .build();
 * </pre></blockquote>
 * <p>
 * A vault has a default <code>RemovalPolicy</code> set to <code>RETAIN</code>. Note that removing a vault
 * that contains recovery points will fail.
 * <p>
 * You can assign policies to backup vaults and the resources they contain. Assigning policies allows
 * you to do things like grant access to users to create backup plans and on-demand backups, but limit
 * their ability to delete recovery points after they're created.
 * <p>
 * Use the <code>accessPolicy</code> property to create a backup vault policy:
 * <p>
 * <blockquote><pre>
 * BackupVault vault = BackupVault.Builder.create(this, "Vault")
 *         .accessPolicy(PolicyDocument.Builder.create()
 *                 .statements(List.of(
 *                     PolicyStatement.Builder.create()
 *                             .effect(Effect.DENY)
 *                             .principals(List.of(new AnyPrincipal()))
 *                             .actions(List.of("backup:DeleteRecoveryPoint"))
 *                             .resources(List.of("*"))
 *                             .conditions(Map.of(
 *                                     "StringNotLike", Map.of(
 *                                             "aws:userId", List.of("user1", "user2"))))
 *                             .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternativately statements can be added to the vault policy using <code>addToAccessPolicy()</code>.
 * <p>
 * Use the <code>blockRecoveryPointDeletion</code> property or the <code>blockRecoveryPointDeletion()</code> method to add
 * a statement to the vault access policy that prevents recovery point deletions in your vault:
 * <p>
 * <blockquote><pre>
 * BackupVault backupVault;
 * BackupVault.Builder.create(this, "Vault")
 *         .blockRecoveryPointDeletion(true)
 *         .build();
 * backupVault.blockRecoveryPointDeletion();
 * </pre></blockquote>
 * <p>
 * By default access is not restricted.
 * <p>
 * Use the <code>lockConfiguration</code> property to enable <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/vault-lock.html">AWS Backup Vault Lock</a>:
 * <p>
 * <blockquote><pre>
 * BackupVault.Builder.create(this, "Vault")
 *         .lockConfiguration(LockConfiguration.builder()
 *                 .minRetention(Duration.days(30))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Importing existing backup vault</h2>
 * <p>
 * To import an existing backup vault into your CDK application, use the <code>BackupVault.fromBackupVaultArn</code> or <code>BackupVault.fromBackupVaultName</code>
 * static method. Here is an example of giving an IAM Role permission to start a backup job:
 * <p>
 * <blockquote><pre>
 * IBackupVault importedVault = BackupVault.fromBackupVaultName(this, "Vault", "myVaultName");
 * 
 * Role role = Role.Builder.create(this, "Access Role").assumedBy(new ServicePrincipal("lambda.amazonaws.com")).build();
 * 
 * importedVault.grant(role, "backup:StartBackupJob");
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.backup;
