/**
 * <h1>Amazon EventBridge Scheduler Construct Library</h1>
 * <p>
 * <a href="https://aws.amazon.com/blogs/compute/introducing-amazon-eventbridge-scheduler/">Amazon EventBridge Scheduler</a> is a feature from Amazon EventBridge
 * that allows you to create, run, and manage scheduled tasks at scale. With EventBridge Scheduler, you can schedule millions of one-time or recurring tasks across various AWS services without provisioning or managing underlying infrastructure.
 * <p>
 * <ol>
 * <li><strong>Schedule</strong>: A schedule is the main resource you create, configure, and manage using Amazon EventBridge Scheduler. Every schedule has a schedule expression that determines when, and with what frequency, the schedule runs. EventBridge Scheduler supports three types of schedules: rate, cron, and one-time schedules. When you create a schedule, you configure a target for the schedule to invoke.</li>
 * <li><strong>Target</strong>: A target is an API operation that EventBridge Scheduler calls on your behalf every time your schedule runs. EventBridge Scheduler
 * supports two types of targets: templated targets and universal targets. Templated targets invoke common API operations across a core groups of
 * services. For example, EventBridge Scheduler supports templated targets for invoking AWS Lambda Function or starting execution of Step Functions state
 * machine. For API operations that are not supported by templated targets you can use customizable universal targets. Universal targets support calling
 * more than 6,000 API operations across over 270 AWS services.</li>
 * <li><strong>Schedule Group</strong>: A schedule group is an Amazon EventBridge Scheduler resource that you use to organize your schedules. Your AWS account comes
 * with a default scheduler group. A new schedule will always be added to a scheduling group. If you do not provide a scheduling group to add to, it
 * will be added to the default scheduling group. You can create up to 500 schedule groups in your AWS account. Groups can be used to organize the
 * schedules logically, access the schedule metrics and manage permissions at group granularity (see details below). Schedule groups support tagging.
 * With EventBridge Scheduler, you apply tags to schedule groups, not to individual schedules to organize your resources.</li>
 * </ol>
 * <p>
 * <h2>Defining a schedule</h2>
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * LambdaInvoke target = LambdaInvoke.Builder.create(fn)
 *         .input(ScheduleTargetInput.fromObject(Map.of(
 *                 "payload", "useful")))
 *         .build();
 * 
 * Schedule schedule = Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(10)))
 *         .target(target)
 *         .description("This is a test schedule that invokes a lambda function every 10 minutes.")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Schedule Expressions</h3>
 * <p>
 * You can choose from three schedule types when configuring your schedule: rate-based, cron-based, and one-time schedules.
 * <p>
 * Both rate-based and cron-based schedules are recurring schedules. You can configure each recurring schedule type using a schedule expression.
 * <p>
 * For
 * cron-based schedules you can specify a time zone in which EventBridge Scheduler evaluates the expression.
 * <p>
 * <blockquote><pre>
 * LambdaInvoke target;
 * 
 * 
 * Schedule rateBasedSchedule = Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(10)))
 *         .target(target)
 *         .description("This is a test rate-based schedule")
 *         .build();
 * 
 * Schedule cronBasedSchedule = Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.cron(CronOptionsWithTimezone.builder()
 *                 .minute("0")
 *                 .hour("23")
 *                 .day("20")
 *                 .month("11")
 *                 .timeZone(TimeZone.AMERICA_NEW_YORK)
 *                 .build()))
 *         .target(target)
 *         .description("This is a test cron-based schedule that will run at 11:00 PM, on day 20 of the month, only in November in New York timezone")
 *         .build();
 * </pre></blockquote>
 * <p>
 * A one-time schedule is a schedule that invokes a target only once. You configure a one-time schedule by specifying the time of day, date,
 * and time zone in which EventBridge Scheduler evaluates the schedule.
 * <p>
 * <blockquote><pre>
 * LambdaInvoke target;
 * 
 * 
 * Schedule oneTimeSchedule = Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.at(
 *         new Date(2022, 10, 20, 19, 20, 23), TimeZone.AMERICA_NEW_YORK))
 *         .target(target)
 *         .description("This is a one-time schedule in New York timezone")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Grouping Schedules</h3>
 * <p>
 * Your AWS account comes with a default scheduler group. You can access the default group in CDK with:
 * <p>
 * <blockquote><pre>
 * IScheduleGroup defaultScheduleGroup = ScheduleGroup.fromDefaultScheduleGroup(this, "DefaultGroup");
 * </pre></blockquote>
 * <p>
 * You can add a schedule to a custom scheduling group managed by you. If a custom group is not specified, the schedule is added to the default group.
 * <p>
 * <blockquote><pre>
 * LambdaInvoke target;
 * 
 * 
 * ScheduleGroup scheduleGroup = ScheduleGroup.Builder.create(this, "ScheduleGroup")
 *         .scheduleGroupName("MyScheduleGroup")
 *         .build();
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(10)))
 *         .target(target)
 *         .scheduleGroup(scheduleGroup)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Disabling Schedules</h3>
 * <p>
 * By default, a schedule will be enabled. You can disable a schedule by setting the <code>enabled</code> property to false:
 * <p>
 * <blockquote><pre>
 * LambdaInvoke target;
 * 
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(10)))
 *         .target(target)
 *         .enabled(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Configuring a start and end time of the Schedule</h3>
 * <p>
 * If you choose a recurring schedule, you can set the start and end time of the Schedule by specifying the <code>start</code> and <code>end</code>.
 * <p>
 * <blockquote><pre>
 * LambdaInvoke target;
 * 
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.hours(12)))
 *         .target(target)
 *         .start(new Date("2023-01-01T00:00:00.000Z"))
 *         .end(new Date("2023-02-01T00:00:00.000Z"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Scheduler Targets</h2>
 * <p>
 * The <code>aws-cdk-lib/aws-scheduler-targets</code> module includes classes that implement the <code>IScheduleTarget</code> interface for
 * various AWS services. EventBridge Scheduler supports two types of targets:
 * <p>
 * <ol>
 * <li><strong>Templated targets</strong> which invoke common API
 * operations across a core groups of services, and</li>
 * <li><strong>Universal targets</strong> that you can customize to call more
 * than 6,000 operations across over 270 services.</li>
 * </ol>
 * <p>
 * A list of supported targets can be found at <code>aws-cdk-lib/aws-scheduler-targets</code>.
 * <p>
 * <h3>Input</h3>
 * <p>
 * Targets can be invoked with a custom input. The <code>ScheduleTargetInput</code> class supports free-form text input and JSON-formatted object input:
 * <p>
 * <blockquote><pre>
 * ScheduleTargetInput input = ScheduleTargetInput.fromObject(Map.of(
 *         "QueueName", "MyQueue"));
 * </pre></blockquote>
 * <p>
 * You can include context attributes in your target payload. EventBridge Scheduler will replace each keyword with
 * its respective value and deliver it to the target. See
 * <a href="https://docs.aws.amazon.com/scheduler/latest/UserGuide/managing-schedule-context-attributes.html">full list of supported context attributes</a>:
 * <p>
 * <ol>
 * <li><code>ContextAttribute.scheduleArn()</code> – The ARN of the schedule.</li>
 * <li><code>ContextAttribute.scheduledTime()</code> – The time you specified for the schedule to invoke its target, e.g., 2022-03-22T18:59:43Z.</li>
 * <li><code>ContextAttribute.executionId()</code> – The unique ID that EventBridge Scheduler assigns for each attempted invocation of a target, e.g., d32c5kddcf5bb8c3.</li>
 * <li><code>ContextAttribute.attemptNumber()</code> – A counter that identifies the attempt number for the current invocation, e.g., 1.</li>
 * </ol>
 * <p>
 * <blockquote><pre>
 * String text = String.format("Attempt number: %s", ContextAttribute.getAttemptNumber());
 * ScheduleTargetInput input = ScheduleTargetInput.fromText(text);
 * </pre></blockquote>
 * <p>
 * <h3>Specifying an execution role</h3>
 * <p>
 * An execution role is an IAM role that EventBridge Scheduler assumes in order to interact with other AWS services on your behalf.
 * <p>
 * The classes for templated schedule targets automatically create an IAM role with all the minimum necessary
 * permissions to interact with the templated target. If you wish you may specify your own IAM role, then the templated targets
 * will grant minimal required permissions. For example, the <code>LambdaInvoke</code> target will grant the
 * IAM execution role <code>lambda:InvokeFunction</code> permission to invoke the Lambda function.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("scheduler.amazonaws.com"))
 *         .build();
 * 
 * LambdaInvoke target = LambdaInvoke.Builder.create(fn)
 *         .input(ScheduleTargetInput.fromObject(Map.of(
 *                 "payload", "useful")))
 *         .role(role)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Specifying an encryption key</h3>
 * <p>
 * EventBridge Scheduler integrates with AWS Key Management Service (AWS KMS) to encrypt and decrypt your data using an AWS KMS key.
 * EventBridge Scheduler supports two types of KMS keys: AWS owned keys, and customer managed keys.
 * <p>
 * By default, all events in Scheduler are encrypted with a key that AWS owns and manages.
 * If you wish you can also provide a customer managed key to encrypt and decrypt the payload that your schedule delivers to its target using the <code>key</code> property.
 * Target classes will automatically add AWS <code>kms:Decrypt</code> permission to your schedule's execution role permissions policy.
 * <p>
 * <blockquote><pre>
 * Key key;
 * Function fn;
 * 
 * 
 * LambdaInvoke target = LambdaInvoke.Builder.create(fn)
 *         .input(ScheduleTargetInput.fromObject(Map.of(
 *                 "payload", "useful")))
 *         .build();
 * 
 * Schedule schedule = Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(10)))
 *         .target(target)
 *         .key(key)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/scheduler/latest/UserGuide/data-protection.html">Data protection in Amazon EventBridge Scheduler</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Configuring flexible time windows</h2>
 * <p>
 * You can configure flexible time windows by specifying the <code>timeWindow</code> property.
 * Flexible time windows are disabled by default.
 * <p>
 * <blockquote><pre>
 * LambdaInvoke target;
 * 
 * 
 * Schedule schedule = Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.hours(12)))
 *         .target(target)
 *         .timeWindow(TimeWindow.flexible(Duration.hours(10)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/scheduler/latest/UserGuide/managing-schedule-flexible-time-windows.html">Configuring flexible time windows</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Error-handling</h2>
 * <p>
 * You can configure how your schedule handles failures, when EventBridge Scheduler is unable to deliver an event
 * successfully to a target, by using two primary mechanisms: a retry policy, and a dead-letter queue (DLQ).
 * <p>
 * A retry policy determines the number of times EventBridge Scheduler must retry a failed event, and how long
 * to keep an unprocessed event.
 * <p>
 * A DLQ is a standard Amazon SQS queue EventBridge Scheduler uses to deliver failed events to, after the retry
 * policy has been exhausted. You can use a DLQ to troubleshoot issues with your schedule or its downstream target.
 * If you've configured a retry policy for your schedule, EventBridge Scheduler delivers the dead-letter event after
 * exhausting the maximum number of retries you set in the retry policy.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Queue dlq = Queue.Builder.create(this, "DLQ")
 *         .queueName("MyDLQ")
 *         .build();
 * 
 * LambdaInvoke target = LambdaInvoke.Builder.create(fn)
 *         .deadLetterQueue(dlq)
 *         .maxEventAge(Duration.minutes(1))
 *         .retryAttempts(3)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Monitoring</h2>
 * <p>
 * You can monitor Amazon EventBridge Scheduler using CloudWatch, which collects raw data
 * and processes it into readable, near real-time metrics. EventBridge Scheduler emits
 * a set of metrics for all schedules, and an additional set of metrics for schedules that
 * have an associated dead-letter queue (DLQ). If you configure a DLQ for your schedule,
 * EventBridge Scheduler publishes additional metrics when your schedule exhausts its retry policy.
 * <p>
 * <h3>Metrics for all schedules</h3>
 * <p>
 * The <code>Schedule</code> class provides static methods for accessing all schedules metrics with default configuration, such as <code>metricAllErrors</code> for viewing errors when executing targets.
 * <p>
 * <blockquote><pre>
 * Alarm.Builder.create(this, "SchedulesErrorAlarm")
 *         .metric(Schedule.metricAllErrors())
 *         .threshold(0)
 *         .evaluationPeriods(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Metrics for a Schedule Group</h3>
 * <p>
 * To view metrics for a specific group you can use methods on class <code>ScheduleGroup</code>:
 * <p>
 * <blockquote><pre>
 * ScheduleGroup scheduleGroup = ScheduleGroup.Builder.create(this, "ScheduleGroup")
 *         .scheduleGroupName("MyScheduleGroup")
 *         .build();
 * 
 * Alarm.Builder.create(this, "MyGroupErrorAlarm")
 *         .metric(scheduleGroup.metricTargetErrors())
 *         .evaluationPeriods(1)
 *         .threshold(0)
 *         .build();
 * 
 * // Or use default group
 * IScheduleGroup defaultScheduleGroup = ScheduleGroup.fromDefaultScheduleGroup(this, "DefaultScheduleGroup");
 * Alarm.Builder.create(this, "DefaultScheduleGroupErrorAlarm")
 *         .metric(defaultScheduleGroup.metricTargetErrors())
 *         .evaluationPeriods(1)
 *         .threshold(0)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See full list of metrics and their description at
 * <a href="https://docs.aws.amazon.com/scheduler/latest/UserGuide/monitoring-cloudwatch.html">Monitoring Using CloudWatch Metrics</a>
 * in the <em>AWS EventBridge Scheduler User Guide</em>.
 */
package software.amazon.awscdk.services.scheduler;
