/**
 * <h1>Amazon CloudWatch Logs Construct Library</h1>
 * <p>
 * This library supplies constructs for working with CloudWatch Logs.
 * <p>
 * <h2>Log Groups/Streams</h2>
 * <p>
 * The basic unit of CloudWatch is a <em>Log Group</em>. Every log group typically has the
 * same kind of data logged to it, in the same format. If there are multiple
 * applications or services logging into the Log Group, each of them creates a new
 * <em>Log Stream</em>.
 * <p>
 * Every log operation creates a "log event", which can consist of a simple string
 * or a single-line JSON object. JSON objects have the advantage that they afford
 * more filtering abilities (see below).
 * <p>
 * The only configurable attribute for log streams is the retention period, which
 * configures after how much time the events in the log stream expire and are
 * deleted.
 * <p>
 * The default retention period if not supplied is 2 years, but it can be set to
 * one of the values in the <code>RetentionDays</code> enum to configure a different
 * retention period (including infinite retention).
 * <p>
 * <blockquote><pre>
 * // Configure log group for short retention
 * LogGroup logGroup = LogGroup.Builder.create(stack, "LogGroup")
 *         .retention(RetentionDays.ONE_WEEK)
 *         .build();// Configure log group for infinite retention
 * LogGroup logGroup = LogGroup.Builder.create(stack, "LogGroup")
 *         .retention(Infinity)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>LogRetention</h2>
 * <p>
 * The <code>LogRetention</code> construct is a way to control the retention period of log groups that are created outside of the CDK. The construct is usually
 * used on log groups that are auto created by AWS services, such as <a href="https://docs.aws.amazon.com/lambda/latest/dg/monitoring-cloudwatchlogs.html">AWS
 * lambda</a>.
 * <p>
 * This is implemented using a <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html">CloudFormation custom
 * resource</a>
 * which pre-creates the log group if it doesn't exist, and sets the specified log retention period (never expire, by default).
 * <p>
 * By default, the log group will be created in the same region as the stack. The <code>logGroupRegion</code> property can be used to configure
 * log groups in other regions. This is typically useful when controlling retention for log groups auto-created by global services that
 * publish their log group to a specific region, such as AWS Chatbot creating a log group in <code>us-east-1</code>.
 * <p>
 * By default, the log group created by LogRetention will be retained after the stack is deleted. If the RemovalPolicy is set to DESTROY, then the log group will be deleted when the stack is deleted.
 * <p>
 * <h2>Log Group Class</h2>
 * <p>
 * CloudWatch Logs offers two classes of log groups:
 * <p>
 * <ol>
 * <li>The CloudWatch Logs Standard log class is a full-featured option for logs that require real-time monitoring or logs that you access frequently.</li>
 * <li>The CloudWatch Logs Infrequent Access log class is a new log class that you can use to cost-effectively consolidate your logs. This log class offers a subset of CloudWatch Logs capabilities including managed ingestion, storage, cross-account log analytics, and encryption with a lower ingestion price per GB. The Infrequent Access log class is ideal for ad-hoc querying and after-the-fact forensic analysis on infrequently accessed logs.</li>
 * </ol>
 * <p>
 * For more details please check: <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatch_Logs_Log_Classes.html">log group class documentation</a>
 * <p>
 * <h2>Resource Policy</h2>
 * <p>
 * CloudWatch Resource Policies allow other AWS services or IAM Principals to put log events into the log groups.
 * A resource policy is automatically created when <code>addToResourcePolicy</code> is called on the LogGroup for the first time:
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup = new LogGroup(this, "LogGroup");
 * logGroup.addToResourcePolicy(PolicyStatement.Builder.create()
 *         .actions(List.of("logs:CreateLogStream", "logs:PutLogEvents"))
 *         .principals(List.of(new ServicePrincipal("es.amazonaws.com")))
 *         .resources(List.of(logGroup.getLogGroupArn()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Or more conveniently, write permissions to the log group can be granted as follows which gives same result as in the above example.
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup = new LogGroup(this, "LogGroup");
 * logGroup.grantWrite(new ServicePrincipal("es.amazonaws.com"));
 * </pre></blockquote>
 * <p>
 * Similarly, read permissions can be granted to the log group as follows.
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup = new LogGroup(this, "LogGroup");
 * logGroup.grantRead(new ServicePrincipal("es.amazonaws.com"));
 * </pre></blockquote>
 * <p>
 * Be aware that any ARNs or tokenized values passed to the resource policy will be converted into AWS Account IDs.
 * This is because CloudWatch Logs Resource Policies do not accept ARNs as principals, but they do accept
 * Account ID strings. Non-ARN principals, like Service principals or Any principals, are accepted by CloudWatch.
 * <p>
 * <h2>Encrypting Log Groups</h2>
 * <p>
 * By default, log group data is always encrypted in CloudWatch Logs. You have the
 * option to encrypt log group data using a AWS KMS customer master key (CMK) should
 * you not wish to use the default AWS encryption. Keep in mind that if you decide to
 * encrypt a log group, any service or IAM identity that needs to read the encrypted
 * log streams in the future will require the same CMK to decrypt the data.
 * <p>
 * Here's a simple example of creating an encrypted Log Group using a KMS CMK.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * LogGroup.Builder.create(this, "LogGroup")
 *         .encryptionKey(new Key(this, "Key"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See the AWS documentation for more detailed information about <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/encrypt-log-data-kms.html">encrypting CloudWatch
 * Logs</a>.
 * <p>
 * <h2>Subscriptions and Destinations</h2>
 * <p>
 * Log events matching a particular filter can be sent to either a Lambda function
 * or a Kinesis stream.
 * <p>
 * If the Kinesis stream lives in a different account, a <code>CrossAccountDestination</code>
 * object needs to be added in the destination account which will act as a proxy
 * for the remote Kinesis stream. This object is automatically created for you
 * if you use the CDK Kinesis library.
 * <p>
 * Create a <code>SubscriptionFilter</code>, initialize it with an appropriate <code>Pattern</code> (see
 * below) and supply the intended destination:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.destinations.*;
 * 
 * Function fn;
 * LogGroup logGroup;
 * 
 * 
 * SubscriptionFilter.Builder.create(this, "Subscription")
 *         .logGroup(logGroup)
 *         .destination(new LambdaDestination(fn))
 *         .filterPattern(FilterPattern.allTerms("ERROR", "MainThread"))
 *         .filterName("ErrorInMainThread")
 *         .build();
 * </pre></blockquote>
 * <p>
 * When you use <code>KinesisDestination</code>, you can choose the method used to
 * distribute log data to the destination by setting the <code>distribution</code> property.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.destinations.*;
 * import software.amazon.awscdk.services.kinesis.*;
 * 
 * Stream stream;
 * LogGroup logGroup;
 * 
 * 
 * SubscriptionFilter.Builder.create(this, "Subscription")
 *         .logGroup(logGroup)
 *         .destination(new KinesisDestination(stream))
 *         .filterPattern(FilterPattern.allTerms("ERROR", "MainThread"))
 *         .filterName("ErrorInMainThread")
 *         .distribution(Distribution.RANDOM)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When you use <code>FirehoseDestination</code>, you can choose the method used to
 * distribute log data to the destination by setting the <code>distribution</code> property.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.destinations.*;
 * import software.amazon.awscdk.services.kinesisfirehose.*;
 * 
 * IDeliveryStream deliveryStream;
 * LogGroup logGroup;
 * 
 * 
 * SubscriptionFilter.Builder.create(this, "Subscription")
 *         .logGroup(logGroup)
 *         .destination(new FirehoseDestination(deliveryStream))
 *         .filterPattern(FilterPattern.allEvents())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Metric Filters</h2>
 * <p>
 * CloudWatch Logs can extract and emit metrics based on a textual log stream.
 * Depending on your needs, this may be a more convenient way of generating metrics
 * for you application than making calls to CloudWatch Metrics yourself.
 * <p>
 * A <code>MetricFilter</code> either emits a fixed number every time it sees a log event
 * matching a particular pattern (see below), or extracts a number from the log
 * event and uses that as the metric value.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * MetricFilter.Builder.create(this, "MetricFilter")
 *         .logGroup(logGroup)
 *         .metricNamespace("MyApp")
 *         .metricName("Latency")
 *         .filterPattern(FilterPattern.all(FilterPattern.exists("$.latency"), FilterPattern.regexValue("$.message", "=", "bind: address already in use")))
 *         .metricValue("$.latency")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Remember that if you want to use a value from the log event as the metric value,
 * you must mention it in your pattern somewhere.
 * <p>
 * A very simple MetricFilter can be created by using the <code>logGroup.extractMetric()</code>
 * helper function:
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup;
 * 
 * logGroup.extractMetric("$.jsonField", "Namespace", "MetricName");
 * </pre></blockquote>
 * <p>
 * Will extract the value of <code>jsonField</code> wherever it occurs in JSON-structured
 * log records in the LogGroup, and emit them to CloudWatch Metrics under
 * the name <code>Namespace/MetricName</code>.
 * <p>
 * <h3>Exposing Metric on a Metric Filter</h3>
 * <p>
 * You can expose a metric on a metric filter by calling the <code>MetricFilter.metric()</code> API.
 * This has a default of <code>statistic = 'avg'</code> if the statistic is not set in the <code>props</code>.
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup;
 * 
 * MetricFilter mf = MetricFilter.Builder.create(this, "MetricFilter")
 *         .logGroup(logGroup)
 *         .metricNamespace("MyApp")
 *         .metricName("Latency")
 *         .filterPattern(FilterPattern.exists("$.latency"))
 *         .metricValue("$.latency")
 *         .dimensions(Map.of(
 *                 "ErrorCode", "$.errorCode"))
 *         .unit(Unit.MILLISECONDS)
 *         .build();
 * 
 * //expose a metric from the metric filter
 * Metric metric = mf.metric();
 * 
 * //you can use the metric to create a new alarm
 * //you can use the metric to create a new alarm
 * Alarm.Builder.create(this, "alarm from metric filter")
 *         .metric(metric)
 *         .threshold(100)
 *         .evaluationPeriods(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Metrics for IncomingLogs and IncomingBytes</h3>
 * <p>
 * Metric methods have been defined for IncomingLogs and IncomingBytes within LogGroups. These metrics allow for the creation of alarms on log ingestion, ensuring that the log ingestion process is functioning correctly.
 * <p>
 * To define an alarm based on these metrics, you can use the following template:
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup = new LogGroup(this, "MyLogGroup");
 * Metric incomingEventsMetric = logGroup.metricIncomingLogEvents();
 * Alarm.Builder.create(this, "HighLogVolumeAlarm")
 *         .metric(incomingEventsMetric)
 *         .threshold(1000)
 *         .evaluationPeriods(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup = new LogGroup(this, "MyLogGroup");
 * Metric incomingBytesMetric = logGroup.metricIncomingBytes();
 * Alarm.Builder.create(this, "HighDataVolumeAlarm")
 *         .metric(incomingBytesMetric)
 *         .threshold(5000000) // 5 MB
 *         .evaluationPeriods(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Patterns</h2>
 * <p>
 * Patterns describe which log events match a subscription or metric filter. There
 * are three types of patterns:
 * <p>
 * <ul>
 * <li>Text patterns</li>
 * <li>JSON patterns</li>
 * <li>Space-delimited table patterns</li>
 * </ul>
 * <p>
 * All patterns are constructed by using static functions on the <code>FilterPattern</code>
 * class.
 * <p>
 * In addition to the patterns above, the following special patterns exist:
 * <p>
 * <ul>
 * <li><code>FilterPattern.allEvents()</code>: matches all log events.</li>
 * <li><code>FilterPattern.literal(string)</code>: if you already know what pattern expression to
 * use, this function takes a string and will use that as the log pattern. For
 * more information, see the <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/FilterAndPatternSyntax.html">Filter and Pattern
 * Syntax</a>.</li>
 * </ul>
 * <p>
 * <h3>Text Patterns</h3>
 * <p>
 * Text patterns match if the literal strings appear in the text form of the log
 * line.
 * <p>
 * <ul>
 * <li><code>FilterPattern.allTerms(term, term, ...)</code>: matches if all of the given terms
 * (substrings) appear in the log event.</li>
 * <li><code>FilterPattern.anyTerm(term, term, ...)</code>: matches if all of the given terms
 * (substrings) appear in the log event.</li>
 * <li><code>FilterPattern.anyTermGroup([term, term, ...], [term, term, ...], ...)</code>: matches if
 * all of the terms in any of the groups (specified as arrays) matches. This is
 * an OR match.</li>
 * </ul>
 * <p>
 * Examples:
 * <p>
 * <blockquote><pre>
 * // Search for lines that contain both "ERROR" and "MainThread"
 * IFilterPattern pattern1 = FilterPattern.allTerms("ERROR", "MainThread");
 * 
 * // Search for lines that either contain both "ERROR" and "MainThread", or
 * // both "WARN" and "Deadlock".
 * IFilterPattern pattern2 = FilterPattern.anyTermGroup(List.of("ERROR", "MainThread"), List.of("WARN", "Deadlock"));
 * </pre></blockquote>
 * <p>
 * <h2>JSON Patterns</h2>
 * <p>
 * JSON patterns apply if the log event is the JSON representation of an object
 * (without any other characters, so it cannot include a prefix such as timestamp
 * or log level). JSON patterns can make comparisons on the values inside the
 * fields.
 * <p>
 * <ul>
 * <li><strong>Strings</strong>: the comparison operators allowed for strings are <code>=</code> and <code>!=</code>.
 * String values can start or end with a <code>*</code> wildcard.</li>
 * <li><strong>Numbers</strong>: the comparison operators allowed for numbers are <code>=</code>, <code>!=</code>,
 * <code>&lt;</code>, <code>&lt;=</code>, <code>&gt;</code>, <code>&gt;=</code>.</li>
 * </ul>
 * <p>
 * Fields in the JSON structure are identified by identifier the complete object as <code>$</code>
 * and then descending into it, such as <code>$.field</code> or <code>$.list[0].field</code>.
 * <p>
 * <ul>
 * <li><code>FilterPattern.stringValue(field, comparison, string)</code>: matches if the given
 * field compares as indicated with the given string value.</li>
 * <li><code>FilterPattern.regexValue(field, comparison, string)</code>: matches if the given
 * field compares as indicated with the given regex pattern.</li>
 * <li><code>FilterPattern.numberValue(field, comparison, number)</code>: matches if the given
 * field compares as indicated with the given numerical value.</li>
 * <li><code>FilterPattern.isNull(field)</code>: matches if the given field exists and has the
 * value <code>null</code>.</li>
 * <li><code>FilterPattern.notExists(field)</code>: matches if the given field is not in the JSON
 * structure.</li>
 * <li><code>FilterPattern.exists(field)</code>: matches if the given field is in the JSON
 * structure.</li>
 * <li><code>FilterPattern.booleanValue(field, boolean)</code>: matches if the given field
 * is exactly the given boolean value.</li>
 * <li><code>FilterPattern.all(jsonPattern, jsonPattern, ...)</code>: matches if all of the
 * given JSON patterns match. This makes an AND combination of the given
 * patterns.</li>
 * <li><code>FilterPattern.any(jsonPattern, jsonPattern, ...)</code>: matches if any of the
 * given JSON patterns match. This makes an OR combination of the given
 * patterns.</li>
 * </ul>
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Search for all events where the component field is equal to
 * // "HttpServer" and either error is true or the latency is higher
 * // than 1000.
 * JsonPattern pattern = FilterPattern.all(FilterPattern.stringValue("$.component", "=", "HttpServer"), FilterPattern.any(FilterPattern.booleanValue("$.error", true), FilterPattern.numberValue("$.latency", "&gt;", 1000)), FilterPattern.regexValue("$.message", "=", "bind address already in use"));
 * </pre></blockquote>
 * <p>
 * <h2>Space-delimited table patterns</h2>
 * <p>
 * If the log events are rows of a space-delimited table, this pattern can be used
 * to identify the columns in that structure and add conditions on any of them. The
 * canonical example where you would apply this type of pattern is Apache server
 * logs.
 * <p>
 * Text that is surrounded by <code>"..."</code> quotes or <code>[...]</code> square brackets will
 * be treated as one column.
 * <p>
 * <ul>
 * <li><code>FilterPattern.spaceDelimited(column, column, ...)</code>: construct a
 * <code>SpaceDelimitedTextPattern</code> object with the indicated columns. The columns
 * map one-by-one the columns found in the log event. The string <code>"..."</code> may
 * be used to specify an arbitrary number of unnamed columns anywhere in the
 * name list (but may only be specified once).</li>
 * </ul>
 * <p>
 * After constructing a <code>SpaceDelimitedTextPattern</code>, you can use the following
 * two members to add restrictions:
 * <p>
 * <ul>
 * <li><code>pattern.whereString(field, comparison, string)</code>: add a string condition.
 * The rules are the same as for JSON patterns.</li>
 * <li><code>pattern.whereNumber(field, comparison, number)</code>: add a numerical condition.
 * The rules are the same as for JSON patterns.</li>
 * </ul>
 * <p>
 * Multiple restrictions can be added on the same column; they must all apply.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Search for all events where the component is "HttpServer" and the
 * // result code is not equal to 200.
 * SpaceDelimitedTextPattern pattern = FilterPattern.spaceDelimited("time", "component", "...", "result_code", "latency").whereString("component", "=", "HttpServer").whereNumber("result_code", "!=", 200);
 * </pre></blockquote>
 * <p>
 * <h2>Logs Insights Query Definition</h2>
 * <p>
 * Creates a query definition for CloudWatch Logs Insights.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * QueryDefinition.Builder.create(this, "QueryDefinition")
 *         .queryDefinitionName("MyQuery")
 *         .queryString(QueryString.Builder.create()
 *                 .fields(List.of("&#64;timestamp", "&#64;message"))
 *                 .parseStatements(List.of("&#64;message \"[*] *\" as loggingType, loggingMessage", "&#64;message \"&lt;*&gt;: *\" as differentLoggingType, differentLoggingMessage"))
 *                 .filterStatements(List.of("loggingType = \"ERROR\"", "loggingMessage = \"A very strange error occurred!\""))
 *                 .statsStatements(List.of("count(loggingMessage) as loggingErrors", "count(differentLoggingMessage) as differentLoggingErrors"))
 *                 .sort("&#64;timestamp desc")
 *                 .limit(20)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Data Protection Policy</h2>
 * <p>
 * Creates a data protection policy and assigns it to the log group. A data protection policy can help safeguard sensitive data that's ingested by the log group by auditing and masking the sensitive log data. When a user who does not have permission to view masked data views a log event that includes masked data, the sensitive data is replaced by asterisks.
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/mask-sensitive-log-data.html">Protect sensitive log data with masking</a>.
 * <p>
 * For a list of types of managed identifiers that can be audited and masked, see <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/protect-sensitive-log-data-types.html">Types of data that you can protect</a>.
 * <p>
 * If a new identifier is supported but not yet in the <code>DataIdentifiers</code> enum, the name of the identifier can be supplied as <code>name</code> in the constructor instead.
 * <p>
 * To add a custom data identifier, supply a custom <code>name</code> and <code>regex</code> to the <code>CustomDataIdentifiers</code> constructor.
 * For more information on custom data identifiers, see <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CWL-custom-data-identifiers.html">Custom data identifiers</a>.
 * <p>
 * Each policy may consist of a log group, S3 bucket, and/or Firehose delivery stream audit destination.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kinesisfirehose.*;
 * 
 * 
 * LogGroup logGroupDestination = LogGroup.Builder.create(this, "LogGroupLambdaAudit")
 *         .logGroupName("auditDestinationForCDK")
 *         .build();
 * 
 * Bucket bucket = new Bucket(this, "audit-bucket");
 * S3Bucket s3Destination = new S3Bucket(bucket);
 * 
 * DeliveryStream deliveryStream = DeliveryStream.Builder.create(this, "Delivery Stream")
 *         .destination(s3Destination)
 *         .build();
 * 
 * DataProtectionPolicy dataProtectionPolicy = DataProtectionPolicy.Builder.create()
 *         .name("data protection policy")
 *         .description("policy description")
 *         .identifiers(List.of(DataIdentifier.DRIVERSLICENSE_US,  // managed data identifier
 *             new DataIdentifier("EmailAddress"),  // forward compatibility for new managed data identifiers
 *             new CustomDataIdentifier("EmployeeId", "EmployeeId-\\d{9}"))) // custom data identifier
 *         .logGroupAuditDestination(logGroupDestination)
 *         .s3BucketAuditDestination(bucket)
 *         .deliveryStreamNameAuditDestination(deliveryStream.getDeliveryStreamName())
 *         .build();
 * 
 * LogGroup.Builder.create(this, "LogGroupLambda")
 *         .logGroupName("cdkIntegLogGroup")
 *         .dataProtectionPolicy(dataProtectionPolicy)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Field Index Policies</h2>
 * <p>
 * Creates or updates a field index policy for the specified log group. You can use field index policies to create field indexes on fields found in log events in the log group. Creating field indexes lowers the costs for CloudWatch Logs Insights queries that reference those field indexes, because these queries attempt to skip the processing of log events that are known to not match the indexed field. Good fields to index are fields that you often need to query for and fields that have high cardinality of values.
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatchLogs-Field-Indexing.html">Create field indexes to improve query performance and reduce costs</a>.
 * <p>
 * Only log groups in the Standard log class support field index policies.
 * Currently, this array supports only one field index policy object.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * FieldIndexPolicy fieldIndexPolicy = FieldIndexPolicy.Builder.create()
 *         .fields(List.of("Operation", "RequestId"))
 *         .build();
 * 
 * LogGroup.Builder.create(this, "LogGroup")
 *         .logGroupName("cdkIntegLogGroup")
 *         .fieldIndexPolicies(List.of(fieldIndexPolicy))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Transformer</h2>
 * <p>
 * A log transformer enables transforming log events into a different format, making them easier
 * to process and analyze. You can transform logs from different sources into standardized formats
 * that contain relevant, source-specific information. Transformations are performed at the time of log ingestion.
 * Transformers support several types of processors which can be chained into a processing pipeline (subject to some restrictions, see <a href="#usage-limits">Usage Limits</a>).
 * <p>
 * <h3>Processor Types</h3>
 * <p>
 * <ol>
 * <li><strong>Parser Processors</strong>: Parse string log events into structured log events. These are configurable parsers created using <code>ParserProcessor</code>, and support conversion to a format like Json, extracting fields from CSV input, converting vended sources to <a href="https://schema.ocsf.io/1.1.0/">OCSF</a> format, regex parsing using Grok patterns or key-value parsing. Refer <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatch-Logs-Transformation-Processors.html#CloudWatch-Logs-Transformation-Configurable">configurable parsers</a> for more examples.</li>
 * <li><strong>Vended Log Parsers</strong>: Parse log events from vended sources into structured log events. These are created using <code>VendedLogParser</code>, and support conversion from sources such as AWS WAF, PostGres, Route53, CloudFront and VPC. These parsers are not configurable, meaning these can be added to the pipeline but do not accept any properties or configurations. Refer <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatch-Logs-Transformation-Processors.html#CloudWatch-Logs-Transformation-BuiltIn">vended log parsers</a> for more examples.</li>
 * <li><strong>String Mutators</strong>: Perform operations on string values in a field of a log event and are created using <code>StringMutatorProcessor</code>. These can be used to format string values in the log event such as changing case, removing trailing whitespaces or extracting values from a string field by splitting the string or regex backreferences. Refer <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatch-Logs-Transformation-Processors.html#CloudWatch-Logs-Transformation-StringMutate">string mutators</a> for more examples.</li>
 * <li><strong>JSON Mutators</strong>: Perform operation on JSON log events and are created using <code>JsonMutatorProcessor</code>. These processors can be used to enrich log events by adding new fields, deleting, moving, renaming fields, copying values to other fields or converting a list of key-value pairs to a map. Refer <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatch-Logs-Transformation-Processors.html#CloudWatch-Logs-Transformation-JSONMutate">JSON mutators</a> for more examples.</li>
 * <li><strong>Data Converters</strong>: Convert the data into different formats and are created using <code>DataConverterProcessor</code>. These can be used to convert values in a field to datatypes such as integers, string, double and boolean or to convert dates and times to different formats. Refer <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatch-Logs-Transformation-Processors.html#CloudWatch-Logs-Transformation-Datatype">datatype processors</a> for more examples.</li>
 * </ol>
 * <p>
 * <h3>Usage Limits</h3>
 * <p>
 * <ul>
 * <li>A transformer can have a maximum of 20 processors</li>
 * <li>At least one parser-type processor is required</li>
 * <li>Maximum of 5 parser-type processors allowed</li>
 * <li>AWS vended log parser (if used) must be the first processor</li>
 * <li>Only one parseToOcsf processor, one grok processor, one addKeys processor, and one copyValue processor allowed per transformer</li>
 * <li>Transformers can only be used with log groups in the Standard log class</li>
 * </ul>
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * // Create a log group
 * LogGroup logGroup = new LogGroup(this, "MyLogGroup");
 * 
 * // Create a JSON parser processor
 * ParserProcessor jsonParser = ParserProcessor.Builder.create()
 *         .type(ParserProcessorType.JSON)
 *         .build();
 * 
 * // Create a processor to add keys
 * JsonMutatorProcessor addKeysProcessor = JsonMutatorProcessor.Builder.create()
 *         .type(JsonMutatorType.ADD_KEYS)
 *         .addKeysOptions(AddKeysProperty.builder()
 *                 .entries(List.of(AddKeyEntryProperty.builder()
 *                         .key("metadata.transformed_in")
 *                         .value("CloudWatchLogs")
 *                         .build()))
 *                 .build())
 *         .build();
 * 
 * // Create a transformer with these processors
 * // Create a transformer with these processors
 * Transformer.Builder.create(this, "Transformer")
 *         .transformerName("MyTransformer")
 *         .logGroup(logGroup)
 *         .transformerConfig(List.of(jsonParser, addKeysProcessor))
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more details on CloudWatch Logs transformation processors, refer to the <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatch-Logs-Transformation-Processors.html">AWS documentation</a>.
 * <p>
 * <h2>Notes</h2>
 * <p>
 * Be aware that Log Group ARNs will always have the string <code>:*</code> appended to
 * them, to match the behavior of <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#aws-resource-logs-loggroup-return-values">the CloudFormation <code>AWS::Logs::LogGroup</code>
 * resource</a>.
 */
package software.amazon.awscdk.services.logs;
