package software.amazon.awscdk;

/**
 * Possible values for a resource's Removal Policy.
 * <p>
 * The removal policy controls what happens to the resource if it stops being
 * managed by CloudFormation. This can happen in one of three situations:
 * <p>
 * <ul>
 * <li>The resource is removed from the template, so CloudFormation stops managing it;</li>
 * <li>A change to the resource is made that requires it to be replaced, so CloudFormation stops
 * managing it;</li>
 * <li>The stack is deleted, so CloudFormation stops managing all resources in it.</li>
 * </ul>
 * <p>
 * The Removal Policy applies to all above cases.
 * <p>
 * Many stateful resources in the AWS Construct Library will accept a
 * <code>removalPolicy</code> as a property, typically defaulting it to <code>RETAIN</code>.
 * <p>
 * If the AWS Construct Library resource does not accept a <code>removalPolicy</code>
 * argument, you can always configure it by using the escape hatch mechanism,
 * as shown in the following example:
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * CfnResource cfnBucket = (CfnResource)bucket.node.findChild("Resource");
 * cfnBucket.applyRemovalPolicy(RemovalPolicy.DESTROY);
 * </pre></blockquote>
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = Bucket.Builder.create(this, "memoryBucket")
 *         .bucketName("test-memory")
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .autoDeleteObjects(true)
 *         .build();
 * Topic topic = new Topic(this, "topic");
 * // Create a custom semantic memory strategy
 * SelfManagedMemoryStrategy selfManagedStrategy = MemoryStrategy.usingSelfManaged(SelfManagedStrategyProps.builder()
 *         .name("selfManagedStrategy")
 *         .description("self managed memory strategy")
 *         .historicalContextWindowSize(5)
 *         .invocationConfiguration(InvocationConfiguration.builder()
 *                 .topic(topic)
 *                 .s3Location(Location.builder()
 *                         .bucketName(bucket.getBucketName())
 *                         .objectKey("memory/")
 *                         .build())
 *                 .build())
 *         .triggerConditions(TriggerConditions.builder()
 *                 .messageBasedTrigger(1)
 *                 .timeBasedTrigger(Duration.seconds(10))
 *                 .tokenBasedTrigger(100)
 *                 .build())
 *         .build());
 * // Create memory with custom strategy
 * Memory memory = Memory.Builder.create(this, "MyMemory")
 *         .memoryName("my-custom-memory")
 *         .description("Memory with custom strategy")
 *         .expirationDuration(Duration.days(90))
 *         .memoryStrategies(List.of(selfManagedStrategy))
 *         .build();
 * </pre></blockquote>
 */
@javax.annotation.Generated(value = "jsii-pacmak/1.118.0 (build 02eec31)", date = "2025-11-10T13:39:56.889Z")
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
@software.amazon.jsii.Jsii(module = software.amazon.awscdk.$Module.class, fqn = "aws-cdk-lib.RemovalPolicy")
public enum RemovalPolicy {
    /**
     * When this removal policy is applied, the resource will be physically destroyed when it is removed from the stack or when the stack is deleted.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    DESTROY,
    /**
     * This uses the 'Retain' DeletionPolicy, which will cause the resource to be retained in the account, but orphaned from the stack.
     * <p>
     * Most resources default to this removal policy.
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    RETAIN,
    /**
     * This retention policy deletes the resource, but saves a snapshot of its data before deleting, so that it can be re-created later.
     * <p>
     * Only available for some stateful resources,
     * like databases, EC2 volumes, etc.
     * <p>
     * @see <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options">https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options</a>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    SNAPSHOT,
    /**
     * Resource will be retained when they are requested to be deleted during a stack delete request or need to be replaced due to a stack update request.
     * <p>
     * Resource are not retained, if the creation is rolled back.
     * <p>
     * The result is that new, empty, and unused resources are deleted,
     * while in-use resources and their data are retained.
     * <p>
     * This uses the 'RetainExceptOnCreate' DeletionPolicy,
     * and the 'Retain' UpdateReplacePolicy, when <code>applyToUpdateReplacePolicy</code> is set.
     * <p>
     * @see <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options">https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options</a>
     */
    @software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
    RETAIN_ON_UPDATE_OR_DELETE,
}
