/**
 * <h1>Amazon API Gateway Construct Library</h1>
 * <p>
 * Amazon API Gateway is a fully managed service that makes it easy for developers
 * to publish, maintain, monitor, and secure APIs at any scale. Create an API to
 * access data, business logic, or functionality from your back-end services, such
 * as applications running on Amazon Elastic Compute Cloud (Amazon EC2), code
 * running on AWS Lambda, or any web application.
 * <p>
 * <h2>Table of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#amazon-api-gateway-construct-library">Amazon API Gateway Construct Library</a>
 * <p>
 * <ul>
 * <li><a href="#table-of-contents">Table of Contents</a></li>
 * <li><a href="#defining-apis">Defining APIs</a>
 * <p>
 * <ul>
 * <li><a href="#breaking-up-methods-and-resources-across-stacks">Breaking up Methods and Resources across Stacks</a></li>
 * </ul></li>
 * <li><a href="#aws-lambda-backed-apis">AWS Lambda-backed APIs</a></li>
 * <li><a href="#aws-stepfunctions-backed-apis">AWS StepFunctions backed APIs</a></li>
 * <li><a href="#integration-targets">Integration Targets</a></li>
 * <li><a href="#usage-plan--api-keys">Usage Plan &amp; API Keys</a>
 * <p>
 * <ul>
 * <li><a href="#adding-an-api-key-to-an-imported-restapi">Adding an API Key to an imported RestApi</a></li>
 * <li><a href="#%EF%B8%8F-multiple-api-keys">⚠️ Multiple API Keys</a></li>
 * <li><a href="#rate-limited-api-key">Rate Limited API Key</a></li>
 * </ul></li>
 * <li><a href="#working-with-models">Working with models</a></li>
 * <li><a href="#default-integration-and-method-options">Default Integration and Method Options</a></li>
 * <li><a href="#proxy-routes">Proxy Routes</a></li>
 * <li><a href="#authorizers">Authorizers</a>
 * <p>
 * <ul>
 * <li><a href="#iam-based-authorizer">IAM-based authorizer</a></li>
 * <li><a href="#lambda-based-token-authorizer">Lambda-based token authorizer</a></li>
 * <li><a href="#lambda-based-request-authorizer">Lambda-based request authorizer</a></li>
 * <li><a href="#cognito-user-pools-authorizer">Cognito User Pools authorizer</a></li>
 * </ul></li>
 * <li><a href="#mutual-tls-mtls">Mutual TLS (mTLS)</a></li>
 * <li><a href="#deployments">Deployments</a>
 * <p>
 * <ul>
 * <li><a href="#deploying-to-an-existing-stage">Deploying to an existing stage</a>
 * <p>
 * <ul>
 * <li><a href="#using-restapi">Using RestApi</a></li>
 * <li><a href="#using-specrestapi">Using SpecRestApi</a></li>
 * </ul></li>
 * <li><a href="#controlled-triggering-of-deployments">Controlled triggering of deployments</a></li>
 * <li><a href="#deep-dive-invalidation-of-deployments">Deep dive: Invalidation of deployments</a></li>
 * </ul></li>
 * <li><a href="#custom-domains">Custom Domains</a>
 * <p>
 * <ul>
 * <li><a href="#custom-domains-with-multi-level-api-mapping">Custom Domains with multi-level api mapping</a></li>
 * </ul></li>
 * <li><a href="#access-logging">Access Logging</a></li>
 * <li><a href="#cross-origin-resource-sharing-cors">Cross Origin Resource Sharing (CORS)</a></li>
 * <li><a href="#endpoint-configuration">Endpoint Configuration</a></li>
 * <li><a href="#private-integrations">Private Integrations</a></li>
 * <li><a href="#gateway-response">Gateway response</a></li>
 * <li><a href="#openapi-definition">OpenAPI Definition</a>
 * <p>
 * <ul>
 * <li><a href="#endpoint-configuration-1">Endpoint configuration</a></li>
 * </ul></li>
 * <li><a href="#metrics">Metrics</a></li>
 * <li><a href="#apigateway-v2">APIGateway v2</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>Defining APIs</h2>
 * <p>
 * APIs are defined as a hierarchy of resources and methods. <code>addResource</code> and
 * <code>addMethod</code> can be used to build this hierarchy. The root resource is
 * <code>api.root</code>.
 * <p>
 * For example, the following code defines an API that includes the following HTTP
 * endpoints: <code>ANY /</code>, <code>GET /books</code>, <code>POST /books</code>, <code>GET /books/{book_id}</code>, <code>DELETE /books/{book_id}</code>.
 * <p>
 * <blockquote><pre>
 * RestApi api = new RestApi(this, "books-api");
 * 
 * api.root.addMethod("ANY");
 * 
 * Resource books = api.root.addResource("books");
 * books.addMethod("GET");
 * books.addMethod("POST");
 * 
 * Resource book = books.addResource("{book_id}");
 * book.addMethod("GET");
 * book.addMethod("DELETE");
 * </pre></blockquote>
 * <p>
 * To give an IAM User or Role permission to invoke a method, use <code>grantExecute</code>:
 * <p>
 * <blockquote><pre>
 * RestApi api;
 * User user;
 * 
 * 
 * Method method = api.root.addResource("books").addMethod("GET");
 * method.grantExecute(user);
 * </pre></blockquote>
 * <p>
 * <h3>Breaking up Methods and Resources across Stacks</h3>
 * <p>
 * It is fairly common for REST APIs with a large number of Resources and Methods to hit the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html">CloudFormation
 * limit</a> of 500 resources per
 * stack.
 * <p>
 * To help with this, Resources and Methods for the same REST API can be re-organized across multiple stacks. A common
 * way to do this is to have a stack per Resource or groups of Resources, but this is not the only possible way.
 * The following example uses sets up two Resources '/pets' and '/books' in separate stacks using nested stacks:
 * <p>
 * <blockquote><pre>
 * import software.constructs.Construct;
 * import software.amazon.awscdk.App;
 * import software.amazon.awscdk.CfnOutput;
 * import software.amazon.awscdk.NestedStack;
 * import software.amazon.awscdk.NestedStackProps;
 * import software.amazon.awscdk.Stack;
 * import software.amazon.awscdk.services.apigateway.Deployment;
 * import software.amazon.awscdk.services.apigateway.Method;
 * import software.amazon.awscdk.services.apigateway.MockIntegration;
 * import software.amazon.awscdk.services.apigateway.PassthroughBehavior;
 * import software.amazon.awscdk.services.apigateway.RestApi;
 * import software.amazon.awscdk.services.apigateway.Stage;
 * 
 * /**
 *  * This file showcases how to split up a RestApi's Resources and Methods across nested stacks.
 *  *
 *  * The root stack 'RootStack' first defines a RestApi.
 *  * Two nested stacks BooksStack and PetsStack, create corresponding Resources '/books' and '/pets'.
 *  * They are then deployed to a 'prod' Stage via a third nested stack - DeployStack.
 *  *
 *  * To verify this worked, go to the APIGateway
 *  *&#47;
 * 
 * public class RootStack extends Stack {
 *     public RootStack(Construct scope) {
 *         super(scope, "integ-restapi-import-RootStack");
 * 
 *         RestApi restApi = RestApi.Builder.create(this, "RestApi")
 *                 .cloudWatchRole(true)
 *                 .deploy(false)
 *                 .build();
 *         restApi.root.addMethod("ANY");
 * 
 *         PetsStack petsStack = new PetsStack(this, new ResourceNestedStackProps()
 *                 .restApiId(restApi.getRestApiId())
 *                 .rootResourceId(restApi.getRestApiRootResourceId())
 *                 );
 *         BooksStack booksStack = new BooksStack(this, new ResourceNestedStackProps()
 *                 .restApiId(restApi.getRestApiId())
 *                 .rootResourceId(restApi.getRestApiRootResourceId())
 *                 );
 *         new DeployStack(this, new DeployStackProps()
 *                 .restApiId(restApi.getRestApiId())
 *                 .methods(petsStack.methods.concat(booksStack.getMethods()))
 *                 );
 * 
 *         CfnOutput.Builder.create(this, "PetsURL")
 *                 .value(String.format("https://%s.execute-api.%s.amazonaws.com/prod/pets", restApi.getRestApiId(), this.region))
 *                 .build();
 * 
 *         CfnOutput.Builder.create(this, "BooksURL")
 *                 .value(String.format("https://%s.execute-api.%s.amazonaws.com/prod/books", restApi.getRestApiId(), this.region))
 *                 .build();
 *     }
 * }
 * 
 * public class ResourceNestedStackProps extends NestedStackProps {
 *     private String restApiId;
 *     public String getRestApiId() {
 *         return this.restApiId;
 *     }
 *     public ResourceNestedStackProps restApiId(String restApiId) {
 *         this.restApiId = restApiId;
 *         return this;
 *     }
 * 
 *     private String rootResourceId;
 *     public String getRootResourceId() {
 *         return this.rootResourceId;
 *     }
 *     public ResourceNestedStackProps rootResourceId(String rootResourceId) {
 *         this.rootResourceId = rootResourceId;
 *         return this;
 *     }
 * }
 * 
 * public class PetsStack extends NestedStack {
 *     public final Method[] methods;
 * 
 *     public PetsStack(Construct scope, ResourceNestedStackProps props) {
 *         super(scope, "integ-restapi-import-PetsStack", props);
 * 
 *         IRestApi api = RestApi.fromRestApiAttributes(this, "RestApi", RestApiAttributes.builder()
 *                 .restApiId(props.getRestApiId())
 *                 .rootResourceId(props.getRootResourceId())
 *                 .build());
 * 
 *         Method method = api.root.addResource("pets").addMethod("GET", MockIntegration.Builder.create()
 *                 .integrationResponses(List.of(IntegrationResponse.builder()
 *                         .statusCode("200")
 *                         .build()))
 *                 .passthroughBehavior(PassthroughBehavior.NEVER)
 *                 .requestTemplates(Map.of(
 *                         "application/json", "{ \"statusCode\": 200 }"))
 *                 .build(), MethodOptions.builder()
 *                 .methodResponses(List.of(MethodResponse.builder().statusCode("200").build()))
 *                 .build());
 * 
 *         this.methods.push(method);
 *     }
 * }
 * 
 * public class BooksStack extends NestedStack {
 *     public final Method[] methods;
 * 
 *     public BooksStack(Construct scope, ResourceNestedStackProps props) {
 *         super(scope, "integ-restapi-import-BooksStack", props);
 * 
 *         IRestApi api = RestApi.fromRestApiAttributes(this, "RestApi", RestApiAttributes.builder()
 *                 .restApiId(props.getRestApiId())
 *                 .rootResourceId(props.getRootResourceId())
 *                 .build());
 * 
 *         Method method = api.root.addResource("books").addMethod("GET", MockIntegration.Builder.create()
 *                 .integrationResponses(List.of(IntegrationResponse.builder()
 *                         .statusCode("200")
 *                         .build()))
 *                 .passthroughBehavior(PassthroughBehavior.NEVER)
 *                 .requestTemplates(Map.of(
 *                         "application/json", "{ \"statusCode\": 200 }"))
 *                 .build(), MethodOptions.builder()
 *                 .methodResponses(List.of(MethodResponse.builder().statusCode("200").build()))
 *                 .build());
 * 
 *         this.methods.push(method);
 *     }
 * }
 * 
 * public class DeployStackProps extends NestedStackProps {
 *     private String restApiId;
 *     public String getRestApiId() {
 *         return this.restApiId;
 *     }
 *     public DeployStackProps restApiId(String restApiId) {
 *         this.restApiId = restApiId;
 *         return this;
 *     }
 * 
 *     private Method[] methods;
 *     public Method[] getMethods() {
 *         return this.methods;
 *     }
 *     public DeployStackProps methods(Method[] methods) {
 *         this.methods = methods;
 *         return this;
 *     }
 * }
 * 
 * public class DeployStack extends NestedStack {
 *     public DeployStack(Construct scope, DeployStackProps props) {
 *         super(scope, "integ-restapi-import-DeployStack", props);
 * 
 *         Deployment deployment = Deployment.Builder.create(this, "Deployment")
 *                 .api(RestApi.fromRestApiId(this, "RestApi", props.getRestApiId()))
 *                 .build();
 *         if (props.getMethods()) {
 *             for (Object method : props.getMethods()) {
 *                 deployment.node.addDependency(method);
 *             }
 *         }
 *         Stage.Builder.create(this, "Stage").deployment(deployment).build();
 *     }
 * }
 * 
 * new RootStack(new App());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Warning:</strong> In the code above, an API Gateway deployment is created during the initial CDK deployment.
 * However, if there are changes to the resources in subsequent CDK deployments, a new API Gateway deployment is not
 * automatically created. As a result, the latest state of the resources is not reflected. To ensure the latest state
 * of the resources is reflected, a manual deployment of the API Gateway is required after the CDK deployment. See <a href="#controlled-triggering-of-deployments">Controlled triggering of deployments</a> for more info.
 * <p>
 * </blockquote>
 * <p>
 * <h2>AWS Lambda-backed APIs</h2>
 * <p>
 * A very common practice is to use Amazon API Gateway with AWS Lambda as the
 * backend integration. The <code>LambdaRestApi</code> construct makes it easy:
 * <p>
 * The following code defines a REST API that routes all requests to the
 * specified AWS Lambda function:
 * <p>
 * <blockquote><pre>
 * Function backend;
 * 
 * LambdaRestApi.Builder.create(this, "myapi")
 *         .handler(backend)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also supply <code>proxy: false</code>, in which case you will have to explicitly
 * define the API model:
 * <p>
 * <blockquote><pre>
 * Function backend;
 * 
 * LambdaRestApi api = LambdaRestApi.Builder.create(this, "myapi")
 *         .handler(backend)
 *         .proxy(false)
 *         .build();
 * 
 * Resource items = api.root.addResource("items");
 * items.addMethod("GET"); // GET /items
 * items.addMethod("POST"); // POST /items
 * 
 * Resource item = items.addResource("{item}");
 * item.addMethod("GET"); // GET /items/{item}
 * 
 * // the default integration for methods is "handler", but one can
 * // customize this behavior per method or even a sub path.
 * item.addMethod("DELETE", new HttpIntegration("http://amazon.com"));
 * </pre></blockquote>
 * <p>
 * Additionally, <code>integrationOptions</code> can be supplied to explicitly define
 * options of the Lambda integration:
 * <p>
 * <blockquote><pre>
 * Function backend;
 * 
 * 
 * LambdaRestApi api = LambdaRestApi.Builder.create(this, "myapi")
 *         .handler(backend)
 *         .integrationOptions(LambdaIntegrationOptions.builder()
 *                 .allowTestInvoke(false)
 *                 .timeout(Duration.seconds(1))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>AWS StepFunctions backed APIs</h2>
 * <p>
 * You can use Amazon API Gateway with AWS Step Functions as the backend integration, specifically Synchronous Express Workflows.
 * <p>
 * The <code>StepFunctionsRestApi</code> only supports integration with Synchronous Express state machine. The <code>StepFunctionsRestApi</code> construct makes this easy by setting up input, output and error mapping.
 * <p>
 * The construct sets up an API endpoint and maps the <code>ANY</code> HTTP method and any calls to the API endpoint starts an express workflow execution for the underlying state machine.
 * <p>
 * Invoking the endpoint with any HTTP method (<code>GET</code>, <code>POST</code>, <code>PUT</code>, <code>DELETE</code>, ...) in the example below will send the request to the state machine as a new execution. On success, an HTTP code <code>200</code> is returned with the execution output as the Response Body.
 * <p>
 * If the execution fails, an HTTP <code>500</code> response is returned with the <code>error</code> and <code>cause</code> from the execution output as the Response Body. If the request is invalid (ex. bad execution input) HTTP code <code>400</code> is returned.
 * <p>
 * To disable default response models generation use the <code>useDefaultMethodResponses</code> property:
 * <p>
 * <blockquote><pre>
 * IStateMachine machine;
 * 
 * 
 * StepFunctionsRestApi.Builder.create(this, "StepFunctionsRestApi")
 *         .stateMachine(machine)
 *         .useDefaultMethodResponses(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The response from the invocation contains only the <code>output</code> field from the
 * <a href="https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartSyncExecution.html#API_StartSyncExecution_ResponseSyntax">StartSyncExecution</a> API.
 * In case of failures, the fields <code>error</code> and <code>cause</code> are returned as part of the response.
 * Other metadata such as billing details, AWS account ID and resource ARNs are not returned in the API response.
 * <p>
 * By default, a <code>prod</code> stage is provisioned.
 * <p>
 * In order to reduce the payload size sent to AWS Step Functions, <code>headers</code> are not forwarded to the Step Functions execution input. It is possible to choose whether <code>headers</code>,  <code>requestContext</code>, <code>path</code>, <code>querystring</code>, and <code>authorizer</code> are included or not. By default, <code>headers</code> are excluded in all requests.
 * <p>
 * More details about AWS Step Functions payload limit can be found at https://docs.aws.amazon.com/step-functions/latest/dg/limits-overview.html#service-limits-task-executions.
 * <p>
 * The following code defines a REST API that routes all requests to the specified AWS StepFunctions state machine:
 * <p>
 * <blockquote><pre>
 * Pass stateMachineDefinition = new Pass(this, "PassState");
 * 
 * IStateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definition(stateMachineDefinition)
 *         .stateMachineType(StateMachineType.EXPRESS)
 *         .build();
 * 
 * StepFunctionsRestApi.Builder.create(this, "StepFunctionsRestApi")
 *         .deploy(true)
 *         .stateMachine(stateMachine)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When the REST API endpoint configuration above is invoked using POST, as follows -
 * <p>
 * <blockquote><pre>
 * curl -X POST -d '{ "customerId": 1 }' https://example.com/
 * </pre></blockquote>
 * <p>
 * AWS Step Functions will receive the request body in its input as follows:
 * <p>
 * <blockquote><pre>
 * {
 *   "body": {
 *     "customerId": 1
 *   },
 *   "path": "/",
 *   "querystring": {}
 * }
 * </pre></blockquote>
 * <p>
 * When the endpoint is invoked at path '/users/5' using the HTTP GET method as below:
 * <p>
 * <blockquote><pre>
 * curl -X GET https://example.com/users/5?foo=bar
 * </pre></blockquote>
 * <p>
 * AWS Step Functions will receive the following execution input:
 * <p>
 * <blockquote><pre>
 * {
 *   "body": {},
 *   "path": {
 *      "users": "5"
 *   },
 *   "querystring": {
 *     "foo": "bar"
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * Additional information around the request such as the request context, authorizer context, and headers can be included as part of the input
 * forwarded to the state machine. The following example enables headers to be included in the input but not query string.
 * <p>
 * <blockquote><pre>
 * StepFunctionsRestApi.Builder.create(this, "StepFunctionsRestApi")
 *         .stateMachine(machine)
 *         .headers(true)
 *         .path(false)
 *         .querystring(false)
 *         .authorizer(false)
 *         .requestContext(RequestContext.builder()
 *                 .caller(true)
 *                 .user(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * In such a case, when the endpoint is invoked as below:
 * <p>
 * <blockquote><pre>
 * curl -X GET https://example.com/
 * </pre></blockquote>
 * <p>
 * AWS Step Functions will receive the following execution input:
 * <p>
 * <blockquote><pre>
 * {
 *   "headers": {
 *     "Accept": "...",
 *     "CloudFront-Forwarded-Proto": "...",
 *   },
 *   "requestContext": {
 *      "accountId": "...",
 *      "apiKey": "...",
 *   },
 *   "body": {}
 * }
 * </pre></blockquote>
 * <p>
 * <h2>Integration Targets</h2>
 * <p>
 * Methods are associated with backend integrations, which are invoked when this
 * method is called. API Gateway supports the following integrations:
 * <p>
 * <ul>
 * <li><code>MockIntegration</code> - can be used to test APIs. This is the default
 * integration if one is not specified.</li>
 * <li><code>AwsIntegration</code> - can be used to invoke arbitrary AWS service APIs.</li>
 * <li><code>HttpIntegration</code> - can be used to invoke HTTP endpoints.</li>
 * <li><code>LambdaIntegration</code> - can be used to invoke an AWS Lambda function.</li>
 * <li><code>SagemakerIntegration</code> - can be used to invoke Sagemaker Endpoints.</li>
 * </ul>
 * <p>
 * The following example shows how to integrate the <code>GET /book/{book_id}</code> method to
 * an AWS Lambda function:
 * <p>
 * <blockquote><pre>
 * Function getBookHandler;
 * Resource book;
 * 
 * 
 * LambdaIntegration getBookIntegration = new LambdaIntegration(getBookHandler);
 * book.addMethod("GET", getBookIntegration);
 * </pre></blockquote>
 * <p>
 * Integration options can be optionally be specified:
 * <p>
 * <blockquote><pre>
 * Function getBookHandler;
 * LambdaIntegration getBookIntegration;
 * 
 * 
 * LambdaIntegration getBookIntegration = LambdaIntegration.Builder.create(getBookHandler)
 *         .contentHandling(ContentHandling.CONVERT_TO_TEXT) // convert to base64
 *         .credentialsPassthrough(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Method options can optionally be specified when adding methods:
 * <p>
 * <blockquote><pre>
 * Resource book;
 * LambdaIntegration getBookIntegration;
 * 
 * 
 * book.addMethod("GET", getBookIntegration, MethodOptions.builder()
 *         .authorizationType(AuthorizationType.IAM)
 *         .apiKeyRequired(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * It is possible to also integrate with AWS services in a different region. The following code integrates with Amazon SQS in the
 * <code>eu-west-1</code> region.
 * <p>
 * <blockquote><pre>
 * AwsIntegration getMessageIntegration = AwsIntegration.Builder.create()
 *         .service("sqs")
 *         .path("queueName")
 *         .region("eu-west-1")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Usage Plan &amp; API Keys</h2>
 * <p>
 * A usage plan specifies who can access one or more deployed API stages and methods, and the rate at which they can be
 * accessed. The plan uses API keys to identify API clients and meters access to the associated API stages for each key.
 * Usage plans also allow configuring throttling limits and quota limits that are enforced on individual client API keys.
 * <p>
 * The following example shows how to create and associate a usage plan and an API key:
 * <p>
 * <blockquote><pre>
 * LambdaIntegration integration;
 * 
 * 
 * RestApi api = new RestApi(this, "hello-api");
 * 
 * Resource v1 = api.root.addResource("v1");
 * Resource echo = v1.addResource("echo");
 * Method echoMethod = echo.addMethod("GET", integration, MethodOptions.builder().apiKeyRequired(true).build());
 * 
 * UsagePlan plan = api.addUsagePlan("UsagePlan", UsagePlanProps.builder()
 *         .name("Easy")
 *         .throttle(ThrottleSettings.builder()
 *                 .rateLimit(10)
 *                 .burstLimit(2)
 *                 .build())
 *         .build());
 * 
 * IApiKey key = api.addApiKey("ApiKey");
 * plan.addApiKey(key);
 * </pre></blockquote>
 * <p>
 * To associate a plan to a given RestAPI stage:
 * <p>
 * <blockquote><pre>
 * UsagePlan plan;
 * RestApi api;
 * Method echoMethod;
 * 
 * 
 * plan.addApiStage(UsagePlanPerApiStage.builder()
 *         .stage(api.getDeploymentStage())
 *         .throttle(List.of(ThrottlingPerMethod.builder()
 *                 .method(echoMethod)
 *                 .throttle(ThrottleSettings.builder()
 *                         .rateLimit(10)
 *                         .burstLimit(2)
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Existing usage plans can be imported into a CDK app using its id.
 * <p>
 * <blockquote><pre>
 * IUsagePlan importedUsagePlan = UsagePlan.fromUsagePlanId(this, "imported-usage-plan", "&lt;usage-plan-key-id&gt;");
 * </pre></blockquote>
 * <p>
 * The name and value of the API Key can be specified at creation; if not
 * provided, a name and value will be automatically generated by API Gateway.
 * <p>
 * <blockquote><pre>
 * RestApi api;
 * 
 * IApiKey key = api.addApiKey("ApiKey", ApiKeyOptions.builder()
 *         .apiKeyName("myApiKey1")
 *         .value("MyApiKeyThatIsAtLeast20Characters")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Existing API keys can also be imported into a CDK app using its id.
 * <p>
 * <blockquote><pre>
 * IApiKey importedKey = ApiKey.fromApiKeyId(this, "imported-key", "&lt;api-key-id&gt;");
 * </pre></blockquote>
 * <p>
 * The "grant" methods can be used to give prepackaged sets of permissions to other resources. The
 * following code provides read permission to an API key.
 * <p>
 * <blockquote><pre>
 * ApiKey importedKey;
 * Function lambdaFn;
 * 
 * importedKey.grantRead(lambdaFn);
 * </pre></blockquote>
 * <p>
 * <h3>Adding an API Key to an imported RestApi</h3>
 * <p>
 * API Keys are added to ApiGateway Stages, not to the API itself. When you import a RestApi
 * it does not have any information on the Stages that may be associated with it. Since adding an API
 * Key requires a stage, you should instead add the Api Key to the imported Stage.
 * <p>
 * <blockquote><pre>
 * IRestApi restApi;
 * 
 * IStage importedStage = Stage.fromStageAttributes(this, "imported-stage", StageAttributes.builder()
 *         .stageName("myStageName")
 *         .restApi(restApi)
 *         .build());
 * 
 * importedStage.addApiKey("MyApiKey");
 * </pre></blockquote>
 * <p>
 * <h3>⚠️ Multiple API Keys</h3>
 * <p>
 * It is possible to specify multiple API keys for a given Usage Plan, by calling <code>usagePlan.addApiKey()</code>.
 * <p>
 * When using multiple API keys, a past bug of the CDK prevents API key associations to a Usage Plan to be deleted.
 * If the CDK app had the <a href="https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html">feature flag</a> - <code>&#64;aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId</code> - enabled when the API
 * keys were created, then the app will not be affected by this bug.
 * <p>
 * If this is not the case, you will need to ensure that the CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html">logical ids</a> of the API keys that are not
 * being deleted remain unchanged.
 * Make note of the logical ids of these API keys before removing any, and set it as part of the <code>addApiKey()</code> method:
 * <p>
 * <blockquote><pre>
 * UsagePlan usageplan;
 * ApiKey apiKey;
 * 
 * 
 * usageplan.addApiKey(apiKey, AddApiKeyOptions.builder()
 *         .overrideLogicalId("...")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Rate Limited API Key</h3>
 * <p>
 * In scenarios where you need to create a single api key and configure rate limiting for it, you can use <code>RateLimitedApiKey</code>.
 * This construct lets you specify rate limiting properties which should be applied only to the api key being created.
 * The API key created has the specified rate limits, such as quota and throttles, applied.
 * <p>
 * The following example shows how to use a rate limited api key :
 * <p>
 * <blockquote><pre>
 * RestApi api;
 * 
 * 
 * RateLimitedApiKey key = RateLimitedApiKey.Builder.create(this, "rate-limited-api-key")
 *         .customerId("hello-customer")
 *         .apiStages(List.of(UsagePlanPerApiStage.builder().stage(api.getDeploymentStage()).build()))
 *         .quota(QuotaSettings.builder()
 *                 .limit(10000)
 *                 .period(Period.MONTH)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Working with models</h2>
 * <p>
 * When you work with Lambda integrations that are not Proxy integrations, you
 * have to define your models and mappings for the request, response, and integration.
 * <p>
 * <blockquote><pre>
 * Function hello = Function.Builder.create(this, "hello")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("hello.handler")
 *         .code(Code.fromAsset("lambda"))
 *         .build();
 * 
 * RestApi api = RestApi.Builder.create(this, "hello-api").build();
 * Resource resource = api.root.addResource("v1");
 * </pre></blockquote>
 * <p>
 * You can define more parameters on the integration to tune the behavior of API Gateway
 * <p>
 * <blockquote><pre>
 * Function hello;
 * 
 * 
 * LambdaIntegration integration = LambdaIntegration.Builder.create(hello)
 *         .proxy(false)
 *         .requestParameters(Map.of(
 *                 // You can define mapping parameters from your method to your integration
 *                 // - Destination parameters (the key) are the integration parameters (used in mappings)
 *                 // - Source parameters (the value) are the source request parameters or expressions
 *                 // &#64;see: https://docs.aws.amazon.com/apigateway/latest/developerguide/request-response-data-mappings.html
 *                 "integration.request.querystring.who", "method.request.querystring.who"))
 *         .allowTestInvoke(true)
 *         .requestTemplates(Map.of(
 *                 // You can define a mapping that will build a payload for your integration, based
 *                 //  on the integration parameters that you have specified
 *                 // Check: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
 *                 "application/json", JSON.stringify(Map.of("action", "sayHello", "pollId", "$util.escapeJavaScript($input.params('who'))"))))
 *         // This parameter defines the behavior of the engine is no suitable response template is found
 *         .passthroughBehavior(PassthroughBehavior.NEVER)
 *         .integrationResponses(List.of(IntegrationResponse.builder()
 *                 // Successful response from the Lambda function, no filter defined
 *                 //  - the selectionPattern filter only tests the error message
 *                 // We will set the response status code to 200
 *                 .statusCode("200")
 *                 .responseTemplates(Map.of(
 *                         // This template takes the "message" result from the Lambda function, and embeds it in a JSON response
 *                         // Check https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
 *                         "application/json", JSON.stringify(Map.of("state", "ok", "greeting", "$util.escapeJavaScript($input.body)"))))
 *                 .responseParameters(Map.of(
 *                         // We can map response parameters
 *                         // - Destination parameters (the key) are the response parameters (used in mappings)
 *                         // - Source parameters (the value) are the integration response parameters or expressions
 *                         "method.response.header.Content-Type", "'application/json'",
 *                         "method.response.header.Access-Control-Allow-Origin", "'*'",
 *                         "method.response.header.Access-Control-Allow-Credentials", "'true'"))
 *                 .build(), IntegrationResponse.builder()
 *                 // For errors, we check if the error message is not empty, get the error data
 *                 .selectionPattern("(\n|.)+")
 *                 // We will set the response status code to 200
 *                 .statusCode("400")
 *                 .responseTemplates(Map.of(
 *                         "application/json", JSON.stringify(Map.of("state", "error", "message", "$util.escapeJavaScript($input.path('$.errorMessage'))"))))
 *                 .responseParameters(Map.of(
 *                         "method.response.header.Content-Type", "'application/json'",
 *                         "method.response.header.Access-Control-Allow-Origin", "'*'",
 *                         "method.response.header.Access-Control-Allow-Credentials", "'true'"))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can define models for your responses (and requests)
 * <p>
 * <blockquote><pre>
 * RestApi api;
 * 
 * 
 * // We define the JSON Schema for the transformed valid response
 * Model responseModel = api.addModel("ResponseModel", ModelOptions.builder()
 *         .contentType("application/json")
 *         .modelName("ResponseModel")
 *         .schema(JsonSchema.builder()
 *                 .schema(JsonSchemaVersion.DRAFT4)
 *                 .title("pollResponse")
 *                 .type(JsonSchemaType.OBJECT)
 *                 .properties(Map.of(
 *                         "state", JsonSchema.builder().type(JsonSchemaType.STRING).build(),
 *                         "greeting", JsonSchema.builder().type(JsonSchemaType.STRING).build()))
 *                 .build())
 *         .build());
 * 
 * // We define the JSON Schema for the transformed error response
 * Model errorResponseModel = api.addModel("ErrorResponseModel", ModelOptions.builder()
 *         .contentType("application/json")
 *         .modelName("ErrorResponseModel")
 *         .schema(JsonSchema.builder()
 *                 .schema(JsonSchemaVersion.DRAFT4)
 *                 .title("errorResponse")
 *                 .type(JsonSchemaType.OBJECT)
 *                 .properties(Map.of(
 *                         "state", JsonSchema.builder().type(JsonSchemaType.STRING).build(),
 *                         "message", JsonSchema.builder().type(JsonSchemaType.STRING).build()))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * And reference all on your method definition.
 * <p>
 * <blockquote><pre>
 * LambdaIntegration integration;
 * Resource resource;
 * Model responseModel;
 * Model errorResponseModel;
 * 
 * 
 * resource.addMethod("GET", integration, MethodOptions.builder()
 *         // We can mark the parameters as required
 *         .requestParameters(Map.of(
 *                 "method.request.querystring.who", true))
 *         // we can set request validator options like below
 *         .requestValidatorOptions(RequestValidatorOptions.builder()
 *                 .requestValidatorName("test-validator")
 *                 .validateRequestBody(true)
 *                 .validateRequestParameters(false)
 *                 .build())
 *         .methodResponses(List.of(MethodResponse.builder()
 *                 // Successful response from the integration
 *                 .statusCode("200")
 *                 // Define what parameters are allowed or not
 *                 .responseParameters(Map.of(
 *                         "method.response.header.Content-Type", true,
 *                         "method.response.header.Access-Control-Allow-Origin", true,
 *                         "method.response.header.Access-Control-Allow-Credentials", true))
 *                 // Validate the schema on the response
 *                 .responseModels(Map.of(
 *                         "application/json", responseModel))
 *                 .build(), MethodResponse.builder()
 *                 // Same thing for the error responses
 *                 .statusCode("400")
 *                 .responseParameters(Map.of(
 *                         "method.response.header.Content-Type", true,
 *                         "method.response.header.Access-Control-Allow-Origin", true,
 *                         "method.response.header.Access-Control-Allow-Credentials", true))
 *                 .responseModels(Map.of(
 *                         "application/json", errorResponseModel))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Specifying <code>requestValidatorOptions</code> automatically creates the RequestValidator construct with the given options.
 * However, if you have your RequestValidator already initialized or imported, use the <code>requestValidator</code> option instead.
 * <p>
 * If you want to use <code>requestValidatorOptions</code> in multiple <code>addMethod()</code> calls
 * then you need to set the <code>&#64;aws-cdk/aws-apigateway:requestValidatorUniqueId</code>
 * feature flag. When this feature flag is set, each <code>RequestValidator</code> will have a unique generated id.
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note</strong> if you enable this feature flag when you have already used
 * <code>addMethod()</code> with <code>requestValidatorOptions</code> the Logical Id of the resource
 * will change causing the resource to be replaced.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * LambdaIntegration integration;
 * Resource resource;
 * Model responseModel;
 * Model errorResponseModel;
 * 
 * 
 * resource.node.setContext("&#64;aws-cdk/aws-apigateway:requestValidatorUniqueId", true);
 * 
 * resource.addMethod("GET", integration, MethodOptions.builder()
 *         // we can set request validator options like below
 *         .requestValidatorOptions(RequestValidatorOptions.builder()
 *                 .requestValidatorName("test-validator")
 *                 .validateRequestBody(true)
 *                 .validateRequestParameters(false)
 *                 .build())
 *         .build());
 * 
 * resource.addMethod("POST", integration, MethodOptions.builder()
 *         // we can set request validator options like below
 *         .requestValidatorOptions(RequestValidatorOptions.builder()
 *                 .requestValidatorName("test-validator2")
 *                 .validateRequestBody(true)
 *                 .validateRequestParameters(false)
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Default Integration and Method Options</h2>
 * <p>
 * The <code>defaultIntegration</code> and <code>defaultMethodOptions</code> properties can be used to
 * configure a default integration at any resource level. These options will be
 * used when defining method under this resource (recursively) with undefined
 * integration or options.
 * <p>
 * <blockquote>
 * <p>
 * If not defined, the default integration is <code>MockIntegration</code>. See reference
 * documentation for default method options.
 * <p>
 * </blockquote>
 * <p>
 * The following example defines the <code>booksBackend</code> integration as a default
 * integration. This means that all API methods that do not explicitly define an
 * integration will be routed to this AWS Lambda function.
 * <p>
 * <blockquote><pre>
 * LambdaIntegration booksBackend;
 * 
 * RestApi api = RestApi.Builder.create(this, "books")
 *         .defaultIntegration(booksBackend)
 *         .build();
 * 
 * Resource books = api.root.addResource("books");
 * books.addMethod("GET"); // integrated with `booksBackend`
 * books.addMethod("POST"); // integrated with `booksBackend`
 * 
 * Resource book = books.addResource("{book_id}");
 * book.addMethod("GET");
 * </pre></blockquote>
 * <p>
 * A Method can be configured with authorization scopes. Authorization scopes are
 * used in conjunction with an <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html#apigateway-enable-cognito-user-pool">authorizer that uses Amazon Cognito user
 * pools</a>.
 * Read more about authorization scopes
 * <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationscopes">here</a>.
 * <p>
 * Authorization scopes for a Method can be configured using the <code>authorizationScopes</code> property as shown below -
 * <p>
 * <blockquote><pre>
 * Resource books;
 * 
 * 
 * books.addMethod("GET", new HttpIntegration("http://amazon.com"), MethodOptions.builder()
 *         .authorizationType(AuthorizationType.COGNITO)
 *         .authorizationScopes(List.of("Scope1", "Scope2"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Proxy Routes</h2>
 * <p>
 * The <code>addProxy</code> method can be used to install a greedy <code>{proxy+}</code> resource
 * on a path. By default, this also installs an <code>"ANY"</code> method:
 * <p>
 * <blockquote><pre>
 * Resource resource;
 * Function handler;
 * 
 * ProxyResource proxy = resource.addProxy(ProxyResourceOptions.builder()
 *         .defaultIntegration(new LambdaIntegration(handler))
 * 
 *         // "false" will require explicitly adding methods on the `proxy` resource
 *         .anyMethod(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Authorizers</h2>
 * <p>
 * API Gateway <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-to-api.html">supports several different authorization types</a>
 * that can be used for controlling access to your REST APIs.
 * <p>
 * <h3>IAM-based authorizer</h3>
 * <p>
 * The following CDK code provides 'execute-api' permission to an IAM user, via IAM policies, for the 'GET' method on the <code>books</code> resource:
 * <p>
 * <blockquote><pre>
 * Resource books;
 * User iamUser;
 * 
 * 
 * Method getBooks = books.addMethod("GET", new HttpIntegration("http://amazon.com"), MethodOptions.builder()
 *         .authorizationType(AuthorizationType.IAM)
 *         .build());
 * 
 * iamUser.attachInlinePolicy(Policy.Builder.create(this, "AllowBooks")
 *         .statements(List.of(
 *             PolicyStatement.Builder.create()
 *                     .actions(List.of("execute-api:Invoke"))
 *                     .effect(Effect.ALLOW)
 *                     .resources(List.of(getBooks.getMethodArn()))
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Lambda-based token authorizer</h3>
 * <p>
 * API Gateway also allows <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">lambda functions to be used as authorizers</a>.
 * <p>
 * This module provides support for token-based Lambda authorizers. When a client makes a request to an API's methods configured with such
 * an authorizer, API Gateway calls the Lambda authorizer, which takes the caller's identity as input and returns an IAM policy as output.
 * A token-based Lambda authorizer (also called a token authorizer) receives the caller's identity in a bearer token, such as
 * a JSON Web Token (JWT) or an OAuth token.
 * <p>
 * API Gateway interacts with the authorizer Lambda function handler by passing input and expecting the output in a specific format.
 * The event object that the handler is called with contains the <code>authorizationToken</code> and the <code>methodArn</code> from the request to the
 * API Gateway endpoint. The handler is expected to return the <code>principalId</code> (i.e. the client identifier) and a <code>policyDocument</code> stating
 * what the client is authorizer to perform.
 * See <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">here</a> for a detailed specification on
 * inputs and outputs of the Lambda handler.
 * <p>
 * The following code attaches a token-based Lambda authorizer to the 'GET' Method of the Book resource:
 * <p>
 * <blockquote><pre>
 * Function authFn;
 * Resource books;
 * 
 * 
 * TokenAuthorizer auth = TokenAuthorizer.Builder.create(this, "booksAuthorizer")
 *         .handler(authFn)
 *         .build();
 * 
 * books.addMethod("GET", new HttpIntegration("http://amazon.com"), MethodOptions.builder()
 *         .authorizer(auth)
 *         .build());
 * </pre></blockquote>
 * <p>
 * A full working example is shown below.
 * <p>
 * <blockquote><pre>
 * public class MyStack extends Stack {
 *     public MyStack(Construct scope, String id) {
 *         super(scope, id);
 * 
 *         Function authorizerFn = Function.Builder.create(this, "MyAuthorizerFunction")
 *                 .runtime(Runtime.NODEJS_LATEST)
 *                 .handler("index.handler")
 *                 .code(AssetCode.fromAsset(join(__dirname, "integ.token-authorizer.handler")))
 *                 .build();
 * 
 *         TokenAuthorizer authorizer = TokenAuthorizer.Builder.create(this, "MyAuthorizer")
 *                 .handler(authorizerFn)
 *                 .build();
 * 
 *         RestApi restapi = RestApi.Builder.create(this, "MyRestApi")
 *                 .cloudWatchRole(true)
 *                 .defaultMethodOptions(MethodOptions.builder()
 *                         .authorizer(authorizer)
 *                         .build())
 *                 .defaultCorsPreflightOptions(CorsOptions.builder()
 *                         .allowOrigins(Cors.ALL_ORIGINS)
 *                         .build())
 *                 .build();
 * 
 *         restapi.root.addMethod("ANY", MockIntegration.Builder.create()
 *                 .integrationResponses(List.of(IntegrationResponse.builder().statusCode("200").build()))
 *                 .passthroughBehavior(PassthroughBehavior.NEVER)
 *                 .requestTemplates(Map.of(
 *                         "application/json", "{ \"statusCode\": 200 }"))
 *                 .build(), MethodOptions.builder()
 *                 .methodResponses(List.of(MethodResponse.builder().statusCode("200").build()))
 *                 .build());
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * By default, the <code>TokenAuthorizer</code> looks for the authorization token in the request header with the key 'Authorization'. This can,
 * however, be modified by changing the <code>identitySource</code> property.
 * <p>
 * Authorizers can also be passed via the <code>defaultMethodOptions</code> property within the <code>RestApi</code> construct or the <code>Method</code> construct. Unless
 * explicitly overridden, the specified defaults will be applied across all <code>Method</code>s across the <code>RestApi</code> or across all <code>Resource</code>s,
 * depending on where the defaults were specified.
 * <p>
 * <h3>Lambda-based request authorizer</h3>
 * <p>
 * This module provides support for request-based Lambda authorizers. When a client makes a request to an API's methods configured with such
 * an authorizer, API Gateway calls the Lambda authorizer, which takes specified parts of the request, known as identity sources,
 * as input and returns an IAM policy as output. A request-based Lambda authorizer (also called a request authorizer) receives
 * the identity sources in a series of values pulled from the request, from the headers, stage variables, query strings, and the context.
 * <p>
 * API Gateway interacts with the authorizer Lambda function handler by passing input and expecting the output in a specific format.
 * The event object that the handler is called with contains the body of the request and the <code>methodArn</code> from the request to the
 * API Gateway endpoint. The handler is expected to return the <code>principalId</code> (i.e. the client identifier) and a <code>policyDocument</code> stating
 * what the client is authorizer to perform.
 * See <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">here</a> for a detailed specification on
 * inputs and outputs of the Lambda handler.
 * <p>
 * The following code attaches a request-based Lambda authorizer to the 'GET' Method of the Book resource:
 * <p>
 * <blockquote><pre>
 * Function authFn;
 * Resource books;
 * 
 * 
 * RequestAuthorizer auth = RequestAuthorizer.Builder.create(this, "booksAuthorizer")
 *         .handler(authFn)
 *         .identitySources(List.of(IdentitySource.header("Authorization")))
 *         .build();
 * 
 * books.addMethod("GET", new HttpIntegration("http://amazon.com"), MethodOptions.builder()
 *         .authorizer(auth)
 *         .build());
 * </pre></blockquote>
 * <p>
 * A full working example is shown below.
 * <p>
 * <blockquote><pre>
 * import path.*;
 * import software.amazon.awscdk.services.lambda.*;
 * import software.amazon.awscdk.App;
 * import software.amazon.awscdk.Stack;
 * import software.amazon.awscdk.services.apigateway.MockIntegration;
 * import software.amazon.awscdk.services.apigateway.PassthroughBehavior;
 * import software.amazon.awscdk.services.apigateway.RestApi;
 * import software.amazon.awscdk.services.apigateway.RequestAuthorizer;
 * import software.amazon.awscdk.services.apigateway.IdentitySource;
 * 
 * // Against the RestApi endpoint from the stack output, run
 * // `curl -s -o /dev/null -w "%{http_code}" &lt;url&gt;` should return 401
 * // `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: deny' &lt;url&gt;?allow=yes` should return 403
 * // `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: allow' &lt;url&gt;?allow=yes` should return 200
 * 
 * App app = new App();
 * Stack stack = new Stack(app, "RequestAuthorizerInteg");
 * 
 * Function authorizerFn = Function.Builder.create(stack, "MyAuthorizerFunction")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(AssetCode.fromAsset(join(__dirname, "integ.request-authorizer.handler")))
 *         .build();
 * 
 * RestApi restapi = RestApi.Builder.create(stack, "MyRestApi").cloudWatchRole(true).build();
 * 
 * RequestAuthorizer authorizer = RequestAuthorizer.Builder.create(stack, "MyAuthorizer")
 *         .handler(authorizerFn)
 *         .identitySources(List.of(IdentitySource.header("Authorization"), IdentitySource.queryString("allow")))
 *         .build();
 * 
 * RequestAuthorizer secondAuthorizer = RequestAuthorizer.Builder.create(stack, "MySecondAuthorizer")
 *         .handler(authorizerFn)
 *         .identitySources(List.of(IdentitySource.header("Authorization"), IdentitySource.queryString("allow")))
 *         .build();
 * 
 * restapi.root.addMethod("ANY", MockIntegration.Builder.create()
 *         .integrationResponses(List.of(IntegrationResponse.builder().statusCode("200").build()))
 *         .passthroughBehavior(PassthroughBehavior.NEVER)
 *         .requestTemplates(Map.of(
 *                 "application/json", "{ \"statusCode\": 200 }"))
 *         .build(), MethodOptions.builder()
 *         .methodResponses(List.of(MethodResponse.builder().statusCode("200").build()))
 *         .authorizer(authorizer)
 *         .build());
 * 
 * restapi.root.resourceForPath("auth").addMethod("ANY", MockIntegration.Builder.create()
 *         .integrationResponses(List.of(IntegrationResponse.builder().statusCode("200").build()))
 *         .passthroughBehavior(PassthroughBehavior.NEVER)
 *         .requestTemplates(Map.of(
 *                 "application/json", "{ \"statusCode\": 200 }"))
 *         .build(), MethodOptions.builder()
 *         .methodResponses(List.of(MethodResponse.builder().statusCode("200").build()))
 *         .authorizer(secondAuthorizer)
 *         .build());
 * </pre></blockquote>
 * <p>
 * By default, the <code>RequestAuthorizer</code> does not pass any kind of information from the request. This can,
 * however, be modified by changing the <code>identitySource</code> property, and is required when specifying a value for caching.
 * <p>
 * Authorizers can also be passed via the <code>defaultMethodOptions</code> property within the <code>RestApi</code> construct or the <code>Method</code> construct. Unless
 * explicitly overridden, the specified defaults will be applied across all <code>Method</code>s across the <code>RestApi</code> or across all <code>Resource</code>s,
 * depending on where the defaults were specified.
 * <p>
 * <h3>Cognito User Pools authorizer</h3>
 * <p>
 * API Gateway also allows <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html">Amazon Cognito user pools as authorizer</a>
 * <p>
 * The following snippet configures a Cognito user pool as an authorizer:
 * <p>
 * <blockquote><pre>
 * Resource books;
 * UserPool userPool = new UserPool(this, "UserPool");
 * 
 * CognitoUserPoolsAuthorizer auth = CognitoUserPoolsAuthorizer.Builder.create(this, "booksAuthorizer")
 *         .cognitoUserPools(List.of(userPool))
 *         .build();
 * books.addMethod("GET", new HttpIntegration("http://amazon.com"), MethodOptions.builder()
 *         .authorizer(auth)
 *         .authorizationType(AuthorizationType.COGNITO)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Mutual TLS (mTLS)</h2>
 * <p>
 * Mutual TLS can be configured to limit access to your API based by using client certificates instead of (or as an extension of) using authorization headers.
 * <p>
 * <blockquote><pre>
 * Object acm;
 * 
 * 
 * DomainName.Builder.create(this, "domain-name")
 *         .domainName("example.com")
 *         .certificate(acm.Certificate.fromCertificateArn(this, "cert", "arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d"))
 *         .mtls(MTLSConfig.builder()
 *                 .bucket(new Bucket(this, "bucket"))
 *                 .key("truststore.pem")
 *                 .version("version")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Instructions for configuring your trust store can be found <a href="https://aws.amazon.com/blogs/compute/introducing-mutual-tls-authentication-for-amazon-api-gateway/">here</a>.
 * <p>
 * <h2>Deployments</h2>
 * <p>
 * By default, the <code>RestApi</code> construct will automatically create an API Gateway
 * <a href="https://docs.aws.amazon.com/apigateway/api-reference/resource/deployment/">Deployment</a> and a "prod" <a href="https://docs.aws.amazon.com/apigateway/api-reference/resource/stage/">Stage</a> which represent the API configuration you
 * defined in your CDK app. This means that when you deploy your app, your API will
 * have open access from the internet via the stage URL.
 * <p>
 * The URL of your API can be obtained from the attribute <code>restApi.url</code>, and is
 * also exported as an <code>Output</code> from your stack, so it's printed when you <code>cdk deploy</code> your app:
 * <p>
 * <blockquote><pre>
 * $ cdk deploy
 * ...
 * books.booksapiEndpointE230E8D5 = https://6lyktd4lpk.execute-api.us-east-1.amazonaws.com/prod/
 * </pre></blockquote>
 * <p>
 * To disable this behavior, you can set <code>{ deploy: false }</code> when creating your
 * API. This means that the API will not be deployed and a stage will not be
 * created for it. You will need to manually define a <code>apigateway.Deployment</code> and
 * <code>apigateway.Stage</code> resources.
 * <p>
 * Use the <code>deployOptions</code> property to customize the deployment options of your
 * API.
 * <p>
 * The following example will configure API Gateway to emit logs and data traces to
 * AWS CloudWatch for all API calls:
 * <p>
 * <blockquote>
 * <p>
 * Note: whether or not this is enabled or disabled by default is controlled by the
 * <code>&#64;aws-cdk/aws-apigateway:disableCloudWatchRole</code> feature flag. When this feature flag
 * is set to <code>false</code> the default behavior will set <code>cloudWatchRole=true</code>
 * <p>
 * </blockquote>
 * <p>
 * This is controlled via the <code>&#64;aws-cdk/aws-apigateway:disableCloudWatchRole</code> feature flag and
 * is disabled by default. When enabled (or <code>&#64;aws-cdk/aws-apigateway:disableCloudWatchRole=false</code>),
 * an IAM role will be created and associated with API Gateway to allow it to write logs and metrics to AWS CloudWatch.
 * <p>
 * <blockquote><pre>
 * RestApi api = RestApi.Builder.create(this, "books")
 *         .cloudWatchRole(true)
 *         .deployOptions(StageOptions.builder()
 *                 .loggingLevel(MethodLoggingLevel.INFO)
 *                 .dataTraceEnabled(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: there can only be a single apigateway.CfnAccount per AWS environment
 * so if you create multiple <code>RestApi</code>s with <code>cloudWatchRole=true</code> each new <code>RestApi</code>
 * will overwrite the <code>CfnAccount</code>. It is recommended to set <code>cloudWatchRole=false</code>
 * (the default behavior if <code>&#64;aws-cdk/aws-apigateway:disableCloudWatchRole</code> is enabled)
 * and only create a single CloudWatch role and account per environment.
 * <p>
 * </blockquote>
 * <p>
 * You can specify the CloudWatch Role and Account sub-resources removal policy with the
 * <code>cloudWatchRoleRemovalPolicy</code> property, which defaults to <code>RemovalPolicy.RETAIN</code>.
 * This option requires <code>cloudWatchRole</code> to be enabled.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * RestApi api = RestApi.Builder.create(this, "books")
 *         .cloudWatchRole(true)
 *         .cloudWatchRoleRemovalPolicy(RemovalPolicy.DESTROY)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deploying to an existing stage</h3>
 * <p>
 * <h4>Using RestApi</h4>
 * <p>
 * If you want to use an existing stage to deploy your <code>RestApi</code>, first set <code>{ deploy: false }</code> so the construct doesn't automatically create new <code>Deployment</code> and <code>Stage</code> resources.  Then you can manually define a <code>apigateway.Deployment</code> resource and specify the stage name for your existing stage using the <code>stageName</code> property.
 * <p>
 * Note that as long as the deployment's logical ID doesn't change, it will represent the snapshot in time when the resource was created. To ensure your deployment reflects changes to the <code>RestApi</code> model, see <a href="#controlled-triggering-of-deployments">Controlled triggering of deployments</a>.
 * <p>
 * <blockquote><pre>
 * RestApi restApi = RestApi.Builder.create(this, "my-rest-api")
 *         .deploy(false)
 *         .build();
 * 
 * // Use `stageName` to deploy to an existing stage
 * Deployment deployment = Deployment.Builder.create(this, "my-deployment")
 *         .api(restApi)
 *         .stageName("dev")
 *         .retainDeployments(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Using SpecRestApi</h4>
 * <p>
 * If you want to use an existing stage to deploy your <code>SpecRestApi</code>, first set <code>{ deploy: false }</code> so the construct doesn't automatically create new <code>Deployment</code> and <code>Stage</code> resources. Then you can manually define a <code>apigateway.Deployment</code> resource and specify the stage name for your existing stage using the <code>stageName</code> property.
 * <p>
 * To automatically create a new deployment that reflects the latest API changes, you can use the <code>addToLogicalId()</code> method and pass in your OpenAPI definition.
 * <p>
 * <blockquote><pre>
 * AssetApiDefinition myApiDefinition = ApiDefinition.fromAsset("path-to-file.json");
 * SpecRestApi specRestApi = SpecRestApi.Builder.create(this, "my-specrest-api")
 *         .deploy(false)
 *         .apiDefinition(myApiDefinition)
 *         .build();
 * 
 * // Use `stageName` to deploy to an existing stage
 * Deployment deployment = Deployment.Builder.create(this, "my-deployment")
 *         .api(specRestApi)
 *         .stageName("dev")
 *         .retainDeployments(true)
 *         .build();
 * 
 * // Trigger a new deployment on OpenAPI definition updates
 * deployment.addToLogicalId(myApiDefinition);
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: If the <code>stageName</code> property is set but a stage with the corresponding name does not exist, a new stage resource will be created with the provided stage name.
 * <p>
 * </blockquote><blockquote>
 * <p>
 * Note: If you update the <code>stageName</code> property, you should be triggering a new deployment (i.e. with an updated logical ID and API changes). Otherwise, an error will occur during deployment.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Controlled triggering of deployments</h3>
 * <p>
 * By default, the <code>RestApi</code> construct deploys changes immediately. If you want to
 * control when deployments happen, set <code>{ deploy: false }</code> and create a <code>Deployment</code> construct yourself. Add a revision counter to the construct ID, and update it in your source code whenever you want to trigger a new deployment:
 * <p>
 * <blockquote><pre>
 * RestApi restApi = RestApi.Builder.create(this, "my-api")
 *         .deploy(false)
 *         .build();
 * 
 * Number deploymentRevision = 5; // Bump this counter to trigger a new deployment
 *  // Bump this counter to trigger a new deployment
 * Deployment.Builder.create(this, String.format("Deployment%s", deploymentRevision))
 *         .api(restApi)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Deep dive: Invalidation of deployments</h3>
 * <p>
 * API Gateway deployments are an immutable snapshot of the API. This means that we
 * want to automatically create a new deployment resource every time the API model
 * defined in our CDK app changes.
 * <p>
 * In order to achieve that, the AWS CloudFormation logical ID of the
 * <code>AWS::ApiGateway::Deployment</code> resource is dynamically calculated by hashing the
 * API configuration (resources, methods). This means that when the configuration
 * changes (i.e. a resource or method are added, configuration is changed), a new
 * logical ID will be assigned to the deployment resource. This will cause
 * CloudFormation to create a new deployment resource.
 * <p>
 * By default, old deployments are <em>deleted</em>. You can set <code>retainDeployments: true</code>
 * to allow users revert the stage to an old deployment manually.
 * <p>
 * In order to also create a new deployment when changes are made to any authorizer attached to the API,
 * the <code>&#64;aws-cdk/aws-apigateway:authorizerChangeDeploymentLogicalId</code> <a href="https://docs.aws.amazon.com/cdk/v2/guide/featureflags.html">feature flag</a> can be enabled. This can be set
 * in the <code>cdk.json</code> file.
 * <p>
 * <blockquote><pre>
 * {
 *   "context": {
 *     "&#64;aws-cdk/aws-apigateway:authorizerChangeDeploymentLogicalId": true
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h2>Custom Domains</h2>
 * <p>
 * To associate an API with a custom domain, use the <code>domainName</code> configuration when
 * you define your API:
 * <p>
 * <blockquote><pre>
 * Object acmCertificateForExampleCom;
 * 
 * 
 * RestApi api = RestApi.Builder.create(this, "MyDomain")
 *         .domainName(DomainNameOptions.builder()
 *                 .domainName("example.com")
 *                 .certificate(acmCertificateForExampleCom)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will define a <code>DomainName</code> resource for you, along with a <code>BasePathMapping</code>
 * from the root of the domain to the deployment stage of the API. This is a common
 * set up.
 * <p>
 * To route domain traffic to an API Gateway API, use Amazon Route 53 to create an
 * alias record. An alias record is a Route 53 extension to DNS. It's similar to a
 * CNAME record, but you can create an alias record both for the root domain, such
 * as <code>example.com</code>, and for subdomains, such as <code>www.example.com</code>. (You can create
 * CNAME records only for subdomains.)
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.route53.*;
 * import software.amazon.awscdk.services.route53.targets.*;
 * 
 * RestApi api;
 * Object hostedZoneForExampleCom;
 * 
 * 
 * ARecord.Builder.create(this, "CustomDomainAliasRecord")
 *         .zone(hostedZoneForExampleCom)
 *         .target(RecordTarget.fromAlias(new ApiGateway(api)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also define a <code>DomainName</code> resource directly in order to customize the default behavior:
 * <p>
 * <blockquote><pre>
 * Object acmCertificateForExampleCom;
 * 
 * 
 * DomainName.Builder.create(this, "custom-domain")
 *         .domainName("example.com")
 *         .certificate(acmCertificateForExampleCom)
 *         .endpointType(EndpointType.EDGE) // default is REGIONAL
 *         .securityPolicy(SecurityPolicy.TLS_1_2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Once you have a domain, you can map base paths of the domain to APIs.
 * The following example will map the URL <a href="https://example.com/go-to-api1">https://example.com/go-to-api1</a>
 * to the <code>api1</code> API and <a href="https://example.com/boom">https://example.com/boom</a> to the <code>api2</code> API.
 * <p>
 * <blockquote><pre>
 * DomainName domain;
 * RestApi api1;
 * RestApi api2;
 * 
 * 
 * domain.addBasePathMapping(api1, BasePathMappingOptions.builder().basePath("go-to-api1").build());
 * domain.addBasePathMapping(api2, BasePathMappingOptions.builder().basePath("boom").build());
 * </pre></blockquote>
 * <p>
 * By default, the base path URL will map to the <code>deploymentStage</code> of the <code>RestApi</code>.
 * You can specify a different API <code>Stage</code> to which the base path URL will map to.
 * <p>
 * <blockquote><pre>
 * DomainName domain;
 * RestApi restapi;
 * 
 * 
 * Deployment betaDeploy = Deployment.Builder.create(this, "beta-deployment")
 *         .api(restapi)
 *         .build();
 * Stage betaStage = Stage.Builder.create(this, "beta-stage")
 *         .deployment(betaDeploy)
 *         .build();
 * domain.addBasePathMapping(restapi, BasePathMappingOptions.builder().basePath("api/beta").stage(betaStage).build());
 * </pre></blockquote>
 * <p>
 * It is possible to create a base path mapping without associating it with a
 * stage by using the <code>attachToStage</code> property. When set to <code>false</code>, the stage must be
 * included in the URL when invoking the API. For example,
 * <a href="https://example.com/myapi/prod">https://example.com/myapi/prod</a> will invoke the stage named <code>prod</code> from the
 * <code>myapi</code> base path mapping.
 * <p>
 * <blockquote><pre>
 * DomainName domain;
 * RestApi api;
 * 
 * 
 * domain.addBasePathMapping(api, BasePathMappingOptions.builder().basePath("myapi").attachToStage(false).build());
 * </pre></blockquote>
 * <p>
 * If you don't specify <code>basePath</code>, all URLs under this domain will be mapped
 * to the API, and you won't be able to map another API to the same domain:
 * <p>
 * <blockquote><pre>
 * DomainName domain;
 * RestApi api;
 * 
 * domain.addBasePathMapping(api);
 * </pre></blockquote>
 * <p>
 * This can also be achieved through the <code>mapping</code> configuration when defining the
 * domain as demonstrated above.
 * <p>
 * Base path mappings can also be created with the <code>BasePathMapping</code> resource.
 * <p>
 * <blockquote><pre>
 * RestApi api;
 * 
 * 
 * IDomainName domainName = DomainName.fromDomainNameAttributes(this, "DomainName", DomainNameAttributes.builder()
 *         .domainName("domainName")
 *         .domainNameAliasHostedZoneId("domainNameAliasHostedZoneId")
 *         .domainNameAliasTarget("domainNameAliasTarget")
 *         .build());
 * 
 * BasePathMapping.Builder.create(this, "BasePathMapping")
 *         .domainName(domainName)
 *         .restApi(api)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you wish to setup this domain with an Amazon Route53 alias, use the <code>targets.ApiGatewayDomain</code>:
 * <p>
 * <blockquote><pre>
 * Object hostedZoneForExampleCom;
 * DomainName domainName;
 * 
 * import software.amazon.awscdk.services.route53.*;
 * import software.amazon.awscdk.services.route53.targets.*;
 * 
 * 
 * ARecord.Builder.create(this, "CustomDomainAliasRecord")
 *         .zone(hostedZoneForExampleCom)
 *         .target(RecordTarget.fromAlias(new ApiGatewayDomain(domainName)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Custom Domains with multi-level api mapping</h3>
 * <p>
 * Additional requirements for creating multi-level path mappings for RestApis:
 * <p>
 * (both are defaults)
 * <p>
 * <ul>
 * <li>Must use <code>SecurityPolicy.TLS_1_2</code></li>
 * <li>DomainNames must be <code>EndpointType.REGIONAL</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Object acmCertificateForExampleCom;
 * RestApi restApi;
 * 
 * 
 * DomainName.Builder.create(this, "custom-domain")
 *         .domainName("example.com")
 *         .certificate(acmCertificateForExampleCom)
 *         .mapping(restApi)
 *         .basePath("orders/v1/api")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To then add additional mappings to a domain you can use the <code>addApiMapping</code> method.
 * <p>
 * <blockquote><pre>
 * Object acmCertificateForExampleCom;
 * RestApi restApi;
 * RestApi secondRestApi;
 * 
 * 
 * DomainName domain = DomainName.Builder.create(this, "custom-domain")
 *         .domainName("example.com")
 *         .certificate(acmCertificateForExampleCom)
 *         .mapping(restApi)
 *         .build();
 * 
 * domain.addApiMapping(secondRestApi.getDeploymentStage(), ApiMappingOptions.builder()
 *         .basePath("orders/v2/api")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Access Logging</h2>
 * <p>
 * Access logging creates logs every time an API method is accessed. Access logs can have information on
 * who has accessed the API, how the caller accessed the API and what responses were generated.
 * Access logs are configured on a Stage of the RestApi.
 * Access logs can be expressed in a format of your choosing, and can contain any access details, with a
 * minimum that it must include either 'requestId' or 'extendedRequestId'. The list of  variables that
 * can be expressed in the access log can be found
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference">here</a>.
 * Read more at <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/set-up-logging.html">Setting Up CloudWatch API Logging in API
 * Gateway</a>
 * <p>
 * <blockquote><pre>
 * // production stage
 * LogGroup prodLogGroup = new LogGroup(this, "PrdLogs");
 * RestApi api = RestApi.Builder.create(this, "books")
 *         .deployOptions(StageOptions.builder()
 *                 .accessLogDestination(new LogGroupLogDestination(prodLogGroup))
 *                 .accessLogFormat(AccessLogFormat.jsonWithStandardFields())
 *                 .build())
 *         .build();
 * Deployment deployment = Deployment.Builder.create(this, "Deployment").api(api).build();
 * 
 * // development stage
 * LogGroup devLogGroup = new LogGroup(this, "DevLogs");
 * Stage.Builder.create(this, "dev")
 *         .deployment(deployment)
 *         .accessLogDestination(new LogGroupLogDestination(devLogGroup))
 *         .accessLogFormat(AccessLogFormat.jsonWithStandardFields(JsonWithStandardFieldProps.builder()
 *                 .caller(false)
 *                 .httpMethod(true)
 *                 .ip(true)
 *                 .protocol(true)
 *                 .requestTime(true)
 *                 .resourcePath(true)
 *                 .responseLength(true)
 *                 .status(true)
 *                 .user(true)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following code will generate the access log in the <a href="https://en.wikipedia.org/wiki/Common_Log_Format">CLF format</a>.
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup = new LogGroup(this, "ApiGatewayAccessLogs");
 * RestApi api = RestApi.Builder.create(this, "books")
 *         .deployOptions(StageOptions.builder()
 *                 .accessLogDestination(new LogGroupLogDestination(logGroup))
 *                 .accessLogFormat(AccessLogFormat.clf())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also configure your own access log format by using the <code>AccessLogFormat.custom()</code> API.
 * <code>AccessLogField</code> provides commonly used fields. The following code configures access log to contain.
 * <p>
 * <blockquote><pre>
 * LogGroup logGroup = new LogGroup(this, "ApiGatewayAccessLogs");
 * RestApi.Builder.create(this, "books")
 *         .deployOptions(StageOptions.builder()
 *                 .accessLogDestination(new LogGroupLogDestination(logGroup))
 *                 .accessLogFormat(AccessLogFormat.custom(String.format("%s %s %s%n      %s %s", AccessLogField.contextRequestId(), AccessLogField.contextErrorMessage(), AccessLogField.contextErrorMessageString(), AccessLogField.contextAuthorizerError(), AccessLogField.contextAuthorizerIntegrationStatus())))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can use the <code>methodOptions</code> property to configure
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html#apigateway-api-level-throttling-in-usage-plan">default method throttling</a>
 * for a stage. The following snippet configures the a stage that accepts
 * 100 requests per minute, allowing burst up to 200 requests per minute.
 * <p>
 * <blockquote><pre>
 * RestApi api = new RestApi(this, "books");
 * Deployment deployment = Deployment.Builder.create(this, "my-deployment").api(api).build();
 * Stage stage = Stage.Builder.create(this, "my-stage")
 *         .deployment(deployment)
 *         .methodOptions(Map.of(
 *                 "/*&#47;*", MethodDeploymentOptions.builder() // This special path applies to all resource paths and all HTTP methods
 *                         .throttlingRateLimit(100)
 *                         .throttlingBurstLimit(200).build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Configuring <code>methodOptions</code> on the <code>deployOptions</code> of <code>RestApi</code> will set the
 * throttling behaviors on the default stage that is automatically created.
 * <p>
 * <blockquote><pre>
 * RestApi api = RestApi.Builder.create(this, "books")
 *         .deployOptions(StageOptions.builder()
 *                 .methodOptions(Map.of(
 *                         "/*&#47;*", MethodDeploymentOptions.builder() // This special path applies to all resource paths and all HTTP methods
 *                                 .throttlingRateLimit(100)
 *                                 .throttlingBurstLimit(1000).build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * To write access log files to a Firehose delivery stream destination use the <code>FirehoseLogDestination</code> class:
 * <p>
 * <blockquote><pre>
 * Bucket destinationBucket = new Bucket(this, "Bucket");
 * Role deliveryStreamRole = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("firehose.amazonaws.com"))
 *         .build();
 * 
 * CfnDeliveryStream stream = CfnDeliveryStream.Builder.create(this, "MyStream")
 *         .deliveryStreamName("amazon-apigateway-delivery-stream")
 *         .s3DestinationConfiguration(S3DestinationConfigurationProperty.builder()
 *                 .bucketArn(destinationBucket.getBucketArn())
 *                 .roleArn(deliveryStreamRole.getRoleArn())
 *                 .build())
 *         .build();
 * 
 * RestApi api = RestApi.Builder.create(this, "books")
 *         .deployOptions(StageOptions.builder()
 *                 .accessLogDestination(new FirehoseLogDestination(stream))
 *                 .accessLogFormat(AccessLogFormat.jsonWithStandardFields())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note:</strong> The delivery stream name must start with <code>amazon-apigateway-</code>.
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-logging-to-kinesis.html">Logging API calls to Amazon Data Firehose</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Cross Origin Resource Sharing (CORS)</h2>
 * <p>
 * <a href="https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS">Cross-Origin Resource Sharing (CORS)</a> is a mechanism
 * that uses additional HTTP headers to tell browsers to give a web application
 * running at one origin, access to selected resources from a different origin. A
 * web application executes a cross-origin HTTP request when it requests a resource
 * that has a different origin (domain, protocol, or port) from its own.
 * <p>
 * You can add the CORS <a href="https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS#Preflighted_requests">preflight</a> OPTIONS
 * HTTP method to any API resource via the <code>defaultCorsPreflightOptions</code> option or by calling the <code>addCorsPreflight</code> on a specific resource.
 * <p>
 * The following example will enable CORS for all methods and all origins on all resources of the API:
 * <p>
 * <blockquote><pre>
 * RestApi.Builder.create(this, "api")
 *         .defaultCorsPreflightOptions(CorsOptions.builder()
 *                 .allowOrigins(Cors.ALL_ORIGINS)
 *                 .allowMethods(Cors.ALL_METHODS)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following example will add an OPTIONS method to the <code>myResource</code> API resource, which
 * only allows GET and PUT HTTP requests from the origin <a href="https://amazon.com.">https://amazon.com.</a>
 * <p>
 * <blockquote><pre>
 * Resource myResource;
 * 
 * 
 * myResource.addCorsPreflight(CorsOptions.builder()
 *         .allowOrigins(List.of("https://amazon.com"))
 *         .allowMethods(List.of("GET", "PUT"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * See the
 * <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/&#64;aws-cdk_aws-apigateway.CorsOptions.html"><code>CorsOptions</code></a>
 * API reference for a detailed list of supported configuration options.
 * <p>
 * You can specify defaults this at the resource level, in which case they will be applied to the entire resource sub-tree:
 * <p>
 * <blockquote><pre>
 * Resource resource;
 * 
 * 
 * Resource subtree = resource.addResource("subtree", ResourceOptions.builder()
 *         .defaultCorsPreflightOptions(CorsOptions.builder()
 *                 .allowOrigins(List.of("https://amazon.com"))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * This means that all resources under <code>subtree</code> (inclusive) will have a preflight
 * OPTIONS added to them.
 * <p>
 * See <a href="https://github.com/aws/aws-cdk/issues/906">#906</a> for a list of CORS
 * features which are not yet supported.
 * <p>
 * <h2>Endpoint Configuration</h2>
 * <p>
 * API gateway allows you to specify an
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-endpoint-types.html">API Endpoint Type</a>.
 * To define an endpoint type for the API gateway, use <code>endpointConfiguration</code> property:
 * <p>
 * <blockquote><pre>
 * RestApi api = RestApi.Builder.create(this, "api")
 *         .endpointConfiguration(EndpointConfiguration.builder()
 *                 .types(List.of(EndpointType.EDGE))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also configure <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-ip-address-type.html">endpoint IP address type</a>.
 * The default value is <code>IpAddressType.DUAL_STACK</code> for private API, and <code>IpAddressType.IPV4</code> for regional and edge-optimized API.
 * <p>
 * <blockquote><pre>
 * RestApi api = RestApi.Builder.create(this, "api")
 *         .endpointConfiguration(EndpointConfiguration.builder()
 *                 .types(List.of(EndpointType.REGIONAL))
 *                 .ipAddressType(IpAddressType.DUAL_STACK)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: If creating a private API, the <code>IPV4</code> IP address type is not supported.
 * <p>
 * You can also create an association between your Rest API and a VPC endpoint. By doing so,
 * API Gateway will generate a new
 * Route53 Alias DNS record which you can use to invoke your private APIs. More info can be found
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/associate-private-api-with-vpc-endpoint.html">here</a>.
 * <p>
 * Here is an example:
 * <p>
 * <blockquote><pre>
 * IVpcEndpoint someEndpoint;
 * 
 * 
 * RestApi api = RestApi.Builder.create(this, "api")
 *         .endpointConfiguration(EndpointConfiguration.builder()
 *                 .types(List.of(EndpointType.PRIVATE))
 *                 .vpcEndpoints(List.of(someEndpoint))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * By performing this association, we can invoke the API gateway using the following format:
 * <p>
 * <blockquote><pre>
 * https://{rest-api-id}-{vpce-id}.execute-api.{region}.amazonaws.com/{stage}
 * </pre></blockquote>
 * <p>
 * To restrict access to the API Gateway to only the VPC endpoint, you can use the <code>grantInvokeFromVpcEndpointsOnly</code> method to <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/private-api-tutorial.html#private-api-tutorial-attach-resource-policy">add resource policies</a> to the API Gateway:
 * <p>
 * <blockquote><pre>
 * IVpcEndpoint apiGwVpcEndpoint;
 * 
 * 
 * RestApi api = RestApi.Builder.create(this, "PrivateApi")
 *         .endpointConfiguration(EndpointConfiguration.builder()
 *                 .types(List.of(EndpointType.PRIVATE))
 *                 .vpcEndpoints(List.of(apiGwVpcEndpoint))
 *                 .build())
 *         .build();
 * api.grantInvokeFromVpcEndpointsOnly(List.of(apiGwVpcEndpoint));
 * </pre></blockquote>
 * <p>
 * <h2>Private Integrations</h2>
 * <p>
 * A private integration makes it simple to expose HTTP/HTTPS resources behind an
 * Amazon VPC for access by clients outside of the VPC. The private integration uses
 * an API Gateway resource of <code>VpcLink</code> to encapsulate connections between API
 * Gateway and targeted VPC resources.
 * The <code>VpcLink</code> is then attached to the <code>Integration</code> of a specific API Gateway
 * Method. The following code sets up a private integration with a network load
 * balancer -
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * 
 * Vpc vpc = new Vpc(this, "VPC");
 * NetworkLoadBalancer nlb = NetworkLoadBalancer.Builder.create(this, "NLB")
 *         .vpc(vpc)
 *         .build();
 * VpcLink link = VpcLink.Builder.create(this, "link")
 *         .targets(List.of(nlb))
 *         .build();
 * 
 * Integration integration = Integration.Builder.create()
 *         .type(IntegrationType.HTTP_PROXY)
 *         .integrationHttpMethod("ANY")
 *         .options(IntegrationOptions.builder()
 *                 .connectionType(ConnectionType.VPC_LINK)
 *                 .vpcLink(link)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The uri for the private integration, in the case of a VpcLink, will be set to the DNS name of
 * the VPC Link's NLB. If the VPC Link has multiple NLBs or the VPC Link is imported or the DNS
 * name cannot be determined for any other reason, the user is expected to specify the <code>uri</code>
 * property.
 * <p>
 * Any existing <code>VpcLink</code> resource can be imported into the CDK app via the <code>VpcLink.fromVpcLinkId()</code>.
 * <p>
 * <blockquote><pre>
 * IVpcLink awesomeLink = VpcLink.fromVpcLinkId(this, "awesome-vpc-link", "us-east-1_oiuR12Abd");
 * </pre></blockquote>
 * <p>
 * <h2>Gateway response</h2>
 * <p>
 * If the Rest API fails to process an incoming request, it returns to the client an error response without forwarding the
 * request to the integration backend. API Gateway has a set of standard response messages that are sent to the client for
 * each type of error. These error responses can be configured on the Rest API. The list of Gateway responses that can be
 * configured can be found <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html">here</a>.
 * Learn more about <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-gatewayResponse-definition.html">Gateway
 * Responses</a>.
 * <p>
 * The following code configures a Gateway Response when the response is 'access denied':
 * <p>
 * <blockquote><pre>
 * RestApi api = new RestApi(this, "books-api");
 * api.addGatewayResponse("test-response", GatewayResponseOptions.builder()
 *         .type(ResponseType.ACCESS_DENIED)
 *         .statusCode("500")
 *         .responseHeaders(Map.of(
 *                 // Note that values must be enclosed within a pair of single quotes
 *                 "Access-Control-Allow-Origin", "'test.com'",
 *                 "test-key", "'test-value'"))
 *         .templates(Map.of(
 *                 "application/json", "{ \"message\": $context.error.messageString, \"statusCode\": \"488\", \"type\": \"$context.error.responseType\" }"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>OpenAPI Definition</h2>
 * <p>
 * CDK supports creating a REST API by importing an OpenAPI definition file. It currently supports OpenAPI v2.0 and OpenAPI
 * v3.0 definition files. Read more about <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-import-api.html">Configuring a REST API using
 * OpenAPI</a>.
 * <p>
 * The following code creates a REST API using an external OpenAPI definition JSON file -
 * <p>
 * <blockquote><pre>
 * Integration integration;
 * 
 * 
 * SpecRestApi api = SpecRestApi.Builder.create(this, "books-api")
 *         .apiDefinition(ApiDefinition.fromAsset("path-to-file.json"))
 *         .build();
 * 
 * Resource booksResource = api.root.addResource("books");
 * booksResource.addMethod("GET", integration);
 * </pre></blockquote>
 * <p>
 * It is possible to use the <code>addResource()</code> API to define additional API Gateway Resources.
 * <p>
 * <strong>Note:</strong> Deployment will fail if a Resource of the same name is already defined in the Open API specification.
 * <p>
 * <strong>Note:</strong> Any default properties configured, such as <code>defaultIntegration</code>, <code>defaultMethodOptions</code>, etc. will only be
 * applied to Resources and Methods defined in the CDK, and not the ones defined in the spec. Use the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions.html">API Gateway
 * extensions to OpenAPI</a>
 * to configure these.
 * <p>
 * There are a number of limitations in using OpenAPI definitions in API Gateway. Read the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-known-issues.html#api-gateway-known-issues-rest-apis">Amazon API Gateway important
 * notes for REST APIs</a>
 * for more details.
 * <p>
 * <strong>Note:</strong> When starting off with an OpenAPI definition using <code>SpecRestApi</code>, it is not possible to configure some
 * properties that can be configured directly in the OpenAPI specification file. This is to prevent people duplication
 * of these properties and potential confusion.
 * <p>
 * However, you can control how API Gateway handles resource updates using the <code>mode</code> property. Valid values are:
 * <p>
 * <ul>
 * <li><code>overwrite</code> - The new API definition replaces the existing one. The existing API identifier remains unchanged.</li>
 * <li><code>merge</code> - The new API definition is merged with the existing API.</li>
 * </ul>
 * <p>
 * If you don't specify this property, a default value is chosen:
 * <p>
 * <ul>
 * <li>For REST APIs created before March 29, 2021, the default is <code>overwrite</code></li>
 * <li>For REST APIs created after March 29, 2021, the new API definition takes precedence, but any container types such as endpoint configurations and binary media types are merged with the existing API.</li>
 * </ul>
 * <p>
 * Use the default mode to define top-level RestApi properties in addition to using OpenAPI.
 * Generally, it's preferred to use API Gateway's OpenAPI extensions to model these properties.
 * <p>
 * <blockquote><pre>
 * SpecRestApi api = SpecRestApi.Builder.create(this, "books-api")
 *         .apiDefinition(ApiDefinition.fromAsset("path-to-file.json"))
 *         .mode(RestApiMode.MERGE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <code>SpecRestApi</code> also supports binary media types, similar to <code>RestApi</code>:
 * <p>
 * <blockquote><pre>
 * SpecRestApi api = SpecRestApi.Builder.create(this, "books-api")
 *         .apiDefinition(ApiDefinition.fromAsset("path-to-file.json"))
 *         .binaryMediaTypes(List.of("image/png", "application/pdf"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Endpoint configuration</h3>
 * <p>
 * By default, <code>SpecRestApi</code> will create an edge optimized endpoint.
 * <p>
 * This can be modified as shown below:
 * <p>
 * <blockquote><pre>
 * ApiDefinition apiDefinition;
 * 
 * 
 * SpecRestApi api = SpecRestApi.Builder.create(this, "ExampleRestApi")
 *         .apiDefinition(apiDefinition)
 *         .endpointTypes(List.of(EndpointType.PRIVATE))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note:</strong> For private endpoints you will still need to provide the
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/openapi-extensions-policy.html"><code>x-amazon-apigateway-policy</code></a> and
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-endpoint-configuration.html"><code>x-amazon-apigateway-endpoint-configuration</code></a>
 * in your openApi file.
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * The API Gateway service sends metrics around the performance of Rest APIs to Amazon CloudWatch.
 * These metrics can be referred to using the metric APIs available on the <code>RestApi</code>, <code>Stage</code> and <code>Method</code> constructs.
 * Note that detailed metrics must be enabled for a stage to use the <code>Method</code> metrics.
 * Read more about <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-metrics-and-dimensions.html">API Gateway metrics</a>, including enabling detailed metrics.
 * The APIs with the <code>metric</code> prefix can be used to get reference to specific metrics for this API. For example:
 * <p>
 * <blockquote><pre>
 * RestApi api = new RestApi(this, "my-api");
 * Stage stage = api.getDeploymentStage();
 * Method method = api.root.addMethod("GET");
 * 
 * Metric clientErrorApiMetric = api.metricClientError();
 * Metric serverErrorStageMetric = stage.metricServerError();
 * Metric latencyMethodMetric = method.metricLatency(stage);
 * </pre></blockquote>
 * <p>
 * <h2>APIGateway v2</h2>
 * <p>
 * APIGateway v2 APIs are now moved to its own package named <code>aws-apigatewayv2</code>. Previously, these APIs were marked
 * deprecated but retained for backwards compatibility. The deprecated usage of APIGateway v2 APIs within this module
 * <code>aws-apigateway</code> has now been removed from the codebase.
 * <p>
 * The reason for the removal of these deprecated Constructs is that CloudFormation team is releasing AWS resources
 * like <code>AWS::APIGateway::DomainNameV2</code> and this would cause compatibility issue with the deprecated <code>CfnDomainNameV2</code>
 * resource defined in <code>apigatewayv2.ts</code> file during the L1 generation.
 * <p>
 * Move to using <code>aws-apigatewayv2</code> to get the latest APIs and updates.
 * <p>
 * <hr>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 */
package software.amazon.awscdk.services.apigateway;
