/**
 * <h1>AWS Certificate Manager Construct Library</h1>
 * <p>
 * AWS Certificate Manager (ACM) handles the complexity of creating, storing, and renewing public and private SSL/TLS X.509 certificates and keys that
 * protect your AWS websites and applications. ACM certificates can secure singular domain names, multiple specific domain names, wildcard domains, or
 * combinations of these. ACM wildcard certificates can protect an unlimited number of subdomains.
 * <p>
 * This package provides Constructs for provisioning and referencing ACM certificates which can be used with CloudFront and ELB.
 * <p>
 * After requesting a certificate, you will need to prove that you own the
 * domain in question before the certificate will be granted. The CloudFormation
 * deployment will wait until this verification process has been completed.
 * <p>
 * Because of this wait time, when using manual validation methods, it's better
 * to provision your certificates either in a separate stack from your main
 * service, or provision them manually and import them into your CDK application.
 * <p>
 * <strong>Note:</strong> There is a limit on total number of ACM certificates that can be requested on an account and region within a year.
 * The default limit is 2000, but this limit may be (much) lower on new AWS accounts.
 * See https://docs.aws.amazon.com/acm/latest/userguide/acm-limits.html for more information.
 * <p>
 * <h2>DNS validation</h2>
 * <p>
 * DNS validation is the preferred method to validate domain ownership, as it has a number of advantages over email validation.
 * See also <a href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html">Validate with DNS</a>
 * in the AWS Certificate Manager User Guide.
 * <p>
 * If Amazon Route 53 is your DNS provider for the requested domain, the DNS record can be
 * created automatically:
 * <p>
 * <blockquote><pre>
 * HostedZone myHostedZone = HostedZone.Builder.create(this, "HostedZone")
 *         .zoneName("example.com")
 *         .build();
 * Certificate.Builder.create(this, "Certificate")
 *         .domainName("hello.example.com")
 *         .certificateName("Hello World Service") // Optionally provide an certificate name
 *         .validation(CertificateValidation.fromDns(myHostedZone))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If Route 53 is not your DNS provider, the DNS records must be added manually and the stack will not complete
 * creating until the records are added.
 * <p>
 * <blockquote><pre>
 * Certificate.Builder.create(this, "Certificate")
 *         .domainName("hello.example.com")
 *         .validation(CertificateValidation.fromDns())
 *         .build();
 * </pre></blockquote>
 * <p>
 * When working with multiple domains, use the <code>CertificateValidation.fromDnsMultiZone()</code>:
 * <p>
 * <blockquote><pre>
 * HostedZone exampleCom = HostedZone.Builder.create(this, "ExampleCom")
 *         .zoneName("example.com")
 *         .build();
 * HostedZone exampleNet = HostedZone.Builder.create(this, "ExampleNet")
 *         .zoneName("example.net")
 *         .build();
 * 
 * Certificate cert = Certificate.Builder.create(this, "Certificate")
 *         .domainName("test.example.com")
 *         .subjectAlternativeNames(List.of("cool.example.com", "test.example.net"))
 *         .validation(CertificateValidation.fromDnsMultiZone(Map.of(
 *                 "test.example.com", exampleCom,
 *                 "cool.example.com", exampleCom,
 *                 "test.example.net", exampleNet)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Email validation</h2>
 * <p>
 * Email-validated certificates (the default) are validated by receiving an
 * email on one of a number of predefined domains and following the instructions
 * in the email.
 * <p>
 * See <a href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html">Validate with Email</a>
 * in the AWS Certificate Manager User Guide.
 * <p>
 * <blockquote><pre>
 * Certificate.Builder.create(this, "Certificate")
 *         .domainName("hello.example.com")
 *         .validation(CertificateValidation.fromEmail())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Cross-region Certificates</h2>
 * <p>
 * ACM certificates that are used with CloudFront -- or higher-level constructs which rely on CloudFront -- must be in the <code>us-east-1</code> region.
 * CloudFormation allows you to create a Stack with a CloudFront distribution in any region. In order
 * to create an ACM certificate in us-east-1 and reference it in a CloudFront distribution is a
 * different region, it is recommended to perform a multi stack deployment.
 * <p>
 * Enable the Stack property <code>crossRegionReferences</code>
 * in order to access the cross stack/region certificate.
 * <p>
 * <blockquote>
 * <p>
 * <strong>This feature is currently experimental</strong>
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudfront.*;
 * import software.amazon.awscdk.services.cloudfront.origins.*;
 * App app;
 * 
 * 
 * Stack stack1 = Stack.Builder.create(app, "Stack1")
 *         .env(Environment.builder()
 *                 .region("us-east-1")
 *                 .build())
 *         .crossRegionReferences(true)
 *         .build();
 * Certificate cert = Certificate.Builder.create(stack1, "Cert")
 *         .domainName("*.example.com")
 *         .validation(CertificateValidation.fromDns(PublicHostedZone.fromHostedZoneId(stack1, "Zone", "ZONE_ID")))
 *         .build();
 * 
 * Stack stack2 = Stack.Builder.create(app, "Stack2")
 *         .env(Environment.builder()
 *                 .region("us-east-2")
 *                 .build())
 *         .crossRegionReferences(true)
 *         .build();
 * 
 * Distribution.Builder.create(stack2, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(new HttpOrigin("example.com"))
 *                 .build())
 *         .domainNames(List.of("dev.example.com"))
 *         .certificate(cert)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Requesting private certificates</h2>
 * <p>
 * AWS Certificate Manager can create <a href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-request-private.html">private certificates</a> issued by <a href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaWelcome.html">Private Certificate Authority (PCA)</a>. Validation of private certificates is not necessary.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.acmpca.*;
 * 
 * 
 * PrivateCertificate.Builder.create(this, "PrivateCertificate")
 *         .domainName("test.example.com")
 *         .subjectAlternativeNames(List.of("cool.example.com", "test.example.net")) // optional
 *         .certificateAuthority(CertificateAuthority.fromCertificateAuthorityArn(this, "CA", "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/023077d8-2bfa-4eb0-8f22-05c96deade77"))
 *         .keyAlgorithm(KeyAlgorithm.RSA_2048)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Requesting public SSL/TLS certificates exportable to use anywhere</h2>
 * <p>
 * AWS Certificate Manager can issue an exportable public certificate. There is a charge at certificate issuance and again when the certificate renews. See <a href="https://docs.aws.amazon.com/acm/latest/userguide/acm-exportable-certificates.html">opting out of certificate transparency logging</a> for details.
 * <p>
 * <blockquote><pre>
 * Certificate.Builder.create(this, "Certificate")
 *         .domainName("test.example.com")
 *         .allowExport(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Requesting certificates without transparency logging</h2>
 * <p>
 * Transparency logging can be opted out of for AWS Certificate Manager certificates. See <a href="https://docs.aws.amazon.com/acm/latest/userguide/acm-bestpractices.html#best-practices-transparency">opting out of certificate transparency logging</a> for limits.
 * <p>
 * <blockquote><pre>
 * Certificate.Builder.create(this, "Certificate")
 *         .domainName("test.example.com")
 *         .transparencyLoggingEnabled(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Key Algorithms</h2>
 * <p>
 * To specify the algorithm of the public and private key pair that your certificate uses to encrypt data use the <code>keyAlgorithm</code> property.
 * <p>
 * Algorithms supported for an ACM certificate request include:
 * <p>
 * <ul>
 * <li><code>RSA_2048</code></li>
 * <li><code>EC_prime256v1</code></li>
 * <li><code>EC_secp384r1</code></li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Certificate.Builder.create(this, "Certificate")
 *         .domainName("test.example.com")
 *         .keyAlgorithm(KeyAlgorithm.EC_PRIME256V1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.aws.amazon.com/acm/latest/userguide/acm-certificate.html#algorithms.title">Key algorithms</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Importing</h2>
 * <p>
 * If you want to import an existing certificate, you can do so from its ARN:
 * <p>
 * <blockquote><pre>
 * String arn = "arn:aws:...";
 * ICertificate certificate = Certificate.fromCertificateArn(this, "Certificate", arn);
 * </pre></blockquote>
 * <p>
 * <h2>Sharing between Stacks</h2>
 * <p>
 * To share the certificate between stacks in the same CDK application, simply
 * pass the <code>Certificate</code> object between the stacks.
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * The <code>DaysToExpiry</code> metric is available via the <code>metricDaysToExpiry</code> method for
 * all certificates. This metric is emitted by AWS Certificates Manager once per
 * day until the certificate has effectively expired.
 * <p>
 * An alarm can be created to determine whether a certificate is soon due for
 * renewal ussing the following code:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudwatch.*;
 * 
 * HostedZone myHostedZone;
 * 
 * Certificate certificate = Certificate.Builder.create(this, "Certificate")
 *         .domainName("hello.example.com")
 *         .validation(CertificateValidation.fromDns(myHostedZone))
 *         .build();
 * certificate.metricDaysToExpiry().createAlarm(this, "Alarm", CreateAlarmOptions.builder()
 *         .comparisonOperator(ComparisonOperator.LESS_THAN_THRESHOLD)
 *         .evaluationPeriods(1)
 *         .threshold(45)
 *         .build());
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.certificatemanager;
