/**
 * <h1>Amazon FSx Construct Library</h1>
 * <p>
 * <a href="https://docs.aws.amazon.com/fsx/?id=docs_gateway">Amazon FSx</a> provides fully managed third-party file systems with the
 * native compatibility and feature sets for workloads such as Microsoft Windows–based storage, high-performance computing,
 * machine learning, and electronic design automation.
 * <p>
 * Amazon FSx supports two file system types: <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/index.html">Lustre</a> and
 * <a href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/index.html">Windows</a> File Server.
 * <p>
 * <h2>FSx for Lustre</h2>
 * <p>
 * Amazon FSx for Lustre makes it easy and cost-effective to launch and run the popular, high-performance Lustre file
 * system. You use Lustre for workloads where speed matters, such as machine learning, high performance computing (HPC),
 * video processing, and financial modeling.
 * <p>
 * The open-source Lustre file system is designed for applications that require fast storage—where you want your storage
 * to keep up with your compute. Lustre was built to solve the problem of quickly and cheaply processing the world's
 * ever-growing datasets. It's a widely used file system designed for the fastest computers in the world. It provides
 * submillisecond latencies, up to hundreds of GBps of throughput, and up to millions of IOPS. For more information on
 * Lustre, see the <a href="http://lustre.org/">Lustre website</a>.
 * <p>
 * As a fully managed service, Amazon FSx makes it easier for you to use Lustre for workloads where storage speed matters.
 * Amazon FSx for Lustre eliminates the traditional complexity of setting up and managing Lustre file systems, enabling
 * you to spin up and run a battle-tested high-performance file system in minutes. It also provides multiple deployment
 * options so you can optimize cost for your needs.
 * <p>
 * Amazon FSx for Lustre is POSIX-compliant, so you can use your current Linux-based applications without having to make
 * any changes. Amazon FSx for Lustre provides a native file system interface and works as any file system does with your
 * Linux operating system. It also provides read-after-write consistency and supports file locking.
 * <p>
 * <h3>Installation</h3>
 * <p>
 * Import to your project:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.fsx.*;
 * </pre></blockquote>
 * <p>
 * <h3>Basic Usage</h3>
 * <p>
 * Setup required properties and create:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * LustreFileSystem fileSystem = LustreFileSystem.Builder.create(this, "FsxLustreFileSystem")
 *         .lustreConfiguration(LustreConfiguration.builder().deploymentType(LustreDeploymentType.SCRATCH_2).build())
 *         .storageCapacityGiB(1200)
 *         .vpc(vpc)
 *         .vpcSubnet(vpc.getPrivateSubnets()[0])
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>File System Type Version</h3>
 * <p>
 * You can set <a href="https://docs.aws.amazon.com/ja_jp/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html#cfn-fsx-filesystem-filesystemtypeversion">the Lustre version for the file system</a>. To do this, use the <code>fileSystemTypeVersion</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * LustreFileSystem fileSystem = LustreFileSystem.Builder.create(this, "FsxLustreFileSystem")
 *         .lustreConfiguration(LustreConfiguration.builder().deploymentType(LustreDeploymentType.SCRATCH_2).build())
 *         .storageCapacityGiB(1200)
 *         .vpc(vpc)
 *         .vpcSubnet(vpc.getPrivateSubnets()[0])
 *         .fileSystemTypeVersion(FileSystemTypeVersion.V_2_15)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: The <code>fileSystemTypeVersion</code> has a restrictions on the values that can be set based on the <code>deploymentType</code>.
 * <p>
 * <ul>
 * <li><code>V_2_10</code> is supported by the Scratch and <code>PERSISTENT_1</code> deployment types.</li>
 * <li><code>V_2_12</code> is supported by all Lustre deployment types.</li>
 * <li><code>V_2_15</code> is supported by all Lustre deployment types and is recommended for all new file systems.</li>
 * </ul>
 * <p>
 * <strong>Note</strong>: The default value of <code>fileSystemTypeVersion</code> is <code>V_2_10</code> except for <code>PERSISTENT_2</code> deployment type where the default value is <code>V_2_12</code>.
 * <p>
 * <h3>Connecting</h3>
 * <p>
 * To control who can access the file system, use the <code>.connections</code> attribute. FSx has a fixed default port, so you don't
 * need to specify the port. This example allows an EC2 instance to connect to a file system:
 * <p>
 * <blockquote><pre>
 * LustreFileSystem fileSystem;
 * Instance instance;
 * 
 * 
 * fileSystem.connections.allowDefaultPortFrom(instance);
 * </pre></blockquote>
 * <p>
 * <h3>Mounting</h3>
 * <p>
 * The LustreFileSystem Construct exposes both the DNS name of the file system as well as its mount name, which can be
 * used to mount the file system on an EC2 instance. The following example shows how to bring up a file system and EC2
 * instance, and then use User Data to mount the file system on the instance at start-up:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * Vpc vpc;
 * 
 * Map&lt;String, LustreDeploymentType&gt; lustreConfiguration = Map.of(
 *         "deploymentType", LustreDeploymentType.SCRATCH_2);
 * 
 * LustreFileSystem fs = LustreFileSystem.Builder.create(this, "FsxLustreFileSystem")
 *         .lustreConfiguration(lustreConfiguration)
 *         .storageCapacityGiB(1200)
 *         .vpc(vpc)
 *         .vpcSubnet(vpc.getPrivateSubnets()[0])
 *         .build();
 * 
 * Instance inst = Instance.Builder.create(this, "inst")
 *         .instanceType(InstanceType.of(InstanceClass.T2, InstanceSize.LARGE))
 *         .machineImage(AmazonLinuxImage.Builder.create()
 *                 .generation(AmazonLinuxGeneration.AMAZON_LINUX_2)
 *                 .build())
 *         .vpc(vpc)
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .build();
 * fs.connections.allowDefaultPortFrom(inst);
 * 
 * // Need to give the instance access to read information about FSx to determine the file system's mount name.
 * inst.role.addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName("AmazonFSxReadOnlyAccess"));
 * 
 * String mountPath = "/mnt/fsx";
 * String dnsName = fs.getDnsName();
 * String mountName = fs.getMountName();
 * 
 * inst.userData.addCommands("set -eux", "yum update -y", "amazon-linux-extras install -y lustre2.10", String.format("mkdir -p %s", mountPath), String.format("chmod 777 %s", mountPath), String.format("chown ec2-user:ec2-user %s", mountPath), String.format("echo \"%s&#64;tcp:/%s %s lustre defaults,noatime,flock,_netdev 0 0\" &gt;&gt; /etc/fstab", dnsName, mountName, mountPath), "mount -a");
 * </pre></blockquote>
 * <p>
 * <h3>Importing an existing Lustre filesystem</h3>
 * <p>
 * An FSx for Lustre file system can be imported with <code>fromLustreFileSystemAttributes(this, id, attributes)</code>. The
 * following example lays out how you could import the SecurityGroup a file system belongs to, use that to import the file
 * system, and then also import the VPC the file system is in and add an EC2 instance to it, giving it access to the file
 * system.
 * <p>
 * <blockquote><pre>
 * ISecurityGroup sg = SecurityGroup.fromSecurityGroupId(this, "FsxSecurityGroup", "{SECURITY-GROUP-ID}");
 * IFileSystem fs = LustreFileSystem.fromLustreFileSystemAttributes(this, "FsxLustreFileSystem", FileSystemAttributes.builder()
 *         .dnsName("{FILE-SYSTEM-DNS-NAME}")
 *         .fileSystemId("{FILE-SYSTEM-ID}")
 *         .securityGroup(sg)
 *         .build());
 * 
 * IVpc vpc = Vpc.fromVpcAttributes(this, "Vpc", VpcAttributes.builder()
 *         .availabilityZones(List.of("us-west-2a", "us-west-2b"))
 *         .publicSubnetIds(List.of("{US-WEST-2A-SUBNET-ID}", "{US-WEST-2B-SUBNET-ID}"))
 *         .vpcId("{VPC-ID}")
 *         .build());
 * 
 * Instance inst = Instance.Builder.create(this, "inst")
 *         .instanceType(InstanceType.of(InstanceClass.T2, InstanceSize.LARGE))
 *         .machineImage(AmazonLinuxImage.Builder.create()
 *                 .generation(AmazonLinuxGeneration.AMAZON_LINUX_2)
 *                 .build())
 *         .vpc(vpc)
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .build();
 * 
 * fs.connections.allowDefaultPortFrom(inst);
 * </pre></blockquote>
 * <p>
 * <h3>Lustre Data Repository Association support</h3>
 * <p>
 * The LustreFilesystem Construct supports one <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/fsx-data-repositories.html">Data Repository Association</a> (DRA) to an S3 bucket.  This allows Lustre hierarchical storage management to S3 buckets, which in turn makes it possible to use S3 as a permanent backing store, and use FSx for Lustre as a temporary high performance cache.
 * <p>
 * Note: CloudFormation does not currently support for <code>PERSISTENT_2</code> filesystems, and so neither does CDK.
 * <p>
 * The following example illustrates setting up a DRA to an S3 bucket, including automated metadata import whenever a file is changed, created or deleted in the S3 bucket:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * 
 * Vpc vpc;
 * Bucket bucket;
 * 
 * 
 * Map&lt;String, Object&gt; lustreConfiguration = Map.of(
 *         "deploymentType", LustreDeploymentType.SCRATCH_2,
 *         "exportPath", bucket.s3UrlForObject(),
 *         "importPath", bucket.s3UrlForObject(),
 *         "autoImportPolicy", LustreAutoImportPolicy.NEW_CHANGED_DELETED);
 * 
 * LustreFileSystem fs = LustreFileSystem.Builder.create(this, "FsxLustreFileSystem")
 *         .vpc(vpc)
 *         .vpcSubnet(vpc.getPrivateSubnets()[0])
 *         .storageCapacityGiB(1200)
 *         .lustreConfiguration(lustreConfiguration)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Compression</h3>
 * <p>
 * By default, transparent compression of data within FSx for Lustre is switched off.  To enable it, add the following to your <code>lustreConfiguration</code>:
 * <p>
 * <blockquote><pre>
 * Map&lt;String, LustreDataCompressionType&gt; lustreConfiguration = Map.of(
 *         // ...
 *         "dataCompressionType", LustreDataCompressionType.LZ4);
 * </pre></blockquote>
 * <p>
 * When you turn data compression on for an existing file system, only newly written files are compressed.  Existing files are not compressed. For more information, see <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-compression.html#migrate-compression">Compressing previously written files</a>.
 * <p>
 * <h3>Backups</h3>
 * <p>
 * You can take daily automatic backups by setting <code>automaticBackupRetention</code> to a non-zero day in the <code>lustreConfiguration</code>.
 * <p>
 * Additionally, you can set the backup window by specifying the <code>dailyAutomaticBackupStartTime</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * Map&lt;String, Object&gt; lustreConfiguration = Map.of(
 *         // ...
 *         "automaticBackupRetention", Duration.days(3),  // backup retention
 *         "copyTagsToBackups", true,  // if true, tags are copied to backups
 *         "dailyAutomaticBackupStartTime", DailyAutomaticBackupStartTime.Builder.create().hour(11).minute(30).build());
 * </pre></blockquote>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/using-backups-fsx.html">Working with backups
 * </a>.
 * <p>
 * <h3>Storage Type</h3>
 * <p>
 * By default, FSx for Lustre uses SSD storage. To use HDD storage, specify <code>storageType</code>:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * LustreFileSystem fileSystem = LustreFileSystem.Builder.create(this, "FsxLustreFileSystem")
 *         .lustreConfiguration(LustreConfiguration.builder().deploymentType(LustreDeploymentType.PERSISTENT_1).build())
 *         .storageCapacityGiB(1200)
 *         .vpc(vpc)
 *         .vpcSubnet(vpc.getPrivateSubnets()[0])
 *         .storageType(StorageType.HDD)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note:</strong> The HDD storage type is only supported for <code>PERSISTENT_1</code> deployment types.
 * <p>
 * To improve the performance of frequently accessed files by caching up to 20% of the total storage capacity of the file system, set <code>driveCacheType</code> to <code>READ</code>:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * LustreFileSystem fileSystem = LustreFileSystem.Builder.create(this, "FsxLustreFileSystem")
 *         .lustreConfiguration(LustreConfiguration.builder()
 *                 .deploymentType(LustreDeploymentType.PERSISTENT_1)
 *                 .driveCacheType(DriveCacheType.READ)
 *                 .build())
 *         .storageCapacityGiB(1200)
 *         .vpc(vpc)
 *         .vpcSubnet(vpc.getPrivateSubnets()[0])
 *         .storageType(StorageType.HDD)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>FSx for Windows File Server</h2>
 * <p>
 * The L2 construct for the FSx for Windows File Server has not yet been implemented. To instantiate an FSx for Windows
 * file system, the L1 constructs can be used as defined by CloudFormation.
 */
package software.amazon.awscdk.services.fsx;
