/**
 * <h1>Amazon DocumentDB Construct Library</h1>
 * <p>
 * <h2>Starting a Clustered Database</h2>
 * <p>
 * To set up a clustered DocumentDB database, define a <code>DatabaseCluster</code>. You must
 * always launch a database in a VPC. Use the <code>vpcSubnets</code> attribute to control whether
 * your instances will be launched privately or publicly:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser") // NOTE: 'admin' is reserved by DocumentDB
 *                 .excludeCharacters("\"&#64;/:") // optional, defaults to the set "\"&#64;/" and is also used for eventually created rotations
 *                 .secretName("/myapp/mydocdb/masteruser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.MEMORY5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .copyTagsToSnapshot(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, the master password will be generated and stored in AWS Secrets Manager with auto-generated description.
 * <p>
 * Your cluster will be empty by default.
 * <p>
 * <h2>Serverless Clusters</h2>
 * <p>
 * DocumentDB supports serverless clusters that automatically scale capacity based on your application's needs.
 * To create a serverless cluster, specify the <code>serverlessV2ScalingConfiguration</code> instead of <code>instanceType</code>:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .vpc(vpc)
 *         .serverlessV2ScalingConfiguration(ServerlessV2ScalingConfiguration.builder()
 *                 .minCapacity(0.5)
 *                 .maxCapacity(2)
 *                 .build())
 *         .engineVersion("5.0.0")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: DocumentDB serverless requires engine version 5.0.0 or higher and is not compatible with all features. See the <a href="https://docs.aws.amazon.com/documentdb/latest/developerguide/docdb-serverless-limitations.html">AWS documentation</a> for limitations.
 * <p>
 * <h2>Connecting</h2>
 * <p>
 * To control who can access the cluster, use the <code>.connections</code> attribute. DocumentDB databases have a default port, so
 * you don't need to specify the port:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * cluster.connections.allowDefaultPortFromAnyIpv4("Open to the world");
 * </pre></blockquote>
 * <p>
 * The endpoints to access your database cluster will be available as the <code>.clusterEndpoint</code> and <code>.clusterReadEndpoint</code>
 * attributes:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * String writeAddress = cluster.getClusterEndpoint().getSocketAddress();
 * </pre></blockquote>
 * <p>
 * If you have existing security groups you would like to add to the cluster, use the <code>addSecurityGroups</code> method. Security
 * groups added in this way will not be managed by the <code>Connections</code> object of the cluster.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * DatabaseCluster cluster;
 * 
 * 
 * SecurityGroup securityGroup = SecurityGroup.Builder.create(this, "SecurityGroup")
 *         .vpc(vpc)
 *         .build();
 * cluster.addSecurityGroups(securityGroup);
 * </pre></blockquote>
 * <p>
 * <h2>Deletion protection</h2>
 * <p>
 * Deletion protection can be enabled on an Amazon DocumentDB cluster to prevent accidental deletion of the cluster:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.MEMORY5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .deletionProtection(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Rotating credentials</h2>
 * <p>
 * When the master password is generated and stored in AWS Secrets Manager, it can be rotated automatically:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * cluster.addRotationSingleUser();
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(stack, "Database")
 *         .masterUser(Login.builder()
 *                 .username("docdb")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.R5, InstanceSize.LARGE))
 *         .vpc(vpc)
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .build();
 * 
 * cluster.addRotationSingleUser();
 * </pre></blockquote>
 * <p>
 * The multi user rotation scheme is also available:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.secretsmanager.*;
 * 
 * Secret myImportedSecret;
 * DatabaseCluster cluster;
 * 
 * 
 * cluster.addRotationMultiUser("MyUser", RotationMultiUserOptions.builder()
 *         .secret(myImportedSecret)
 *         .build());
 * </pre></blockquote>
 * <p>
 * It's also possible to create user credentials together with the cluster and add rotation:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * DatabaseSecret myUserSecret = DatabaseSecret.Builder.create(this, "MyUserSecret")
 *         .username("myuser")
 *         .masterSecret(cluster.getSecret())
 *         .build();
 * ISecret myUserSecretAttached = myUserSecret.attach(cluster); // Adds DB connections information in the secret
 * 
 * cluster.addRotationMultiUser("MyUser", RotationMultiUserOptions.builder() // Add rotation using the multi user scheme
 *         .secret(myUserSecretAttached).build());
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: This user must be created manually in the database using the master credentials.
 * The rotation will start as soon as this user exists.
 * <p>
 * See also <a href="https://github.com/aws/aws-cdk/blob/main/packages/aws-cdk-lib/aws-secretsmanager/README.md">aws-cdk-lib/aws-secretsmanager</a> for credentials rotation of existing clusters.
 * <p>
 * <h2>Audit and profiler Logs</h2>
 * <p>
 * Sending audit or profiler needs to be configured in two places:
 * <p>
 * <ol>
 * <li>Check / create the needed options in your ParameterGroup for <a href="https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html#event-auditing-enabling-auditing">audit</a> and
 * <a href="https://docs.aws.amazon.com/documentdb/latest/developerguide/profiling.html#profiling.enable-profiling">profiler</a> logs.</li>
 * <li>Enable the corresponding option(s) when creating the <code>DatabaseCluster</code>:</li>
 * </ol>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * import software.amazon.awscdk.services.logs.*;
 * 
 * Role myLogsPublishingRole;
 * Vpc vpc;
 * 
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.MEMORY5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .exportProfilerLogsToCloudWatch(true) // Enable sending profiler logs
 *         .exportAuditLogsToCloudWatch(true) // Enable sending audit logs
 *         .cloudWatchLogsRetention(RetentionDays.THREE_MONTHS) // Optional - default is to never expire logs
 *         .cloudWatchLogsRetentionRole(myLogsPublishingRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Enable Performance Insights</h2>
 * <p>
 * By enabling this feature it will be cascaded and enabled in all instances inside the cluster:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.MEMORY5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .enablePerformanceInsights(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * ## Removal Policy
 * <p>
 * This resource supports the snapshot removal policy.
 * To specify it use the <code>removalPolicy</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.MEMORY5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .removalPolicy(RemovalPolicy.SNAPSHOT)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: A <code>RemovalPolicy.DESTROY</code> removal policy will be applied to the
 * cluster's instances and security group by default as they don't support the snapshot
 * removal policy.
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html">DeletionPolicy</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * To specify a custom removal policy for the cluster's instances, use the
 * <code>instanceRemovalPolicy</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.MEMORY5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .removalPolicy(RemovalPolicy.SNAPSHOT)
 *         .instanceRemovalPolicy(RemovalPolicy.RETAIN)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To specify a custom removal policy for the cluster's security group, use the
 * <code>securityGroupRemovalPolicy</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.MEMORY5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .removalPolicy(RemovalPolicy.SNAPSHOT)
 *         .securityGroupRemovalPolicy(RemovalPolicy.RETAIN)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>CA certificate</h2>
 * <p>
 * Use the <code>caCertificate</code> property to specify the <a href="https://docs.aws.amazon.com/documentdb/latest/developerguide/ca_cert_rotation.html">CA certificate</a> to use for all instances inside the cluster:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.MEMORY5, InstanceSize.LARGE))
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PUBLIC)
 *                 .build())
 *         .vpc(vpc)
 *         .caCertificate(CaCertificate.RDS_CA_RSA4096_G1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Storage Type</h2>
 * <p>
 * You can specify <a href="https://docs.aws.amazon.com/documentdb/latest/developerguide/db-cluster-storage-configs.html">storage type</a> for the cluster.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .masterUser(Login.builder()
 *                 .username("myuser")
 *                 .build())
 *         .instanceType(InstanceType.of(InstanceClass.MEMORY5, InstanceSize.LARGE))
 *         .vpc(vpc)
 *         .storageType(StorageType.IOPT1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: <code>StorageType.IOPT1</code> is supported starting with engine version 5.0.0.
 * <p>
 * <strong>Note</strong>: For serverless clusters, storage type is managed automatically and cannot be specified.
 */
package software.amazon.awscdk.services.docdb;
